#define HSA_VERSION	0.5	/* current version number of this routine */
//
// HP Spectrum Analyzer GPIB Routines (HP 8593A)  488 VERSION!!!
//		Models supported:
//			HP8562A		not completely tested, but no known problems
//			HP8593A		not fully tested, but no known problems
//			HP8561A		not completely tested, but no known problems
//			HP8563E		not completely tested, but no known problems
//
//	NOTE: The application program MUST CALL "hsa_preset" to initialize the
//		the proper globals!!!!!
//
//
//		6/28/96  VERSION 0.5 csp
//			Added B trace functions: 
//				hsa_max_holdB (dev) - set B trace to max hold mode
//				hsa_clear_writeB (dev)	clear B trace, resets from max_hold
//				hsa_blank_B (dev)	- set trace B blank.
//				hsa_sweep_auto (dev)		set sweep time to auto
//				on hsa_sweep_time, had to remove the S; (seconds) for it to
//					work on the HP8593A analyzer w/o giving an error.
//				hsa_get_trace_size - returns the number of elements in 
//					a returned trace for this analyzer.  Takes value from
//					'model' table, does not ask analyzer for the value.
//				hsa_init - Like hsa_preset, but it does not do an "IP"
//					to reset the analyzer.  This should only be used with
//					caution.  The procedure does determine the analyzer type
//					and sets the variables accordingly.
//
//		2/2/93  VERSION 0.4  csp
//			No checks only the model number (first 7 characters) of the
//				returned model code.  It ignores extension.
//			Also, fixed bug in returning 602 (or 402) points in hsa_M_FORMAT_to_int
//				procedure.
//
//		1/25/93 VERSION 0.3  csp
//			1) added support for the HP 8563E spectrum Analyzer.
//			2) Fixed bug in hsa_M_FORMAT_to_int routine on last point.
//				[Was giving a 0x0a (\r) as last character.]
//
//		4/15/92 VERSION 0.2.1  csp
//			Added analyzer type HP8561,003 (cause that's what it said). csp
//			untested with other analyzers.
//	POTENTIAL BUG UNCOVERED: (with 601 point models at least), dimension the
//		integer trace buffer (passed to hsa_M_FORMAT_to_int) as at least one
//		element larger that what is expected (my example used a point of 605
//		points instead of 601 pts).  This routine appears to do one too many
//		points overwriting whatever the following variable is.
//		4/09/92 VERSION 0.2  csp
//			0.2.1  Made more universal for HP spectrum analyzers by using the
//				hsa_preset to initialize the number of points per trace into the
//				global variable "???" based on the spectrum analyzer model.
//				Models supported listed above.
//			0.1.4 Within the hsa_preset function, the type of analyzer is determined
//				using the ID? command.  Based on the response, the following global 
//				variables are set:
//					hsa_points_per_trace = number of points returned in a trace
//					hsa_M_top = returned units for top of screen (for M_format)
//					hsa_M_divisions = number of y divisions on screen
//
//				ADDING A SPECTRUM ANALYZER IS DONE THROUGH THE STRUCTURE "MODELS"
//					and incrementing the variable HSA_NUMBER_MODELS.
//
//	  4/09/92  VERSION 0.1  
//			0.1.1 Revised trigger routine to use DONE? function for
//			proper synchronization since the ONEOS is not supported by all our
//			spectrum analyzers.
//			0.1.2 Added new functions: hsa_set_time_date to set the analyzer's
//				clock and calendar (if unit is HP8562A, then skips as it has no
//				time and date transfer).
//			0.1.3 Added "Hz" to all frequency set commands to make it work with
//				HP8562 spectrum analyzer properly.
//
//  (OBSOLETE) 4/17/91 Added some logic to help the trigger situation properly using the
//   ONEOS command.  Also added global variable hsa_mode_global;  Affects
//		procedures hsa_sweep_mode and hsa_get_traceA.  The routine, "hsa_sweep"
//		has been removed since triggering is now part of the _get_traceA.
//
//  Routines
//		hsa_preset (dev)		set to preset value and INITIALIZE the system (IP)
//		hsa_init (dev)		Init system globals based on analyze type, no IP
//		hsa_get_trace_size() - returns the number of elements in a trace
//		hsa_sweep_mode (dev, mode)		set sweep mode
//  	hsa_sweep				trigger a sweep (and wait till sweep complete)
//		hsa_cf (dev, freq)		set center frequency
//		hsa_ask_cf (dev)		ask center frequency
//		hsa_sf (dev, freq)		set starting frequency
//		hsa_ask_sf (dev)		ask starting frequency
//		hsa_ef (dev, freq)		set ending frequency
//		hsa_ask_ef (dev)		ask ending frequency
//		hsa_span (dev, span)		set span
//		hsa_ask_span (dev)		ask span
//		hsa_ref_level (dev, level)		set reference level
//		hsa_ask_ref_level (dev)		ask reference level
//		hsa_trace_format (dev, format)   set trace format
//		hsa_get_traceA (dev, buffer)		get entire buffer for trace A
//		hsa_M_FORMAT_to_int (in, ref, out)  convert from buffer to int dBm array
//		hsa_max_holdA (dev)		set A trace to max hold mode
//		hsa_clear_writeA (dev)	clear A trace, resets from max_hold
//		hsa_max_holdB (dev)		set B trace to max hold mode
//		hsa_clear_writeB (dev)	clear B trace, resets from max_hold
//		hsa_blank_B (dev)	blank B trace, resets from max_hold
//		hsa_units (dev, measure)	set units of measure
//*		hsa_ask_units (dev)	set units of measure
//		hsa_atten (dev, value)	set attenuation (value is in dB and integer)
//		hsa_ask_atten (dev)		ask attenuation level (dB and integer)
//		hsa_atten_auto (dev)		set attenuation to auto
//		hsa_sweep_time (dev, time)		set sweep time
//		hsa_sweep_auto (dev)		set sweep time to auto
//		hsa_ask_sweep_time (dev)		ask sweep time
//		hsa_rb (dev, value)		set resolution bandwidth
//		hsa_ask_rb (dev)		ask resolution bandwidth
//		hsa_vb (dev, value)		set video bandwidth
//		hsa_ask_vb (dev)		ask video bandwidth
//		hsa_hold (dev)			do HOLD operation (clear screen)
//		hsa_title (dev, string)	write title
//		hsa_set_time_date(dev)	set time/date on analyzer to PC time/date (if possible)


// Be sure functions are defined properly (those that a non-integer)
double hsa_ask_cf();
double hsa_ask_sf();
double hsa_ask_ef();
double hsa_ask_span();
double hsa_ask_ref_level();
double hsa_ask_sweep_time();
double hsa_ask_rb();
double hsa_ask_vb();
int hsa_get_trace_size ();

/* define the models supported here */
struct	MODELS {
	char	model_name[15];	/* model number (in response to ID?) */
	int	points;	/* number points per trace */
	int	top;	/* top of screen in screen units (for M_format) */
	int	divisions_per_screen;	/* number y division per screen */
}
	models[] = {{"HP8562A", 601, 600, 10},
				 {"HP8593A", 401, 8000, 8},
				 {"HP8561B", 601, 600, 10},
				 {"HP8563E", 601, 600, 10},
				 {"HP8560A", 601, 600, 10}

				/* add additional models here --be sure to change HSA_NUMBER_MODELS */
				};	/* end spectrum analyzer models */
#define	HSA_NUMBER_MODELS	5	/* number of models in list */

#define HSA_NUM_LENGTH	24		// max number of char returned (numeric data)
#define HSA_BUF_LENGTH	4096	// max characters in get trace (works with 601 traces)

// Constants for spectrum analyzer routines
#define	DBM	1		// units of measure cases for AUNITS
#define	DBMV	2
#define	DBUV	3
#define	V	4
#define	W	5		// end units of measure cases for AUNITS
#define	SWEEP_CONTINUOUS	6		// used to set sweep continuous
#define	SWEEP_SINGLE	7		// set sweep in single mode
#define	P_FORMAT	8		// trace format,  parameter units
#define	A_FORMAT	9		// trace format,  a-block format
#define	M_FORMAT 10		// trace format,  measurement units
#define	I_FORMAT 11		// trace format,  i-block units
#define	B_FORMAT 12		// trace format,  binary units

#include <stdio.h>
#include <math.h>
#include <string.h>
#include	<time.h>

// =================================================================

// ---------- Global Variables for hsa_prog routines ----------------
int	hsa_mode_global = SWEEP_CONTINUOUS;	// current mode of sweeping
int	hsa_number = -1;	/* index of model number (0 = first in table, etc.) */
int	hsa_points_per_trace = 401;	/* default here */
int	hsa_M_top;	/* holds the reference for top of screen (used in M format decode) */
int	hsa_M_divisions;	/* holds divisions per y axis for this analyzer */

hsa_preset (dev)		// set analyzer and initialize to preset state
int dev;
{
	int	len,i;
	char	buffer[16];

	gpib_write (dev, "IP;");
//	fprintf(gpib_out, "Output %d;IP;\n", dev);

	hsa_init();	/* 0.5, init below */

#ifdef DELETED
/* See if the unit is a HP8562A, if so skip the set time date function,
   it is not supported in this unit */

	gpib_write(dev,"ID?;");

	len = gpib_read (dev, buffer, 15);
	buffer[len] = 0;		// terminate string properly

// printf ("ID? yeileded <%s>\n", buffer); getch();

	for (i=0; i< HSA_NUMBER_MODELS; i++) {
//printf ("Checking <%s> with <%s>\n", models[i].model_name, buffer);
//		if (strstr (buffer, models[i].model_name) != NULL)
/*  See if the model number (first 7 characters) matches a table entry */
		if (strnicmp (buffer, models[i].model_name, 7) == 0)
		{  /* Model matches */
// printf ("Match found at i = %d (%s %d)\n", i, models[i].model_name, models[i].points);
			printf ("Analyzer Type: %s\n", models[i].model_name);
			hsa_number = i;	/* save model number index */
			hsa_points_per_trace = models[i].points;
			hsa_M_top = models[i].top	;	/* holds the reference for top of screen (used in M format decode) */
			hsa_M_divisions = models[i].divisions_per_screen ;	/* holds divisions per y axis for this analyzer */
			return (0);
		}	/* endif */
	}	/* end for -- no match, error, abort */
	printf ("The spectrum analyzer, model <%s>, is invalid, program aborted\n", buffer);
	exit (3);
#endif

}		// end hsa_preset

hsa_init (int dev)
/* Initialize globals based on analyzer type */
{
	int	len,i;
	char	buffer[16];
/* See if the unit is a HP8562A, if so skip the set time date function,
   it is not supported in this unit */

	gpib_write(dev,"ID?;");

	len = gpib_read (dev, buffer, 15);
	buffer[len] = 0;		// terminate string properly

// printf ("ID? yeileded <%s>\n", buffer); getch();

	for (i=0; i< HSA_NUMBER_MODELS; i++) {
//printf ("Checking <%s> with <%s>\n", models[i].model_name, buffer);
//		if (strstr (buffer, models[i].model_name) != NULL)
/*  See if the model number (first 7 characters) matches a table entry */
		if (strnicmp (buffer, models[i].model_name, 7) == 0)
		{  /* Model matches */
// printf ("Match found at i = %d (%s %d)\n", i, models[i].model_name, models[i].points);
			printf ("Analyzer Type: %s\n", models[i].model_name);
			hsa_number = i;	/* save model number index */
			hsa_points_per_trace = models[i].points;
			hsa_M_top = models[i].top	;	/* holds the reference for top of screen (used in M format decode) */
			hsa_M_divisions = models[i].divisions_per_screen ;	/* holds divisions per y axis for this analyzer */
			return (0);
		}	/* endif */
	}	/* end for -- no match, error, abort */
	printf ("The spectrum analyzer, model <%s>, is invalid, program aborted\n", buffer);
	exit (3);


}	/* hsa_init */

int hsa_get_trace_size ()
{
			return (hsa_points_per_trace);
}

hsa_sweep_mode (dev, mode)		// set sweep mode where
int dev;
int mode;  // SWEEP_CONTINUOUS, SWEEP_SINGLE
{
	switch (mode) {
		case SWEEP_CONTINUOUS :
			gpib_write (dev, "CONTS;");
			hsa_mode_global = SWEEP_CONTINUOUS;
//			fprintf(gpib_out, "Output %d;CONTS;\n", dev);
			break;
		case SWEEP_SINGLE :
			gpib_write (dev, "SNGLS;");
			hsa_mode_global = SWEEP_SINGLE;
//			fprintf(gpib_out, "Output %d;SNGLS;\n", dev);
			break;
		default:
			printf ("Incorrect sweep mode, entry = %d\n", mode);
			break;
	}		// end case
}		// end hsa_sweep_mode

hsa_sweep (dev)		// trigger a sweep and wait till complete (revision 0.01)
int	dev;
{
	int	len;
	char	string[4];

//	gpib_write (dev, "TS;DONE?;");  /* worked better splitting out into 2 lines */
	gpib_write (dev, "TS;");
	gpib_write (dev, "DONE?;");
//	fprintf(gpib_out, "Output %d;TS;\n", dev);
	len = gpib_read (dev, string, 4);	/* wait for reply */

}		// end hsa_sweep

hsa_cf (dev, freq)		// set center frequency
	int	dev;
	double	freq;		// center frequency in Hz
{
	char	string[64];
//	sprintf (string, "CF %g;", freq);	0.1.3
	sprintf (string, "CF %gHZ;", freq);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;CF %g;\n", dev, freq);
}		// end hsa_cf

double hsa_ask_cf (dev)		// ask center frequency (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "CF?;");
//	fprintf(gpib_out, "Output %d;CF?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_cf

hsa_sf (dev, freq)		// set starting frequency
	int	dev;
	double	freq;		// starting frequency in Hz
{
	char	string[64];
//	sprintf (string, "FA %g;", freq);	0.1.3
	sprintf (string, "FA %gHZ;", freq);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;FA %g;\n", dev, freq);
}		// end hsa_sf

double hsa_ask_sf (dev)		// ask start frequency (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "FA?;");
//	fprintf(gpib_out, "Output %d;FA?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_sf

hsa_ef (dev, freq)		// set ending frequency
	int	dev;
	double	freq;		// ending frequency in Hz
{
	char	string[64];
//	sprintf (string, "FB %g;", freq);	0.1.3
	sprintf (string, "FB %gHZ;", freq);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;FB %g;\n", dev, freq);
}		// end hsa_ef

double hsa_ask_ef (dev)		// ask ending frequency (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "FB?;");
//	fprintf(gpib_out, "Output %d;FB?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_ef

hsa_span (dev, span)		// set span 
	int	dev;
	double	span;		// center frequency in Hz
{
	char	string[64];
//	sprintf (string, "SP %g;", span); 0.1.3
	sprintf (string, "SP %gHZ;", span);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;SP %g;\n", dev, span);
}		// end hsa_span

double hsa_ask_span (dev)		// ask span (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "SP?;");
//	fprintf(gpib_out, "Output %d;SP?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_span

hsa_ref_level (dev, level)		// set reference level
	int	dev;
	double	level;		// Reference level to set
{
	char	string[64];
	sprintf (string, "RL %gDB;", level); /* must be DB for 8593A */
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;RL %g;\n", dev, level);
}		// end hsa_ref_level

double hsa_ask_ref_level (dev)		// ask reference level
int	dev;
{
	double	value;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "RL?;");
//	fprintf(gpib_out, "Output %d;RL?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	value = atof (string);
	return (value);
}		// end hsa_ask_ref_level

hsa_trace_format (dev, format)		// set trace format
int	dev;
int	format;	//	P_FORMAT, A_FORMAT, M_FORMAT, I_FORMAT, B_FORMAT
{
	switch (format) {
		case P_FORMAT:
			gpib_write (dev, "TDF P;");
//			fprintf(gpib_out, "Output %d;TDF P;\n", dev);
			break;
		case A_FORMAT:
			gpib_write (dev, "TDF A;");
//			fprintf(gpib_out, "Output %d;TDF A;\n", dev);
			break;
		case M_FORMAT:
			gpib_write (dev, "TDF M;");
//			fprintf(gpib_out, "Output %d;TDF M;\n", dev);
			break;
		case I_FORMAT:
			gpib_write (dev, "TDF I;");
//			fprintf(gpib_out, "Output %d;TDF I;\n", dev);
			break;
		case B_FORMAT:
			gpib_write (dev, "TDF B;");
//			fprintf(gpib_out, "Output %d;TDF B;\n", dev);
			break;
	}		// end switch
}	// end hsa_trace_format

hsa_get_traceA (dev, buffer)		// get entire buffer for trace A
int	dev;
char	buffer[];		// buffer of returned values
// NOTE:  This routine is not extensively tested under all modes, but I
//  believe it accepts only text, not a binary based format.  "C" will
//  probably reject a binary fromat (nulls, CR, LF imbeded).
//	NOTE: Version 0.01 (4/09/92) revised to use hsa_sweep (which waits till
//		sweep is complete) to trigger trace.
{
	int	len, i,j;
//	char	string[HSA_NUM_LENGTH];

	switch (hsa_mode_global) {
		case (SWEEP_CONTINUOUS) :
			hsa_sweep (dev);	/* trigger a trace and return when complete */
			gpib_write (dev, "TRA?;");	/* ask for trace */
			len = gpib_read (dev, buffer, HSA_BUF_LENGTH);
			buffer[len] = 0;		// terminate string properly
//			gpib_write (dev, "DISPOSE ONEOS;");
			return (len);
			break;
		case (SWEEP_SINGLE) :
			hsa_sweep (dev);	/* trigger a trace and return when trace complete */
			gpib_write (dev, "TRA?;");	/* ask for trace results */
			len = gpib_read (dev, buffer, HSA_BUF_LENGTH);
			buffer[len] = 0;		// terminate string properly

#ifdef	XXXX
printf ("In hsa_get_trace, len, buffer length = %d, %d:\n", len, strlen(buffer));
	for (i = 0; i < strlen(buffer);i = i + 80) {
		for (j=i; j < i+80; j++) {
			printf ("%c", buffer[j]);
		}
		getch();
		printf ("\n");
	}
getch();
#endif
			return (len);
			break;
	}		// end switch
}		// end hsa_get_traceA

hsa_M_FORMAT_to_int (in, ref, out)  // convert from buffer to int dBm array
//  NOTE:  Returns in M_format type dBm units, I.E. -10000 equals -100.00
//   dBm.  Adjustment factor is "ref, which should be obtained by routine
//	  hsa_ask_ref_level before calling this routine.  To simply convert the
//   character sequence in "in" to integer without adjustment, set ref = +8000.0
//  NOTE2: Revised to us the number of bins on vertical axis (hsa_M_divisions)
//		properly and give the dBm value times 100.
//  1/26/93  Revised the SEPARATORS below to include \r, was giving 0x0a as last
//    character.
char	in[];		// input is string of M_FORMAT values xx,xxx,xxx,...
double ref;		// reference value for top line of scope, obtain with ask_ref_level
					//  before calling this routine
int	out[];	// output is in integer form (short)

#define M_FORMAT_SEPARATORS ",\n\r"	// possible separation characters 
/* #define M_FORMAT_SEPARATORS ",\n"	// possible separation characters  */
{
	double	ref_int;		// reference in integer
	int	i;
	int	number_done;		// count of number done
	char	*index;
	double	slope, y_intercept;	/* conversion factors */

	ref_int = (int) ref * 100;		// convert to integer units xx.xx = xxxx
	y_intercept = hsa_M_divisions * 1000.0;
	slope = y_intercept / hsa_M_top;

	number_done = 0;
	index = in;
	index--;
//	index = strpbrk (&in[0], M_FORMAT_SEPARATORS);
	while ((index != NULL) && (number_done < hsa_points_per_trace)) {
		index++;
//		out[number_done] = ref_int - 8000 + atoi (index);
		out[number_done] = ref_int - y_intercept + (int) ((double) atoi (index) * slope);
//printf ("String <%s>; ref_int %lf; factor %lf; atoi(index) %d\n",
//	index, ref_int, factor, atoi(index)); getch();
// printf ("number_done, index, out[number_done] : %d %p %d\n",number_done, index, out[number_done]);
		number_done++;
		index = strpbrk (index, M_FORMAT_SEPARATORS);
	}		// end while
	return (number_done - 1);
}		// end hsa_M_FORMAT_to_int

hsa_max_holdA (dev)		// set A trace to max hold mode
int	dev;
// NOTE: Added ONEOS to execute at end of trace.
{

	gpib_write (dev, "MXMH TRA;");
//	gpib_write (dev, "ONEOS!MXMH TRA;");	// removed 0.01
//	fprintf(gpib_out, "Output %d;ONEOS!MXMH TRA;!;\n", dev);
}		// end hsa_max_holdA

hsa_clear_writeA (dev)	// clear A trace, resets from max_hold
int	dev;
{
	gpib_write (dev, "CLRW TRA;");
//	fprintf(gpib_out, "Output %d;CLRW TRA;\n", dev);
}		// end hsa_clear_writeA

hsa_max_holdB (dev)		// set B trace to max hold mode
int	dev;
// NOTE: Added ONEOS to execute at end of trace.
{

	gpib_write (dev, "MXMH TRB;");  /* 0.5 */
}		// end hsa_max_holdB

hsa_clear_writeB (dev)	// clear B trace, resets from max_hold
int	dev;
{
	gpib_write (dev, "CLRW TRB;");  /* 0.5 */
}		// end hsa_clear_writeB

hsa_blank_B (dev)
int	dev;
{
	gpib_write (dev, "BLANK TRB;");  /* 0.5 */
}		// end hsa_clear_writeB


//		hsa_blank_B (dev)	blank B trace, resets from max_hold

hsa_units (dev, measure)	// set units of measure
int	dev;
int	measure;		// DBM, DBMV, DBUV, V, W
{
	switch (measure) {
		case DBM:
			gpib_write (dev, "AUNITS DBM;");
//			fprintf(gpib_out, "Output %d;AUNITS DBM;\n", dev);
			break;
		case DBMV:
			gpib_write (dev, "AUNITS DBMV;");
//			fprintf(gpib_out, "Output %d;AUNITS DBMV;\n", dev);
			break;
		case DBUV:
			gpib_write (dev, "AUNITS DBUV;");
//			fprintf(gpib_out, "Output %d;AUNITS DBUV;\n", dev);
			break;
		case V:
			gpib_write (dev, "AUNITS V;");
//			fprintf(gpib_out, "Output %d;AUNITS V;\n", dev);
			break;
		case W:
			gpib_write (dev, "AUNITS W;");
//			fprintf(gpib_out, "Output %d;AUNITS W;\n", dev);
			break;
	}		// end switch
}	// end hsa_units

hsa_atten (dev, value)	// set attenuation
int	dev;
int	value;		// attenuation to set (INTEGER in dB)
{
	char	string[64];
	sprintf (string, "AT %dDB;", value);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;AT %dDB;\n", dev, value);
}		// end hsa_atten

hsa_ask_atten (dev)		// ask Attenuation (returns INTEGER in dBm)
	int	dev;
{
	int	value;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "AT?;");
//	fprintf(gpib_out, "Output %d;AT?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	value = atoi (string);
	return (value);
}		// end hsa_ask_atten

// hsa_ask_atten (dev)		ask attenuation level

hsa_atten_auto (dev)		// set attenuation to auto
int	dev;
{
	gpib_write (dev, "AT AUTO;");
//	fprintf(gpib_out, "Output %d;AT AUTO;\n", dev);
}		// end hsa_atten_auto

hsa_sweep_time (dev, value)		// set sweep time
int	dev;
double	value;		// sweep time in seconds
{
	char	string[64];
//	sprintf (string, "ST %gS;", value);  removed, the 'S;" would not work on 8593A
	sprintf (string, "ST %g;", value);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;ST %g;\n", dev, value);
}		// end hsa_sweep_time

hsa_sweep_auto (dev)		// set sweep time
int	dev;
{
	char	string[64];
	sprintf (string, "ST AUTO;");
	gpib_write (dev, string);
}		// end hsa_sweep_time


double hsa_ask_sweep_time (dev)		// ask sweep time (sec)
	int	dev;
{
	double	time;		//  time in seconds
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "ST?;");
//	fprintf(gpib_out, "Output %d;ST?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	time = atof (string);
	return (time);
}		// end hsa_ask_sweep_time

hsa_rb (dev, value)		// set resolution bandwidth
int	dev;
double	value;		// bandwidth (in Hz) legal: 1 kHz to 3Mhz, step {1,3,10}
{
	char	string[64];
	sprintf (string, "RB %gHZ;", value);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;RB %g;\n", dev, value);
}		// end hsa_rb

double hsa_ask_rb (dev)		// ask resolution bandwidth (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "RB?;");
//	fprintf(gpib_out, "Output %d;RB?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_rb

hsa_vb (dev, value)		// set video bandwidth
int	dev;
double	value;		// bandwidth (in Hz) legal: 30 Hz to 3 Mhz, step {1,3,10}
{
	char	string[64];
	sprintf (string, "VB %gHZ;", value);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;VB %g;\n", dev, value);
}		// end hsa_vb

double hsa_ask_vb (dev)		// ask video bandwidth (Hz)
	int	dev;
{
	double	freq;		//  frequency in Hz
	char	string[HSA_NUM_LENGTH];
	int	len;

	gpib_write (dev, "VB?;");
//	fprintf(gpib_out, "Output %d;VB?;\n", dev);

	len = gpib_read (dev, string, HSA_NUM_LENGTH);
	string[len] = 0;		// terminate string properly

	freq = atof (string);
	return (freq);
}		// end hsa_ask_vb

hsa_hold (dev)
	int	dev;
{
	gpib_write (dev, "HD;");
//	fprintf(gpib_out, "Output %d;HD;\n", dev);
}		// end hsa_hold

hsa_title (dev, title)
	int	dev;
	char	title[];
{
	char	string[64];
	sprintf (string, "TITLE \"%s\";", title);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;TITLE \"%s\";\n", dev, title);
}		// end hsa_title

hsa_set_time_date(int dev)
//   Procedure to set time and date on spectrum analyzer.
//   Commands to set time and date are stored in strings
//   time_cmd and date_cmd, respectively.
{
	int	len;
	char	string[15];
	char time_cmd[25]="SETTIME ";
	char date_cmd[25]="SETDATE ";
	char time_buf[9], date_buf[9];
	char *time_hr, *time_min, *time_sec;
	char *date_yr, *date_mo, *date_day;

/* if the model number is HP8562(index = 0), 
		of if model is POH 8561B,003, 
	skip the date_time function, it is not
	suported on this model */
	if (hsa_number == 0 || hsa_number == 2) {
		return (0);
	}

//   Get time, put in HHMMSS form, and store in string
 	_strtime(time_buf);
	time_hr=strtok(time_buf,":");
	time_min=strtok(NULL,":");
	time_sec=strtok(NULL,"\0");
	strcat(time_hr, time_min);
	strcat(time_hr, time_sec);
	strcat(time_cmd, time_hr);
//	printf ("time_cmd = %s\n", time_cmd); getch();
	gpib_write (dev, time_cmd);	/* write out time */

//   Get date, put in YYMODA form, and store in string
	_strdate(date_buf);
	date_mo=strtok(date_buf,"/");
	date_day=strtok(NULL,"/");
	date_yr=strtok(NULL,"\0");
	strcat(date_yr, date_mo);
	strcat(date_yr, date_day);
	strcat(date_cmd, date_yr);
//	printf ("date_cmd = %s\n", date_cmd); getch();
	gpib_write (dev, date_cmd);	/* write out date */
}	/* end sa_set_time_date */


