#define	PROGRAM_NAME	"fdcch_ci.c"
#define	VERSION	1.01

// fdcch_ci.c
//
//   FDCCH C/I PERFORMANCE TEST PROCEDURE
//
//   Developed to quantify the C/I performance of the Forward DCCH
//   and the effects of various Access Parameters on that performance.
//
//		Usage:
//			fd n b
//
//		Where:
//			n = file number (see file naming convention below)
//			b = band to measure (see below)
//
//		Output File Naming convention
//			Files are named according to date, number (supplied above),
//			The file is stored in the DATA directory of the current
//			 working directory (DATA\) with name of:
//				mmddnn.log   <test results log file>
//			where:
//				mmdd is the captured month, date
//				nn is the file number specified as command line argument
//
//		NOTE: Will not overwrite an existing file
//
//		File Format (Output)
//			All fields are seperated by tabs and units of measure are
//				km/hr, dBm, and ratio values depending upon type of
//				measurement.
//			The format of each line is:
//			  Channel \t Doppler(km/hr) \t Correlation \t ARQ \t DIC \t 
//			  Max Retries\t Max Repetitions \t Max Busy/Reserved \t
//			  Max Counters \t C/I ratio \t BER \t WER \t RSS(dBm) \t
//         Audit Failures \t SPACH Failures \t RDATA Failures \n.
//		User is prompted for the following information:
//
//	Ver 0.1  Sourced from test.c from hna_cart.c
//

// =================================================================


#include <stdio.h>
#include <stdlib.h>
#include	<math.h>
#include	<string.h>
#include	<time.h>

#include "gpib_lib.c"		// basic library
#include "ifr_prog.c"		// HP network Analyzer routines


// Define control parameters
#define	TRUE	1
#define	FALSE	0

// User entered information and key variables
FILE	*data_fileg, *data_filed, *data_files;		// data file definition
#define OUTPUT_PATH "DATA\\"
#define LOG_EXT ".log"

#define IFR_DEV 2

// Define file formats
#define	STD_HEAD1	"#  Proprietray Information\n#  Property of SOUTHWESTERN BELL TECHNOLOGY RESOURCES INC\n"
#define	STD_HEAD2	"#  written by program %s Version %.2f\n", PROGRAM_NAME, VERSION



int process_command_line_arguments (int argc, char **argv, int *meas_number)
/* read and do any and all command line arguments from dos */
/*  Returns TRUE if all arguments were valid */
{
	/* Required are three arguments */
	if (argc != 2)
	{
		printf ("Arguments Required: number(1-99) \n");
		return (!TRUE);
	}

	/* Transfer measurement number back */
	*meas_number = atoi (argv[1]);
	}

	return (TRUE);
}	/* end process_command_lines */





build_base_file_name (int number, int band, char *output)
/* Builds the base file name of mmddnnbn */
{
	char date_buffer[10];
	char	temp_buffer[10];

	/* Build mmdd */
	_strdate(date_buffer);	/* get current date */
	strncpy (temp_buffer, date_buffer, (size_t) 2);	/* get month (mm) */
	strncpy (&temp_buffer[2], &date_buffer[3], (size_t) 2);	/* get date (dd) */
	temp_buffer[4] = '\0';	/* end of string */

	strcpy (output, OUTPUT_PATH);
	strcat (output, temp_buffer);

	/* Add the measurement number */
	sprintf (temp_buffer, "%2d", number);
	if (temp_buffer[0] == ' ')
		temp_buffer[0] = '0';  /* add leading zero if needed */
	strcat (output, temp_buffer);

}  /* build_data_file_name */

int open_all_files(int number, int band)
/* Open all output files checking to see if it exists first.  Returns
 *   TRUE if all files opened OK.
*/
{
	char	base_file_name[64];
	char	file_name[64];
	int	any_exist = !TRUE;

	build_base_file_name (number, band, base_file_name);

printf ("Base name %s\n", base_file_name);

/* 1. Open log file */
	strcpy (file_name, base_file_name);
	strcat (file_name, LOG_EXT);  /* summary file extension added */

	/* See if the file exists by opening as a read only */
	data_fileg = fopen (file_name, "r");

	if (data_fileg == NULL)
	{  /* File does not exist */
		/* create it, just leave it blank */
		data_fileg = fopen (file_name, "w");
		printf ("%s Opened.\n", file_name);
	}
	else
	{  /* file exists, need to pick out next Meas_ID */
		fclose (data_fileg);	/* close it */
		any_exist = TRUE;
		printf ("ERROR: File Exists: %s\n", file_name);
	}

		fclose (data_fileg);	/* close it */
		any_exist = TRUE;
	}

	return (!any_exist);
}



close_files(void)
{
	fclose (data_fileg);
}

write_output_files (int mb, double a[], double s[], double d[])
/* Write the output files */
{
}

main(int argc, char **argv)
{
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int ifr_ud ;		// GPIB handler for new 488 routines for IFR

	int	meas_number;
	int	in_key, i;


	printf ("%s Version %.2f\n", PROGRAM_NAME, VERSION);
	printf ("\nCopyright Southwestern Bell Technology Resources 1996\n\n\n\n\n");

/* process any command line arguments */
	if (!process_command_line_arguments (argc, argv, &meas_number))
	{
		printf ("Command Line Error, Program aborted\n");
		exit (1);
	}

	/* Open all files */
	if (!open_all_files(meas_number))
	{
		printf ("File(s) for this number already exist\n");
		exit (1);
	}


	ifr_ud = gpib_open(ifr_dev);		// get the ud for the IFR 

/* Initialize DCCH */
	initialize_dcch(hna_ud);

/* 1. Set up for the following:
 *			Top:  Gain (S21), capture and store in Gain array
 *			Bot:  SWR (S22), capture and store in Swr Array
 *       Then wait for key to be pressed
*/
	hna_dual_channel (hna_ud, TRUE);  /* set dual channel mode */
 /* Top Display */
	hna_select_channel (hna_ud, 1);  /* Channel 1 */
	hna_start_freq (hna_ud, Band[meas_band].start_freq);
	hna_stop_freq (hna_ud, Band[meas_band].stop_freq);
	gpib_write (hna_ud, "S21; LOGM;");  /* set S21 Gain */
	hna_trigger (hna_ud);		// trigger a single sweep
	hna_trigger (hna_ud);		// trigger a single sweep
	printf ("TOP: Gain (S21) Captured\n");
	get_analyzer_values (hna_ud, Ampl);

 /* Bottom Display */
	hna_select_channel (hna_ud, 2);  /* Channel 2 */
	hna_start_freq (hna_ud, Band[meas_band].start_freq);
	hna_stop_freq (hna_ud, Band[meas_band].stop_freq);
	gpib_write (hna_ud, "S22; SWR;");  /* set S22 SWR */
	hna_trigger (hna_ud);		// trigger a single sweep
	hna_trigger (hna_ud);		// trigger a single sweep
	printf ("BOTTOM: SWR (S22) Captured\n");
	get_analyzer_values (hna_ud, Swr);

	printf ("Press RETURN to Continue Capture ");
	getch();
	printf ("\n");

/* 2. Now do the following:
 *			Top:  Leave with previous Gain (S21)
 *			Bot:  Delay (S21), capture and store in Delay Array
 *       Then wait for key to be pressed
*/
 /* Bottom Display */
	hna_select_channel (hna_ud, 2);  /* Channel 2 */
	hna_start_freq (hna_ud, Band[meas_band].start_freq);
	hna_stop_freq (hna_ud, Band[meas_band].stop_freq);
	gpib_write (hna_ud, "S21; DELA;");  /* set S22 SWR */
	hna_trigger (hna_ud);		// trigger a single sweep
	hna_trigger (hna_ud);		// trigger a single sweep
	printf ("BOTTOM: Delay Spread (S21) Captured\n");
	get_analyzer_values (hna_ud, Delay);

/* 3. Write the output files */
	write_output_files (meas_band, Ampl, Swr, Delay);

	printf ("Data Files Written Number/Band %d/%d\n",
		meas_number, meas_band+1);

	hna_continuous_sweep (hna_ud);  // sets to continuous sweep

	hna_reset_analyzer (hna_ud);

	close_files();
  
}		// end main

