#define HSA_VERSION     1.1     /* current version number of this routine */
//
// HP Spectrum Analyzer GPIB Routines (HP 8593A)  NI488.2 VERSION!!!
//              Models supported:
//                      HP8562A         not completely tested, but no known problems
//                      HP8593A         not fully tested, but no known problems
//                      HP8561A         not completely tested, but no known problems
//                      HP70900B       not completely tested, but no known problems
//
//      NOTE: The application program MUST CALL "hsa_preset" to initialize the
//                   the proper globals!!!!!
//
//              7/8/96 VERSION 1.1.0  sah
//                      Changed HSA_BUF_LENGTH to 5,000 from 4,096 to handle P-Format
//                      Converted additional functions to ANSI C and added back in
//                      Commented out delay in hsa_sweep() to speed up instrumentation
//                      Cleaned up hsa_get_traceA() to take out superfluous code
//                      Added hsa_vid_trig() to set video trigger mode and level
//
//              9/02/92 VERSION 1.0.0  sah
//                      Added analyzer type HP70900B (untested with others)
//                      Converted to IEEE 488.2 gpib calls for simplicity
//                      Added hsa_ask, hsa_ask_mkf, and hsa_ask_mka
//                      Converted some routines to ANSI C format
//                      Deleted some routines for now
//                      Changed format to %f from %g in hsa_cf due to incompatibility with HP70900B
//                      Deleted "DONE?;" request in hsa_sweep due to timeout problem during long sweeps
//                      Added DevClear to hsa_preset to clear gpib for new setup 
//
//              4/15/92 VERSION 0.2.1  csp
//                      Added analyzer type HP8561,003 (cause that's what it said). csp
//                      untested with other analyzers.
//
//      POTENTIAL BUG UNCOVERED: (with 601 point models at least), dimension the
//              integer trace buffer (passed to hsa_M_FORMAT_to_int) as at least one
//              element larger that what is expected (my example used a point of 605
//              points instead of 601 pts).  This routine appears to do one too many
//              points overwriting whatever the following variable is.
//
//              4/09/92 VERSION 0.2  csp
//                      0.2.1  Made more universal for HP spectrum analyzers by using the
//                              hsa_preset to initialize the number of points per trace into the
//                              global variable "???" based on the spectrum analyzer model.
//                              Models supported listed above.
//
//                      0.1.4 Within the hsa_preset function, the type of analyzer is determined
//                              using the ID? command.  Based on the response, the following global 
//                              variables are set:
//                                      hsa_points_per_trace = number of points returned in a trace
//                                      hsa_M_top = returned units for top of screen (for M_format)
//                                      hsa_M_divisions = number of y divisions on screen
//
//        ADDING A SPECTRUM ANALYZER IS DONE THROUGH THE STRUCTURE "MODELS"
//                                      and incrementing the variable HSA_NUMBER_MODELS.
//
//        4/09/92  VERSION 0.1  
//                      0.1.1 Revised trigger routine to use DONE? function for
//                      proper synchronization since the ONEOS is not supported by all our
//                      spectrum analyzers.
//
//                      0.1.2 Added new functions: hsa_set_time_date to set the analyzer's
//                              clock and calendar (if unit is HP8562A, then skips as it has no
//                              time and date transfer).
//
//                      0.1.3 Added "Hz" to all frequency set commands to make it work with
//                              HP8562 spectrum analyzer properly.
//
//  (OBSOLETE) 
//  4/17/91 Added some logic to help the trigger situation properly using the
//               ONEOS command.  Also added global variable hsa_mode_global;  Affects
//               procedures hsa_sweep_mode and hsa_get_traceA.  The routine, "hsa_sweep"
//               has been removed since triggering is now part of the _get_traceA.
//
//  Routines that have not yet been prototyped:
//  (those marked with * have not yet been converted to v1.1 or higher)
//              hsa_preset (dev)                set to preset value and INITIALIZE the system (IP)
//              hsa_sweep_mode (dev, mode)              set sweep mode
//              hsa_sweep                         trigger a sweep (and wait till sweep complete)
//*              hsa_sf (dev, freq)              set starting frequency
//*              hsa_ef (dev, freq)              set ending frequency
//              hsa_span (dev, span)        set span
//*              hsa_M_FORMAT_to_int (in, ref, out)  convert from buffer to int dBm array
//*              hsa_max_holdA (dev)             set A trace to max hold mode
//*              hsa_clear_writeA (dev)           clear A trace, resets from max_hold
//*              hsa_units (dev, measure)        set units of measure
//*              hsa_ask_units (dev)    set units of measure
//*              hsa_atten_auto (dev)    set attenuation to auto
//              hsa_sweep_time (dev, time)              set sweep time
//              hsa_rb (dev, value)            set resolution bandwidth
//              hsa_vb (dev, value)           set video bandwidth
//*              hsa_hold (dev)                  do HOLD operation (clear screen)
//*              hsa_title (dev, string)       write title
//*              hsa_set_time_date(dev)  set time/date on analyzer to PC time/date (if possible)


/* Function Prototype Declarations  */
double hsa_ask_cf(int hsa_addr);                /* get center frequency */
double hsa_ask_sf(int hsa_addr);                /* get starting frequency */
double hsa_ask_ef(int hsa_addr);                /* get ending frequency */
double hsa_ask_span(int hsa_addr);              /* get frequency span */
double hsa_ask_ref_level(int hsa_addr);         /* get reference level (dBm) */
double hsa_ask_sweep_time(int hsa_addr);        /* get sweep time */
double hsa_ask_rb(int hsa_addr);                /* get resolution bandwidth */
double hsa_ask_vb(int hsa_addr);                /* get video bandwidth */
double hsa_ask_mka(int hsa_addr);               /* get amplitude of trace marker */
double hsa_ask_mkf(int hsa_addr);               /* get frequency of trace marker */
int    hsa_ask_atten (int hsa_addr);            /* get attenuation level (dB) */

void hsa_mkpk(int hsa_addr);                            /* marker finds peak signal on trace */
void hsa_cf (int hsa_addr, double freq);                /*set center frequency */
void hsa_trace_format (int hsa_addr, int format);       /* set trace format */
void hsa_get_traceA (int hsa_addr, char buffer[]);      /* get entire buffer for trace A */
void hsa_ref_level (int hsa_addr, double level);        /* set reference level (dBm) */
void hsa_atten (int hsa_addr, int value);               /* set attenuation (dB) */
void hsa_vid_trig (int dev, double value);              /* set video trig level (dBm)  */           


/* Define a Structure for the Equipment Models Supported */
#define HSA_NUMBER_MODELS 4   /* number of models in list */

struct  MODELS {
	char  model_name[15]; /* model number (in response to ID?) */
	int     points; /* number points per trace */
	int     top;    /* top of screen in screen units (for M_format) */
	int     divisions_per_screen;   /* number y division per screen */
	}

	models[] = {{"HP8562A", 601, 600, 10},
			{"HP8593A", 401, 8000, 8},
			{"HP8561B,003", 601, 600, 10},
			{"HP70900B",801,1000,10}
		      }; /* add additional models here */



/*  Constant Definitions  */
#define HSA_NUM_LENGTH  24      // max number of char returned (numeric data)
#define HSA_BUF_LENGTH  5000    // max chars in get trace (works with 601 traces)
#define HSA_EOS         10      // end of string termination (LF) v1.0 sah

#define DBM     1               // units of measure cases for AUNITS
#define DBMV    2
#define DBUV    3
#define V       4
#define W       5               // end units of measure cases for AUNITS

#define SWEEP_CONTINUOUS 6      // used to set sweep continuous
#define SWEEP_SINGLE    7       // set sweep in single mode

#define P_FORMAT        8       // trace format,  parameter units
#define A_FORMAT        9       // trace format,  a-block format
#define M_FORMAT 10             // trace format,  measurement units
#define I_FORMAT 11             // trace format,  i-block units
#define B_FORMAT 12             // trace format,  binary units

/* NOTE:  Source file must define GPIB_BOARD and HSA_ADDR  */
/*               prior to calling HSA library functions.  For example,        */
/*               include the following lines in your code:                            */
/*                                                                                                           */
/*      .  #define GPIB_BOARD  0   (id number for the GPIB board)   */
/*         #define HSA_ADDR      7   (GPIB address for the Sp. An. )   */




/*  External Libraries to be Included  */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include "sah_lib.c"
/* NOTE:  Source file must include gpib library  c:\at-gpib\decl.h */


// ============================================================

// ---------- Global Variables for hsa_prog routines ----------------
int     hsa_mode_global = SWEEP_CONTINUOUS;     // current mode of sweeping
int     hsa_number = -1;        /* index of model number (0 = first in table, etc.) */
int     hsa_points_per_trace = 401;     /* default here */
int     hsa_M_top;      /* holds the reference for top of screen (used in M format decode) */
int     hsa_M_divisions;        /* holds divisions per y axis for this analyzer */

// ============================================================


/**********************************************************************/
/***********Begin HP Spectrum Analyzer Function Definitions*******************/
/**********************************************************************/
/**********************************************************************/
void hsa_preset (int dev)  // set analyzer and initialize to preset state
{
	int     i,exists;
	char    buffer[16];
	int     addr_list[2] = {dev, NOADDR};

	DevClear(GPIB_BOARD, dev);
	EnableRemote(GPIB_BOARD, addr_list);
	Send(GPIB_BOARD, dev, "IP;",3L,NLend);

/* See if the unit is a HP8562A, if so skip the set time date function,
   it is not supported in this unit */

	Send(GPIB_BOARD,dev,"ID?;",4L,NLend);
	// delay(2.0);        
	Receive(GPIB_BOARD,dev,buffer,24,HSA_EOS);
	buffer[ibcnt-1] = 0;   // terminate string properly

	exists=0;
	for (i=0; i< HSA_NUMBER_MODELS; i++) {
		if (strstr (buffer, models[i].model_name) != NULL)
		{  /* Model Matches */
		  exists=1;
		  printf ("Analyzer Type: \t%s\n", models[i].model_name);
		  hsa_number = i; /* save model number index */
		  hsa_points_per_trace = models[i].points;
		  hsa_M_top = models[i].top;       /* holds the ref for top of screen (used in M format) */
		  hsa_M_divisions = models[i].divisions_per_screen ;    /* holds y axis tics */
		}       /* endif */
	}       /* end for */

	if (exists==0)
	{
	  printf ("The spectrum analyzer, model <%s>, is invalid, program aborted\n", buffer);
	  exit (3);
	} /* end if */

}               // end hsa_preset
/**********************************************************************/



/**********************************************************************/
void hsa_sweep_mode (int dev, int mode)              // set sweep mode where
{
	switch (mode) {
		case SWEEP_CONTINUOUS :
			Send(GPIB_BOARD,dev,"CONTS;",6L,NLend);
			hsa_mode_global = SWEEP_CONTINUOUS;
			break;
		case SWEEP_SINGLE :
			Send(GPIB_BOARD,dev,"SNGLS;",6L,NLend);
			hsa_mode_global = SWEEP_SINGLE;
			break;
		default:
			printf ("Incorrect sweep mode, entry = %d\n", mode);
			break;
	} // end case
}  // end hsa_sweep_mode
/**********************************************************************/



/**********************************************************************/
void hsa_sweep (int dev) // trigger a sweep and wait till complete (revision 0.01)
{
	char    string[4];

	Send(GPIB_BOARD,dev,"TS;",3L,NLend);

/*      delay(hsa_ask_sweep_time(HSA_ADDR)); this adds delay, not sure if needed*/
/*                                                        think I had trouble w HP7090A    */
/*                                                                                Commented out in rev 1.1.0        */

      Send(GPIB_BOARD,dev,"DONE?;",6L,NLend);  // this was in early rev but think
      Receive(GPIB_BOARD,dev,string,4,HSA_EOS); // I had trouble on long sweeps
//                                                                                           with HP7090A
	
} // end hsa_sweep
/**********************************************************************/



/**********************************************************************/
double hsa_ask_sf (int dev)         // ask start frequency (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"FA?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	freq = atof (string);
	return (freq);
} // end hsa_ask_sf
/**********************************************************************/



/**********************************************************************/
double hsa_ask_ef (int dev)         // ask stop (end) frequency (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"FB?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	freq = atof (string);
	return (freq);
} // end hsa_ask_ef
/**********************************************************************/



/**********************************************************************/
void hsa_cf (int dev, double freq)              // set center frequency
{
	char    string[64];

	sprintf (string, "CF %fHZ;", freq);
	Send(GPIB_BOARD,dev,string,(long) strlen(string),NLend);
} // end hsa_cf
/**********************************************************************/



/**********************************************************************/
double hsa_ask_cf (int dev)         // ask center frequency (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"CF?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	freq = atof (string);
	return (freq);
} // end hsa_ask_cf
/**********************************************************************/



/**********************************************************************/
void hsa_span (int dev, double span)            // set span 
{
	char    string[64];
	sprintf (string, "SP %gHZ;", span);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

} // end hsa_span
/**********************************************************************/



/**********************************************************************/
double hsa_ask_span (int dev)               // ask span (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"SP?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly
	
	freq = atof (string);
	return (freq);
} // end hsa_ask_span
/**********************************************************************/



/**********************************************************************/
void hsa_ref_level (int dev, double level)         // set reference level (dBm)
{
	char    string[64];
	
	sprintf (string, "RL %gDB;", level);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

} // end hsa_ref_level
/**********************************************************************/



/**********************************************************************/
double hsa_ask_ref_level (int dev)         // ask reference level (aunits)
{
	double  value;           //  level in amplitude units
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"RL?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	value = atof (string);
	return (value);
} // end hsa_ask_ref_level
/**********************************************************************/



/**********************************************************************/
void hsa_atten (int dev, int value)         // set attenuator (dB)
{
	char    string[64];
	
	sprintf (string, "AT %dDB;", value);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

} // end hsa_atten
/**********************************************************************/



/**********************************************************************/
int hsa_ask_atten (int dev)         // get attenuator setting (integer dB)
{
	int  value;           //  attenuator setting in dB (integer)
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"AT?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	value = atoi (string);
	return (value);
} // end hsa_ask_atten
/**********************************************************************/



/**********************************************************************/
void hsa_sweep_time (int dev, double value)             // set sweep time
{
	char    string[64];
	
	sprintf (string, "ST %gS;", value);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

} // end hsa_ sweep_time 
/**********************************************************************/



/**********************************************************************/
double hsa_ask_sweep_time (int dev)         // ask sweep time (sec)
{
	double  time;           //  time in seconds
	char    string[HSA_NUM_LENGTH];

	Send(GPIB_BOARD,dev,"ST?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly
	
	time = atof (string);
	return (time);
}               // end hsa_ask_sweep_time
/**********************************************************************/



/**********************************************************************/
void hsa_rb (int dev, double value)             // set resolution bandwidth
// bandwidth (in Hz) legal: 1 kHz to 3Mhz, step {1,3,10}
{
	char    string[64];
	
	sprintf (string, "RB %gHZ;", value);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

}               // end hsa_rb
/**********************************************************************/



/**********************************************************************/
double hsa_ask_rb (int dev)         // ask resolution bandwidth (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];
	
	Send(GPIB_BOARD,dev,"RB?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly

	freq = atof (string);
	return (freq);
}               // end hsa_ask_rb
/**********************************************************************/



/**********************************************************************/
hsa_vb (int dev, double value)             // set video bandwidth
// bandwidth (in Hz) legal: 30 Hz to 3 Mhz, step {1,3,10}
{
	char    string[64];
	
	sprintf (string, "VB %gHZ;", value);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

}               // end hsa_vb
/**********************************************************************/



/**********************************************************************/
double hsa_ask_vb (int dev)         // ask video bandwidth (Hz)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];
	
	Send(GPIB_BOARD,dev,"VB?;",4L,NLend);

	Receive(GPIB_BOARD,dev,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly
	
	freq = atof (string);
	return (freq);
}               // end hsa_ask_vb
/**********************************************************************/



/**********************************************************************/
double hsa_ask_mka(int hsa_addr)
{
	double  ampl;           //  amplitude in dB
	char    string[HSA_NUM_LENGTH];
	
	Send(GPIB_BOARD,hsa_addr,"MKA?;",5L,NLend);

	Receive(GPIB_BOARD,hsa_addr,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly
	
	ampl = atof (string);
	return (ampl);
}               // end hsa_ask_mka
/**********************************************************************/



/**********************************************************************/
double hsa_ask_mkf(int hsa_addr)
{
	double  freq;           //  frequency in Hz
	char    string[HSA_NUM_LENGTH];
	
	Send(GPIB_BOARD,hsa_addr,"MKF?;",5L,NLend);

	Receive(GPIB_BOARD,hsa_addr,string,HSA_NUM_LENGTH,HSA_EOS);
	string[ibcnt-1] = 0;                // terminate string properly
	
	freq = atof (string);
	return (freq);
}               // end hsa_ask_mkf
/**********************************************************************/



/**********************************************************************/
void hsa_mkpk(int hsa_addr)
{
  Send(GPIB_BOARD,hsa_addr,"MKPK HI;",8L,NLend);
} // end hsa_mkpk
/**********************************************************************/



/**********************************************************************/
void hsa_trace_format (int dev, int format)     // set trace data format
// valid formats: P_FORMAT, A_FORMAT, M_FORMAT, I_FORMAT, B_FORMAT
{
	switch (format) {
		case P_FORMAT:  // real number output format
			Send(GPIB_BOARD,dev,"TDF P;",6L,NLend);
			break;
		case A_FORMAT:  // A-block format
			Send(GPIB_BOARD,dev,"TDF A;",6L,NLend);
			break;
		case M_FORMAT:  // ASCII data format (machine units)
			Send(GPIB_BOARD,dev,"TDF M;",6L,NLend);
			break;
		case I_FORMAT:  // I-block format
			Send(GPIB_BOARD,dev,"TDF I;",6L,NLend);
			break;
		case B_FORMAT:  // Binary data format
			Send(GPIB_BOARD,dev,"TDF B;",6L,NLend);
			break;
		}  // end switch
}  // end hsa_trace_format
/**********************************************************************/



/**********************************************************************/
void hsa_get_traceA (int dev, char buffer[])    // get entire buffer for trace A
//=====================================================
//  NOTE 1:     This routine is not extensively tested under all trace formats, but I
//                      believe it accepts only text, not a binary based format.  "C" will
//                      probably reject a binary fromat (nulls, CR, LF imbeded).
//  NOTE 2:     Version 0.01 (4/09/92) revised to use hsa_sweep (which waits till
//                      sweep is complete) to trigger trace.
//=====================================================
{
	hsa_sweep (dev);        /* trigger a trace and return when complete */

	Send(GPIB_BOARD,dev,"TRA?;",5L,NLend);  /* ask for trace */

	Receive(GPIB_BOARD,dev,buffer,HSA_BUF_LENGTH,HSA_EOS);
	buffer[ibcnt-1] = 0;    // terminate string properly

}  // end hsa_get_traceA
/**********************************************************************/



/**********************************************************************/
void hsa_vid_trig (int dev, double value)   // set video trigger level (dBm)
{
	char    string[64];
	
	sprintf (string, "TM VID;VTL %gDBM;", value);
	Send(GPIB_BOARD,dev,string,strlen(string),NLend);

}  // end hsa_vid_trig
/**********************************************************************/



/**********************************************************************/
/************************End of HSA.LIB********************************/
/**********************************************************************/

