/********************************************************************
	$Workfile:   scl.h  $

	Copyright Northern Telecom Limited. 1989
	All Rights Reserved under the Copyright Laws.

	Header file for the SCL.  This file is broken down
	into the following main areas

	- General constants and types
	- Constants for encoding/decoding functional messages
	- Macros for encoding/decoding functional messages
	- Function prototypes and other definitions for determining
	  information about the PCI card
	- Function prototypes and other definitions for buffering.
	- General SCL function prototypes
	- Function prototypes and other definitions for timers.
	- Constants for error codes.


	$Revision:   1.39  $

	$Author:   CHARLES  $

	$Date:   29 Sep 1989 15:05:30  $


**********************************************************************/


/*
**	=====================================================================
**
**		General constants and types 
**
**	=====================================================================
*/


#ifndef FALSE
#define FALSE		0
#endif

#ifndef TRUE
#define TRUE		~FALSE
#endif

#ifndef NULL
/* define NULL data pointer for different memory models */

	#if (defined(M_I86SM) || defined(M_I86MM))
		#define  NULL    0
	#elif (defined(M_I86CM) || defined(M_I86LM) || defined(M_I86HM))
		#define  NULL    0L
	#endif
#endif

#ifndef NULLF
	/* define NULLF function pointer for different memory models */

	#if (defined(M_I86SM) || defined(M_I86CM))
		#define  NULLF    0
	#elif (defined(M_I86MM) || defined(M_I86LM) || defined(M_I86HM))
		#define  NULLF    0L
	#endif
#endif

/*
**	type definitions
*/

typedef unsigned char byte;
typedef unsigned int word;


/*
**	=====================================================================
**
**		Constants for encoding/decoding messages
**
**	The following prefixes apply
**
**		cp ==> CP Message related
**		ie ==> Info Element related
**		cs ==> Call States
**		dv ==> Device Types
**		im ==> INFO Message related
**		ad ==> ADMIN Message related
**		au ==> AUDIT Message related
**		ft ==> FEAT Message related
**		fta => FEAT User Action related
**		ftx => FEAT xFunction related
**		fi ==> Feature Indices
**		sv ==> Fixed LAD's for Servers
**
**	=====================================================================
*/

/*
**	message types
*/

#define cpMSG		0x71
#define imMSG		0x72
#define mtMSG		0x73
#define adMSG		0x74
#define auMSG		0x75
#define ftMSG		0x76

/*
**	Info element bit values
*/

#define ieRABit		0x08		/* redirect address */
#define ieNMBit		0x04		/* network manager */
#define ieBIBit		0x01		/* broadcast information */

/*
**	Call States
*/

#define csIdle			0x00
#define csEstablished		0x01	
#define csIncoming		0x02
#define csOutgoing		0x03
#define csHeld			0x04
#define csHeldExclusive	0x27
#define csHold			0x05
#define csIntDialTone		0x06
#define csCollectDN		0x07
#define csWaitOriginateResp	0x08
#define csPager			0x0b
#define csHoldOpen		0x0c
#define csHoldAnother		0x0d
#define csOverflow		0x10
#define csBusyTone		0x11
#define csMutualHold		0x12
#define csMastEstab		0x13
#define csMastHold		0x14
#define csMastHeld		0x15
#define csMastMutualHold	0x16
#define csSlaveEstab		0x17
#define csSlaveHold		0x18
#define csSlaveHeld		0x19
#define csSlaveMutualHold	0x1a


/*
**	Device Types
*/

#define dvATA			0x00
#define dvM7208			0x02
#define dvM7310			0x03
#define dvM7324			0x04
#define dvM7900_S		0x05
#define dvTrunkCO		0x40
#define dvRAD			0x80
#define dvPCID			0x81
#define dvDCA			0x82
#define dvM7900_F		0x83
#define dvPCIB			0x84
#define dvMVOX1			0x85
#define dvMVOX2			0x86
#define dvMVOX3			0x87


/*
**	CP message commands
*/

#define cpOriginate	0x00
#define cpAcceptCall	0x01
#define cpAlerting	0x02
#define cpReject		0x03
#define cpReroute	0x04
#define cpConference	0x05
#define cpTransfer	0x06
#define cpHold		0x07
#define cpUnhold		0x08
#define cpRelease	0x09
#define cpSelect		0x0a
#define cpConnect	0x0b
#define cpDisconnect	0x0c
#define cpXselect	0x0d
#define cpRequestTone	0x0f
#define cpExtend		0x10
#define cpSeize		0x11
#define cpSolicit	0x12


/*
**	CP message subcommands and parms
*/

#define cpSignal			0x00	/* cpOriginate command */
#define cpVoice			0x01	
#define cpSignal1		0x02
#define cpPriority		0x03
#define cpData			0x04
#define cpRerouted		0x07
#define cpParkedCallback  	0x0d

#define cpVisualAlrt		0x00	/* cpAlerting command */
#define cpAudioAlrt		0x01
#define cpDNDAlrt		0x03
#define cpOtherCallAlrt	0x04
#define cpPriorityAlrt		0x0a

#define cpAccCO			0x00  /* cpAcceptCall, cpSelect */
#define cpAccSetNoData		0x02  /* Terminal Capabilities */
#define cpAccSetWithData	0x03
#define cpAccNoData		0xff	/* B channel Protocols */
#define cpAccTLINK		0x01
#define cpAccBCM			0x02

#define cpRerouteFwd		0x06	/* cpReroute command */
#define cpRerouteXfer		0x09
#define cpRerouteCallback	0x08
#define cpRerouteExtR7		0x07
#define cpRerouteExtR21	0x15
#define cpRerouteExtR22	0x16
#define cpRerouteExtR23	0x17
#define cpRerouteExtR24	0x18
#define cpRerouteExtR25	0x19
#define cpRerouteExtR26	0x1a
#define cpRerouteExtR27	0x1b

#define cpRejBusy		0x00	/* cpReject command */
#define cpRejNoPriority	0x01
#define cpRejFwd			0x06
#define cpRejConn		0x07

#define cpHoldNorm		0x00	/* cpHold command */
#define cpHoldEx			0x01
#define cpHoldAutoEx		0x03
#define cpHoldAutoConf		0x04
#define cpHoldConf		0x05
#define cpHoldAuto		0x06

#define cpUnholdNorm		0x00	/* cpUnhold command */
#define cpUnholdConf		0x05

#define cpRlsNorm		0x00	/* cpRelease command */
#define cpRlsReselect		0x01
#define cpRlsRestrict		0x02
#define cpRlsRerouted		0x04

#define cpConfSplit		0x00	/* cpConference command */
#define cpConfJoin		0x01
#define cpConfHold		0x07
#define cpConfUnhold		0x08

#define cpConnSingle		0x00	/* cpConnect command */
#define cpConnJoin		0x01
#define cpConnPage		0x02

#define cpDiscNormal		0x00	/* cpDisconnect command */

#define cpXselResend		0x80	/* cpXselect command */
#define cpXselStateMask	0x7f

#define cpRToneDial		0x00	/* cpRequestTone command */
#define cpRToneBusy		0x01
#define cpRToneReorder		0x02
#define cpRToneHold		0x03
#define cpRToneMusic		0x04
#define cpRToneRing		0x05
#define cpToneDig0		0x06
#define cpToneDig1		0x07
#define cpToneDig2		0x08
#define cpToneDig3		0x09
#define cpToneDig4		0x0a
#define cpToneDig5		0x0b
#define cpToneDig6		0x0c
#define cpToneDig7		0x0d
#define cpToneDig8		0x0e
#define cpToneDig9		0x0f
#define cpToneDigStar		0x10	/* Star (*) digit */
#define cpToneDigOct		0x11	/* Octothorpe (#) digit */

#define cpExtFwd			0x06	/* cpExtend command */
#define cpExtFwdReroute  	0x07
#define cpExtCallNoPlace	0x08
#define cpExtCallDestBusy	0x09
#define cpExtCallDestDND	0x0a
#define cpExtCallDestNoAns	0x0b
#define cpExtCallDestDead	0x0c
#define cpExtCallParkNoRetr	0x0d
#define cpExtTransR1		0x0e
#define cpExtTransR2		0x0f
#define cpExtTransR3		0x10
#define cpExtTransR4		0x11
#define cpExtTransR5		0x12
#define cpExtTransR6		0x13
#define cpExtTransR7		0x14
#define cpExtFwdR1		0x15
#define cpExtFwdR2		0x16
#define cpExtFwdR3		0x17
#define cpExtFwdR4		0x18
#define cpExtFwdR5		0x19
#define cpExtFwdR6		0x1a
#define cpExtFwdR7		0x1b


/*
**	INFO message commands
*/

#define imDigit			0x01
#define imRingInfo		0x04
#define imAssign			0x06
#define imMinUpdate		0x07
#define imRequest		0x09
#define imResponse		0x0a
#define imFeature		0x0b
#define imRoute			0x0e
#define imUserInfo		0x11
#define imStatus			0x0d
#define imReserved		0x0f
#define imAccessDenied		0x12


/*
**	INFO message subcommands and parms
*/

#define imNormalTone		0x00	/* imDigit command */
#define imLongTone		0x20
#define imShortTone		0x40
#define imDiscTone		0x60
#define imDigitStar		0x0a	/* Star (*) digit */
#define imDigitOct		0x0b	/* Octothorpe (#) digit */
#define imDigitCOSup		0x0c	/* CO Supervision */
#define imDigitFlash		0x01	/* CO Sup subcommands */
#define imDigitPause		0x08
#define imDigitTimedRls 	0x02
#define imDigitTypeMask	0xe0	/* imDigit decoding masks */
#define imDigitLenMask		0x1f

#define imRingOn			0x01	/* imRingInfo command */
#define imRingOff		0x02

#define imStatQuery		0x00	/* imStatus command */
#define imStatResp		0x0a
#define imStatQVoicePath	0x00	/* Query byte[1] */
#define imStatQCallState	0x01
#define imStatQCallLAD		0x02
#define imStatQAppLAD		0x03
#define imStatErrIncomplMsg	0xfc	/* Response byte[1] errors */
#define imStatErrQCO		0xfd	
#define imStatErrQInvalid	0xfe
#define imStatVPIdle		0x00	/* Voice Path Response */
#define imStatVPHF		0x10
#define imStatVPHandset	0x20
#define imStatVPHeadset	0x30
#define imStatVPHeadsetIn	0x08
#define imStatVPHFOff		0x00
#define imStatVPHFMute		0x02
#define imStatVPHFOn		0x06
#define imStatVPOffHook	0x01
#define imStatVPStateMask	0x30	/* Voice path response masks */
#define imStatHFStateMask	0x06

#define imMinTimeDate		0x00	/* imMinUpdate command */

#define imReqTimeDate		0x00	/* imRequest command */
#define imReqName		0x02
#define imReqSpeedNum		0x04
#define imReqCheckNum		0x05
#define imReqAdd			0x06
#define imReqRemove		0x07
#define imReqAddress		0x08
#define imReqPermission	0x09
#define imReqDNToLAD		0x0a
#define imReqLADToDN		0x0b

#define imResTimeDate		0x00	/* imResponse command */
#define imResEmpty		0x01
#define imResName		0x02
#define imResSpeedNum		0x04
#define imResDenied		0x05
#define imResAddress		0x08
#define imResPermission	0x09
#define imResDNToLAD		0x0a
#define imResLADToDN		0x0b
#define imResDenBadDNLen	0x00	/* imResDenied reasons */
#define imResDenInvalidDN	0x01
#define imResDenInvalidLAD	0x02
#define imResDenInvalidTandD	0x00
#define imResDenNoResource	0x02
#define imResDenInvalidID	0x03
#define imResDenBIRequest	0x04

#define imRouteCancel		0x00	/* imRoute command */
#define imRouteAccept		0x01
#define imRouteReject		0x03
#define imRouteRequest		0x09

#define imFeatInvoke		0x00	/* imFeature command */
#define imFeatAccept		0x01
#define imFeatReject		0x03
#define imFeatRelease		0x09

#define imAccDenInvalidPswd	0x01	/* imAccessDenied command */
#define imAccDenRestrictNum	0x02


/*
**	Admin message commands
*/

#define adQuery                 0x0b  /* admin data access commands */
#define adResponse              0x0c
#define adVersionNum            0x10

/*
** Admin message subcommands and constants
*/

#define adPLADQuery             0x00  /* adQuery, adResponse subcommands */
#define adSessionQuery          0x01
#define adTemplateQuery         0x02
#define adHWidQuery             0x03

#define adKSUswVersion          0x06
#define adStnxToTN              0x07
#define adTNtoStnx              0x08
#define adStnxToPDN             0x09
#define adPDNtoStnx             0x0a
#define adSecondToPrim          0x0b
#define adStnxToTermType        0x0e
#define adPDNvalidation         0x12
#define adDeviceType            0x13

#define adStnxTypeEmpty         0x00  /* admin station index type */
#define adStnxTypeB1Reloc       0x01
#define adStnxTypeB2Reloc       0x02
#define adStnxTypeNonReloc      0x03


/*
**	Audit message commands
*/

#define auSessionState		0x00
#define auActiveSession	0x01
#define auFault			0x0e

/*
**	Audit message subcommands and constants
*/

#define auBadAuditMsg		0x06	/* auFault command */
#define auNoAuditMsg		0x07
#define auRemoteAuditFault	0x08
#define auOnHookMsg		0x09


/*
**	FEAT message commands and types
*/

#define ftOriginate		0x00		/* commands */
#define ftAccept			0x01
#define ftReject			0x03
#define ftRelease		0x09
#define ftRequest		0x0c
#define ftResponse		0x0d
#define ftTermCapReq		0x0e
#define ftTermCapResp		0x0f
#define ftUserActionResp	0x12
#define ftUserActionAck	0x13
#define ftXCSResponse		0x14
#define ftXCSRequest		0x15
#define ftXCSError		0x16
#define ftC4InfoReq		0x17
#define ftC4InfoResp		0x18

#define ftOrigClassMask	0xf0	/* ftOriginate and ftAccept masks */
#define ftOrigPriorityMask	0x0f

#define ftRejIntFeatInvoke	0x01	/* ftReject reasons */
#define ftRejIntFeatActive	0x02
#define ftRejHiPriActive	0x03
#define ftRejWaitingForResp	0x04
#define ftRejNoClassSupport	0x05
#define ftRejTermNoSession	0x07
#define ftRejCmndNotSupport	0x08
#define ftRejIncompleteMsg	0x09

#define ftRlsTimeOrig		0x00	/* ftRelease reasons */
#define ftRlsTimeRelease	0x03
#define ftRlsTimeFeature	0x04
#define ftRlsHiPriFeat		0x05
#define ftRlsUserAction	0x06
#define ftRlsReceivedRls	0x07

						/* Term Cap constants */
#define ftTermCapEnglish	0x00	/* terminal languages */
#define ftTermCapFrench	0x01
#define ftTermCapSpanish	0x02
#define ftTermCapClass0	0x00	/* Terminal Classes */
#define ftTermCapClass1	0x10
#define ftTermCapClass2	0x20
#define ftTermCapClass3	0x30
#define ftTermCapClass4  	0x40
#define ftTermCapATA		0x00	/* Class 1 term types */
#define ftTermCapM7208		0x00	/* Class 2 term types */
#define ftTermCapM7310		0x00	/* Class 3 term types */
#define ftTermCapM7324		0x08
#define ftTermCapM7900		0x00	/* Class 4 term types */
#define ftTermCapClassMask	0xf0	/* term cap decoding masks */
#define ftTermCapTypeMask	0x0f
#define ftTermCapLangMask	0x1f

#define ftaRespRequired	0x80
#define ftaRespTypeMask	0x7f
#define ftaRespPKIDepr		0x00	/* ftUserActionResp commands */
#define ftaRespPKDepr		0x01
#define ftaRespKeyPad		0x02
#define ftaRespSoftKey		0x03
#define ftaRespHardKey		0x04
#define ftaRespError		0x05

#define ftPKIIntLine		0x00	/* Programmable keys with indicators */
#define ftPKIExtLine		0x01
#define ftPKIAnswer		0x02
#define ftPKFeature		0x10	/* Programmable keys */
#define ftPKExtFeat		0x11
#define ftPKDSS			0x12
#define ftPKAutoDial		0x13
#define ftPKBlank		0x14
#define ftPKFunction		0x15
#define ftHKHold			0x00	/* Hard Keys */
#define ftHKFeature		0x01
#define ftHKRelease		0x02
#define ftHKShift		0x03
#define ftHKHeadsetIn		0x04
#define ftHKHeadsetOut		0x05
#define ftHKOnHook		0x06
#define ftHKOffHook		0x07
#define ftHKHandsfree		0x08

#define ftaErrTransient	0x00	/* USER ACTION Errors */
#define ftaErrPermanent	0x80
#define ftaErrNoFreeInPool	0x00
#define ftaErrNoFreeKey	0x01
#define ftaErrAccDenToPool	0x02
#define ftaErrDestIsSrcDN	0x03
#define ftaErrDestDNDead	0x04
#define ftaErrAlreadyJoin	0x05
#define ftaErrInvalidDN	0x06
#define ftaErrLineNotSelect	0x07

#define ftaAckNoSession	0x00	/* USER_ACTION_ACK Error codes */
#define ftaAckBadUAType	0x01
#define ftaAckBadUserKey	0x02
#define ftaAckWrongNumParms	0x03

#define ftXCSAssignLAD		0x00	/* ftXCSRequest, ftXCSResponse */
#define ftXCSDeleteLAD		0x01	/* and ftXCSError subcommands */
#define ftXCSQueryLAD		0x02
#define ftXCSQueryFC		0x03
#define ftXCSQueryAppID	0x04

#define ftXCSErrNoSession	0x00	/* ftXCSError codes */
#define ftXCSErrAppIDInUse	0x01
#define ftXCSErrInvalidLAD	0x02
#define ftXCSErrTableFull	0x03
#define ftXCSErrInvalidParm	0x04
#define ftXCSErrLADInUse	0x05
#define ftXCSErrOutOfBounds	0x06
#define ftXCSErrInSession	0x07
#define ftXCSErrInvalidAppID	0x08

#define ftC4ReqB2LAD		0x01	/* ftC4InfoReq command */

#define ftC4RespB2LAD		0x02	/* ftC4InfoResp command */

/*
**	xFunctions
*/

#define ftxSelectLine		0x00	/* SELECT LINE  */
#define ftxCallLAD		0x01	/* CALL_LAD  */
#define ftxHoldCall		0x02	/* HOLD_CALL  */
#define ftxReleaseCall		0x03	/* RELEASE_CALL  */
#define ftxAnswerCall		0x04	/* ANSWER_CALL  */
#define ftxUnholdCall		0x05	/* UNHOLD_CALL  */
#define ftxDialDigits		0x06	/* DIAL_DIGITS  */
#define ftxForward		0x1e	/* FORWARD  */
#define ftxCancelForward	0x1f	/* CANCEL_FORWARD  */
#define ftxdndOn			0x20	/* DND_ON  */
#define ftxdndOff		0x21	/* DND_OFF  */
#define ftxCollectDigits	0x3c	/* COLLECT_DIGITS  */
#define ftxCollectDN		0x3d	/* COLLECT_DN  */
#define ftxGetNumFromReg	0x3e	/* GET_NUMBER_FROM_REGISTER  */
#define ftxStopDigitCollect	0x3f	/* STOP_DIGIT_COLLECTION  */
#define ftxUpdateDisplayReg	0x50	/* UPDATE_DISPLAY_REGISTER  */
#define ftxUpdateSoftReg	0x51	/* UPDATE_SOFTKEY_REGISTER  */
#define ftxClearReg		0x52	/* CLEAR_REGISTER  */
#define ftxCursor		0x55	/* CURSOR  */
#define ftxProgressTone	0x56	/* CALL_PROGRESS_TONE  */
#define ftxUserAlertTone	0x57	/* USER_ALERT_TONE  */
#define ftxIndicator		0x58	/* INDICATOR  */
#define ftxUpdateHFState	0x64	/* UPDATE_HANDSFREE_STATUS  */
#define ftxUpdateMode		0x69	/* UPDATE_MODE  */
#define ftxC4			0xfd	/* CLASS_4 */
#define ftxExtension		0xfe	/* XFUNCTION_EXTENSION */


#define ftxFeatNotInstall	0xf8	/* general xFunction Results */
#define ftxMsgIncomplete	0xf9	
#define ftxNotProvided		0xfa	
#define ftxResSessRequired	0xfb
#define ftxResInvalidXFID	0xfc
#define ftxResInvalidREF	0xfd
#define ftxResInvalidLAD	0xfe

#define ftxInternalLine	0x00	/* ftxSelectLine */
#define ftxPrimeExtLine	0x01
#define ftxAppearanceLAD	0x02
#define ftxLinePool		0x03
#define ftxLineSelOK		0x00	/* results */
#define ftxInvalidLineType	0x01
#define ftxIntLineDialTone	0x02
#define ftxNoFreeIntKey	0x03
#define ftxExtActiveHere	0x04
#define ftxExtActiveThere	0x05
#define ftxNoAppearOfLAD	0x05
#define ftxNoAccessPool	0x07
#define ftxNoFreeInPool	0x08
#define ftxNoFreeIntForPool	0x09
#define ftxKSUSoftErr		0x0a
#define ftxPrimeNotAssigned	0x0b
#define ftxPrimeIsIC		0x0c

#define ftxVoiceCall		0x00	/* ftxCallLAD */
#define ftxSignalCall		0x01
#define ftxCallingLAD		0x00	/* results */
#define ftxInvalidCallType	0x01
#define ftxIntLineDialTone	0x02
#define ftxNoFreeIntKey	0x03
#define ftxAlreadyJoined	0x04
#define ftxCallingOwnLAD	0x05

#define ftxHoldNorm		0x00	/* ftxHoldCall */
#define ftxHoldEx		0x01
#define ftxCallNoAppear	0x00	/* results */
#define ftxCallNotActive	0x01
#define ftxInvalidHoldType	0x02
#define ftxCannotExHold	0x03

#define ftxCallNoAppear	0x00	/* ftxReleaseCall */ /* results */
#define ftxCallNotActive	0x01

#define ftxCallNoAppear	0x00	/* ftxAnswerCall */ /* results */
#define ftxCallNotAnsState	0x01

#define ftxCallNoAppear	0x00	/* ftxUnholdCall */ /* results */
#define ftCallNotHoldState	0x01

#define ftxCallNoAppear	0x00	/* ftxDialDigits */ /* results */
#define ftxCallNotActive	0x01
#define ftxCallNotCO		0x02
#define ftxInvalidDigitLen	0x03
#define ftxBadDigit		0x04

#define ftxFwdToThatLAD	0x00	/* ftxForward */ /* results */
#define ftxFwdToYourLAD	0x01
#define ftxFwdToAnotherLAD	0x02

#define ftxSetNotForwarded	0x00	/* ftxCancelForward */ /* results */

#define ftxdndIsOn		0x00	/* ftxdndOn */ /* results */

#define ftxdndIsOff		0x00	/* ftxdndOff */ /* results */

#define ftxCollectEcho		0x01	/* ftxCollectDigits */
#define ftxInvalidDigCount	0x00	/* ftxCollectDigts */ /* results */
#define ftxInvalidDelimeter	0x01
#define ftxCollectedDigits 	0x03

#define ftxCollectedDN		0x00	/* ftxCollectDN */ /* results */
#define ftxUserError		0x01
#define ftxInvalidDN		0x02

#define ftxLastIntReg		0x00	/* ftxGetNumFromReg */
#define ftxLastExtReg		0x10
#define ftxDigitReg		0x20
#define ftxCollectedDN		0x00	/* results */
#define ftxInvalidReg		0x01
#define ftxEmptyReg		0x02
#define ftxCollectedDigits	0x03
#define ftxNoCollPending	0x04

#define ftxNoCollPending	0x04	/* ftxStopDigitCollect */ /* results */

#define ftxTransient		0x00	/* ftxUpdateDisplayReg */
#define ftxPermanent		0x01
#define ftxSticky		0x03
#define ftxStrTooLong		0x00	/* results */

#define ftxStrTooLong		0x00	/* ftxUpdateSoftReg */ /* results */

#define ftxDisplayReg		0x00	/* ftxClearReg */
#define ftxSoftkeyReg		0x01
#define ftxBothRegs		0x02
#define ftxInvalidRegister	0x00	/* results */

#define ftxCursorOn		0x00	/* ftxCursor */
#define ftxCursorOff		0x01
#define ftxInvalidCursState	0x00	/* results */

#define ftxDialTone		0x00	/* ftxProgressTone */
#define ftxBusyTone		0x01
#define ftxOverflowTone	0x02
#define ftxMusic			0x04
#define ftxRingback		0x05
#define ftxToneOff		0x06
#define ftxInvalidTone		0x00	/* results */

#define ftxAttentionAlrt	0x00	/* ftxUserAlertTone */
#define ftxErrorAlrt		0x01
#define ftxHeldLineRemAlrt	0x02
#define ftxMicOnAlrt		0x03
#define ftxPageAlrt		0x04
#define ftxConfirmationAlrt	0x05
#define ftxPriorityAlrt	0x06
#define ftxInvalidTone		0x00	/* results */

#define ftxIndIdle		0x00	/* ftxIndicator */
#define ftxMyIndicator		0xff
#define ftxIndActive		0x01
#define ftxIndAlert		0x02
#define ftxIndIHold		0x03
#define ftxIndUHold		0x04
#define ftxInvalidIndNum	0x00	/* results */
#define ftxInvalidIndState	0x01
#define ftxFeatHasNoInd	0x02
#define ftxCPLockNotSet	0x03

#define ftxHandsfreeOn		0x00	/* ftxUpdateHFState */
#define ftxHandsfreeMuteOn	0x01
#define ftxHandsfreeMuteOff	0x02
#define ftxHandsfreeOff	0x03
#define ftxInvalidHFState	0x00	/* results */
#define ftxHFNotEnabled	0x01
#define ftxNoLineSelected	0x02

#define ftxCPLockOff		0x00	/* ftxUpdateMode */ /* Prog_Mode */
#define ftxCPLockOn		0x10
#define ftxPKandPKI		0x00
#define ftxOnlyPK		0x04
#define ftxOnlyPKI		0x08
#define ftxIntMode		0x00
#define ftxPassNextKey		0x01
#define ftxExtMode		0x02
#define ftxEchoOff		0x00	/* Key_Pad Mode */
#define ftxEchoOn		0x01
#define ftxProcLocally		0x00
#define ftxDiscard		0x02
#define ftxPassToFeat		0x04
#define ftxSoftDisable		0x00	/* Hard_Key Mode */
#define ftxSoftEnable		0x01
#define ftxShiftDisable	0x00
#define ftxShiftEnable		0x02
#define ftxHookDisable		0x00
#define ftxHookEnableExt	0x04
#define ftxHookEnableInt	0x08
#define ftxHoldDisable		0x00
#define ftxHoldEnableExt	0x10
#define ftxHoldEnableInt	0x20
#define ftxInvalidProgMode	0x00	/* results */
#define ftxInvalidPKMode	0x01
#define ftxInvalidHKMode	0x02


/*
**	Feature Indexes for KSU resident features
*/

#define fiSpeedDial		0x00
#define fiLeavMsgWaiting	0x01
#define fiRingAgain		0x02
#define fiConference		0x03
#define fiCallForward		0x04
#define fiLNR			0x05
#define fiPageGeneral		0x06
#define fiPageInternal		0x07
#define fiPageExternal		0x08
#define fiPageIntExt		0x09
#define fiLinePool		0x0a
#define fiReplyMsgWaiting	0x0b
#define fiVoiceCall		0x0c
#define fiSavedNumberRedial	0x0d
#define fiExecOverride		0x0e
#define fiPriorityCall		0x0f

#define fiBlindTransfer	0x10
#define fiHookFlash		0x11
#define fiTimedRelease		0x12
#define fiSysCallPark		0x14
#define fiGroupCallPickup	0x15
#define fiDirectCallPickup	0x16
#define fiCalltimer		0x17
#define fiPause			0x18
#define fiExclusiveHold	0x19
#define fiDoNotDisturb		0x1f

#define fiBackgroundMusic	0x20
#define fiNightService		0x21
#define fiVoiceCallDeny	0x22
#define fiPreselection		0x23
#define fiKeyInquiry		0x24
#define fiProgramAutoDial	0x25
#define fiProgramDSS		0x26
#define fiProgramFeature	0x27
#define fiProgramSpeedDial	0x28
#define fiProgramLanguage1	0x29
#define fiProgramLanguage2	0x2a
#define fiProgramLanguage3	0x2b
#define fiRingType		0x2c
#define fiContrastControl	0x2d
#define fiRingVolume		0x2e
#define fiRunStopCode		0x2f

#define fiCancelLeaveMsgWait	0x30
#define fiCancelRingAgain	0x31
#define fiCancelCallFwd	0x33
#define fiCancelMsgWaiting	0x34
#define fiCancelDoNotDisturb	0x35
#define fiCancelBackgrndMus	0x36
#define fiCancelNightService	0x37
#define fiCancelVoiceDeny	0x38
#define fiStartupNVRAMClear	0x39
#define fiSystemConfig		0x3a
#define fiGeneralAdmin		0x3b
#define fiChecksumDebug	0x3c
#define fiMaintenance		0x3d
#define fiDebug			0x3e
#define fiPrimeSet		0x3f

#define fiHOLDKey		0x40
#define fiRLSKey			0x41  
#define fiFEAT			0x43
#define fiLATER			0x44
#define fiTimeStamp		0x45
#define fiOffHook		0x46
#define fiOnHook			0x47
#define fiAlarmTerminal	0x49
#define fiRetrSysPark		0x4a
#define fiRetrSetPark		0x4b
#define fiRetrGeneralPark	0x4c	
#define fiHandsfree		0x4d
#define fiCallback		0x4e

#define fiTAFAS			0x50
#define fiLongTone		0x58

#define fiSetAdmin		0x64
#define fiLineAdmin		0x65

#define fiHLRPrime		0xa3
#define fiTxLvMsg		0xa6
#define fiRxLvMsg		0xa7
#define fiRxCanLvMsg		0xa8
#define fiRxCanReplyMsg	0xa9
#define fiFwdHLRPrime		0xac

#define fiTxRAG			0xb4
#define fiRxRAG			0xb5
#define fiCancelRAG		0xb6
#define fiTxOffer		0xb7
#define fiRxOffer		0xb8
#define fiRxWaitOver		0xb9
#define fiRxNoRAG		0xba
#define fiDNDActive		0xbb
#define fiPriCallBlocked	0xbe



/*
**	Fixed LAD's for Servers
**
**  NOTE--These constants are in byte-swapped Intel format
**  when porting this code to a non-Intel environment, you must
**  swap around the order of the bytes.
**
**	The NIL LAD and the NIL REF are also defined here because
**	they are fixed addresses of sorts.
*/

#define nilLAD				0xffef
#define nilREF				0xffff

#define svNetworkManager		0x00c0
#define svTimeServer			0x01c0
#define svTableServer			0x02c0
#define svAdminServer			0x03c0
#define svSpeedDialServer		0x04c0
#define svOverrideServer		0x05c0
#define svAuxRingerServer		0x06c0
#define svPageServer			0x07c0
#define svLinePoolServer		0x08c0
#define svBLFServer			0x09c0
#define svCallPickupServer		0x0ac0
#define svMTCServer			0x0cc0
#define svCallParkServer		0x0dc0
#define svFeatCodeServer		0x0ec0
#define svTranslateServer		0x10c0


/*
**	=====================================================================
**
**		Macros and functions for encoding and decoding 
**		functional messages.
**
**		In these cases, message is a pointer to a functional
**		message stored in a byte array.
**
**	NOTE --  Macros and functions which begin with the prefix "NT" are 
**		   for SCL use only.  These NEVER need be called directly by
**		   application programs.
**	=====================================================================
*/

#define NTlsb( theWord )  	*( ((byte *) &theWord) + 1)	/* internal use */
#define NTmsb( theWord )  	*( (byte *) &theWord)		/* internal use */
int NTgetStatID( word );						/* internal use */
void NTsetStatID( word *, word );					/* internal use */

#define getControlByte( message )		message[0]
#define getLengthByte( message )		message[1]
#define getCmndByte( message )		message[11]
#define getInfoByte( message )		message[10]

#define getSrcLAD( message )			*( (word *) (message + 4) )
#define getSrcREF( message )			*( (word *) (message + 2) )
#define getDestLAD( message )		*( (word *) (message + 8) )
#define getDestREF( message )		*( (word *) (message + 6) )

#define getNodeID( ref )			( (NTmsb(ref) & 0xfc) >> 2 )
#define getStatID( ref )			NTgetStatID( ref )
#define getSrcNodeID( message )  		getNodeID( getSrcREF( message ))
#define getDestNodeID( message )		getNodeID( getDestREF( message ))
#define getSrcStatID( message )		getStatID( getSrcREF( message ))
#define getDestStatID( message )		getStatID( getDestREF( message ))

#define getVar( message )		(message + 12)
#define getVarLen( message)		(message[1] - 12)

#define getByte( ptr, offset ) 		*( ptr + offset)
#define getWord( ptr, offset ) 		*( (word *) ((byte *) ptr + offset))

#define setControlByte( message, theByte )    	message[0] = theByte
#define setCmndByte( message, theByte )	     	message[11] = theByte
#define setInfoByte( message, theByte )	     	message[10] = theByte

#define setSrcLAD( message, theWord )	     	*( (word *) (message + 4) ) = theWord
#define setSrcREF( message, theWord )	     	*( (word *) (message + 2) ) = theWord
#define setDestLAD( message, theWord )	     	*( (word *) (message + 8) ) = theWord
#define setDestREF( message, theWord )	     	*( (word *) (message + 6) ) = theWord

#define setNodeID( ref, id )			NTmsb( ref ) = (NTmsb(ref) & 0x03) | (id << 2)
#define setStatID( ref, id )			NTsetStatID( &ref, id )
#define setSrcNodeID( message, id )     	setNodeID( getSrcREF(message), id )
#define setDestNodeID( message, id )	setNodeID( getDestREF(message), id )
#define setSrcStatID( message, id )	setStatID( getSrcREF(message), id )
#define setDestStatID( message, id )	setStatID( getDestREF(message), id )

#define assignNextByte( message, parm ) 	  	*( message + (*(message+1))++ ) = parm 
byte *assignNextWord( byte * msg, word theWord);
byte *assignNextByteArray( byte * msg, int len, byte * data);
byte *assignNextCstr( byte * msg, char * theStr);

word strToLAD( char *ladStr );
char *ladToStr( word lad, char *ladStr );


/*
**	=====================================================================
**
**	Constants and functions for determining the type and hardware
**	ID of the PCI card.
**
**	=====================================================================
*/

#define noPCInfo              0xFFFF	/* PCI Card Types */
#define pciD_Old              0x0DFD
#define pciB_DChan_NoDMA      0x0000
#define pciB_DChan_DMA        0x0100
#define pciB_BChan_NoDMA      0x0200
#define pciB_BChan_DMA        0x0300

int initPCInfo ( void );

word getPCIVersion( void );

byte *getHardwareID( void );


/*
**	=====================================================================
**
**		Function prototypes, etc. used for buffering
**
**	=====================================================================
*/

int checkFreeBuffers( void );

int checkFreeB2Buffers( void );

int setDefaultSrcLAD( word theLAD );

int setDefaultSrcREF( word theREF );

int initBuffers( byte * data, int maxNumber );

int initB2Buffers( byte * data, int maxNumber );

byte *getMessageBuffer ( int *err );

byte *getB2MessageBuffer ( byte type, int *err );

int sendMessage ( byte *msg );

int sendB2Message ( byte *msg, int number );


/*
**	=====================================================================
**
**		General SCL function prototypes
**
**	=====================================================================
*/

char *SCL_Version( void );

void setKSUVersion( char *version );

int initSCL( void );

int setMessageHandler( int (*handler) () );

int setB2MessageHandler( int (*handler) () );

int (*getMessageHandler( void )) ();

int (*getB2MessageHandler( void )) ();

int setErrorHandler( int (*handler) () );

int processEvent( struct eventRec *evt );



/*
**	=====================================================================
**
**		Function prototypes, etc. used for timers
**
**	=====================================================================
*/

#define	SECOND	18		/* timer ticks in a second */

#define oneShotTimer	0		/* timer types */
#define periodicTimer	1

int checkFreeTimers( void );

int initTimers( byte *data, int maxNumber );

int startTimer( int timerType, int (*handler) (), 
                 int tickCount, int *err );

int updateTimer( int timerID,  int tickCount );



/*
**	=====================================================================
**
**		Error and return codes
**
**	=====================================================================
*/

/*
**	error types 
*/

#define opErr		1	/* operational error */
#define raErr		2	/* resource allocation error */
#define idErr		3	/* invalid data error */

/*
**	error codes
*/

#define noErr			0	/* no error */

#define pciReset			2	/* Norstar System Software errors */
#define rxOverflow		10
#define evqOverflow		32
#define exqOverflow		8
#define ctrlAltDel		5
#define timerQCorrupt		42
#define frxOverflow		70
#define fevqOverflow		71

#define pciNotResponding    	101	/* PCI card not responding */

#define noFreeBuffers		150	/* no free buffers exist */
#define hasOneBuffer		151	/* one buffer already allocated */
#define bufferNotAllocated	152	/* buffer is not allocated */
#define invalidMsgLen		154	/* message length invalid */
#define invalidBufferInit	155	/* attempt to initialize buffers */
						/* failed because of invalid data */
#define cannotSendMessage	156	/* Nortalk could not send message */
#define bufferWasDeallocated	666	/* app had a buffer allocated when */
						/* its message handler finished */
#define invalidByteArrayLen	158	/* byte array too large to fit in */
						/* a functional message, or byte */
						/* array <= 0. */
#define invalidMsgBuffer	159	/* attempt to xmit an invalid buffer */

#define noTimersAvailable	170	/* no timer could be allocated */
#define invalidTimerID		173	/* timer ID is invalid */
#define noFreeNorstarTimers	174	/* could not allocate a Nortalk */
						/* timer for use by the SCL */
#define invalidTimerInit	175	/* attempt to initialize timers failed */
						/* because of invalid data */
#define invalidTickCount	176	/* tick count out of range */
#define cannotRegister		177	/* cannot register with NSS */
#define appAlreadyRegistered	178	/* app already registered with NSS */



