;***********************************************************************
;*                                                                     *
;* BJACK.ASP  (C) 1990 DATASTORM TECHNOLOGIES, INC.                    *
;*                                                                     *
;* An ASPECT script designed to play Blackjack.  Define macro DANCEINC *
;* to include the code for the dancers who entertain the user while    *
;* the cards are being shuffled.                                       *
;*                                                                     *
;***********************************************************************

;define FORCEMONO                      ;* define to force monochrome display

define DANCEINC                        ;* define this to include dancers

;***********************************************************************
;* General macro definitions                                           *
;***********************************************************************

define getkey call keyget2 with

define TRUE       1                    ;* boolean true
define FALSE      0                    ;* boolean false

define ESC       27                    ;* value returned by Esc key

define CASH     500                    ;* user's cash reserve
define BET       10                    ;* default bet per hand

define CARDS     52                    ;* size of deck
define RANKS     13                    ;* number of ranks
define SUITS      4                    ;* number of suits.

;***********************************************************************
;* Display values for card suit and rank                               *
;***********************************************************************

define HEARTS    ''
define DIAMONDS  ''
define CLUBS     ''
define SPADES    ''

define TEN       'T'
define JACK      'J'
define QUEEN     'Q'
define KING      'K'
define ACE       'A'

;***********************************************************************
;* Global variables                                                    *
;***********************************************************************

integer CARDATTR, REDCARD, BLACARD, BOXATTR, USRATTR, MSGATTR, ERRATTR
integer GALATTR, GUYATTR
integer isounds, deckcnt = 0, dealcnt, usercnt, dealscor, userscor
integer ante = BET, dollars = CASH, hands = 0
string carddeck, dealhand, userhand, username

;***********************************************************************
;* Main Procedure                                                      *
;***********************************************************************

proc main
   set keys on
   set rxdata on
   call intro                          ;* title screen and name prompt
   call clrscrn                        ;* clear playing area
   while forever
      if (deckcnt < 15)
         call shuffle                  ;* shuffle cards
      endif
      call playhand                    ;* deal a hand
      call playmenu                    ;* prompt to play again
   endwhile
endproc

;***********************************************************************
;* Clear the playing area                                              *
;***********************************************************************

proc clrscrn
   scroll 0 1 1 23 78 BOXATTR
endproc

;***********************************************************************
;* Clear the message area                                              *
;***********************************************************************

proc clrmsg
   scroll 0 9 2 15 77 BOXATTR
endproc

;***********************************************************************
;* Get a key from the user (alphabetic forced upper case)              *
;***********************************************************************

proc keyget2
intparm key

   keyget key
   if (key == ESC)
      call restore with 0
   endif
   if ((key >= 'a') && (key <= 'z'))
      key -= 32
   endif
endproc

;***********************************************************************
;* This procedure manages the game.  It accepts the deck string, the   *
;* user and dealer handstrings, and handles user input throughout game *
;***********************************************************************

proc playhand
   integer pay = -ante
   string msg

   userhand = $null
   dealhand = $null
   usercnt = dealcnt = userscor = dealscor = 0
   call usercard
   call dealcard
   call usercard
   call dealcard
   if (userscor == 21)
      msg = "   Blackjack! You win!"
      pay = ante * 2
      call drawcard with dealhand dealcnt 1 1
   else
      call userplay
      if (userscor <= 21)
         call dealerplay
         if (dealscor <= 21)
            if (dealscor == userscor)
               msg = "Dealer pushes. House wins."
            elseif (dealscor < userscor)
               msg = "  Dealer lost. You win!"
               pay = ante
            else
               msg = "  You lost. House wins."
            endif
         else
            msg = " Dealer busted! You win!"
            pay = ante
         endif
      else
         msg = "You're busted! House wins."
         call drawcard with dealhand dealcnt 1 1
      endif
   endif
   dollars += pay
   hands++
   if isounds
      if pay >= 0
         sound 440 2
         sound 880 2
      else
         sound 220 5
      endif
   endif
   atsay 11 27 MSGATTR msg
   atsay 13 27 MSGATTR "Press any key to continue."
   keyget
endproc

;***********************************************************************
;* Allows user to take cards until busted or standing pat              *
;***********************************************************************

proc userplay
   integer key

   atsay 11 30 MSGATTR "1. Take a card"
   atsay 12 30 MSGATTR "2. Stand pat"
   while (userscor < 21)
      key = 0
      while ((key < '1') || (key > '2'))
         getkey &key
      endwhile
      if (key == '1')
         call usercard
      else
         exitwhile
      endif
   endwhile
   call clrmsg
endproc

;***********************************************************************
;* Dealer must draw cards until score is 17 or greater (dealer must    *
;* hit on a "soft" 17)                                                 *
;***********************************************************************

proc dealerplay
   call drawcard with dealhand dealcnt 1 1
   while(dealscor < 17)
      mspause 500
      call dealcard
   endwhile
endproc

;***********************************************************************
;* Draw last card in hand                                              *
;***********************************************************************

proc drawcard
strparm cardhand
intparm cardcnt, topdisp, showit
   string thiscard
   integer cardint, cardcolor
   integer x1 = 3, y1, x2, y2

   cardcnt--                           ;* use 0-based index
   if !topdisp
      x1 += 13
   endif
   x2 = x1 + 5
   y1 = (cardcnt * 8) + 19
   y2 = y1 + 5
   box x1 y1 x2 y2 CARDATTR
   if showit
      strpeek cardhand cardcnt cardint
      call whatcard with &thiscard &cardcolor cardint
      x1++
      y1++
      x2--
      y2 -= 2
      atsay x1 y1 cardcolor thiscard
      atsay x2 y2 cardcolor thiscard
   endif
endproc

;***********************************************************************
;* Get next card into user's hand, display it, and score it            *
;***********************************************************************

proc usercard
   call nextcard with &userhand &usercnt
   call drawcard with userhand usercnt 0 1
   call scorehand with 0 userhand usercnt &userscor
endproc

;***********************************************************************
;* Get next card into dealer's hand, display it, and score it          *
;***********************************************************************

proc dealcard
   integer showit = 1

   call nextcard with &dealhand &dealcnt
   if (dealcnt == 2)
      showit--
   endif
   call drawcard with dealhand dealcnt 1 showit
   call scorehand with 1 dealhand dealcnt &dealscor
endproc

;***********************************************************************
;* Removes a card from the deck and places it into a particular hand   *
;***********************************************************************

proc nextcard
strparm cardhand
intparm cardcnt
   integer ncard

   strpeek carddeck 0 ncard
   substr carddeck carddeck 1 80
   strfmt cardhand "%s%c" cardhand ncard
   deckcnt--
   cardcnt++
endproc

;***********************************************************************
;* Proc scorehand scores the cards in the user's hand returning the    *
;* value in the integer parameter shandscor.  ACEs are assumed a value *
;* of 11 for user's hand unless score is > 21;  Dealer's hand must use *
;* an ACE value of 1 on "soft" 17                                      *
;***********************************************************************

proc scorehand
intparm dealer
strparm cardhand
intparm cardcnt, handscor
   integer ace11 = 0                   ;* ACE scored as 11 counter
   integer index                       ;* location in string
   integer peekint                     ;* current card

   cardcnt--                           ;* 0-based index
   handscor = 0
   for index = 0 upto cardcnt
      strpeek cardhand index peekint
      peekint = ((peekint - 1) % RANKS) + 1
      switch peekint
         case 1                        ;* ACE
            handscor += 11             ;* score as 11
            inc ace11                  ;* increment ACE counter
         endcase
         case 11                       ;* JACK
         case 12                       ;* QUEEN
         case 13                       ;* KING
            handscor += 10
         endcase
         default                       ;* TWO through TEN
            handscor += peekint
         endcase
      endswitch
   endfor
   while ((handscor > 21) && ace11)
      handscor -= 10                   ;* subtract 10 (which leaves 1 for ace)
      dec ace11                        ;* decrement ace11 counter
   endwhile
   if (dealer && (handscor == 17) && ace11)
      handscor -= 10                   ;* no soft 17's allowed
   endif
endproc

;***********************************************************************
;* Proc whatcard returns a display string representing the card and    *
;* an integer which represents the display color to use.               *
;***********************************************************************

proc whatcard
strparm wthiscard
intparm wcardcolor, wcardint
   integer selector

   wcardint--                          ;* make card value 0-based
   selector = (wcardint % RANKS) + 1   ;* get card rank (1 to 13)
   switch selector
      case 1
         selector = ACE
      endcase
      case 10
         selector = TEN
      endcase
      case 11
         selector = JACK
      endcase
      case 12
         selector = QUEEN
      endcase
      case 13
         selector = KING
      endcase
      default
         selector += '0';              ;* TWO through NINE
      endcase
   endswitch
   wthiscard = "  "                    ;* initialize to blanks
   strpoke wthiscard 0 selector        ;* assign rank
   div wcardint 13 selector
   selector = (wcardint / 13) + 3
   strpoke wthiscard 1 selector        ;* assign suit
   if (selector < CLUBS)
      wcardcolor = REDCARD
   else
      wcardcolor = BLACARD
   endif
endproc

;***********************************************************************
;* This menu prompts to the user for continued play                    *
;***********************************************************************

proc playmenu
   integer key
   string msg

   if isounds
      sound 300 1
      sound 900 1
      sound 100 1
      sound 1200 1
   endif
   call clrscrn                        ;* clear playing area
   strfmt msg "%s, you have $%d after %d hand(s)" username dollars hands
   strlen msg key
   key = ((80 - key) >> 1);
   atsay 10 key MSGATTR msg
   atsay 12 33 MSGATTR "1. Play again"
   atsay 13 33 MSGATTR "2. Change bet"
   atsay 14 33 MSGATTR "3. Quit Game"
   atsay 15 33 MSGATTR "4. Exit PCPLUS"
   while ((key < '1') || (key > '4'))
      getkey &key
   endwhile
   switch key
      case '2'                         ;* change bet
         call userbet
      endcase
      case '3'                         ;* quit
         call restore with 0
      endcase
      case '4'                         ;* exit
         call restore with 1
      endcase
   endswitch
   call clrscrn                        ;* clear playing area
endproc

;***********************************************************************
;* Prompts for a new betting level which cannot exceed house limit     *
;***********************************************************************

proc userbet
   call clrmsg
   fatsay 11 22 MSGATTR "Current betting level is $%d per hand" ante
   atsay  12 22 MSGATTR "Enter a new value (Limit $100):"
   while forever
      atget 12 54 USRATTR 3 ante
      if failure
         call restore with 0
      endif
      if ((ante > 0) && (ante <= 100))
         call clrmsg
         return
      endif
      if isounds
         sound 220 5
      endif
   endwhile
endproc

;***********************************************************************
;* Title screen and introductory procedure                             *
;***********************************************************************

proc intro
   integer key

   call setcolor                       ;* initialize display attributes
   curoff
   vidsave 0
   set statline off
   box 0 0 24 79 BOXATTR
   atsay  5  3 MSGATTR "                                                        "
   atsay  6  3 MSGATTR "                                                             "
   atsay  7  3 MSGATTR "                                               "
   atsay  8  3 MSGATTR "                                                   "
   atsay  9  3 MSGATTR "                                             "
   atsay 10  3 MSGATTR "                                                       "
   atsay 11  3 MSGATTR "                                                     "
   atsay 12  3 MSGATTR "                                     "
   atsay 13  3 MSGATTR "                                             "
   atsay 14  3 MSGATTR "                                            "
   atsay 15  3 MSGATTR "                                         "
   atsay 18 17 ERRATTR "COPYRIGHT (C) 1990 DATASTORM TECHNOLOGIES, INC."
   atsay 21 17 MSGATTR "Please enter your name:"
   atget 21 41 USRATTR 23 username
   if failure
      call restore with 0
   endif
   scroll 0 21 17 21 78 BOXATTR
   atsay 21 27 MSGATTR "Do you want sounds? (Y/N)"
   getkey &key
   scroll 0 21 17 21 78 BOXATTR
   if (key == 'Y')
      isounds = TRUE
   else
      isounds = FALSE
   endif
endproc

;***********************************************************************
;* Set up display attributes                                           *
;***********************************************************************

proc setcolor
   integer x = 0

   $ifdef FORCEMONO
      inc x
   $endif
   if (mono || x)
      CARDATTR = 112
      REDCARD  = 112
      BLACARD  = 112
      BOXATTR  =  15
      USRATTR  = 112
      MSGATTR  =  15
      ERRATTR  =  15
      GUYATTR  =  14
      GALATTR  =  15
   else
      CARDATTR = 113
      REDCARD  = 116
      BLACARD  = 112
      BOXATTR  =  31
      USRATTR  = 112
      MSGATTR  =  31
      ERRATTR  =  30
      GUYATTR  =  48
      GALATTR  =  62
   endif
endproc

;***********************************************************************
;* Clean up (if necessary) and exit the script file or program         *
;***********************************************************************

proc restore
intparm exitmode

   if exitmode
      quit
   endif
   clear
   set statline on
   vidrest 0
   curon
   exit
endproc

;***********************************************************************
;* Shuffle the deck of cards                                           *
;* ASCII values 1 through 52 are used to represent a deck of cards.    *
;* Rank is determined from (card value - 1) mod 13                     *
;* Suit is determined from (card value - 1) div 13                     *
;***********************************************************************

proc shuffle
   integer cindex, dindex, x
   $ifdef DANCEINC
   integer dancing = 0
   $endif

   vidsave 1
   x = CARDS + 1
   strset carddeck 0 x                 ;* initialize new deck to nulls
   $ifdef DANCEINC
      call clrscrn
      atsay 21 28 MSGATTR ">>> Shuffling cards <<<"
   $else
      call clrmsg
      atsay 12 28 MSGATTR ">>> Shuffling cards <<<"
   $endif
   for cindex = 1 upto CARDS
      $ifdef DANCEINC
         call dancekarl with dancing 7 19
         call dancekyla with dancing 7 45
         dancing = (dancing + 1) % 3
      $endif
      call rand with &dindex CARDS
      while forever
         strpeek carddeck dindex x
         if !x                         ;* if no card currently assigned
            strpoke carddeck dindex cindex
            if isounds
               sound 440 1
               sound 2000 1
            endif
            exitwhile
         endif
         dindex = (dindex + 7) % CARDS
      endwhile
   endfor
   deckcnt = CARDS
   vidrest 1
endproc

;***********************************************************************
;* Return a "random" value between 0 and (modulus - 1)                 *
;***********************************************************************

proc rand
intparm value, modulus
   integer x

   mempeek 0 0x046C value
   mempeek 0 0x0440 x
   value = (value * x) % modulus
   if (value < 0)
      value = -value
   endif
endproc

$ifdef DANCEINC

;***********************************************************************
;* Karl, the dancing guy (and card shuffler)                           *
;***********************************************************************

proc dancekarl
intparm whozatguy
intparm wherex
intparm wherey

   switch whozatguy
      case 0
        atsay wherex wherey GUYATTR "      mmmmm     "
        inc wherex
        atsay wherex wherey GUYATTR "   m O O m    "
        inc wherex
        atsay wherex wherey GUYATTR "   (    )    "
        inc wherex
        atsay wherex wherey GUYATTR "    \  /    "
        inc wherex
        atsay wherex wherey GUYATTR "  \-**  **-/  "
        inc wherex
        atsay wherex wherey GUYATTR "    ******    "
        inc wherex
        atsay wherex wherey GUYATTR "      ****    "
        inc wherex
        atsay wherex wherey GUYATTR "      ******    "
        inc wherex
        atsay wherex wherey GUYATTR "          "
        inc wherex
        atsay wherex wherey GUYATTR "            "
        inc wherex
        atsay wherex wherey GUYATTR "            "
        inc wherex
        atsay wherex wherey GUYATTR "          "
     endcase
     case 1
        atsay wherex wherey GUYATTR "      mmmmm     "
        inc wherex
        atsay wherex wherey GUYATTR "     m   m    "
        inc wherex
        atsay wherex wherey GUYATTR "     (    )    "
        inc wherex
        atsay wherex wherey GUYATTR "      \  /    "
        inc wherex
        atsay wherex wherey GUYATTR "   /-**  **-/ "
        inc wherex
        atsay wherex wherey GUYATTR "   U  ******   "
        inc wherex
        atsay wherex wherey GUYATTR "    ****    "
        inc wherex
        atsay wherex wherey GUYATTR "   ******     "
        inc wherex
        atsay wherex wherey GUYATTR "   ߲     "
        inc wherex
        atsay wherex wherey GUYATTR "            "
        inc wherex
        atsay wherex wherey GUYATTR "           "
        inc wherex
        atsay wherex wherey GUYATTR "             "
     endcase
     case 2
        atsay wherex wherey GUYATTR "      mmmmm     "
        inc wherex
        atsay wherex wherey GUYATTR "     m   m    "
        inc wherex
        atsay wherex wherey GUYATTR "     (    )    "
        inc wherex
        atsay wherex wherey GUYATTR " \ = / "
        inc wherex
        atsay wherex wherey GUYATTR "   \-**  **-/   "
        inc wherex
        atsay wherex wherey GUYATTR "     ******     "
        inc wherex
        atsay wherex wherey GUYATTR "      ****    "
        inc wherex
        atsay wherex wherey GUYATTR "       ******   "
        inc wherex
        atsay wherex wherey GUYATTR "         "
        inc wherex
        atsay wherex wherey GUYATTR "            "
        inc wherex
        atsay wherex wherey GUYATTR "           "
        inc wherex
        atsay wherex wherey GUYATTR "             "
     endcase
  endswitch
endproc


;***********************************************************************
;* Kyla, the dancing girl!                                             *
;***********************************************************************

proc dancekyla
intparm whozatgirl
intparm wherex
intparm wherey

   switch whozatgirl
      case 0
        atsay wherex wherey GALATTR "     mmmmm     "
        inc wherex
        atsay wherex wherey GALATTR "    m O O m    "
        inc wherex
        atsay wherex wherey GALATTR "    m    m    "
        inc wherex
        atsay wherex wherey GALATTR "    \  /    "
        inc wherex
        atsay wherex wherey GALATTR "   \-**  **-/  "
        inc wherex
        atsay wherex wherey GALATTR "     ****    "
        inc wherex
        atsay wherex wherey GALATTR "      ****     "
        inc wherex
        atsay wherex wherey GALATTR "     ******    "
        inc wherex
        atsay wherex wherey GALATTR "    ((((((((   "
        inc wherex
        atsay wherex wherey GALATTR "      () ()    "
        inc wherex
        atsay wherex wherey GALATTR "      || ||    "
        inc wherex
        atsay wherex wherey GALATTR "         "
     endcase
     case 1
        atsay wherex wherey GALATTR "     mmmmm     "
        inc wherex
        atsay wherex wherey GALATTR "    m   m    "
        inc wherex
        atsay wherex wherey GALATTR "    mm   mm   "
        inc wherex
        atsay wherex wherey GALATTR "     \  /    "
        inc wherex
        atsay wherex wherey GALATTR "    /-**  **-/ "
        inc wherex
        atsay wherex wherey GALATTR "   U  ****   "
        inc wherex
        atsay wherex wherey GALATTR "      ****     "
        inc wherex
        atsay wherex wherey GALATTR "    ******     "
        inc wherex
        atsay wherex wherey GALATTR "  ((((((((     "
        inc wherex
        atsay wherex wherey GALATTR "      () ()    "
        inc wherex
        atsay wherex wherey GALATTR "      ||    "
        inc wherex
        atsay wherex wherey GALATTR "            "
     endcase
     case 2
        atsay wherex wherey GALATTR "     mmmmm     "
        inc wherex
        atsay wherex wherey GALATTR "    m   m    "
        inc wherex
        atsay wherex wherey GALATTR "   mm   mm    "
        inc wherex
        atsay wherex wherey GALATTR "    \  /     "
        inc wherex
        atsay wherex wherey GALATTR "  \-**  **-\   "
        inc wherex
        atsay wherex wherey GALATTR "    ****  U  "
        inc wherex
        atsay wherex wherey GALATTR "      ****     "
        inc wherex
        atsay wherex wherey GALATTR "      ******   "
        inc wherex
        atsay wherex wherey GALATTR "      )))))))) "
        inc wherex
        atsay wherex wherey GALATTR "      () ()    "
        inc wherex
        atsay wherex wherey GALATTR "      ||    "
        inc wherex
        atsay wherex wherey GALATTR "            "
     endcase
  endswitch
endproc

$endif

