/******************************** wvfmtest.c ********************************/
/*                                                                          */
/*     This program makes use of the data acquisition library Counter and   */
/*  Waveform Functions. While illustrating the use of these functions, this */
/*  program also performs the useful task of determining the minimum update */
/*  interval for waveform generation on your particular computer. The two   */
/*  main factors governing the minimum update interval are the interrupt    */
/*  latency and the execution speed of the interrupt service routine. When  */
/*  the update interval approaches the sum of these two quantities          */
/*  interrupts begin to be missed.                                          */
/*                                                                          */
/*     The program works in this way. A waveform is generated that is a 50% */
/*  duty cycle square wave. The low value is 0 volts and the high value is  */
/*  5 volts. This produces a TTL signal that can be counted by the onboard  */
/*  counters on the MIO-16. Counter 5 is set up to count the signal. The    */
/*  length of time to count a certain number of rising edges (default is    */
/*  10000) is computed. This length of time depends on the update interval  */
/*  used to produce the square wave. Counter 1 is set to produce a pulse of */
/*  this duration. The pulse is used to turn (gate) Counter 5 on and off.   */
/*  The number of rising edges counted can then be compared to the number of*/
/*  edges expected.                                                         */
/*                                                                          */
/****************************************************************************/

#define  EDGES_TO_CNT  10000.0      /* the number of rising edges to count */
#define  MISS_LIMIT    20           /* missing MISS_LIMIT or more edges causes
                                       the program to prompt for termination */
#define  WV_BUF_SIZE   50           /* the size of the waveform buffer */
#define  TIME_OUT_VAL  50000L       /* the program will wait for a finite
                                       time for the gating pulse to begin
                                       and end */

static double wvVolts[WV_BUF_SIZE]; /* the voltage waveform buffer */
static    int wvBuf[WV_BUF_SIZE];   /* the digital waveform buffer */

void errCheck (int, char *, int);
double tb_to_sec (int);
/*
 * This routine converts a timebase code to the timebase resolution in
 * seconds
 */

double tb_to_sec (tb)
int tb;
{
    int i, multiplier;

    multiplier = 1;
    for (i=1; i<tb; i++) multiplier *= 10;
    return 0.000001 * multiplier;
}

/*
 * The main routine
 */

main()
{
    int
       i, j, err, cr,
       board,           /* the slot/ID of the MIO-16 or MIO-16F-5 */
       number_reps,     /* the number of tests to perform at each update
                           interval */
       PLStimebase,     /* the timebase to be used to produce the gating
                           pulse */
       WFtimebase,      /* the timebase to be used to produce the waveform */
       CTR1_state,      /* the state (high or low) of the gating pulse */
       CTR5_overflow,   /* the overflow status of Counter 5 */
       CTR5_count,     /* the number of rising edges counted by Counter 5 */
       abort_test;      /* a flag used to terminate early */

   long
       start_update,    /* the beginning update interval in the range to be
                           tested */
       stop_update,     /* the final update interval in the range to be
                           tested */
       step_size,       /* the number of update interval values to skip
                           between tests */
       update_int,      /* the update interval being tested */
       pulse_width,     /* the pulse width of the gating pulse */
       CTR5_lcount,     /* the number of rising edges counted by Counter 5 */
       edges_expected;  /* the number of rising edges expected calculated
                           from the update interval and the length of the
                           gating pulse */

   long
       waiting;         /* when this value reaches 0, the wait loops time out */

   double
       secs_per_edge,   /* these are all variables used in calculating the */
       edges_per_sec,   /* width of the gating pulse and the number of edges */
       pls_width_in_sec,   /* expected */
       temp;

/* ------------------------------ begin ----------------------------------- */

   FmtOut("\nThis example will operate on an MIO-16");
   FmtOut("\nEnter the slot number of a MIO-16:");
   ScanIn("%d", &board);

   FmtOut("\n\nThis routine uses the counters to count rising edges produced by");
   FmtOut("\nWF_Start and can be used to detect missed updates.");
   FmtOut("\nConnect OUT1 (pin 43) to GATE5 (pin 48).");
   FmtOut("\nConnect DAC0OUT (pin 20) to SOURCE5 (pin 47).");

   FmtOut("\n\nEnter maximum update interval to be tested (microsec): ");
   ScanIn("%d[b4]",&start_update);
   FmtOut("\n\nWARNING: A sufficiently small update interval will cause your");
   FmtOut("\n         computer to lock up. If this happens, simply reboot.");
   FmtOut("\n\nEnter minimum update interval to be tested (microsec): ");
   ScanIn("%d[b4]",&stop_update);
   FmtOut("\nEnter the step size to use between tests (microsec): ");
   ScanIn("%d[b4]",&step_size);
   FmtOut("\nEnter number of repetitions for each update interval: ");
   ScanIn("%d",&number_reps);

/* configure counter 5 for high level gating, count rising edges, ...
   Since the board has been verified by the above call to CTR_Reset,
   we may ignore the error return value. */
   CTR_Config (board, 5, 0, 1, 0, 0);

/* fill the waveform buffer with the values, scale and load it */
   for (i=0; i<WV_BUF_SIZE-1; i+=2)
   {
      wvVolts[i] = 0.0;
      wvVolts[i+1] = 5.0;
   }
   err = WF_Scale (board, 0, WV_BUF_SIZE, 1.0, wvVolts, wvBuf);
   errCheck (board, "WF_Scale", err);

   err = WF_Load (board, 0, wvBuf, WV_BUF_SIZE, 0);
   errCheck (board, "WF_Load", err);

   WFtimebase = 1;   /* use the 1 micro second timebase for the waveform */

/* enter test loop */
   abort_test = 0;
   for (update_int =  start_update;
        update_int >= stop_update && abort_test == 0;
        update_int -= step_size)
   {

   /* calculate pulse width to count approx EDGES_TO_CNT edges */
      secs_per_edge = tb_to_sec (WFtimebase) * ((double)update_int) * 2.0;
      pls_width_in_sec = EDGES_TO_CNT * secs_per_edge;
      PLStimebase = 0;
      temp = 0;
      while ((temp < 3.0 || temp > 65535.0) && PLStimebase < 5)
      {
         PLStimebase++;
         temp = pls_width_in_sec / tb_to_sec (PLStimebase);
      }

   /* adding 0.5 and then casting is a rounding technique */
      pulse_width = (long) (temp + 0.5);
      if (PLStimebase != 6 && pulse_width != 0)
      {
         /* run each test several times */
         i = 0;
         while (i < number_reps)
         {
        /* set the ouptut state of counter 1 to logical low */
            err = CTR_Reset (board, 1, 1);
            errCheck (board, "CTR_Reset", err);

         /* configure counter 5 for event counting */
            CTR_EvCount (board, 5, 10, 0);
            errCheck (board, "CTR_EvCount", err);

         /* start the waveform */
            err = WF_Start (board, WFtimebase, (int) update_int);
            errCheck (board, "WF_Start", err);

         /* initiate the gating pulse */
            err = CTR_Pulse (board, 1, PLStimebase, 3, (int) pulse_width);
            errCheck (board, "CTR_Pulse", err);

         /* wait loop for the gating pulse to begin */
         /* the pulse has started when Counter 1's output is high */
            waiting = PLStimebase * TIME_OUT_VAL;
            CTR1_state = 0;
            err = 0;
            while (!CTR1_state && !err && waiting)
            {
               waiting--;
               err = CTR_State (board, 1, &CTR1_state);
            }

            if (waiting == 0L)
            {
               FmtOut("\nUpdate %d: Wait loop for the gating pulse to begin timed out.",update_int);
               i = number_reps;
               /* try next update_interval */
            }
            else
            {
            /* wait for the gating pulse to finish */
            /* the pulse is finished when Counter 1's output is low */
               waiting = PLStimebase * TIME_OUT_VAL;
               CTR1_state = 1;
               err = 0;
               while (CTR1_state && !err && waiting)
               {
               waiting--;
               err = CTR_State (board, 1, &CTR1_state);
               }

               if (waiting == 0L)
               {
                  FmtOut("\nUpdate %d: Wait loop for the gating pulse to finish timed out.",update_int);
                  i = number_reps;
               /* try next update_interval */
               }
               else
               {
               /* pausing the waveform speeds up the following operations */
                  WF_Pause (board);
               /* read the number of rising edges counted */
                  CTR_EvRead (board, 5, &CTR5_overflow, &CTR5_count);
                  if (CTR5_overflow)
                  {
                     FmtOut("\nCounter 5 overflowed.");
                     i = number_reps;
                  /* try next update interval */
                  }
                  else
                  {
                     if (CTR5_count < 0)
                        CTR5_lcount = CTR5_count + 65536;
                     else
                        CTR5_lcount = CTR5_count;
                     edges_per_sec = 1.0 / secs_per_edge;
                     pls_width_in_sec = tb_to_sec (PLStimebase) * ((double)pulse_width);
                     edges_expected = (long) (edges_per_sec * pls_width_in_sec + 0.5);
                     FmtOut("\nUpdate interval = %d[b4], expected edges = %d[b4], and counted edges = %d[b4]",
                            update_int,edges_expected,CTR5_lcount);
                     if (edges_expected - CTR5_lcount >= MISS_LIMIT)
                     {
                        FmtOut("\nInterrupts are being missed, do you wish to continue? (y/n): ");
                        cr = getkey();
                        if (cr != 'y' && cr != 'Y')
                        {
                           abort_test = 1;
                           i = number_reps;
                        }
                     }
                  }
               }
            }
         i++;
         }
      }
      FmtOut("\n");
   }
   WF_Clear (board);
}

/*
 * This procedure calls ErrPrint to print an error message. If the error is
 * a serious one (as opposed to a warning) this procedure terminates the
 * program. Note that WF_Clear is called before termination.
 */

void errCheck (board, proc_name, err_num)
int board;
char *proc_name;
int err_num;
{
   if (err_num < 0)
   {
      FmtOut("%s returned an error of %d\n", proc_name, err_num);
      WF_Clear (board);
      exit(1);
   }
}
