/*=========================================================================*/
/*  This program illustrates the use of the selection list routines        */
/*  available in the User Interface Library.                               */
/*=========================================================================*/

/*= INCLUDES ==============================================================*/
/*                                                                         */
/*  Remember, never modify the contents of include files generated by the  */
/*  User Interface Editor.                                                 */
/*=========================================================================*/

#include "uil_ex7.h"

/*= DEFINES ===============================================================*/

#define  TRUE   1
#define  FALSE  0

/*= FUNCTION DECLARATIONS =================================================*/

int AddEntry (int, int);

void EditEntry (int);
void ClearFields (void);
void DisplayEntry (void);

/*= STATIC VARIABLES ======================================================*/

static int Panel;
static int NoEdit;
static int ListAltered;

static char ProdName[21];
static char SerialNo[18];
static char TestName[51];
static char TestDate[26];
static char TestTime[18];
static char Operator[51];
static char Entry[200];
static char Dirname[68];
static char Pathname[80];

/*= MAIN PROGRAM ==========================================================*/

main()

{
  int id,ndx,handle;

  /*-----------------------------------------------------------------------*/
  /*  Load the panel from the resource file.  Use the constant assigned    */
  /*  in the editor to refer to the panel.  The handle returned by         */
  /*  LoadPanel must be used to reference the panel in all subsequent      */
  /*  function calls. If the panel handle is negative, the load failed,    */
  /*  so print a message and exit the program.                             */
  /*-----------------------------------------------------------------------*/
  Panel = LoadPanel("uil_ex7.uir",INFO);
  if (Panel < 0) {
    FmtOut("Unable to load the required panel from the resource file.\n");
    return;
  }
  /*-----------------------------------------------------------------------*/
  /*  The test log data is stored in a panel state file which is recalled  */
  /*  before defaulting and displaying the panel.   This panel state file  */
  /*  is maintained in the same directory as the program.                  */
  /*-----------------------------------------------------------------------*/
  GetProgramDir(Dirname);
  MakePathname(Dirname,"uil_ex7.log",Pathname);
  RecallPanelState(Panel,Pathname);
  DefaultPanel(Panel);
  DisplayPanel(Panel);

  NoEdit      = TRUE;
  ListAltered = FALSE;

  while (TRUE) {
    /*---------------------------------------------------------------------*/
    /*  Obtain the index of current list item.  The index will be -1 if    */
    /*  the list is empty.                                                 */
    /*---------------------------------------------------------------------*/
    GetListItemIndex(Panel,INFO_LIST,&ndx);
    /*---------------------------------------------------------------------*/
    /*  Enable or disable input on the test log list and the six push      */
    /*  buttons based on the following criteria:                           */
    /*                                                                     */
    /*    ADD    - enabled unless an entry is being edited.                */
    /*    EDIT   - enabled unless the list is empty.                       */
    /*    CANCEL - enabled only when an entry is being edited.             */
    /*    DELETE - enabled unless the list is empty or an entry is being   */
    /*             edited.                                                 */
    /*    EXPAND - enabled unless the list is empty or an entry is being   */
    /*             edited.                                                 */
    /*    QUIT   - enabled unless an entry is being edited.                */
    /*---------------------------------------------------------------------*/
    SetInputMode(Panel,INFO_ADD,NoEdit);
    SetInputMode(Panel,INFO_EDIT,(ndx >= 0));
    SetInputMode(Panel,INFO_CANCEL,!NoEdit);
    SetInputMode(Panel,INFO_DELETE,NoEdit && (ndx >= 0));
    SetInputMode(Panel,INFO_EXPAND,NoEdit && (ndx >= 0));
    SetInputMode(Panel,INFO_QUIT,NoEdit);
    /*---------------------------------------------------------------------*/
    /*  Call GetUserEvent with the wait parameter set to TRUE.  This will  */
    /*  cause the function to wait for an event.  When an event occurs,    */
    /*  the id variable will match one of the control ID constants         */
    /*  assigned in the editor.  Since we only have one panel loaded, we   */
    /*  do not have to examine the handle value returned by GetUserEvent.  */
    /*---------------------------------------------------------------------*/
    GetUserEvent(1,&handle,&id);
    /*---------------------------------------------------------------------*/
    /*  If we are not editing Since the user might have changed the current list item, we must   */
    /*  obtain the list item index again.                                  */
    /*---------------------------------------------------------------------*/
    GetListItemIndex(Panel,INFO_LIST,&ndx);
    /*---------------------------------------------------------------------*/
    /*  Remember, only those push buttons that are enabled can generate    */
    /*  events.                                                            */
    /*---------------------------------------------------------------------*/
    switch (id) {
      case INFO_OPERATOR :
        /*-----------------------------------------------------------------*/
        /*  Since the operator control is the last data field of an        */
        /*  entry, it is made "hot" so that it will generate events when   */
        /*  ENTER is pressed while it is active.  If the entry is not      */
        /*  being edited, it is added to the list.  Otherwise, the list    */
        /*  is updated with the new entry information.                     */
        /*-----------------------------------------------------------------*/
        if (NoEdit)
          AddEntry(ndx,FALSE);
        else
          EditEntry(ndx);
        break;
      case INFO_ADD :
        AddEntry(ndx,FALSE);
        break;
      case INFO_EXPAND :
        DisplayEntry();
        break;
      case INFO_DELETE :
        /*-----------------------------------------------------------------*/
        /*  Delete the current list item and redraw the list to reflect    */
        /*  the change.                                                    */
        /*-----------------------------------------------------------------*/
        ListAltered = TRUE;
        DeleteListItem(Panel,INFO_LIST,ndx,1);
        RefreshListCtrl(Panel,INFO_LIST);
        break;
      case INFO_EDIT :
        EditEntry(ndx);
        break;
      case INFO_CANCEL :
        NoEdit = TRUE;
        ClearFields();
        break;
      case INFO_CLEAR :
        ClearFields();
        break;
      case INFO_QUIT :
        /*-----------------------------------------------------------------*/
        /*  If the information log was altered and the user wants to save  */
        /*  it, save the panel state to file.                              */
        /*-----------------------------------------------------------------*/
        if (ListAltered && (ConfirmPopup("Log was modified. Do you want to save it?") == 1))
          SavePanelState (Panel, Pathname);
        exit(0);
        break;
    }
  }
}

/*= ADD ENTRY =============================================================*/
/*                                                                         */
/*  Adds the current entry to the test log list.                           */
/*=========================================================================*/

int AddEntry (ndx, replace)

int ndx;                                         /* list index             */
int replace;                                     /* replace flag           */

{
  int result;

  /*-----------------------------------------------------------------------*/
  /*  Query the panel for the value of the product name.  If the name is   */
  /*  empty, display an error message, make the product name control       */
  /*  active, and return with an error.                                    */
  /*-----------------------------------------------------------------------*/
  GetCtrlVal(Panel,INFO_PRODUCT,ProdName);
  if (StringLength(ProdName) == 0) {
    MessagePopup("Entry requires at least a product name");
    SetActiveCtrl(INFO_PRODUCT);
    return(-1);
  }

  /*-----------------------------------------------------------------------*/
  /*  Query the panel for the product name, serial number, test name,      */
  /*  test date, test time, and test operator.                             */
  /*-----------------------------------------------------------------------*/
  GetCtrlVal(Panel,INFO_SERNO,SerialNo);
  GetCtrlVal(Panel,INFO_TEST_NAME,TestName);
  GetCtrlVal(Panel,INFO_TEST_DATE,TestDate);
  GetCtrlVal(Panel,INFO_TEST_TIME,TestTime);
  GetCtrlVal(Panel,INFO_OPERATOR,Operator);

  /*-----------------------------------------------------------------------*/
  /*  Format all of the information into a single string which will be     */
  /*  added to the test log list.                                          */
  /*-----------------------------------------------------------------------*/
  Fmt(Entry,"%s[w23]< %s",ProdName);
  Fmt(Entry,"%s[w18a]<%s",SerialNo);
  Fmt(Entry,"%s[w51a]<%s",TestName);
  Fmt(Entry,"%s[w26a]<%s",TestDate);
  Fmt(Entry,"%s[w18a]<%s",TestTime);
  Fmt(Entry,"%s[w50a]<%s",Operator);

  /*-----------------------------------------------------------------------*/
  /*  If the replace flag is set, we are editing an existing list item so  */
  /*  replace it with the updated entry. Otherwise, add the new entry to   */
  /*  the list.                                                            */
  /*-----------------------------------------------------------------------*/
  if (replace)
    result = ReplaceListItem(Panel,INFO_LIST,ndx,Entry);
  else {
    /*---------------------------------------------------------------------*/
    /*  If the index is negative, the list is empty so change the index so */
    /*  the InsertListItem call will work.  Otherwise, increment the index */
    /*  so the new item will be added after the current item.              */
    /*---------------------------------------------------------------------*/
    if (ndx < 0)
      ndx = 0;
    else
      ndx++;
    result = InsertListItem(Panel,INFO_LIST,ndx,Entry);
  }

  /*-----------------------------------------------------------------------*/
  /*  If the entry was successfully updated or added, set the list altered */
  /*  flag, redraw the list control to reflect the change, and clear the   */
  /*  entry fields.                                                        */
  /*-----------------------------------------------------------------------*/
  if (result == 0) {
    ListAltered = TRUE;
    RefreshListCtrl(Panel,INFO_LIST);
    ClearFields();
  }

  return(result);
}

/*= EDIT ENTRY ============================================================*/
/*                                                                         */
/*  Edits the current entry in the product list.                           */
/*=========================================================================*/

void EditEntry (ndx)

int ndx;

{
  /*-----------------------------------------------------------------------*/
  /*  If we are not currently editing an entry, toggle the flag, display   */
  /*  the entry for editing, and make the last name control active.        */
  /*  Otherwise, we have completed editing an entry so we must replace it  */
  /*  in the product list.  This is done by calling AddEntry with the      */
  /*  replace flag set to TRUE.                                            */
  /*-----------------------------------------------------------------------*/
  if (NoEdit) {
    NoEdit = FALSE;
    DisplayEntry();
    SetActiveCtrl(INFO_PRODUCT);
  }
  else if (AddEntry(ndx,TRUE) == 0)
    NoEdit = TRUE;
}

/*= DISPLAY ENTRY =========================================================*/
/*                                                                         */
/*  Displays the current entry in the product list in expanded form.       */
/*=========================================================================*/

void DisplayEntry (void)

{
  if (GetCtrlVal(Panel,INFO_LIST,Entry) < 0)
    return;

  Scan(Entry," %s[w20t-]  %s[w17t-] %s[w50t-] %s[w25t-] %s[w17t-] %s[w50t-]",
              ProdName, SerialNo, TestName, TestDate, TestTime, Operator);

  SetCtrlVal(Panel,INFO_PRODUCT,ProdName);
  SetCtrlVal(Panel,INFO_SERNO,SerialNo);
  SetCtrlVal(Panel,INFO_TEST_NAME,TestName);
  SetCtrlVal(Panel,INFO_TEST_DATE,TestDate);
  SetCtrlVal(Panel,INFO_TEST_TIME,TestTime);
  SetCtrlVal(Panel,INFO_OPERATOR,Operator);
}

/*= CLEAR FIELDS ==========================================================*/
/*                                                                         */
/*  Clear the product name, serial nnumber, test name, test date, test     */
/*  time, and operator fields and make the product name control active.    */
/*=========================================================================*/

void ClearFields (void)

{
  DefaultCtrl(Panel,INFO_PRODUCT);
  DefaultCtrl(Panel,INFO_SERNO);
  DefaultCtrl(Panel,INFO_TEST_NAME);
  DefaultCtrl(Panel,INFO_TEST_DATE);
  DefaultCtrl(Panel,INFO_TEST_TIME);
  DefaultCtrl(Panel,INFO_OPERATOR);
  SetActiveCtrl(INFO_PRODUCT);
}
