#include "scanxmpl.h"

/****************************************************************************
   This is an example of a LabWindows application that calls the data
   acquisition library functions for programming multiple A/D conversions.
   The following Analog Input defaults are assumed:

        For MIO-16:
            input mode: differential (8 double ended input channels)
            input range: 20 volts
            polarity: bipolar (-10 to +10 volts)
            Gain = 1
            No external conversion pulses or triggering

        For Lab-PC:
            input mode: single ended
            input range: 10 volts
            polarity: bipolar (-5 to +5 volts)
            Gain = 1
            No external conversion pulses or triggering

        For LPM-16:
            input mode: single ended
            input range: 10 volts
            polarity: bipolar (-5 to +5 volts)
            Gain = 1
            No external conversion pulses

****************************************************************************/

#define     NUMCHANS        4       /* the # of channels we're scanning */
#define     PTSPERCHAN      500     /* the number of data points to acquire
                                       per channel */
#define     NUMPOINTS       (NUMCHANS * PTSPERCHAN)
                                    /* total number of points to acquire */
#define     RATE            2000.0  /* sampling rate in points per second */

#define LABPC_BRD_CODE 9
#define LPM16_BRD_CODE 13

int data_buf[NUMPOINTS];            /* buffer for storing acquired data */

main()
{
    int i,                          /* loop counter and array index */
        board,                      /* slot/ID number of board */
        boardType,                  /* board code */
        err,                        /* holds DOS LabDriver error code */
        chans[NUMCHANS],            /* array storing channels */
        gains[NUMCHANS],            /* array storing gains */
        finalScanOrder[NUMCHANS],   /* for Lab_SCAN_Op call */
        quit,                       /* loop control flag */
        hPanel,                     /* user panel handler */
        pan,                        /* panel handler */
        ctrl;                       /* control ID */

    long
        numTimeOutTicks;            /* tick count for Timeout_Config call */
/*---------------------------------------------------------------------*/
    /* open and display panel */
    err = OpenInterfaceManager ();
    hPanel = LoadPanel ("scanxmpl.uir", SP);
    err = DisplayPanel (hPanel);

    /* initialization */
    GetCtrlVal (hPanel, SP_cBoard, &board);
    quit = 0;
    /* fill the channel and gain arrays for MIO-16 */
    for (i=0; i<NUMCHANS; i++)
    {
        chans[i] = i;
        gains[i] = 1;
    }
    /* get board type */
    err = Init_DA_Brds(board, &boardType);
    SetCtrlVal (hPanel, SP_iError, err);
    /* calculate and set a timeout limit - use a minimum of approx 1 second */
    numTimeOutTicks = (NUMPOINTS / RATE) * 20;
    if (numTimeOutTicks < 20L) numTimeOutTicks = 20L;
    err = Timeout_Config (board, numTimeOutTicks);
    while (!quit)
    {
        err = GetUserEvent (0, &pan, &ctrl);
        switch (ctrl)
        {
            case SP_cBoard :
            /* user changes the board number */
            GetCtrlVal(hPanel, SP_cBoard, &board);
            err = Init_DA_Brds(board, &boardType);
            SetCtrlVal (hPanel, SP_iError, err);
            err = Timeout_Config (board, numTimeOutTicks);
            break;

            case SP_cQuit :
            /* user presses the quit button */
            quit = 1;
            break;

            case SP_cGo :
            /* user presses the Go butten */
            /* Check board type. Use the appropriate call. */
            if ((boardType == LPM16_BRD_CODE) || (boardType == LABPC_BRD_CODE))
            {
                err = Lab_SCAN_Op (board, NUMCHANS, 1, data_buf, NUMPOINTS, RATE,
                                   finalScanOrder);
            }
            else
            {
                err = SCAN_Op (board, NUMCHANS, chans, gains, data_buf, NUMPOINTS,
                               RATE, 0.0);
            }
            SetCtrlVal (hPanel, SP_iError, err);
            if (!err)
            {
                /* erase previous plot */
                DeletePlots (hPanel, SP_iStrip);
                /* plot new graph */
                PlotStripChart (hPanel, SP_iStrip, data_buf, NUMPOINTS, 0, 0, 1);
            }
            break;

            default :
            break;
        }
    }
    /* done with program */
    err = Init_DA_Brds (board, &boardType);
    err = CloseInterfaceManager ();
}
