
/*****************************************************************************
    This sample program interactively generates and graphs waveforms.
    It uses the LabWindows User Interface Library to create an instrument
    panel which the user operates.
*****************************************************************************/

#include "sample7.h"

#define FALSE               0
#define TRUE                1
#define WFM_BUFFER_SIZE    1000

void Generate_and_Plot_Waveform (void);
void Generate_Ramp (int size, double min, double max, double wfm_buffer[]);
void Generate_Triangle (int size, double min, double max, int num_cycles, double wfm_buffer[]);
void Generate_Square (int size, double min, double max, int num_cycles, double wfm_buffer[]);
void Generate_SineWave (int size, double min, double max, int num_cycles, double wfm_buffer[]);
void Generate_Noise (int size, double min, double max, double wfm_buffer[]);
void generate_tone (double freq, double duration);

double wfm[WFM_BUFFER_SIZE];
int p;

main ()
{
  int panel, ctrl, power_switch;

  p = LoadPanel ("sample7.uir", P1);
  DisplayPanel (p);

/* Main event loop */
  while (TRUE)  {
    GetUserEvent (1, &panel, &ctrl);
    switch (ctrl)  {
      case P1_POWER :
          generate_tone (1.0, .0005);
          GetCtrlVal (p, P1_POWER, &power_switch);
          SetCtrlVal (p, P1_LED, power_switch);
          SetInputMode (p, P1_WAVEFORM_TYPE, power_switch);
          SetInputMode (p, P1_CURVE_COLOR, power_switch);
          SetInputMode (p, P1_MIN_VAL, power_switch);
          SetInputMode (p, P1_MAX_VAL, power_switch);
          SetInputMode (p, P1_NUM_PTS, power_switch);
          SetInputMode (p, P1_NUM_CYCLES, power_switch);
          if (!power_switch)
            return;
          break;
    }
    Generate_and_Plot_Waveform ();
  }
}

void Generate_and_Plot_Waveform ()
{
  double min, max;
  int num_pts, num_cycles, wfm_type, curv_color;

  GetCtrlVal (p, P1_WAVEFORM_TYPE, &wfm_type);
  GetCtrlVal (p, P1_NUM_CYCLES, &num_cycles);
  GetCtrlVal (p, P1_MIN_VAL, &min);
  GetCtrlVal (p, P1_MAX_VAL, &max);
  GetCtrlVal (p, P1_NUM_PTS, &num_pts);
  GetCtrlVal (p, P1_CURVE_COLOR, &curv_color);
  switch (wfm_type)  {
    case 0 :
      SetInputMode (p, P1_NUM_CYCLES, FALSE);
      Generate_Ramp (num_pts, min, max, wfm);
      break;
    case 1 :
      SetInputMode (p, P1_NUM_CYCLES, TRUE);
      Generate_Triangle (num_pts, min, max, num_cycles, wfm);
      break;
    case 2:
      SetInputMode (p, P1_NUM_CYCLES, TRUE);
      Generate_Square (num_pts, min, max, num_cycles, wfm);
      break;
    case 3 :
      SetInputMode (p, P1_NUM_CYCLES, TRUE);
      Generate_SineWave (num_pts, min, max, num_cycles, wfm);
      break;
    case 4:
      SetInputMode (p, P1_NUM_CYCLES, FALSE);
      Generate_Noise (num_pts, min, max, wfm);
      break;
  }
  DeletePlots (p, P1_GRAPH);
  PlotY (p, P1_GRAPH, wfm, num_pts, 4, 0, 0, 1, curv_color);
}


/*****************************************************************************
    The following support functions are used to generate the desired
    waveform data.
*****************************************************************************/

void Generate_Ramp (int size, double min, double max, double wfm_buffer[])
{
  int i;
  double increment;

  wfm_buffer[0] = min;
  if (size <= 1)
    return;
  wfm_buffer[size - 1] = max;
  if (size == 2)
    return;
  increment = (max - min) / (size - 1);
  for (i = 1; i < (size - 1); i++)  {
    wfm_buffer[i] = (wfm_buffer[i - 1] + increment);
  }
}

void Generate_Triangle (int size, double min, double max, int num_cycles, double wfm_buffer[])
{
  int i;
  double increment, point;

  wfm_buffer[0] = min;
  if (size <= 1)
    return;
  increment = ((double)num_cycles / (double)(size - 1));
  wfm_buffer[size - 1] = min;
  if (size == 2)
    return;
  point = increment;
  for (i = 1; i < (size - 1); i++)  {
    while (point > 1.0)
      point -= 1.0;
    if (point <= 0.5)
      wfm_buffer[i] = min + (point * (max - min) / 0.5);
    else
      wfm_buffer[i] = min + ((1 - point) * (max - min) / 0.5);
    point += increment;
  }
}

void Generate_Square (int size, double min, double max, int num_cycles, double wfm_buffer[])
{
  int i;
  double increment, point;

  wfm_buffer[0] = min;
  if (size <= 1)
    return;
  increment = ((double)num_cycles / (double)(size - 1));
  wfm_buffer[size - 1] = min;
  if (size == 2)
    return;
  point = increment;
  for (i = 1; i < (size - 1); i++)  {
    while (point > 1.0)
      point -= 1.0;
    if (point <= 0.5)
      wfm_buffer[i] = min;
    else
      wfm_buffer[i] = max;
    point += increment;
  }
}

void Generate_SineWave (int size, double min, double max, int num_cycles, double wfm_buffer[])
{
  int i;
  double increment, point;

  wfm_buffer[0] = min + ((max - min) / 2);
  if (size <= 1)
    return;
  increment = (double)num_cycles / (double)(size - 1);
  wfm_buffer[size - 1] = wfm_buffer[0];
  if (size == 2)
    return;
  point = increment;
  for (i = 1; i < (size - 1); i++)  {
    wfm_buffer[i] = wfm_buffer[0] + (fabs (max - min) / 2 * sin (2 * point * 3.1415926));
    point += increment;
  }
}

void Generate_Noise (int size, double min, double max, double wfm_buffer[])
{
  int i;

  for (i = 0; i < size; i++)
    wfm_buffer[i] = (min + ((max - min) * (double)rand () / 32767.0));
}


/*****************************************************************************
    generate_tone  is used to create the 'switch' sound when the Power
    control switch is toggled.
*****************************************************************************/
void generate_tone (freq, duration)
double freq, duration;
{
  int low, high, port;
  long count;

  if (freq <= 0.0)
    return;
  count = (long)(1193280.0 / freq);
  low   = count % 256;
  high  = count / 256;
  outp (67,182);
  outp (66,low);
  outp (66,high);
  port = inp (97) | 3;
  outp (97,port);
  delay (duration);
  port = inp (97) & 0xFC;
  outp (97,port);
}
