#include "freqxmpl.h"

/***************************************************************************/
/*  This is an example of a LabWindows application that calls data         */
/*  acquisition library functions to make frequency and duty cycle         */
/*  measurements with a MIO-16, MIO-16F-5, or TIO-10.                      */
/*                                                                         */
/*  The signal to be measured should be connected to GATE1 on the I/O      */
/*  connector.  This is pin 42 on the MIO-16 & MIO-16F-5, and pin 2 on the */
/*  TIO-10.  The signal ground should be connected to pin 33 on the MIO-16,*/
/*  MIO-16F-5, or TIO-10.                                                  */
/*                                                                         */
/*  When the program is running, the frequency and duty cycle measured will*/
/*  be displayed continuously if good measurements are made.  In case of   */
/*  bad measurement due to improper choice of timebase, no value will be   */
/*  displayed.                                                             */
/***************************************************************************/

#define     PERIOD_CTR              1 /* counter used for period measurement*/
#define     PULSE_CTR               2 /* counter used for pulse width meas. */
#define     TOTAL_NUMBER_OF_CTR     2 /* num of counters used */

#define     RISING_EDGE_SOURCE      0 /*code for rising edge source counting*/
#define     NO_GATE                 0 /* code for no gating */
#define     RISING_EDGE_GATE        3 /* code for rising edge gate counting */
#define     H_LVL_GATE_MINUS_1      7 /* high level gate count of counter-1 */
#define     TC_TOGGLE               0 /* toggle OUT line on terminal count */
#define     TC_PULSE                1 /* pulse OUT line on terminal count */
#define     POSITIVE_POLARITY       0 /* active is high */

/* CTR_Simul_Op operating modes */
#define     OP_CANCEL               0 
#define     OP_RESERVE              1
#define     OP_START                2
#define     OP_STOP                 3

/* TOLERANCE is lower limit of a good reading.  Actually, a reading less
   than this number can still be a valid reading.  However, note that as
   the reading gets smaller, the accuracy decreases.  Always try to use
   the fastest timebase and get the largest reading possible for the best
   accuracy. */
#define     TOLERANCE               15
/***************************************************************************/
main()
{
        int brdCode;                        /* board code of selected board */
        int ctr_list[TOTAL_NUMBER_OF_CTR];  /* array of counters used by CTR_Simul_Op */
        int period_count;                   /* period counter value */
        int pulse_count;                    /* pulse width counter value */
        int overflow;                       /* overflow flag */
        int err;                            /* error code */
        int hPanel;                         /* user panel handler */
        int boardType;                      /* board code */
        int timebase;                       /* counter timebase */
        int board;                          /* slot number of the board */
        int quit_hit;                       /* loop control flag */
        int ctrl;                           /* control ID */
        int pan;                            /* panel handler */
        int status;                         /* counter status */
        double factor;                      /* use for period-to-frequency conversion */
        long lfreq;                         /* frequency measured */
        long lduty;                         /* duty cycle measured */
        long lperiod_count;                 /* period counter value */
        long lpulse_count;                  /* pulse width counter value */
        char freq_buffer[10];               /* for printing frequency */
        char duty_buffer[10];               /* for printing duty cycle */
/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
    /* tell user to make the connections */
    cls ();
    FmtOut("For a MIO-16 or AT-MIO-16F-5, connect:\n") ;
    FmtOut("    Signal to GATE1 (pin 42)\n") ;
    FmtOut("    Signal Ground to GND (pin 33)\n") ;
    FmtOut("\n") ;
    FmtOut("For a PC-TIO-10, connect:\n") ;
    FmtOut("    Signal to GATE1 (pin 2)\n") ;
    FmtOut("    Signal Ground to GND (pin 33)\n") ;
    FmtOut("\n") ;
    FmtOut("Press a key when ready...") ;
    getkey ();

    /* open and display user panel */
    err = OpenInterfaceManager ();
    hPanel = LoadPanel ("freqxmpl.uir", P0);
    err = DisplayPanel (hPanel);

    /* building up array of counter for CTR_SIMUL_OP */
    ctr_list[0] = PERIOD_CTR;
    ctr_list[1] = PULSE_CTR;

    /* initialization */
    board = 1;
    err = Init_DA_Brds(board, &boardType);
    SetCtrlVal (hPanel, P0_I_error, err);
    quit_hit = 0;
    status = 0;
    timebase = 1;
    factor = pow(10.0, 7.0 - timebase);
    while (!quit_hit)
    {
        err = GetUserEvent (0, &pan, &ctrl);
        switch (ctrl)
        {
            case P0_C_board :
            /* user changes board number */
            /* stop all counters */
            err = CTR_Simul_Op (board, TOTAL_NUMBER_OF_CTR, ctr_list, OP_STOP);

            /* get new board number nad initialize the board */
            GetCtrlVal(hPanel, P0_C_board, &board) ;

            err = Init_DA_Brds (board, &boardType);
            SetCtrlVal (hPanel, P0_I_error, err);
            status = 0;
            break;

            case P0_C_timebase :
            /* user changes timebase */
            GetCtrlVal(hPanel, P0_C_timebase, &timebase) ;
            factor = pow(10.0, 7.0 - timebase);
            /* re-program counters if they are busy */
            if (status)
            {
                err = CTR_Period (board, PERIOD_CTR, timebase);
                err = CTR_Period (board, PULSE_CTR, timebase);
                err = CTR_Simul_Op (board, TOTAL_NUMBER_OF_CTR, ctr_list, OP_START);
            }
            break;

            case P0_C_go :
            /* user presses the Go button */
            err = CTR_Simul_Op (board, TOTAL_NUMBER_OF_CTR, ctr_list, OP_RESERVE);
            SetCtrlVal (hPanel, P0_I_error, err);
            if (err >= 0)
            {
                /* config to count edges of counter */
                err = CTR_Config (board, PERIOD_CTR, RISING_EDGE_SOURCE, RISING_EDGE_GATE,
                                  TC_PULSE, POSITIVE_POLARITY);
                /* config to count level of counter-1 */
                err = CTR_Config (board, PULSE_CTR, RISING_EDGE_SOURCE, H_LVL_GATE_MINUS_1,
                                  TC_PULSE, POSITIVE_POLARITY);
                err = CTR_Period (board, PERIOD_CTR, timebase);
                err = CTR_Period (board, PULSE_CTR, timebase);
                err = CTR_Simul_Op (board, TOTAL_NUMBER_OF_CTR, ctr_list, OP_START);
                status = 1;
            }
            break;

            case P0_C_quit:
            /* user presses the Quit button */
            quit_hit = 1;
            break;

            default:
            if (status)
            {
                /* read counters if they are busy */
                err = CTR_EvRead (board, PERIOD_CTR, &overflow, &period_count);
                err = CTR_EvRead (board, PULSE_CTR, &overflow, &pulse_count);

                /* convert int values read to long */
                if (period_count >= 0) lperiod_count = period_count;
                else lperiod_count = ((long) period_count) + 65536;

                if (pulse_count >= 0) lpulse_count = pulse_count;
                else lpulse_count = ((long) pulse_count) + 65536;

                /* calculate frequency and duty cycle */
                if (lperiod_count > TOLERANCE)
                {
                    /* good reading */
                    lfreq = factor / lperiod_count;
                    lduty = (lpulse_count * 100.0 / lperiod_count);
                    Fmt (freq_buffer, "%d[b4]", lfreq);
                    Fmt (duty_buffer, "%d[b4]", lduty);
                }
                else
                {
                    /* bad reading */
                    Fmt (freq_buffer, "-----");
                    Fmt (duty_buffer, "--");
                }

                /* update user panel */
                SetCtrlVal (hPanel, P0_I_freq, freq_buffer);
                SetCtrlVal (hPanel, P0_I_duty, duty_buffer);
            }
            break;
        }
    }
    err = Init_DA_Brds (board, &boardType);
    err = CloseInterfaceManager ();
}
