/*
    FILTERS.C
    This program interactively computes and plots frequency response
    curves for the various LabWindows Advanced Analysis Library filter
    functions.  It is also capable of calculating the Fourier transforms
    for each of the window types.

    Note:   This program is intended for use only within the LabWindows
            interactive program.  Because this filters program requires
            a large amount of memory, you might not be able to run it
            as a standalone executable.
*/

#include "filters.h"

#define FALSE   0
#define TRUE    1
#define OFF     0
#define ON      1

#define SIZE 1024
double x[SIZE], y[SIZE], z[SIZE], coef[SIZE / 2];
double axis[SIZE];
double win[SIZE];

#define MAX_NUM_PLOTS   10
double plot0[SIZE], plot1[SIZE], plot2[SIZE], plot3[SIZE], plot4[SIZE];
double plot5[SIZE], plot6[SIZE], plot7[SIZE], plot8[SIZE], plot9[SIZE];

double plot0xl[2], plot0yl[2], plot1xl[2], plot1yl[2];
double plot2xl[2], plot2yl[2], plot3xl[2], plot3yl[2];
double plot4xl[2], plot4yl[2], plot5xl[2], plot5yl[2];
double plot6xl[2], plot6yl[2], plot7xl[2], plot7yl[2];
double plot8xl[2], plot8yl[2], plot9xl[2], plot9yl[2];

double plot0xu[2], plot0yu[2], plot1xu[2], plot1yu[2];
double plot2xu[2], plot2yu[2], plot3xu[2], plot3yu[2];
double plot4xu[2], plot4yu[2], plot5xu[2], plot5yu[2];
double plot6xu[2], plot6yu[2], plot7xu[2], plot7yu[2];
double plot8xu[2], plot8yu[2], plot9xu[2], plot9yu[2];

double fs;
int n;
int p, pp, ppp;
int order;
int num_coefs;
int fl_is_valid, fu_is_valid;
double fl, fu;
double beta;
int error;
char err_buf[100];
int curv_color;
int num_plots;

#define MAX_NUM_BANDS   20
int num_bands;
double A[MAX_NUM_BANDS], cutoffs[MAX_NUM_BANDS * 2];

int uil_err;
#define UIM_OPEN_ERR    -1
#define FONT_LOAD_ERR   -2
#define NO_MEM_ERR      -12
#define FILE_OPEN_ERR   -19

#define LCF_COLOR   12
#define UCF_COLOR   14

#define BW_LP        0
#define BW_HP        1
#define BW_BP        2
#define BW_BS        3
#define CH_LP        4
#define CH_HP        5
#define CH_BP        6
#define CH_BS        7
#define INVCH_LP     8
#define INVCH_HP     9
#define INVCH_BP     10
#define INVCH_BS     11
#define ELP_LP       12
#define ELP_HP       13
#define ELP_BP       14
#define ELP_BS       15
#define KSR_LP       16
#define KSR_HP       17
#define KSR_BP       18
#define KSR_BS       19
#define WIND_LP      20
#define WIND_HP      21
#define WIND_BP      22
#define WIND_BS      23
#define EQUIRPL_LP   24
#define EQUIRPL_HP   25
#define EQUIRPL_BP   26
#define EQUIRPL_BS   27
#define EQUIRIPPLE   28
#define DIVIDER      100
#define RECT_WIN     101
#define TRI_WIN      102
#define HAN_WIN      103
#define HAM_WIN      104
#define BKMAN_WIN    105
#define KSR_WIN      106

void Generate_and_Plot_Filter (void);
void setup_input_modes (int filter, int filter_type);
void plot_fl_and_fu (void);
void generate_tone (double freq, double duration);
void dsp_error (void);
void Get_Equi_Ripple_Data (void);
void report_uil_error (int, int);

main () {
    int panel, ctrl, power_switch, filter, filter_type;

    error = FALSE;
    n = SIZE;
    num_bands = 0;
    num_plots = 0;
    p = LoadPanel ("filters.uir", P1);
    if (p < 0)
        report_uil_error (p, TRUE);
    pp = LoadPanel ("filters.uir", P2);
    if (pp < 0)
        report_uil_error (pp, TRUE);
    ppp = LoadPanel ("filters.uir", P3);
    if (ppp < 0)
        report_uil_error (ppp, TRUE);
    DisplayPanel (p);
    GetCtrlVal (p, P1_FILTER, &filter);
    GetCtrlVal (p, P1_FILTER_TYPE, &filter_type);
    GetCtrlVal (p, P1_CURV_COLOR, &curv_color);
    setup_input_modes (filter, filter_type);

    /* Main event loop */
    while (TRUE)  {
        GetUserEvent (1, &panel, &ctrl);
        switch (ctrl)  {

          case P1_QUIT:
            return;
            break;

          case P1_FILTER:
            GetCtrlVal (p, P1_FILTER, &filter);
            GetCtrlVal (p, P1_FILTER_TYPE, &filter_type);
            setup_input_modes (filter, filter_type);
            break;

          case P1_FILTER_TYPE:
            GetCtrlVal (p, P1_FILTER, &filter);
            GetCtrlVal (p, P1_FILTER_TYPE, &filter_type);
            setup_input_modes (filter, filter_type);
            break;

          case P1_DEFBANDS:
            Get_Equi_Ripple_Data ();
            break;

          case P1_PLOT:
            if (num_plots >= MAX_NUM_PLOTS)  {
                Fmt (err_buf, "Unable to plot more than %d Response Curves.", MAX_NUM_PLOTS);
                MessagePopup (err_buf);
            }
            else  {
                Generate_and_Plot_Filter ();
                if (num_plots != -2)  {
                    SetInputMode (p, P1_FS, OFF);
                    num_plots++;
                }
            }
            break;

          case P1_CLEAR:
            SetInputMode (p, P1_FS, ON);
            DeletePlots (p, P1_GRAPH);
            num_plots = 0;
            break;

          case P1_CURV_COLOR:
            GetCtrlVal (p, P1_CURV_COLOR, &curv_color);
            break;
        }
    }
}

void Generate_and_Plot_Filter ()
{
    int i;
    int filter, filter_type, window_type;
    double ripple, atten, delta;
    double wts[MAX_NUM_BANDS];
    double min, max;
    int imin, imax;
    double x_array[2], y_array[2];
    double axis_scale;
    double y0;

    InstallPopup (ppp);
    GetCtrlVal (p, P1_FILTER, &filter);
    GetCtrlVal (p, P1_FILTER_TYPE, &filter_type);
    GetCtrlVal (p, P1_ORDER, &order);
    GetCtrlVal (p, P1_BETA, &beta);
    GetCtrlVal (p, P1_COEFS, &num_coefs);
    GetCtrlVal (p, P1_WINDOW, &window_type);
    GetCtrlVal (p, P1_LCF, &fl);
    GetCtrlVal (p, P1_UCF, &fu);
    GetCtrlVal (p, P1_FS, &fs);
    GetCtrlVal (p, P1_RIPPLE, &ripple);
    GetCtrlVal (p, P1_ATTEN, &atten);
    for (i = 0; i < n; i++)  {
        x[i] = 0.0;
        y[i] = 0.0;
        z[i] = 0.0;
        win[i] = 1.0;
    }
    x[0] = 1.0;
    for (i = 0; i < MAX_NUM_BANDS; i++)
        wts[i] = 1.0;
    if (filter < EQUIRIPPLE)
        filter += filter_type;
    switch (filter)  {
        case BW_LP:
            Bw_LPF (x, n, fs, fl, order, y);
            break;
        case BW_HP:
            Bw_HPF (x, n, fs, fu, order, y);
            break;
        case BW_BP:
            Bw_BPF (x, n, fs, fl, fu, order, y);
            break;
        case BW_BS:
            Bw_BSF (x, n, fs, fl, fu, order, y);
            break;
        case CH_LP:
            Ch_LPF (x, n, fs, fl, ripple, order, y);
            break;
        case CH_HP:
            Ch_HPF (x, n, fs, fu, ripple, order, y);
            break;
        case CH_BP:
            Ch_BPF (x, n, fs, fl, fu, ripple, order, y);
            break;
        case CH_BS:
            Ch_BSF (x, n, fs, fl, fu, ripple, order, y);
            break;
        case INVCH_LP:
            InvCh_LPF (x, n, fs, fl, atten, order, y);
            break;
        case INVCH_HP:
            InvCh_HPF (x, n, fs, fu, atten, order, y);
            break;
        case INVCH_BP:
            InvCh_BPF (x, n, fs, fl, fu, atten, order, y);
            break;
        case INVCH_BS:
            InvCh_BSF (x, n, fs, fl, fu, atten, order, y);
            break;
        case ELP_LP:
            Elp_LPF (x, n, fs, fl, ripple, atten, order, y);
            break;
        case ELP_HP:
            Elp_HPF (x, n, fs, fu, ripple, atten, order, y);
            break;
        case ELP_BP:
            Elp_BPF (x, n, fs, fl, fu, ripple, atten, order, y);
            break;
        case ELP_BS:
            Elp_BSF (x, n, fs, fl, fu, ripple, atten, order, y);
            break;
        case KSR_LP:
            Ksr_LPF (fs, fl, num_coefs, coef, beta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case KSR_HP:
            Ksr_HPF (fs, fu, num_coefs, coef, beta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs+ 1), y);
            break;
        case KSR_BP:
            Ksr_BPF (fs, fl, fu, num_coefs, coef, beta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case KSR_BS:
            Ksr_BSF (fs, fl, fu, num_coefs, coef, beta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case WIND_LP:
            Wind_LPF (fs, fl, num_coefs, coef, window_type);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case WIND_HP:
            Wind_HPF (fs, fu, num_coefs, coef, window_type);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case WIND_BP:
            Wind_BPF (fs, fl, fu, num_coefs, coef, window_type);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case WIND_BS:
            Wind_BSF (fs, fl, fu, num_coefs, coef, window_type);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case EQUIRPL_LP:
            EquiRpl_LPF (fs, fl, fu, num_coefs, coef, &delta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case EQUIRPL_HP:
            EquiRpl_HPF (fs, fl, fu, num_coefs, coef, &delta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case EQUIRPL_BP:
            EquiRpl_BPF (fs, cutoffs[1], cutoffs[2], cutoffs[3], cutoffs[4], num_coefs, coef, &delta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case EQUIRPL_BS:
            EquiRpl_BSF (fs, cutoffs[1], cutoffs[2], cutoffs[3], cutoffs[4], num_coefs, coef, &delta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case EQUIRIPPLE:
            Equi_Ripple (num_bands, A, wts, fs, cutoffs, window_type, num_coefs, coef, &delta);
            if (dsperr)
                dsp_error ();
            else
                Cxy (coef, num_coefs, x, (n - num_coefs + 1), y);
            break;
        case DIVIDER:
            error = TRUE;
            break;
        case RECT_WIN:
            Copy1D (win, order, y);
            break;
        case TRI_WIN:
            TriWin (win, order);
            if (dsperr)
                dsp_error ();
            else
                Copy1D (win, order, y);
            break;
        case HAN_WIN:
            HanWin (win, order);
            if (dsperr)
                dsp_error ();
            else
                Copy1D (win, order, y);
            break;
        case HAM_WIN:
            HamWin (win, order);
            if (dsperr)
                dsp_error ();
            else
                Copy1D (win, order, y);
            break;
        case BKMAN_WIN:
            BkmanWin (win, order);
            if (dsperr)
                dsp_error ();
            else
                Copy1D (win, order, y);
            break;
        case KSR_WIN:
            KsrWin (win, order, beta);
            if (dsperr)
                dsp_error ();
            else
                Copy1D (win, order, y);
            break;
    }
    if (error || dsperr)  {
        if ((num_plots == -2) || (filter >= EQUIRPL_BP))  {
            DeletePlots (p, P1_GRAPH);
            if (num_plots == -2)  {
                if (filter < EQUIRPL_BP)
                    num_plots = 0;
            }
            else  {
                SetInputMode (p, P1_FS, ON);
                num_plots = -2;
            }
        }
        if (!error && dsperr)
            dsp_error ();
        if (num_plots == 0)
            num_plots = -2;
        if (num_plots != -2)
            num_plots--;
        error = FALSE;
        return;
    }
    FFT (y, z, n);
    for (i = 0; i < n; i++)  {
        y[i] = sqrt (y[i] * y[i] + z[i] * z[i]);
    }
    if (filter > DIVIDER)  {
        y0 = y[0];
        for (i = 0; i < n; i++)
            y[i] /= y0;
    }
    Clip (y, n, 10.0, 1.0e-10, y);
    if ((filter == EQUIRIPPLE) && (window_type != 1))
        ;
    else  {
        for (i=0; i < n; i++)
            y[i] = 20 * log10 (y[i]);
    }
    axis_scale = (fs / SIZE);
    for (i = 0; i < SIZE; i++)
        axis[i] = i * axis_scale;
    RemovePopup (0);
    if ((num_plots == -2) || (filter >= EQUIRPL_BP))  {
        DeletePlots (p, P1_GRAPH);
        if (num_plots == -2)  {
            if (filter < EQUIRPL_BP)
                num_plots = 0;
        }
        else  {
            SetInputMode (p, P1_FS, ON);
            num_plots = -2;
        }
    }
    switch (num_plots)  {
        case 0:
        case -2:
            Copy1D (y, SIZE, plot0);
            if (filter > DIVIDER)
                ConfigureAxes (p, P1_GRAPH, 0, 0.0, fs, -1, 0.0, 0.0);
            else
                ConfigureAxes (p, P1_GRAPH, 0, 0.0, (fs / 2), -1, 0.0, 0.0);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot0, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 1:
            Copy1D (y, SIZE, plot1);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot1, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 2:
            Copy1D (y, SIZE, plot2);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot2, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 3:
            Copy1D (y, SIZE, plot3);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot3, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 4:
            Copy1D (y, SIZE, plot4);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot4, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 5:
            Copy1D (y, SIZE, plot5);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot5, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 6:
            Copy1D (y, SIZE, plot6);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot6, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 7:
            Copy1D (y, SIZE, plot7);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot7, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 8:
            Copy1D (y, SIZE, plot8);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot8, n, 4, 4, 0, 0, 1, curv_color);
            break;
        case 9:
            Copy1D (y, SIZE, plot9);
            uil_err = PlotXY (p, P1_GRAPH, axis, plot9, n, 4, 4, 0, 0, 1, curv_color);
            break;
    }
    if (uil_err < 0)  {
        report_uil_error (uil_err, FALSE);
        return;
    }
    if ((filter == EQUIRPL_BP) || (filter == EQUIRPL_BS)
     || (filter == EQUIRIPPLE))  {
        MaxMin1D (y, n, &max, &imax, &min, &imin);
        for (i = 0; i < num_bands; i++)  {
            x_array[0] = cutoffs[2 * i];
            x_array[1] = cutoffs[2 * i];
            y_array[0] = min;
            y_array[1] = max;
            PlotXY (p, P1_GRAPH, x_array, y_array, 2, 4, 4, 0, 0, 1, LCF_COLOR);
            x_array[0] = cutoffs[(2 * i) + 1];
            x_array[1] = cutoffs[(2 * i) + 1];
            PlotXY (p, P1_GRAPH, x_array, y_array, 2, 4, 4, 0, 0, 1, UCF_COLOR);
        }
    }
    else
        plot_fl_and_fu ();
}

void setup_input_modes (filter, filter_type)
int filter;
int filter_type;
{
    SetInputMode (p, P1_FILTER_TYPE, ON);
    SetInputMode (p, P1_ORDER, ON);
    SetInputMode (p, P1_BETA, OFF);
    SetInputMode (p, P1_COEFS, OFF);
    SetInputMode (p, P1_WINDOW, OFF);
    SetInputMode (p, P1_LCF, ON);
    SetInputMode (p, P1_UCF, ON);
    SetInputMode (p, P1_RIPPLE, ON);
    SetInputMode (p, P1_ATTEN, ON);
    SetInputMode (p, P1_PLOT, ON);
    SetInputMode (p, P1_DEFBANDS, OFF);
    fl_is_valid = TRUE;
    fu_is_valid = TRUE;
    if (filter < EQUIRIPPLE)
        filter += filter_type;
    if ((num_plots == 0) || (num_plots == -2))
        SetInputMode (p, P1_FS, ON);
    else
        SetInputMode (p, P1_FS, OFF);
    switch (filter)  {
        case BW_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case BW_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case BW_BP:
        case BW_BS:
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case CH_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case CH_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case CH_BP:
        case CH_BS:
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case INVCH_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            break;
        case INVCH_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            break;
        case INVCH_BP:
        case INVCH_BS:
            SetInputMode (p, P1_RIPPLE, OFF);
            break;
        case ELP_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_UCF, OFF);
            break;
        case ELP_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_LCF, OFF);
            break;
        case ELP_BP:
        case ELP_BS:
            break;
        case KSR_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_BETA, ON);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case KSR_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_BETA, ON);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case KSR_BP:
        case KSR_BS:
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_BETA, ON);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case WIND_LP:
            fu_is_valid = FALSE;
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_WINDOW, ON);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case WIND_HP:
            fl_is_valid = FALSE;
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_WINDOW, ON);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case WIND_BP:
        case WIND_BS:
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_WINDOW, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case EQUIRPL_LP:
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case EQUIRPL_HP:
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case EQUIRPL_BP:
        case EQUIRPL_BS:
            fl_is_valid = FALSE;
            fu_is_valid = FALSE;
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_FS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            SetInputMode (p, P1_DEFBANDS, ON);
            break;
        case EQUIRIPPLE:
            fl_is_valid = FALSE;
            fu_is_valid = FALSE;
            SetInputMode (p, P1_FILTER_TYPE, OFF);
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_COEFS, ON);
            SetInputMode (p, P1_WINDOW, ON);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_FS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            SetInputMode (p, P1_DEFBANDS, ON);
            break;
        case DIVIDER:
            fl_is_valid = FALSE;
            fu_is_valid = FALSE;
            SetInputMode (p, P1_FILTER_TYPE, OFF);
            SetInputMode (p, P1_ORDER, OFF);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_FS, OFF);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            SetInputMode (p, P1_PLOT, OFF);
            break;
        case RECT_WIN:
        case TRI_WIN:
        case HAN_WIN:
        case HAM_WIN:
        case BKMAN_WIN:
            fl_is_valid = FALSE;
            fu_is_valid = FALSE;
            SetInputMode (p, P1_FILTER_TYPE, OFF);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_FS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
        case KSR_WIN:
            fl_is_valid = FALSE;
            fu_is_valid = FALSE;
            SetInputMode (p, P1_BETA, ON);
            SetInputMode (p, P1_FILTER_TYPE, OFF);
            SetInputMode (p, P1_LCF, OFF);
            SetInputMode (p, P1_UCF, OFF);
            SetInputMode (p, P1_FS, ON);
            SetInputMode (p, P1_RIPPLE, OFF);
            SetInputMode (p, P1_ATTEN, OFF);
            break;
    }
}


void plot_fl_and_fu (void)
{
    double min, max;
    int imin, imax;

    MaxMin1D (y, n, &max, &imax, &min, &imin);
    if (fl_is_valid)  {
        switch (num_plots)  {
            case 0:
            case -2:
                plot0xl[0] = fl;
                plot0xl[1] = fl;
                plot0yl[0] = min;
                plot0yl[1] = max;
                PlotXY (p, P1_GRAPH, plot0xl, plot0yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 1:
                plot1xl[0] = fl;
                plot1xl[1] = fl;
                plot1yl[0] = min;
                plot1yl[1] = max;
                PlotXY (p, P1_GRAPH, plot1xl, plot1yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 2:
                plot2xl[0] = fl;
                plot2xl[1] = fl;
                plot2yl[0] = min;
                plot2yl[1] = max;
                PlotXY (p, P1_GRAPH, plot2xl, plot2yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 3:
                plot3xl[0] = fl;
                plot3xl[1] = fl;
                plot3yl[0] = min;
                plot3yl[1] = max;
                PlotXY (p, P1_GRAPH, plot3xl, plot3yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 4:
                plot4xl[0] = fl;
                plot4xl[1] = fl;
                plot4yl[0] = min;
                plot4yl[1] = max;
                PlotXY (p, P1_GRAPH, plot4xl, plot4yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 5:
                plot5xl[0] = fl;
                plot5xl[1] = fl;
                plot5yl[0] = min;
                plot5yl[1] = max;
                PlotXY (p, P1_GRAPH, plot5xl, plot5yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 6:
                plot6xl[0] = fl;
                plot6xl[1] = fl;
                plot6yl[0] = min;
                plot6yl[1] = max;
                PlotXY (p, P1_GRAPH, plot6xl, plot6yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 7:
                plot7xl[0] = fl;
                plot7xl[1] = fl;
                plot7yl[0] = min;
                plot7yl[1] = max;
                PlotXY (p, P1_GRAPH, plot7xl, plot7yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 8:
                plot8xl[0] = fl;
                plot8xl[1] = fl;
                plot8yl[0] = min;
                plot8yl[1] = max;
                PlotXY (p, P1_GRAPH, plot8xl, plot8yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
            case 9:
                plot9xl[0] = fl;
                plot9xl[1] = fl;
                plot9yl[0] = min;
                plot9yl[1] = max;
                PlotXY (p, P1_GRAPH, plot9xl, plot9yl, 2, 4, 4, 0, 0, 1, LCF_COLOR);
                break;
        }
    }
    if (fu_is_valid)  {
        switch (num_plots)  {
            case 0:
            case -2:
                plot0xu[0] = fu;
                plot0xu[1] = fu;
                plot0yu[0] = min;
                plot0yu[1] = max;
                PlotXY (p, P1_GRAPH, plot0xu, plot0yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 1:
                plot1xu[0] = fu;
                plot1xu[1] = fu;
                plot1yu[0] = min;
                plot1yu[1] = max;
                PlotXY (p, P1_GRAPH, plot1xu, plot1yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 2:
                plot2xu[0] = fu;
                plot2xu[1] = fu;
                plot2yu[0] = min;
                plot2yu[1] = max;
                PlotXY (p, P1_GRAPH, plot2xu, plot2yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 3:
                plot3xu[0] = fu;
                plot3xu[1] = fu;
                plot3yu[0] = min;
                plot3yu[1] = max;
                PlotXY (p, P1_GRAPH, plot3xu, plot3yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 4:
                plot4xu[0] = fu;
                plot4xu[1] = fu;
                plot4yu[0] = min;
                plot4yu[1] = max;
                PlotXY (p, P1_GRAPH, plot4xu, plot4yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 5:
                plot5xu[0] = fu;
                plot5xu[1] = fu;
                plot5yu[0] = min;
                plot5yu[1] = max;
                PlotXY (p, P1_GRAPH, plot5xu, plot5yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 6:
                plot6xu[0] = fu;
                plot6xu[1] = fu;
                plot6yu[0] = min;
                plot6yu[1] = max;
                PlotXY (p, P1_GRAPH, plot6xu, plot6yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 7:
                plot7xu[0] = fu;
                plot7xu[1] = fu;
                plot7yu[0] = min;
                plot7yu[1] = max;
                PlotXY (p, P1_GRAPH, plot7xu, plot7yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 8:
                plot8xu[0] = fu;
                plot8xu[1] = fu;
                plot8yu[0] = min;
                plot8yu[1] = max;
                PlotXY (p, P1_GRAPH, plot8xu, plot8yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
            case 9:
                plot9xu[0] = fu;
                plot9xu[1] = fu;
                plot9yu[0] = min;
                plot9yu[1] = max;
                PlotXY (p, P1_GRAPH, plot9xu, plot9yu, 2, 4, 4, 0, 0, 1, UCF_COLOR);
                break;
        }
    }
}

void dsp_error (void)
{
    char err_buf[20];

    error = TRUE;
    RemovePopup (0);
    Fmt (err_buf, "DSP ERROR:  %d", dsperr);
    MessagePopup (err_buf);
}


void Get_Equi_Ripple_Data (void)
{
    int i, index, ctrl;
    char entry[50];
    char band_buf[10];
    double band;
    double stop_freq, start_freq;

    DefaultPanel (pp);
    InstallPopup (pp);
    SetActiveCtrl (P2_BAND);
    while (TRUE)  {
        GetPopupEvent (1, &ctrl);
        GetListItemIndex (pp, P2_LIST, &index);
        switch (ctrl)  {
            case P2_STOP_FREQ:
            case P2_ADD:
                GetCtrlVal (pp, P2_BAND, &band);
                GetCtrlVal (pp, P2_START_FREQ, &start_freq);
                GetCtrlVal (pp, P2_STOP_FREQ, &stop_freq);
                if (band == 0.0)
                    Fmt (entry, "%s[w12]<%s", "Stop");
                else
                    Fmt (entry, "%s[w12]<%s", "Pass");
                Fmt (entry, "%s[w12a]<%f", start_freq);
                Fmt (entry, "%s[w12a]<%f", stop_freq);
                if (index < 0)
                    index = 0;
                else
                    index++;
                if (InsertListItem (pp, P2_LIST, index, entry) == 0)
                    num_bands++;
                RefreshListCtrl (pp, P2_LIST);
                DefaultCtrl (pp, P2_START_FREQ);
                DefaultCtrl (pp, P2_STOP_FREQ);
                SetActiveCtrl (P2_BAND);
                break;

            case P2_CUT:
            case P2_LIST:
                GetCtrlVal (pp, P2_LIST, entry);
                Scan (entry, "%s %f %f", band_buf, &start_freq, &stop_freq);
                SetCtrlVal (pp, P2_BAND, band_buf[0] == 'P');
                SetCtrlVal (pp, P2_START_FREQ, start_freq);
                SetCtrlVal (pp, P2_STOP_FREQ, stop_freq);
                DeleteListItem (pp, P2_LIST, index, 1);
                RefreshListCtrl (pp, P2_LIST);
                num_bands--;
                break;

            case P2_PASTE:
                GetCtrlVal (pp, P2_BAND, &band);
                GetCtrlVal (pp, P2_START_FREQ, &start_freq);
                GetCtrlVal (pp, P2_STOP_FREQ, &stop_freq);
                if (band == 0.0)
                    Fmt (entry, "%s[w12]<%s", "Stop");
                else
                    Fmt (entry, "%s[w12]<%s", "Pass");
                Fmt (entry, "%s[w12a]<%f", start_freq);
                Fmt (entry, "%s[w12a]<%f", stop_freq);
                if (index < 0)
                    index = 0;
                InsertListItem (pp, P2_LIST, index, entry);
                RefreshListCtrl (pp, P2_LIST);
                num_bands++;
                break;

            case P2_CLEAR:
                DeleteListItem (pp, P2_LIST, 0, num_bands);
                RefreshListCtrl (pp, P2_LIST);
                DefaultCtrl (pp, P2_START_FREQ);
                DefaultCtrl (pp, P2_STOP_FREQ);
                DefaultCtrl (pp, P2_BAND);
                num_bands = 0;
                break;

            case P2_DONE:
                RemovePopup (0);
                for (i = 0; (i < (2 * MAX_NUM_BANDS)); i++)
                    cutoffs[i] = 0.0;
                for (i = 0; i < num_bands; i++)  {
                    SetCtrlVal (pp, P2_LIST, i);
                    GetCtrlVal (pp, P2_LIST, entry);
                    Scan (entry, "%s %f %f", band_buf, &cutoffs[2 * i], &cutoffs[(2 * i) + 1]);
                    A[i] = (band_buf[0] == 'P');
                }
                SetInputMode (p, P1_PLOT, ON);
                SetActiveCtrl (P1_PLOT);
                return;
                break;
        }
        SetInputMode (pp, P2_ADD, (num_bands < MAX_NUM_BANDS));
        SetInputMode (pp, P2_PASTE, (num_bands < MAX_NUM_BANDS));
        GetListItemIndex (pp, P2_LIST, &index);
        SetInputMode (pp, P2_CUT, (index >= 0));
        SetInputMode (pp, P2_LIST, (index >= 0));
        SetInputMode (pp, P2_CLEAR, (index >= 0));
    }
}


/*****************************************************************************
    generate_tone  is used to create the 'switch' sound when the Power
    control switch is toggled.
*****************************************************************************/
void generate_tone (freq, duration)
double freq, duration;
{
  int low, high, port;
  long count;

  if (freq <= 0.0)
    return;
  count = (long)(1193280.0 / freq);
  low   = count % 256;
  high  = count / 256;
  outp (67,182);
  outp (66,low);
  outp (66,high);
  port = inp (97) | 3;
  outp (97,port);
  delay (duration);
  port = inp (97) & 0xFC;
  outp (97,port);
}

void report_uil_error (err_code, abort)
int err_code;
int abort;
{
    switch (err_code)  {
        case UIM_OPEN_ERR:
            FmtOut ("Unable to open the user interface manager.\n");
            break;
        case FONT_LOAD_ERR:
            MessagePopup ("Unable to load the system font.");
            break;
        case NO_MEM_ERR:
            if (abort)
                MessagePopup ("Not enough memory to continue... Aborting program.");
            else
                MessagePopup ("Not enough memory to plot graph.");
            break;
        case FILE_OPEN_ERR:
            MessagePopup ("Unable to open .UIR resource file.");
            break;
    }
    if (abort)
        exit (1);
}
