#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <graph.h>
#include <math.h>
#include <time.h>
#include <dos.h>
#include <io.h>

#include "c:\krs\adprogs\v3.h"
#include "c:\krs\adprogs\ioformat.h"
#include "c:\lw\include\dataacq.h"
#include "c:\lw\include\formatio.h"
#include "c:\lw\include\lwsystem.h"
#include "c:\lw\include\lw_const.h"

/****************************************************************************
*	Function:		int Initialize_GPS_Receiver(void)
*	Description:	Initialize the GPS receiver in warning mode through the 
*	serial port connection.  This routine waits for the GPS receiver to lock 
*	onto three satellites.  If lock cannot be made, the user has the option to
*	abort the program.
****************************************************************************/
int Initialize_GPS_Receiver(void)
{
  int  i,
       quit;
  struct rccoord textpos;

  gps_state = 2;

  /*  Initialize GPS Receiver */
  if (gps_state > 0)
  {
    tans_init();
    time (&gps_last_warning_time);		/* initialize warning timer */
    if (gps_state == 2)						/* wait for lock in warning mode */
    {
		_settextposition(HEADER_ROW1,HEADER_COL1);
      printf("Initializing GPS receiver - waiting for satellite lock\n");
      printf("Press 'Esc' to exit or 'Enter' to continue\n");
      quit = !TRUE;            
      Empty_Keyboard_Buffer();
      while (!quit)
      {
        if (Display_GPS_Status() == 0) quit = TRUE;
        if (kbhit())
        {
          i = getch();
          if (i == 0x1b)
          {
            _settextposition(MESSAGE_ROW1,MESSAGE_COL1);
            printf("Program aborted before GPS lock was established.\n");
            tans_restore();
            exit(0);  /* abort program */
          }
          else if (i == '\r')
            quit = TRUE;
        }  /* endif */
      } /* end while waiting for lock */
    }  /* end if warning set */
  } /* if gps is enabled */
}  

/****************************************************************************
*	Function:	int Check_Free_Disk_Space(void)
*	Description:	Gets the free disk space (from Disk_Free_kb procedure which 
*	checks the specified drive and returns true if the available space is less 
*	than specified by MIN_FREE_DISK_SIZE. Also display message and waits for
*	operator	to acknowledge this error.
****************************************************************************/
int Check_Free_Disk_Space (void)
{
  int  retval = TRUE;
  if (Disk_Free_kb() <= (long) MIN_FREE_DISK_SIZE)
  {  /* low disk space condition is here */
    retval = !TRUE;
    printf ("\a");  /* alert */
    printf("The data disk is nearly full\n");
    printf("Program operation halted.  Press any key to continue\n");
    printf ("\a");  /* alert */
    getch();
  }  /* end low disk space */
  return (retval);
}  /* end Check_Free_Disk_Space */

/****************************************************************************
*	Function:	unsigned long Disk_Free_kb(void)
*	Description:	Returns kbytes free (rounded down) as a long integer value.
*	Note:  This function currently checks only the C drive.  It may be req'd
*	to change this function so other drives (e.g., Bernoulli) are checked.
****************************************************************************/
unsigned long Disk_Free_kb(void)
{
  unsigned long free_space, bytes_per_cluster;
  struct	diskfree_t	dfinfo;

  if (_dos_getdiskfree(3, &dfinfo) != 0) /* can't do this disk */
  {
    printf ("Error in function\n");
    return (0L);
  }
  bytes_per_cluster = dfinfo.sectors_per_cluster * dfinfo.bytes_per_sector;
  free_space = dfinfo.avail_clusters * bytes_per_cluster;
  free_space = free_space / 1000L;
  return (free_space);
}	/* end Disk_Free_kb */

/*****************************************************************************
*	Function:		Display_GPS_Status(void)
*	Description:	Take the current GPS reading an load the structure (through
*	tans_get_gps_status) and display the result on some standardized display.
*	Returns current gps status byte (0 == locked o.k.)
*****************************************************************************/
int Display_GPS_Status()
{
  char	descr[50];
  time_t	now;	/* for current time */
  int	i, j;	/* used for delay loop */
  int	ret_status;	/* returned status, if =-1, then not all information correct */

  if (gps_state == 2) { /* warning line */
    _settextposition(GPS_HEADER_ROW,GPS_HEADER_COL);
    printf(GPS_HEADER_WARN);
  }
  else {
    _settextposition(GPS_HEADER_ROW,GPS_HEADER_COL);
    printf(GPS_HEADER);
  }
  switch (gps_state) {
    case 1:	/* gps on */
    case 2:	/* gps set to warning */
      ret_status = tans_get_gps_status (&gps_info);	/* get the gps information 07.08 */
      if (gps_state == 2) {	/* do warning checks */
        time(&now);	/* get current time */
        if (ret_status == 0)		/* 07.08 */
        { /* doing fixes */
          gps_last_warning_time = now;
        }
        else
        {  /* gps not locked, beep if time exceeded */
          if (fabs(difftime(now, gps_last_warning_time)) > (double) GPS_WARNING_TIME) 
          { /* time to warn them, do a nasty display thing too... */
            strcpy (descr, "GPS NOT LOCKED");
				_settextposition(MESSAGE_ROW1,MESSAGE_COL1);
				printf("\n%s\n",descr);
            printf ("\a");  /* sound bell here */
            gps_last_warning_time = now;
          }
        }
      }  /* end gps warning check */
      switch (ret_status) {  /* 07.08 */
        case -1:  /* still waiting for all packets info to accompany a fix 07.08 */
          sprintf (descr, "Wait   %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 0:  /* doing fixes */
          sprintf (descr, "Locked %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 1:  /* no GPS time yet */
          sprintf (descr, "NoTime %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 8:  /* no sats */
          sprintf (descr, "NO Sat %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 9:  /* 1 satellite */
          sprintf (descr, "1 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 0xa:  /* 2 satellites */
          sprintf (descr, "2 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        case 0xb:  /* 3 satellites */
          sprintf (descr, "3 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
        default:
          sprintf (descr, "Unknwn %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
            printf("%s",descr);
          break;
      }  /* end switch on status */
      break;
    case 0:  /* no gps in use */
      strcpy (descr, "Off");
      _settextposition(GPS_TEXT_ROW,GPS_TEXT_COL);
      printf("%s",descr);
      break;
  } /* end switch on gps_state */
  if (gps_state == 0) /* no gps */
    return (0);  /* return all o.k. in this case */
  else
    return (ret_status);  /* return current status (including -1) 07.08 */
}  /* end Display_GPS_Status */
  
/*****************************************************************************
*	Function:		int Process_Error_Code(int errNum)
*	Description:	Function process the error codes returned by the LabWindows
*	Data Acquisition C Libraries.  The warning and error codes are described in
*	the LabWindows documentation.
*****************************************************************************/
int Process_Error_Code(int errNum)
{
  switch (errNum)
  {
    case 0:
      //  Operation completed.  No error(s) detected.
      return(TRUE);
    break;
    case 5:
      // Warning message - calibrationFailed.
		_settextposition(MESSAGE_ROW2,MESSAGE_COL2);
      printf("Warning - Analog input channel calibration failed.\n");
      return(TRUE);
    break;
    case 12:
      // Warning - overWriteBeforeCopy.
		_settextposition(MESSAGE_ROW2,MESSAGE_COL2);
      printf("Warning - Buffer overwritten before copied.\n");
      return(!TRUE);
    break;
    case -60:
      //  Operation incomplete - NotOurBrdErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Data Acquisition board not recognized.\n");
      return(!TRUE);
    break;
    case -61:
      //  Operation incomplete - badBrdNumErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Incorrect data acquisition board number.\n");
      return(!TRUE);
    break;
    case -62:
      //  Operation incomplete - badGainErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Incorrect A/D gain value specified.\n");
      return(!TRUE);
    break;
    case -63:
      // Operation incomplete - badChanErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Invalid analog input channel specified.\n");
      return(!TRUE);
    break;
    case -64:
      // Operation incomplete - noSupportErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - No support available.\n");
      return(!TRUE);
    break;
    case -69:
      // Operation incomplete - badInputValError
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
		printf("Error - Bad input value\n");
      return(!TRUE);
    break;
    case -70:
      // Operation incomplete - timeOutErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Time out error.\n");
      return(!TRUE);
    break;
    case -71:
      // Operation incomplete - outOfRangeErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Data out of range.\n");
      return(!TRUE);
    break;
    case -72:
      // Operation incomplete - daqInProgErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Data acquisition in progress.\n");
      return(!TRUE);
    break;
    case -73:
      // Operation incomplete - counterInUseErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Counter in use.\n");
      return(!TRUE);
    break;
    case -75:
      // Operation incomplete - overFlowErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Buffer overflow error.\n");
      return(!TRUE);
    break;
    case -76:
      // Operation incomplete - overRunErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Buffer overrun error.\n");
      return(!TRUE);
    break;
    case -77:
      // Operation incomplete - badCntErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Bad counter error\n");
      return(!TRUE);
    break;
    case -91:
      // Operation incomplete - badPreTrigCntError.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Bad pretrigger counter error\n");
      return(!TRUE);
    break;
    case -93:
      // Operation incomplete - intLevelNoSupportErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - No internal level support\n");
      return(!TRUE);
    break;
    case -94:
      // Operation incomplete - extConvErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - External conversion error\n");
      return(!TRUE);
    break;
    case -96:
      // Operation incomplete - noDbDaqErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Double buffered data acquisition not initialized.\n");
      return(!TRUE);
    break;
    case -97:
      // Operation incomplete - overWriteErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Buffer overwrite error.\n");
      return(!TRUE);
	 break;
    case -98:
      //  Operation incomplete - memErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Memory error.\n");
      return(!TRUE);
    break;
    case -99:
      //  Operation incomplete - noConfigFile
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Cannot find configuration file.\n");
      return(!TRUE);
    break;
 	 case -101:
      //  Operation incomplete - intLevelInUse
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
    	printf("Error - Level in use.\n");
      return(!TRUE);
    break;
    case -102:
      // Operation incomplete - DMAChanInUse
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - DMA channel in use.\n");
      return(!TRUE);
    break;
    case -104:
      // Operation incomplete - lowScanIntervalErr
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Low scan interval error.\n");
      return(!TRUE);
    break;
    case -118:
      // Operation incomplete - DMADisabledErr
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - DMA channel disabled.\n");
      return(!TRUE);
    break;
    case -119:
      //  Operation incomplete - invalidConfigErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Invalid configuration data.\n");
      return(!TRUE); 
    break;
    case -127:
      // Operation incomplete - keyNotFoundErr.
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Key code not found.\n");
      return(!TRUE);
    break;
    default:
      //  Operation incomplete - default error code
		_settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Error - Error unknown.\n");
      exit(0);
    break;
  }
}

/*****************************************************************************
*	Function:		Empty_Keyboard_Buffer(void)
*	Description:	Function to empty keyboard buffer to enable a program escape
*	sequence.  This function is used prior to entering and key controlled loop.
*****************************************************************************/
Empty_Keyboard_Buffer(void)
{
  while( kbhit() )
    getch();
}

/*****************************************************************************
*	Function:		int Open_Data_File()
*	Description:	Ask the operator for the file number to store the data in.
*	This function creates a filename of the form mmddfxxx.. Where 'mm' is 
*	the current month, 'dd' is the current day, 'f' is the measurement band
*	(1 = 890 MHz and 2 = 2.1 GHz), and 'xxx' is the file number entered by the
*	the operator.  The '.acq' extension is standard for all of the	data files
*****************************************************************************/
int Open_Data_File(int fnumber)
{
  int 	i;
  char  	datebfr[10],filestr[32],tmpstr[10];

  /* Get the system date and operator entries */
  _strdate(datebfr);
  strncpy(filestr,datebfr,(size_t) 2);
  strncpy(&filestr[2],&datebfr[3],(size_t) 2);
  filestr[4] = '\0';
  for (i = 0; i < params.channels; i++)
    strcpy(datafilename[i],DATAPATH);
  for (i = 0; i < params.channels; i++)
    strcat(datafilename[i],filestr);
  for (i = 0; i < params.channels; i++) {
	 sprintf(tmpstr,"%1d",i);
    strcat(datafilename[i],tmpstr);
    sprintf(tmpstr,"%.3d",fnumber);
    strcat(datafilename[i],tmpstr);
  }
  for (i = 0; i < params.channels; i++)
    strcat(datafilename[i],DATAEXT);
  for (i = 0; i < params.channels; i++) {
    printf("Checking for file <%s>\n",datafilename[i]);
    datafileptr[i] = fopen(datafilename[i],"r");
    if (datafileptr[i] != NULL) { 
      printf("Data file <%s> exists\n",datafilename);
      fclose(datafileptr[i]);
      return(!TRUE);
    }
    printf("Opening data file <%s>\n\n",datafilename[i]);
    if ( (datafileptr[i] = fopen(datafilename[i],"w+")) == NULL) {
      printf("Unable to create data file <%s>\n",datafilename[i]);
      return(!TRUE);
    }
  }
  return(TRUE);
}

/****************************************************************************
*	Function:		int Open_Temporary_File(int fnumber)
*	Description:	Open a temporary file to store raw (unscaled) data in.  One
*	file is opened for each of the channels scanned.  The acquired data is 
*	demuxed before it is written to file to simplify later data processing.
*	Temporary files are deleted upon successful program completion.
****************************************************************************/
int Open_Temporary_File(int fnumber)
{

  int 	i;
  char  	operinpt[32],datebfr[10],filestr[32],tmpstr[10];

  for (i = 0; i < params.channels; i++) {
    _strdate(datebfr);
    strncpy(filestr,datebfr,(size_t) 2);
    strncpy(&filestr[2],&datebfr[3],(size_t) 2);
    filestr[4] = '\0';
    strcpy(tmpfilename[i],TMPPATH);
    strcat(tmpfilename[i],filestr);
    sprintf(filestr,"%1d",i);
    strcat(tmpfilename[i],filestr);
    sprintf(filestr,"%.3d",fnumber);
    strcat(tmpfilename[i],filestr);
    strcat(tmpfilename[i],TMPEXT);
    if ( (tmpfileptr[i] = fopen(tmpfilename[i],"w+")) == NULL) {
      printf("Unable to create temporary file <%s>\n",tmpfilename[i]);
      return(!TRUE);
    }
    else 
      printf("Temporary file <%s> opened\n",tmpfilename);
  }
  return(TRUE);
}

/****************************************************************************
*	Function:		int Get_File_Number()
*	Description:	Ask the operator for a file number.  The number can be up
*	to three digits.  This file number is incorported into the temporary and
*	final data filenames.
****************************************************************************/
int Get_File_Number()
{
  int fnumber;
  char operinpt[32];

  operinpt[3] = 0;
  printf("\n\nEnter the Output File Number (maximum of three digits):  ");
  gets(operinpt);
  printf("\n");
  if (strlen(operinpt) == 0) 
    return(!TRUE);
  fnumber = atoi(operinpt);
  if (fnumber < 1 || fnumber > 999) { /* error in range specified */
    printf("Number entered (%i) is out of range\n");
    return(!TRUE);
  }
  else
    return(fnumber);
}

/*****************************************************************************
*	Function:		int Write_Raw_Data(int ptsTfr,int *dataMuxed)
*	Description:	Demux the raw (unscaled) data and write each data stream to
*	its respective temporary file.  The data will be processed later.
*****************************************************************************/
int Write_Raw_Data(int ptsTfr,int *dataMuxed)
{
  int errNum;
  int index,marker;
  unsigned int bufsize;

  for (index = 0; index < params.channels; index++) {
    for (marker = index; marker < params.ptsTfr; marker += params.channels)
      fprintf(tmpfileptr[index],"%i\t",dataMuxed[marker]);
  }
  return(TRUE);
} 

/*****************************************************************************
*	Function:		int Write_End_Record()
*	Description:	Attach an end record to each of the data files.  The end
*	record verifies successful data acquisition and date/time staps the final
*	data file.
*****************************************************************************/
int Write_End_Record()
{
  int 		index;
  time_t 	end;
  char		dateBuffer[10],timeBuffer[10];

  _strdate(dateBuffer);
  _strtime(timeBuffer);
  for (index = 0; index < params.channels; index++)	{
    fprintf(datafileptr[index],COMMENT_FORMAT,COMMENT_END_DATA_RECORD);
    fprintf(datafileptr[index],END_DATA_RECORD,dateBuffer,timeBuffer);
  }
  for (index = 0; index < params.channels; index++) {
    fprintf(datafileptr[index],COMMENT_FORMAT,COMMENT_END_FILE);
    fprintf(datafileptr[index],END_DATA_FILE,index);
  }
}

/*****************************************************************************
*	Function:		int Process_Raw_Data()
*	Description:	Read the raw data stored in a temporary file, scale the 
*	raw (binary) value to an actual voltage level, and write this processed
*	to file.  This operation is conducted for each of the temporary files
*	(e.g., channels scanned).  
*****************************************************************************/
int Process_Raw_Data()
{
  int channel;
  int readval;
  double voltage;

  _settextposition(MESSAGE_ROW1,MESSAGE_COL1);
  printf("Processing temporary files.\n");

  for (channel = 0; channel < params.channels; channel++) {
    if ( (tmpfileptr[channel] = fopen(tmpfilename[channel],"r")) == NULL ) {
      _settextposition(MESSAGE_ROW3,MESSAGE_COL3);
      printf("Unable to open temporary file for data processing.\n");
      printf("Program aborted.\n");
      exit(0);
    }
    while (!feof(tmpfileptr[channel])) {
      fscanf(tmpfileptr[channel],"%i\t",&readval);
      voltage = Scale_Data(readval,params.gain[channel]);
      fprintf(datafileptr[channel],"%lf\t",voltage);
    }
    Write_End_Record();
    if (tmpfileptr == NULL)
      printf("NULL file temporary file pointer discovered\n");
    fclose(tmpfileptr[channel]);
    if (datafileptr[channel] == NULL)
      printf("NULL data file pointer discovered\n");
    fclose(datafileptr[channel]);
    remove(tmpfilename[channel]);
  }
  return(TRUE);
}

/*****************************************************************************
*	Function:		double Scale_Data(int readval,int gain)
*	Description:	Scale the A/D output (binary value) to a measured voltage.
*	This operation depends on the polarity setting on the DA board.
*****************************************************************************/
double Scale_Data(int readval,int gain)
{
  double voltage;
  
  switch(daqbrd.polarity) {
    case 0:
      /*  Bipolar mode */
      voltage = ((double)readval/2048.0) * (daqbrd.inputRange/gain);
    break;
    case 1:
      /* Unipolar mode */
      voltage = ((double)readval/4096.0) * (daqbrd.inputRange/gain);
    break;
    default:
      voltage = 0;
    break;
  }
  return(voltage);
}

/*****************************************************************************
*	Function:		int Default_Parameters()
*	Description:	Set up default DA board parameters and data acquisition
*	parameters.  This is hard coded so that parameters not specified in the
*	configuration file will have a "workable" value. 
*****************************************************************************/
int Default_Parameters()
{
  daqbrd.board = 2;				// Board installed on slot 2 of EISA Bus
  daqbrd.daqBufMode = 1; 		// Enable double buffering
  daqbrd.inputRange = 10;
  daqbrd.polarity = 1;
  daqbrd.inputMode = 0;

  params.samples = 2000;
  params.channels = 2;
  params.points = params.samples*params.channels;
  params.buffer = 2.0*(int)params.points;
  params.ptsTfr = params.samples / 2;

  params.sampTimebase = 3;		// Timebase (resolution) for sample interval counter
  params.sampInterval = 2;		// Length of sample interval (elapsed time between A/D conversions)
  params.scanTimebase = 4;    // Timebase for scan interval counter
  params.scanInterval = 0;    // Length of scan interval (elasped time between scan sequences)
  params.scanchannel[0] = 0;	// Analog channels to scan
  params.scanchannel[1] = 1;
  params.gain[0] = 1;		   // Default analog channel gain factors
  params.gain[1] = 1;
  return(TRUE);
} 

/*****************************************************************************
*	Function:		int Set_Graphics_Mode(void)
*	Description:	Initialize the graphics driver based on the graphics mode
*	supported by the machine.
*****************************************************************************/
int Set_Graphics_Mode(void)
{
  if (!graphics_mode) {
    graphics_mode = TRUE;
    if (_setvideomode(_VRES16COLOR) != 0)
      return(TRUE);
    if (_setvideomode(_HRES16COLOR) != 0)
      return(TRUE);
    if (_setvideomode(_MRES4COLOR) != 0)
      return(TRUE);
    printf("No graphics capability available...\n");
    return(!TRUE);
  }
}

/*****************************************************************************
*	Function:		int Create_Header_File(int fnumber)
*	Description:	Create a header file for the measurement.  One header file
*	is created for a single measurement session, so there may be several data
*	files corresponding to one header file (if multiple channels are scanned).
*	Pertinent measurement information, necessary for later data processing, is
*	stored in this file.
*****************************************************************************/
int Create_Header_File(int fnumber,char *configname)
{
  int		index,lines;
  char  	operinpt[32],datebfr[10],timebfr[10],filestr[32],tmpstr[10];
  char 	ch,configfilename[64],readstr[81],buffer[8][81];
  FILE 	*configfileptr;

  _strdate(datebfr);
  _strtime(timebfr);
  strncpy(filestr,datebfr,(size_t) 2);
  strncpy(&filestr[2],&datebfr[3],(size_t) 2);
  filestr[4] = '\0';
  sprintf(tmpstr,"%.4d",fnumber);
  strcat(filestr,tmpstr);
  strcpy(hdrfilename,HDRPATH);
  strcat(hdrfilename,filestr);
  strcat(hdrfilename,HDREXT);
  if ( (hdrfileptr = fopen(hdrfilename,"w+")) == NULL) {
    printf("Unable to create header file <%s>\n",hdrfilename);
    return(!TRUE);
  }
  printf("\nCreating header file <%s>\n",hdrfilename);

  /* Print legal/proprietary information */
  fprintf(hdrfileptr,LEGAL_HEADER1);  
  fprintf(hdrfileptr,LEGAL_HEADER2);
  fprintf(hdrfileptr,LEGAL_HEADER3);
  fprintf(hdrfileptr,LEGAL_HEADER4);
  fprintf(hdrfileptr,LEGAL_HEADER5);
  fprintf(hdrfileptr,LEGAL_HEADER6);
  fprintf(hdrfileptr,LEGAL_HEADER7,PROGRAM_NAME);
  fprintf(hdrfileptr,LEGAL_HEADER8,timebfr,datebfr);
  fprintf(hdrfileptr,LEGAL_HEADER9,configname);
  fprintf(hdrfileptr,LEGAL_HEADER10,fnumber);
  fprintf(hdrfileptr,LEGAL_HEADER11);

  /* Echo configuration file */
  strcpy(configfilename,CONFIGPATH);
  strcat(configfilename,configname);
  if ( (configfileptr = fopen(configfilename,"r+")) == NULL) {
    printf("Unable to open config file <%s>\n",configfilename);
    return(!TRUE);
  }
  while ( !feof(configfileptr) ) {
    fgets(readstr,80,configfileptr);
    fprintf(hdrfileptr,readstr);
  }

  /* Get user specific data */
  printf("\n\n****  Enter Measurement Information ****\n");
  printf("Enter measurement site description (up to 8 lines are allowed):\n");
  for (index = 0; index < 8; index++) {
    lines = index;
    gets(&buffer[index][0]);
    if (strlen(&buffer[index][0]) == 0)
      break;
  }
  for (index = 0; index < lines; index++) {
    if (strlen(&buffer[index][0]) != 0)
      fprintf(hdrfileptr,COMMENT_FORMAT,HEADER_SITE_DESCR,buffer[index]);
  }
    
  printf("Enter measurement site classification: ");
  index = 0;
  while ( ch != '\r') {
    ch = getch();
    putch(ch);
    tmpstr[index] = toupper(ch);
    index++;
  }
  tmpstr[index] = '\0';
  fprintf(hdrfileptr,COMMENT_FORMAT,HEADER_SITE_CLASS,tmpstr); 

  fclose(configfileptr);
  fclose(hdrfileptr);
  return(TRUE);
}

/*****************************************************************************
*	Function:		int Read_Config_File(char filename[])
*	Description:	Read the configuration file specified as a command line
*	argument.  Assign the measurement parameter from the configuration file
*	to its respective variable.
*****************************************************************************/
int Read_Config_File(char filename[])
{
  int		i,tmpval;
  char 	configfilename[64],readstr[81],*tmpstr;
  FILE 	*configfileptr;

  strcpy(configfilename,CONFIGPATH);
  strcat(configfilename,filename);
  if ( (configfileptr = fopen(configfilename,"r+")) == NULL) {
    printf("Unable to open config file <%s>\n",configfilename);
    return(!TRUE);
  }
  printf("Contents of configuration file <%s>\n\n",configfilename);

  while ( !feof(configfileptr) ) {
    fgets(readstr,80,configfileptr);
    printf("config > %s",readstr);
    tmpstr = strtok(readstr,"\t");
    if ( (tmpval = atoi(tmpstr)) <= 4)
      continue;
    else  {
      switch (tmpval) {
        case DA_BOARD_CONFIG:	/* Board Specific Parameters */
          tmpstr = strtok(NULL,"\t");
          daqbrd.board = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          daqbrd.inputRange = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          daqbrd.polarity = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          daqbrd.inputMode = atoi(tmpstr);
        break;
        case DA_BUFFER_CONFIG: /* Buffering Mode */
          tmpstr = strtok(NULL,"\t");
			 daqbrd.daqBufMode = atoi(tmpstr);
        break;
		  case DA_PARAMS: /* Data Acquisition Parameters */
          tmpstr = strtok(NULL,"\t");
          params.channels = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          params.samples = atoi(tmpstr);
          for (i = 0; i < params.channels; i++) {
            tmpstr = strtok(NULL,"\t");
            params.scanchannel[i] = atoi(tmpstr);
          }
        break;
        case ANALOG_CHANNEL_CONFIG: /* Analog Channel Gains */
          for (i = 0; i < params.channels; i++)	{
            tmpstr = strtok(NULL,"\t");
            params.gain[i] = atoi(tmpstr);
          }
        break;
        case SAMPLING_PARAMS:  /* Sampling Parameters */
          tmpstr = strtok(NULL,"\t");
          params.sampTimebase = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          params.sampInterval = atoi(tmpstr);
        break;
        case SCANNING_PARAMS: /* Scan Parameters */
          tmpstr = strtok(NULL,"\t");
          params.scanTimebase = atoi(tmpstr);
          tmpstr = strtok(NULL,"\t");
          params.scanInterval = atoi(tmpstr);
        break;
        default:
        break;
      }
    }
  }

  /* Additional variables to initialize */
  params.points = params.samples*params.channels;
  params.buffer = 2.0*(int)params.points;
  params.ptsTfr = params.samples / 2;

  printf("\n\nPress any key to continue...\n");
  getch();
  return(TRUE);
}

/****************************************************************************
*	Function:		Print_Parameters()
*	Description:	Print the DA board parameters.  Used for debugging only.
*****************************************************************************/
Print_Parameters()
{
  printf("daqbrd.board = %i\n",daqbrd.board); 
  printf("daqbrd.daqBufMode = %i\n",daqbrd.daqBufMode);
  printf("daqbrd.inputRange = %i\n",daqbrd.inputRange);
  printf("daqbrd.polarity = %i\n",daqbrd.polarity);
  printf("daqbrd.inputMode = %i\n",daqbrd.inputMode);

  printf("params.samples = %u\n",params.samples);
  printf("params.channels = %i\n",params.channels);
  printf("params.points = %i\n",params.points);
  printf("params.buffer = %i\n",params.buffer);
  printf("params.ptsTfr = %i\n",params.ptsTfr);

  printf("params.sampTimebase = %i\n",params.sampTimebase);
  printf("params.sampInterval = %u\n",params.sampInterval);
  printf("params.scanTimebase = %i\n",params.scanTimebase);
  printf("params.scanInterval = %u\n",params.scanInterval);
  printf("params.scanchannel[0] = %i\n",params.scanchannel[0]);
  printf("params.scanchannel[1] = %i\n",params.scanchannel[1]);
  printf("params.gain[0] = %i\n",params.gain[0]);
  printf("params.gain[1] = %i\n",params.gain[1]);

  getch();

  return(TRUE);
}

/*****************************************************************************
*	Function:		Initial_Screen()
*	Description:	Display a set-up screen describing program operation
*****************************************************************************/
int Initial_Screen()
{
  printf("\tͻ\n");
  printf("\t                                                        \n");
  printf("\t            Narrow Band Measurement Software            \n");
  printf("\t                                                        \n");
  printf("\t   This program is used to capture narrow band signal   \n");
  printf("\t   strength data.  This software is configured for      \n");
  printf("\t   multi-frequency moving measurements.  The multiple   \n");
  printf("\t   frequency measurements are performed using a high    \n");
  printf("\t   speed multi-channel DA board.  Location fixes for    \n");
  printf("\t   the moving measurements are provided by a GPS rcvr.  \n");
  printf("\t   Specific measurement parameters are defined in the   \n");
  printf("\t   configuration file specified at the command line.    \n");
  printf("\t                                                        \n");
  printf("\tͼ\n");
  printf("\n\n");
  printf("\tThe current data file path is <%s>\n",DATAPATH);
  printf("\tThe current header file path is <%s>\n",HDRPATH);
  printf("\tThe current temporary file path is <%s>\n",TMPPATH);
  printf("\n\n\tPress any key to continue...\n");
  getch();
  return(TRUE);
}

void Annotate(int xdispl,int ydispl,char annotate[])
{
  _moveto(xdispl,ydispl);
  _outgtext(annotate);
}
 
void Annotate_W(float xdispl,float ydispl,char annotate[])
{
  _moveto_w(xdispl,ydispl);
  _outgtext(annotate);
}

int Status_Screen()
{
  int maxx,maxy;
  char annotate[80];
  
  _getvideoconfig(&videoInfo);
  maxx = videoInfo.numxpixels - 1;
  maxy = videoInfo.numypixels - 1;
  _registerfonts("c:\\qc25\\samples\\*.fon");
  _setfont("t'Tms Rmn' h18 w18 b");
  _setcolor(15);
  _settextcolor(5);
  sprintf(annotate,"%s","Measurement Status");
  Annotate(10,100,annotate);
  getch();
}

void reset_graphics (void)
{
  _unregisterfonts();
  _setvideomode(_DEFAULTMODE);
  graphics_mode = !TRUE;
  _clearscreen(_GCLEARSCREEN);
}

/*****************************************************************************
*	Function:		main(int argc, char *argv[])
*	Description:	Main program subroutine
*****************************************************************************/
main(int argc, char *argv[])
{
  int i,diskfree;
  int errNum,status;
  int code,quit,fnumber,exitloop;
  int *dataBuffer,*dataHalfBuffer;
  long numTimeOutTicks;
  
  if (!Set_Graphics_Mode())
    exit(0);
  _clearscreen(_GCLEARSCREEN);
  Initial_Screen();

  if (argc < 2) {
    printf("\n\tNo configuration file was specified.  Program aborted.\n");
    exit(0);
  }

  Check_Free_Disk_Space();
  if ( !Check_Free_Disk_Space() ) {
    printf("Insufficient disk space...Program aborted\n");
    exit(0);
  }
 
  quit = !TRUE;
  Default_Parameters();
  _clearscreen(_GCLEARSCREEN);
  if ( !Read_Config_File(argv[1]) )
    exit(0);
  if (DEBUG)
    Print_Parameters();
  _clearscreen(_GCLEARSCREEN);
  if ( !(fnumber = Get_File_Number()) )
    exit(0);
  if ( !Open_Temporary_File(fnumber) )
    exit(0);
  if ( !Open_Data_File(fnumber) )
    exit(0);
  if ( !Create_Header_File(fnumber,argv[1]) )
    exit(0);

  printf("\nPress any key to continue...\n");
    getch();
  _clearscreen(_GCLEARSCREEN);
  Initialize_GPS_Receiver();
  errNum = Get_DA_Brds_Info(daqbrd.board,&daqbrd.boardCode,&daqbrd.baseAddr,
                            &daqbrd.irq1Lvl,&daqbrd.irq2Lvl,&daqbrd.irqTrigMode,
                            &daqbrd.dma1Lvl,&daqbrd.dma2Lvl,&daqbrd.daqMode);
  Process_Error_Code(errNum);
  printf("\n\nParameters retrieved for data acquisition board.\n");
  errNum = Init_DA_Brds(daqbrd.board,&daqbrd.boardCode);
  Process_Error_Code(errNum);
  errNum = AI_Config(daqbrd.board,daqbrd.inputMode,daqbrd.inputRange,daqbrd.polarity);
  Process_Error_Code(errNum);
  if (numTimeOutTicks < 20L) numTimeOutTicks = 20L;
  errNum = Timeout_Config(daqbrd.board,numTimeOutTicks);
  errNum = DAQ_DB_Config(daqbrd.board,daqbrd.daqBufMode);
  dataBuffer = (int *) calloc(params.buffer,sizeof(int));
  dataHalfBuffer = (int *) calloc(params.buffer,sizeof(int));
  printf("Data acquisition board and buffers initialized.\n");

  errNum = SCAN_Setup(daqbrd.board,params.channels,params.scanchannel,params.gain);
  if (errNum == 0)
    printf("Multi-channel scan setup complete.\n");
  printf("\nReady to begin scanning operation - Press any key to continue.\n");
  getch();
  Empty_Keyboard_Buffer();
  _clearscreen(_GCLEARSCREEN);
  Status_Screen();
  _settextposition(HEADER_ROW1,HEADER_COL1);
  printf("Press 'Enter' to Exit\t'Esc' to Abort\t'P' to Pause\n");
  errNum = SCAN_Start(daqbrd.board,dataBuffer,params.points,
                        params.sampTimebase,params.sampInterval,
                        params.scanTimebase,params.scanInterval);
  Process_Error_Code(errNum);
  while (!quit) 
  {
    errNum = DAQ_DB_Transfer(daqbrd.board,dataHalfBuffer,&params.ptsTfr,&status);
	 if (errNum != TRUE)
      Process_Error_Code(errNum);
    Display_GPS_Status();
	 Write_Raw_Data((int)params.ptsTfr,dataHalfBuffer);
    _settextposition(STATUS_HEADER_ROW,STATUS_HEADER_COL);
    printf(STATUS_HEADER);
    _settextposition(STATUS_TEXT_ROW,STATUS_TEXT_COL);
    printf("MEASURING\t\t");
    if ( kbhit() )
    {
      code = getch();
      Empty_Keyboard_Buffer();
      if (code == 0x1b)
      {
        errNum = DAQ_Clear(daqbrd.board);
        exit(0);
      }
      else if (code == '\r')
      {
        errNum = DAQ_Clear(daqbrd.board);
        _settextposition(STATUS_TEXT_ROW,STATUS_TEXT_COL);
        printf("HALTED\t\t");
        free(dataHalfBuffer);
        free(dataBuffer);
        for (i = 0; i < params.channels; i++)
          fclose(tmpfileptr[i]);
        Process_Raw_Data();
        exit(0);
      }
      else if (code == 'p')
	   {
        _settextposition(STATUS_TEXT_ROW,STATUS_TEXT_COL);
        printf("PAUSED\t\t");
        errNum = DAQ_DB_Transfer(daqbrd.board,dataHalfBuffer,&params.ptsTfr,&status);
		  Empty_Keyboard_Buffer();
        while (!quit) {
          if ( kbhit() ) {
            code = getch();
            if (code == 0x1b) {
              errNum = DAQ_Clear(daqbrd.board);
              exit(0);
            }
            else if (code == '\r')
            {
              quit = TRUE;
              errNum = DAQ_Clear(daqbrd.board);
				  _settextposition(MESSAGE_ROW1,MESSAGE_COL1);
              printf("\nResuming data acqusition\n");
              Display_GPS_Status();
	           Write_Raw_Data((int)params.ptsTfr,dataHalfBuffer);
            }
          }
        }
        quit = !TRUE;
      }
	 }
  }
}
