#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <dos.h>

#include "c:\krs\adprogs\acquire.h"
#include "c:\lw\include\dataacq.h"
#include "c:\lw\include\formatio.h"
#include "c:\lw\include\lwsystem.h"
#include "c:\lw\include\lw_const.h"

// GPS variable definitions, functions, and global variables
#include		"tans_def.h"				/* declare variables for tans gps routines */
#include		"tans_pro.c"				/* tans procedures */
#define		GPS_WARNING_TIME	10		/* warning bell every 10 (or setting) seconds */
int			gps_state;					/* GPS state marker */
time_t		gps_last_warning_time;	/* holds time of last gps warning */
struct		TANS_BUFFER	gps_info;	/* store the results here */

// Function prototypes
int check_free_disk_space (void);
unsigned long Disk_Free_kb(void);
int Initialize_GPS_Receiver(void);
int Display_GPS_Status(void);
int Check_Free_Disk_Space (void);
int Display_GPS_Status(void);
unsigned long Disk_Free_kb(void);
int Open_Data_File();
int Open_Temporary_File();

/****************************************************************************
*	Function:		int Initialize_GPS_Receiver(void)
*	Description:	Initialize the GPS receiver in warning mode through the 
*	serial port connection.  This routine waits for the GPS receiver to lock 
*	onto three satellites.  If lock cannot be made, the user has the option to
*	abort the program.
****************************************************************************/
int Initialize_GPS_Receiver(void)
{
  int  i,
       quit;

  gps_state = 2;

  /*  Initialize GPS Receiver */
  if (gps_state > 0)
  {
    tans_init();
    time (&gps_last_warning_time);		/* initialize warning timer */
    if (gps_state == 2)						/* wait for lock in warning mode */
    {
      printf("Waiting for GPS to lock onto satellites\n");
      printf("'Esc' to exit or 'Enter' to continue\n");
      quit = !TRUE;            
      Empty_Keyboard_Buffer();
      while (!quit)
      {
        if (Display_GPS_Status() == 0) quit = TRUE;  
      if (kbhit())
        {
          i = getch();
          if (i == 0x1b)
          {
            printf("Program aborted before GPS lock was established.\n");
            tans_restore();
            exit(0);  /* abort program */
          }
          else if (i == '\r')
            quit = TRUE;
        }  /* endif */
      } /* end while waiting for lock */
    }  /* end if warning set */
  } /* if gps is enabled */
}  

/****************************************************************************
*	Function:	int Check_Free_Disk_Space(void)
*	Description:	Gets the free disk space (from Disk_Free_kb procedure which 
*	checks the specified drive and returns true if the available space is less 
*	than specified by MIN_FREE_DISK_SIZE. Also display message and waits for
*	operator	to acknowledge this error.
****************************************************************************/
int Check_Free_Disk_Space (void)
{
  int  retval = !TRUE;
  if (Disk_Free_kb() <= (long) MIN_FREE_DISK_SIZE)
  {  /* low disk space condition is here */
    retval = TRUE;
    printf ("\a");  /* alert */
    printf("The data disk is nearly full\n");
    printf("Program operation halted.  Press any key to continue\n");
    printf ("\a");  /* alert */
    getch();
  }  /* end low disk space */
  return (retval);
}  /* end Check_Free_Disk_Space */

/****************************************************************************
*	Function:	unsigned long Disk_Free_kb(void)
*	Description:	Returns kbytes free (rounded down) as a long integer value.
*	Note:  This function currently checks only the C drive.  It may be req'd
*	to change this function so other drives (e.g., Bernoulli) are checked.
****************************************************************************/
unsigned long Disk_Free_kb(void)
{
  unsigned long free_space, bytes_per_cluster;
  struct	diskfree_t	dfinfo;

  if (_dos_getdiskfree(3, &dfinfo) != 0) /* can't do this disk */
  {
    printf ("Error in function\n");
    return (0L);
  }
  bytes_per_cluster = dfinfo.sectors_per_cluster * dfinfo.bytes_per_sector;
  free_space = dfinfo.avail_clusters * bytes_per_cluster;
  free_space = free_space / 1000L;
  return (free_space);
}	/* end Disk_Free_kb */

/*****************************************************************************
*	Function:		Display_GPS_Status(void)
*	Description:	Take the current GPS reading an load the structure (through
*	tans_get_gps_status) and display the result on some standardized display.
*	Returns current gps status byte (0 == locked o.k.)
*****************************************************************************/
int Display_GPS_Status(void)
{
  char	descr[50];
  time_t	now;	/* for current time */
  int	i, j;	/* used for delay loop */
  int	ret_status;	/* returned status, if =-1, then not all information correct */

  if (gps_state == 2) /* warning line */
    printf("GPS_TEXT_LINE_WARN\n");
  else
    printf("GPS_TEXT_LINE\n");
  switch (gps_state) {
    case 1:	/* gps on */
    case 2:	/* gps set to warning */
      ret_status = tans_get_gps_status (&gps_info);	/* get the gps information 07.08 */
      if (gps_state == 2) {	/* do warning checks */
        time(&now);	/* get current time */
        if (ret_status == 0)		/* 07.08 */
        { /* doing fixes */
          gps_last_warning_time = now;
        }
        else
        {  /* gps not locked, beep if time exceeded */
          if (fabs(difftime(now, gps_last_warning_time)) > (double) GPS_WARNING_TIME) 
          {  /* time to warn them, do a nasty display thing too... */
            /* do a funky message here */
            for (i = 0; i < 20; i++ ){
              strnset (descr, '.', 40);
              strcpy (&descr[i], "GPS NOT LOCKED.");
              printf("%s\n",descr);
              for (j=0; j < 15000; j++)  ;  /* delay loop */
            }  /* end of funky message */
            printf ("\a");  /* sound bell here */
            for (i = 20; i > 0; --i ){ /* now do backwards */
              strnset (descr, '.', 40);
              strcpy (&descr[i], "GPS NOT LOCKED.");
              printf("%s\n",descr);
              for (j=0; j < 15000; j++)  ;  /* delay loop */
            }  /* end of funky message */

            gps_last_warning_time = now;
          }
        }
      }  /* end gps warning check */
      switch (ret_status) {  /* 07.08 */
        case -1:  /* still waiting for all packets info to accompany a fix 07.08 */
          sprintf (descr, "Wait   %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 0:  /* doing fixes */
          sprintf (descr, "Locked %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 1:  /* no GPS time yet */
          sprintf (descr, "NoTime %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 8:  /* no sats */
          sprintf (descr, "NO Sat %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 9:  /* 1 satellite */
          sprintf (descr, "1 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 0xa:  /* 2 satellites */
          sprintf (descr, "2 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        case 0xb:  /* 3 satellites */
          sprintf (descr, "3 Sat  %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
        default:
          sprintf (descr, "Unknwn %2d.%2d.%2d.%2d %3.1f %7.4lf %9.4lf",
            gps_info.c_sats[0],
            gps_info.c_sats[1],
            gps_info.c_sats[2],
            gps_info.c_sats[3],
            gps_info.c_pdop,
            gps_info.last_lat,
            gps_info.last_lon);
            printf("%s\n",descr);
          break;
      }  /* end switch on status */
      break;
    case 0:  /* no gps in use */
      strcpy (descr, "Off");
      printf("%s\n",descr);
      break;
  } /* end switch on gps_state */
  if (gps_state == 0) /* no gps */
    return (0);  /* return all o.k. in this case */
  else
    return (ret_status);  /* return current status (including -1) 07.08 */
}  /* end Display_GPS_Status */
  
/*****************************************************************************
*	Function:		int Process_Error_Code(int errNum)
*	Description:	Function process the error codes returned by the LabWindows
*	Data Acquisition C Libraries.  The warning and error codes are described in
*	the LabWindows documentation.
*****************************************************************************/
int Process_Error_Code(int errNum)
{
  switch (errNum)
  {
    case 0:
      //  Operation completed.  No error(s) detected.
      return(TRUE);
    break;
    case 5:
      // Warning message - calibrationFailed.
      printf("Warning - Analog input channel calibration failed.\n");
      return(TRUE);
    break;
    case 12:
      // Warning - overWriteBeforeCopy.
      printf("Warning - Buffer overwritten before copied.\n");
      return(!TRUE);
    break;
    case -60:
      //  Operation incomplete - NotOurBrdErr.
      printf("Error - Data Acquisition board not recognized.\n");
      return(!TRUE);
    break;
    case -61:
      //  Operation incomplete - badBrdNumErr.
      printf("Error - Incorrect data acquisition board number.\n");
      return(!TRUE);
    break;
    case -62:
      //  Operation incomplete - badGainErr.
      printf("Error - Incorrect A/D gain value specified.\n");
      return(!TRUE);
    break;
    case -63:
      // Operation incomplete - badChanErr.
      printf("Error - Invalid analog input channel specified.\n");
      return(!TRUE);
    break;
    case -64:
      // Operation incomplete - noSupportErr.
      printf("Error - No support available.\n");
      return(!TRUE);
    break;
    case -69:
      // Operation incomplete - badInputValError
		printf("Error - Bad input value\n");
      return(!TRUE);
    break;
    case -70:
      // Operation incomplete - timeOutErr.
      printf("Error - Time out error.\n");
      return(!TRUE);
    break;
    case -71:
      // Operation incomplete - outOfRangeErr.
      printf("Error - Data out of range.\n");
      return(!TRUE);
    break;
    case -72:
      // Operation incomplete - daqInProgErr.
      printf("Error - Data acquisition in progress.\n");
      return(!TRUE);
    break;
    case -73:
      // Operation incomplete - counterInUseErr.
      printf("Error - Counter in use.\n");
      return(!TRUE);
    break;
    case -75:
      // Operation incomplete - overFlowErr.
      printf("Error - Buffer overflow error.\n");
      return(!TRUE);
    break;
    case -76:
      // Operation incomplete - overRunErr.
      printf("Error - Buffer overrun error.\n");
      return(!TRUE);
    break;
    case -77:
      // Operation incomplete - badCntErr.
      printf("Error - Bad counter error\n");
      return(!TRUE);
    break;
    case -91:
      // Operation incomplete - badPreTrigCntError.
      printf("Error - Bad pretrigger counter error\n");
      return(!TRUE);
    break;
    case -93:
      // Operation incomplete - intLevelNoSupportErr.
      printf("Error - No internal level support\n");
      return(!TRUE);
    break;
    case -94:
      // Operation incomplete - extConvErr.
      printf("Error - External conversion error\n");
      return(!TRUE);
    break;
    case -96:
      // Operation incomplete - noDbDaqErr.
      printf("Error - Double buffered data acquisition not initialized.\n");
      return(!TRUE);
    break;
    case -97:
      // Operation incomplete - overWriteErr.
      printf("Error - Buffer overwrite error.\n");
      return(!TRUE);
	 break;
    case -98:
      //  Operation incomplete - memErr.
      printf("Error - Memory error.\n");
      return(!TRUE);
    break;
    case -99:
      //  Operation incomplete - noConfigFile
      printf("Error - Cannot find configuration file.\n");
      return(!TRUE);
    break;
 	 case -101:
      //  Operation incomplete - intLevelInUse
    	printf("Error - Level in use.\n");
      return(!TRUE);
    break;
    case -102:
      // Operation incomplete - DMAChanInUse
      printf("Error - DMA channel in use.\n");
      return(!TRUE);
    break;
    case -104:
      // Operation incomplete - lowScanIntervalErr
      printf("Error - Low scan interval error.\n");
      return(!TRUE);
    break;
    case -118:
      // Operation incomplete - DMADisabledErr
      printf("Error - DMA channel disabled.\n");
      return(!TRUE);
    break;
    case -119:
      //  Operation incomplete - invalidConfigErr.
      printf("Error - Invalid configuration data.\n");
      return(!TRUE); 
    break;
    case -127:
      // Operation incomplete - keyNotFoundErr.
      printf("Error - Key code not found.\n");
      return(!TRUE);
    break;
    default:
      //  Operation incomplete - default error code
      printf("Error - Error unknown.\n");
      exit(0);
    break;
  }
}

/*****************************************************************************
*	Function:		Empty_Keyboard_Buffer(void)
*	Description:	Function to empty keyboard buffer to enable a program escape
*	sequence.  This function is used prior to entering and key controlled loop.
*****************************************************************************/
Empty_Keyboard_Buffer(void)
{
  while( kbhit() )
    getch();
}

/*****************************************************************************
*	Function:		int Open_Data_File()
*	Description:	Ask the operator for the file number to store the data in.
*	This function creates a filename of the form mmddfxxx.. Where 'mm' is 
*	the current month, 'dd' is the current day, 'f' is the measurement band
*	(1 = 890 MHz and 2 = 2.1 GHz), and 'xxx' is the file number entered by the
*	the operator.  The '.acq' extension is standard for all of the	data files
*****************************************************************************/
int Open_Data_File(int fnumber)
{
  int 	i;
  char  	datebfr[10],filestr[32],tmpstr[10];

  /* Get the system date and operator entries */
  _strdate(datebfr);
  strncpy(filestr,datebfr,(size_t) 2);
  strncpy(&filestr[2],&datebfr[3],(size_t) 2);
  filestr[4] = '\0';
  for (i = 0; i < params.channels; i++) {
    strcpy(datafilename[i],DATAPATH);
    printf("%s\n",datafilename[i]);
  }
  for (i = 0; i < params.channels; i++)
    strcat(datafilename[i],filestr);
  for (i = 0; i < params.channels; i++) {
	 sprintf(tmpstr,"%1d",i);
    strcat(datafilename[i],tmpstr);
    sprintf(tmpstr,"%.3d",fnumber);
    strcat(datafilename[i],tmpstr);
  }
  for (i = 0; i < params.channels; i++)
    strcat(datafilename[i],DATAEXT);
  for (i = 0; i < params.channels; i++) {
    printf("Checking for file <%s>\n",datafilename[i]);
    datafileptr[i] = fopen(datafilename[i],"r");
    if (datafileptr[i] != NULL) { 
      printf("Data file <%s> exists\n",datafilename);
      fclose(datafileptr[i]);
      return(!TRUE);
    }
    printf("Opening data file <%s>\n",datafilename[i]);
    if ( (datafileptr[i] = fopen(datafilename[i],"w+")) == NULL) {
      printf("Unable to create data file <%s>\n",datafilename[i]);
      return(!TRUE);
    }
  }
  printf("Data files opened...\n");
  return(TRUE);
}

/****************************************************************************
*
****************************************************************************/
int Open_Temporary_File(int fnumber)
{

  int 	i;
  char  	operinpt[32],datebfr[10],filestr[32],tmpstr[10];

  _strdate(datebfr);
  strncpy(filestr,datebfr,(size_t) 2);
  strncpy(&filestr[2],&datebfr[3],(size_t) 2);
  filestr[4] = '\0';
  sprintf(tmpstr,"%.4d",fnumber);
  strcat(filestr,tmpstr);
  strcpy(tmpfilename,TMPPATH);
  strcat(tmpfilename,filestr);
  strcat(tmpfilename,TMPEXT);
  if ( (tmpfileptr = fopen(tmpfilename,"w+")) == NULL) {
    printf("Unable to open temporary file <%s>\n",tmpfilename);
    return(!TRUE);
  }
  else {
    printf("Temporary file <%s> opened\n",tmpfilename);
    return(TRUE);
  }
}
/****************************************************************************
*
****************************************************************************/
int Get_File_Number()
{
  int fnumber;
  char operinpt[32];

  operinpt[3] = 0;
  printf("\n\nEnter the Output File Number (maximum of three digits):");
  gets(operinpt);
  if (strlen(operinpt) == 0) 
    return(!TRUE);
  fnumber = atoi(operinpt);
  if (fnumber < 1 || fnumber > 999) { /* error in range specified */
    printf("Number entered (%i) is out of range\n");
    return(!TRUE);
  }
  else
    return(fnumber);
}

/*****************************************************************************
*
*****************************************************************************/
unsigned int Write_Demuxed_Data(int ptsTfr,int *dataMuxed,int * rawChanData,double *voltage)
{
  int errNum;
  int index,marker;
  unsigned int bufsize;

  for (index = 0; index < params.channels; index++) {
    bufsize = 0;
    for (marker = index; marker < ptsTfr; marker += params.channels) {
      rawChanData[bufsize] = dataMuxed[marker];
      bufsize++;
    }
    errNum = DAQ_Scale(daqbrd.board,params.gain[index],bufsize,rawChanData,voltage);
	 if (errNum != TRUE)
      Process_Error_Code(errNum);
    for (marker = 0; marker < bufsize; marker++) {
      fprintf(tmpfileptr,"%i\t%lf\t",index,voltage[marker]);
    }
  }
  return(TRUE);
} 

/*****************************************************************************
*
*****************************************************************************/
int Process_Tempfile()
{
  int channel;
  double readval;

  printf("Processing temporary file\n");

  if ( (tmpfileptr = fopen(tmpfilename,"r")) == NULL ) {
    printf("Unable to open temporary file for data processing.\n");
    printf("Program aborted.\n");
    exit(0);
  }

  while (!feof(tmpfileptr)) {
    fscanf(tmpfileptr,"%i\t%lf\t",&channel,&readval);
    if (datafileptr[channel] == NULL)
      printf("Warning - Null file pointer assignment\n");
    fprintf(datafileptr[channel],"%i\t%lf\t",channel,readval);
  }

  if (tmpfileptr == NULL)
    printf("NULL file temporary file pointer discovered\n");
  fclose(tmpfileptr);
  for (channel = 0; channel < params.channels; channel++) {
    if (datafileptr[channel] == NULL)
      printf("NULL data file pointer discovered\n");
    fclose(datafileptr[channel]);
  }
  return(TRUE);
}

main()
{
  int i;
  int errNum,status;
  int code,quit,fnumber;
  int *dataBuffer,*dataHalfBuffer;
  int *rawChanData;
  double *voltage;
  unsigned int ptsTfr;
  long numTimeOutTicks;
  
  quit = FALSE;
  daqbrd.board = 2;				// Board installed on slot 2 of EISA Bus
  daqbrd.daqBufMode = 1; 		// Enable double buffering

  params.samples = 5000;
  params.channels = 2;
  params.points = params.samples*params.channels;

  params.sampTimebase = 3;		// Timebase (resolution) for sample interval counter
  params.sampInterval = 2;		// Length of sample interval (elapsed time between A/D conversions)
  params.scanTimebase = 4;    // Timebase for scan interval counter
  params.scanInterval = 0;    // Length of scan interval (elasped time between scan sequences)
  params.scanchannel[0] = 0;	// Analog channels to scan
  params.scanchannel[1] = 1;
  params.gain[0] = 1;		   // Default analog channel gain factors
  params.gain[1] = 1;
  ptsTfr = params.samples / 2;
  
  if ( !(fnumber = Get_File_Number()) )
    exit(0);
  if ( !Open_Temporary_File(fnumber) )
    exit(0);
  if ( !Open_Data_File(fnumber) )
    exit(0);
  errNum = Get_DA_Brds_Info(daqbrd.board,&daqbrd.boardCode,&daqbrd.baseAddr,
                            &daqbrd.irq1Lvl,&daqbrd.irq2Lvl,&daqbrd.irqTrigMode,
                            &daqbrd.dma1Lvl,&daqbrd.dma2Lvl,&daqbrd.daqMode);
  Process_Error_Code(errNum);
  printf("Parameters retrieved for data acquisition board\n");
  errNum = Init_DA_Brds(daqbrd.board,&daqbrd.boardCode);
  Process_Error_Code(errNum);
  errNum = AI_Config(daqbrd.board,0,10,1);
  Process_Error_Code(errNum);
  printf("Data acquisition board initialized\n");
  params.points = params.channels * params.samples;
  params.buffer = 2.0*(int)params.points;
  rawChanData = (int *) calloc(params.points,sizeof(int));
  voltage = (double *) calloc(params.points,sizeof(double));
  dataBuffer = (int *) calloc(params.buffer,sizeof(int));
  dataHalfBuffer = (int *) calloc(params.buffer,sizeof(int));
  
  if (numTimeOutTicks < 20L) numTimeOutTicks = 20L;
  errNum = Timeout_Config(daqbrd.board,numTimeOutTicks);
  errNum = DAQ_DB_Config(daqbrd.board,daqbrd.daqBufMode);
  if (errNum == 0)
    printf("Configured for double buffered operation\n");
  errNum = SCAN_Setup(daqbrd.board,params.channels,params.scanchannel,params.gain);
  if (errNum == 0)
    printf("Scan setup complete\n");
  Empty_Keyboard_Buffer();
  printf("Scanning channels...\n");
  errNum = SCAN_Start(daqbrd.board,dataBuffer,params.points,
                        params.sampTimebase,params.sampInterval,
                        params.scanTimebase,params.scanInterval);
  Process_Error_Code(errNum);
  while (!quit) 
  {
    errNum = DAQ_DB_Transfer(daqbrd.board,dataHalfBuffer,&ptsTfr,&status);
	 if (errNum != TRUE)
      Process_Error_Code(errNum);
	 Write_Demuxed_Data((int)ptsTfr,dataHalfBuffer,rawChanData,voltage);
    if ( kbhit() )
    {
      code = getch();
      Empty_Keyboard_Buffer();
      if (code == 0x1b)
      {
        errNum = DAQ_Clear(daqbrd.board);
        printf("Data acquisition complete...\n");
        fclose(tmpfileptr);
        free(dataHalfBuffer);
        free(dataBuffer);
        Process_Tempfile();
        exit(0);
      }
      else if (code == '\r')
      {
        errNum = DAQ_Clear(daqbrd.board);
        fclose(tmpfileptr);
        exit(0);
      }
    }
  }
}

