/*  TANS_PRO.C

	TANS GPS Receiver SERVICE ROUTINES -- C Source programs to in #INCLUDED
	in user programs to talk to the GPS.

	REVISION #1 - 4/2/92:  Added a simplified structure for user programs
	to access the current status of the GPS receiver.  This simplifies the
	user access, but complicates software.  See routine "tans_get_gps_status".

	Revision #2 - Revision #1 would cause crashes and stuff...very nasty.
	This revision appends to the double lla packet the packets 41, 44, and
	46 to hold the status at the time of fix.  This required hard coded
	fixes several places (noted by "rev2")

	Revision #3 - In tans_init, set the unused pointers to a junk_buffer
	instead of null.  This is attempted to see if this eliminates the
	occasional "runtime error R6001 -- Null Pointer Assignment" error that
	occasionally occurs. 4/12/92

	Revision #4 - Revised the procedure "get_tans_gps_status" to return
	a value indicating the current status OR IF NOT ALL THE NECESSARY
	TANS PACKETS (41,44,46) HAVE BEEN RECEIVED, RETURNS a -1!  This enables
	the application program to determine if ALL satellite parameters (status, time_of_week,
	satellites, pdop, etc) are contained in the gps structure along with
	any current position fix (lat,lon). 4/12/92

   Revision #5 - removed the check for packet 41 when determing the
   "wait" (-1) status from tans_get_status.  3/14/94.

   Original February 19, 1992  Craig S. Palmer

   Used lsr_prog.c as model to develop this module

NOTE: Doesn't work properly if run from the QuickC environment.  Run
programs from DOS.

	Note that TANS_DEF.H must be included prior to including this file
	in user program.  All global variables are defined in TANS_DEF.H.

Revision History:
	REVISION #1 - 4/2/92:  Added a simplified structure for user programs
		to access the current status of the GPS receiver.  This simplifies the
		user access, but complicates software.  See routine "tans_get_gps_status".
	02-19-92  Started this program, based on lsr_prog.c and initial
      development of tans7.c code to build and test algorithm.
	02-21-92  Added write on initialization packets to the Tans (in
		tans_init()) to set the Tans to double precision and lat/lon
		mode.
	02-24-92  Wouldn't work on Compaq laptop (but worked on Compaq
		portables.  Revised the interrupt enable logic to set the
		interupt on received character available.
	02-26-92  Added routines copy_float and copy_double to convert
		character buffer (passed from the TANS) to Microsoft C
		floating and double real variables.  The TANS sends as MSB,...,LSB
		but Microsoft C wants as LSB,...,MSB.

	User Program callable Routines specified in this module:
		int tans_init();		initialize and set-up for TANS GPS receiver

		int tans_get_gps_status (struct TANS_BUFFER buffer);
			returns the current status and last known position including
			flags for position is stale (old) and which satellites, etc.

		int tans_ask_position (double *lat, double *lon, double *height,
			int *number_satellites, int sats[6], float 

		int tans_get_packet (int id, int &length, char packet_data[]);
			Returns the latest packet for packet type identified by 'id'.
			Returned value can be the following values(defined in tans_def.h):
				invalid packet id code;
				packet id code is not stored;
				packet id is valid -- no new reading, last packet contents returned
				packet id is valid -- new reading, new packet contents returned
			If a new reading is available, the 'new' reading flag is reset after
			reading.

		int tans_restore();	Called upon termination of program to restore
			most unchanged registers.  This (hopefully) should place the
			MS-DOS system back into a known and predictable state.  
			Abnormal termination may cause strange things to happen
			requiring system reset to operate properly.

		void copy_float (char *dest, char *source);
			Copy a floating point number in MSB, ... , LSB (as sent by the
			TANS) to the Microsoft C format LSB,...,MSB.

		void copy_double (char *dest, char *source);
			Copy a double point number in MSB, ... , LSB (as sent by the
			TANS) to the Microsoft C format LSB,...,MSB


	Interrupt Service Routine not callable by user:
	
		void interrupt far tans_isr(void);  services COM1 interrupts according
					to algorithm and updates structure contained in TANS_DEF.H.
					The user should not need to use this structure as the above
					user callable procedures should suffice for the entire
					interface.


For SAMPLE PROGRAM, see file "tans_samp.c" (future)

---------------------------  end comments  ----------------------------*/

/* define constants for this program */
#define	DLE	0x10
#define	ETX	0x03
/* the following two packets are sent to tans in tans_init(): */
static char TANS_INIT_PACKET_2C[] = {DLE, 0x2c, 1, 0xbf, 0x80, 0, 0,
	0xbf, 0x80, 0, 0, 0xbf, 0x80, 0, 0, 0x80, 0xbf, 0, 0, DLE, ETX};
   /* initialize packet 2ch to
	land velocity, and four -1's (float) to keep the elevation angle, signal
	level mask, pdop mask, and pdop switches the same.  Length is 21 bytes */
static char TANS_INIT_PACKET_35[] = {DLE, 0x35, 0x13, 1, 0, 0, DLE, ETX};
	/* initilze packet 35 sets the i/o options to (1) lat/lon and double
	precision (for single precision, change byte 3 from 0x13 to 0x03),
	(2) velocity default, (3) timing default, (4) aux default. 8 bytes long*/

/* Define COM1 port addresses */
/* define com port vector (NOTE: Non-relocatable, fix in future) */
#define com1_vector 0x0C		// interrupt vector for com1
#define com1_base 0x3F8		// port address for start of com1,
#define com1_data	0			// received data port, offset from com1_base
#define com1_enable 1		// enable port for com1, offset
#define com1_id 2			// interrupt id for com1, offset
#define com1_line 3		// com1 line control (baud, etc.), offset
#define com1_modem 4		// com1 modem status, offset
#define com1_status 5		// com1 status (errors), offset
#define com1_status2 6		// com1, 2nd status, offset
#define UART_ADDR1 0x20		// UART chip, data port (it must be used to work)
#define UART_ADDR2 0x21		// UART chip, enable interrupt port

/* define the 5 states specific to the TANS packets, only used in the
   tans interrupt service routine */
/* define the possible waiting states when a new character is received */
#define	TANS_NEW_PACKET	0	/* awaiting start of new packet */
#define	TANS_ID 1	/* await packet id character */
#define	TANS_PACKET_DATA 2  /* awaiting packet data characters */
#define	TANS_ENDING_DLE 3	/* awaiting DLE at end of packet */
#define	TANS_ENDING_ETX 4  /* awaiting ETX to terminate packet */

/* Some globals here, only variables kept during duration of program */
	void (interrupt far *com1_old_vector) ();
		 /* save old com1 interrupt vector from _dos_getvect  pg 615 QC bible*/
	int old_uart_register;			// holds old uart register

/* =================================================================
	TANS INTERRUPT SERVICE ROUTINE

	Services interrupt and sets flags according to algorithm detailed 
	in lab notebook.  This is a bit harry and might be coded in assembly
	language to speed up the process.  Hopefully, 9600 baud wont hurt
   things.

===================================================================== */

void interrupt far tans_isr(void)
{
	int char_in;		// holds character read form COM1:
	int ch_error;		// holds error bits, if non-zero, then error
	int	index, len;
//	_disable;

/* Get the character and the status (if error) of COM1: */
	char_in = inp(com1_base+com1_data);		// get character
	char_in = char_in & 0x00ff;  /* remove any sign extended bits */
	ch_error = inp (com1_base+com1_status) & 0x1E; // mask to any error bits (1-4)

/* see if there is a temporary buffer waiting, and if so, see if the
    packet is no longer in_use.  If not in use, copy it into the appropriate
    buffer (assumes valid ) 
	REV2: If the packet is 84, then use a revised length instead of the
	tans packet length because the status and stuff was appended to the*/
	if (tans_temporary_in_use)  /* check if buffer waiting */
	{  /* buffer waiting */
		index = tans_temporary_buffer[0] & 0xff;  /* watch char to int conversion */
		len = TANS_PACKET_LENGTH[index-TANS_ID_LOW];	/* length of buffer */
		if (index == 0x84) len = 69; /* rev2 -- include status, etc. */
		if (!tans_packet_data[index-TANS_ID_LOW].packet_in_use)
		{  /* packet is no longer in use, transfer over temporary */
			memcpy (  /* continued next lines */
			  tans_packet_data[index-TANS_ID_LOW].buffer_pointer,
			  &tans_temporary_buffer[1],
			  len );
			tans_packet_data[index-TANS_ID_LOW].new_packet_available=TRUE;
			tans_temporary_in_use = !TRUE;  /* reset temporar in_use */
		}  /* end if packet no longer in use */
	}  /* end if temporary_buffer in use */		

/* now act on received character depending upon state of system */
	switch (tans_state) { /* what type character are we awaiting? */
		case (TANS_NEW_PACKET):	/* awaiting start of new packet */
			if (char_in == DLE)
				{	/* start of packet received */
				tans_state = TANS_ID;  /* start of packet received */
				}
			else
				{  /* did'nt start a packet, error occured, try again */
				tans_errored_characters++;
				tans_error_1++;   /* error level 1 count */
				}  
			break;

		case (TANS_ID):		/* await packet id character */
			if ((char_in < TANS_ID_LOW) | (char_in > TANS_ID_HIGH))
				{  /* packet id out of range */
				tans_state = TANS_NEW_PACKET;
				tans_errored_characters++;
				tans_error_2++;
				break;  /* exit this loop */
				}
			if (TANS_PACKET_TYPE[char_in-TANS_ID_LOW] == TANS_INVALID)
				{  /* the id is a type that is invalid/illegal, error */
				tans_state = TANS_NEW_PACKET;
				tans_errored_characters++;
				tans_error_2++;
				break;  /* exit this loop */
				}
			else
				{  /* this packet is valid, save the packet id as first element
                   in the buffer */
				tans_inbuffer[0] = char_in;
				tans_buffer_size = 1;  /* one received character in packet */
				tans_expected_packet_size = TANS_PACKET_LENGTH[char_in-TANS_ID_LOW];
				tans_double_dle = !TRUE; /* double dle reset */
				tans_state = TANS_PACKET_DATA;
				}
			break;

		case (TANS_PACKET_DATA):	  /* awaiting packet data characters */
			if (tans_double_dle)
				{  /* this is second part of DLE, it must be a dle or else error */
					if (char_in == DLE)
						{	/* what we expected */
							tans_buffer_size++;  /* count now (so exit check works ) */
							tans_double_dle = !TRUE; /* double dle reset */
							/* go onto check if end-of-buffer reached yet */
						}
						else
						{  /* wasn't a DLE, error */
							tans_errored_characters++;
							tans_error_3++;   /* error level count */
							tans_state = TANS_NEW_PACKET;
							break;
						}  

				}
			else
				{  /* we have a new packet character, now process it */
					
					tans_inbuffer[tans_buffer_size] = char_in;
/* handling of DLE (0x10) is special, if the data is a 0x10, then it is
    sent twice.  Use the variable tans_double_dle to denote that we
    received the first DLE (0x10), and use special handling (as above) to
    see that a 2nd is sent.  Insert into buffer here, but count the char
    when the second one is received to keep the packet in sync.  */
					if (char_in == DLE)
						{ /* we have (first) DLE of two.  Do not count it yet. */
							tans_double_dle = TRUE; /* set that first DLE came accross*/
						}
					else
						{  /* not a DLE, count it now */
					tans_buffer_size++;  /* one more received character in packet */
						}	
				}
			if (tans_buffer_size > tans_expected_packet_size) /* allow ID in count */
				{  /* this was last character of packet, next state */
					tans_state = TANS_ENDING_DLE;
				}
			break;

		case (TANS_ENDING_DLE):		/* awaiting DLE at end of packet */
			if (char_in == DLE)
				{	/* we got terminatin DLE as expected */
				tans_state = TANS_ENDING_ETX;  
				}
			else
				{  /* did'nt start a packet, error occured, try again */
				tans_errored_characters++;
				tans_error_4++;   /* error level 1 count */
				tans_state = TANS_NEW_PACKET;  
				}  
			break;

		case (TANS_ENDING_ETX):	  /* awaiting ETX to terminate packet */
			if (char_in == ETX)
				{	/* we got terminatin ETX as expected */
					tans_state = TANS_NEW_PACKET;
					tans_valid_packets_received++;
					index = tans_inbuffer[0] & 0xff;  /* watch char to int. conversio */
					tans_packet_data[index-TANS_ID_LOW].number_packets_received++;
					if (TANS_PACKET_TYPE[index-TANS_ID_LOW] == TANS_VALID)
					{  /* packet is a valid one, store in appropriate buffer */
						len = TANS_PACKET_LENGTH[index-TANS_ID_LOW] ;

/* REV2: If the packet is a fix, append the contents of buffer 41, 44, and 46 to
	the buffer to indicate the fix's state at the time of fix */
						if (index == 0x84) {	/* double precision lat/lon */
							memcpy (&tans_inbuffer[37], &tans_id_41[0], 10); /* packet 41 */
							memcpy (&tans_inbuffer[47], &tans_id_44[0], 21); /* packet 44 */
							memcpy (&tans_inbuffer[68], &tans_id_46[0], 2); /* packet 46 */
							len = 69;  /* size of buffer */
						}	/* end case 84 */

						if (tans_packet_data[index-TANS_ID_LOW].packet_in_use)
						{  /* this packet is in_use, store in temporary buffer instead */
							memcpy (  /* continued next lines */
							  &tans_temporary_buffer[0],
							  &tans_inbuffer[0],
							  len + 1 );
							tans_temporary_in_use = TRUE;  /* set we're waiting */
						} /* endif packet_in_use */
						else
						{  /* store into regular buffer, not in_use */
							memcpy (  /* continued next lines */
							  tans_packet_data[index-TANS_ID_LOW].buffer_pointer,
							  &tans_inbuffer[1],
							  len );
							tans_packet_data[index-TANS_ID_LOW].new_packet_available=TRUE;
						}  /* end else */
					}  /* end transfer valid packet to buffer */
				} /* end got valid ETX */
			else
				{  /* did'nt start a packet, error occured, try again */
				tans_errored_characters++;
				tans_error_5++;   /* error level 5 count */
				tans_state = TANS_NEW_PACKET;  
				}  
				break;

	}  /* switch tans_state */

/* reset the uart for the next interrupt */
	outp (UART_ADDR1, 0x20);

}   // end tans_isr  interrupt service routine **************************

/* ================  tans_get_gps_status  ========================= */
/*  Public access to tans status through this routine.  Uses structure to
		return variables.		*/

int tans_get_gps_status (struct TANS_BUFFER *buffer)
{

	char	lbuffer[128];	/* read packets into this buffer */
	int	status, length;
	float f;

/* get packet 44, the current Satellite selection table */
	status = tans_get_packet(0x44, &length, lbuffer);

	buffer->c_mode = (unsigned int) lbuffer[0]; /* mode */
	buffer->c_sats[0] = (unsigned int) lbuffer[1];	/* satellite value */
	buffer->c_sats[1] = (unsigned int) lbuffer[2];	/* satellite value */
	buffer->c_sats[2] = (unsigned int) lbuffer[3];	/* satellite value */
	buffer->c_sats[3] = (unsigned int) lbuffer[4];	/* satellite value */
	float_copy (&buffer->c_pdop, &lbuffer[5]);  
	float_copy (&buffer->c_hdop, &lbuffer[9]);  
	float_copy (&buffer->c_vdop, &lbuffer[13]);  
	float_copy (&buffer->c_tdop, &lbuffer[17]);  

/* get packet 46, the health of the tans */
	status = tans_get_packet(0x46, &length, lbuffer);
	buffer->c_status = (unsigned int) lbuffer[0];	/* health code */

/* get the double precision lla packet, which includes the state at the
	time of the reading (41,44,46) */
	status = tans_get_packet(0x84, &length, lbuffer);
	if (status == TANS_NEW)
		buffer->new = TRUE; /* set if a new reading since last time */
	else
		buffer->new = !TRUE;

	double_copy (&buffer->last_lat, &lbuffer[0]);
	double_copy (&buffer->last_lon, &lbuffer[8]);
	double_copy (&buffer->last_alt, &lbuffer[16]);
	double_copy (&buffer->last_bias, &lbuffer[24]);
	float_copy (&buffer->last_tof, &lbuffer[32]);
/* packet 41 at tof */
	swab ( &lbuffer[40], (void *)buffer->last_week_number, 2);  /* gps week number */

/* packet 44 at tof */
	buffer->last_mode = (unsigned int) lbuffer[46]; /* mode */
	buffer->last_sats[0] = (unsigned int) lbuffer[47];	/* satellite value */
	buffer->last_sats[1] = (unsigned int) lbuffer[48];	/* satellite value */
	buffer->last_sats[2] = (unsigned int) lbuffer[49];	/* satellite value */
	buffer->last_sats[3] = (unsigned int) lbuffer[50];	/* satellite value */

	float_copy (&buffer->last_pdop, &lbuffer[51]);  
	float_copy (&buffer->last_hdop, &lbuffer[55]);  
	float_copy (&buffer->last_vdop, &lbuffer[59]);  
	float_copy (&buffer->last_tdop, &lbuffer[63]);  
/* packet 46 at tof */
	buffer->last_status = (unsigned int) lbuffer[67];
//	memcpy (&buffer->last_status, (void *) lbuffer[67], 1);  

/* convert to degrees */
	buffer->last_lat = buffer->last_lat * (180.0/PI);
	buffer->last_lon = buffer->last_lon * (180.0/PI);

/* finally set stale flag if c_status (current status) <> 0 */
	if (buffer->c_status != 0)
		buffer->stale = TRUE;
	else
		buffer->stale = !TRUE;

/* Setup the return variable, which is
	-1 if no all packets (41,44,46) have received at least valid packet,
	= c_status if all packets have been received once.
*/
	if (/*(tans_packet_data[01].number_packets_received == 0L) |*/ /* packet 41 */
		(tans_packet_data[04].number_packets_received == 0L) | /* packet 44 */
		(tans_packet_data[06].number_packets_received == 0L)) /* packet 46 */
			return (-1);	/* not all necessary packets received yet */
	else
		return (buffer->c_status);	/* return current_status */

}	/* end tans_get_gps_status */


/* ====================================================================
   INITIALIZE TANS SYSTEM    tans_init()
		initializes the lsr structure
		sets the baud rate (does not save previous speed, etc.)
		saves the old interrupt vector
		sets the new interrupt service vector as lsr_isr
		enables interrupts thought com1 port addresses
		enables UART interrupts directly (saving old register contents)
=======================================================================
*/

int tans_init(void)
/*  Initialize the data structure (where the packets are stored) and
     set the COM1 port to 9600,o,8,1 settings.  */
{
	int i;
	union REGS xr;		// define registers for dos call

/*  INITIALIZE TANS PACKET STRUCTURE  */
/*  Initialize the packet_data table:
     set number_of_packets received = 0
     set new_packet_available = !TRUE
     set packet_in_use = !TRUE
     set pointers to appropriate buffers
*/

	for (i=0; i<=TANS_ID_HIGH-TANS_ID_LOW; i++) { /* clear all IDs */
		tans_packet_data[i].number_packets_received = 0L;
		tans_packet_data[i].new_packet_available = !TRUE;
		tans_packet_data[i].packet_in_use = !TRUE;
//		tans_packet_data[i].buffer_pointer = 0;  /* set null */
		tans_packet_data[i].buffer_pointer = tans_junk_buffer;  /* set to tans_junk_buffer Rev #3 */
	} /* for i */
/* now set pointers for valid packets */
	tans_packet_data[1].buffer_pointer = &tans_id_41[0];	/* 41h  GPS time */
	tans_packet_data[2].buffer_pointer = &tans_id_42[0];	/* 42h  Single Precision XYZ */
	tans_packet_data[4].buffer_pointer = &tans_id_44[0];	/* 44h  Satellite selection, PDOP, mode */
	tans_packet_data[6].buffer_pointer = &tans_id_46[0]; 	/* 46h  health of TANS */
	tans_packet_data[8].buffer_pointer = &tans_id_48[0];	/* 48h  GPS System Message */
	tans_packet_data[9].buffer_pointer = &tans_id_49[0];	/* 49h  Almanac Health Page for all sats   */
	tans_packet_data[0xa].buffer_pointer = &tans_id_4a[0];	/* 4ah  single precision LLA position */
	tans_packet_data[0x16].buffer_pointer = &tans_id_56[0];	/* 56h  Velocity fix (ENU) */
	tans_packet_data[0x42].buffer_pointer = &tans_id_82[0]; 	/* 82h  Differential position fix mode */
	tans_packet_data[0x43].buffer_pointer = &tans_id_83[0];	/* 83h  double precision XYZ */
	tans_packet_data[0x44].buffer_pointer = &tans_id_84[0];	/* 84h  double precision LLA */
	tans_packet_data[0x45].buffer_pointer = &tans_id_85[0];	/* 85h  differential correction status */

/*  SET COM1 PORT */
/* set the baud rate use DOS_BIOS routine (only once, so slow is o.k.) */
	xr.h.ah = 0;		// init serial port function 0
	xr.h.al = 0xeb;	// 9600 baud, odd parity, 1 stop bit, 8 data bits
	xr.x.dx = 0;                     // serial port number 0 is com1:
	int86 (0x14, &xr, &xr);	// dos call to set serial port

/* now send packets to the tans to set the precision, lat/lon, and
   land type velocities */
	tans_init_unit();	/* initialize the unit */

/* Save the old interrupt vector, insert out vector into table */
	_disable();			// disable interrupts while we do interupt vector change
	com1_old_vector = _dos_getvect (com1_vector);  // get/save current vector
	_dos_setvect (com1_vector, tans_isr);	// insert int. srvc. rout. into table
	_enable();			// re-enable system interrupts

/* enable interrupts through COM1: ports (not sure if all these are necessary,
but appeared to be so when developing test program)  */
	i = inp (com1_base+com1_line);	// get current line status and reset MSB
	i = i & 0x7F;		// to disable baud rate divisor.
	outp (com1_base+com1_line, i);     // 

//        outp (com1_base+com1_enable, 0x05);  // interpt on rcvd character or error
//	outp (com1_base+com1_enable, 1);        // revised interrupt, data rdy
// Note, the 5 below was required to make it work with Compaq Portable.
	outp (com1_base+com1_enable, 5);        // enable interrupt, received line status 2/24/92

	outp (com1_base+com1_modem, 0x0B);      //set DTR/CTS, enable interrupts

/* now we must (by experiment) enable UART through this port */
	old_uart_register = inp(UART_ADDR2);  // get/save current UART register
	outp (UART_ADDR2, old_uart_register & 0xEF);  //force bit 4 low to enable

/* return some value */
	return (TRUE);
}   //  END tans_init *********************************************

tans_init_unit(void)
/* initilize the tans unit by sending two packets: 
		TANS_INIT_PACKET_2C[] 
		TANS_INIT_PACKET_35[] 
*/
{
	int	i;

/* packet 2C */
	for (i=0; i < 21; i++) {
		while ((inp(com1_base+com1_status) & 0x20) == 0)
			;  /* wait for UART ready for character bit is set = 1 (bit 5) */
		outp (com1_base, TANS_INIT_PACKET_2C[i]);  /* send the character */
	}

/* packet 35 */
	for (i=0; i < 8; i++) {
		while ((inp(com1_base+com1_status) & 0x20) == 0)
			;  /* wait for UART ready for character bit is set = 1 (bit 5) */
		outp (com1_base, TANS_INIT_PACKET_35[i]);  /* send the character */
	}
}  /* end tans_init_unit */

/* ==================================================================
		int tans_get_packet (int id, int &length, char packet_data[]);
			Returns the latest packet for packet type identified by 'id'.
			Returned value can be the following values(defined in tans_def.h):
				TANS_INVALID :   invalid packet id code;
				TANS_IGNORE :    packet id code is not stored;
				TANS_OLD :       packet id is valid -- no new reading, last packet contents returned
				TANS_NEW :       packet id is valid -- new reading, new packet contents returned
			If a new reading is available, the 'new' reading flag is reset after
			reading.
===================================================================*/
int	tans_get_packet (id, returned_length, packet_data_out)
	int	id;  /* packet id to lookup */
	int	*returned_length;  /* returns the length of returned packet */
	char	packet_data_out[];  /* packet buffer returned */
{
	int	return_code, len;

	if (id >= 0x40 && id < 0x85)
		return_code = TANS_PACKET_TYPE[id-TANS_ID_LOW];  /* lookup code type */
	else
		return_code = TANS_INVALID;

	if (return_code == TANS_VALID)
	{  /* this is valid code, see if it is new */
		tans_packet_data[id-TANS_ID_LOW].packet_in_use = TRUE; /* set in-use so interrupt doesn't change it */

		if (tans_packet_data[id-TANS_ID_LOW].new_packet_available)
		{  /* new packet is available */
			return_code = TANS_NEW;
		}
		else
		{  /* new packet not available */
			return_code = TANS_OLD;
		}
/* copy over the buffer into the return packet */
		len = TANS_PACKET_LENGTH[id-TANS_ID_LOW];
		if (id == 0x84) len = 69;	/* rev2, include status, etc */
		memcpy (  /* continued next lines */
		  &packet_data_out[0],
		  tans_packet_data[id-TANS_ID_LOW].buffer_pointer,
		  len );

		tans_packet_data[id-TANS_ID_LOW].new_packet_available = !TRUE; /* set no longer new */
		tans_packet_data[id-TANS_ID_LOW].packet_in_use = !TRUE; /* set not-in-use so interrupt can use */
//		*returned_length = TANS_PACKET_LENGTH[id-TANS_ID_LOW]; /* length of packet */
		*returned_length = len; /* length of packet */

	} /* end valid code, others just passed back */
	return (return_code);

}  /* end tans_get_packet */

/* ======================================================================
	RESTORE COM1 Settings
	Called upon termination of user program to set the com1: ports back to
original values so system is well behaved (hopefully!!!)
	1) resets the UART to the original uart value
	2) restores the original interrupt vector to the table
	NOTE: Does not reset the baud rate, etc.
========================================================================= */

int tans_restore()
{
/* restore the original UART register to the UART */
//	outp (com1_base+com1_modem, 0x03);      //set DTR/CTS, disable interrupts
//printf ("\n_restore - 1");
	_disable();		// disable interrupts while we do this
	outp (UART_ADDR2, old_uart_register);  // put the old back out there
//printf ("\n_restore - 2");

/* restore dos's interrupt vector table */
	_dos_setvect (com1_vector, com1_old_vector);  // restore original value
//printf ("\n_restore - 3");

	_enable();
/* return some value */
	return (TRUE);
}  //   END PROCEDURE lsr_restore ****************************************

float_copy (char *dest, char *source)
/*  Copy a floating point number in MSB, ... , LSB (as sent by the
     TANS) to the Microsoft C format LSB,...,MSB  */
{

	int	i;
/* move pointer to MSB (highest memory address */
	dest++;
	dest++;
	dest++;

	for (i=0; i < 4; i++) {

		*dest = *source;
		dest--;
		source++;
	}
} /* float_copy */

double_copy (char *dest, char *source)
/*  Copy a double point number in MSB, ... , LSB (as sent by the
     TANS) to the Microsoft C format LSB,...,MSB  */
{

	int	i;
/* move pointer to MSB (highest memory address */
	dest++;
	dest++;
	dest++;
	dest++;
	dest++;
	dest++;
	dest++;

	for (i=0; i < 8; i++) {

		*dest = *source;
		dest--;
		source++;
	}
} /* double_copy */





