/*  tans_def.h    Definition of variables used the interface to
                   the Trimble TANS series of GPS receivers.
                  Started 2/13/92 csp                                  
                  Rev 1 2/18/92  Added buffers and pointers for valid packets.
						Rev 2 2/19/92  Added TANS_NEW and TANS_OLD for _get_packet.
						Rev 3 4/02/92  Added structure for new user access
						Rev 4 4/04/92  Another method of new user access
						Rev 5 4/12/92  added junk buffer, see Revision #3 note in tans_prog.c
*/

/* define pi per the GPS standard */
#define	 PI	3.1415926535898

/* define the standard access structure for user to access through routine
	"tans_get_gps_status".	*/

struct	TANS_BUFFER	{
/* Flags about the last known state */
	int	stale;	/* last known position is now stale (unreliable) */
	int	new;	/* new position since last read */

/* current status (what is currently going on in tans, see last_* below for
		the last known position_ */
	int	c_status;	/* current status (packet 46 - Health) */
	int	c_mode;	/* current mode (auto/manual 0, 2, or 3_D) packet 44 */
	int	c_sats[4];	/* current list of satellites that can be seen */
	float	c_pdop;	/* current pdop */
	float	c_hdop;	/* current hdop */
	float	c_vdop;	/* current vdop */
	float	c_tdop;	/* current tdop */

/* last known position (along with satellites, mode, etc when fix obtained */
	double	last_lat;	/* last known latitude (degrees) [84] */
	double	last_lon;	/* last known longitude (degrees) [84] */
	double	last_alt;	/* last known altiude (meters) [84] */
	double	last_bias;	/* meters of bias [84] */
	float		last_tof;	/* time of fix (seconds) [84] */
	int	last_status;	/* last status (packet 46 - Health ) */
	int	last_week_number;	/* week number at last fix (packet 41) */
	int	last_mode;	/* mode at fix (auto/manual 0, 2, or 3_D) packet 44 */
	int	last_sats[4];	/* list of satellites from which fix obtained */
	float	last_pdop;	/* fix time pdop */
	float	last_hdop;	/* fix time hdop */
	float	last_vdop;	/* fix time vdop */
	float	last_tdop;	/* fix time tdop */
};	/* end structure TANS_BUFFER */

/* packet ID types :
               invalid means the TANS cannot send this packet id
               ignore means the TANS can send this code, but we do not 
                save packet contents
               valid means the TANS can send this code, and we do save
                the packet contents.                                 */
            
#define	TANS_INVALID -1   /* this packet type will not be sent by TANS */
#define	TANS_IGNORE 0	/* packet type can be sent, but do not save contents */
#define	TANS_VALID 1  /* save packet contents */
#define	TANS_OLD 2  /* return code from tans_get_packet, no new packet avail */
#define	TANS_NEW 3  /* return code from tans_get_packet, new packet avail */

/* define the range of packet IDs valid in Tans to user packets */
#define  TANS_ID_LOW 0x40  /* base packet ID sent from TANS */
#define  TANS_ID_HIGH 0x85  /* highest packet ID sent from TANS */

/* this is table of how many bytes are contained in each packet by ID */
static int TANS_PACKET_LENGTH[] =
/*  x0  x1  x2  x3  x4  x5  x6  x7  x8  x9  xA  xB  xC  xD  xE  xF  */
{   39, 10, 16, 20, 21, 10,  2,  0, 22, 32, 20,  3,  0,  4,  1,  0,  /* 40-4F */
     0,  0,  0, 32, 12,  4, 20,  8,  0,  0, 25, 16, 24,  0,  0,  0,  /* 50-5F */
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 60-6F */
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 70-7F */
     0,  0,  1, 36, 36, 22, 0 /* 80-86 */
};  /* end packet_lengths */

/* this table defines if the packet ID is valid, ignored, or invalid. */
/*  make temporary definitions to simplify populating the table */
#define	X  TANS_INVALID /* set "X" = invalid ID type */
#define	I	TANS_IGNORE  /* set "i" = ignore ID type */
#define	VL  TANS_VALID /* set "V" = valid ID type */
//#define	V  TANS_VALID /* set "V" = valid ID type */
static int TANS_PACKET_TYPE[] =
/*  x0  x1  x2  x3  x4  x5  x6  x7  x8  x9  xA  xB  xC  xD  xE  xF  */
{    I,  VL,  VL,  I,  VL,  I,  VL,  X,  VL,  VL,  VL,  I,  X,  I,  X,  X,  /* 40-4F */
     X,  X,  X,  X,  I,  I,  VL,  I,  X,  X,  I,  I,  I,  X,  X,  X,  /* 50-5F */
     X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  /* 60-6F */
     X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  X,  /* 70-7F */
     X,  X,  VL,  VL,  VL,  VL,  X /* 80-86 */
};  /* end packet_types */
/* remove temporary definitions */
#undef	X
#undef	I
#undef	VL
//#undef	V

/* Packet Table of what packets have been received (per ID):
     Number of packets received (long)
     New Packet Available (if TRUE)
     Packet in use by application program (if TRUE) -- Note: this 
      condition can occur when the application program, through the
      routine "tans_request_packet" , is requesting the contents of this
      ID.  When requesting, we don't want the interrupt service routine
      updating the buffer when the packet is in use.
     Time last packet received (not implemented at this time )
     Pointer to packet buffer (implemented later )
*/

struct TANS_RECORD {
	long number_packets_received; /* number of packets received */
	int	new_packet_available; /* new packet is available */
	int   packet_in_use; /* packet in use by application "tans_request_packet" */
	char	*buffer_pointer;  /* points to location of buffer (valid pckts only) */
};

struct TANS_RECORD tans_packet_data[TANS_ID_HIGH-TANS_ID_LOW+1];

/* running statistics, such as errors, etc. */
	long	tans_valid_packets_received = 0L; /* number valid pckts rcvd */
	long tans_errored_characters = 0L;  /* total error characters received */
	long	tans_error_1 = 0L;	/* level 1, awaiting new packet */
	long	tans_error_2 = 0L; /* level 2, invalid packet ID received */
	long	tans_error_3 = 0L; /* level 3, expecting 0x03, got different */
	long	tans_error_4 = 0L; /* level 4, expecting terminating DLE */
	long	tans_error_5 = 0L; /* level 5, expecting terminating ETX */

/* packet buffers which store packets at differing points in their lives */
char	tans_inbuffer[48];  /* incoming packets stored here */
int	tans_buffer_size;  /* current number of characters in packet */
int	tans_expected_packet_size;  /* size of upcoming packet (based on id) */
char	tans_temporary_buffer[75];  /* store packet here until non in_use */
//int 	tans_temporary_buffer_length;  /* size of temporary packet */

char	tans_id_41[10];	/* 41h  GPS time */
char	tans_id_42[16];	/* 42h  Single Precision XYZ */
char	tans_id_44[21];	/* 44h  Satellite selection, PDOP, mode */
char	tans_id_46[2];		/* 46h  health of TANS */
char	tans_id_48[22];	/* 48h  GPS System Message */
char	tans_id_49[32];	/* 49h  Almanac Health Page for all sats   */
char	tans_id_4a[20];	/* 4ah  single precision LLA position */
char	tans_id_56[20];	/* 56h  Velocity fix (ENU) */
char	tans_id_82[1];		/* 82h  Differential position fix mode */
char	tans_id_83[36];	/* 83h  double precision XYZ */
char	tans_id_84[70];	/* 84h  double precision LLA (rev2) */
char	tans_id_85[22];	/* 85h  differential correction status */
char	tans_junk_buffer[16]; /* revision 5, will this eliminate R6001 runtime error? */

/* states of the TANS packet routines */
int	tans_state;	/* state for next incoming character */
int	tans_double_dle;  /* for the case where data is a dle (0x10) which
                            the tans send twice (reference TANS manual) */
int	tans_temporary_in_use = !TRUE;  /* set when temporary buffer in-use */


