//  IFR_LIB.C
//
//   Library routines for IFR CSA1600.
//
// 	Note: TMACS must be stored and loaded in IFR before running program
// 			TMACS are located in directory c:\ifr_new\tmacs
// 			Store TMACS in IFR if not currently stored
// 			Run program load_mac to load TMACS fdinit(), fdreg2(), fdaudit() 
// 				fdinit initializes fdcch w/ correct access parameters
// 				fdreg2 checks to see if phone has registered on power up
// 				fdaudit sends out an audit and waits for audit confirmation & MACA report
//
//	  Routines:
//		void ifr_init_FDCCH(int ud, ACCESS_PARAMS *acc, int ch_num) initializes access parameters and sends BCCH
//		int ifr_ask_RDCCH_msg(int ud, char msg[], char rsp[]) sends msg to IFR and waits for response (assumes downlink faded)
//		int ifr_ask2_RDCCH_msg(int ud, char msg[], char rsp[], MACA_PARAMS *maca) sends msg to IFR and waits for response (assumes uplink faded)
//		int ifr_ask_mess_rcvd(int ud, char response[], char confirmation[]) returns TRUE if received msg = expected response
//		int registered(int ud, int burst) returns true if phone registers
//		double decode_WER ( int encoded_WER )  returns decoded WER
//		double decode_BER ( int encoded_BER )  returns decoded BER
//		double decode_RSS (double encoded_RSS)	returns decoded RSS
//		ifr_rebuild_bcch(int ud, ACCESS_PARAMS *acc)  rebuilds DCCH w/ new access params
//		ifr_regaccept(int ud) send registration accept message to mobile
//		ifr_load_fdreg(int ud) loads tmac fdreg2 into IFR memory
//		ifr_send_bcch(int ud, ACCESS_PARAMS *acc, int ch_num)	sends BCCH message to IFR

// Define control parameters
#define	TRUE	1
#define	FALSE	0
#define  LOG_TO_LN 2.30258	 	// conversion from log(x) to ln(x)
#define  TIMEOUT_SEC 10		   // max time to wait between audits
#define MAX_MESSAGE_LENGTH 128  // length of gpib message response
#define  IFR_DEV 8					/* IFR controller */

/* TYPEDEFS */

/* Access Parameters */
typedef struct {
					  int	burst;
                 int MaxRetries;
                 int	MaxBusy;
                 int	MaxReps;
                 int	MaxStop;
                 int	DelayInterval;
               } ACCESS_PARAMS;

/* Maca/Audit Reports */
typedef struct {
						int passed;
						double avg_BER;
						double avg_WER;
						double avg_RSS;
						int audit_conf_cnt;
						int rpt_cnt;
						double audit_conf_fail_rate;
						double rpt_cnt_fail_rate;
						double SPACH_conf_fail_rate;
						double rdata_acpt_fail_rate;
						double max_audit_conf_fail_rate;
						int synch_fail_cnt;
					}MACA_PARAMS;

void ifr_init_FDCCH(int ifr_ud, ACCESS_PARAMS *acc, int ch_num)
{
/* build DCCH with appropriate access parameters */

	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	char message[24];	 /* message string to send to IFR */
	int mask = TIMO;	 /* timeout mask */
 
   printf ( "DCCH CHANNEL = %d\n", ch_num);

	/* wait timeout period to setup DCCH */
	gpib_tmo(ifr_ud,TIMEOUT_SEC);

	/* load bcch initialization macros */
	ifr_load_fdinit(ifr_ud);

	/* broadcast bcch */
	ifr_send_bcch(ifr_ud, &acc, ch_num);

	ifr_ud = gpib_open(ifr_dev);		// get the ud for the IFR 

	/* load registration/audit macros */
	ifr_load_fdreg(ifr_ud);

}


int ifr_ask_RDCCH_msg(int ifr_ud, char message[], char response[]){
/* assumes forward link faded */
	int mask = TIMO | RQS;					// gpib timeout or request to send
	int resp_len = MAX_MESSAGE_LENGTH;	// message length
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int status;
	int mask_recvd;
	int error;
	char ibrsp_msg[MAX_MESSAGE_LENGTH];

	ifr_ud = gpib_open(ifr_dev);		// get the ud for the IFR 

	gpib_tmo(ifr_ud, 30);

	/* send message to IFR */
	strset(response,' ');
   gpib_write(ifr_ud, message);

	/* wait for 30 secs or Request to Send from IFR */
	mask_recvd = gpib_wait(ifr_ud,mask);

	if(mask_recvd == RQS_CMPL){
		gpib_write(ifr_ud, "disp_msgs");

		/* record response from IFR */
		gpib_read(ifr_ud,response,resp_len);
		error = FALSE;
	}
	else{
			error = TRUE;
	}

	ifr_regaccept(ifr_ud);

	/* change mask to timeout */
	gpib_tmo(ifr_ud, 3);
	mask = TIMO;

	mask_recvd = gpib_wait(ifr_ud,mask);

	/* serial poll gpib device until rqs cleared */
  	while(mask_recvd == END_RQS_CMPL | mask_recvd == TIMO_END_RQS_CMPL){
		printf("RQS\\");
		ibrsp(ifr_ud,ibrsp_msg); 
		mask_recvd = gpib_wait(ifr_ud,mask);
	}

	return(!error);
	
}

int ifr_ask2_RDCCH_msg(int ifr_ud, char message[], char response[], MACA_PARAMS *maca){
/* assumes reverse link faded */
	int mask = TIMO;					// gpib timeout or request to send
	int resp_len = MAX_MESSAGE_LENGTH;	// message length
	int ifr_dev = IFR_DEV;		// GPIB logical device number for IFR 1600CSA
	int status;
	int mask_recvd;
	int error = FALSE;
	int iwait=0;
	int exit_loop = FALSE;
	char ibrsp_msg[MAX_MESSAGE_LENGTH];

	ifr_ud = gpib_open(ifr_dev);		// get the ud for the IFR 

	gpib_tmo(ifr_ud, 10);

	/* send message to IFR */
	strset(response,' ');
   gpib_write(ifr_ud, message);

	while (!exit_loop & (iwait < 3)){
  		gpib_write(ifr_ud, "disp_msgs");

  		/* record response from IFR */
  		gpib_read(ifr_ud,response,resp_len);

	   if(parse_confirmation_msg(ifr_ud, response, maca)){
				exit_loop = TRUE;
		}
		else {
			/* wait for 1 sec */
			mask_recvd = gpib_wait(ifr_ud,mask);
			printf("%d %s\n", iwait, response);
			iwait++;
		}
	}

	ifr_regaccept(ifr_ud);

	/* change mask to timeout */
	gpib_tmo(ifr_ud, 3);
	mask = TIMO;

	mask_recvd = gpib_wait(ifr_ud,mask);

	/* serial poll gpib device until rqs cleared */
  	while(mask_recvd == END_RQS_CMPL | mask_recvd == TIMO_END_RQS_CMPL){
//		printf("RQS\\");
		ibrsp(ifr_ud,ibrsp_msg); 
		mask_recvd = gpib_wait(ifr_ud,mask);
	}

	return(!exit_loop);
	
}



int ifr_ask_mess_rcvd(int ifr_ud, char response[], char confirmation[]){
	int received = FALSE;
	char *result;

	/* check to see if response contains expected string */

   if((result = strstr(response,confirmation)) != NULL)
			received = TRUE;
	return(received);
}

int kb_exit_loop(void){

/* check for keyboard input and exit loop if key hit */
	int exit_loop = FALSE;
	int i;

	/* Check for keyboard input */
		if(kbhit())
		{
			i = getch();
			if (i == 'q'){
				exit_loop = TRUE;
			}
			else
				exit_loop = TRUE;
		} /* keyboard input */

	return(exit_loop);
}			

int registered(int ifr_ud, int burst){

	int exit_loop = FALSE, registered = FALSE;
	char message[MAX_MESSAGE_LENGTH];
	char response[MAX_MESSAGE_LENGTH];
	int resp_len = MAX_MESSAGE_LENGTH;
	char result;

	/* power up and register phone */
    printf( " Power up the phone, please.\n ");
    printf( " Hit any key to continue.\n" );

   while (!exit_loop){
		/* Check for keyboard input */
		exit_loop = kb_exit_loop();
	}

    printf( " Waiting for Camping then Registration.\n");
  
	 /* Loop through until phone registers or users hits key */
    sprintf(message,"fdreg2 %d", burst);

   if (ifr_ask_RDCCH_msg (ifr_ud, message, response))
	{
		/* check for registration message */
		if(ifr_ask_mess_rcvd(ifr_ud, response,"REGISTRATION")){
			registered = TRUE;
			printf( " Phone registered.\n");
		}
	}
	else
		registered = FALSE;
				
	return(registered);
}

double decode_WER ( int encoded_WER ){
	double  WER;
	// WER field is coded as follows
	// 0 		 0%
	// 1 		>0-2%
	// 2		>2-4%
	// 3		>4-8%
	// 4		>8-16%
	// 5		>16-32%
	// 6		>32-64%
	//	7		>64%

	/* WER coded from midpoint of estimated range */
	switch (encoded_WER)
	{
		case 0: WER = 0.0;
				  break;
		case 1: WER = 1.0;
				  break;
		case 2: WER = 3.0;
				  break;
		case 3: WER = 6.0;
				  break;
		case 4: WER = 12.0;
				  break;
		case 5: WER = 24.0;
				  break;
		case 6: WER = 48.0;
				  break;
		case 7: WER = 64.0;
				  break;
		default: WER = -99;
					break;
	}

	return(WER);
}

double decode_BER ( int encoded_BER ){
	double  BER;

	// BER field is coded as follows
	// 0 		<.01%
	// 1 		.01 -< .1%
	// 2		.1 - < .5%
	// 3		.5 -< 1.0%
	// 4		1.0 -< 2.0%
	// 5		2.0 - < 4.0%
	// 6		4.0 - < 8.0%

	/* BER coded from midpoint of estimated range */
	switch (encoded_BER)
	{
		case 0: BER = 0.0;
				  break;
		case 1: BER = 0.05;
				  break;
		case 2: BER = 0.3;
				  break;
		case 3: BER = .75;
				  break;
		case 4: BER = 1.5;
				  break;
		case 5: BER = 3.0;
				  break;
		case 6: BER = 6.0;
				  break;
		default: BER = -99;
					break;
	}

	return(BER);
}

double decode_RSS (double encoded_RSS){
	double RSS_pow; 		/* received signal strength (pow) */
	double RSS_dBm;			/* received signal strength (dBm) */

	/* encode dBm */
	RSS_dBm = -113.0 + encoded_RSS*2;	 

	/* convert to power */
	RSS_pow = exp(LOG_TO_LN*RSS_dBm/10);
	return (RSS_pow);

}


ifr_rebuild_bcch(int ifr_ud, ACCESS_PARAMS *acc){

	int mask = TIMO;	 /* timeout mask */
	char message[40];	 /* message string to send to IFR */


	/* initialize command string */
	sprintf(message,"reset_access %d,%d,%d,%d,%d,%d\n", acc->MaxRetries, acc->MaxReps, 
	acc->MaxBusy, acc->MaxStop, acc->DelayInterval, acc->burst);

	/* send command to IFR to update access parameters*/
	gpib_write(ifr_ud, message);

	/* send command to IFR to rebuild bcch */
	gpib_write(ifr_ud, "rebuild_bcch");

}

ifr_regaccept(int ifr_ud){

	gpib_write(ifr_ud, "reg_accept");

}
ifr_load_fdreg(int ifr_ud){

	int mask= TIMO;

	gpib_tmo(ifr_ud, 3);

	/* load macro to perform registration */
	gpib_write(ifr_ud,":mmem:load:macro \"\", \"fdreg\" ");

 	/* wait timeout period before sending audit */
	gpib_wait(ifr_ud, mask);


}

ifr_load_fdinit(int ifr_ud){

	int mask = TIMO;

	gpib_tmo(ifr_ud, 3);

	/* load macro to perform dcch initialization */
	gpib_write(ifr_ud,":mmem:load:macro \"\", \"fdinit\" ");

	/* wait timeout period before sending audit */
	gpib_wait(ifr_ud, mask);

}

ifr_send_bcch(int ifr_ud, ACCESS_PARAMS *acc, int ch_num){
	int mask = TIMO;
	char message[MAX_MESSAGE_LENGTH];

	/* initialize command string */
	sprintf(message,"fdinit %d,%d,%d,%d,%d,%d,%d\n", ch_num, acc->MaxRetries, acc->MaxReps, 
	acc->MaxBusy, acc->MaxStop, acc->DelayInterval, acc->burst);

	/* send command to IFR */
	gpib_write(ifr_ud, message);
	printf ("Setting up DCCH, please wait ...\n");

	/* wait timeout period to setup DCCH */
	gpib_wait(ifr_ud,mask);

}
