/*   HP11759B.C
 *
 *   GPIB_Routines for the HP 11759B Fading Simulator under control
 *    of the GPIB to Serial "Black Box" (I/O TECH) converter.  The
 *    fading simulator must use the "rchansim" program (already running).
 *
 *   SEE NOTES BELOW for running the channel simulator and the switch
 *    settings required on the Black Box.
 *
 *	  The Black Box settings are defined as follows:
 *			Baud Rate : 300
 *			Data Bits : 7
 *			Stop Bits : 1
 *			Parity Disabled
 *			Echo Disabled
 *			Serial Handshake : RTS/CTS
 *			Serial Terminator : CR-LF
 *			EOI Enabled
 *			IEEE-488 Address : 11
 *			Peripheral Pass-Thru Operation (IEEE-488 to RS-232)
 *			SRQ on Terminator Disabled
 *
 *
 *   Started 3/6/97  csp
 *   Source: HP8657b.c (under TMA)
 *
 *   Procedures:
 *		hp11759b_check_OK_primitive (int ud) : Checks for the "OK" response from the fading simulator.
 *		hp11759b_set_simulation (int ud) : Sets the channel simulator in simulation mode.
 *		hp11759b_recall_user_profile (int ud, int n) : recall user profile "n"
 *		hp11759b_recall_standard_profile (int ud, int n) : recall standard profile "n"
 *		hp11759b_set_main_menu (int ud) : return to main menu
 *		hp11759b_exit_program (int ud) : stop remote channel simulation
 *
 *
*/

#define HP11759B_MAX_BUFFER_SIZE 512

/* Be sure TRUE is defined */
#ifndef TRUE
#define TRUE 1
#endif



int hp11759b_check_OK_primitive(int dev)
/* Checks for the "OK" response from fading simulator after implementing a word command.
 * Input parameter is defined as follows:
 *			dev = gpib device number.
 *	Function returns(1) if successful, returns(0) if errors occured.
 */
 {
	 int len;
	 char	reply_buffer[HP11759B_MAX_BUFFER_SIZE];

	 /* Be sure we get the OK response */
	 len = gpib_read (dev, reply_buffer, HP11759B_MAX_BUFFER_SIZE);
	 reply_buffer[len] = '\0';		// terminate string properly

	 /* See if it is "OK" */
	 if(!strncmp(reply_buffer,"OK",2))
	   return (TRUE);

	 return(!TRUE); // return(0) if errors occured.
}


int hp11759b_set_simulation(int dev)
/* Sets the fading simulator to the simulation menu. 
 *	Input parameter is defined as follows:
 *			dev = gpib device number.
 * Function returns(1) if successful, returns(0) if errors occured.
 */
{

	 int len;
	 char reply_buffer[HP11759B_MAX_BUFFER_SIZE];

	 gpib_write(dev,"MENU?\n");    // Query the current menu
	 len = gpib_read (dev, reply_buffer, HP11759B_MAX_BUFFER_SIZE);
	 reply_buffer[len] = '\0';		// terminate string properly

	 if(!strncmp(reply_buffer,"MAIN",4))  // Determine if "MAIN MENU" is the current menu 
	 {
		gpib_read(dev,	reply_buffer, HP11759B_MAX_BUFFER_SIZE); // Flush "OK" from buffer
		gpib_write (dev, "MENU SIMULATION\n");	 // set fading simulator to SIMULATION MENU
		return(hp11759b_check_OK_primitive(dev));	 // Check for "OK" response
	 }

	 else if(!strncmp(reply_buffer, "SIMULATION",10))  // Determine if "SIMULATION MENU" is the current menu
	 {
		gpib_read(dev,	reply_buffer, HP11759B_MAX_BUFFER_SIZE); // Flush "OK" from buffer
		return(TRUE);
	 }
	 else
		return(!TRUE);	// return(0) if errors occured
}


int hp11759b_set_main_menu(int dev)
/* Sets the fading simulator to the MAIN MENU. 
 *	Input parameter is defined as follows:
 *		   dev = gpib device number.
 * Function returns(1) if successful, returns(0) if errors occured. 
 */
{
	 int len;
	 char reply_buffer[HP11759B_MAX_BUFFER_SIZE];

	 gpib_write(dev,"MENU?\n"); // Query the current menu
	 len = gpib_read (dev, reply_buffer, HP11759B_MAX_BUFFER_SIZE);
	 reply_buffer[len] = '\0';	

	 if(!strncmp(reply_buffer,"MAIN",4)) // Determine if "MAIN MENU" is the current menu
	 {
		gpib_read(dev,	reply_buffer, HP11759B_MAX_BUFFER_SIZE); // Flush "OK" from buffer
		return(TRUE);
	 }
	 else if(!strncmp(reply_buffer, "SIMULATION", 10))	// Determine if "SIMULATION MENU" is the current menu
	 {
		gpib_read(dev,	reply_buffer, HP11759B_MAX_BUFFER_SIZE); // Flush "OK" from buffer
		gpib_write(dev, "EXIT\n");	// Set fading simulator to the "MAIN MENU"
		return(hp11759b_check_OK_primitive(dev));	// Check for "OK" response
	 }	
		
     else
	   return(!TRUE);	// Return(0) if errors occured.
}


int hp11759b_recall_user_profile(int dev, int profile_num)
/* Recalls a user defined profile. The current menu must be the "SIMMULATION MENU".
 *	Input parameters are defined as follows:
 *		   dev = gpib device number
 *			profile_num = user defined profile number.
 *	Function returns(1) if successful, returns(0) if errors occured. 
 */
{
	  
	 char reply_buffer[HP11759B_MAX_BUFFER_SIZE], command_string[32];
	 /* Load command_string with word command "URECALL" */
	 sprintf(command_string, "%s %d\n", "URECALL", profile_num);

	 gpib_write(dev, command_string); // Implement word command "URECALL"
	 return(hp11759b_check_OK_primitive(dev)); // Check for "OK" response.
}


int hp11759b_recall_standard_profile(int dev, int profile_num, float hz_rayleigh)
/* Recalls a standard profile. The current menu must be the "SIMMULATION MENU".
 *	Input parameters are defined as follows:
 *		   dev = gpib device number
 *			profile_num = user defined profile number.
 *	Function returns(1) if successful, returns(0) if errors occured. 
 */

{
	 char reply_buffer[HP11759B_MAX_BUFFER_SIZE], command_string[32];
	 /* Load command_string with word command "SRECALL" */ 
	 sprintf(command_string, "%s %d %d\n", "SRECALL", profile_num, hz_rayleigh);

	 gpib_write(dev, command_string); // Implement word command "SRECALL"
	 return(hp11759b_check_OK_primitive(dev)); // Check for "OK" response
}


int hp11759b_exit_program(int dev)
/* Halts remote channel simmulation.
 * Input parameter is defined as follows:
 *			dev = gpib device number.
 * Function returns(1) if successful, returns(0) if errors occured.
 */
{
	 int len;
	 char reply_buffer[HP11759B_MAX_BUFFER_SIZE];

	 gpib_write(dev,"MENU?\n"); // Query the current menu
	 len = gpib_read (dev, reply_buffer, HP11759B_MAX_BUFFER_SIZE);
	 reply_buffer[len] = '\0';	

	 if(!strncmp(reply_buffer, "MAIN",4)) // Check if "MAIN MENU" is the current menu.
	 {
		gpib_write(dev, "EXIT\n");	// Exit program
		return(hp11759b_check_OK_primitive(dev));	// Check for "OK" response
	 }
	 else if(!strncmp(reply_buffer, "SIMULATION",10)) // Check if "SIMULATION MENU" is the current menu.
	 {
		gpib_write(dev, "EXIT\n");	// "Exit" from "SIMULATION MENU"

		switch(hp11759b_check_OK_primitive(dev)) // Check for "OK" response
		{
			case TRUE:
				gpib_write(dev, "EXIT\n");	// "Exit" from program
				return(hp11759b_check_OK_primitive(dev));	// Check for "OK" response
				  
			case !TRUE:	// errors occured
				return(!TRUE);
		}
	 }

	 
		return(!TRUE);	// errors occured
}

/**************************************************************************/
#ifdef SKIP_EXAMPLES

hna_trigger (ud)		// trigger a sweep
/* Triggers a sweep (typically is already in single sweep mode)
 *  and returns when the sweep is complete */
int	ud;
{
	int len;
	char	reply_buffer[16];

	gpib_write (ud, "OPC?;SING;");

	len = gpib_read (ud, reply_buffer, 16);
	reply_buffer[len] = '\0';		// terminate string properly
// printf ("hna_trigger:Number characters read = %d (%s)\n", len, reply_buffer); getch();
}		// end hna_trigger

void hp8657b_mod_cw (int dev)
/* Set the modulation OFF (i.e. CW) */
{
	char	string[64];		  
	sprintf (string, "AM S4");
	gpib_write (dev, string);
}

void hp8657b_rf_on (int dev)
/*     Turn RF ON  */
{
	char	string[64];		  
	sprintf (string, "R3");
	gpib_write (dev, string);
}

void hp8657b_rf_off (int dev)
/*     Turn RF OFF  */
{
	char	string[64];		  
	sprintf (string, "R2");
	gpib_write (dev, string);
}


void hp8657b_freq (int dev, double freq)
/*  Set CW frequency (input in MHz) */
{
	char	string[64];		  
	sprintf (string, "FR %.8lf MZ", freq);
	gpib_write (dev, string);
}


void hp8657b_level_dBm (int dev, double amplitude)
/*  Set power level in dBm */
{
	char	string[64];		  
	sprintf (string, "AP %.1lf DM", amplitude);
	gpib_write (dev, string);
}


void hp8657b_reset (int dev)
/* Reset to factory settings */
/* NOTE: Had to move this after the first define of hp8657b_rf_off to keep
 *  the turbo c compiler from complaining to me. */
{
	ibclr (dev);   /* This resets the unit */
	hp8657b_rf_off (dev);  /* be sure it is off! */
}			  
/* IGNORE THE FOLLOWING CODE, IT WAS FROM THE FLUKE GENERATOR AND
 *  MAY BE USEFUL FOR EXPANSION OF HP8657B CODE 
*/

// .c
//
//  Modified for the 488 routines // NON ULI (universal language interface)
//
//
//This file contains the functions for setting the Fluke
//6060B RF Signal Generator

#include <stdio.h>

// defines for fluke signal generators
#define	ON	1
#define	OFF 0
#define	HZ400 0
#define	HZ1K 1
#define	NONE 0
#define	INTERNAL 1
#define	EXTERNAL 2


fsg_freq (dev, freq)				 //Set Freq
	int dev;							//device
	double freq;					//frequency in Hz
{
	char	string[64];
	sprintf (string, "FR%10.9eHZ", freq);
	gpib_write (dev, string);
}										//end fsg_freq

fsg_ampdb (dev, value)			// Set amp. in dB
	int dev;							//device
	double value;				  //amplitude in dB
{
	char	string[64];
	sprintf (string, "AP%6.1fDB", value);
	gpib_write (dev, string);
//	fprintf(gpib_out,"Output %d;AP%6.1fDB\n",dev,value);
}										//end fsg_ampdb


fsg_ampV (dev,value)				//Set amp. in uVolts
	int dev;
	double value;
{
	char	string[64];
	sprintf (string, "AP%6.5fV", value);
	gpib_write (dev, string);
//	fprintf(gpib_out, "Output %d;AP%6.5fV\n",dev,value);
}										//end fsg_ampV


fsg_rf (dev,value)				//Set RF output On or OFF
	int	dev,value;
{
	char	string[64];		  
	sprintf (string, "RO%1d", value);
	gpib_write (dev, string);
//	fprintf (gpib_out,"Output %d;RO%1d\n", dev, value);
}										//end fsg_rf

fsg_fixmodfreq (dev,value)		//Set internal modulation			
	int dev, value;
{
	char	string[64];		  
	sprintf (string, "MR%1d", value);
	gpib_write (dev, string);
//	fprintf (gpib_out,"Output %d;MR%1d\n", dev, value);
}										//end fsg_fixmodfreq


fsg_ammod (dev,value)			//Set AM modulation
	int dev, value;
{

	switch (value)
		{
			case NONE:
						gpib_write (dev, "AI0");
						gpib_write (dev, "AE0");
//						fprintf(gpib_out,"Output %d;AI0\n",dev);	
//						fprintf(gpib_out,"Output %d;AE0\n",dev);
						break;
			case INTERNAL:
						gpib_write (dev, "AI1");
						gpib_write (dev, "AE0");
//						fprintf(gpib_out,"Output %d;AI1\n",dev);
//						fprintf(gpib_out,"Output %d;AE0\n",dev);
						break;
			case EXTERNAL:
						gpib_write (dev, "AI0");
						gpib_write (dev, "AE1");
//						fprintf(gpib_out,"Output %d;AI0\n",dev);
//						fprintf(gpib_out,"Output %d;AE1\n",dev);
						break;
		}

}											//end AM modulation


fsg_amdepth (dev,value)				//Set AM depth in %
	int dev;
	double value;
{
	fprintf(gpib_out,"Output %d;AM%3.2fPC\n", dev, value);
//	printf("Output %d;AM%3.2fPC\n", dev, value);
	fflush (gpib_out);
}											//end AM modulation

fsg_fmmod (dev,value)					//Set FM modulation
	int dev, value;
{
	switch (value)
		{
			case NONE:
						gpib_write (dev, "FI0");
						gpib_write (dev, "FE0");
//						fprintf(gpib_out,"Output %d;FI0\n",dev);	
//						fprintf(gpib_out,"Output %d;FE0\n",dev);
						break;
			case INTERNAL:
						gpib_write (dev, "FI1");
						gpib_write (dev, "FE0");
//						fprintf(gpib_out,"Output %d;FI1\n",dev);
//						fprintf(gpib_out,"Output %d;FE0\n",dev, value);
						break;
			case EXTERNAL:
						gpib_write (dev, "FI0");
						gpib_write (dev, "FE1");
//						fprintf(gpib_out,"Output %d;FI0\n",dev);
//						fprintf(gpib_out,"Output %d;FE1\n",dev);
						break;
		}

}											//end fmmod

fsg_fmdeviation(dev,value)			//Set FM deviation in Hz
	int dev;
	double value;
{
	char	string[64];		  
	sprintf (string, "FM%8.3eHZ", value);
	gpib_write (dev, string);

//	fprintf (gpib_out, "Output %d;FM%8.3eHZ\n", dev, value);
}										//end fsg_modfreq

fsg_spcl(dev,value)				//Set special functions
	int dev;
	double value;
{
	char	string[64];		  
	sprintf (string, "SP%2f", value);
	gpib_write (dev, string);

//	fprintf(gpib_out, "Output %d;SP%2f\n", dev, value);
}										//end fsg_spcl

fsg_text (dev,text)				//Set "other" commands
	int dev;
	char text[];
{
	char	string[64];		  
	gpib_write (dev, text);
//	fprintf(gpib_out, "Output %d;%s\n", dev, text);
}										//end fsg_text


#endif
