#ifdef PASCAL_XFACE
#ifndef DEBUGI
#define DEBUGI 1
#endif
#include <stdio.h>
#include <string.h>
#include "Clib/basictypes.h"
#include "Clib/fault.h"
#include "pascal.h"
#include "erule.h"
#include "dwg_table.h"        // For dump

#ifdef SVS
#define REVERSE
#endif


#ifdef DEBUGI
static boolean debugging = FALSE;
const char* NULLS = "<null>";
inline const char* checknull(const char* s) { return s ? s : NULLS; }
#endif


pascal_table_entry pascal_table[SENTINEL_PFUNC];
				// initialized by erule_init() below

pascal_ptr_parameter pascal_expression(EXPRESSION);
pascal_int_parameter pascal_err(ERR);
pascal_int_parameter pascal_assert(ERR_ASSERTION_FAILURE);
pascal_int_parameter pascal_err_int(ERR_INT);
pascal_no_parameters pascal_err_indent(ERR_INDENT);
pascal_ptr_parameter pascal_err_xtring(ERR_XTRING);
pascal_no_parameters pascal_err_context(ERR_CONTEXT);
pascal_no_parameters pascal_err_crlf(ERR_CRLF);
pascal_no_parameters pascal_err_sdl(ERR_SDL);


// Expects x to be non-0
void to_cstring(register char* c, register const char* x)
{
    register const char* end = x + *(unsigned char*)x;
    while (x < end) *c++ = *++x;
    *c = 0;
}


// copy alpha a to cstring c, return 0 if null_alpha, else return c
char* alpha_to_cstring(register char* c, register const char* a)
{
    char* val = c;
    register char* end = (char*)a + 16;
    while (a < end && *a != ' ') *c++ = *a++;
    *c = 0;
    return c == val ? 0 : val;
}


// If c string is 0, returns a nullstring
void to_xtring(char* xbuf, register const char* c)
{
    if (!c) { *xbuf = 0;  return; }

    register char* x = xbuf + 1;
    register char* end = xbuf + 256;
    while (*c && x < end) *x++ = *c++;
    *xbuf = x - xbuf - 1;
}


// The compiler knows how to dump xtrings efficiently, so it's better to
// convert C strings on this side of the border.
void pascal_err_string(register const char* s)
{
    char buf[256];
    to_xtring(buf, s);
    pascal_err_xtring(buf);
}


// Need to insure static initialization
static char* _initialized = 0;
class check_static_init {
public:
    check_static_init();
    operator boolean()
        { return _initialized ? TRUE : FALSE; }
} static_init;

check_static_init::check_static_init()
{ 
    _initialized = "done";   // NOT inline -- must require static initializer
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "pascal.C static init...");
#endif
}


/******************  Functions callable by the PASCAL ValidCOMPILER *******/

#ifdef S32
// Truncate entry points to 8 chars for SVS m68k PASCAL compiler
#define er_select er_selec
#define er_special er_speci
#define er_extension er_exten
#define er_version er_versi
#define er_filename er_filen
#define er_isprim er_ispri
#define er_force_to_prim er_force
#define er_release er_relea
//#define er_samedir er_samed
#define er_samedwg er_samed
#define er_issimple er_issim
#define er_except er_excep
#define er_dirtype er_dirty
#endif


// PASCAL passes each error procedure represented by the enumerated type
// pfunc, in that order (with the exception of SENTINEL_PFUNC -- it is
// never called).

void er_init(
	     // varargs doesn't work on all platforms -- use brute force.
#if PMAX
	     void* l0, pfuncP f0, void* l1, pfuncP f1, void* l2, pfuncP f2, 
             void* l3, pfuncP f3, void* l4, pfuncP f4, void* l5, pfuncP f5, 
             void* l6, pfuncP f6, void* l7, pfuncP f7, void* l8, pfuncP f8
#endif
#if SUN
	     pfuncP f0, pfuncP f1, pfuncP f2, pfuncP f3, pfuncP f4, pfuncP f5, 
             pfuncP f6, pfuncP f7, pfuncP f8, 
             void* l0, void* l1, void* l2, void* l3, void* l4, void* l5, 
             void* l6, void* l7, void* l8
#endif
#if VAX
	     void* l0, void* l1, void* l2, void* l3, void* l4, void* l5, 
             void* l6, void* l7, void* l8, 
	     pfuncP f0, pfuncP f1, pfuncP f2, pfuncP f3, pfuncP f4, pfuncP f5, 
             pfuncP f6, pfuncP f7, pfuncP f8
#endif
#if SVS
             void* l8, pfuncP f8, void* l7, pfuncP f7, void* l6, pfuncP f6, 
             void* l5, pfuncP f5, void* l4, pfuncP f4, void* l3, pfuncP f3, 
             void* l2, pfuncP f2, void* l1, pfuncP f1, void* l0, pfuncP f0
#endif
)
{
    extern void _Cplusplus_static_init();

#ifdef DEBUGI
    if (debugging) fprintf(stderr,  "er_init(...)\n");
#endif
    if (!static_init) _Cplusplus_static_init();
    if (!static_init) fault("C++ static initialization is broken");
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "static init done...");
#endif

    // note comment re. static links in pascal.cl

    pascal_table[0].func = f0;
    pascal_table[0].link = l0;

    pascal_table[1].func = f1;
    pascal_table[1].link = l1;

    pascal_table[2].func = f2;
    pascal_table[2].link = l2;

    pascal_table[3].func = f3;
    pascal_table[3].link = l3;

    pascal_table[4].func = f4;
    pascal_table[4].link = l4;

    pascal_table[5].func = f5;
    pascal_table[5].link = l5;

    pascal_table[6].func = f6;
    pascal_table[6].link = l6;

    pascal_table[7].func = f7;
    pascal_table[7].link = l7;

    pascal_table[8].func = f8;
    pascal_table[8].link = l8;
       
#ifdef DEBUGI
    if (debugging) {
	fprintf(stderr, "er_init done:\n");
	for (pfunc i = 0; i < SENTINEL_PFUNC; i++)
	    fprintf(stderr, "ptable[%d] func=0x%x link=0x%x\n", 
		i, pascal_table[i].func, pascal_table[i].link);
    }
#endif
}


static erule* er = 0;


void er_type(char* compile_type)	// Set compile type
{
    char buf[17];
    char* ctype = alpha_to_cstring(buf, compile_type);
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_type(%s)...", checknull(ctype));
#endif
    er = new erule(ctype);
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "done\n");
#endif
}


boolean er_read(char* filename)		// Read rules file
{
    char fname[256];
    to_cstring(fname, filename);
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_read(%s)...", fname);
#endif
    boolean val = er->read(fname);
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "done\n");
#endif
    return val;
}


class pascal_selection: public erule_selection {
    erule_drawing_page* _page;
public:
    pascal_selection(char* dname, char* context, boolean ignore_non_existence)
    : (dname, context, ignore_non_existence)
    { 
	AS(!this);
	this = (pascal_selection*)erule_heap.mem(sizeof(*this));
	_page = 0; 
    }
    ~pascal_selection()
	{  erule_heap.memfree(this, sizeof(*this));  this = 0; }
    int page_itr()
    {
        _page = _page ? next_page(_page) : first_page();
        return _page ? _page->page_number() : 0;
    }
};

#ifndef REVERSE
pascal_selection* er_select(char* drawing_name, char* context, boolean ignore_non_existence)	// Perform selection per rules
#else
pascal_selection* er_select(boolean ignore_non_existence, char* context, char* drawing_name)	// Perform selection per rules
#endif
{
    char dname[256],cont[256];
    to_cstring(dname, drawing_name);
    if (context) { to_cstring(cont, context);  context = cont; }
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_select(%s,%s,%d)", dname, 
			   checknull(context), ignore_non_existence);
#endif
    pascal_selection *s = new pascal_selection(dname, context, ignore_non_existence);
    if (!(s->extension()) && !(s->special())) { delete s;  s = 0; }
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->0x%x\n", s);
#endif
    return s;
}


void er_release(pascal_selection** selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_release(0x%x)...", *selection);
#endif
    if (*selection) { delete *selection;  *selection = 0; }
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "done\n");
#endif
}


static char retbuf[256];


char* er_special(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_special(0x%x)", selection);
#endif
    if (!selection) return 0;
    string nm = selection->special();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%s\n", checknull(nm));
#endif
    if (!nm) return 0;
    to_xtring(retbuf, nm);
    return retbuf;
}


char* er_extension(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_extension(0x%x)", selection);
#endif
    if (!selection) return 0;
    string nm = selection->extension();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%s\n", checknull(nm));
#endif
    if (!nm) return 0;
    to_xtring(retbuf, nm);
    return retbuf;
}


char* er_dirtype(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_dirtype(0x%x)", selection);
#endif
    string nm = selection->directory_type();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%s\n", checknull(nm));
#endif
    if (!nm) return 0;
    to_xtring(retbuf, nm);
    return retbuf;
}


int er_page(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_page(0x%x)", selection);
#endif
    int pg = !selection ? 0 : selection->page_itr();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", pg);
#endif
    return pg;
}


int er_version(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_version(0x%x)", selection);
#endif
    int v = !selection ? 0: selection->version();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", v);
#endif
    return v;
}


int er_isprim(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_isprim(0x%x)", selection);
#endif
    int val = !selection ? FALSE : selection->is_primitive();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", val);
#endif
    return val;
}


int er_force_to_prim(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_force_to_prim(0x%x)", selection);
#endif
    int val = !selection ? FALSE : selection->force_to_primitive();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", val);
#endif
    return val;
}


int er_issimple(pascal_selection* selection)
{
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "er_issimple(0x%x)", selection);
#endif
    int val = !selection ? FALSE : selection->is_simple_primitive();
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", val);
#endif
    return val;
}


// this is commutative, so ignore parameter reversal
int er_samedwg(pascal_selection* selection1, pascal_selection* selection2)
{
#ifdef DEBUGI
    if (debugging) 
	fprintf(stderr, "er_samedir(0x%x, 0x%x)", selection1, selection2);
#endif
    //int val = selection1->same_directory(selection2);  // NULLs handled therein
    int val = selection1->same_drawing(selection2);  // NULLs handled therein
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%d\n", val);
#endif
    return val;
}


// These MUST match ordinal values with the pascal type file_types within
// the compiler
enum pascal_filetype  { CONNECTIVITY = 1, 
			SPECIAL,
			SCHEMA,
                        EXPANSION, 
			LIST,
			LINKER_LIST,
			PFILE_SENTINEL };

char*
#ifdef REVERSE
er_filename(char* context, unsigned page, pascal_filetype filetype, 
            pascal_selection* module)
#else
er_filename(pascal_selection* module, pascal_filetype filetype, unsigned page,
	    char* context)
#endif
{
    // Indexed by pascal_filetype
    static erule_filetype ftypes[] = {	ERULE_UNDEF_FILETYPE,
					ERULE_CONNECTIVITY,
					ERULE_SPECIAL,
					ERULE_SCHEMA,
					ERULE_EXPANSION,
					ERULE_LIST,
					ERULE_LINKER_LIST };

    char _context[256];
    if (context) { to_cstring(_context, context);  context = _context; }
#ifdef DEBUGI
    if (debugging) 
	fprintf(stderr, "er_filename(0x%x, %d, %d, %s)",
			module, filetype, page, checknull(context));
#endif
    AS(filetype<PFILE_SENTINEL);
    string name = module->filename(ftypes[filetype], page, context);

    static char _name[256];
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "->%s\n", checknull(name));
#endif
    to_xtring(_name, name);
    return _name;
}


void er_dump()
{
    ((erule*)0)->dump(cerr);
}


#ifndef REVERSE
void er_except(const char* compile_type, const char* drawing, 
	       const char* context,
               const char* extension, const char* attribute)
#else
void er_except(const char* attribute, const char* extension, 
	       const char* context,
               const char* drawing, const char* compile_type)
#endif
{
    char ctype_buf[17], dwg[256], cont[256], ext_buf[17], att_buf[17];
    to_cstring(dwg, drawing);
    if (context) { to_cstring(cont, context);  context = cont; }
    char* ctype = alpha_to_cstring(ctype_buf, compile_type);
    char* ext = alpha_to_cstring(ext_buf, extension);
    char* att = alpha_to_cstring(att_buf, attribute);
#ifdef DEBUGI
    if (debugging) 
	fprintf(stderr, "er_except(%s, %s, %s, %s, %s)...",
	    checknull(ctype), checknull(dwg), checknull(context),
		checknull(ext), checknull(att));
#endif
    ((erule*)0)->except(ctype, dwg, context, ext, att);
#ifdef DEBUGI
    if (debugging) fprintf(stderr, "done.\n");
#endif
}


void er_debug()
{
#ifdef DEBUGI
    debugging = TRUE;
#endif
}
#endif
