#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <libc.h>

#include "Clib/heap.h"
heap arena;

#define protected public /* Till this c++ feature works */
enum boolean { FALSE = 0, TRUE };
const char *INPUT_FLAG = "-i";
const char *OUTPUT_FLAG = "-o";
const char *DEBUG_FLAG = "-d";
const char *DECOLON_FLAG = "-:";
const char *DECARET_FLAG = "-^";
const char *DETAB_FLAG = "-t";
const char *DECOMMENT_FLAG = "-c";
const char SKIP = '\1';
const NUM_FLAGS = 10;
const LBRACE = '{';
const RBRACE = '}';
const LPAREN = '(';
const RPAREN = ')';
const TAB = '\t';

const OTHERWISE_LENGTH = 9;
const char *otherwise = "otherwise";
const char *caps_otherwise = "OTHERWISE";

char *input_file = NULL;
char *output_file = NULL;
boolean decomment = FALSE;
boolean decolon_otherwise = FALSE;
boolean replace_carets = FALSE;
boolean debug = FALSE;
boolean detab_on_input = FALSE;

int max_flag_length = 0;

inline boolean isidchar(char c)
{
    return isalpha(c) || c == '_' || c == '$';
}


class flags {
    int num, maxnum;
    char **strings;
public:
    flags();
    void insert(char *);
    boolean exists(char *);
    void dump(FILE *f = stderr);
    int number() { return num; }
} decomment_flags, delete_flags;

flags::flags()
{
    num = 0;
    maxnum = NUM_FLAGS;
    strings = (char **)arena.cmem(sizeof(char*), NUM_FLAGS);
}

void
flags::insert(char *str)
{
    if (num >= maxnum) {
        strings = (char **)arena.rmem(strings, maxnum, maxnum + NUM_FLAGS);
	maxnum += NUM_FLAGS;
    }
    strings[num++] = str;
    int len = strlen(str);
    if (len > max_flag_length) max_flag_length = len;
}

boolean
flags::exists(char *str)
{
    int i;
    if (str)
	for (i = 0; i < num; i++) if (!strcmp(strings[i], str)) return TRUE;
    return FALSE;
}

void
flags::dump(FILE *f)
{
    const NUM_PER_LINE = 5;
    const char *indent = "   ";
    int i;
    int count = NUM_PER_LINE;
    fprintf(f, indent);
    for (i = 0; i < num; i++) {
        if (!count--) {
            fprintf(f, "\n%s", indent);
	    count = NUM_PER_LINE;
	}
	fprintf(f, " %s", strings[i]);
    }
    fprintf(f, "\n");
}


class buffer {
    char *buf;
    char *pos;
    char *end;
    int incr;
    void extend();
public:
    buffer(int increment);
    ~buffer() { arena.memfree(buf, ++end - buf); }
    void append(char c) 
        { if (pos >= end) extend(); *pos++ = c; }
    void start() { pos = buf; }
    void terminate() { *pos = '\0'; }
    operator char*() { return buf; }
};

buffer::buffer(int increment)
{
    incr = increment;  
    buf = (char *)arena.mem(incr);  
    end = buf + incr - 1;
}

void buffer::extend()
{
    int size = ++end - buf + incr;
    int offset = pos - buf;
    buf = (char *)arena.rmem(buf, end - buf, size);
    end = buf + (size - 1);
    pos = buf + offset;
}


class line {
    buffer l;
    buffer flag;
    buffer include;
    boolean in_comment;
    boolean last_line_blank;
    boolean last_tok_otherwise;
    char comment_char;
protected:
    void reset() 
    { 
	in_comment = FALSE;  
	last_line_blank = FALSE;  
	last_tok_otherwise = FALSE;
    }
public:
    line();
    boolean read(FILE *f);
    void write(FILE *f);
    void handle_otherwise();
    void strip_comments();
    void uncomment();
    char *flag_comment();    
    char *include_file();
    void do_carets();
};

line::line() : l(80), flag(16), include(15) 
{
    reset();
    last_line_blank = TRUE;
}

boolean
line::read(FILE *f)
{
    register int c = getc(f);
    if (c == EOF) return FALSE;

    l.start();
    if (detab_on_input) {
        int col = 0;
	while (c != EOF && c != '\n') {
	    col++;
	    if (c == TAB) {
	        l.append(' ');
	        while (col % 8) { l.append(' '); col++; }
	    }
	    else l.append(c);
	    c = getc(f);
	}
    }
    else 
	while (c != EOF && c != '\n') {
	    l.append(c);
	    c = getc(f);
	}
    l.terminate();
    return TRUE;
}

void
line::write(FILE *f)
{
    register char *p;
    boolean blank = TRUE;
    for (p = l; *p; p++) if (!isspace(*p) && *p != SKIP) {
        blank = FALSE;
	break;
    }
    if (blank && last_line_blank) return; // collapse blank lines to 1
    last_line_blank = blank;
    for (p = l; *p ; p++) if (*p != SKIP) putc(*p, f);
    putc('\n', f);
}

void 
line::handle_otherwise()
{
    register char *p;
    int j = 0;
    boolean in_non_otherwise_id = FALSE;

    for (p = l; *p; p++) {
        if (!in_non_otherwise_id &&
            ((*p == otherwise[j]) || (*p == caps_otherwise[j]))) {
            if (++j == OTHERWISE_LENGTH) {
                last_tok_otherwise = TRUE;
		j = 0;
            }
	}
        else {
            j = 0;
            if (last_tok_otherwise && (*p == ':')) *p = ' ';
            if (!isspace(*p)) last_tok_otherwise = FALSE;
            in_non_otherwise_id = (isidchar(*p));
        }
    }
}

void
line::strip_comments()
{
    boolean in_quotes = FALSE;
    register char *last = 0;
    register char *p;

    for (p = l; *p; p++) {
        if (in_comment) {
            // skip chars while looking for end of comment
	    if (*p == comment_char) {
	        if (comment_char == RBRACE) in_comment = FALSE;
		else {
		    *p = SKIP;
		    if (*++p == RPAREN) in_comment = FALSE;
		    else (p--);  // in case we're at end of line
		}
	    }
	    *p = SKIP;
	}
        else if (in_quotes) {
            if (*p == '\'') in_quotes = FALSE;
	}
        else {
            if (*p == '\'') in_quotes = TRUE;
            else if (*p == LBRACE) {
		comment_char = RBRACE;
                in_comment = TRUE;  
		*p = SKIP;
	    }
            else if (*p == LPAREN) {
                if (*++p == '*') {
		    comment_char = '*';
		    in_comment = TRUE;  
		    *p = *(p-1) = SKIP;
		}
		else p--;
            }
	}
    }
}  
  
// ASSUMPTION -- if there is an include directive, it is the first thing
// on the line and there is at most 1 per line.
//
// ALSO -- file names do not contain the character '*'
char *
line::include_file()
{
    register char *p = (char *)l;
    while (*p && isspace(*p)) p++;

    char term_char;
    switch (*p) {
    case LBRACE: 
        term_char = RBRACE;  break;
    case LPAREN:
        if (*++p != '*') return NULL;
	term_char = '*';
	break;
    default: return NULL;
    }

    if (*++p != '$' && *p != '%') return NULL;
    if (*++p != 'i' && *p != 'I') return NULL;
    p++;
    if (!isspace(*p)) return NULL;
    for (p++; *p && isspace(*p); p++);

    register char *startc = p;
    while (*p && *p != term_char) p++;
    if (*p != term_char || p == startc) return NULL;

    include.start();
    while (startc < p) include.append(*startc++);
    include.terminate();
    return include;
}

void
line::uncomment()
{
    boolean in_string = FALSE;
    register char *p;
    p = (char *)l;
    if (*p == LBRACE && *(p+1) == '#') *p++ = SKIP; /* for "#include"s */
    for ( ; *p; p++) {
        switch (*p) {
        case LBRACE: 
        case RBRACE: 
	    if (!in_string) *p = ' ';  break;
	case '\'':
	    in_string = !in_string;  break;
	case LPAREN:
	    if (!in_string && *(p+1) == '*') {
	        *p++ = ' ';  *p = LBRACE;
	    }
	    break;
	case '*':
	    if (!in_string && *(p+1) == RPAREN) {
	        *p++ = RBRACE;  *p = ' ';
	    }
	    break;
	default:  break;
	}
    }
}

inline void
line::do_carets()
{
    register char *p;
    for (p = l; *p; p++) if (*p == '^') *p = '@';
}


char *
line::flag_comment()
{
    char *ln = (char *)l;
    register char *p;
    for (p = ln + strlen(l) - 1; p >= ln && isspace(*p); p--) ;

    char start_char;
    if (*p == RBRACE) start_char = LBRACE;
    else if (*p == RPAREN) {
        if (--p >= ln && *p == '*') start_char = '*';
	else return NULL;
    }
    else return NULL;

    char *last;
    for (last = --p; p >= ln && last - p <= max_flag_length; p--) {
        if (*p == start_char &&
	    (  (start_char == LBRACE) ||
	       (p > ln && *(p-1) == LPAREN))) {
	    flag.start();
	    for (p++; p <= last; p++) flag.append(*p);
	    flag.terminate();
	    return flag;
	}
    }

    return NULL;
}

class source : line {
    FILE *f;
    boolean status;
public:
    source(char *name);
    ~source() { if (f) { fclose(f); } }
    boolean process();
};

source::source(char *name)
{
    if (name) f = fopen(name, "r");
    else f = stdin;

    if (!f) fprintf(stderr, "Can't open %s\n", name);
    status = f ? TRUE : FALSE;
}

boolean
source::process()
{
    if (!status) return FALSE;
    char *flag;
    while (read(f)) {
        flag = flag_comment();
	if (delete_flags.exists(flag)) reset();
	else {
	    char *fname = include_file();
	    if (fname) {
	        source included(fname);
		if (!included.process()) status = FALSE;
	    }
	    else {
	        if (decomment_flags.exists(flag)) uncomment();
		if (decomment) strip_comments();
		if (decolon_otherwise) handle_otherwise();
		if (replace_carets) do_carets();
		write(stdout);
	    }
	}
    }
    return status;
}


void
print_usage()
{
    fprintf(stderr, "               [+][CMS  ]\n");
    fprintf(stderr, "Usage: bigfile [-][VAX  ] [-c] [-:] [-^] [-t]\n");
    fprintf(stderr, "                  [UNIX ]\n");
    fprintf(stderr, "                  [etc.]\n");
    exit(1);
}


void
command_flags(int argc, char **argv)
{
    int i;
    for (i = 1; i < argc ; i++) {
	if (!strcmp(argv[i], INPUT_FLAG)) {
	    if (input_file || ++i >= argc) print_usage();
	    input_file = argv[i];
        }
	else if (!strcmp(argv[i], OUTPUT_FLAG)) {
	    if (input_file || ++i >= argc) print_usage();
	    output_file = argv[i];
        }
        else if (!strcmp(argv[i], DECOMMENT_FLAG)) decomment = TRUE;
        else if (!strcmp(argv[i], DECOLON_FLAG)) decolon_otherwise = TRUE;
        else if (!strcmp(argv[i], DECARET_FLAG)) replace_carets = TRUE;
        else if (!strcmp(argv[i], DETAB_FLAG)) detab_on_input = TRUE;
        else if (!strcmp(argv[i], DEBUG_FLAG)) debug = TRUE;
	else if (argv[i][0] == '-') delete_flags.insert(&(argv[i][1]));
	else if (argv[i][0] == '+') decomment_flags.insert(&(argv[i][1]));
	else print_usage();
    }
}


void
dump_flags()
{
    fprintf(stderr, "Deleting lines flagged:\n");
    delete_flags.dump(stderr);
    fprintf(stderr, "Uncommenting lines flagged:\n");
    decomment_flags.dump(stderr);

    if (decomment) fprintf(stderr, "Stripping comments\n");
    if (decolon_otherwise) fprintf(stderr, "Deleteing : from otherwise\n");
    if (replace_carets) fprintf(stderr, "Changing ^ to @\n");
}


main(int argc, char **argv)
{
    char output_buffer[BUFSIZ];
    char input_buffer[BUFSIZ];

    command_flags(argc, argv);
    if (!debug && !isatty(0)) setbuf(stdin, input_buffer);
    if (!debug && !isatty(1)) setbuf(stdout, output_buffer);

    source input(input_file);
    exit(input.process() ? 0 : 1);
}
