function get_EXPR_property: xtring;
  { read the EXPR property from the file and return it }
begin
  get_EXPR_property := nullstring;

  if id.name^.name <> 'EXPR            ' then assert(17 { expected EXPR } )
  else
    begin
      insymbol;     { eat the identifier }
      if sy = EQUAL then insymbol;
      if sy <> strings then assert(18 { expected a string })
      else
        begin
          get_EXPR_property := lex_string;
          insymbol;   { eat the string }
        end;

      if sy = SEMI then insymbol else assert(35 { expected a ; });
    end;
end { get_EXPR_property } ;


function check_for_special_body(body_name: xtring): body_type;
  { check the body name for a special body name and return its type }
  var
    type_of_body: body_type;  { type of the body }
    found: boolean;           { TRUE if body found }
begin
  type_of_body := succ(FIRST_BODY);  found := FALSE;
  while (type_of_body < LAST_BODY) and not found do
    if special_body_list[type_of_body] = body_name then found := TRUE
    else type_of_body := succ(type_of_body);

  if found then check_for_special_body := type_of_body
           else check_for_special_body := USER_BODY;
end { check_for_special_body } ;


function is_comment_body(property_list: property_ptr): boolean;
  { return TRUE if the body with the given properties is a COMMENT }
  var
    prop: property_ptr;      { property returned from the search }
begin
  is_comment_body := FALSE;

  if find_property(property_list, COMMENT_BODY_prop_name, prop) then
    is_comment_body := TRUE

  else if find_property(property_list, BODY_TYPE_prop_name, prop) then
    if prop^.text = COMMENT_string then
      is_comment_body := TRUE;
end { is_comment_body } ;


procedure check_for_abbreviation(macro: macro_def_ptr);
  { check for the ABBREV property on the macro.  If it is present,
    make sure it is legal.  If it is not present, then concoct it
    if this is a primitive. }
  var
    abbrev: property_ptr;     { pointer into property list }
    i: string_range;          { index into the abbreviation }
    done: boolean;            { TRUE when search in abbrev is complete }
begin
  if find_property(macro^.properties, ABBREV_prop_name, abbrev) then
    with abbrev^ do
      begin
        if text=nullstring then
          begin
	    { This is not the best error message for this case,
	      but it should be a rare problem, as the user
	      would have to change the connectivity file by hand
	      to do this. }
            error(128 { must be only letters, digits, & _ });
            error_dump_macro_def(macro);
            error_dump_indent(indent);
            error_dump_alpha('Abbreviation=   ');
            error_dump_alpha('<null>          ');
            error_dump_CRLF;
  
            text := concoct_abbrev(macro^.macro_name);
            error_dump_indent(indent);
            error_dump_alpha('New abbrev=     ');
            error_dump_string(text);
	    error_dump_CRLF;
          end;
	  
        i := 1;  done := FALSE;
        while (i <= ord(text^[0])) and not done do
          if not isidentchar[text^[i]] then
            begin
              error(128 { must be only letters, digits, & _ });
              error_dump_macro_def(macro);
              error_dump_indent(indent);
              error_dump_alpha('Abbreviation=   ');
              error_dump_string(text);
              error_dump_CRLF;
  
              text := concoct_abbrev(macro^.macro_name);
              error_dump_indent(indent);
              error_dump_alpha('New abbrev=     ');
              error_dump_string(text);
              error_dump_CRLF;
  
              done := TRUE;
            end
          else
            i := i + 1;
      end;
end { check_for_abbreviation } ;


function generate_unique_PATH_name(var unique_number: natural_number): xtring;
  { generate a unique PATH property for a body without one.  Use the given
    unique number. }
  var
    path: alpha;                { PATH element being created }
    pos: 0..ID_LENGTH;          { index into the new element }


  procedure add_the_number(n: natural_number);
    { add the given number to the path element recursively }
  begin
    if n > 9 then add_the_number(n div 10);
    pos := pos + 1;
    path[pos] := chr((n mod 10) + ord('0'));
  end { add_the_number } ;


begin { generate_unique_PATH_name }
  pos := 0;
  path := null_alpha;
  unique_number := unique_number + 1;
  add_the_number(unique_number);

  generate_unique_PATH_name := make_and_enter_string(path);
end { generate_unique_PATH_name } ;


(**)     { ------- MACRO definition parse ------- }


procedure parse_macro_definition(macro: macro_def_ptr);
  { read in and parse a macro definition file.  The drawing being parsed is
    defined by MACRO. }
  var
    unique_body_number: natural_number;  { serves to make all bodies unique }
    body_name: xtring;                   { current body being parsed }


  procedure display_invoke_error;
    { display the invoking macro and invoked macro names }
  begin
    error_dump_macro_def(macro);
    error_dump_body_name(body_name);
  end { display_invoke_error } ;


  function find_pin_name(var list: bindings_list_ptr; pin_name: xtring):
                                                            bindings_list_ptr;
    { find the pin name in the given list of pin names }
    var
      found: boolean;                  { TRUE with pin name is found }
      binding: bindings_list_ptr;      { pin name list element }
  begin
    binding := list;  found := FALSE;
    while (binding <> NIL) and not found do
      if pin_name = binding^.formal_parameter then found := TRUE
      else binding := binding^.next;

    if found then find_pin_name := binding
    else
      begin
        new_bindings_list(list);
        list^.formal_parameter := pin_name;
        find_pin_name := list;
      end;
  end { find_pin_name } ;
    

  procedure parse_properties(body: body_type);
    { parse the properties associated with a body }
    var
      allowed_properties: property_set;    { set of permitted properties }
      termsys: setofsymbols;               { terminal symbols for SKIP }
      property_name: name_ptr;             { current property name }
      property_value: xtring;              { current property value }
      prop_element: bindings_list_ptr;     { element for pin properties }


    function get_property: boolean;
      { read a property name/property text pair.  If there is no error,
        return FALSE otherwise return TRUE. }
      var
        found_error: boolean;      { TRUE if error found }
    begin
      found_error := TRUE;
      if sy <> IDENT then found_error := FALSE
      else
        begin
          property_name := id.name;   insymbol;
          if sy = EQUAL then insymbol else assert(178 { expected = });
          if sy <> STRINGS then
            begin
              assert(108 { expected a string });  found_error := FALSE;
              skip(termsys);
            end
          else
            begin  property_value := lex_string;  insymbol;  end;
        end;

      get_property := found_error;
    end { get_property } ;

      
    procedure copy_pin_properties(var property_list: property_ptr);
      { copy pin properties from input to property list }
    begin
      repeat
        if get_property then
          if PERMIT_PIN IN property_name^.kind then
            add_to_prop_list(property_list, property_name, property_value)
          else
            begin
              error(153 { not permitted on a pin });
              display_invoke_error;
              error_dump_property(property_name, nullstring);
              error_dump_pin_name_string(prop_element^.formal_parameter);
            end;

        if sy = SEMI then insymbol else assert(35 { expected ; });
      until sy <> IDENT;
    end { copy_pin_properties } ;


(**)     { ------- DEFINE body parsing ------ }


    procedure read_text_macros;
      { read the text macro definitions for the macro and enter into a list }
      var
        dummy: property_ptr;      { dummy value returned from find_property }
    begin
      if debug then disp_line('enter read_define');

      while sy = IDENT do
        begin
          if get_property then
            if find_property(macro^.text_macros, property_name, dummy) then
              begin
                if property_value <> dummy^.text then
		  begin
                    error(114 { text macro already exists });
                    error_dump_macro_def(macro);
                    error_dump_body_name(DEFINE_string);
                    error_dump_text_macro(property_name);
		  end;
              end
            else
              if RESERVED IN property_name^.kind then
                begin
                  error(105 { reserved TM name });
                  error_dump_macro_def(macro);
                  error_dump_body_name(DEFINE_string);
                  error_dump_text_macro(property_name);
                end
              else
                begin
                  add_to_prop_list(macro^.text_macros,
                                   property_name, property_value);

                  if macro = root_macro_def then
		    enter_local_tm(paged_schema_of_this_page,
		                   property_name, property_value);
                end;
          if sy = SEMI then insymbol else assert(35 { expected ; });
        end;

      if debug then disp_line('read_text_macros ');
    end { read_text_macros } ;


(**)     { ------- DRAWING body parsing ------- }


    procedure read_drawing_properties;
      { process the DRAWING body of the macro: the drawing properties }
      var
        dummy: property_ptr;    { dummy property pointer for property search }


      procedure display_error(directory_value: xtring);
        { display the old and new strings to the error files }
      begin
        error_dump_indent(indent);
        error_dump_alpha('Directory refs: ');
        error_dump_string(directory_value);
        error_dump_CRLF;

        error_dump_indent(indent);
        error_dump_alpha('Macro specifies:');
        error_dump_string(property_value);
        error_dump_CRLF;
      end { display_error } ;


    begin { read_drawing_properties }
      if debug then disp_line('enter read_drawin');

      while sy = IDENT do
        begin
          if get_property then
            if property_name = TITLE_prop_name then
              begin
                if macro^.macro_name <> property_value then
                  begin
                    error(182 { it doesn't match! });
                    error_dump_macro_def(macro);
                    display_error(macro^.macro_name);
                  end;
              end
            else if property_name = EXPR_prop_name then
              begin
                check_and_add_to_prop_list(macro^.properties,
                                           EXPR_prop_name, property_value);
              end
            else if property_name = ABBREV_prop_name then
              begin
                if not find_property(macro^.properties,
                                     property_name, dummy) then
                  add_to_prop_list(macro^.properties,
                                   property_name, property_value);
              end
            else
              add_to_prop_list(macro^.properties,
                               property_name, property_value);

          if sy = SEMI then insymbol else assert(35 { expected ; });
        end;

      if debug then disp_line('read_drawing_prop');
    end { read_drawing_properties } ;


(**)     { ------- USER macro invocation parsing ------- }


    procedure read_user_body_properties(invokes: invoke_list_ptr;
                                        reading_parameters: boolean);
      { read the property or parameter list on a user defined macro
        invocation.  If a property is found in the property list that
        has the IS_PARAMETER or IS_INT_PARAMETER attribute, then add it to
        the parameter list. }


      procedure check_and_add(invoke: invoke_list_ptr; is_param: boolean);
        { check the property list and if the property is not there, add it }
        var
          dummy: property_ptr;        { dummy property from find_property }
      begin
        if not (PERMIT_BODY IN property_name^.kind) then
          begin
            error(152 { not permitted on a body });
            display_invoke_error;
            error_dump_property(property_name, nullstring);
          end
        else
	  if property_name = PATH_prop_name then
            if invoke^.path <> nullstring then
              begin
                error(133 { prop already defined });
                display_invoke_error;
                error_dump_property(property_name, nullstring);
              end
	    else
	      begin
	        invoke^.path := property_value;
		current_path_prop := property_value;
	      end
          else if is_param then
            if find_property(invoke^.parameters, property_name, dummy) then
              begin
                error(180 { parameter declared twice });
                display_invoke_error;
                error_dump_property(property_name, nullstring);
              end
            else
	      add_to_prop_list(invoke^.parameters, property_name, 
	                       property_value)
          else { normal body property }
	    if find_property(invoke^.properties, property_name, dummy) then
              begin
                error(133 { prop already defined });
                display_invoke_error;
                error_dump_property(property_name, nullstring);
              end
            else 
	      add_to_prop_list(invoke^.properties, property_name, 
	                       property_value);
      end { check_and_add } ;


    begin { read_user_body_properties }
      if debug then disp_line('enter read_user_b');

      while sy = IDENT do
        begin
          if get_property then
            if reading_parameters or
               (parameter_attributes * property_name^.kind <> []) then
              check_and_add(invokes, TRUE)
            else check_and_add(invokes, FALSE);

          if sy = SEMI then insymbol else assert(35 { expected ; });
        end;

      if debug then disp_line('read_user_body_pr');
    end { read_user_body_properties } ;

      
(**)     { ------- parse properties of macro ------- }


  begin { parse_properties }
    if debug then disp_line('enter parse_prope');

    if debug_20 then writeln(outfile, '        Starting to parse properties');

    insymbol;     { eat the PROPERTY symbol }

    if body = USER_BODY then 
      allowed_properties := [BODY_PROPERTY, PIN_PROPERTY, PARAMETER_PROPERTY]
    else if body = DRAWING_BODY then
      allowed_properties := [BODY_PROPERTY]
    else if body IN bodies_with_bindings then
      allowed_properties := []
    else
      allowed_properties := [BODY_PROPERTY];

    repeat

      { check for BODY properties } 

      if sy = BODYSY then 
        begin
          if not (body_property IN allowed_properties) then 
            begin
              error(109 { not allowed here });
              skip([ENDBODYSY]);
              display_invoke_error;
            end
          else
            begin
              insymbol;  termsys := [SEMI,endbodysy];
              case body of
		MENU_BODY:       skip([ENDBODYSY]);  { ignore it }
                DEFINE_BODY:     read_text_macros;
                DRAWING_BODY:    read_drawing_properties;
                USER_BODY:       read_user_body_properties
                                         (macro^.invokes, READ_AS_PROPERTIES);
              end;
            end;

          if sy = ENDBODYSY then insymbol else assert(12 { no END_BODY });
          if sy = SEMI then insymbol else assert(35 { expected ; });
        end

      { check for PIN properties } 

      else if sy = PINSY then
        begin
          if not (pin_property IN allowed_properties) then
            begin
              error(125 { not allowed here });  skip([ENDPINSY]);
              display_invoke_error;
            end
          else
            begin
              insymbol;  termsys := [SEMI,ENDPINSY];
              repeat
                if sy <> STRINGS then 
                  begin
                    assert(14 { expected pin name });
                    skip([ENDPINSY,STRINGS]);
                  end
                else
                  begin
                    prop_element := find_pin_name(macro^.invokes^.bindings,
                                                  lex_string);
                    insymbol;
                    if sy = COLON then insymbol
                                  else assert(163 { expected : });

                    copy_pin_properties(prop_element^.pin_properties);
                  end;
              until sy <> STRINGS;
            end;
          if sy = ENDPINSY then insymbol else assert(15 { expected END_PIN });
          if sy = SEMI then insymbol else assert(35 { expected ; });
        end

      { check for PARAMETERS }

      else if sy = PARAMETERSY then
        begin
          if not (parameter_property IN allowed_properties) then
            begin
              error(188 { not allowed here });
              skip([ENDPARAMETERSY]);
              display_invoke_error;
            end
          else
            begin
              insymbol;
              termsys := [SEMI,ENDPARAMETERSY];
              read_user_body_properties(macro^.invokes, READ_AS_PARAMETERS);
            end;

          if sy = ENDPARAMETERSY then insymbol else assert(16 { wrong });
          if sy = SEMI then insymbol else assert(35 { expected ; });
        end

      { check for NULL property section or garbage }

      else if sy <> ENDPROPERTYSY then
        begin
          assert(20 { unexpected symbol in property body });
          skip(propbeginsys + [ENDPROPERTYSY]);
        end;

    until not (sy IN propbeginsys);

    if sy = ENDPROPERTYSY then insymbol else assert(21 { expected END_PROP });
    if sy = SEMI then insymbol else assert(35 { expected ; });

    if debug then disp_line('parse_properties ');
  end { parse_properties } ;


(**)     { ------- signal BINDINGs parsing ------- }


  procedure parse_bindings(body: body_type);
    { parse the formal/actual parameter bindings list for the macro or
      special body. }
    var
      pin_name: xtring;                 { pin nam in binding }
      actual_signal: xtring;            { actual signal connected to the pin }
      formal: bindings_list_ptr;        { formal/actual binding }
      actual_properties: property_ptr;  { signal (actual) properties }
      signals_NET_ID: xtring;           { NN property from current prop list }


    function read_pin_and_signal(var pin_name, signal_name: xtring;
                                 var properties: property_ptr): boolean;
      { read and return the next pin name, its actual signal, and any
        properties from the BINDINGS section.  Return TRUE if there
        are no parse errors. }


      procedure read_property_list(var properties: property_ptr);
        { read a property list of the form:
                name = 'text', name = 'text', ..., name = 'text';
          and return in the list PROPERTIES. }
        var
          done: boolean;            { TRUE if property list parsing is done }
          property_name: name_ptr;  { property name }
          property_value: xtring;   { property value }
      begin
        signals_NET_ID := nullstring;

        done := FALSE;
        while (sy = IDENT) and not done do
          begin
            property_name := id.name;
            insymbol;   { eat the identifier }

            if sy = EQUAL then insymbol;

            if sy <> STRINGS then assert(49 { expected a string })
            else
              begin
                property_value := lex_string;
                insymbol;

                if property_name = NET_ID_prop_name then
                  signals_NET_ID := property_value
                else if PERMIT_SIGNAL IN property_name^.kind then
                  add_to_prop_list(properties, property_name, property_value)
                else
                  begin
                    error(151 { not permitted on a signal });
                    error_dump_macro_def(macro);
                    error_dump_body_name(body_name);
                    error_dump_property(property_name, nullstring);
                    error_dump_signal_name_string(signal_name);
                  end;
              end;

            if sy = COMMA then insymbol else done := TRUE;
          end;
      end { read_property_list } ;


    begin { read_pin_and_signal }
      read_pin_and_signal := FALSE;
      properties := NIL;
      pin_name := nullstring;
      signal_name := nullstring;
      signals_NET_ID := nullstring;

      { get the pin name }

      if sy <> STRINGS then 
        begin  assert(23 { expected formal parameter } );  skip([SEMI]);  end
      else
        begin
          pin_name := lex_string;
          insymbol;
          if sy = EQUAL then insymbol else error(2 { expected = });
          if sy <> STRINGS then
            begin  assert(24 { expected actual param });  skip([SEMI]);  end
          else 
            begin
              signal_name := lex_string;
              insymbol;
              if sy = COLON then     { read signal properties }
                begin
                  insymbol;  { eat the : }
                  read_property_list(properties);
                end;

              read_pin_and_signal := TRUE;
            end;
        end;

      if sy = SEMI then insymbol else assert(35 { expected ; });
    end { read_pin_and_signal } ;


    procedure read_declare_bindings;
      { read in the DECLAREd signals for this macro }
    begin
      if debug then disp_line('enter read_declar');

      if macro^.is_leaf_macro then 
        skip([ENDBINDSY])    { we ignore declared signals }
      else
        begin
          error(219 { DECLARE bodies are no longer supported });
          display_invoke_error;
          skip([ENDBINDSY]);
        end;
      if debug then disp_line('read_declares    ');
    end { read_declare_bindings } ;


    procedure read_pin_name_bindings;
      { read in the PIN NAMEs for the macro }
      var
        dummy,                      { dummy string for pin name }
        signal_name: xtring;        { the PIN NAME for the body }
        properties: property_ptr;   { properties of the signal }
    begin
      if debug then disp_line('enter read_pin_na');

      if macro^.is_leaf_macro then
        skip([ENDBINDSY])    { we ignore pin names }
      else
        repeat
          if read_pin_and_signal(dummy, signal_name, properties) then
            begin
              release_entire_property_list(properties);
              new_signal_list(macro^.params);
              macro^.params^.signal_name := fix_signal_name(signal_name);
            end;
        until sy <> STRINGS;

      if debug then disp_line('read_pin_names   ');
    end { read_pin_name_bindings } ;


    function create_NET_ID: xtring;
      { create a unique NET_ID property and return it }
      var
        temp_string: xtring;      { temporary string }
    begin
      unique_NET_ID_number := unique_NET_ID_number + 1;

      create_a_string(temp_string, MAX_STRING_LENGTH);
      temp_string^[0] := chr(0);

      if add_char_to_string(temp_string, UNIQUE_PREFIX_CHAR) then;
      if add_number_to_string(temp_string, unique_NET_ID_number) then;

      create_NET_ID := enter_string(temp_string);

      temp_string^[0] := chr(MAX_STRING_LENGTH);
      release_string(temp_string);
    end { create_NET_ID } ;


  begin { parse_bindings }
    if debug then disp_line('enter parse_bindi');

    if debug_20 then writeln(outfile, '        Starting to parse bindings');

    if sy = BINDINGSY then insymbol else assert(22 { expected BINDING });

    case body of
      DECLARE_BODY:   read_declare_bindings;
      PIN_NAMES_BODY: read_pin_name_bindings;
      USER_BODY:
        repeat
          if read_pin_and_signal(pin_name, actual_signal,
                                 actual_properties) then
            begin
              formal := find_pin_name(macro^.invokes^.bindings, pin_name);
              new_clear_text_actual_list(formal^.actual_parameter);

              if net_processing and (signals_NET_ID = nullstring) then
                signals_NET_ID := create_NET_ID;

              with formal^.actual_parameter^ do
                begin
                  actual_parameter := fix_signal_name(actual_signal);
                  properties := actual_properties;
                  net_id := signals_NET_ID;
                end;
            end;
        until sy <> STRINGS;    
    end { case } ;

    if sy = ENDBINDSY then insymbol else assert(25 { expected END_BINDING });
    if sy = SEMI then insymbol else assert(35 { expected ; });

    if debug then disp_line('parse_bindings   ');
  end { parse_bindings } ;


(**)     { ------- macro INVOKE parsing ------- }


  procedure parse_invoke;
    { parse an macro invocation of the form INVOKE 'name'; ... END_INVOKE; }
    var
      body: body_type;          { type of body being parsed }
      temp: invoke_list_ptr;    { invoke to be released }


    procedure check_for_PATH_property;
      { make sure the PATH property is in the property list of a user macro }
    begin
      if macro^.invokes^.path = nullstring then
        begin
          error(197 { PATH property not found });
          display_invoke_error;
          macro^.invokes^.path := 
	    generate_unique_PATH_name(unique_body_number);
        end;
    end { check_for_PATH_property } ;


    procedure check_for_SIZE_parameter;
      { make sure the SIZE parameter exists in the parameter list }
      var
        prop: property_ptr;         { property returned from search for SIZE }
        search_prop: property_ptr;  { property returned from other searches }
    begin
      if find_property(macro^.invokes^.parameters, SIZE_prop_name, prop) then
        begin
          if find_property(macro^.invokes^.properties,
                           NEEDS_NO_SIZE_prop_name, search_prop) then
            begin
              error(144 { can't have one on this body! });
              display_invoke_error;
            end
          else if find_property(macro^.invokes^.properties,
                                HAS_FIXED_SIZE_prop_name, search_prop) then
            begin
              error(144 { can't have one on this body! });
              display_invoke_error;
              prop^.text := search_prop^.text;
            end
        end
      else { found no SIZE property }
        if find_property(macro^.invokes^.properties,
                         HAS_FIXED_SIZE_prop_name, search_prop) then
          begin
            add_to_prop_list(macro^.invokes^.parameters,
                             SIZE_prop_name, search_prop^.text);
          end;
    end { check_for_SIZE_parameter } ;


    procedure check_pin_names;
      { check all pin names of the body to make sure that all have actual
        parameters specified and "fix" all pin signal names. }
      var
        last,                           { previous element in bindings list }
        pin_name: bindings_list_ptr;    { pin name list element }
        fixed_name: xtring;             { pin name after being fixed }
    begin
      pin_name := macro^.invokes^.bindings;  last := NIL;
      while pin_name <> NIL do
        if pin_name^.actual_parameter = NIL then
          begin
            assert(165 { pin property on non-existent pin });
            write(CmpLog, 'Pin name = ');
            print_string(CmpLog, pin_name^.formal_parameter);
            writeln(CmpLog);
            if last = NIL then macro^.invokes^.bindings := pin_name^.next
            else last^.next := pin_name^.next;
            pin_name := pin_name^.next;
          end
        else
          begin
            fixed_name := fix_signal_name(pin_name^.formal_parameter);
            pin_name^.formal_parameter := fixed_name;
            last := pin_name;  pin_name := pin_name^.next;
          end;
    end { check_pin_names } ;


  begin { parse_invoke }
    if debug then disp_line('enter parse_invok');

    if debug_20 then writeln(outfile, '      Starting to parse an invoke');

    body_name := nullstring;  insymbol;      { eat the INVOKE symbol }

    if sy <> STRINGS then
      begin  assert(26 { missing macro name });  skip([ENDINVOKESY]); end
    else
      begin
        body_name := lex_string;

        { set up current environment for error reporting }

        push_error_info;
        current_body_name := body_name;

        body := check_for_special_body(body_name);

	insymbol;
	if sy = SEMI then insymbol else assert(35 { expected ; });

	if body = USER_BODY then
	  begin
	    new_invoke_list(macro^.invokes);
	    macro^.invokes^.macro_name := body_name;
	    macro^.invokes^.page_number := current_page;
	  end;

	if sy = PROPERTYSY then parse_properties(body);

	if sy = BINDINGSY then
	  if not (body IN bodies_with_bindings) then 
	    begin
	      error(136 { not allowed });
	      skip([ENDINVOKESY]);
	      display_invoke_error;
	    end
	  else
	    begin
	      parse_bindings(body);
	      if body = USER_BODY then check_pin_names;
	    end;

	if body = USER_BODY then
	  if is_comment_body(macro^.invokes^.properties) then
	    begin
	      { body is a COMMENT; ignore it completely! }

	      temp := macro^.invokes;
	      macro^.invokes := temp^.next;
	      release_invoke_list(temp);
	    end
	  else
	    begin
	      check_for_PATH_property;
	      check_for_SIZE_parameter;
	    end;

        pop_error_info;
      end;

    if sy = ENDINVOKESY then insymbol else assert(27 { no END_INVOKE });
    if sy = SEMI then insymbol else assert(35 { expected ; });

    if debug then disp_line('parse_invoke     ');
  end { parse_invoke } ;


begin { parse_macro_definition }
  if debug then disp_line('enter parse_macro');
  if debug_20 then writeln(outfile, '    Starting to parse the macro def');

  allowed_key_words := macrodef_keysys;

  if sy <> MACROSY then assert(28 { expected MACRO })
  else
    begin
      insymbol;     { eat the MACRO symbol }

      unique_body_number := 0;

      while (sy = INVOKESY) do parse_invoke;
    end;

  check_for_abbreviation(macro);

  if sy = ENDMACROSY then insymbol else assert(30 { expected END_MACRO });
  if sy <> PERIOD then assert(44 { expected . });

  if debug_20 then writeln(outfile, '    Done parsing the macro def');
  if debug then disp_line('parse_macro_defin');
end { parse_macro_definition } ;

        
(**)     { ------- read a post 6.0 GED connectivity file ------- }


function parse_connectivity_file(macro: macro_def_ptr): boolean;
  { read a connectivity file produced by a post 6.0 GED.  Check the selection
    expression.  If it exists and the directory has none, or is different,
    this means that a selection was placed one this page alone.  Evaluate
    it.  If it evaluates FALSE, do not process the drawing and return FALSE.
    If it evaluates TRUE, process the drawing and return TRUE. }
  var
    expr: xtring;                        { selection expression for drawing }
    net_table: net_table_ptr;            { head of net number table }
    unique_body_number: natural_number;  { serves to make all bodies unique }


  procedure dump_net_table;
    { dump the current net table to the debug file }
    var
      current_group: net_table_ptr;    { current group in the table }
      index: net_group_range;          { index into the group table }
  begin
    writeln(outfile);
    writeln(outfile, '---- Net table dump ----');

    current_group := net_table;
    while current_group <> NIL do
      begin
        for index := 0 to NET_GROUP_SIZE do
          if current_group^.nets[index] <> NIL then
            with current_group^.nets[index]^ do
              begin
                write(outfile, current_group^.group_number+index:2, '=');
                print_string(outfile, net_name);
                write(outfile, ' (');
                print_string(outfile, net_id);
                writeln(outfile, ')');
                dump_property_list(outfile, properties);
              end;

        current_group := current_group^.next;
      end;

    writeln(outfile);
  end { dump_net_table } ;


  function read_property_list(permission: name_types;
                              object: xtring): property_ptr;
    { read a property list from the input and return it.  Return NIL if
      there are no properties in the input file.  Make sure that every
      property in the list has the specified permission.  If an error
      occurs, use the specified OBJECT in the error message. }
    var
      prop_list: property_ptr;     { current property list }
      property_name: name_ptr;     { name of the property }
      property_value: xtring;      { value of the current property }
  begin
    prop_list := NIL;

    if sy = COLON then insymbol;

    while sy = IDENT do
      begin
        property_name := id.name;
        insymbol;                    { eat the property name ID }

        if sy <> STRINGS then
          begin  assert(18 { expected a string });  skip([IDENT,SEMI]);  end
        else
          begin
            property_value := lex_string;
            insymbol;                { eat the property value string }

            if permission in property_name^.kind then
              add_to_prop_list(prop_list, property_name, property_value)
            else
              begin
                if permission = PERMIT_BODY then
                  error(152 { not permitted on a body })
                else if permission = PERMIT_SIGNAL then
                  error(151 { not permitted on a signal })
                else if permission = PERMIT_PIN then
                  error(153 { not permitted on a body });

                error_dump_current_parse_environment;
                error_dump_property(property_name, property_value);

                if permission = PERMIT_PIN then
                  error_dump_pin_name_string(object)
                else if permission = PERMIT_SIGNAL then
                  error_dump_signal_name_string(object);
              end;
          end;
      end;

    read_property_list := prop_list;
  end { read_property_list } ;


  function find_net_in_table(net_number: net_number_range;
                             var net: net_descriptor_ptr): boolean;
    { find the given net in the table.  If it does not exist, create it
      and return it.  If the net is not already in the table, return FALSE. }
    var    
      current_group: net_table_ptr;      { current group in table }
      last: net_table_ptr;               { last group in table }
      done: boolean;                     { TRUE if group has been found or
                                           created. }
      net_index: net_group_range;        { index into table of nets }
  begin
    { find the table base for the net }

    current_group := net_table;  done := FALSE;  last := NIL;
    while (current_group <> NIL) and not done do
      if (net_number <= current_group^.group_number + NET_GROUP_SIZE) then
        begin
          done := TRUE;
	  if (net_number < current_group^.group_number) then
	    begin  { too far -- need to insert new element }
              new_net_table(current_group);  
              if last <> NIL then
                last^.next := current_group
              else
                net_table := current_group;

              current_group^.group_number :=
                net_number - (net_number MOD (NET_GROUP_SIZE+1));
            end;
	end
      else
        begin
          last := current_group;  current_group := current_group^.next;
        end;

    { create a new table base if not already there }

    if not done then
      begin
        new_net_table(current_group);
        if last <> NIL then
          last^.next := current_group
        else
          net_table := current_group;

        current_group^.group_number :=
                             net_number - (net_number MOD (NET_GROUP_SIZE+1));
      end;
        
    { get the net from the table }

    net_index := net_number - current_group^.group_number;
      
    net := current_group^.nets[net_index];
    if net <> NIL then
      find_net_in_table := TRUE
    else
      begin
        new_net_descriptor(net);
        current_group^.nets[net_index] := net;

        find_net_in_table := FALSE;
      end;
  end { find_net_in_table } ;


  function build_net_id(net_number: net_number_range): xtring;
    { create a net id property from the given net number and current page }
    var
      temp_string: xtring;      { temporary string }
  begin
    if current_page = 1 then
      create_a_string(temp_string, width_of_integer(net_number))
    else
      create_a_string(temp_string,
               width_of_integer(current_page)+width_of_integer(net_number)+1);

    temp_string^[0] := chr(0);

    if current_page <> 1 then
      begin
        if add_number_to_string(temp_string, current_page) then;
        if add_char_to_string(temp_string, '.') then;
      end;
    if add_number_to_string(temp_string, net_number) then;

    build_net_id := enter_and_release_string(temp_string);
  end { build_net_id } ;


  procedure read_nets;
    { read the nets from the input file }
    var
      net_number: net_number_range;    { number of the net being processed }
      net_name: xtring;                { name of the net }
      properties: property_ptr;        { properties of the net }
      net: net_descriptor_ptr;         { descriptor for current net }
  begin
    while sy = CONSTANT do
      begin
        net_number := 0;  net_name := nullstring;  properties := NIL;

        if sy <> CONSTANT then
          begin  assert(19 { expected a constant });  skip([SEMI]);  end
        else
          begin
            net_number := const_val;
            insymbol;                   { eat the net number }

            if sy <> STRINGS then
              begin  assert(18 { expected a string });  skip([SEMI]);  end
            else
              begin
                net_name := lex_string;
                insymbol;               { eat the net name string }

                properties := read_property_list(PERMIT_SIGNAL, net_name);
              end;
          end;

        if find_net_in_table(net_number, net) then
          assert(108 { net already specified })
        else
          begin
            net^.net_name := fix_signal_name(net_name);
            net^.properties := properties;
            net^.net_id := build_net_id(net_number);
          end;

        if sy = SEMI then insymbol else assert(35 { expected a ; });
      end;
  end { read_nets } ;


  procedure read_body;
    { read a body from the input connectivity file }
    var
      body_name: xtring;         { name of the body being read }
      body: body_type;           { type of body }
      body_version: xtring;      { version of the body used }
      XY_position: xtring;       { XY position of the body on the page }
      rotation: xtring;          { rotation of the body }
      directory: xtring;         { directory from which the body was taken }
      path: xtring;              { PATH property on the body }
      parameters: property_ptr;  { parameters attached to the body }
      body_properties:
                  property_ptr;  { body properties attached to the body }


    procedure generate_body_error(body: body_type);
      { generate an error - this body is not allowed }
    begin
      if body = MENU_BODY then error(142 { MENUs not supported })
      else if body = DECLARE_BODY then error(219 { DECLAREs not supported });
      error_dump_current_parse_environment;

      release_entire_property_list(body_properties);
      release_entire_property_list(parameters);
    end { generate_body_error } ;


    procedure set_up_properties(body: invoke_list_ptr;
                                var property: property_ptr);
      { add the properties from the given list (PROPERTY) to the
        given body invocation.  If the property has the PARAMETER attribute,
        add it to the invocation's parameter list otherwise add it to the
        property list. }
      var
        next: property_ptr;           { next property in the list }
    begin
      while property <> NIL do
        begin
          next := property^.next;

          if (parameter_attributes * property^.name^.kind) <> [] then
            check_and_add_to_prop_list(body^.parameters,
                                       property^.name, property^.text)
          else
            check_and_add_to_prop_list(body^.properties,
                                       property^.name, property^.text);

          release_property(property);

          property := next;
        end;

      add_to_prop_list(body^.properties, XY_prop_name, XY_position);
      add_to_prop_list(body^.properties, DIRECTORY_prop_name, directory);
      add_to_prop_list(body^.properties, VER_prop_name, body_version);
    end { set_up_properties } ;


    procedure fix_up_PATH_property(var path: xtring; body: invoke_list_ptr);
      { make sure there is a PATH property and add to the property list }
    begin
      if path = nullstring then
        begin
          error(197 { PATH property not found });
          error_dump_current_parse_environment;
          path := generate_unique_path_name(unique_body_number);
        end;

      body^.path := path;
    end { fix_up_PATH_property } ;


    procedure fix_up_SIZE_parameter(body: invoke_list_ptr);
      { check for the presence of HAS_FIXED_SIZE }
      var
        prop: property_ptr;          { property returned from the search }
        search_prop: property_ptr;   { property returned from search }
    begin
      if find_property(body^.parameters, SIZE_prop_name, prop) then
        begin
          if find_property(body^.properties,
                           NEEDS_NO_SIZE_prop_name, search_prop) then
            begin
              error(144 { can't have one on this body! });
              error_dump_current_parse_environment;
            end
          else if find_property(macro^.invokes^.properties,
                                HAS_FIXED_SIZE_prop_name, search_prop) then
            begin
              error(144 { can't have one on this body! });
              error_dump_current_parse_environment;
              prop^.text := search_prop^.text;
            end
        end

      else { found no SIZE property }
        if find_property(body^.properties,
                         HAS_FIXED_SIZE_prop_name, search_prop) then
          begin
            add_to_prop_list(body^.parameters,
                             SIZE_prop_name, search_prop^.text);
          end;
    end { fix_up_SIZE_parameter } ;


    procedure fix_up_ROTATION_property(rot: xtring; body: invoke_list_ptr);
      { check for a legal rotation and add rotation as property to list.
        Possible rotations are:
             0:  0   degree rotation (default).   LEGAL
             1:  90  degree rotation (up).        LEGAL
             2:  mirror of 0 degrees (left).      LEGAL
             3:  mirror of 90 degrees (down).     LEGAL
             4:  180 degree rotation (left).      ILLEGAL
             5:  270 degree rotation (down).      ILLEGAL }
    begin
      if not (rot^[1] IN ['0','1','2','3']) then
        begin
          error(143 { illegal rotation });
          error_dump_current_parse_environment;
          error_dump_indent(indent);
          if rot^[1] = '4' then
            error_dump_alpha('180 deg rotation')
          else if rot^[1] = '5' then
            error_dump_alpha('270 deg rotation');
          error_dump_CRLF;
        end;

      if (parameter_attributes * ROTATION_prop_name^.kind) <> [] then
        add_to_prop_list(body^.parameters, ROTATION_prop_name, rot)
      else
        add_to_prop_list(body^.properties, ROTATION_prop_name, rot);
    end { fix_up_ROTATION_property } ;


    function read_bindings: bindings_list_ptr;
      { read the bindings section of the current body and return them }
      var
        bindings: bindings_list_ptr;  { list of bindings for the body }
        pin_name: xtring;             { name of the pin }
        properties: property_ptr;     { properties on the pin }
        done: boolean;                { TRUE if done processing nets }
        net_number: net_number_range; { net identifying number }


      procedure set_up_actual_parameter(var actual: clear_text_actual_list_ptr;
                                        net_number: net_number_range);
        { set up the signal name for the given actual (ACTUAL) and the given
          net number (NET_NUMBER). }
        var
          net: net_descriptor_ptr;               { pointer to the actual net }
      begin
        { find the specified net in the NET table }

        if not find_net_in_table(net_number, net) then
          assert(107 { net number not in table })
        else
          begin
            actual^.actual_parameter := net^.net_name;
            actual^.properties := net^.properties;
            actual^.net_id := net^.net_id;
          end;
      end { set_up_actual_parameter } ;


    begin { read_bindings }
      bindings := NIL;

      while sy = STRINGS do
        begin
          pin_name := lex_string;
          insymbol;

          properties := read_property_list(PERMIT_PIN, pin_name);

          new_bindings_list(bindings);
          bindings^.formal_parameter := fix_signal_name(pin_name);
          bindings^.pin_properties := properties;

          done := FALSE;
          repeat
            if sy = CONSTANT then net_number := const_val
                             else assert(9 { expected a net constant });
            insymbol;

            new_clear_text_actual_list(bindings^.actual_parameter);
            set_up_actual_parameter(bindings^.actual_parameter, net_number);

            if sy = COMMA then insymbol else done := TRUE;
          until done;

          if sy = SEMI then insymbol else assert(35 { expected ; });
        end;

      read_bindings := bindings;
    end { read_bindings } ;


    procedure process_DEFINE_body(var property_list: property_ptr);
      { process properties attached to the DEFINE body.  The property list
        is released. }
      var
        next: property_ptr;          { next text macro in the list }
        text_macro: property_ptr;    { current text macro }
        dummy: property_ptr;         { dummy procedure return for search }
    begin
      text_macro := property_list;
      while text_macro <> NIL do
        begin
          next := text_macro^.next;

          if find_property(macro^.text_macros, text_macro^.name, dummy) then
            begin
              if text_macro^.text <> dummy^.text then
	        begin
                  error(114 { text macro already exists });
                  error_dump_current_parse_environment;
                  error_dump_text_macro(text_macro^.name);
		end;
            end
          else if RESERVED IN text_macro^.name^.kind then
            begin
              error(105 { reserved TM name });
              error_dump_current_parse_environment;
              error_dump_text_macro(text_macro^.name);
            end
          else
	    begin
              add_to_prop_list(macro^.text_macros,
                               text_macro^.name, text_macro^.text);
              if macro = root_macro_def then
	        enter_local_tm(paged_schema_of_this_page,
		               text_macro^.name, text_macro^.text);
            end;
		  
          release_property(text_macro);

          text_macro := next;
        end;
    end { process_DEFINE_body } ;


    procedure process_DRAWING_body(var property_list: property_ptr);
      { process the properties attached to the DRAWING body.  The property
        list is released. }
      var
        property: property_ptr;     { current property in the list }
        next: property_ptr;         { next property in the list }
    begin
      property := property_list;
      while property <> NIL do
        begin
          next := property^.next;

          if property^.name = TITLE_prop_name then
            begin
              if macro^.macro_name <> property^.text then
                begin
                  error(182 { it doesn't match! });
                  error_dump_current_parse_environment;
                  error_dump_indent(indent);
                  error_dump_alpha('TITLE prop=     ');
                  error_dump_string(property^.text);
                  error_dump_CRLF;
                end;
            end

          else if property^.name = EXPR_prop_name then
            { ignore this }

          else check_and_add_to_prop_list(macro^.properties,
                                          property^.name, property^.text);

          release_property(property);

          property := next;
        end;
    end { process_DRAWING_body } ;


  begin { read_body }
    insymbol;     { read the body start flag (%) }
    directory := nullstring; 

    if sy <> STRINGS then
      begin  assert(26 { expected macro name });  skip([PERCENT,ENDSY]);  end
    else
      begin
        body_name := lex_string;
        insymbol;             { eat the body name string }

        push_error_info;  current_body_name := body_name;

        body := check_for_special_body(body_name);

        if sy = STRINGS then body_version := lex_string
                        else assert(18 { expected a string });
        insymbol;
        if sy = COMMA then insymbol else assert(10 { expected , });

        if sy = STRINGS then XY_position := lex_string
                        else assert(18 { expected a string });
        insymbol;
        if sy = COMMA then insymbol else assert(10 { expected , });

        if sy = STRINGS then rotation := lex_string
                        else assert(18 { expected a string });
        insymbol;
        if sy = COMMA then insymbol else assert(10 { expected , });

        if sy = STRINGS then directory := lex_string
                        else assert(18 { expected a string });
        insymbol;
        if sy = COMMA then insymbol else assert(10 { expected , });

        if sy = STRINGS then path := lex_string
                        else assert(18 { expected a string });
        insymbol;

        current_path_prop := path;

        if sy = SEMI then insymbol else assert(35 { expected ; });

        parameters := read_property_list(PERMIT_BODY, nullstring);
        if sy = SEMI then insymbol else assert(35 { expected ; });

        body_properties := read_property_list(PERMIT_BODY, nullstring);
        if sy = SEMI then insymbol else assert(35 { expected ; });

        if is_comment_body(body_properties) then
          skip([PERCENT,ENDSY])
        else
          case body of
            MENU_BODY:       generate_body_error(body);

            DEFINE_BODY:     process_DEFINE_body(body_properties);

            DRAWING_BODY:    process_DRAWING_body(body_properties);

            PIN_NAMES_BODY:  begin
                               skip([PERCENT,ENDSY]);
                               release_entire_property_list(parameters);
                               release_entire_property_list(body_properties);
                             end;

            USER_BODY:       begin
                               new_invoke_list(macro^.invokes);
                               macro^.invokes^.macro_name := body_name;
                               macro^.invokes^.page_number := current_page;

                               macro^.invokes^.parameters := parameters;
                               set_up_properties(macro^.invokes,
                                                 body_properties);

                               macro^.invokes^.bindings := read_bindings;

                               fix_up_PATH_property(path, macro^.invokes);
                               fix_up_SIZE_parameter(macro^.invokes);
                               fix_up_ROTATION_property(rotation,
                                                        macro^.invokes);
                             end;

            DECLARE_BODY:    generate_body_error(body);
          end;
          
        pop_error_info;
      end;
  end { read_body } ;


begin { parse_connectivity_file }
  if debug then disp_line('enter parse_conne');

  if debug_20 then
    writeln(outfile, '    Starting to parse the connectivity file');

  net_table := NIL;
  unique_body_number := 0;
  allowed_key_words := directory_keysys;

  if sy = IDENT then expr := get_EXPR_property
                else expr := nullstring;

  if not evaluate_selection_expression(expr) then
    parse_connectivity_file := FALSE
  else
    begin
      parse_connectivity_file := TRUE;

      if sy = CONSTANT then read_nets;

      while sy = PERCENT do read_body;
    
      check_for_abbreviation(macro);

      if expr <> nullstring then
        check_and_add_to_prop_list(macro^.properties, EXPR_prop_name, expr);

      if sy = ENDSY then insymbol else assert(34 { expected END });
      if sy <> PERIOD then assert(44 { expected . });

      if debug_8 or printmacros_ok then dump_net_table;

      release_entire_net_table(net_table);
    end;

  if debug_20 then
    writeln(outfile, '    done parsing the connectivity file');
  if debug then disp_line('parse_connectivit');
end { parse_connectivity_file } ;


(**)     { ------- read a specified macro definition ------- }


function read_macro_def(macro_name: xtring): macro_def_ptr;
  { read the specified macro.  If there is more than one version, read
    the one whose selection expression is TRUE.  Return a pointer to the
    descriptor of the macro definition. }
  var
    MDP: macro_def_ptr;               { macro def to be returned }
    version: macro_module_ptr;        { version of the macro }
    plumbing: plumbing_module_ptr;


  function get_the_macro(version: plumbing_module_ptr;
                         modl: macro_module_ptr): macro_def_ptr;
    { read the pages of the plumbing body into the given macro def }
    var
      curr_file: plumbing_page_ptr;      { current file being read }
      current_file_type: file_types;     { type of the file being read }
      macro_def: macro_def_ptr;          { macro def being read }
      error_detected: boolean;           { TRUE if file error detected }
      read_one_page: boolean;            { TRUE if at least one page read }
  begin
    if debug_20 then writeln(outfile, '  Starting to get the macro');

    new_macro_def(macro_def);

    macro_def^.is_leaf_macro := er_isprim(modl);

    macro_def^.macro_name := macro_name;
    macro_def^.version := modl;
    version^.macro := macro_def;

    push_error_info;
    current_macro_def := macro_def;

    error_detected := FALSE;
    read_one_page := FALSE;

    curr_file := version^.pages;
    while curr_file <> NIL do
      begin
        allowed_key_words := macrodef_keysys + directory_keysys;

        { set up the current parse environment }

        current_page := curr_file^.page_number;
        current_file_name := curr_file^.filename;

        if not open_a_file(curr_file^.filename, STANDARD_FILE) then
          begin
            error(205 { cannot open this file });
            error_dump_current_parse_environment;

            error_detected := TRUE;
          end
        else
          begin
            current_file_type := get_file_type;

            if current_file_type = MACRO_DEFINITION then
              begin
                parse_macro_definition(macro_def);
                read_one_page := TRUE;
              end
            else if current_file_type = CONNECTIVITY then
              begin
                if parse_connectivity_file(macro_def) then
                  read_one_page := TRUE;
              end
            else error(86 { invalid file type });

            if not close_parse_file(STANDARD_FILE) then
              begin
                error(168 { cannot close the file });
                error_dump_file_name(curr_file^.filename);
              end;
          end;

        curr_file := curr_file^.next;
      end;
    allowed_key_words := [];

    if not read_one_page then
      begin
        error(177 { selection expression is false });
        error_dump_macro_def(macro_def);
      end;

    if error_detected or not read_one_page then
      begin
         get_the_macro := NIL;
	 release_complete_macro_def(macro_def);
      end
    else
      begin
        get_the_macro := macro_def;
        macro_def^.next := macro_def_list_root;
        macro_def_list_root := macro_def;
      end;

    pop_error_info;
  end { get_the_macro } ;


  function enter_plumbing_def(name: xtring;
                              version: macro_module_ptr): plumbing_module_ptr;
    { find or create (entering all pages).  Assume it's a drawing. 
      Record all pages in the dependency list of the current root page. }
    var
      ext: name_ptr;
      vers: version_range;
      current: plumbing_module_ptr;
      hash: longint;
      found: boolean;
      i: string_range;


    procedure record_dependencies(pages: plumbing_page_ptr);
    begin
      while pages <> NIL do
        begin
	  add_to_dependency_list(pages);
	  pages := pages^.next;
	end;
    end { record_dependencies } ;


    function enter_pages: plumbing_page_ptr;
      var
	head: plumbing_page_ptr;
	page_num: page_range;
	last_page: plumbing_page_ptr;


      function insert_page(num: page_range; var where: plumbing_page_ptr):
	plumbing_page_ptr;
	var
	  fname: xtring;
	  pg: plumbing_page_ptr;
      begin
	fname := 
	  enter_string(er_filename(version, ord(CONNECTIVITY), num, NIL));
	if fname = nullstring then pg := where
	else
	  begin
	    new(pg);
	    increment_heap_count(HEAP_PLUMBING_PAGE, 
	                         2*POINTER_SIZE+2*INT_SIZE);
	    with pg^ do
	      begin
		next := where;  where := pg;
      
		page_number := num;
		filename := fname;
		last_modified_time := 0;
	      end;
	  end;
	insert_page := pg;
      end { insert_page } ;


    begin { enter_pages }
      if debug_24 then
        begin
	  write(Outfile, 'Enter_pages of plumbing ');
	  dump_string(Outfile, name);
	  write(Outfile, '.');
	  dump_string(Outfile, er_extension(version));
	  writeln(Outfile, '.', er_version(version):1);
	end;

      head := NIL;  last_page := NIL;
      page_num := er_page(version);
      while page_num <> 0 do
	begin
	  if last_page = NIL then
	    last_page := insert_page(page_num, head)
	  else 
	    last_page := insert_page(page_num, last_page^.next);
	  page_num := er_page(version);
	end;
      enter_pages := head;
    end { enter_pages } ;


  begin { enter_plumbing_def }
    ext := name_from_string(er_extension(version));
    vers := er_version(version);

    hash := 0;
    for i := 1 to ord(name^[0]) do hash := hash + ord(name^[i]);
    hash := hash mod (LAST_PLUMBING_BUCKET + 1);
    current := plumbing_table[hash];  found := FALSE;
    while (current <> NIL) and not found do
      if (current^.macro_name = name) and
	 (current^.extension = ext) and 
	 (current^.version_number = vers) then found := TRUE
      else current := current^.next;

    if current = NIL then
      begin
        new(current);
	increment_heap_count(HEAP_PLUMBING_MODULE, 5*POINTER_SIZE+INT_SIZE);
	with current^ do
	  begin
	    next := plumbing_table[hash];
	    plumbing_table[hash] := current;

	    macro_name := name;
	    extension := ext;
	    version_number := vers;
	    pages := NIL;
	    macro := NIL;

            pages := enter_pages;
	  end;
      end;

    record_dependencies(current^.pages);
    enter_plumbing_def := current;
  end { enter_plumbing_def } ;


  function concoct_forced_primitive(name: xtring;
				    version: macro_module_ptr): macro_def_ptr;
    { Build data structures that whould have been built if the connectivity
      file existed for PRIM.1.1 }
    var
      macro: macro_def_ptr;   { Value for return }
  begin
    new_macro_def(macro);
    macro^.next := root_macro_def;  root_macro_def := macro;
  
    macro^.is_leaf_macro := TRUE;
    macro^.macro_name := name;
    macro^.version := version;
  
    add_to_prop_list(macro^.properties,ABBREV_prop_name, concoct_abbrev(name));
  
    { ignore properties to define the non-graphical model -- this is 
      an error }

    concoct_forced_primitive := macro;
  end { concoct_forced_primitive } ;


begin { read_macro_def }
  if debug then
    disp_line('enter read_Mdef  ');

  if debug_20 then writeln(outfile, 'Starting to read a macro def');

  MDP := NIL;
  version := select_module(macro_name, nullstring, 0);
  if version <> NIL then
    begin
      if er_extension(version) = NIL then
	{ illegal, but possible -- error emitted elsewhere }

        { !!! CHANGE this to emit the error here, or else }
	MDP := concoct_forced_primitive(macro_name, version)
      else
        begin
	  plumbing := enter_plumbing_def(macro_name, version);
	  MDP := plumbing^.macro;
	  if MDP = NIL then  { macro hasn't yet been read in }
	    begin
	      MDP := get_the_macro(plumbing, version);
	      if printmacros_ok and (MDP <> NIL) then
	        print_macro(outfile, MDP);
	    end;
	end;
    end;

  read_macro_def := MDP;

  if debug then disp_line('read_Mdef        ');
end { read_macro_def } ;
