#if VAX
  procedure uxclose(BYVALUE fd: Cint); external;
  function uxread(BYVALUE fd: Cint; %REF buf: char_array;
                  BYVALUE nbytes: Cint): Cint; external;
  procedure uxwrite(BYVALUE fd: Cint; %REF buf: [UNSAFE] char_array;
                    BYVALUE nbytes: Cint); external;
#endif


procedure print_alpha_continue(var f: textfile; name: alpha);
  { print the given alpha (NAME) to the given file (F).  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
  var
    i,len: id_range;
begin
  len := alpha_length(name);
  if (len > 0) and (len + column < MAX_OUTPUT_FILE_LENGTH - 1) then
    begin
      write(f, name:len);
      column := column + len;
    end
  else
    for i := 1 to len do
      begin
	if column >= MAX_OUTPUT_FILE_LENGTH-2 then
	  begin  writeln(f, CONTINUATION_CHAR);  column := 0;  end;
  
	write(f, name[i]);  column := column + 1;
      end;
end { print_alpha_continue } ;


procedure print_char_continue(var f: textfile; ch: char);
  { print the given string (STR) to the given file (F).  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
begin
  if column >= MAX_OUTPUT_FILE_LENGTH-2 then
    begin  writeln(f, CONTINUATION_CHAR);  column := 0;  end;

  write(f, ch);  column := column + 1;
end { print_char_continue } ;


procedure print_string_quoted_continue(var f: textfile; str: xtring);
  { print the given string (STR) quoted to the given file (F).  Double
    any embedded quoted in the string.  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
  var
    start,len: string_range;
    stop: integer;  { may exceed MAX_STRING_LENGTH by 1 }
    hack: string_hack;
    found_quote: boolean;
begin
  print_char_continue(f, OUTPUT_QUOTE_CHAR);

  len := ord(str^[0]);  
  stop := 1;
  while stop <= len do
    begin
      if column >= MAX_OUTPUT_FILE_LENGTH-2 then
	begin  writeln(f, CONTINUATION_CHAR);  column := 0;  end;
      start := stop;  found_quote := FALSE;
      while (stop <= len) and (column < MAX_OUTPUT_FILE_LENGTH - 2) and
        not found_quote do
	begin
	  column := column + 1;
	  found_quote := str^[stop] = OUTPUT_QUOTE_CHAR;
	  stop := stop + 1;
	end;
      hack.i := ord(str) + start;
      write(f, hack.s^:(stop - start));
      if found_quote then print_char_continue(f, OUTPUT_QUOTE_CHAR);
    end;

  print_char_continue(f, OUTPUT_QUOTE_CHAR);
end { print_string_quoted_continue } ;


procedure pipe_init(var p: pipe);
  { initialize the pipe to "undefined" }
begin
  p.which := FIRST_PIPE;
  p.state := PIPE_NOT_OPEN;
  p.is_pipe := FALSE;
  p.fd := -1;
  p.name := nullstring;
  p.written := FALSE;
end { pipe_init } ;


procedure dump_pipe(var f: textfile; var p: pipe; 
                    indentation: natural_number);
  { Dump the pipe record to file f (for debugging). }
begin
  if indentation > 0 then write(f, ' ':indentation);
  case p.which of
    SCHEMA_PIPE: write(f, 'SCHEMA_PIPE ');
    EXPANSION_PIPE: write(f, 'EXPANSION_PIPE ');
    TO_ET_PIPE: write(f, 'TO_ET_PIPE ');
    TEMP_PIPE: write(f, 'TEMP_PIPE ');
    FROM_ET_PIPE: write(f, 'FROM_ET_PIPE ');
    OTHERWISE write(f, 'UNKNOWN_PIPE ');
  end;
  case p.state of
    PIPE_NOT_OPEN: write(f, 'unopened');
    PIPE_OPEN_FOR_READ: write(f, 'reading from ');
    PIPE_OPEN_FOR_WRITE: write(f, 'writing to ');
  end;
  if p.state <> PIPE_NOT_OPEN then
    if p.is_pipe then write(f, 'fd ', p.fd:1)
    else
      if p.name = nullstring then write(f, 'logical file')
      else writestring(f, p.name);
  writeln(f, '.');
end { dump_pipe } ;


function pipe_readln(var p: pipe; line: xtring): read_result;
  { Read a line from the pipe/file.  Return TRUE if this has been
    done (FALSE if eof or not open). }
  label
    90;  { return }
  const
    C_EOF = 2;         { eof in Creadln }
    C_OVF = 1;         { line overflow in Creadln }
    C_OK = 0;          { Creadln succeeded }
  var
    Cval: Cint;         { condition code returned by Creadln }


#if VAX
  function readline(var f: textfile; line: xtring): read_result;
    { read a line from the specified input file into the already
      allocated line. }
    var
      ch: char;             { current char }
      i: string_range;      { index into the line }
  begin
    if eof(f) then readline := READ_EOF
    else
      begin
        readline := READ_OK;
        i := 0;
        while not eoln(f) and not eof(f) do
          begin
            read(f, ch);
            if i < MAX_STRING_LENGTH then
              begin  i := i + 1;  line^[i] := ch;  end
            else
              begin
                readline := READ_OVERFLOW;
                while not eof(f) and not eoln(f) do get(f);
              end;
          end;
	if not eof(f) then readln(f);
      end;
    line^[0] := chr(i);  
  end { readline } ;


  function fd_readline(fd: Cint; line: xtring): read_result;
    { perform readline using "unix" reads on a file descriptor (which
      presumably is a pipe from et). }
    var
      ch: char_array;   { ch[0] is current char }
      i: string_range;  { index into line }
      num: -1..1;       { number of chars read or -1 }
  begin
    num := uxread(fd, ch, 1);
    if num < 1 then fd_readline := READ_EOF
    else fd_readline := READ_OK;
    i := 0;
    while (num > 0) and (ch[0] <> chr(PIPE_EOL)) do
      begin
        if i < MAX_STRING_LENGTH then
          begin  i := i + 1;  line^[i] := ch[0];  end
        else fd_readline := READ_OVERFLOW;
        num := uxread(fd, ch, 1);
      end;
    line^[0] := chr(i);
  end { fd_readline } ;
#endif VAX

begin { pipe_readln }
  { This routine assumes for now that there is only 1 possible input
    pipe -- FROM_ET_PIPE.  If more are added, then case statements will
    be necessary here. }

  pipe_readln := READ_EOF;
  if p.state <> PIPE_OPEN_FOR_READ then
    begin
      assert(0);
      writeln(Cmplog, ' Reading an pipe not open for read !!');
      goto 90 { return } ;
    end;
  if p.which <> FROM_ET_PIPE then
    begin
      assert(0);
      writeln(Cmplog, ' Read(Unknown PIPE/file ', ord(p.which):1, ')');
      goto 90 { return } ;
    end;

#if UNIX
  Cval := creadln(CmpDraw, line);
  case Cval of
    C_EOF: pipe_readln := READ_EOF;
    C_OVF: pipe_readln := READ_OVERFLOW;
    C_OK: pipe_readln := READ_OK;
  end;
#endif
#if VAX
  if p.is_pipe then pipe_readln := fd_readline(p.fd, line)
  else pipe_readln := readline(CmpDraw, line); 
#endif
90:
end { pipe_readln } ;


procedure pipe_dump_string_quoted(p: pipe;  str: xtring);
  { dump the string, quoted (and parseable) to the given pipe/file }


#if VAX
  procedure fd_dump_string_quoted(fd: Cint; str: xtring);
    { dump the string to the descriptor }
    var
      quote_buf: nolength_string;
      len, start: string_range;
      stop: integer;  { may go up to MAX_STRING_LENGTH + 1 }
      found_quote: boolean;
      hack: string_hack;
  begin
    quote_buf[1] := OUTPUT_QUOTE_CHAR;
    uxwrite(fd, quote_buf, 1);

    len := ord(str^[0]);
    stop := 1;
    while stop <= len do
      begin
	start := stop;
        found_quote := FALSE;
        while (stop <= len) and not found_quote do
	  begin
	    found_quote := str^[stop] = OUTPUT_QUOTE_CHAR;
	    stop := stop + 1;
	  end;
	hack.i := ord(str) + start;
        uxwrite(fd, hack.s^, stop - start);
	if found_quote then uxwrite(fd, quote_buf, 1);
      end;

    uxwrite(fd, quote_buf, 1);
  end { fd_dump_string_quoted } ;
#endif VAX


begin
  p.written := TRUE;
  case p.which of 
    EXPANSION_PIPE:
      begin
#if VAX
        if p.is_pipe then fd_dump_string_quoted(p.fd, str)
        else
#endif
          print_string_quoted_continue(CmpExp, str);
      end;
    SCHEMA_PIPE:
      begin
#if VAX
        if p.is_pipe then fd_dump_string_quoted(p.fd, str)
        else
#endif
          print_string_quoted_continue(CmpSchem, str);
      end;
    TO_ET_PIPE:
      begin
#if VAX
        if p.is_pipe then fd_dump_string_quoted(p.fd, str)
        else
#endif
          print_string_quoted_continue(Design, str);
      end;
    TEMP_PIPE:
      begin
#if VAX
        if p.is_pipe then fd_dump_string_quoted(p.fd, str)
        else
#endif
          print_string_quoted_continue(CmpTmp, str);
      end;
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
end { pipe_dump_string_quoted } ;


procedure pipe_dump_alpha(var p: pipe; name: alpha);
  { Write the name to the pipe }
begin
  p.written := TRUE;
  case p.which of 
    EXPANSION_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, name, alpha_length(name))
        else
#endif
          print_alpha_continue(CmpExp, name);
      end;
    SCHEMA_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, name, alpha_length(name))
        else
#endif
          print_alpha_continue(CmpSchem, name);
      end;
    TO_ET_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, name, alpha_length(name))
        else
#endif
        print_alpha_continue(Design, name);
      end;
    TEMP_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, name, alpha_length(name))
        else
#endif
        print_alpha_continue(CmpTmp, name);
      end;
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
end { pipe_dump_alpha } ;


procedure pipe_dump_integer(var p: pipe; i: longint);
  { Write the integer to the pipe }
var
  numb: alpha;     { buffer for number }
  index: id_range; { index into numb }


  procedure build_number(n: natural_number);
    { add the given number to the string }
  begin
    if n > 9 then build_number(n DIV 10);
    numb[index] := chr((n mod 10) + ord('0'));
    index := index + 1;
  end { build_number } ;


begin { pipe_dump_integer }
  { It is assumed that alphas are always more than long enough to hold 
    the textual representation of a longint. }
  p.written := TRUE;
  numb := NULL_ALPHA;
  if i < 0 then
    begin  i := -i;  numb[1] := '-';  index := 2;  end
  else index := 1;
  build_number(i);
  pipe_dump_alpha(p, numb);
end { pipe_dump_integer } ;


procedure pipe_dump_char(var p: pipe; ch: char);
  { Write the char to the pipe }
#if VAX
  var
    buf: char_array;  { buffer for writing to real pipe }
#endif
begin
  p.written := TRUE;
#if VAX
  if p.is_pipe then buf[0] := ch;
#endif
  case p.which of 
    EXPANSION_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, buf, 1)
        else
#endif
          print_char_continue(CmpExp, ch);
      end;
    SCHEMA_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, buf, 1)
        else
#endif
          print_char_continue(CmpSchem, ch);
      end;
    TO_ET_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, buf, 1)
        else
#endif
          print_char_continue(Design, ch);
      end;
    TEMP_PIPE:
      begin
#if VAX
        if p.is_pipe then uxwrite(p.fd, buf, 1)
        else
#endif
          print_char_continue(CmpTmp, ch);
      end;
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
end { pipe_dump_char } ;


procedure pipe_dump_CRLF(var p: pipe);
  { Write CRLF to the pipe }
begin
  p.written := TRUE;
#if VAX
  if p.is_pipe then pipe_dump_char(p, chr(PIPE_EOL))
  else
#endif
  case p.which of 
    EXPANSION_PIPE:
      begin
        writeln(CmpExp);
      end;
    SCHEMA_PIPE:
      begin
        writeln(CmpSchem);
      end;
    TO_ET_PIPE:
      begin
        writeln(Design);
      end;
    TEMP_PIPE:
      begin
        writeln(CmpTmp);
      end;
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  column := 0;
end { pipe_dump_CRLF } ;


function pipe_is_pipe(var p: pipe): boolean;
  { return TRUE if pipe is actually opened to inter-process communication
    (rather than to a file) }
begin
  pipe_is_pipe := p.is_pipe;
end { pipe_is_pipe } ;


function pipe_reset_file(var p: pipe; which: pipe_files; name: xtring): 
  boolean;
  { perform reset file on the file associated with "which" and format the
    pipe record to indicate this.  Return TRUE iff successful.  Name is
    presumed to remain a viable string pointer until after pipe is closed. }
begin
  pipe_init(p);
  p.is_pipe := FALSE;
  p.which := which;
  p.name := name;
  case which of
    FROM_ET_PIPE: 
      if reset_file(name, CMPDRAW_FILE) then p.state := PIPE_OPEN_FOR_READ;
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  pipe_reset_file := (p.state = PIPE_OPEN_FOR_READ);
  if debug_22 then
    begin
      write(Outfile, 'pipe_reset_file');
      if (p.state = PIPE_OPEN_FOR_READ) then write(Outfile, '(T): ')
                                        else write(Outfile, '(F): ');
      dump_pipe(Outfile, p, 0);
    end;
end { pipe_reset_file } ;


function pipe_rewrite_file(var p: pipe; which: pipe_files; name: xtring): 
  boolean;
  { perform rewrite file on the file associated with "which" and format the
    pipe record to indicate this.  Return TRUE iff successful. Name is
    presumed to remain a viable string pointer until after pipe is closed. }
begin
  pipe_init(p);
  p.is_pipe := FALSE;
  p.which := which;
  p.name := name;
  case which of
    EXPANSION_PIPE: 
      if rewrite_file(CmpExp, name, CMPEXP_FILE_NAME) then
        p.state := PIPE_OPEN_FOR_WRITE;
    TO_ET_PIPE: 
      if rewrite_file(Design, name, DESIGN_FILE_NAME) then
        p.state := PIPE_OPEN_FOR_WRITE;
    TEMP_PIPE: 
      if rewrite_file(CmpTmp, name, CMPTMP_FILE_NAME) then
        p.state := PIPE_OPEN_FOR_WRITE;
    OTHERWISE
      begin
        assert(0);

        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  pipe_rewrite_file := (p.state = PIPE_OPEN_FOR_WRITE);
  debug_22 := TRUE;
  if debug_22 then
    begin
      write(Outfile, 'pipe_rewrite_file');
      if (p.state = PIPE_OPEN_FOR_WRITE) then write(Outfile, '(T): ')
                                         else write(Outfile, '(F): ');
      dump_pipe(Outfile, p, 0);
    end;
end { pipe_rewrite_file } ;


function pipe_reset_descriptor(var p: pipe; which: pipe_files; fd: Cint): 
  boolean;
  { Set up pipe p to read from the given (presumed open) descriptor. 
    Return TRUE iff successful. }
begin
  pipe_init(p);
  p.is_pipe := TRUE;
  p.which := which;
  p.fd := fd;
  case which of
    FROM_ET_PIPE: 
#if UNIX
      if dup_inputfile_descriptor(CmpDraw, fd) then
        begin
          p.state := PIPE_OPEN_FOR_READ;
	  p.fd := cfdsc(CmpDraw);
	end;
#endif
#if VAX
      p.state := PIPE_OPEN_FOR_READ;
#endif
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  pipe_reset_descriptor := (p.state = PIPE_OPEN_FOR_READ);
  if debug_22 then
    begin
      write(Outfile, 'pipe_reset_desc');
      if (p.state = PIPE_OPEN_FOR_READ) then write(Outfile, '(T): ')
                                        else write(Outfile, '(F): ');
      dump_pipe(Outfile, p, 0);
    end;
end { pipe_reset_descriptor } ;


function pipe_rewrite_descriptor(var p: pipe; which: pipe_files; fd: Cint): 
  boolean;
  { Set up pipe p to write to the given (presumed open) descriptor.
    Return TRUE iff successful. }
begin
  pipe_init(p);
  p.is_pipe := TRUE;
  p.which := which;
  p.fd := fd;
  case which of
    EXPANSION_PIPE: 
#if UNIX
      if dup_file_descriptor(CmpExp, fd, UNBUF_WRITE) then
        begin
          p.state := PIPE_OPEN_FOR_WRITE;
	  p.fd := text_file_descr(CmpExp);
	end;
#endif
#if VAX
      p.state := PIPE_OPEN_FOR_WRITE;
#endif
    SCHEMA_PIPE:
#if UNIX
      if dup_file_descriptor(CmpSchem, fd, UNBUF_WRITE) then
        begin
          p.state := PIPE_OPEN_FOR_WRITE;
	  p.fd := text_file_descr(CmpSchem);
	end;
#endif
#if VAX
      p.state := PIPE_OPEN_FOR_WRITE;
#endif
    TO_ET_PIPE: 
#if UNIX
      if dup_file_descriptor(Design, fd, UNBUF_WRITE) then
        begin
          p.state := PIPE_OPEN_FOR_WRITE;
	  p.fd := text_file_descr(Design);
	end;
#endif
#if VAX
      p.state := PIPE_OPEN_FOR_WRITE;
#endif
    TEMP_PIPE: 
#if UNIX
      if dup_file_descriptor(CmpTmp, fd, UNBUF_WRITE) then
        begin
          p.state := PIPE_OPEN_FOR_WRITE;
	  p.fd := text_file_descr(CmpTmp);
	end;
#endif
#if VAX
      p.state := PIPE_OPEN_FOR_WRITE;
#endif
    OTHERWISE
      begin
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  pipe_rewrite_descriptor := (p.state = PIPE_OPEN_FOR_WRITE);
  if debug_22 then
    begin
      write(Outfile, 'pipe_rewrite_desc');
      if (p.state = PIPE_OPEN_FOR_WRITE) then write(Outfile, '(T): ')
                                         else write(Outfile, '(F): ');
      dump_pipe(Outfile, p, 0);
    end;
end { pipe_rewrite_descriptor } ;


procedure pipe_close(var p: pipe);
  { close the pipe -- it is not an error to attempt to close an
    unopened pipe. }
begin
  if debug_22 then
    begin
      write(Outfile, 'Closing ');  dump_pipe(Outfile, p, 0);
    end;

#if VAX
  if (p.state = PIPE_OPEN_FOR_WRITE) and not p.written then
    uxwrite(p.fd, nullstring^, 0);
#endif

  if (p.state = PIPE_OPEN_FOR_READ) or (p.state = PIPE_OPEN_FOR_WRITE) then
    case p.which of
      EXPANSION_PIPE:
#if VAX
        if p.is_pipe then uxclose(p.fd)
        else
#endif
          close_file(CmpExp, CMPEXP_FILE_NAME, p.name);
      SCHEMA_PIPE:
#if VAX
        if p.is_pipe then uxclose(p.fd)
        else
#endif
          close_file(CmpSchem, CMPSCHEM_FILE_NAME, p.name);
      TEMP_PIPE:
#if VAX
        if p.is_pipe then uxclose(p.fd)
        else
#endif
          close_file(CmpTmp, CMPTMP_FILE_NAME, p.name);
      FROM_ET_PIPE:
#if VAX
        if p.is_pipe then
          uxclose(p.fd)
        else
          close_file(CmpDraw, CMPDRAW_FILE_NAME, p.name);
#endif
#if UNIX
        if cfclose(CmpDraw) then ;
#endif
      TO_ET_PIPE:
#if VAX
        if p.is_pipe then uxclose(p.fd)
        else
#endif
          close_file(Design, DESIGN_FILE_NAME, p.name);
      OTHERWISE
        begin
          assert(0);
          writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
        end;
    end;
  pipe_init(p);
end { pipe_close } ;


procedure pipe_from_open_file(var p: pipe; which: pipe_files; name: xtring);
  { Construct pipe record for access to the indicated file as a "pipe".
    The file is presumed open.  Name is presumed to remain a viable
    string pointer until after pipe is closed. }
begin
  pipe_init(p);
  p.is_pipe := FALSE;
  p.which := which;
  p.name := name;
  case which of
    SCHEMA_PIPE: p.state := PIPE_OPEN_FOR_WRITE;
    EXPANSION_PIPE: p.state := PIPE_OPEN_FOR_WRITE;
    TO_ET_PIPE: p.state := PIPE_OPEN_FOR_WRITE;
    TEMP_PIPE: p.state := PIPE_OPEN_FOR_WRITE;
    FROM_ET_PIPE: p.state := PIPE_OPEN_FOR_READ;
    OTHERWISE
      begin
        p.state := PIPE_NOT_OPEN;
        assert(0);
        writeln(CmpLog, ' Unknown pipe/file ', ord(p.which):1);
      end;
  end;
  if debug_22 then
    begin
      write(Outfile, 'Pipe_from_file: ');  dump_pipe(Outfile, p, 0);
    end;
end { pipe_from_open_file } ;

  
