(**)     { ------- lexical analyzer ------- }


procedure insymbol;
  { parse the next token from the input string expanding any text macros
    that are found  If copy_input then ignore tokenization - just expand
    text macros. }
  type
    line_read_type = (NORMAL, CONTINUATION);

  var
    ch,                             { last character read in }
    next: char;                     { the next character in the input stream }
    looking_for_string,             { TRUE if reading in a string }
    done: boolean;                  { TRUE when symbol has been parsed }
    upper_case_on_input: boolean;   { TRUE if upper casing of all characters }
    delimited: boolean;             { TRUE if ch is not proceeded by a legal
                                      identifier char, thus it can begin an
				      identifier if it is a letter.  This is
				      only used if copy_input and not
				      tokenize_params -- it should otherwise
				      be considered undefined. }


  procedure copy_to_buffer(ch: char);
    { copy the character to a global buffer }
  begin
    if (copy_pos < MAX_STRING_LENGTH) and (ch >= ' ') and not copy_error then
      begin
        copy_pos := copy_pos + 1;  copy_buffer[copy_pos] := ch;
      end
    else if not copy_error and (ch >= ' ') then
      begin
        error(116 { too big });  copy_error := TRUE;
      end;
  end { copy_to_buffer } ;


  function get_line(which: line_read_type): boolean;
    { read the next line from the input file.  Return TRUE if a string
      was popped from the stack instead of reading in another line.
      If a string was popped, the state of the lexical analyzer was
      restored from stack.  Otherwise, the character returned is space.
      Don't read in a new line if currently reading a string. }
#ifndef SCALD_COMPILER
    var
      pipe_result: read_result;   { condition of line read from pipe }
#endif

#if !UNIX
    procedure read_a_line(var f: inputfile); 
      { read a line from the specified input file }
      var
        i: string_range;      { index into the parse string }
    begin
      while eoln(f) and not eof(f) do readln(f);

      i := 0;
      while not eoln(f) and not eof(f) do
        begin
          read(f, ch);  
          if i < MAX_STRING_LENGTH then
            begin  i := i + 1;  instring^[i] := ch;  end
          else
            begin
              instring^[0] := chr(MAX_STRING_LENGTH);
              error(53 { input line length exceeded });
              while not eof(f) and not eoln(f) do get(f);
            end;
        end;
      if eof(f) then ch := chr(EOL) else ch := ' ';
      instring^[0] := chr(i);  
    end { read_a_line } ;

#else
    procedure read_a_line(var f: inputfile); 
      { read a line from the specified input file }
      const
        END_OF_FILE = 2;
        OVERFLOW_ERR = 1;
        SUCCESS = 0;
      var
        result: 0..2;  { SUCCESS,OVERFLOW,END_OF_FILE returns from C }
    begin
      repeat
        result := creadln(f, instring);
      until (result = END_OF_FILE) or (ord(instring^[0]) > 0);
      case result of
        END_OF_FILE: ch := chr(EOL);
        OVERFLOW_ERR:
          begin
            error(53 { line too long });
            ch := ' ';
          end;
        SUCCESS: ch := ' ';
      end;
    end { read_a_line } ;
#endif UNIX


  begin { get_line }
    get_line := FALSE;

    if parse_stack_pointer > 1 then       { pop the stack }
      if how_to_parse = PARSE_TRANSPARENTLY then
        begin
          virtual_pop_string;
          get_line := TRUE;

          if copy_input and (read_state = FGOT_CHAR) then
            copy_to_buffer(last_char);
        end
      else
        ch := chr(EOL)    { end of string }

    { don't allow strings to cross lines }

    else if looking_for_string and (which = NORMAL) then
      ch := chr(EOL)

    else
      begin
        case current_file of
          DIRECTIVES_FILE:  read_a_line(infile);
          STANDARD_FILE:    read_a_line(CmpStan);
#ifndef SCALD_COMPILER
          CMPDRAW_FILE:
            begin
	      repeat
	        pipe_result := pipe_readln(CmpDraw_pipe, instring);
              until (pipe_result = READ_EOF) or (ord(instring^[0]) <> 0);
	      case pipe_result of
	        READ_EOF: ch := CHR(EOL);
		READ_OVERFLOW: ch := ' ';
		READ_OK: ch := ' ';
	      end;
	    end;
#if UNIX
          CMPSCHEM_FILE:    read_a_line(CmpSchemI); 
#else
          CMPSCHEM_FILE:    read_a_line(CmpSchem);
#endif
#else !SCALD_COMPILER
#if UNIX
          SYNONYM_FILE:     read_a_line(CmpTmpI);
#else
          SYNONYM_FILE:     read_a_line(CmpTmp);
#endif UNIX
#endif !SCALD_COMPILER

          UNKNOWN_FILE:     assert(173 { no file has been opened });
        end;
        line_pos := 0;  read_state := FINPUT;
      end;

    if debug then
      begin 
        dump_string(outfile, instring);
        writeln(outfile);
      end;
  end { get_line } ;


  procedure get_char(var ch: char);
    { read the next char from the input buffer } 
    var
      need_a_char: boolean; {  whether or not we still need a character } 
  begin
    repeat
      if parse_stack_pointer <= stack_top then
        if (read_state = FGOT_CHAR) or (line_pos < ord(instring^[0])) then
          begin
            if read_state = finput then last_sym_pos := line_pos
            else
              if line_pos >= 1 then last_sym_pos := line_pos-1
                               else last_sym_pos := 0;
	    fix_parse_stack;
	  end;
      if read_state = FGOT_CHAR then
        begin  ch := last_char;  need_a_char := FALSE;  end
      else
        if (read_state = FINIT) or (line_pos >= ord(instring^[0])) then 
          need_a_char := get_line(NORMAL)
        else
          begin
            line_pos := line_pos + 1;
            ch := instring^[line_pos];
            if (ch = CONTINUATION_CHAR) and (line_pos = ord(instring^[0])) then
              begin
                need_a_char := get_line(CONTINUATION);
                need_a_char := TRUE;
              end
            else
              begin
                if ch = chr(TAB_char) then ch := ' '
                else if not islegal[ch] then
                  begin
                    error(32);  ch := ' ';
                  end
                else if upper_case_on_input then ch := upshift[ch];
                if copy_input then copy_to_buffer(ch);
                need_a_char := FALSE;
              end;
          end;
    until not need_a_char;
    read_state := FINPUT;
    last_char := ch;
  end { get_char } ;


  procedure nextchar(var next: char);
    { get the next char. Set a flag indicating that the next char was read } 
  begin
    get_char(ch);  next := ch;
    if ch <> chr(EOL) then read_state := FGOT_CHAR;
  end { nextchar } ;
        

(**)     { ------- process text macros ------- }


  procedure process_text_macro(text_macro_name: name_ptr);
    { if the given name is really a text macro, then (1) expand it
      (2) log it as an expandable id (ValidCOMPILER, not SCALD compiler). }
    var 
      temp: char_array;            { expanded text macro }
      i,                           { index of temp for copy }
      pos: string_range;           { last character in temp }
      saved_copy_input: boolean;   { save of copy_input global }
      curr_char: char;             { save for last_char global }
      saved_state: parse_state;    { save of read_state }
      TM_def,                      { definition for the text macro passed in }
      str: xtring;                 { TM string to be returned }
      parameter_found: boolean;    { TRUE if parameter found }
      ovf_error: boolean;          { TRUE if TM expands too big }


    procedure init;
      { initialize pointers into the buffers and global flags }
    begin
      pos := 0;
      saved_copy_input := copy_input;
      copy_input := FALSE;
      curr_char := last_char;
      saved_state := read_state;
      read_state := finput;
      parameter_found := FALSE;
      ovf_error := FALSE;
    end { init } ;


    procedure expand_with_parameters(TM_name: name_ptr; definition: xtring);
      { expand the text macro and its parameters }
      type 
        param_range = 1..MAX_TM_PARAMETERS;
        parameters = record
                       parameter_value: alpha;
		       param_name: name_ptr; { defined only if an id }
		       param_tm_def: xtring;
                       parameter_length: 0..ID_LENGTH;
                     end;
      var
        parameter_list: array [param_range] of parameters;  { params read }
        found_error: boolean;             { TRUE if error found }
        i,                                { index used to copy parameter }
        spos: string_range;               { index into text macro defn }
        def_char: char;                   { last char read from definition }
        last_read: 0..max_TM_parameters;  { max last read param number }
        num: {natural_number}longint;     { parameter number needed }


      procedure nextchar(var next: char);
        { get the next character from the text macro definition } 
      begin
        if spos < ord(definition^[0]) then
          begin  spos := spos + 1;  next := definition^[spos];  end
        else next := chr(EOL);
      end { nextchar } ;


      procedure read_parameter_list(parameter_num: param_range);
        { read parameters from the input string until the specified parameter
          is found.  Text macro parameters are delimited with spaces. }
        var
          i: param_range;                { current parameter # }


        procedure eat_separators;
          { swallow up all the separators }
        begin
          while ch = ' ' do get_char(ch);
        end { eat_separators } ;


        procedure check_for_text_macro(parameter_num: param_range);
          { check the parameter just read and see if it is a text macro.  If
            it is, process it.  ValidCOMPILER logs all identifers found here
	    as expandable ids. }
	  

          var
            i: 0..ID_LENGTH;          { index into parameter }
            ok: boolean;              { TRUE if name still identifier }
        begin
          with parameter_list[parameter_num] do
            begin
              i := 0;  ok := TRUE;
              while (i < parameter_length) and ok do
                begin
                  i := i + 1;
                  if not isidentchar[parameter_value[i]] then
                    ok := FALSE;
                end;

              if ok and isupper[parameter_value[1]] then
                begin
                  param_name := enter_name(parameter_value);
#ifndef SCALD_COMPILER
                  enter_expandable_id(param_name);
#endif
                  if RESERVED in param_name^.kind then
                    param_tm_def := param_name^.definition;

                  if param_tm_def = nullstring then
                    param_tm_def := search_id(param_name);

                  if debug then if param_tm_def <> nullstring then
                    disp_line('found TM         ');
                end;
            end { with } ;
        end { check_for_text_macro } ;


      begin { read_parameter_list }
        for i := last_read+1 to parameter_num do
          with parameter_list[i] do
            begin
              eat_separators;

              parameter_length := 0;  parameter_value := null_alpha;
	      param_name := NIL;  param_tm_def := nullstring;

              while (ch <> ' ') and (ch <> chr(EOL)) do
                if parameter_length >= ID_LENGTH then
                  begin
                    error(43 { too long });
                    while (ch <> ' ') and (ch <> chr(EOL)) do  get_char(ch);
                  end
                else
                  begin
                    parameter_length := parameter_length + 1;
                    parameter_value[parameter_length] := ch;
                    get_char(ch);
                  end;

              if parameter_length > 0 then check_for_text_macro(i)
              else
                begin
                  error(56 { text macro parameter not found });
                  error_dump_text_macro(TM_name);
                  error_dump_indent(indent);
                  error_dump_alpha('Parameter number');
                  error_dump_char('=');
                  error_dump_integer(i);
                  error_dump_CRLF;
                end;
            end;

        last_read := parameter_num;
        parameter_found := TRUE;
      end { read_parameter_list } ;


      procedure display_error;
        { display length error, the macro name, and the definition }
      begin
        error(117 { text macro + parameters is too long });
        error_dump_text_macro(TM_name);
      end { display_error } ;


    begin { expand_with_parameters }
      if TM_depth >= MAX_TM_RECURSION then
        begin
          error(64 { TM recursion depth exceeded });
          error_dump_text_macro(text_macro_name);

          found_error := TRUE;
        end
      else
        found_error := FALSE;

      if not found_error then
        begin
          TM_depth := TM_depth + 1;  spos := 0;  last_read := 0;

          if definition = nullstring then
            begin
              error(110 { undefined text macro });
              error_dump_text_macro(TM_name);
            end;

          while (spos < ord(definition^[0])) and not found_error do
            begin
              nextchar(def_char);
              if def_char = TM_parameter_prefix_char then
                begin
                  nextchar(def_char);
                  num := ord(def_char) - ord('0');

                  if (num < 1) or (num > MAX_TM_PARAMETERS) then
                    error(35 { parameter value out of range })
                  else
                    begin
                      if num > last_read then read_parameter_list(num);

                      if not found_error then with parameter_list[num] do
			if param_tm_def <> nullstring then
			  begin
			    expand_with_parameters(param_name, param_tm_def);
                            if ovf_error then found_error := TRUE;
			  end
			else
			  begin
			    i := 0;
			    while (i < parameter_length) and
			          not found_error do
			      begin
				i := i + 1;
				if pos+i > MAX_STRING_LENGTH then
				  begin
				    display_error;
				    found_error := TRUE;
				  end
				else
				  temp[pos+i] := parameter_value[i];
			      end;

			    pos := pos + parameter_length;

			  end;
                    end;
                end
              else
                if def_char <> chr(EOL) then
                  if pos >= MAX_STRING_LENGTH then
                    begin  display_error;  found_error := TRUE;  end
                  else 
                    begin  pos := pos + 1;  temp[pos] := def_char;  end;
            end { while } ;

          if not found_error then
            if TM_depth <= 1 then assert(160 { underflow!!! })
                             else TM_depth := TM_depth - 1;

        end { if not found_error } ;
    end { expand_with_parameters } ;


  begin { process_text_macro }
#ifndef SCALD_COMPILER
    enter_expandable_id(text_macro_name);
#endif
    if TM_depth >= MAX_TM_RECURSION then
      begin
        error(64 { TM recursion depth exceeded });
        error_dump_text_macro(text_macro_name);
      end
    else
      begin
        TM_depth := TM_depth + 1;

        TM_def := nullstring;

        if RESERVED in text_macro_name^.kind then
          TM_def := text_macro_name^.definition;

        if TM_def = nullstring then
          TM_def := search_id(text_macro_name);

        if TM_def <> nullstring then
          begin
            if copy_input then current_pos := current_pos-1;
            if debug then
              if RESERVED in text_macro_name^.kind then
                disp_line('found reserved TM')
              else
                disp_line('found TM         ');

            init;

            expand_with_parameters(text_macro_name, TM_def);

            if ovf_error then str := nullstring
            else
              if parameter_found and not ovf_error then
                begin
                  create_a_string(str, pos);
                  for i := 1 to pos do str^[i] := temp[i];
                  str := enter_and_release_string(str);
                end
              else str := TM_def;

            copy_input := saved_copy_input;
            last_char := curr_char;
            if not parameter_found then read_state := saved_state;
            parse_string(str, PARSE_TRANSPARENTLY);
          end;

        if TM_depth <= 1 then assert(160 { underflow!!! })
        else TM_depth := TM_depth - 1;
      end;
  end { process_text_macro } ;


(**)     { ------- scan for an identifier ------- }


  procedure get_identifier;
    { read in an identifier }
    var
      i: 0..ID_LENGTH;            { index into the identifier }
      temp: alpha;                { identifier being parsed }
      id_error: boolean;          { TRUE iff id too long }
  begin
    temp := NULL_ALPHA;  id.name := NIL;

    i := 0;  sy := IDENT;  id_error := FALSE;
    repeat
      if i >= ID_LENGTH then
        begin
          if not copy_input or tokenize_params then
            error(41 { identifier length exceeded });
          id_error := TRUE;
          while isidentchar[ch] do get_char(ch);
        end
      else
        begin  
          i := i + 1;  temp[i] := ch;  get_char(ch);
        end;
    until not isidentchar[ch];

    if ch <> chr(EOL) then read_state := FGOT_CHAR;
    
    if debug then disp_line('identifier       ');

    if not copy_input or tokenize_params or not id_error then
      begin
        id.name := enter_name(temp);

        if (not copy_input) and (KEY_WORD in id.name^.kind) then
          if id.name^.sy in allowed_key_words then sy := id.name^.sy;

        if (sy = IDENT) and allow_TM_expansion then
	  process_text_macro(id.name);
      end;
  end { get_identifier } ;

    
(**)     { ------- scan for constant ------- }


  procedure get_constant;
    { read in one of three different constant types }
    var
      new_radix: natural_number;     { radix specified in constant }


    procedure skip_to_end_of_constant(number_radix: radix_range);
      { skip to the end of the constant;  error recovery }
    begin
      while ch in valid_chars[number_radix] do get_char(ch);
    end { skip_to_end_of_constant } ;


    function build_number(radix: radix_range): natural_number;
      { build a number with the specified radix }
      var
        temp: natural_number;      { value of the function to be returned }
        next_digit: 0..MAX_RADIX;  { numeric value of current digit }
    begin
      temp := 0;  const_width := 0;
      repeat
        const_width := const_width + 1;
        if ch <= '9' then  next_digit := ord(ch) - ord('0')
                     else  next_digit := ord(ch) - ord('A') + 10;

        if (temp > MAXINT DIV radix) or 
           ((temp = MAXINT DIV radix) and
            (next_digit > MAXINT MOD radix)) then
          begin  
            error(24 { ovf });
            skip_to_end_of_constant(radix);
          end
        else
          begin  temp := radix * temp + next_digit;  get_char(ch);  end;
      until not (ch IN valid_chars[radix]);

      const_width := const_width * radix_width[radix];

      build_number := temp;
    end { build_number } ;


  begin { get_constant }
    sy := CONSTANT;
    const_val := build_number(10);

    if parse_SCALDconstants then
      if isupper[ch] then
        begin
          repeat
            if copy_input and tokenize_params then
              copy_pos := copy_pos - 1; { don't copy SCALDconstant characters }

            get_char(ch);
          until not isupper[ch];
        end

      else
        begin
          if ch = '#' then
            begin
              new_radix := const_val;
              if (new_radix < min_radix) or (new_radix > max_radix) then
                begin  error(61 { out of range });  new_radix := 10;  end;

              get_char(ch);
              const_val := build_number(new_radix);

              sy := SIGNALCONST;
            end;

          if ch = '(' then    { width specification }
            begin
              get_char(ch);
              const_width := build_number(10);

              if (const_width <= 0) or (const_width > max_bit_value) then
                begin  error(44 { invalid width });  const_width := 1;  end;

              if ch = ')' then get_char(ch) else error(7 { expected ) });

              sy := SIGNALCONST;
            end;
        end;

    read_state := FGOT_CHAR;

    if debug then disp_line('constant         ');
  end { get_constant } ;


(**)     { ------- scan for string ------- }


  procedure get_string(stopper: char);
    { read a string }
    var
      len: string_range;   { length of the string read in }
      done: boolean;       { TRUE when end of the string has been found }
      nch: char;           { next charactar }
  begin
    len := 0;  done := FALSE;  looking_for_string := TRUE;
#if UNIX
    if not upper_case_strings then upper_case_on_input := FALSE; 
#endif
    repeat
      get_char(ch);
      if ch = stopper then
        begin
	  nextchar(nch);
          if nch = stopper then get_char(ch) else done := TRUE;
	end;

      if (ch = chr(EOL)) and not done then
        begin  error(89 { string not closed });  done := TRUE;  end;

      if not done then
        if len >= MAX_STRING_LENGTH then
          begin error(22 { string length exceeded });
            while (ch <> stopper) and (ch <> chr(EOL)) do get_char(ch);
          end
        else
          begin len := len + 1;  input_buffer^[len] := ch;  end;
    until done;

    input_buffer^[0] := chr(len);
    sy := STRINGS;
    looking_for_string := FALSE;
    upper_case_on_input := TRUE;

    lex_string := enter_string(input_buffer);

    if debug then disp_line('string           ');
  end { get_string } ;


(**)     { ------- main lexical analyzer ------- }


begin { insymbol }
  looking_for_string := FALSE;
  upper_case_on_input := TRUE;
  copy_pos := current_pos;


  if parse_stack_pointer >= stack_top then
    begin
      if read_state = finput then last_sym_pos := line_pos
      else
        if line_pos >= 1 then last_sym_pos := line_pos-1
                         else last_sym_pos := 0;
    end
  else with stack[stack_top] do
    begin
      if state = FINPUT then last_pos := pos
      else
        if pos >= 1 then last_pos := pos-1
                         else last_pos := 0;
    end;

  if copy_input and not tokenize_params then
    begin
      delimited := TRUE;
      repeat
        get_char(ch);
	current_pos := copy_pos;
        if isupper[ch] then
	  begin
            { only delimited (not preceeded by a legal identifier char)
	      letters can begin an identifier in this mode }
	    if delimited then get_identifier;
	    delimited := FALSE;
	  end
        else delimited := not isidentchar[ch];
      until ch = chr(EOL);
      sy := ENDOFDATASY;
    end
  else
    repeat
      done := TRUE;
      get_char(ch);
      while ch = ' ' do get_char(ch);  current_pos := copy_pos;
  
      if ch = chr(EOL) then sy := ENDOFDATASY
      else
	case ch of
	  '!':  sy := EXCLAMATION;
	  '"':  get_string(ch);
	  '#':  sy := SHARP;
	  '$':  sy := DOLLAR;
	  '%':  sy := PERCENT;
	  '&':  sy := AMPERSAND;
	 '''':  get_string(ch);
	  '(':  sy := LPAREN;
	  ')':  sy := RPAREN;
	  '*':  sy := ASTERISK;
	  '+':  sy := PLUS;
	  ',':  sy := COMMA;
	  '-':  sy := MINUS;
	  '.':  begin
		  nextchar(next);
		  if next = '.' then
		    begin
		      sy := DOTDOTSY;  read_state := FINPUT;
		    end
		  else sy := PERIOD;
		end;
	  '/':  sy := SLASH;
	  '0','1','2','3','4','5','6','7','8','9':  get_constant;
	  ':':  begin
		  nextchar(next);
		  if next = ':' then
		    begin
		      sy := COLONCOLONSY;  read_state := FINPUT;
		    end
		  else sy := COLON;
		end;
	  ';':  sy := SEMI;
	  '<':  begin
		  nextchar(next);
		  if next = '=' then
		    begin
		      sy := LESY;  read_state := FINPUT;
		    end
		  else if next = '>' then
		    begin
		      sy := NESY;  read_state := FINPUT;
		    end
		  else sy := LESSTHAN;
		end;
	  '=':  sy := EQUAL;
	  '>':  begin
		  nextchar(next);
		  if next = '=' then
		    begin
		      sy := GESY;  read_state := FINPUT;
		    end
		  else sy := GREATERTHAN;
		end;
	  '?':  sy := QUESTION;
  {       '@':  sy := ATSY;      this symbol is not used: @=^ in EBCDIC! }
	  'A','B','C','D','E','F','G','H','I','J','K','L','M','N',
	  'O','P','Q','R','S','T','U','V','W','X','Y','Z':  get_identifier;
	  '[':  sy := LBRACKET;
#if SVS
	  '\\': sy := BACKSLASH;
#else
	  '\': sy := BACKSLASH;
#endif 
	  ']':  sy := RBRACKET;
	  '^':  sy := CIRCUMFLEX;
	  '_':  sy := UNDERBAR;
	  '`':  sy := ACCENTGRAVE;
	  '{':  begin
		  repeat
		    get_char(ch)
		  until (ch='}')  or (ch=chr(EOL));
  
		  if ch = chr(EOL) then error(34 { comment not closed });
		  done := FALSE;
		end;
	  '|':  sy := VERTICALBAR;
	  '}':  begin  error(20 { unmatched symbol });  done := FALSE;  end;
	  '~':  sy := TILDA;
	  OTHERWISE
	    begin
		error(23 { illegal character in input });
		error_dump_alpha('Character       ');
		error_dump_integer(ord(ch));
		error_dump_CRLF;
	    end;
	end;
    until done;

  if parse_stack_pointer <= stack_top then
    stack[stack_top].last_pos := stack[stack_top].pos;

  if debug then writeln(outfile, 'insymbol: ', ord(sy)); 
end { insymbol } ;
