procedure check_for_synonyming(sig, comp: signal_definition_ptr);
  { check the bits of the signal against its complement and report
    bits that are synonymed together.  Assume that the basescripts
    have been fixed so that chains are all length 1 or 0. Assume that
    comp is the signal -sig (complement of sig) }
  var
    bad_bits: subscript_ptr;      { bits synonymed together }
    last_bad: subscript_ptr;      { last element of bad_bits }
    sig_BS: basescript_ptr;       { current basescript element of sig }
    comp_BS: basescript_ptr;      { current basescript element of comp (-sig) }
    sig_bit: bit_range;           { current bit of sig }
    comp_bit: bit_range;          { current bit of comp (-sig) }
    sig_base: bit_range;          { base bit of sig_BS^.instance for sig_bit }
    comp_base: bit_range;         { ditto, but for comp (-sig) }


  procedure report_synonymed_bits;
    { report the bad bits -- bad_bits can be assumed to be non-NIL }
  begin
    error(235 { signal synonymed to own complement });
    if sig^.scope <> GLOBAL then
      error_dump_mtree_node(sig^.node);
    error_dump_signal_def(sig);
    error_dump_signal_def(comp);
    if sig^.kind = VECTOR then
      begin
        error_dump_indent(indent);
        error_dump_alpha('Synonymed bits: ');  error_dump_char(' ');
	if ok_to_print_error then
	  if PrintCmpLst then dump_bit_subscript(CmpLst, bad_bits, sig^.kind)
	  else dump_bit_subscript(Monitor, bad_bits, sig^.kind);
	dump_bit_subscript(CmpLog, bad_bits, sig^.kind);
	if debugging then dump_bit_subscript(Outfile, bad_bits, sig^.kind);
	error_dump_CRLF;
      end;
  end { report_synonymed_bits } ;
  
  
  procedure collect_bit(bit: bit_range);
    { add the bit to bad_bits }
  begin
    if bad_bits = NIL then
      begin
        new_subscript(bad_bits);  last_bad := bad_bits;
	last_bad^.left_index := bit;
      end
    else if (left_to_right and (bit <> last_bad^.right_index + 1)) or
            (not left_to_right and (bit <> last_bad^.right_index - 1)) then
      begin
        new_subscript(last_bad^.next);  last_bad := last_bad^.next;
        last_bad^.left_index := bit;
      end;
    last_bad^.right_index := bit;
  end { collect_bit } ;


begin { check_for_synonyming }
  if debug_29 then
    begin
      writeln(Outfile, '-- enter check_for_syonyming --');
      dump_signal_definition_with_basescript(Outfile, sig);
      dump_signal_definition_with_basescript(Outfile, comp);
    end;
  
  if (sig^.synonym_bits <> NIL) or (comp^.synonym_bits <> NIL) then
    begin
      bad_bits := NIL;
      sig_BS := sig^.synonym_bits;  comp_BS := comp^.synonym_bits;
      if sig_BS <> NIL then sig_bit := sig_BS^.left_index
                       else sig_bit := -1 { arbitrary } ;
      if comp_BS <> NIL then comp_bit := comp_BS^.left_index
                        else comp_bit := -1 { arbitrary } ;
      repeat
        if left_to_right then
	  begin
	    if (comp_BS = NIL) or (sig_bit < comp_bit) then 
	      begin 
	        { is comp<sig_bit> the base of sig<sig_bit> ? }

	        if sig_BS^.instance^.defined_by = comp then
		  begin
		    sig_base := 
		      sig_bit - sig_BS^.left_index + sig_BS^.offset + 1;
		    if nth_bit_of_signal_instance(sig_base, 
		                                  sig_BS^.instance) then ;
                    if sig_base = sig_bit then collect_bit(sig_bit);
		  end;

		{ move to next bit of sig }

		if sig_bit < sig_BS^.right_index then sig_bit := sig_bit + 1
		else
		  begin
		    sig_BS := sig_BS^.next;
		    if sig_BS <> NIL then sig_bit := sig_BS^.left_index;
		  end;
	      end
	    else if (sig_BS = NIL) or (sig_bit > comp_bit) then
	      begin
	        { is sig<comp_bit> the base of comp<comp_bit> ? }

	        if comp_BS^.instance^.defined_by = sig then
		  begin
		    comp_base := 
		      comp_bit - comp_BS^.left_index + comp_BS^.offset + 1;
		    if nth_bit_of_signal_instance(comp_base, 
		                                  comp_BS^.instance) then ;
                    if comp_base = comp_bit then collect_bit(comp_bit);
		  end;

		{ move to next bit of comp (or -sig) }

		if comp_bit < comp_BS^.right_index then
		  comp_bit := comp_bit + 1
		else
		  begin
		    comp_BS := comp_BS^.next;
		    if comp_BS <> NIL then comp_bit := comp_BS^.left_index;
		  end;
	      end
	    else { (both basescripts non-NIL) and (sig_bit = comp_bit) }
	      begin
	        { are the bases equivalent ? }

	        if sig_BS^.instance^.defined_by =
		   comp_BS^.instance^.defined_by then
		  begin
		    sig_base := 
		      sig_bit - sig_BS^.left_index + sig_BS^.offset + 1;
		    if nth_bit_of_signal_instance(sig_base, 
		                                  sig_BS^.instance) then ;
		    comp_base := 
		      comp_bit - comp_BS^.left_index + comp_BS^.offset + 1;
		    if nth_bit_of_signal_instance(comp_base, 
		                                  comp_BS^.instance) then ;
		    if sig_base = comp_base then collect_bit(sig_bit);
		  end;

		{ move to next bit of both signals }

		if sig_bit < sig_BS^.right_index then sig_bit := sig_bit + 1
		else
		  begin
		    sig_BS := sig_BS^.next;
		    if sig_BS <> NIL then sig_bit := sig_BS^.left_index;
		  end;
		if comp_bit < comp_BS^.right_index then
		  comp_bit := comp_bit + 1
		else
		  begin
		    comp_BS := comp_BS^.next;
		    if comp_BS <> NIL then comp_bit := comp_BS^.left_index;
		  end;
	      end
	  end
	else { right to left bit order }
	  begin
	    if (comp_BS = NIL) or (sig_bit > comp_bit) then
	      begin
	        { is comp<sig_bit> the base of sig<sig_bit> ? }

	        if sig_BS^.instance^.defined_by = comp then
		  begin
		    sig_base := 
		      sig_bit - sig_BS^.left_index + sig_BS^.offset + 1;
		    if nth_bit_of_signal_instance(sig_base, 
		                                  sig_BS^.instance) then ;
                    if sig_base = sig_bit then collect_bit(sig_bit);
		  end;

		{ move to next bit of sig }

		if sig_bit > sig_BS^.right_index then sig_bit := sig_bit - 1
		else
		  begin
		    sig_BS := sig_BS^.next;
		    if sig_BS <> NIL then sig_bit := sig_BS^.left_index;
		  end;
	      end
	    else if (sig_BS = NIL) or (sig_bit < comp_bit) then
	      begin
	        { is sig<comp_bit> the base of comp<comp_bit> ? }

	        if comp_BS^.instance^.defined_by = sig then
		  begin
		    comp_base := 
		      comp_bit - comp_BS^.left_index + comp_BS^.offset + 1;
		    if nth_bit_of_signal_instance(comp_base, 
		                                  comp_BS^.instance) then ;
                    if comp_base = comp_bit then collect_bit(comp_bit);
		  end;

		{ move to next bit of comp (-sig) }

		if comp_bit > comp_BS^.right_index then
		  comp_bit := comp_bit - 1
		else
		  begin
		    comp_BS := comp_BS^.next;
		    if comp_BS <> NIL then comp_bit := comp_BS^.left_index;
		  end;
	      end
	    else { (both basescripts non-NIL) and (sig_bit = comp_bit) }
	      begin
	        { are the bases equivalent ? }

	        if sig_BS^.instance^.defined_by =
		   comp_BS^.instance^.defined_by then
		  begin
		    sig_base := 
		      sig_BS^.left_index - sig_bit + sig_BS^.offset + 1;
		    if nth_bit_of_signal_instance(sig_base, 
		                                  sig_BS^.instance) then ;
		    comp_base := 
		      comp_BS^.left_index - comp_bit + comp_BS^.offset + 1;
		    if nth_bit_of_signal_instance(comp_base, 
		                                  comp_BS^.instance) then ;
		    if sig_base = comp_base then collect_bit(sig_bit);
		  end;

		{ move to next bit of both signals }

		if sig_bit > sig_BS^.right_index then sig_bit := sig_bit - 1
		else
		  begin
		    sig_BS := sig_BS^.next;
		    if sig_BS <> NIL then sig_bit := sig_BS^.left_index;
		  end;
		if comp_bit > comp_BS^.right_index then
		  comp_bit := comp_bit - 1
		else
		  begin
		    comp_BS := comp_BS^.next;
		    if comp_BS <> NIL then comp_bit := comp_BS^.left_index;
		  end;
	      end
	  end;
      until (sig_BS = NIL) and (comp_BS = NIL);
      if bad_bits <> NIL then
        begin
	  report_synonymed_bits;  release_entire_subscript(bad_bits);
	end;
    end;
  if debug_29 then writeln(Outfile, 'exit check_for_synonyming');
end { check_for_synonyming } ;
  

procedure fix_all_basescripts(node: mtree_node_ptr);
  { Fix all basescripts so that ALL "synonym chain" lengths are 1.
    That is, so that for each signal the base descriptor describes the
    absolute base signal for the def.  Assume that property copying
    has already been done, and so is not necessary.  Report illegal
    synonyming of signals to their complements.  Keep track of before
    and after signal statistics. }
  var
    curr_BD: base_descriptor_ptr;         { describes a bit of basescript }
    base_BD: base_descriptor_ptr;         { base of current bit }
    complement_table: avl_ptr;            { table of complemented signals }
    current: mtree_node_ptr;              { current node }
    virtual_base: signal_definition_ptr;  { current virtual base in node }
    sig: signal_definition_ptr;           { signal being fixed }
    comp: signal_definition_ptr;          { complement of virtual_base }


  procedure fix_basescript(sig_def: signal_definition_ptr);
    { fix basescript so that its max chain length is 1.
      (assume that basescript is non-NIL) 
      LOOK REAL HARD at this and see what, if anything, must be modified
      to handle scalars as well. }
    label
      90; { return }
    var
      direction: -1..1;                   { left to right increment }
      current_BS: basescript_ptr;         { current element of basescript }
      i: bit_range;                       { a bit of current BS }
      result: basescript_ptr;             { fixed basescript }
      last: basescript_ptr;               { last element of result list }
      width_of_last: bit_range;           { width of last element of result }
      done: boolean;                      { TRUE when done with current_BS }
    
  begin
    if debug_27 then
      begin
        writeln(Outfile, '-- enter fix_basescript --');
	dump_signal_definition_with_basescript(Outfile, sig_def);
      end;

    if left_to_right then direction := 1
                          else direction := -1;
    result := NIL;  last := NIL;  current_BS := sig_def^.synonym_bits;
    while current_BS <> NIL do
      begin

        { curr_BD describes each bit of current_BS }

	curr_BD^.instance := current_BS^.instance;
	curr_BD^.offset := current_BS^.offset;
	curr_BD^.width := 1;  { do 1 bit at a time because I don't have time
	                        to fuck with anything more complicated }
        i := current_BS^.left_index;  done := FALSE;
	
        repeat  { for each bit of current_BD }
          if not find_base_of_base_descriptor(curr_BD, base_BD) then
	    begin
	      assert(100 { 1 bit lookup should never fail });
	      release_entire_basescript_list(result);
	      goto 90 { return } ;
	    end;
          if result = NIL then
            begin
              new_basescript(result);
              with result^ do
                begin
                  instance := base_BD^.instance;
                  offset := base_BD^.offset;
                  left_index := i;  right_index := i;
                end;
	      last := result;  width_of_last := 1;
            end
          else if (base_BD^.instance = last^.instance) and 
             (base_BD^.offset = last^.offset + width_of_last) and
	     (i = current_BS^.right_index + direction) then
            begin
              last^.right_index := i;  width_of_last := width_of_last + 1;
            end
          else
            begin
              new_basescript(last^.next);  last := last^.next;
              with last^ do
                begin
                  instance := base_BD^.instance;
                  offset := base_BD^.offset;
                  left_index := i;  right_index := i;
                end;
	      width_of_last := 1;
            end; 
      
          if left_to_right then
            if (i >= current_BS^.right_index) then done := TRUE
	    else
	      begin
	        i := i + 1;
		curr_BD^.offset := curr_BD^.offset + 1;
	      end
          else
            if (i <= current_BS^.right_index) then done := TRUE
	    else
	      begin
	        i := i - 1;
		curr_BD^.offset := curr_BD^.offset + 1;
	      end;
        until done;
        current_BS := current_BS^.next;
      end;

    release_entire_basescript_list(sig_def^.synonym_bits);
    sig_def^.synonym_bits := result;

    if debug_27 then
      begin
        write(Outfile, 'result: ');
	dump_basescript_list(Outfile, sig_def^.synonym_bits);
	writeln(Outfile, '-- exit fix_basescript');
      end;
  90:
  end { fix_basescript } ;


  function find_complement(sig: signal_definition_ptr): signal_definition_ptr;
    { find a signal with the same name in the complement table }
    var
      avlsig: avl_object_ptr;     { for passing sig to avl_find }
      avlfound: avl_ptr;          { found entry }
  begin
  { avlsig.tag := AVL_COMPLEMENTED_ACTUAL;                          }(*AVL*)
    avlsig.complemented_actual := sig;
    avlfound := avl_find(avlsig, complement_table, AVL_COMPLEMENTED_ACTUAL);
    if avlfound = NIL then find_complement := NIL
    else find_complement := avlfound^.object.complemented_actual;
  end { find_complement } ;


  procedure enter_complemented_signal(sig: signal_definition_ptr);
    { enter the signal into the complement table }
    var
      avlsig: avl_object_ptr;     { for passing sig to avl_insert }
      avlentry: avl_ptr;          { new entry }
  begin
  { avlsig.tag := AVL_COMPLEMENTED_ACTUAL;                         }(*AVL*)
    avlsig.complemented_actual := sig;
    avlentry := avl_insert(avlsig, complement_table, AVL_COMPLEMENTED_ACTUAL);
  end { enter_complemented_signal } ;


  procedure check_formals_of_leaf(node: mtree_node_ptr);
    { check the formals of a leaf node for improper synonyming to
      complements }
    var
      current_formal: formal_actual_ptr;  { current formal of node }
      complement_entered: boolean;        { TRUE if entry has been made into
                                            the interface_complement_table }
      comp: complemented_formal_ptr;      { a non NAC complemented formal }
  begin
    if debug_29 then
      begin
        writeln(Outfile, 'enter check_formals_of_leaf');
      end;
    current_formal := node^.params;  complement_entered := FALSE;
    while current_formal <> NIL do with current_formal^ do
      begin
        if not uses_NAC and (polarity = COMPLEMENTED) then
	  begin
	    complement_entered := TRUE;
            with pin_name^ do
              if kind = VECTOR then
                enter_interface_complement(
                  signal_name, bit_subscript^.left_index,
                  bit_subscript^.right_index,
                  copy_PCS(current_formal^.actual_parameter^.signal))
              else enter_interface_complement(
                signal_name, -1, -1,
                copy_PCS(current_formal^.actual_parameter^.signal));
	  end;
        current_formal := next;
      end;

    if debug_29 then 
      dump_complemented_formal_hash_table(Outfile, interface_complement_table);

    if complement_entered then
      begin
        current_formal := node^.params;
        while current_formal <> NIL do with current_formal^ do
	  begin
	    if not uses_NAC and (polarity = NORMAL) then
	      begin
	        comp := find_interface_complement(pin_name^.signal_name);
		if comp <> NIL then with pin_name^ do
		  if kind = VECTOR then
	            check_formal_for_synonymed_complement(
		      node, signal_name,
		      bit_subscript^.left_index, bit_subscript^.right_index,
		      current_formal^.actual_parameter^.signal, comp)
		  else 
		    check_formal_for_synonymed_complement(
		      node, signal_name, -1, -1,
		      current_formal^.actual_parameter^.signal, comp);
	      end;
	    current_formal := current_formal^.next;
      	  end;
	release_complete_complemented_formal_table(interface_complement_table);
      end;
    if debug_29 then writeln(Outfile, 'exit check_formals_of_leaf');
  end { check_formals_of_leaf } ;


begin { fix_all_basescripts }
  if debug_20 or debug_27 or debug_29 then
    writeln(Outfile, '-- enter fix_all_basescripts --');

  { initialize work space -- these vars are actually used by fix_basescript,
    but are newed and released here because PASCAL has neither static
    variables nor an address-of operator }

  new_base_descriptor(base_BD);  new_base_descriptor(curr_BD);
  
  { init table for this node }
  
  complement_table := NIL;

  { Since low level signals are often synonymed to more base higher level
    signals, it is quickest to do a breadth-first traversal. }

  current := node;
  while current <> NIL do
    begin
      if current^.is_leaf_node then check_formals_of_leaf(current);
      virtual_base := current^.signals;
      while virtual_base <> NIL do
        begin
	  sig := virtual_base;
          while sig <> NIL do 
            begin
              calculate_synonym_statistics(
                sig,
                unfixed_number_of_sig_defs_with_synonyms,
                unfixed_number_of_basescripts,
                unfixed_number_of_single_bit_basescripts,
                unfixed_number_of_basescripts_with_non_zero_offset,
                unfixed_total_width_of_basescripts,
                unfixed_number_of_base_signal_instances);

              if sig^.synonym_bits <> NIL then fix_basescript(sig);

              calculate_synonym_statistics(
                sig,
                number_of_sig_defs_with_synonyms,
                number_of_basescripts,
                number_of_single_bit_basescripts,
                number_of_basescripts_with_non_zero_offset,
                total_width_of_basescripts,
                number_of_base_signal_instances);

              sig := sig^.next_virtual_def;
            end;

	  if virtual_base^.polarity = COMPLEMENTED then
	    enter_complemented_signal(virtual_base);

	  virtual_base := virtual_base^.next;
	end;

      { check for synonyming of complements -- check for empty
        complement table, as if it is empty, we can skip it }

      virtual_base := current^.signals;
      if complement_table <> NIL then while virtual_base <> NIL do
        begin
	  if virtual_base^.polarity = NORMAL then
	    begin
	      comp := find_complement(virtual_base);
	      if comp <> NIL then check_for_synonyming(virtual_base, comp);
	    end;
	  virtual_base := virtual_base^.next;
	end;

      current := current^.next;
    end;

  { release table and work space before recursing }

  if debug_29 then 
    begin
      writeln(Outfile, '-- Complement table --');
      dump_avl_tree(Outfile, complement_table, AVL_COMPLEMENTED_ACTUAL);
    end;
  release_entire_avl_tree(complement_table);
  release_base_descriptor(base_BD);  release_base_descriptor(curr_BD);
  
  { begin recursion }

  current := node;
  while current <> NIL do
    begin
      if current^.son <> NIL then fix_all_basescripts(current^.son);
      current := current^.next;
    end;

  if debug_20 or debug_27 or debug_29 then
    writeln(Outfile, '-- end fix_all_basescripts --');
end { fix_all_basescripts } ;



