(**)     { ------- FORWARD procedure declarations ------- }


function rewrite_file(var txtfil: textfile; filename: xtring; 
                      logical: alpha): boolean;  forward;
function close_parse_file(which: parse_file_type): boolean;  forward;
procedure close_file(var f: textfile; file_name: alpha);  forward;
function reset_file(filename: xtring; which: parse_file_type): boolean;  forward;
procedure error(error_num: error_range);                              FORWARD;
function enter_name(name: alpha): name_ptr;                           FORWARD;
function make_and_enter_string(name: alpha): xtring;                  FORWARD;
function enter_and_release_string(str: xtring): xtring;               FORWARD;
function enter_string(str: xtring): xtring;                           FORWARD;
procedure assert(assertion_num: assert_range);                        FORWARD;
function alpha_length(name: alpha): id_range;                         FORWARD;
procedure dump_string(var f: textfile; str: xtring);                  FORWARD;
procedure print_string(var f: textfile; str: xtring);                 FORWARD;
procedure error_dump_string(str: xtring);                             FORWARD;
procedure error_dump_CRLF;                                            FORWARD;
procedure insymbol;                                                   FORWARD;

(**)     { ------- trivia ------- }


procedure halt_with_status(halt_code: longint);
  { halt the program generating some termination status }

#if VAX
procedure SYS$EXIT(BYVALUE halt_code: integer);   external;
#endif

begin
#if VAX
  SYS$EXIT(halt_code);
#endif
#if IBM
  halt;
#endif
#if SVS
  halt(halt_code);
#endif
#if SUN || PMAX
  exit(halt_code);
#endif
end { halt_with_status } ;


function min(a, b: longint): longint;
  { return the min }
begin
  if a < b then   min := a  else  min := b;
end { min } ;


function max(a, b: longint): longint;
  { return the max }
begin
  if a > b then   max := a  else  max := b;
end { max } ;


(**)     { ------- string package routines ------- }


{
               ***********************************
               *                                 *
               *       String description        *
               *                                 *
               ***********************************


   A string is represented as a pointer to a packed array of char:

       string = ^packed array [0..255] of char;

   Each string, however, is usually less than 256 characters.  The
   actual length of the string is found in the first byte:  string^[0].
   The length of the string is static;  it should not be changed once
   the string has been created.

   Strings can be up to 255 characters long.  The programmer must make
   sure that characters are not written beyond the end of the string.

   Strings are created on the heap in quantized lengths.  There are 33
   different length arrays created.  The create_a_string routine
   creates an array on the heap big enough to support the given string. }



procedure new_free_element(var f: freeptr);
  { create a new free element for released strings }
begin
  new(f);
  f^.next := NIL;
  f^.str := NIL;
end { new_free_element } ;


procedure create_a_string(var str: xtring; length: string_range);
  { Create a string on the heap of the given length.  This routine uses a
    variant record to represent strings of various lengths with one
    pointer.  First, the free lists are checked for a string of the
    appropriate length.  If none are available, a string is newed from
    the heap.  This scheme works only if the Pascal compiler creates only
    as much space as needed for a variant when the tag field is specified
    in the new.  }
  type
    size_type = (s4,s8,s12,s16,s20,s24,s28,sz32,s36,s40,s44,
                 s48,s52,s56,s60,s64,s68,s72,s76,s80,s84,s88,
                 s92,s96,s100,s120,s140,s160,s180,s200,s220,s240,s256);

    trick_ptr = ^trick_record;
    trick_record = record case size_type of
                     s4: (f4: packed array [0..3] of char);
                     s8: (f8: packed array [0..7] of char);
                     s12: (f12: packed array [0..11] of char);
                     s16: (f16: packed array [0..15] of char);
                     s20: (f20: packed array [0..19] of char);
                     s24: (f24: packed array [0..23] of char);
                     s28: (f28: packed array [0..27] of char);
                     sz32: (f32: packed array [0..31] of char);
                     s36: (f36: packed array [0..35] of char);
                     s40: (f40: packed array [0..39] of char);
                     s44: (f44: packed array [0..43] of char);
                     s48: (f48: packed array [0..47] of char);
                     s52: (f52: packed array [0..51] of char);
                     s56: (f56: packed array [0..55] of char);
                     s60: (f60: packed array [0..59] of char);
                     s64: (f64: packed array [0..63] of char);
                     s68: (f68: packed array [0..67] of char);
                     s72: (f72: packed array [0..71] of char);
                     s76: (f76: packed array [0..75] of char);
                     s80: (f80: packed array [0..79] of char);
                     s84: (f84: packed array [0..83] of char);
                     s88: (f88: packed array [0..87] of char);
                     s92: (f92: packed array [0..91] of char);
                     s96: (f96: packed array [0..95] of char);
                     s100: (f100: packed array [0..99] of char);
                     s120: (f120: packed array [0..119] of char);
                     s140: (f140: packed array [0..139] of char);
                     s160: (f160: packed array [0..159] of char);
                     s180: (f180: packed array [0..179] of char);
                     s200: (f200: packed array [0..199] of char);
                     s220: (f220: packed array [0..219] of char);
                     s240: (f240: packed array [0..239] of char);
                     s256: (f256: packed array [0..255] of char);
                    end;
var
  k: record case boolean of      { "trick" record to fiddle with pointers }
       TRUE:  (tp: trick_ptr);
       FALSE: (ap: xtring);
     end;
  p: trick_ptr;                  { pointer to the created string }
  fp: freeptr;                   { pointer to head of free strings }
  size: 1..33;                   { the size (index into table) of string }

begin
  if length > 100 then size := ((length+1)+420) DIV 20
                  else size := ((length+1) DIV 4) + 1;
  if free_strings[size] <> NIL then
    begin
      str := free_strings[size]^.str;
      fp := free_strings[size]^.next;
      free_strings[size]^.next := free_pointers;
      free_pointers := free_strings[size];
      free_strings[size] := fp;
    end
  else
    begin
      case s_length[size] of
          4: new(p,s4);
          8: new(p,s8);
         12: new(p,s12);
         16: new(p,s16);
         20: new(p,s20);
         24: new(p,s24);
         28: new(p,s28);
         32: new(p,sz32);
         36: new(p,s36);
         40: new(p,s40);
         44: new(p,s44);
         48: new(p,s48);
         52: new(p,s52);
         56: new(p,s56);
         60: new(p,s60);
         64: new(p,s64);
         68: new(p,s68);
         72: new(p,s72);
         76: new(p,s76);
         80: new(p,s80);
         84: new(p,s84);
         88: new(p,s88);
         92: new(p,s92);
         96: new(p,s96);
        100: new(p,s100);
        120: new(p,s120);
        140: new(p,s140);
        160: new(p,s160);
        180: new(p,s180);
        200: new(p,s200);
        220: new(p,s220);
        240: new(p,s240);
        256: new(p,s256);
      end;
      k.tp := p;  str := k.ap;
    end;
  str^[0] := chr(length);
end { create_a_string } ;


procedure release_string(var str: xtring);
  { free the storage used by the given string and place on free list }
  var
    size: string_range;     { size (index into table) of the string }
    f: freeptr;             { head of list of free strings }
begin
  if str <> nullstring then
    begin
      if ord(str^[0]) > 100 then size := (ord(str^[0])+420) DIV 20
                            else size := (ord(str^[0]) DIV 4) + 1;
      if free_pointers = NIL then new_free_element(f)
      else
        begin f := free_pointers; free_pointers := free_pointers^.next; end;
      f^.next := free_strings[size];
      free_strings[size] := f;  f^.str := str;
      str := nullstring;
    end;
end { release_string } ;


procedure copy_string(source: xtring;  var dest: xtring);
  { copy from the source to the destination.  The destination string must 
    exist (= nullstring or some other string).  If the source length is not
    equal to the destination length the destination string is "free"d and a
    new string of the proper size is created. }
  var
    pos: string_range;        { index into string for copy }
begin
  if source^[0] <> dest^[0] then
    begin
      release_string(dest);  create_a_string(dest, ord(source^[0]));
    end;

  for pos := 1 to ord(source^[0]) do  dest^[pos] := source^[pos];
end { copy_string } ;

    
procedure copy_from_string(str: xtring; var name: alpha);
  { copy from a string to an identifier.  Pad with blanks if the string
    has fewer characters than the identifier;  truncate if longer. }
  var
    min,            { smaller of ID_LENGTH and length(string) }
    i: id_range;    { index into the alpha }
begin
  name := null_alpha;
  if ord(str^[0]) < ID_LENGTH then min := ord(str^[0]) else min := ID_LENGTH;
  for i := 1 to min do name[i] := str^[i];
end { copy_from_string } ;


procedure copy_to_string(name: alpha;  var str: xtring);
  { copy from an alpha to a string.  Trailing blanks are deleted. }
  var
    len: id_range;    { length of the identifer }
    i: id_range;      { index into alpha and string for copy }
begin
  len := alpha_length(name);

  if ord(str^[0]) <> len then 
    begin  release_string(str);  create_a_string(str, len);  end;
  for i := 1 to len do  str^[i] := name[i];
end { copy_to_string } ;


function CmpStrLEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 <= s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] > s2^[0] then 
    begin  min_length := ord(s2^[0]);  CmpStrLEQ := FALSE;  end
  else
    begin  min_length := ord(s1^[0]);  CmpStrLEQ := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then
        begin  CmpStrLEQ := FALSE;  done := TRUE;  end
      else
        if s1^[i] < s2^[i] then
          begin  CmpStrLEQ := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLEQ } ;


function CmpStrLT(s1, s2: xtring): boolean;
  { returns TRUE if s1 < s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrLT := TRUE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrLT := FALSE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then 
        begin  CmpStrLT := FALSE;  done := TRUE;  end
      else if s1^[i] < s2^[i] then
        begin  CmpStrLT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLT } ;


function CmpStrGT(s1, s2: xtring): boolean;
  { returns TRUE if s1 > s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrGT := FALSE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrGT := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then 
        begin  CmpStrGT := FALSE;  done := TRUE;  end
      else if s1^[i] > s2^[i] then
        begin  CmpStrGT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrGT } ;


function CmpStrEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 = s2, FALSE otherwise. }
  var
    i: string_range;        { index into the strings }
    done: boolean;          { TRUE if comparison complete }
begin
  CmpStrEQ := FALSE;

  if s2^[0] = s1^[0] then
    begin
      i := 0;  done := FALSE;
      while (i < ord(s1^[0])) and not done do
        begin
          i := i + 1;
          if s1^[i] <> s2^[i] then done := TRUE;
        end;
      if not done then CmpStrEQ := TRUE;
    end;
end { CmpStrEQ } ;


function compare_strings(s1, s2: xtring): compare_type;
  { compare the strings and return the result }
  var
    min_length,             { minimum length of the two strings }
    i: string_range;        { index into the strings }
    result: compare_type;   { result of the comparison }
    still_equal: boolean;   { TRUE if strings are equal to current position }
begin
  if s1^[0] = s2^[0] then
    begin  min_length := ord(s1^[0]);  result := EQ;  end
  else if s1^[0] < s2^[0] then
    begin  min_length := ord(s1^[0]);  result := LT;  end
  else 
    begin  min_length := ord(s2^[0]);  result := GT;  end;

  i := 0;  still_equal := TRUE;
  while (i < min_length) and still_equal do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then
        begin  result := LT;  still_equal := FALSE;  end
      else if s1^[i] > s2^[i] then
        begin  result := GT;  still_equal := FALSE;  end;
    end;

  compare_strings := result;
end { compare_strings } ;


function add_char_to_string(str: xtring;  ch: char): boolean;
  { add the character to the end of the string.  It is assumed that the
    string has been created with length = MAX_STRING_LENGTH and the 
    current length of the string (STR^[0]) is correct.  Always leave the
    last char of the string empty (for path string closing paren). }
begin
  if ord(str^[0])+1 >= MAX_STRING_LENGTH then
    add_char_to_string := FALSE
  else
    begin
      str^[0] := chr(ord(str^[0]) + 1);  str^[ord(str^[0])] := ch;
      add_char_to_string := TRUE;
    end;
end { add_char_to_string } ;


function add_string_to_string(dest, source: xtring): boolean;
  { add the source string to the end of the destination string.  Return
    FALSE if the destination string length is exceeded. It is assumed that
    the destination string has been created as MAX_STRING_LENGTH long and
    its length can be extended that far.  If the resulting string will be
    too long, copy as much as possible. }
  var
    i: string_range;            { index into the strings }
    source_length,              { length of the source string }
    dest_length: string_range;  { length of the destination string }
begin
  add_string_to_string := TRUE;

  if source <> nullstring then
    begin
      dest_length := ord(dest^[0]);  source_length := ord(source^[0]);
      if source_length + dest_length >= MAX_STRING_LENGTH then
        begin
          add_string_to_string := FALSE;
          source_length := MAX_STRING_LENGTH - dest_length - 1;
        end;

      for i := 1 to source_length do dest^[dest_length+i] := source^[i];

      dest^[0] := chr(dest_length + source_length);
    end;
end { add_string_to_string } ;


function add_alpha_to_string(dest: xtring; ident: alpha): boolean;
  { append the identifier (alpha) to the end of the destination string.
    It is assumed that the string has been created MAX_STRING_LENGTH long
    and its length can be extended to there.  If the resulting string
    exceeds MAX_STRING_LENGTH (leaving the last character empty), do
    as much as possible and return FALSE.  Delete trailing spaces from the
    alpha. }
  var
    i: id_range;                  { next char of the_alpha to copy in }
    identifier_length: id_range;  { length of the identifier }
    dest_length: string_range;    { length of the destination string }
begin
  identifier_length := alpha_length(ident);

  dest_length := ord(dest^[0]);
  if dest_length + identifier_length >= MAX_STRING_LENGTH then
    begin
      add_alpha_to_string := FALSE;
      identifier_length := MAX_STRING_LENGTH - dest_length - 1;
    end
  else add_alpha_to_string := TRUE;

  for i := 1 to identifier_length do
    dest^[dest_length+i] := ident[i];
  dest^[0] := chr(dest_length + identifier_length);
end { add_alpha_to_string } ;


function add_number_to_string(str: xtring; number: longint): boolean;
  { Append the given number (NUMBER) to the end of the input string (STR).
    If the number is < 0 then append a '-' at the start.  It is assumed that
    the input string has been created with length = MAX_STRING_LENGTH and
    that the length of the string (STR^[0]) is the current length.  Always
    leave the last char of the string empty (for path string closing ')'). }


  procedure build_number(n: natural_number);
    { add the given number to the string }
  begin
    if n > 9 then build_number(n DIV 10);
    if ord(str^[0])+1 >= MAX_STRING_LENGTH then
      add_number_to_string := FALSE
    else
      if not add_char_to_string(str, chr((n mod 10) + ord('0'))) then
        add_number_to_string := FALSE;
  end { build_number } ;


begin { add_number_to_string }
  add_number_to_string := TRUE;

  if number < 0 then
    begin
      number := abs(number);
      if not add_char_to_string(str, '-') then
        add_number_to_string := FALSE;
    end;

  build_number(number);
end { add_number_to_string } ;


function number_to_string(numb: longint): xtring;
  { return string representing the number in decimal }
  var
    temp: xtring;   { for building number }
begin
  create_a_string(temp, MAX_STRING_LENGTH);
  temp^[0] := chr(0);
  if add_number_to_string(temp, numb) then ;
  number_to_string := enter_string(temp);
  temp^[0] := chr(MAX_STRING_LENGTH);
  release_string(temp);
end { number_to_string } ;


function string_to_natural_number(str: xtring): natural_number;
  { convert string containing a decimal natural number into that number }
  const
    RADIX = 10;  { decimal number }
  var
    temp: natural_number;        { value of the function to be returned }
    len: string_range;           { length of str }
    i: string_range;             { index into str }
    ch: char;                    { current char }
    next_digit: 0..9;            { numeric value of current digit }
    trailing_junk: boolean;      { TRUE if chars follow the last digit }
    done: boolean;               { TRUE when done with a loop }
begin
  temp := 0;

  { ignore preceeding blanks }
  len := ord(str^[0]);
  i := 0;  done := FALSE;
  while (i < len) and not done do
    begin
      i := i + 1;
      if str^[i] <> ' ' then done := TRUE;
    end;

  { add up the digits }
  if i > 0 then i := i - 1;  done := FALSE;  trailing_junk := FALSE;
  while (i < len) and not done do 
    begin
      i := i + 1;
      ch := str^[i];
      case ch of
	'0': next_digit := 0;  '1': next_digit := 1;
	'2': next_digit := 2;  '3': next_digit := 3;
	'4': next_digit := 4;  '5': next_digit := 5;
	'6': next_digit := 6;  '7': next_digit := 7;
	'8': next_digit := 8;  '9': next_digit := 9;
	OTHERWISE 
	  begin
	    done := TRUE;
	    trailing_junk := TRUE;
	  end;
      end;
      if not done then 
	if (temp > MAXINT div RADIX) or 
	   ((temp = MAXINT div RADIX) and
	    (next_digit > MAXINT mod RADIX)) then
	  begin  assert(240 { ovf });  done := TRUE;   end
	else
	  begin  temp := RADIX * temp + next_digit;  end;
    end { while } ;
  
  { ignore blanks but report other extraneous trailing junk }
  if trailing_junk then 
    begin
      i := i - 1;  done := FALSE;
      while (i < len) and not done do
	begin
	  i := i + 1;
	  if str^[i] <> ' ' then
	    begin
	      done := TRUE;
	      error(50 { extraneous junk at end of number });
	      error_dump_string(str);
	      error_dump_CRLF;
	    end;
	end;
    end;
  
  string_to_natural_number := temp;
end { string_to_natural_number } ;


function UNIX_file_concat(path, name: xtring): xtring;
  { concatenate the two strings PATH and NAME to produce a
    fully rooted file name for the file }
  var
    file_name: xtring;           { name of file containing drawing }
begin
  create_a_string(file_name, MAX_STRING_LENGTH);
  file_name^[0] := chr(0);

  if path <> nullstring then
    begin
      if add_string_to_string(file_name, path) then ;
      if add_char_to_string(file_name, '/') then ;
    end;

  if add_string_to_string(file_name, name) then ;

  UNIX_file_concat := enter_string(file_name);

  file_name^[0] := chr(MAX_STRING_LENGTH);
  release_string(file_name);
end { UNIX_file_concat } ;


function temp_UNIX_file_concat(path, name: xtring): xtring;
  { concatenate the two strings PATH and NAME to produce a
    fully rooted file name for the file in a temp string that is
    not in the string table and so can be released. }
  var
    file_name: xtring;           { name of file containing drawing }
    len: string_range;           { length of string }
begin
  if path = nullstring then len := ord(name^[0])
  else len := min(ord(name^[0]) + ord(path^[0]) + 1, MAX_STRING_LENGTH);

  create_a_string(file_name, len);
  file_name^[0] := chr(0);

  if path <> nullstring then
    begin
      if add_string_to_string(file_name, path) then ;
      if add_char_to_string(file_name, '/') then ;
    end;

  if not add_string_to_string(file_name, name) then
    error(22 { string overflow });

  temp_UNIX_file_concat := file_name;
end { temp_UNIX_file_concat } ;


function upper_case(str: xtring): xtring;
  { upper case the given string }
  var
    i: string_range;         { index into the string }
    new_string: xtring;      { new string to be created }
begin
  create_a_string(new_string, ord(str^[0]));

  for i := 1 to ord(str^[0]) do
    if str^[i] in lower_case then
      new_string^[i] := chr(ord(str^[i]) - ord('a') + ord('A'))
    else
      new_string^[i] := str^[i];

  upper_case := enter_and_release_string(new_string);
end { upper_case } ;


function substring(str : xtring; start,len : string_range): xtring;
  { returns the specified substring of str.  substring starts
    at start and is len chars long.  It will be truncated if 
    start+len-1 > length of str.  Returned string is a string table
    entry. }
  var 
    offset: string_range;      { index offset in str }
    temp: xtring;              { temp for building substring }
    i: string_range;           { index into str and temp }
begin
  if start <= ord(str^[0]) then 
    begin
      offset := start - 1;
      if offset+len > ord(str^[0]) then len := ord(str^[0]) - offset;
      if len > 0 then 
	begin
	  create_a_string(temp, len);
	  for i := 1 to len do temp^[i] := str^[i + offset];
          substring := enter_and_release_string(temp);
	end
      else substring := nullstring;
    end
  else substring := nullstring;
end; {substring}


#include "args.p"
#include "init.p"


(**)     { ------- current parse string output (error) ------- }


procedure print_input_line(var f: textfile; error_num: error_range;
                           indent: boolean);
  { print the input parse line to the given file. If indent, then
    indent 2 extra spaces. }
  var
    position: string_range;            { position to print pointer (^) }
    base_pos,                          { starting position of current string }
    curr_pos: string_range;            { current output string length }


begin { print_input_line }
  base_pos := 0;

  if indent then write(f, '   ')
            else write(f, ' ');

  dump_string(f, instring);
  curr_pos := ord(instring^[0]);
  writeln(f);

  { LAST_SYM_POS points to last character preceding current symbol.  If the
    the compiler does not understand the current symbol, the pointer should
    point to the 1st place in the symbol (e.g., expected > ), hence, need to
    use LAST_SYM_POS+1.  If READ_STATE = FINPUT, then LINE_POS points to the
    last character read in and should be used as is.  If READ_STATE = 
    FGOT_CHAR, then LINE_POS points to the character following the last char
    read in.  If, however, LINE_POS points to the last position in the string,
    use it as is. }

  if error_num IN scan_past_errors then position := last_sym_pos+1
  else
    if read_state = FINPUT then position := line_pos
    else
      if line_pos = ord(instring^[0]) then position := line_pos
      else if line_pos > 1 then position := line_pos-1
                           else position := line_pos;

  position := position {in line} + base_pos {text macro} + 1 {leading space};
  if indent then position := position + 2 { indentation };
  if position > 1 then writeln(f, error_position_char:position)
                  else writeln(f, ' ', error_position_char);
end { print_input_line } ;


(**)     { ------- error routine ------- }


procedure error(*error_num: error_range*);
begin
  writeln(monitor);
  writeln(monitor, error_strings[error_num]);
  halt_with_status(FATAL_COMPLETION);
end { error } ;


#if PMAX
function length(var str: varystring): string_range;
  var
    i: string_range;
    len: string_range;
begin
  len := 0;
  i := MAX_STRING_LENGTH;
  while (len = 0) and (i > 0)
    begin
      if str[i] <> ' ' then len := i
      else i := i - 1;
  end;
  length := len;
end;
#endif


procedure parse_error(err: 
#if SVS
                           svs_string);
#endif
#if VAX || PMAX
                           varystring);
#endif
#if SUN
                           array [lb..ub: integer] of char);
#endif
  var 
    i: string_range;
begin
  writeln(monitor);
  print_input_line(monitor, 0, FALSE);
#if SVS
  write(monitor, err);
#endif
#if SUN
  for i := lb to ub do write(monitor, err[i]);
#endif
#if VAX || PMAX
  for i := 1 to length(err) do write(monitor, err[i]);
#endif
  writeln(monitor);
  writeln(monitor, 'sy = ', ord(sy));
  halt_with_status(FATAL_COMPLETION);
end { parse_error } ;

            
procedure assert(*assertion_num: assert_range*);
begin { assert }
  writeln(monitor);
  writeln(monitor, assert_strings[assertion_num]);
  halt_with_status(FATAL_COMPLETION);
end { assert } ;


(**)     { ------- name  utilities ------- }


function alpha_length(*name: alpha): id_range*);
  { find the length of an identifier by scanning for its end }
  var
    i: id_range;         { index into the identifier }
    done: boolean;       { TRUE when end of alpha found }
begin
  i := ID_LENGTH;  done := FALSE;
  while (i > 1) and not done do
    if name[i] <> ' ' then done := TRUE else i := i - 1;

  alpha_length := i;
end { alpha_length } ;


(**)     { ------- I/O utilities ------- }


function width_of_integer(i: longint): longint;
  { Returns the minimum number of places PASCAL uses to print i }
  var
    width: longint;     { width of the integer i in print positions }
begin
  width := 1;
  if i < 0 then
    begin  width := 2;  i := -1;  end;

  if i < 10 then  { ok as is }
  else if i < 100 then width := width + 1
  else if i < 1000 then width := width + 2
  else if i < 10000 then width := width + 3
  else if i < 100000 then width := width + 4
  else if i < 1000000 then width := width + 5
  else if i < 10000000 then width := width + 6
  else if i < 100000000 then width := width + 7
  else if i < 1000000000 then width := width + 8
  else width := width + 9;

  width_of_integer := width;
end { width_of_integer } ;


procedure dump_string(*var f: textfile; str: xtring*);
  { dump the given string (STR) to the given file (F) as is }
  var
    hack: string_hack;
begin
  if ord(str^[0]) > 0 then
    begin
      hack.i := ord(str) + 1;  write(f, hack.s^:ord(str^[0]));
    end;
end { dump_string } ;
    

procedure writestring(var f: textfile; str: xtring);
  { write the given string (STR) to the given file (F) as is with quotes }
begin
  write(f, OUTPUT_QUOTE_CHAR);

  dump_string(f, str);

  write(f, OUTPUT_QUOTE_CHAR);
end { writestring } ;
    

procedure print_string(*var f: textfile; str: xtring*);
  { print the given string (STR) to the given file (F) }
  var
    len: string_range;
    hack: string_hack;
begin
  { NOTE: Illegal characters can ONLY occur in str^[1] -- used there
    to quickly signify an NC. Illegal chars in input files are filtered
    by insymbol. }
  len := ord(str^[0]);
  if (len > 0) and (str^[1] in legal_chars) then                           
    begin
      hack.i := ord(str) + 1;  write(f, hack.s^:len);
    end
  else if (len > 1) then
    begin
      hack.i := ord(str) + 2;  write(f, hack.s^:len-1);
    end;
end { print_string } ;


procedure print_string_repeat_quotes(var f: textfile; str: xtring);
  { print the string (STR) to the given file (F) doubling up any quotes }
  var
    len, start: string_range;
    stop: integer;  { may overflow string_range by 1 }
    hack: string_hack;
    found_quote: boolean;
begin
  len := ord(str^[0]);  
  stop := 1;
  while stop <= len do
    begin
      start := stop;  found_quote := FALSE;
      while (stop <= len) and not found_quote do
	begin
	  found_quote := str^[stop] = OUTPUT_QUOTE_CHAR;
	  stop := stop + 1;
	end;
      hack.i := ord(str) + start;
      write(f, hack.s^:(stop - start));
      if found_quote then write(f, OUTPUT_QUOTE_CHAR:1);
    end;
end { print_string_repeat_quotes } ;


procedure print_string_with_quotes(var f: textfile; str: xtring);
  { write a string to the output file }
begin
  write(f, OUTPUT_QUOTE_CHAR);

  print_string_repeat_quotes(f, str);

  write(f, OUTPUT_QUOTE_CHAR);
end { print_string_with_quotes } ;
    

procedure print_alpha(var f: textfile; name: alpha);
  { print the given alpha (NAME) to the given file (F) }
begin
  write(f, name:alpha_length(name));
end { print_alpha } ;


procedure writealpha(var f: textfile; name: alpha);
  { write an identifier to the output file removing trailing blanks }
begin
  write(f, name:alpha_length(name));
end { writealpha } ;


procedure dump_left_and_right(var f: textfile; left, right: bit_range);
begin
  if left <> SCALAR_BIT then 
    if left = right then write(f, '<', left:1,  '>')
                    else write(f, '<', left:1, '..', right:1, '>');
end { dump_left_and_right } ;


(**)     { ------- other utilities ------- }


procedure disp_line(message: message_type);
  { write a message and a pointer to the output file }
  var
    width: string_range;    { position to place circumflex }
begin
  if line_pos > 0 then  width := line_pos  else  width := 1;
  writeln(outfile, '^':width, message:message_length+4);
end { disp_line } ;


(**)     { ------- error output routines ------- }


procedure error_dump_CRLF;
  { print the CRLF to the appropriate files }
begin
  writeln(monitor);
end { error_dump_CRLF } ;


procedure error_dump_indent(indentation: natural_number);
  { outputs a specified number of spaces to the error files.  If
    printing a CmpLst, then 2 extra spaces are printed to
    CmpLog (as PrintCmpLst implies that we are in the process of 
    printing a page and error messages are to be indented under the
    page heading). }
begin
  write(monitor, ' ':indentation);
end { error_dump_indent } ;


procedure error_dump_char(ch: char);
  { output a character to the error files }
begin
  write(monitor, ch);
end { error_dump_char } ;


procedure error_dump_integer(int: longint);
  { print the integer on the error files }
begin
  write(monitor, int:1);
end { error_dump_integer } ;


procedure error_dump_alpha(data: alpha);
  { print an alpha to the error files }
begin
  writealpha(monitor, data);
end { error_dump_alpha } ;


procedure error_dump_string(*str: xtring*);
  { print the given string to the error files }
begin
  print_string(monitor, str);
end { error_dump_string } ;


procedure error_dump_ioresult(iores: integer);
  { output an ioresult error message.  Do not print
    anything if iores = 0 (no error). }
begin
  if iores <> 0 then
    begin
      error_dump_indent(indent);
      write_ioresult(monitor, iores);
      error_dump_CRLF;
    end;
end { error_dump_ioresult } ;


procedure error_dump_file_name(name: xtring);
  { dump the name of the file to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('File name=      ');
  error_dump_string(name);
  error_dump_CRLF;
end { error_dump_file_name } ;


procedure error_dump_alpha_file_name(file_name: alpha);
  { dump the name of the file to the error files }
begin
  error_dump_indent(indent);
  error_dump_alpha('File name=      ');
  error_dump_alpha(file_name);
  error_dump_CRLF;
end { error_dump_alpha_file_name } ;


#include "fileio.p"


function open_a_file(file_name: xtring; which: parse_file_type): boolean;
  { open the specified file for read and parse the first token from it.
    If the FILE_NAME is not empty (NULLSTRING), open the file named.  If
    it is empty, open the file as specified in the file equation. }
  var
    ok: boolean;         { TRUE iff file open successful }
begin
  ok := reset_file(file_name, which);

  if ok then  
    begin
      input_line_number := 0;
      read_state := FINIT;
      insymbol;
    end;
  open_a_file := ok;
end { open_a_file } ;


(**)     { ------- string hash table routines ------- }


function make_and_enter_string(*name: alpha): string*);
  { convert an alpha into a string and enter it into the hash table }
  var
    temp_string: xtring;        { temporary string }
    original_string: xtring;    { original string value }
begin
  temp_string := nullstring;
  copy_to_string(name, temp_string);

  original_string := temp_string;
  temp_string := enter_string(temp_string);

  if temp_string <> original_string then
    release_string(original_string);

  make_and_enter_string := temp_string;
end { make_and_enter_string } ;


function enter_and_release_string(*str: string): string*);
  { enter the given string, and release the original pointer }
  var
    new_string: xtring;    { string from the table }
begin
  new_string := enter_string(str);

  if new_string <> str then
    release_string(str);

  enter_and_release_string := new_string;
end { enter_and_release_string } ;

  
function enter_string(*str: string): string*);
  { enter or find the string in the string hash table and return it.
    If not found in the table, the string's value is copied to a new
    string created afresh. }
  var
    i: string_range;           { index into STR }
    sum: natural_number;       { checksum of the string }
    index: hash_string_range;  { index into the string table }
    last,                      { last element checked in list }
    element: hash_string_ptr;  { element in the list of names }
    compare: compare_type;     { result of string compare }
    done: boolean;             { TRUE when place in table found }


  procedure insert_entry(list_element: hash_string_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: hash_string_ptr; { new element to be placed into the list }
  begin
    new(new_element);

    new_element^.str := nullstring;
    copy_string(str, new_element^.str);
    if list_element = NIL then
      begin
        new_element^.next_hash_string := string_table[index];
        string_table[index] := new_element;
      end
    else
      begin
        new_element^.next_hash_string := list_element^.next_hash_string;
        list_element^.next_hash_string := new_element;
      end;

    enter_string := new_element^.str;
  end { insert_entry } ;


begin { enter_string }
  if ord(str^[0]) = 0 then enter_string := nullstring
  else
    begin
      { create a hash index from the specified name }

      sum := 0;  i := 0;
      for i := 1 to ord(str^[0]) do
         sum := sum + ord(str^[i]);

      index := sum MOD (HASH_STRING_TABLE_SIZE+1);

      element := string_table[index];
      if element = NIL then insert_entry(NIL)
      else
        begin
          last := NIL;  done := FALSE;
          repeat
            if element^.next_hash_string = NIL then
              begin
                done := TRUE;
                compare := compare_strings(str, element^.str);
              end
            else
              begin
                compare := compare_strings(str, element^.str);
                if compare <> GT then done := TRUE
                else
                  begin
                    last := element;  element := element^.next_hash_string;
                  end;
              end;
          until done;

          case compare of
            LT:  insert_entry(last);
            EQ:  enter_string := element^.str;
            GT:  insert_entry(element);
          end;
        end;
    end;
end { enter_string } ;


(**)     { ------- identifier name routines ------- }


function compare_identifiers(id1, id2: name_ptr): compare_type;
  { compare the 2 names and return the result }
begin
  if (id1 = NIL) or (id2 = NIL) then 
    begin
      assert(221 { nil name passed });
      writeln(monitor, ' compare_identifiers');
      if id1 <> NIL then compare_identifiers := GT 
      else if id2 <> NIL then compare_identifiers := LT 
      else compare_identifiers := EQ;
    end
  else if (id1^.name < id2^.name) then compare_identifiers := LT
  else if (id1^.name = id2^.name) then compare_identifiers := EQ
  else compare_identifiers := GT;
end { compare_identifiers } ;


function enter_name(*name: alpha): name_ptr*);
  { enter or find the name in the name hash table and return a pointer }
  var
    i: 0..ID_LENGTH;           { index into NAME }
    sum: natural_number;       { checksum of the name }
    index: name_table_range;   { index into the name table }
    last,                      { last element checked in list }
    element: name_ptr;         { element in the list of names }
    done: boolean;             { TRUE when end of alpha found }


  procedure insert_entry(list_element: name_ptr);
    { insert a new entry after the given list element.  If the list element
      is NIL, insert at the head of the list. }
    var
      new_element: name_ptr;  { new element to be placed into the list }
  begin
    new(new_element);

    new_element^.name := name;
    new_element^.kind := default_attributes;
    new_element^.definition := nullstring;
    new_element^.sy := NULLSY;

    if list_element = NIL then
      begin
        new_element^.next := name_table[index];
        name_table[index] := new_element;
      end
    else
      begin
        new_element^.next := list_element^.next;
        list_element^.next := new_element;
      end;

    enter_name := new_element;
  end { insert_entry } ;


begin { enter_name }
  { create a hash index from the specified name }

  sum := 0;  i := 0;  done := FALSE;
  while (i < ID_LENGTH) and not done do
    begin
      i := i + 1;
      if name[i] = ' ' then done := TRUE else sum := sum + ord(name[i]);
    end;
  index := sum MOD (name_table_size+1);

  element := name_table[index];
  if element = NIL then insert_entry(NIL)
  else
    begin
      last := NIL;
      while (name > element^.name) and (element^.next <> NIL) do
        begin  last := element;  element := element^.next;  end;
      if name = element^.name then enter_name := element
      else if name < element^.name then insert_entry(last)
      else insert_entry(element);
    end;
end { enter_name } ;


{------------------------ end procedures --------------------------}

