#include <stream.h>
#include <string.h>
#include "fault.h"
#include "parsebuf.h"

extern int read (int, char *, unsigned);


inline int isblank(char c) { return c == ' ' || c == '\t'; } // NOT '\n'


int parsebuf::doallocate()
{
    if ((base = (char*)h->mem(2*BUFSIZ)) == NULL) return EOF;

    pptr = gptr = base;
    last_pptr = next_pptr = 0;
    base2 = base + BUFSIZ;
    eptr = base2 + BUFSIZ;
    *(eptr-1) = '\n';
    
    return 0;
}


int parsebuf::underflow()
{
    if (!opened || _eof) return EOF;

    if (!base) {
	if (doallocate() == EOF) { _eof = 1;  return EOF; }
    }
    else gptr = pptr > base2 ? base : base2;

    if (next_pptr) {
	last_pptr = pptr;
	pptr = next_pptr;	// Already read
	next_pptr = 0;
    }
    else {
	int count;
	if ((count=read(fd, gptr, BUFSIZ)) < 1) { _eof = 1;  return EOF; }
	last_pptr = pptr;
	pptr = gptr+count;
    }
    return *gptr & 0377;
}


#define in_range(p) ((p) >= base && (p) < eptr)
    // NOTE: this is just a gross check for reasonableness.  If we wanted
    // to be very thorough, we would check p against the regions of the
    // buffer actually known to be read; this would be overkill.


char* parsebuf::prev(char* p)
{
    if (!in_range(p)) return 0;

    if (p == base || p == base2) 
	return last_pptr ? last_pptr - 1 : 0;
    return --p;
}


char* parsebuf::succ(char* p)
{
    if (!in_range(p)) return 0;

    if (p == pptr) {
	if (!next_pptr) return read_ahead();
	return next_pptr <= base2 ? base : base2;
    }
    if (p == last_pptr) return pptr <= base2 ? base : base2;
    if (p == next_pptr) return 0;
    return ++p;
}


int parsebuf::overflow(int)
{ 
    fault("Can't write to a parsebuf");
    return EOF;
}


int parsebuf::column(register char* p)
{
    if (!in_range(p)) {
        fault("position not in range");
	return 1;
    }

    register n = 1;
    for (p = prev(p); p && *p != '\n'; p = prev(p)) n++;
    return n;
}


int parsebuf::lineno(register char* pos) 
{
    if (!in_range(pos)) {
        fault("position not in range");
	return _lineno;
    }

    int num = _lineno;
    register char* p = gptr;
    for (p = gptr; p && p != pos; p = prev(p)) if (*p == '\n') num--;
    return num;
}


// Have not yet read the newline following this token, so we have
// to read the next buffer and then fix things up.
// We can only read ahead BUFSIZ chars, so lines longer than that
// may not be able to be entirely read.
char* parsebuf::read_ahead()
{
    if (next_pptr) return 0;	// Already have read ahead -- don't clobber
				// the current position.

    if (gptr == base && pptr <= base2 || 
        gptr == base2 && pptr > base2) return 0; 
	// must leave room for putback.

    char* save_gptr = gptr;
    char* save_pptr = pptr;

    if (underflow() == EOF) return 0;

    char* val = gptr;

    last_pptr = 0;
    next_pptr = pptr;
    pptr = save_pptr;

    gptr = save_gptr;

    return val;
}


const char* parsebuf::line(char* pos)
{
    copy_buf.clear();

    register char* p;
    register char* lastp;
    for (lastp = pos, p = prev(pos); 
         p && *p != '\n'; 
	 lastp = p, p = prev(p)) ;
    for (p = lastp; p && *p != '\n'; p = succ(p)) copy_buf.put(*p);
    return copy_buf;
}


void parsebuf::sputbackc(int c)
{
    if (c == EOF) return;
    if (gptr == base || gptr == base2)
	if (last_pptr) {
	    *gptr = c;
	    gptr = last_pptr - 1;
	    next_pptr = pptr;
	    last_pptr = 0;
	}
	else fault("too many sputbackc()s");
    else *gptr-- = c;
}


void parsebuf::init(heap* hp)
{
    _filename = base = gptr = next_pptr = last_pptr = 0;
    _lineno = 1;
    _eof = 0;
    h = hp;
}


// If continuation char is legitimate, skip it and return next char.
// otherwise, return the continuation char
int parsebuf::skip_continuation()
{
    AS(*gptr == '~');
    register char* p;
    for (p = succ(gptr); p && isblank(*p); p = succ(p));
    if (!p) {
	// Nonstandard error handling, as we're too low-level, but one
	// would have had to really MAKE this one happen.
	fprintf(stderr, "WARNING -- too many blanks following a '~' char\n");
	fprintf(stderr, "    (will assume it's NOT a continuation char).\n");
	fprintf(stderr, "File %s line %d\n", _filename, lineno(gptr));

        return *gptr;
    }
    if (*p != '\n') return *gptr;  // Not a continuation, just a '~'

    // Come forward in buffer to the newline
    if (gptr < base2 && p >= base2 || gptr >= base2 && p < base2) {
	// Read ahead to next buffer -- switch-em
	if (!next_pptr) fault("confused");
	last_pptr = pptr;
	pptr = next_pptr;
	next_pptr = 0;
    }
    gptr = p;

    // Return the next char -- note that if it's a '~', that
    // fact will be recursively handled.
    return snextc();
}


parsebuf* parsebuf::open(const char* filename)
{
    if (_filename) h->memfree(_filename, strlen(_filename)+1);
    if (!filename) filename = "<unknown>";
    _filename = (char*)h->mem(strlen(filename)+1);
    strcpy(_filename, filename);
    return (parsebuf*)filebuf::open(filename, input);
}


parsebuf::~parsebuf() 
{
    if (base) h->memfree(base, 2*BUFSIZ);
    if (on_heap) {
	h->memfree(this, sizeof(*this));  this = 0;
	// Warning -- this does not call the destructors for the
	// base class nor the member classes
    }
}
