#ifndef gavl_inh_define
/*
    generic homogeneous AVL tree declaration for a key which is a class
    with a cmp() function.  Creates a class derived from the key which is
    suitable for table entry.  (Thus the table entry class inherits all of the
    behaviour of the key.)  It is used as follows:

    // foo.h
    class foo {
        int stuff;
    public: 
        int cmp(const foo&);
    };
    gavl_inh_define(foo); // Declares classes fooEntry and fooTable

    // foo.c
    gavl_inh_search(foo);  // Defines search procedure using foo::cmp(const foo&)

    // bar.h
    class bar {
    public:
        fooTable foos;  // A table of 'fooEntry's which will behave like 'foo's
    };

    OR, in order to make it possible for these (fooEntrys) to appear as members 
    of other classes, the following can be used:

    // foo_is_member.h
    gavl_inh_member_define(foo_member_name,foo_is_member,foo); // Declares many things.
    class foo_is_member {
        junk stuff;
        gavl_inh_member_inst(foo_member_name,foo_is_member,foo);
    public:
    }
    gavl_inh_member_inline(foo_member_name,foo_is_member,foo);

    // grbx.h
    class grbx {
        junk stuff;
    public:
        gavl_inh_member_root(foo_member_name,foo_is_member) foos;
    };
        
    The tables are used like tables built with the macros in gavl.h
*/

#include <generic.h>
#include "avlbase.h"
#include "giterator.h"


// Class names to be created
#define gavl_inh(keytype) name2(keytype,Entry)
#define gavl_inh_baseroot(keytype) name2(keytype,Table)
#define gavl_inh_inorder(keytype) name2(keytype,Inorder)
#define gavl_inh_preorder(keytype) name2(keytype,Preorder)


extern char *already_in_tree; // Fault message for gavlkey(keytype)::operator=()


// Given keytype define classes keytypeEntry, keytypeTable, 
// keytypeInorder, keytypePreorder, including all (inline) member functions
#define gavl_inh_define(keytype) \
class gavl_inh(keytype) : public keytype {\
    friend class gavl_inh_baseroot(keytype);\
    friend class gavl_inh_preorder(keytype);\
    friend class gavl_inh_inorder(keytype);\
    avlbase avl;\
    gavl_inh(keytype)* to_classtype(avlbase*);\
    gavl_inh(keytype)* left();\
    gavl_inh(keytype)* right();\
public:\
    gavl_inh(keytype)(const keytype& key) : (key) {}\
    const keytype& operator=(const keytype& other)\
    {\
	if (avl.in_tree()) fault(already_in_tree);\
	keytype* _this = this;\
	*_this = other;  return other;\
    }\
};\
gavl_inh(keytype)* gavl_inh(keytype)::to_classtype(avlbase* p)\
{\
    return !p ? 0 :\
	(gavl_inh(keytype)*)((char*)p - (int)&(((gavl_inh(keytype)*)0)->avl));\
}\
gavl_inh(keytype)* gavl_inh(keytype)::left()\
    { return to_classtype(avl.left()); }\
gavl_inh(keytype)* gavl_inh(keytype)::right()\
    { return to_classtype(avl.right()); }\
\
\
class gavl_inh_baseroot(keytype) : public avlroot_base {\
    friend class gavl_inh_preorder(keytype);\
    friend class gavl_inh_inorder(keytype);\
    gavl_inh(keytype) *search(const keytype& key, avlpath *path);\
    gavl_inh(keytype)* to_classtype(avlbase* p)\
	{ return ((gavl_inh(keytype)*)0)->to_classtype(p); }\
public:\
    gavl_inh_baseroot(keytype)() {}\
    gavl_inh(keytype) *insert(gavl_inh(keytype) *object)\
    {\
	 avlpath path; gavl_inh(keytype) *found;\
	 if (!(found = search(*object, &path)))\
	     avlroot_base::insert(&object->avl, path);\
	 return found ? found : object;\
    }\
    gavl_inh(keytype) *find(const keytype& key)\
    {\
	 avlpath path;\
	 gavl_inh(keytype) *found = search(key, &path);\
	 return found;\
    }\
    void pick(gavl_inh(keytype) *object)\
    {\
	 avlpath path;\
	 if (!search(*object, &path)) fault("Not found");\
	 avlroot_base::pick(&object->avl, path);\
    }\
    gavl_inh(keytype) *findpick(const keytype& key)\
    {\
	 avlpath path;\
	 gavl_inh(keytype) *found = search(key, &path);\
	 if (found) avlroot_base::pick(&found->avl, path);\
	 return found;\
    }\
};\
\
\
class gavl_inh_inorder(keytype) : public avl_inorder {\
protected:\
    gavl_inh(keytype)* to_classtype(avlbase *p)\
	{ return ((gavl_inh(keytype)*)0)->to_classtype(p); }\
public:\
    gavl_inh_inorder(keytype) (const gavl_inh_baseroot(keytype)& root) : (root) {}\
    gavl_inh(keytype)* operator()()\
	{ return to_classtype(avl_inorder::operator()()); }\
\
    gavl_inh(keytype)* next()\
        { return (*this)(); }\
    gavl_inh(keytype)* first()\
	{ init();  return (*this)(); }\
};\
\
\
class gavl_inh_preorder(keytype) : public avl_preorder {\
protected:\
    gavl_inh(keytype)* to_classtype(avlbase *p)\
	{ return ((gavl_inh(keytype)*)0)->to_classtype(p); }\
public:\
    gavl_inh_preorder(keytype) (const gavl_inh_baseroot(keytype)& root) : (root) {}\
    gavl_inh(keytype)* operator()()\
	{ return to_classtype(avl_preorder::operator()()); }\
\
    gavl_inh(keytype)* next()\
        { return (*this)(); }\
    gavl_inh(keytype)* first()\
	{ init();  return (*this)(); }\
};\
\
\
/*typedef gavl_inh_inorder(keytype) GITERATOR(,gavl_inh(keytype))*/\
typedef gavl_inh_inorder(keytype) name3(_itr_,keytype,Entry)


#define gavl_inh_search(keytype) \
gavl_inh(keytype) *\
gavl_inh_baseroot(keytype)::search(const keytype& key, avlpath *path)\
{\
    gavl_inh(keytype) *p = to_classtype(tree_root());\
    while (p) {\
        int comparison;\
        path->push(comparison = key.cmp(*p));\
	if (!comparison) break;\
	p = (comparison < 0) ? p->left() : p->right();\
    }\
    if (!p) path->push(0);\
    return p;\
}

#endif
