/* Bag.c -- implementation of a Set of Objects with possible duplicates

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2017
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-5363
	uucp: {decvax!}seismo!elsie!cecil!keith
	September, 1985

Function:
	
A Bag is like a Set, except Bags can contain multiple occurrences of
equal objects.  Bags are implemented by using a Dictionary to associate
each object in the Bag with its number of occurrences.

Modification History:
	
06-Oct-86	S. M. Orlow

1.  Added binary I/O constructor, storer, and reader functions

11-Feb-86	K. E. Gorlen

1.  Modify Collection::asBag to return Bag with minimum capacity of
CLTN_DEFAULT_CAPACITY.

31-Jan-86	K. E. Gorlen

1.  Modify add to return pointer to object in Bag.
2.  Modify remove to return pointer to last occurrence of
    object when removed.

9-Jan-86	K. E. Gorlen

1.  Modify storer/Bag(istream&,Bag&) to write/read collection capacity.

7-Jan-86	K. E. Gorlen

1.  Add Bag::species().

2.  Modify Bag::isEqual to use species().


*/

#include "Bag.hxx"
#include "AssocInt.hxx"
#include "Integer.hxx"
#include "oopsIO.hxx"

#define	THIS	Bag
#define	BASE	Collection
DEFINE_CLASS(Bag,Collection,1,NULL,NULL);

extern const int OOPS_REMOVEERR;

Bag::Bag(UNSIGNED size) : contents(size)
{
	count = 0;
}

Bag::Bag(const Bag& b) : contents(b.contents)
{
	count = b.count;
}

void Bag::operator=(const Bag& b)
{
	contents = b.contents;
	count = b.count;
}
	
void Bag::reSize(UNSIGNED newSize)
{
	contents.reSize(newSize);
}

obid Bag::addWithOccurrences(const Object& ob, UNSIGNED n)
{
	register AssocInt* a = (AssocInt*)&contents.assocAt(ob);
	obid o = &ob;
	if (a==nil) contents.add(*new AssocInt(ob,n));
	else {
		register Integer& i = *((Integer*)(a->value()));
		o = a->key();
		i.value(i.value()+n);
	}
	count += n;
	return o;
}

obid Bag::add(const Object& ob)
{
	return addWithOccurrences(ob,1);
}

UNSIGNED Bag::capacity()	{ return contents.capacity(); }

Collection& Bag::addContentsTo(Collection& cltn)
{
	DO(*this,obid,ob) cltn.add(*ob); OD
	return cltn;
}

void Bag::deepenShallowCopy()
{
	BASE::deepenShallowCopy();
	contents.deepenShallowCopy();
}

obid Bag::doNext(DoCltnPos& pos)
{
	if (pos.ptr == 0) {
		pos.ptr = contents.doNext(pos);
		if (pos.ptr == 0) return 0;
		pos.num = 0;
	}
	AssocInt& a = *(AssocInt*)pos.ptr;
	if (pos.num++ < ((Integer*)a.value())->value()) return a.key();
	pos.ptr = 0;
	return doNext(pos);
}

obid Bag::remove(const Object& ob)
{
	register AssocInt* a = (AssocInt*)&contents.assocAt(ob);
	obid rob = &ob;
	if (a==nil) setOOPSerror(OOPS_REMOVEERR,DEFAULT,this,className(),ob.className(),&ob);
	register Integer* i = (Integer*)(a->value());
	register UNSIGNED n = i->value();
	if (--n == 0) {
		rob = a->key();
		delete contents.remove(*a);
	}
	else i->value(n);
	--count;
	return rob;
}

bool Bag::operator==(const Bag& b)
{
	return count==b.count && contents==b.contents;
}

UNSIGNED Bag::hash()	{ return count^contents.hash(); }

bool Bag::isEqual(const Object& p)
{
	return p.isSpecies(class_Bag) && *this==*(Bag*)&p;
}

const Class* Bag::species()	{ return &class_Bag; }

UNSIGNED Bag::occurrencesOf(const Object& ob)
{
	register AssocInt* a = (AssocInt*)&contents.assocAt(ob);
	if (a==nil) return 0;
	else return ((Integer*)a->value())->value();
}

void Bag::printOn(ostream& strm)
{
	UNSIGNED n=0;
	strm << className() << "[\n";
	DO(contents,obid,o)
		if (n>0) strm << "\n";
		o->printOn(strm);
		n++;
	OD
	strm << "]\n";
}

UNSIGNED Bag::size()	{ return count; }

Bag Collection::asBag()
{
	Bag cltn(MAX(size(),CLTN_DEFAULT_CAPACITY));
	addContentsTo(cltn);
	return cltn;
}

static UNSIGNED bag_capacity;

Bag::Bag(istream& strm, Bag& where) : contents((strm >> bag_capacity, bag_capacity))
{
	this = &where;
	count = 0;
	UNSIGNED i,n;
	strm >> n;		// read bag size 
	while (n--) {
		strm >> i;
		addWithOccurrences(*readFrom(strm),i);
	}
}

void Bag::storer(ostream& strm)
{
	Object::storer(strm);
	strm << contents.capacity() << " " << contents.size() << " ";
	DO(contents,AssocInt*,a)
		strm << ((Integer*)(a->value()))->value();
		(a->key())->storeOn(strm);
	OD
}

 Bag::Bag(fileDescTy& fd, Bag& where) : contents((readBin(fd,bag_capacity), bag_capacity))
{
	this = &where;
	count = 0;
	UNSIGNED i,n;
	readBin(fd,n);
	while ( n-- ) {
		readBin(fd,i);
		addWithOccurrences(*readFrom(fd),i);
	}
}

void Bag::storer(fileDescTy& fd) 
{
	Object::storer(fd);
	storeBin(fd,contents.capacity());
	storeBin(fd,contents.size());
	DO(contents,AssocInt*,a)
		storeBin(fd,((Integer*)(a->value()))->value());
		(a->key())->storeOn(fd);
	OD
}
