/*
 * Fps: fast ps
 *
 * Description:
 *	Give a fast but very simple listing of existent processes.
 *	This is ps(1) with a lot of stuff exorcised. Eliminates
 *	much waiting around when the system's slow and all that
 *	is needed is a list of processes.
 *
 * Usage: fps [-acegrRux]
 * 	Options mean same as in ps(1), except 'u' option only
 *	adds symbolic userid to output. 'R' option lists
 *	runnable processes. The 'r' must be used in conjunction
 *	with a symbolic user id (as "r<uid>") and must appear
 *	either last or as a separate argument.
 *
 * Compile:
 *	cc -O -s -DUFILE=\"XXX\" -DNUID=nnnn fps.c -o fps
 *
 *	Note the backslashes to quote the ".
 *
 *	UFILE is the full pathname of a file containing
 *	the OUTPUT of the command:
 *
 *	awk -F: ' { print $3" "$1 } ' /etc/passwd
 *
 *	UFILE should be updated with above command every now 
 *	and then, preferably whenever /etc/passwd changes. The
 *	program might die mysteriously if new users are added
 *	and UFILE is not updated.
 *
 *	NUID should be somewhat larger than the largest
 *	numeric uid in /etc/passwd, say the smallest multiple 
 *	of 512 larger than it (or something like that).
 *
 * Author:
 *	Andrew V. Royappa
 *	Dept. of Computer Science
 *	Purdue University
 *	W. Lafayette, IN47906
 *
 * UUCP address:
 *	{allegra,ihnp4,ucbvax,decvax,pur-ee}!purdue!avr
 */

#include <stdio.h>
#include <ctype.h>
#include <nlist.h>
#include <pwd.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/vm.h>
#include <machine/pte.h>
#include <sys/mbuf.h>

struct nlist nl[] = {
	{ "_proc" },
#define	X_PROC		0
	{ "_Usrptmap" },
#define	X_USRPTMA	1
	{ "_usrpt" },
#define X_USRPT		2
	{ "_nswap" },
#define	X_NSWAP		3
	{ "_nproc" },
#define	X_NPROC		4
	{ "_dmmin" },
#define	X_DMMIN		5
	{ "_dmmax" },
#define	X_DMMAX		6
	{ "" },
};

struct	proc proc[8];		/* 8 = a few, for less syscalls */
struct	proc *mproc;

union {
	struct	user user;
	char	upages[UPAGES][NBPG];
} user;
#define u	user.user

int	aflg, cflg, eflg, gflg, Rflg, uflg, xflg;
char	*index, *rindex(), *strcpy(), *strcat(), *strncat();
long	lseek();
int	nswap;
struct	pte *Usrptma, *usrpt;
int	nproc;
int	dmmin, dmmax;

char	*kmemf, *memf, *swapf, *nlistf = "/vmunix";
int	kmem, mem, swap = -1;

int	argaddr;


#define NMAX 8		/* see <utmp.h> */

#ifdef PURDUE
#define NUID 1024
#define UFILE "/usr/avr/bin/.fps_data"
#endif

char unames[NUID+1][NMAX+1];
char *fmt;
char cmdbuf[CLSIZE*NBPG];

main(argc, argv)
	char **argv;
{
	register int i, j;
	register char *ap;
	int uid;
	time_t cpu;
	off_t procp;

	argc--, argv++;
	uid = getuid();
	while (argc > 0) {
		ap = argv[0];
		while (*ap) switch (*ap++) {
		case '-': break;
		case 'a':
			aflg++;
			break;
		case 'c':
			cflg = !cflg;
			break;
		case 'e':
			eflg++;
			break;
		case 'g':
			gflg++;
			break;
		case 'r':
			if (aflg) {
				fprintf(stderr,"fps: use only one of a,r.\n");
				exit(1);
			}
			uid = getpwnam(ap)->pw_uid;
			for ( ; *ap ; ++ap );
			break;
		case 'R':
			Rflg++;
			break;
		case 'u':
			uflg++;
			readnames();
			break;
		case 'x':
			xflg++;
			break;
		default:
			fprintf(stderr, "options: a, c, e, g, r, R, u, x.\n");
			exit(1);
		}
		argc--, argv++;
	}
	openfiles();
	getkvars();
	printf(uflg ? "USER\t PID\t  TIME\t\tCOMMAND\n" :
		      "PID\t  TIME\t\tCOMMAND\n");
	fmt = uflg ? "%-9s%d\t%3ld:%02ld\t\t%s\n" :
		     "%d\t%3ld:%02ld\t\t%s\n"; /* 9 = NMAX + 1 */
	procp = getw(nl[X_PROC].n_value);
	nproc = getw(nl[X_NPROC].n_value);
	for (i=0; i<nproc; i += 8) {
		klseek(kmem, (long)procp, 0);
		j = nproc - i;
		if (j > 8)
			j = 8;
		j *= sizeof (struct proc);
		if (read(kmem, (char *)proc, j) != j) {
			cantread("proc table", kmemf);
			exit(1);
		}
		procp += j;
		for (j = j / sizeof (struct proc) - 1; j >= 0; j--) {
			mproc = &proc[j];
			if (uid != mproc->p_uid && aflg==0)
				continue;
			if (xflg == 0 && gflg == 0 &&
			    mproc->p_ppid == 1)
				continue;
			if ((mproc->p_pgrp == 0 && xflg == 0) ||
			    mproc->p_pid <= 2 || mproc->p_stat == 0)
				continue;
			if (Rflg && mproc->p_stat != SRUN &&
			    mproc->p_stat != SWAIT &&
			    mproc->p_stat != SIDL)
				continue;
			if (mproc->p_stat != SZOMB && getu() == 0)
				continue;
			getcmd();
			cpu = u.u_ru.ru_utime.tv_sec + u.u_ru.ru_stime.tv_sec;
			if (uflg)
				printf(fmt, unames[mproc->p_uid],
					    mproc->p_pid,
					    cpu / 60, cpu % 60,
					    cmdbuf);
			else
				printf(fmt, mproc->p_pid,
					    cpu / 60, cpu % 60,
					    cmdbuf);
		}
	}
	exit(0);
}

getw(loc)
	unsigned long loc;
{
	long word;

	klseek(kmem, (long)loc, 0);
	if (read(kmem, (char *)&word, sizeof (word)) != sizeof (word))
		printf("error reading kmem at %x\n", loc);
	return (word);
}

klseek(fd, loc, off)
	int fd;
	long loc;
	int off;
{
	(void) lseek(fd, (long)loc, off);
}

openfiles(argc, argv)
	char **argv;
{

	kmemf = "/dev/kmem";
	kmem = open(kmemf, 0);
	if (kmem < 0) {
		perror(kmemf);
		exit(1);
	}
	memf = "/dev/mem";
	mem = open(memf, 0);
	if (mem < 0) {
		perror(memf);
		exit(1);
	}
	swapf = "/dev/drum";
	swap = open(swapf, 0);
	if (swap < 0) {
		perror(swapf);
		exit(1);
	}
}

getkvars(argc, argv)
	char **argv;
{
	register struct nlist *nlp;
#ifndef PURDUE
	nlist(nlistf, nl);
#endif
	if (nl[0].n_type == 0) {
		fprintf(stderr, "%s: No namelist\n", nlistf);
		exit(1);
	}
	usrpt = (struct pte *)nl[X_USRPT].n_value;	/* don't clear!! */
	Usrptma = (struct pte *)nl[X_USRPTMA].n_value;
	klseek(kmem, (long)nl[X_NSWAP].n_value, 0);
	if (read(kmem, (char *)&nswap, sizeof (nswap)) != sizeof (nswap)) {
		cantread("nswap", kmemf);
		exit(1);
	}
	dmmin = getw(nl[X_DMMIN].n_value);
	dmmax = getw(nl[X_DMMAX].n_value);
}

cantread(what, fromwhat)
	char *what, *fromwhat;
{

	fprintf(stderr, "fps: error reading %s from %s\n", what, fromwhat);
}


getu()
{
	struct pte *pteaddr, apte;
	struct pte arguutl[UPAGES+CLSIZE];
	register int i;
	int ncl, size;

	size = sizeof (struct user);
	if ((mproc->p_flag & SLOAD) == 0) {
		if (swap < 0)
			return (0);
		(void) lseek(swap, (long)dtob(mproc->p_swaddr), 0);
		if (read(swap, (char *)&user.user, size) != size) {
			fprintf(stderr, "fps: cant read u for pid %d from %s\n",
			    mproc->p_pid, swapf);
			return (0);
		}
		argaddr = 0;
		return (1);
	}
	pteaddr = &Usrptma[btokmx(mproc->p_p0br) + mproc->p_szpt - 1];
	klseek(kmem, (long)pteaddr, 0);
	if (read(kmem, (char *)&apte, sizeof(apte)) != sizeof(apte)) {
		printf("fps: cant read indir pte to get u for pid %d from %s\n",
		    mproc->p_pid, swapf);
		return (0);
	}
	klseek(mem,
	    (long)ctob(apte.pg_pfnum+1) - (UPAGES+CLSIZE) * sizeof (struct pte),
		0);
	if (read(mem, (char *)arguutl, sizeof(arguutl)) != sizeof(arguutl)) {
		printf("fps: cant read page table for u of pid %d from %s\n",
		    mproc->p_pid, kmemf);
		return (0);
	}
	if (arguutl[0].pg_fod == 0 && arguutl[0].pg_pfnum)
		argaddr = ctob(arguutl[0].pg_pfnum);
	else
		argaddr = 0;
	ncl = (size + NBPG*CLSIZE - 1) / (NBPG*CLSIZE);
	while (--ncl >= 0) {
		i = ncl * CLSIZE;
		klseek(mem, (long)ctob(arguutl[CLSIZE+i].pg_pfnum), 0);
		if (read(mem, user.upages[i], CLSIZE*NBPG) != CLSIZE*NBPG) {
			printf("fps: cant read page %d of u of pid %d from %s\n",
			    arguutl[CLSIZE+i].pg_pfnum, mproc->p_pid, memf);
			return(0);
		}
	}
	return (1);
}

getcmd()
{
	union {
		char	argc[CLSIZE*NBPG];
		int	argi[CLSIZE*NBPG/sizeof (int)];
	} argspac;
	register char *cp;
	register int *ip;
	char c;
	int nbad;
	struct dblock db;
	char *file;

	if (mproc->p_stat == SZOMB || mproc->p_flag&SWEXIT) {
		strcpy(cmdbuf, "<zombie|exiting>");
		return;
	}
	if (cflg) {
		(void) strncpy(cmdbuf, u.u_comm, sizeof (u.u_comm));
		return;
	}
	if ((mproc->p_flag & SLOAD) == 0 || argaddr == 0) {
		if (swap < 0)
			goto retucomm;
		vstodb(0, CLSIZE, &u.u_smap, &db, 1);
		(void) lseek(swap, (long)dtob(db.db_base), 0);
		if (read(swap, (char *)&argspac, sizeof(argspac))
		    != sizeof(argspac))
			goto bad;
		file = swapf;
	} else {
		klseek(mem, (long)argaddr, 0);
		if (read(mem, (char *)&argspac, sizeof (argspac))
		    != sizeof (argspac))
			goto bad;
		file = memf;
	}
	ip = &argspac.argi[CLSIZE*NBPG/sizeof (int)];
	ip -= 2;		/* last arg word and .long 0 */
	while (*--ip)
		if (ip == argspac.argi)
			goto retucomm;
	*(char *)ip = ' ';
	ip++;
	nbad = 0;
	for (cp = (char *)ip; cp < &argspac.argc[CLSIZE*NBPG]; cp++) {
		c = *cp & 0177;
		if (c == 0)
			*cp = ' ';
		else if (c < ' ' || c > 0176) {
			if (++nbad >= 5*(eflg+1)) {
				*cp++ = ' ';
				break;
			}
			*cp = '?';
		} else if (eflg == 0 && c == '=') {
			while (*--cp != ' ')
				if (cp <= (char *)ip)
					break;
			break;
		}
	}
	*cp = 0;
	while (*--cp == ' ')
		*cp = 0;
	cp = (char *)ip;
	(void) strncpy(cmdbuf, cp, &argspac.argc[CLSIZE*NBPG] - cp);
	if (cp[0] == '-' || cp[0] == '?' || cp[0] <= ' ') {
		(void) strcat(cmdbuf, " (");
		(void) strncat(cmdbuf, u.u_comm, sizeof(u.u_comm));
		(void) strcat(cmdbuf, ")");
	}
	return;

bad:
	fprintf(stderr, "fps: error locating command name for pid %d from %s\n",
	    mproc->p_pid, file);
retucomm:
	(void) strcpy(cmdbuf, " (");
	(void) strncat(cmdbuf, u.u_comm, sizeof (u.u_comm));
	(void) strcat(cmdbuf, ")");
	return;
}


/*
 * Given a base/size pair in virtual swap area,
 * return a physical base/size pair which is the
 * (largest) initial, physically contiguous block.
 */
vstodb(vsbase, vssize, dmp, dbp, rev)
	register int vsbase;
	int vssize;
	struct dmap *dmp;
	register struct dblock *dbp;
{
	register int blk = dmmin;
	register swblk_t *ip = dmp->dm_map;

	vsbase = ctod(vsbase);
	vssize = ctod(vssize);
	if (vsbase < 0 || vsbase + vssize > dmp->dm_size)
		panic("vstodb");
	while (vsbase >= blk) {
		vsbase -= blk;
		if (blk < dmmax)
			blk *= 2;
		ip++;
	}
	if (*ip <= 0 || *ip + blk > nswap)
		panic("vstodb *ip");
	dbp->db_size = min(vssize, blk - vsbase);
	dbp->db_base = *ip + (rev ? blk - (vsbase + dbp->db_size) : vsbase);
}

/*ARGSUSED*/
panic(cp)
	char *cp;
{

#ifdef DEBUG
	printf("%s\n", cp);
#endif
}

min(a, b)
{

	return (a < b ? a : b);
}

/*
 * readnames: (numeric uid) -> (symbolic uid) mapping
 * 
 * We come here if the 'u' option is used. First we read
 * UFILE, which consists of lines of the form
 * 	<numeric uid><space><symbolic uid>
 * Then readnames essentially does
 * 	unames[<numeric uid>] = <symbolic uid>
 * so when a numeric uid is seen later on we can instantly map 
 * it to the symbolic one with a single array reference - thus
 * bypassing getpwent. we essentially create a hash table using
 * hash function = identity. So unames[] must have at least 
 * (largest numeric uid + 1) entries. Note that most of unames[]
 * is wasted.
 */

readnames()
{
	register FILE *fp;
	register int  ind;
	register char *c1;
	register char *c2;
	char buf[NMAX+8];

	if ((fp = fopen(UFILE, "r")) == NULL)
		fprintf(stderr, "couldn't open %s.\n", UFILE), exit(1);

/* since format of each line is known and simple, don't use sscanf */

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		ind = 0;
		for (c1 = buf; *c1 != ' '; c1++)
			ind = 10 * ind + *c1 - '0';
		for (c2 = unames[ind]; *++c1 != '\n'; )
			*c2++ = *c1;
		*c2 = '\0';
	}
}
