#
#include <stty.h>

#define	DUMPCHAR	'\0'	/* Character which forces flushing of plot buffer */
#define	VECLEN		200	/* Maximum number of bytes/string */
#define	FLAG		'-'	/* Parameter which signals that what follows are flags */
#define	ERROR		'e'	/* Parameter flag to signal no error message */
#define	SLOW		's'	/* Parameter flag to signal slow mode */
#define	DEBUG		'd'	/* Parameter flag to signal debug mode */
#define	MSINK		2	/* Error output unit for writting */
#define	PAUSE		2	/* Number of seconds between screen erase and plot */
#define	MAXX		1023	/* Maximum x co-ordinate size */
#define	MAXY		1023	/* Maximum y co-ordinate size */
#define	TAGHIY		| 040	/* Add high y tag bits */
#define	TAGLOWY		| 0140	/* Add low y tag bits */
#define	TAGHIX		| 040	/* Add high x tag bits */
#define	TAGLOWX		| 0100	/* Add low x tag bits */
#define	TOALPHA		"\033:"	/* Return to Alphanumeric Mode character */
#define	TOGRAPH		"\033;"	/* Go into Graphics Mode character */
#define	PENDOWN		"\033[1s"	/* lower the pen */
#define	PENUP		"\033[0s"	/* raise the pen */
#define	GS		035	/* Go into Graphics Mode character */
#define	CAN		030	/* Erase screen character */
#define	ERASE		"\033[10;2048;2048p\033[9s"	/* Erase screen sequence */
#define	HOME		0214	/* Home cursor character */
#define	COLOR		033	/* color shift code */
#define	ENDC		-1	/* End of Graphics Mode record character */
#define	ESC		033
#define	LINE		19	/* DC3 */
char	group[5], last[5];
char	tx4662		0;	/* Flag to tell program if output is going to 4662 plotter */
char	gs		0;	/* Flag initial 4 bytes after GS received */
char	slow		1;	/* If slow=1, only transmit bytes which change */
char	debug		0;	/* Debug mode switch */
char	error		0;	/* Error message switch */
char	bflg		0;	/* don't buffer */
int	moving =	1;	/* after pen is raised */
int	finish();
int	attn();
int	tty[3], oldtty[3];	/* Stty buffers */
char	vector[ VECLEN ];	/* Output buffer */
char	*v		vector;	/* Pointer to output buffer */
int	nv		0;	/* number of elements in output buffer */
char plot		0;	/* Flag to signal receiving of a GS character */
int	x, y;

main(argc, argv) int argc; 
char **argv;
{
	register char *c;
	extern int fout, fin;
	int	i;
/* 
 * Set up attention interrupt return code.
*/
	signal(2,&attn);
	signal(15,&attn);

	fout = dup(1);
	gtty(fout,tty);
	oldtty[0] = tty[0];
	oldtty[1] = tty[1];
	oldtty[2] = tty[2];
	tty[2] = (tty[2]&ANYP) | RAW;	/* keep parity etc. */
	/*
	Check to see if any parameters passed to the program.
	If there are, process them.  If not, ignore and proceed.
	*/
	c = group;
	if ( (argc > 1) && (argv[1][0] == FLAG) )
		for ( i=1; (*c=argv[1][i]); i++)
			if ( *c == SLOW ) slow = ! slow;
			else if ( *c == DEBUG ) debug = 1;
			else if ( *c == ERROR ) error = 1;
			else if ( *c == 'b' ) bflg = 1;
			else if ( *c == 'x') tty[2] = 0;	/* use normal mode */
	/*
	 * Parameters checked.  Initialize variables and continue.
	 */
	plot = 0;
	/*
	Loop around reading characters until an EOF is encountered.
	*/
	while ( (*c = getc(&fin)) != -1 )
	{
		if ( *c == GS )			/* Beginning of plot record character */
			sendgs();
		else if ( *c == ENDC ) 		/* End of plot record character */
			sendus();		/* send end of plot sequence */
		else if (*c == COLOR) {
			*c = getc(&fin);
		}
		else if ( *c == CAN )		/* Initialize screen */
		{
			outs( ERASE );	/* Erase screen */
			output( HOME );		/* Home cursor */
			output( DUMPCHAR );
			sleep( PAUSE );		/* Pause to allow screen to erase */
		}
		else if ( plot ) {
			get();		/* get rest of data */
			conv();
			send();		/* If graphics mode, convert to plot characters */
		}
		else if ( !tx4662 ) output( *c );	/* If all else fails, simply pass the character on */
	}
	finish(0);
}

get()
{
/*
 * get next three characters from input stream and convert to x,y co-ordindates
 */
	register int i;
	register char *g, *l;
	g = &group[1];
	for ( i = 0; i < 3; i++)
		*g++ = getchar();	/* Get plot co-ords */
}

conv()
{
/*
Convert input graphics records to 4010-style graphics records.
The input format is:
	1) a GS record
	2) An arbitrary number of plot co-ords consisiting of:
		a) hi x co-ord.
		b) lo x co-ord.
		c) hi y co-ord.
		d) lo y co-ord.
		with all co-ords preceeded by a "tag bit".
	3) A final byte of 0377 ending this record.
 
The output format is:
	1) A GS record.
	2) An arbitrary number of plot co-ords consisting of:
		a) hi y co-ord
		b) lo y co-ord
		c) hi x co-ord
		d) lo x co-ord
		with all co-ords preceeded by 2 "tag bits" as defined by the
		#define TAG? statments.
	3) The final 0377 is converted to a US character.
*/
	register char *g;

	g = group;
	x = ((*(g++) & 077) << 6) + (*(g++) & 077);
	y = ((*(g++) & 077) << 6) + (*g & 077);
	if ( debug ) writef(MSINK,"x=%d  y=%d\n", x, y);
	if ( (x > MAXX) || (y > MAXY) )
		if ( error ) writef(MSINK,"** 4010 plot error\n\tCoordinate too big: x=%.2f  y=%.2f\n\tValue truncated\n", (x-100)/100., (y-100)/100.);
}

send()
{
/*
 * Routine to convert strings of graphic data to TX4010 input and send it.
 * input is already in "group".
*/
	register int i;
	register char *g, *l;

	/*
	if the tx4662 flag is on, conversion of the numbers into tx4010
	style characters is repressed.  At this point, print out the
	raw numbers, and return.
	*/
	if ( tx4662 )
		{
		printf("%d %d ",x,y);
		return( 0 );
		}
	g = group;
	*g++ = ((y >> 5) & 037) TAGHIY;		/* Hi y co-ord. */
	*g++ = (y & 037) TAGLOWY;		/* Lo y co-ord */
	*g++ = ((x >> 5) & 037) TAGHIX;		/* Hi x co-ord */
	*g = (x & 037) TAGLOWX;			/* Lo x co-ord */

	g = group;
	if ( !slow || gs )
		for ( i = 0; i < 4; i++)
			output(*g++);			/* Output 4010-style co-ords. */
	else
	{
		l = last;
		if ( *g != *l++ ) output( *g );		/* First bytes differ */
		if ( (*(++g) != *(l++)) || (*(g+1) != *l) )
			output( *g );				/* Second or third bytes differ */
		if ( *(++g) != *(l++) ) output( *g );	/* Third bytes differ */
		output( *(++g) );			/* Always print fourth byte */
	}
	g = group;
	l = last;
	for ( i=0; i<4; i++) *l++ = *g++;	/* Save old byte values */
	if (moving)
		{
		outs(PENDOWN);
		moving = 0;
		}
	gs = 0;
}

output( ch ) char ch;
/*
  Output routine to do buffered output on unit FOUT.  Stores VECLEN
  characters into array "vector", and does a wholesale write when it
  is full.  Before writting plot data, it puts the output device into
  "raw" mode so no character translation is done.  If a null character
  is passed, it is assumed that the output vector is to be flushed.
*/
{
	extern int fout;
	if ( (ch == DUMPCHAR) || (nv >= VECLEN) )
	{
		stty(fout, tty);
		write(fout, vector, nv);
		stty(fout, oldtty);
		v = vector;
		nv = 0;
#ifdef NONULLS
		if (ch == DUMPCHAR)
			return;
#endif
	}
	*v++ = ch;
	++nv;
}

finish(n)
{
	extern int fout;

	home();
	if ( !tx4662 )
		output( DUMPCHAR );
	else printf("%d ",-5);
	flush();
	stty(fout,oldtty);
	exit(n);
}

home()
{
/*
 * position at top of screen.
 */
	x = 0; y = 767;		/* top left corner */
	sendgs();
	send();
	sendus();		/* back to alpha mode */
}

sendgs()
{
	outs( TOGRAPH );	/* Echo the GS */
	plot = 1;			/* Flag plotting mode ON */
	gs = 1;				/* Initial 4 byte flag set */
}

sendus()
{
	outs( TOALPHA );	/* Return to Alpha mode */
	outs( PENUP );		/* raise the pen*/
	if (bflg)
		output( DUMPCHAR );
	plot = 0;			/* Flag plotting OFF */
	moving = 1;
}

attn()
{
/* interrupted from keyboard or kill command */
finish(1);
}

outs(ptr) register char *ptr;
{
while (*ptr)
	output(*ptr++);
}
