/*****************************************************\
*                                                     *
*             PTEXTBOX customized control             *
*                    Version 1.01                     *
*    Designed for Gerard Vermeulen's Freeware SDK     *
*        Copyright (C)1999 Charles "Randy" Gill       *
*           Released as licenced freeware             *
*                                                     *
\*****************************************************/

/*********************************************************************\
This control may be used as a replacement for PTEXTFIELD in many
instances.  At time this is being written, there is no implementation
of PTEXTFIELD in the GUI library.  This will only display one line of
text, but it does have many options for appearance.

This isn't a true custom control, but rather a major modification of
the PBUTTON class.

Because of the way the BankedAssign macros work, you will need to
assign its handlers in main() after CreateTextBox(), as below:

   BankedAssign1(SomeTextBox->draw, PTEXTBOX_draw);
   BankedAssign2(SomeTextBox->drawStatus, PTEXTBOX_drawStatus);
   BankedAssign2(SomeTextBox->handleEvent, PTEXTBOX_handleEvent);

\*********************************************************************/

#ifndef _PTEXTBOX_H_
#define _PTEXTBOX_H_

#ifndef _AVGUI_H_
#include <avgui.h>
#endif

/* PTEXTBOX types */

#define tbtNoBorder     0x00
#define tbtUnderline    0x01
#define tbtBoxBorder    0x03
#define tbtSolid        0x00
#define tbtDotted       0x04

#define MAX_TEXTBOX_LEN 0x40

typedef struct {
    PVIEW_PTR next;              
    PGROUP_PTR owner;
    unsigned char state;      
    unsigned char options;
    unsigned char left;
    unsigned char top;        
    unsigned char right;      
    unsigned char bottom;     
    void (*setState)();
    BANK_NO setStateBank;
    void (*draw)();
    BANK_NO drawBank;        
    void (*handleEvent)();
    BANK_NO handleEventBank; 
    char* text;
    BANK_NO reserved;
    unsigned int command;
    void (*drawStatus)();
    BANK_NO drawStatusBank;
    unsigned char boxType;
} PTEXTBOX;

void PTEXTBOX_draw(VOID_PTR thisView);
void PTEXTBOX_handleEvent(VOID_PTR thisView, unsigned int *evType,
                          unsigned char x, unsigned char y);
void PTEXTBOX_drawStatus(VOID_PTR thisView, unsigned char status);
VOID_PTR CreateTextBox(unsigned char ID, unsigned char left,
                       unsigned char top, unsigned char right,
                       unsigned char bottom, char *text, unsigned char type);

/*  Function Implementations  */

void PTEXTBOX_draw(VOID_PTR thisView)
{
   PTEXTBOX *TB;
   unsigned char oldFont;
   unsigned char lineStyle;
   unsigned char textX;
   unsigned char textY;
   unsigned char maxLen;

   TB = (PTEXTBOX *) thisView;
   oldFont = GetFontType();
   lineStyle = DRAW_BLACK;
   textX = TB->left + 1;
   textY = TB->top + 1;
   maxLen = TB->right - TB->left - 3;
   
   if (TB->boxType & tbtDotted)
   {
      lineStyle = DRAW_B1DOT;
   }
   if (TB->boxType & 0x02)
   {
      textX += 2;
      maxLen -= 2;
      DrawLine(TB->left, TB->top, TB->right, TB->top, lineStyle);
      DrawLine(TB->left, TB->top, TB->left, TB->bottom, lineStyle);
      DrawLine(TB->right, TB->top, TB->right, TB->bottom, lineStyle);
   }
   if (TB->boxType & tbtUnderline)
   {
      DrawLine(TB->left, TB->bottom, TB->right, TB->bottom, lineStyle);
   }
   SetFontType(PRPFONT11N);
   WriteAlignString(textX, textY, TB->text, maxLen, ALIGN_LEFT, 0);
   SetFontType(oldFont);
}

void PTEXTBOX_handleEvent(VOID_PTR thisView, unsigned int *evType,
                          unsigned char x, unsigned char y)
{
   PTEXTBOX *TB;
   TB = (PTEXTBOX *) thisView;
   if (*evType == evPenDown)
   {
      if (x >= TB->left && y >= TB->top && x <= TB->right && y <= TB->bottom)
      {
         ClearEvent(evType);
         TB->reserved = TB->reserved | 0x01;
      }
      else
      {
         TB->reserved = TB->reserved & 0xFE;
      }
   }

   if (*evType == evPenUp)
   {
      if ((TB->reserved & 0x01) && x >= TB->left && y >= TB->top && x <= TB->right && y <= TB->bottom) 
      {
         PenBeep();
         ExecPicker(TB->text, MAX_TEXTBOX_LEN);
         ClearEvent(evType);
         PutEvent(evCommand, HIGBYTE(cmRedraw), LOWBYTE(cmRedraw));
         TB->reserved = TB->reserved & 0xFE;
      }
      else
      {
         TB->reserved = TB->reserved & 0xFE;
      }
   }
}

void PTEXTBOX_drawStatus(VOID_PTR thisView, unsigned char status)
{
}

VOID_PTR CreateTextBox(unsigned char ID, unsigned char left,
                       unsigned char top, unsigned char right,
                       char *text, unsigned char type)
{
   PTEXTBOX *TB;
   unsigned char bottom;
   bottom = top + 12;
   TB = (PTEXTBOX *) CreateButton(ID, left, top, right, bottom,
      MK_FAR_PTR(text), 0x0000, bttBasic);
   TB->boxType = type;
   TB->reserved = 0;

   return (VOID_PTR) TB;
}

#endif
