#include <f_sdkapi.h>
#include <p100plib.h>
#include <ressdk.h>
#include <math.h>
#include <packet.h>
#include "about.h"
#include "memolist.h"
#include "buffview.h"
#include "memoutil.h"
#include "myapp.h"
#include "prefs.h"


/**********************************************************************
 *	memolist.c:
 *		Displays a deskbox containing a list of memos in the system.
 *		Calls the "viewer" to display the memo contained in buffer.
 **********************************************************************/

 /* To read categories */
extern BANKED short ReadCategoryInfo(BYTE type, CATEGORY_INFO* categoryInfo);


/* Category control */
#define cmCategoryEdit	0x1400
#define cmCategory		0x1500
#define cmCategoryAll	0x1600


/***************************************************************************
* Category info holder
****************************************************************************/
#define CAT_MENU_SIZE		CATEGORY_MAXNUM+2
static const char	catMenuLabel[] = "Choose a category:";
static MENUITEM 	catMenuText[CAT_MENU_SIZE];
static PCONTROLMENU	*catMenu;



/***************************************************************************
* Preference loading & saving
****************************************************************************/
static prefsRec myPrefs;	/* Hold the settings for this app */
static BYTE categoryID;		/* Current category that's set */
static bool filterCategory; /* TRUE when All not set */


/***************************************************************************
* Build Category menu
****************************************************************************/
static void buildCategoryMenu (MENUITEM *menu, unsigned char *defaultCategory);



/***************************************************************************
*	changeCategory:
*		Accept the new category setting and do all processing to update screen
****************************************************************************/
static void changeCategory (unsigned char newCategory, unsigned char newFilter, PVIEW *thisView)
{
	short currentID;

	if ((newCategory != categoryID) || (newFilter != filterCategory))
	{
		/* update the settings */
		categoryID = newCategory;
		filterCategory = newFilter;
		deselectAllMemos();

		/* Get the new "top-of-screen" and starting memo number */
		top = 0;
		firstMemoInCategory = 0;
		for (currentID=0; (currentID < numberMemos); currentID++)
		{
			if (!filterCategory || (memoList[currentID].category == categoryID))
			{
				/* recID of first memo to be shown */
				firstMemoInCategory = currentID;
				top = currentID;
				break;
			}
		}
		/* Update / redraw */
		refreshScreen(thisView);
	}

}



/**********************************************************************************
*
* Deskbox event handler.
* Call the default event handler and then process event commands (evCommand).
*
**********************************************************************************/
void mList_HandleEvent(VOID_PTR thisView, unsigned short *evType, unsigned char x, unsigned char y)
{
	PDESKBOX *dsk;
	unsigned char myCategory;
	unsigned char myFilter;
  	unsigned short myCommand; /* "cmXXX" command code */


	PDESKBOX_handleEvent(thisView, evType, x, y);

	if(*evType == evCommand)
	{
		myCommand = (TOWORD(x,y));
		if ((myCommand >= cmCategory) && (myCommand <= cmCategoryAll))
		{
			/* User has chosen to view a different category */
			/* Update the current category settings */
			myCategory = myCommand - cmCategory;
			myFilter = (myCommand != cmCategoryAll);
			changeCategory(myCategory, myFilter, (PVIEW*)thisView);
			ClearEvent(evType);
			return;
		}
		switch (myCommand)
		{
            case cmCancel:
				ClearEvent(evType);
				break;
      }
	}
}


/***************************************************************************
* Build Category menu
****************************************************************************/
static void buildCategoryMenu (MENUITEM *menu, unsigned char *defaultCategory)
{
	unsigned char i;
	unsigned char thisCat;
	unsigned short totalCats = 0;
	CATEGORY_INFO	myCats;

	/* Set category commands & flag status */
	for (i=0; i<(CAT_MENU_SIZE); i++)
	{
		menu[i].item[0] = 0; /* null */
		menu[i].flag = 0;
		menu[i].command = 0;
	};

	/* Read the categories from the Memo app */
	totalCats = ReadCategoryInfo(TYPE_MEMO, &myCats);

	/* First entry, "All", is always there */
	i = 0; /* Index into the category menu we build */
	strcpy(menu[i].item, "All");
	menu[i].flag = lstDisplay | lstUnderLine;
	menu[i].command = cmCategoryAll;

 	if (totalCats > 0)
	{
		for (thisCat = 0; thisCat<totalCats; thisCat++)
		{
			/* Only use it if deleted bit is NOT set */
			if ( !(myCats.cate[thisCat].status & 0x20) )
			{
				i++;
				strcpy(menu[i].item, myCats.cate[thisCat].name);
				menu[i].command = cmCategory + myCats.cate[thisCat].id;
				if (myCats.cate[thisCat].status & 0x01)
				{
					menu[i].flag = lstDisplay;
				}
				else
				{
					menu[i].flag = lstDisplay | lstBoldFont;
				}
			}
		}
		/* Now underline the last entry */
/* no need		menu[i].flag |= lstUnderLine; */
	}
	/* Now that we are done, add the "edit" command */
/* no need
	i++;
	strcpy(menu[i].item, "Edit Categories...");
	menu[i].flag = 0;
	menu[i].command = cmCategoryEdit;
*/

	/* Find the default menu item to select */
	if (!filterCategory)
	{
		/* "All" is the first entry */
		*defaultCategory = 0;
	}
	else
	{
		/* Find the corresponding entry - if one does not exist then
			destroy the saved categoryID which is apparently invalid */
		for (i=1; i<CAT_MENU_SIZE; i++)
		{
			if ((menu[i].command - cmCategory) == categoryID)
			{
				*defaultCategory = i;
				break;
			}
		};
		if (i == CAT_MENU_SIZE)
		{
			/* Trash it - turn off filter */
			*defaultCategory = 0;
			categoryID = 0;
			filterCategory = FALSE;
		}
	}

};


/********************************************************************************
* Create a deskbox
********************************************************************************/

far VOID_PTR createMemoListDeskbox(void)
{
	VOID_PTR retval;
	PDESKBOX *dsk;
	
	CONST_FAR_PTR fp1;
	CONST_FAR_PTR fpMenu;
	short dummy;
	unsigned char defaultCategory;

	fpMenu = MK_FAR_PTR(myMenu);

	/* Create stuff using current setting of portrait/landscape mode */
	ResourceRotate (landscapeMode);		/* set new dir */
  	SetDisplayDirection (landscapeMode);	/* set new dir */

	fp1 = MK_FAR_PTR(myTitle);
	dsk = (PDESKBOX *) CreateDeskBox('D', 0, 0, maxX[landscapeMode], maxY[landscapeMode], fp1, fpMenu, 0, bafClose|bafDotTitle|bafComLine|bafPlace);
    dsk->options &= ~ofFindable;

	BankedAssign(dsk->handleEvent, mList_HandleEvent);
	BankedAssign(dsk->draw, mList_Draw);


	/* The "Category" menu */
	memset (&catMenuText, 0, sizeof (catMenuText));
    buildCategoryMenu  (catMenuText, &defaultCategory);
    fpMenu	= MK_FAR_PTR(catMenuText);
    fp1 	= MK_FAR_PTR(catMenuLabel);
    /* magicDefault gets set in buildCategory */
    catMenu = (PCONTROLMENU *) CreateControlMenu ('E', 0, 0, 0, 0, fpMenu, fp1, defaultCategory, cmtPrimaryFull); 
    LoadBank(&(dsk->insert));
    dsk->insert((VOID_PTR)dsk, (PVIEW *)catMenu);

    fp1 = MK_FAR_PTR(myTitle);
	btn		= (PBUTTON *)InsertButton(dsk, 'F', fp1, cmRotate, bttRotate, 0);

	fp1 = MK_FAR_PTR(txtFont);
	btnFont	= (PBUTTON *)InsertButton(dsk, 'G', fp1, cmFont, bttBasic, 1);

	fp1 = MK_FAR_PTR(txtDelete);
	btnDelete = (PBUTTON *)InsertButton(dsk, 'H', fp1, cmDelete, bttBasic, 1);

    pgML = (PPAGEARROW *)InsertPageArrow(dsk, 'I', (PVIEW *)dsk);
    DisableCommand (cmPgUp);
	DisableCommand (cmPgDn);
	DisableCommand (cmDelete);

    retval = dsk;
	return retval;

}


/*****************************************************************************
*
* Create a desbox, execute (display) it and then destroy it.
*
*****************************************************************************/
unsigned short listMemos()
{
	PDESKBOX *dsk;
    unsigned short retval;

    /* Get current application settings */
    getPrefs (&myPrefs);
    landscapeMode	= myPrefs.landscapeList;
    currentFont		= myPrefs.fontList;
    categoryID		= myPrefs.categoryID;
    filterCategory	= myPrefs.filterCategory;

    /* Initialize our data structures */
    updateGeometry();
    buildMemoList();

	dsk = (PDESKBOX *) createMemoListDeskbox();
	retval = ExecView ((PGROUP *) dsk);

	Destroy ((VOID_PTR) dsk);

    /* Save application settings currently in use */
    getPrefs (&myPrefs);	/* Read current settings, then alter */
    myPrefs.landscapeList	= landscapeMode;
    myPrefs.fontList		= currentFont;
    myPrefs.categoryID		= categoryID;
    myPrefs.filterCategory	= filterCategory;
	putPrefs (&myPrefs);	/* Save current settings */
		
	return retval;
}

