/*
** File:		AviGlyph.c
** Author:		Randal M. Bates
** Date:		October 15, 1998
** Copyright:	October 15, 1998
** Description:	This file provides all of the functionality for Glyph (v.2) input and
**				translation.
**
** API:
** Modifications:
** 10/26/98	Bates	Added comments before sending the code to Ollivier Civiol for
**					inclusion in his project.
*/


#include "p100plib.h"       
#include "ressdk.h"
#include "math.h"
#include "packet.h"

/*
** Glyph SYLE and MODE constants
*/
#define glyphCHAR	1
#define glyphNUM	2
#define glyphSHIFT	1
#define glyphCAPS	2
#define glyphSPECIAL	3

/*
** The big mother GLYPH structure.  This houses all of the GLYPH globals, this
** (hopefully) avoiding collision with other globals of similar name(s)
*/
struct
{
  int	Style;		/* 0 = none, 1 = char, 2 = num                			*/
  int	Mode;		/* 0 = none, 1 = shift, 2 = caps, 3 = special 			*/
  int	PMode;		/* restore Mode when 'special' completed      			*/
  int	Dir;		/* 0 = portrait, 1 = landscape                			*/
  int	Roof;		/* top Y coord of the Glyph pad               			*/
  int	Floor;		/* botom Y of the Glyph pad                   			*/
  int	CharLeft;	/* left X of the Char Glyph Pad               			*/
  int	CharRight;	/* right X of the Char Glyph Pad              			*/
  int	NumLeft;	/* left X of the Num Glyph Pad                			*/
  int	NumRight;	/* right X of the Num Glyph Pad               			*/
  int	HC;			/* Y coord of the Horizontal Center line      			*/
  					/* values captured/computed during Glyph formation...	*/
  int	VVC, VVL, VHC, VHL;
  int	minX, minY, maxX, maxY, currX, currY;
  int	startX, startY, endX, endY;
  int	Count, Repeat, CrossedVVC, CrossedVHC;
  int	rx, ry, px, py, acl, bcl;
  int	UnderRoof, RTR, RTRFLOOR, CLOSURE;
  int	LC, FC, RC, TC;	/* left, floor, right, top */
} Glyph;



extern unsigned short StartRomBank;


/* ------------------------------------------------------------- */

/*
** Draw the Glyph Pads
*/
BANKED void ResetGlyphPad()
{
  unsigned char y, f, l, x;

  y = ((Glyph.Floor - Glyph.Roof) / 2) + Glyph.Roof;

  /*
  ** Erase the area
  */
  FillRect(Glyph.CharLeft,Glyph.Roof,Glyph.NumRight,Glyph.Floor,DRAW_WHITE);

  /*
  ** Draw the frames
  */
  DrawRect(Glyph.CharLeft,Glyph.Roof,Glyph.CharRight,Glyph.Floor,DRAW_BLACK);
  DrawRect(Glyph.NumLeft,Glyph.Roof,Glyph.NumRight,Glyph.Floor,DRAW_BLACK);

  f = GetFontType();
  SetFontType(PRPFONT7N);

  l = GetStringLength("ALPHA",PRPFONT7N);
  x = ((Glyph.CharRight - Glyph.CharLeft) - l) / 2;
  WriteString(Glyph.CharLeft+x,Glyph.Floor-8,"ALPHA",0);

  l = GetStringLength("NUM",PRPFONT7N);
  x = ((Glyph.NumRight - Glyph.NumLeft) - l) / 2;
  WriteString(Glyph.NumLeft+x,Glyph.Floor-8,"NUM",0);

  SetFontType(f);

  /*
  ** Char area shadow
  */
  DrawLine(Glyph.CharLeft+1, Glyph.Floor+1,
           Glyph.CharRight+1,Glyph.Floor+1,DRAW_BLACK);
  DrawLine(Glyph.CharRight+1,Glyph.Floor+1,
           Glyph.CharRight+1,Glyph.Roof+1,DRAW_BLACK);

  /*
  ** Num area shadow
  */
  DrawLine(Glyph.NumLeft+1, Glyph.Floor+1,
           Glyph.NumRight+1,Glyph.Floor+1,DRAW_BLACK);
  DrawLine(Glyph.NumRight+1,Glyph.Floor+1,
           Glyph.NumRight+1,Glyph.Roof+1,DRAW_BLACK);

  /*
  ** Mode area
  */
  DrawLine(Glyph.CharRight,Glyph.Roof+4,
           Glyph.NumLeft,  Glyph.Roof+4,DRAW_BLACK);
  DrawLine(Glyph.CharRight,Glyph.Roof+18,
           Glyph.NumLeft,  Glyph.Roof+18,DRAW_BLACK);
  DrawLine(Glyph.CharRight,Glyph.Roof+19,
           Glyph.NumLeft,  Glyph.Roof+19,DRAW_BLACK);

  /*
  ** Draw the horizontal rules
  */
  DrawLine(Glyph.CharLeft,y,Glyph.CharRight,y,DRAW_GRAY);
  DrawLine(Glyph.NumLeft,y,Glyph.NumRight,y,DRAW_GRAY);

  Glyph.Style = 0;

  switch (Glyph.Mode)
  {
    case glyphSHIFT:
         WriteChar(Glyph.CharRight+3,Glyph.Roof+5,'S',0);
         break;
    case glyphCAPS:
         WriteChar(Glyph.CharRight+3,Glyph.Roof+5,'C',0);
         break;
    case glyphSPECIAL:
         WriteChar(Glyph.CharRight+3,Glyph.Roof+7,'*',0);
         break;
  }
}

/*
** Select between O, T, +, U, 0 and V
*/
BANKED char Pick_OTUV(int NUM,int SHIFT,int TAIL,int SPEC)
{
  int	width, height;
  char	it;

  width = (Glyph.endX - Glyph.startX) + 4;
  height = (Glyph.endY - Glyph.startY) + 4;

  if (Glyph.minX < Glyph.startX  &&  Glyph.endX > Glyph.startX)
  {
    if (((Glyph.VVC - Glyph.startX) + 8) < 8)
      it = (NUM)? (Glyph.CLOSURE)? '8' : '+' : (SHIFT)? 'T' : 't';
    else if (Glyph.FC + Glyph.RC < 8)
      it = (SHIFT)? 'V' : 'v';
    else
      it = (SHIFT)? 'U' : 'u';
  }
  else if (width < 8  &&  height < 8)
  {
    it = (NUM)? '0' : (SHIFT)? 'O' : 'o';
  }
  else if (Glyph.FC + Glyph.RC < 8)
  {
    it = (SHIFT)? 'V' : 'v';
  }
  else
  {
    it = (SHIFT)? 'U' : 'u';
  }
  return(it);
}

/*
** Select a Glyph that started in the Upper-Left and ended in the Upper-Left
*/
BANKED char Pick_ULUL(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 0)
    it = (SPEC)? '\"' : '\'';
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 2)
    it = (NUM)? '%' : (SHIFT)? 'P' : 'p';
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 3)
    it = (NUM)? '%' : (SHIFT)? 'P' : 'p';
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 4)
    it = (NUM)? '%' : (SHIFT)? 'P' : 'p';
  else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 1)
    it = (NUM)? '0' : (SHIFT)? 'O' : 'o';

  return(it);
}

/*
** Select a Glyph that started in the Upper-Left and ended in the Upper-Right
*/
BANKED char Pick_ULUR(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 0)
  {
    it = (NUM)? (SPEC)? '=' : '-' : ' ';
  }
  else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 1)
  {
    if (Glyph.FC + Glyph.RC < 8)
      it = (NUM)? 0 : (SHIFT)? 'V' : 'v';
    else
      it = (NUM)? 0 : (SHIFT)? 'U' : 'u';
  }
  else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 2)
    it = '^';
  else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 3)
  {
    if (Glyph.acl > 0  &&  Glyph.bcl == 0)
      it = '~';
    else
      it = (NUM)? 0 : (SHIFT)? 'W' : 'w';
  }
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 1)
    it = Pick_OTUV(NUM,SHIFT,TAIL,SPEC);
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 3)
    it = (NUM)? 0 : (SHIFT)? 'W' : 'w';

  return(it);
}

/*
** Select a Glyph that started in the Upper-Left and ended in the Lower-Left
*/
BANKED char Pick_ULDL(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if (Glyph.CrossedVHC == 0)
  {
    switch (Glyph.CrossedVVC)
    {
      case 1:
           if (Glyph.acl > 0  &&  Glyph.bcl == 0)
             it = (SPEC)? '\"' : '\'';
           else if (NUM)
           {
             if (Glyph.RC < 5)
               it = (NUM)? (SPEC)? ']' : '>' : 0;
             else
               it = (NUM)? (SPEC)? '}' : ')' : (SHIFT)? 'D' : 'd';
           }
           else
             it = (SHIFT)? 'D' : 'd';
           break;
      case 3:
      case 5:
           it = (NUM)? '3' : (SHIFT)? 'B' : 'b';
           break;
    }
  }
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 0)
    it = (NUM)? '7' : 0;
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 2)
  {
    if (NUM)
      it = (TAIL)? '4' : '*';	/* also P, Y */
    else
      it = (TAIL)? (SHIFT)? 'Y' : 'y' : (SHIFT)? 'X' : 'x';
  }

  return(it);
}

/*
** Select a Glyph that started in the Upper-Left and ended in the Upper-Right
*/
BANKED char Pick_ULDR(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if (NUM)
  {
    if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 0)
      it = (TAIL)? '4' : '7';
    else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 2)
      it = (TAIL)? '4' : '#';		/* also R, Y */
    else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 0)
      it = (SPEC)? '%' : '2';
    else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 2)
      it = 0;
    else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 2)
      it = 0;
  }
  else
  {
    if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 0)
    {
      if (SHIFT)
        it = (TAIL)? 'Y' : ((Glyph.TC+Glyph.RC) > (Glyph.LC+Glyph.FC))? 'T' : 'L';
      else
        it = (TAIL)? 'y' : ((Glyph.TC+Glyph.RC) > (Glyph.LC+Glyph.FC))? 't' : 'l';
    }
    else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 2)
    {
      if (SHIFT)
        it = (TAIL)? 'Y' : 'H';
      else
        it = (TAIL)? 'y' : 'h';
    }
    else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 0)
      it = (SPEC)? '?' : (SHIFT)? 'Z' : 'z';
    else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 2)
      it = (SHIFT)? 'R' : 'r';
    else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 2)
      it = (SHIFT)? 'R' : 'r';
  }

  return(it);
}


/*
** Select a Glyph that started in the Upper-Rught and ended in the Upper-Left
*/
BANKED char Pick_URUL(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if ((Glyph.acl > 0  &&  Glyph.bcl == 0) || (Glyph.acl == 0  &&  Glyph.bcl > 0))
    it = 0x08;
  else if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 1)
    it = (NUM)? 0 : (SHIFT)? 'V' : 'v';
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 0)
  {
    if (NUM)
    {
      if (TAIL  &&  Glyph.CLOSURE)
        it = '9';
      else
        it = '6';
    }
    else
    {
      it = (Glyph.CLOSURE)? (SHIFT)? 'Q' : 'q' : (SHIFT)? 'G' : 'g';
    }
  }
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 1)
  {
    if (SHIFT)
      it = (Glyph.CLOSURE)? (NUM)? '0' : 'O' : (NUM)? '6' : 'G';
    else
      it = (Glyph.CLOSURE)? (NUM)? '0' : 'o' : (NUM)? '6' : 'g';
  }
  else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 1)
  {
    if (SHIFT)
      it = (Glyph.CLOSURE)? (NUM)? '0' : 'O' : (NUM)? '6' : 'G';
    else
      it = (Glyph.CLOSURE)? (NUM)? '0' : 'o' : (NUM)? '6' : 'g';
  }

  return(it);
}


/*
** Select a Glyph that started in the Upper-Rught and ended in the Upper-Right
*/
BANKED char Pick_URUR(int NUM,int SHIFT,int TAIL,int SPEC)
{
  char it = 0;

  if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 0)
  {
    if (Glyph.acl > 0  &&  Glyph.bcl == 0)
      it = (SPEC)? '"' : '`';
    else if (SHIFT)
      it = (Glyph.CLOSURE)? 'Q' : (NUM)? '6' : 'G';
    else
      it = (Glyph.CLOSURE)? 'q' : (NUM)? '6' : 'g';
  }
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 1)
    it = Pick_OTUV(NUM,SHIFT,TAIL,SPEC);
  else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 3)
    it = (NUM)? '+' : 0;
  else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 1)
    it = (NUM)? '8' : Pick_OTUV(NUM,SHIFT,TAIL,SPEC);

  return(it);
}


/*
** Parse the Glyph.  The resuling character will be placed on the event queue as
** event type 100, x=the char, y=0
*/
BANKED void ParseGlyph()
{
  char	it = 0;
  int	TAIL, NUM, SHIFT, SPEC;
  int	width, height;

  NUM = (Glyph.Style == glyphCHAR)? 0 : 1;
  width = Glyph.maxX - Glyph.minX;
  height = Glyph.maxY - Glyph.minY;

  SPEC = (Glyph.Mode == glyphSPECIAL)? 1 : 0;

  TAIL = 0;
  if (Glyph.RTR  &&  (Glyph.RTRFLOOR < Glyph.endY - 10))
    TAIL = 1;

  it = 0;

  SHIFT = (Glyph.Mode == glyphSHIFT  ||  Glyph.Mode == glyphCAPS)? 1 : 0;

  if (width < 10)
  {
    if (Glyph.acl == 0  &&  Glyph.bcl > 0)
      it = (SPEC)? ':' : '.';
    else if (Glyph.acl > 0  &&  Glyph.bcl == 0)
    {
      it = 0;
      if (Glyph.Mode == glyphSPECIAL)
      {
        Glyph.Mode = Glyph.PMode;
        Glyph.PMode = 0;
      }
      else
      {
        Glyph.PMode = Glyph.Mode;
        Glyph.Mode = glyphSPECIAL;
      }
    }
    else if (Glyph.startY < Glyph.HC  &&  Glyph.endY > Glyph.HC)
    {
      it = (NUM)? '1' : (SPEC)? '!' : (SHIFT)? 'I' : 'i';
    }
    else if (Glyph.startY > Glyph.HC  &&  Glyph.endY < Glyph.HC)
    {
      if (++Glyph.Mode > glyphCAPS)
      {
        Glyph.Mode = 0;
        glyphSHIFT = 0;
      }
    }
  }
  else if (Glyph.startY < Glyph.HC)
  {

    if (Glyph.startX < Glyph.VVC)
    {
      if (Glyph.endY < Glyph.HC)
      {
        if (Glyph.endX < Glyph.VVC)		/* up left up left */
        {
          it = Pick_ULUL(NUM,SHIFT,TAIL,SPEC);
        }
        else					/* up left up right */
        {
          it = Pick_ULUR(NUM,SHIFT,TAIL,SPEC);
        }
      }
      else
      {
        if (Glyph.endX < Glyph.VVC)		/* up left down left */
        {
          it = Pick_ULDL(NUM,SHIFT,TAIL,SPEC);
        }
        else					/* up left down right */
        {
          it = Pick_ULDR(NUM,SHIFT,TAIL,SPEC);
        }
      }
    }
    else
    {
      if (Glyph.endY < Glyph.HC)
      {
        if (Glyph.endX < Glyph.VVC)		/* up right up left */
        {
          it = Pick_URUL(NUM,SHIFT,TAIL,SPEC);
        }
        else					/* up right up right */
        {
          it = Pick_URUR(NUM,SHIFT,TAIL,SPEC);
        }
      }
      else
      {
        if (Glyph.endX < Glyph.VVC)		/* up right down left */
        {
          if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 0)
          {
            it = (NUM)? 0x0A : (SHIFT)? 'F' : 'f';
          }
          else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 2)
          {
            if (SHIFT)
              it = (Glyph.CLOSURE)? 'Q' : (NUM)? (TAIL)? '9' : '6' : 'G';
            else
              it = (Glyph.CLOSURE)? 'q' : (NUM)? (TAIL)? '9' : '6' : 'g';
          }
          else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 0)
            it = (SPEC)? '$' : (NUM)? '5' : (SHIFT)? 'S' : 's';
        }
        else		/* up right down right */
        {
          if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 0)
          {
            if (Glyph.LC + Glyph.FC < 9)
              it = (NUM)? (SPEC)? '[' : '<' : (SHIFT)? 'K' : 'k';
            else if (Glyph.acl > 0  &&  Glyph.bcl == 0)
              it = '`';
            else
              it = (NUM)? (SPEC)? '{' : '(' : (SHIFT)? 'C' : 'c';
          }
          else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 1)
            it = (NUM)? '9' : 0;
          else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 2)
          {
            if (SHIFT)
              it = (Glyph.CLOSURE)? (NUM)? '9' : 'Q' : (NUM)? '6' : 'G';
            else
              it = (Glyph.CLOSURE)? (NUM)? '9' : 'q' : (NUM)? '6' : 'g';
          }
          else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 0)
            it = (NUM)? 0 : (SHIFT)? 'E' : 'e';
          else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 0)
            it = (NUM)? 0 : (SHIFT)? 'E' : 'e';
        }
      }
    }
    Glyph.Mode = (Glyph.Mode == glyphSHIFT)? 0 : Glyph.Mode;
  }
  else
  {
    if (Glyph.startX < Glyph.VVC)
    {
      if (Glyph.endY < Glyph.HC)
      {
        if (Glyph.endX < Glyph.VVC)		/* down left up left */
        {
          if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 2)
            it = (NUM)? '%' : (SHIFT)? 'P' : 'p';
        }
        else		/* down left up right */
        {
          if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 1)
            it = (NUM)? '/' : (SHIFT)? 'J' : 'j';
          else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 3)
            it = (NUM)? 0 : (SHIFT)? 'N' : 'n';
        }
      }
      else
      {
        if (Glyph.endX < Glyph.VVC)		/* down left down left */
        {
          if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 2)
            it = (NUM)? 0 : (SHIFT)? 'D' : 'd';
          else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 2)
            it = (NUM)? 0 : (SHIFT)? 'B' : 'b';
          else if (Glyph.CrossedVVC == 3  &&  Glyph.CrossedVHC == 4)
            it = '*';
        }
        else		/* down left down right */
        {
          if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 0)
            it = (NUM)? 0 : '_';
          else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 2)
            it = (NUM)? 0 : (SPEC)? '@' : (SHIFT)? 'A' : 'a';
          else if (Glyph.CrossedVVC == 0  &&  Glyph.CrossedVHC == 4)
            it = (NUM)? 0 : (SHIFT)? 'M' : 'm';
          else if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 2)
            it = (NUM)? 0 : (SHIFT)? 'R' : 'r';
          else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 2)
            it = '&';
          else if (Glyph.CrossedVVC == 4  &&  Glyph.CrossedVHC == 4)
            it = '&';
        }
      }
    }
    else
    {
      if (Glyph.endY < Glyph.HC)
      {
        if (Glyph.endX < Glyph.VVC)		/* down right up left */
        {
          it = ' ';
        }
        else		/* down right up right */
        {
          if (Glyph.CrossedVVC == 2  &&  Glyph.CrossedVHC == 1)
            it = (NUM)? (SPEC)? '[' : '<' : (SHIFT)? 'K' : 'k';
        }
      }
      else
      {
        if (Glyph.endX < Glyph.VVC)		/* down right down left */
        {
          if (Glyph.CrossedVVC == 1  &&  Glyph.CrossedVHC == 0)
          {
            if (Glyph.acl == 0  &&  Glyph.bcl > 0)
              it = (SPEC)? ';' : ',';
          }
        }
        else		/* down right down right */
        {
          it = ' ';
        }
      }
    }
    Glyph.Mode = (Glyph.Mode == glyphSHIFT)? 0 : Glyph.Mode;
  }

  if (it)
  {
    /*
    ** Send the character to the event queue to be printed.
    */
    PutEvent(100,it,0);
    if (Glyph.Mode == glyphSPECIAL)
    {
      Glyph.Mode = Glyph.PMode;
      Glyph.PMode = 0;
    }
  }
}

BANKED void StepShift()
{
  if ((++Glyph.Mode) > glyphSPECIAL)
    Glyph.Mode = 0;
  PenBeep();
  ResetGlyphPad();
}

/*
** Given the x and y of a pen event and 1=pen-down, 2=pen-move and 3=pen-up,
** process the event as needed...
*/
BANKED void DoGlyph(unsigned char x,unsigned char y,unsigned char pen)
{
  int width, height;

  /*
  ** Ignore movement & penup's if not glyphing
  */
  if (pen > 1  &&  Glyph.Style == 0)
    return;

  /*
  ** Ignore pen down if not within glyph area
  */
  if (pen == 1  &&  y < Glyph.Roof)
    return;

  /*
  ** Ensure that the Y coord is within the pad height
  */
  y = (y < Glyph.Roof)? Glyph.Roof : (y > Glyph.Floor)? Glyph.Floor : y;

  /*
  ** if it is a pendown event, decide which pad (if any) it happened in
  */
  if (pen == 1)
  {
    if (x >= Glyph.CharLeft  &&  x <= Glyph.CharRight)
      Glyph.Style = glyphCHAR;
    else if (x >= Glyph.NumLeft  &&  x <= Glyph.NumRight)
      Glyph.Style = glyphNUM;
    else if (x >= Glyph.CharRight && x <= Glyph.NumLeft)
      StepShift();
  }

  /*
  ** If not glyphing, just return
  */
  if (Glyph.Style == 0)
    return;

  /*
  ** Draw the pixel
  */
  DrawDot(x,y,DRAW_BLACK);
  Glyph.rx = x;
  Glyph.ry = y;

  /*
  ** Parse the pen event
  */
  switch (pen)
  {
    case 1: /* pen down */
         Glyph.minX       = x;
         Glyph.minY       = y;
         Glyph.maxX       = x;
         Glyph.maxY       = y;
         Glyph.startX     = x;
         Glyph.startY     = y;
         Glyph.currX      = x;
         Glyph.currY      = y;
         Glyph.CrossedVVC = 0;
         Glyph.CrossedVHC = 0;
         Glyph.VVC        = x;
         Glyph.VHC        = y;
         Glyph.VVL        = 0;
         Glyph.VHL        = 0;
         Glyph.Repeat     = 0;
         Glyph.Count      = 1;

         Glyph.UnderRoof  = 0;
         Glyph.RTR        = 0;
         Glyph.RTRFLOOR   = 0;
         Glyph.CLOSURE    = 0;
         Glyph.LC         = 0;
         Glyph.FC         = 0;
         Glyph.RC         = 0;
         Glyph.TC         = 0;
         Glyph.acl        = 0;
         Glyph.bcl        = 0;

         Glyph.px = Glyph.rx;
         Glyph.py = Glyph.ry;
         break;

    case 2: /* pen moved */
         /*
         ** If the pen moved far enough to move into a new fatbit pixel...
         */
         Glyph.px = Glyph.rx;
         Glyph.py = Glyph.ry;
         if (x != Glyph.currX  ||  y != Glyph.currY)
         {
           ++Glyph.Count;

           /*
           ** if we are on the same pixel row or column as the previous
           ** pixel then increment the repeat counter.  his is how we will
           ** select between similar characters like 'U' and 'V'
           */
           if (x == Glyph.currX  ||  y == Glyph.currY)
             ++Glyph.Repeat;

           if (y < Glyph.HC)
             ++Glyph.acl;
           else
             ++Glyph.bcl;

           /*
           ** manage the min & max boundaries
           */
           Glyph.minX = (x < Glyph.minX)? x : Glyph.minX;
           Glyph.minY = (y < Glyph.minY)? y : Glyph.minY;
           Glyph.maxX = (x > Glyph.maxX)? x : Glyph.maxX;
           Glyph.maxY = (y > Glyph.maxY)? y : Glyph.maxY;

           /*
           ** manage the edge counters
           */
           if ((x > Glyph.minX) && (x < Glyph.minX + 4))
             ++Glyph.LC;
           else if ((x < Glyph.maxX) && (x > Glyph.maxX - 4))
             ++Glyph.RC;
           if ((y < Glyph.maxY) && (y > Glyph.maxY - 4))
             ++Glyph.FC;
           else if ((y > Glyph.minY) && (y < Glyph.minY + 4))
             ++Glyph.TC;

           if (y > Glyph.minY + 4)
             Glyph.UnderRoof = 1;

           /*
           ** manage the RTR (return to roof) parameters
           */
           if (Glyph.UnderRoof  &&  (y <= Glyph.minY + 4))
           {
             if (Glyph.RTR == 0)
             {
               Glyph.RTR = 1;
               Glyph.RTRFLOOR = Glyph.maxY;
             }
             else if (!Glyph.CLOSURE)
             {
               width = (x - Glyph.startX) + 4;
               height = (y - Glyph.startY) + 4;
               Glyph.CLOSURE = (width < 8  &&  height < 8)? 1 : 0;
             }
           }

           /*
           ** Compute the new virtual virtical center and the
           ** new virtual horizontal center
           */
           Glyph.VVC = ((((Glyph.maxX - Glyph.minX) + 1) / 2) + Glyph.minX) - 1;
           Glyph.VHC = ((((Glyph.maxY - Glyph.minY) + 1) / 2) + Glyph.minY) - 1;

           /*
           ** test to see if the stroke moved across the
           ** virtual virtical center
           */
           if (x < (Glyph.VVC - 2))
           {
             if (Glyph.VVL == 1)
               ++Glyph.CrossedVVC;
             Glyph.VVL = -1;
           }
           else if (x > (Glyph.VVL + 2))
           {
             if (Glyph.VVL == -1)
               ++Glyph.CrossedVVC;
             Glyph.VVL = 1;
           }

           /*
           ** test to see if the stroke moved across the
           ** virtual horizontal center
           */
           if (y < (Glyph.VHC - 2))
           {
             if (Glyph.VHL == 1)
               ++Glyph.CrossedVHC;
             Glyph.VHL = -1;
           }
           else if (y > (Glyph.VHL + 2))
           {
             if (Glyph.VHL == -1)
               ++Glyph.CrossedVHC;
             Glyph.VHL = 1;
           }

           /*
           ** Note the current x & current y
           */
           Glyph.currX = x;
           Glyph.currY = y;
         }
         break;

    case 3: /* pen up */
         Glyph.endX = x;
         Glyph.endY = y;
         ParseGlyph();
         ResetGlyphPad();
         Glyph.px = 0;
         Glyph.py = 0;
         break;
  }
}

/*
** Initialize the Glypg environment.  x & y define the upper-left of the entire pad.
** width & height define the CHAR side.  the NUM side's dimensions will be computed
** at runtime.
*/
BANKED void InitGlyph(int x,int y,int width,int height)
{
  unsigned short retval;

  Glyph.Style     = 0;
  Glyph.Mode      = 0;
  Glyph.Roof      = y;
  Glyph.Floor     = y + height;
  Glyph.HC        = Glyph.Roof + ((Glyph.Floor - Glyph.Roof) / 2);
  Glyph.CharLeft  = x;
  Glyph.CharRight = x+width;
  Glyph.NumLeft   = Glyph.CharRight+10;
  Glyph.NumRight  = Glyph.NumLeft + (width/2);
  Glyph.Dir       = 0;

  /*ResetGlyphPad();*/
}
