#ifndef TOPS_PDA_H
#define TOPS_PDA_H

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************\
*  tops_pda.h
*
*  Header file that defines the data types and functions for the TOPS interface
*  to the PDA-100.
*  TOPS Tables consist of records, and records consist of fields.
*  This header file defines the format of all the fields.
*  Then it defines the records that are made up of those fields, and finally
*  it defines the functions that are used to send and receive the records.
*
\******************************************************************************/

// define options
#ifdef __BORLANDC__
#define DLLEXPORT __export _stdcall
// Borland pragma for assuring byte alignment
#pragma option -a1
#else
#define DLLEXPORT __declspec(dllexport) _stdcall
#define _USERENTRY _cdecl
// MFC pragma's for assuring byte alignment
#pragma pack( push, alignment_before_tops_includes ) //store current alignment
#pragma pack(1)                                      //set up byte alignment
#endif


typedef unsigned long  DWORD;
typedef int            BOOL;
typedef unsigned char  BYTE;
typedef unsigned short WORD;


/*******************************************************************************
// Field Types.
*******************************************************************************/

typedef struct TOPS_UnassignedDef
{
  WORD FieldLength;
  BYTE Fields[128];
} TOPS_Unassigned;


typedef struct TOPS_Text8Def
{
  WORD FieldLength;
  BYTE Text[8];
} TOPS_Text8;


typedef struct TOPS_Text16Def
{
  WORD FieldLength;
  BYTE Text[16];
} TOPS_Text16;

typedef struct TOPS_Text127Def
{
  BYTE FieldLength;
  BYTE Text[127];
} TOPS_Text127;


typedef struct TOPS_Text128Def
{
  WORD FieldLength;
  BYTE Text[128];
} TOPS_Text128;

typedef struct TOPS_Text255Def
{
  BYTE FieldLength;
  BYTE Text[255];
} TOPS_Text255;

typedef struct TOPS_Text256Def
{
  WORD FieldLength;
  BYTE Text[256];
} TOPS_Text256;

typedef struct TOPS_Text512Def
{
  WORD FieldLength;
  BYTE Text[512];
}  TOPS_Text512;

typedef struct TOPS_Text3_5KDef
{
  WORD FieldLength;
  BYTE Text[3584];
}  TOPS_Text3_5K;

typedef BYTE   TOPS_Boolean;
typedef WORD   TOPS_Integer;
typedef BYTE   TOPS_Byte;
typedef WORD   TOPS_Word;
typedef double TOPS_Real;
typedef double TOPS_Currency;
typedef double TOPS_IntegerCustomField;
typedef struct TOPS_DateDef  // definition of data and time
{
  BYTE Year;
  BYTE Month;
  BYTE Day;
}TOPS_Date;


typedef struct TOPS_TimeDef
{
  BYTE Hour;
  BYTE Minute;
  BYTE Second;
}TOPS_Time;

typedef struct TOPS_TaskDateDef
{
  WORD FieldLength;
  TOPS_Date Date[100];
}TOPS_TaskDate;

typedef struct TOPS_ScheduleTimeDef
{
  TOPS_Time StartTime;
  TOPS_Time EndTime;
}TOPS_ScheduleTime;
typedef BYTE TOPS_Alarm;    // 0-reserved, 0=0 min, 1=5,2=10,3=15,4=30,5=45,
                            // 6=1 hour, 7=1.5, 8=2, 0xff=no alarm
typedef struct
{  WORD FieldLength;
   BYTE Drawing[3600];
} TOPS_Drawing;
typedef BYTE TOPS_Category; //range 0-7
typedef BYTE TOPS_Priority; // range 1-3, 0xff=no priority
typedef struct TOPS_RepeatDef
{
  TOPS_Date StartDate;
  TOPS_Date EndDate;
  BYTE Flag;
  BYTE Data1;
  BYTE Data2;
}TOPS_Repeat;
typedef struct TOPS_ListTypeDef
{
  BYTE  FieldLength;
  BYTE Text[16];
}TOPS_ListFieldType;
typedef TOPS_ListFieldType TOPS_ListType1;
typedef TOPS_ListFieldType TOPS_ListType2;
typedef TOPS_ListFieldType TOPS_ListType3;
typedef TOPS_ListFieldType TOPS_ListType4;
typedef TOPS_ListFieldType TOPS_ListType5;
typedef TOPS_ListFieldType TOPS_ListType6;
typedef TOPS_ListFieldType TOPS_ListType7;
typedef TOPS_ListFieldType TOPS_ListType8;
typedef TOPS_ListFieldType TOPS_ListType9;
typedef TOPS_ListFieldType TOPS_ListType10;
typedef TOPS_ListFieldType TOPS_ListType11;
typedef TOPS_ListFieldType TOPS_ListType12;
typedef TOPS_ListFieldType TOPS_ListType13;
typedef TOPS_ListFieldType TOPS_ListType14;
typedef TOPS_ListFieldType TOPS_ListType15;
typedef TOPS_ListFieldType TOPS_ListType16;
typedef TOPS_Date TOPS_DueDate;
typedef TOPS_Date TOPS_RepeatDueDate[100];
typedef TOPS_Byte TOPS_FieldDefName[16];
typedef TOPS_Date TOPS_AdvanceNotice;
typedef struct
{
  WORD FieldLength;
  BYTE Key[128];
}TOPS_SynchronizerKey;
typedef struct TOPS_IconDef
{
  WORD IconLength;
  BYTE Icon[5*1024];
} TOPS_Icon;
typedef struct TOPS_CatLabelTypeDef
{
  BYTE  FieldLength;
  BYTE Text[16];
}TOPS_CategoryLabelType;

#define MAX_LIST_NAME_LENGTH 16
typedef struct TOPS_ListNameDef
{
  BYTE  FieldLength;  // 0..16  (The name can be at most 16 bytes)
  BYTE  Text[MAX_LIST_NAME_LENGTH];
}TOPS_ListName;

#define MAX_NUM_OF_ITEMS_IN_LIST 16
#define MAX_SIZE_OF_ITEM_NAME    16
typedef struct TOPS_ListItemDef
{
  BYTE FieldLength;
  BYTE Text[MAX_SIZE_OF_ITEM_NAME];
}TOPS_ListItems;

typedef struct TOPS_ApplicationRecordDef
{
  DWORD Length;   // Apps size can be DWORD (which could be huge),
  BYTE Binary[];  // so don't use max size.
} TOPS_Application;


typedef struct TOPS_TableStatusRecordDef
{
  WORD  Length;
  BYTE  Data[128];
} TOPS_TableStatus;

typedef struct TOPS_DictionaryRecordDef
{
  DWORD Length;
  BYTE  Data[];
} TOPS_Dictionary;

typedef struct TOPS_LanguageBundleRecordDef
{
  DWORD Length;
  BYTE  Data[];
} TOPS_LanguageBundle;

typedef struct TOPS_PatchRecordDef
{
  DWORD Length;
  BYTE  Data[65535];
} TOPS_Patch;

typedef struct TOPS_ApplicationPreferenceRecordDef
{
  WORD Length;
  BYTE Data[256];
} TOPS_ApplicationPreference;

// the following structure defines the format in which data records are
// actually sent to the handheld.  This is the format that would be seen
// on the wire connecting the PC to the handheld.
//
typedef struct TOPS_GenericRecordFieldDef
{
  WORD FieldId;    // Field Id
  WORD DataLength; // Length of Field Data
  BYTE Data[];     // Data (size=DataLength)
} TOPS_GenericRecordField;


/*******************************************************************************
// Define the records of the differenct Record Tables. There are several
// different Record Tables in the PDA-100.  Each Record Table has a different
// format for its record.  The Record Tables are:
//
// 1. Address Table
// 2. Memo Table
// 3. Schedule Table
// 4. Task Table
// 5. Expense Table
// 6. Sketch Table
// 7. Data Tables(8)
// 8. Custom Downloadable App Record
//
*******************************************************************************/

typedef struct TOPS_AddressRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Category        Category;
  TOPS_Text128         LastName;
  TOPS_Text128         FirstName;
  TOPS_Text128         Company;
  TOPS_Text128         Title;
  TOPS_Text128         BusinessPhone;
  TOPS_Text128         Fax;
  TOPS_Text128         HomePhone;
  TOPS_Text128         Cellular;
  TOPS_Text128         Pager;
  TOPS_Text128         Email;
  TOPS_Text128         BusinessStreet;
  TOPS_Text128         BusinessCity;
  TOPS_Text128         BusinessState;
  TOPS_Text128         BusinessZip;
  TOPS_Text128         BusinessCountry;
  TOPS_Text128         HomeStreet;
  TOPS_Text128         HomeCity;
  TOPS_Text128         HomeState;
  TOPS_Text128         HomeZip;
  TOPS_Text128         HomeCountry;
  TOPS_Date            Birthday;
  TOPS_Text512         Notes;
  TOPS_Integer         IndexBy;
  TOPS_Integer         PrimaryComm;
  TOPS_Unassigned      Custom1;
  TOPS_Unassigned      Custom2;
  TOPS_Unassigned      Custom3;
  TOPS_Unassigned      Custom4;
  TOPS_Unassigned      Custom5;
  TOPS_Unassigned      Custom6;
  TOPS_Unassigned      Custom7;
  TOPS_Unassigned      Custom8;
}TOPS_AddressRecord;


typedef struct TOPS_MemoRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Category        Category;
  TOPS_Text256         Title;
  TOPS_Text3_5K        Memo;
  TOPS_Unassigned      Custom1;
  TOPS_Unassigned      Custom2;
  TOPS_Unassigned      Custom3;
  TOPS_Unassigned      Custom4;
}TOPS_MemoRecord;

typedef struct TOPS_ScheduleRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Category        Category;
  TOPS_Date            Date;
  TOPS_ScheduleTime    Time;
  TOPS_Text256         Description;
  TOPS_Repeat          Repeat;
  TOPS_Alarm           Alarm;
  TOPS_Text512         Note;
  TOPS_Unassigned     Custom1;
  TOPS_Unassigned     Custom2;
  TOPS_Unassigned     Custom3;
  TOPS_Unassigned     Custom4;
  TOPS_Unassigned     Custom5;
  TOPS_Unassigned     Custom6;
  TOPS_Unassigned     Custom7;
  TOPS_Unassigned     Custom8;
}TOPS_ScheduleRecord;

typedef struct TOPS_TaskRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Category        Category;
  TOPS_TaskDate        StartDate;
  TOPS_TaskDate        DueDate;
  TOPS_TaskDate        CompleteDate;
  TOPS_Text256         Description;
  TOPS_Priority        Priority;
  TOPS_Repeat          Repeat;
  TOPS_Text512         Note;
  TOPS_Unassigned     Custom1;
  TOPS_Unassigned     Custom2;
  TOPS_Unassigned     Custom3;
  TOPS_Unassigned     Custom4;
  TOPS_Unassigned     Custom5;
  TOPS_Unassigned     Custom6;
  TOPS_Unassigned     Custom7;
  TOPS_Unassigned     Custom8;
}TOPS_TaskRecord;


typedef struct TOPS_ExpenseRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Category        Category;
  TOPS_Date            DueDate;
  TOPS_Currency        Amount;
  TOPS_ListType1       Type;
  TOPS_Text128         Description;
  TOPS_Text128         Merchant;
  TOPS_Text128         Location;
  TOPS_ListType2       Method;
  TOPS_Text512         Note;
  TOPS_Unassigned      Custom1;
  TOPS_Unassigned      Custom2;
  TOPS_Unassigned      Custom3;
  TOPS_Unassigned      Custom4;
  TOPS_Unassigned      Custom5;
  TOPS_Unassigned      Custom6;
  TOPS_Unassigned      Custom7;
  TOPS_Unassigned      Custom8;
}TOPS_ExpenseRecord;

typedef struct TOPS_SketchRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Integer         ID;
  TOPS_Text16          Title;
  TOPS_Drawing         Drawing;
}TOPS_SketchRecord;

typedef struct TOPS_DataTableRecordDef
{
  TOPS_SynchronizerKey SynchronizerKey;
  TOPS_Unassigned     Custom1;
  TOPS_Unassigned     Custom2;
  TOPS_Unassigned     Custom3;
  TOPS_Unassigned     Custom4;
  TOPS_Unassigned     Custom5;
  TOPS_Unassigned     Custom6;
  TOPS_Unassigned     Custom7;
  TOPS_Unassigned     Custom8;
  TOPS_Unassigned     Custom9;
  TOPS_Unassigned     Custom10;
  TOPS_Unassigned     Custom11;
  TOPS_Unassigned     Custom12;
  TOPS_Unassigned     Custom13;
  TOPS_Unassigned     Custom14;
  TOPS_Unassigned     Custom15;
  TOPS_Unassigned     Custom16;
  TOPS_Unassigned     Custom17;
  TOPS_Unassigned     Custom18;
  TOPS_Unassigned     Custom19;
  TOPS_Unassigned     Custom20;
  TOPS_Unassigned     Custom21;
  TOPS_Unassigned     Custom22;
  TOPS_Unassigned     Custom23;
  TOPS_Unassigned     Custom24;
  TOPS_Unassigned     Custom25;
  TOPS_Unassigned     Custom26;
  TOPS_Unassigned     Custom27;
  TOPS_Unassigned     Custom28;
  TOPS_Unassigned     Custom29;
  TOPS_Unassigned     Custom30;
  TOPS_Unassigned     Custom31;
  TOPS_Unassigned     Custom32;
} TOPS_DataTableRecord;


/*******************************************************************************
// Define the Custom Downloadable App record.  Downloadable Apps can have a
// data table associated with them.  This data table must have a table id
// in the range 128 to 256.  This comment record defines the format for those
// data tables.
*******************************************************************************/

//typedef struct TOPS_CustomRecordDef
//{
//  WORD Length;
//  TOPS_GenericRecordField Field[];  // an array of fields
//}TOPS_CustomRecord;



typedef union TOPS_RecordTableRecordDef
{
  TOPS_AddressRecord          AddressRecord;
  TOPS_MemoRecord             MemoRecord;
  TOPS_ScheduleRecord         ScheduleRecord;
  TOPS_TaskRecord             TaskRecord;
  TOPS_ExpenseRecord          ExpenseRecord;
  TOPS_SketchRecord           SketchRecord;
  TOPS_DataTableRecord        DataTableRecord;
} TOPS_RecordTableRecords;


/*******************************************************************************
// Table Info Table record definition.
// There is only 1 Table Info Table.
// It is used to get/put information about any table in the PDA-100.
// For the Table Info Table the record id corresponds to a table id. For
// example, reading record 1 will get the table info for the Memo Table since
// the Memo Tables Table Id is 1.
*******************************************************************************/

typedef struct TOPS_RecordObjectsTableInfoRecordDef
{
  TOPS_Word Reserved;    // Reserved value, always = 0
  TOPS_Byte ReadOnly;    // 0=Modifiable, 1=ReadOnly
  TOPS_Text255 AppName;  // Name of Application
} TOPS_TableInfoRecord;

/*******************************************************************************
// Field Definition Table record definition.
// There is only 1 Field Definition Table.
// It is used to get/put the field definitions for the records of a table
// in the PDA-100.  The table id is constant (0x8006).
//  The record id corresponds to the table this field definition defines.
*******************************************************************************/

typedef enum
{
  TOPS_UnassignedFieldType   = 0,
  TOPS_Text16FieldType       = 1,
  TOPS_Text128FieldType      = 2,
  TOPS_Text256FieldType      = 3,
  TOPS_Text512FieldType      = 4,
  TOPS_Text3_5KFieldType     = 5,
  TOPS_BooleanFieldType      = 6,
  TOPS_IntegerFieldType      = 7,
  TOPS_RealFieldType         = 8,
  TOPS_CurrencyFieldType     = 9,
  TOPS_DateFieldType         = 10,
  TOPS_TimeFieldType         = 11,
  TOPS_ScheduleTimeFieldType = 12,
  TOPS_AlarmFieldType        = 13,
  TOPS_DrawingFieldType      = 14,
  TOPS_CategoryFieldType     = 15,
  TOPS_PriorityFieldType     = 16,
  TOPS_RepeatFieldType       = 17,
  TOPS_ListType1FieldType    = 18,
  TOPS_ListType2FieldType    = 19,
  TOPS_ListType3FieldType    = 20,
  TOPS_ListType4FieldType    = 21,
  TOPS_ListType5FieldType    = 22,
  TOPS_ListType6FieldType    = 23,
  TOPS_ListType7FieldType    = 24,
  TOPS_ListType8FieldType    = 25,
  TOPS_ListType9FieldType    = 26,
  TOPS_ListType10FieldType   = 27,
  TOPS_ListType11FieldType   = 28,
  TOPS_ListType12FieldType   = 29,
  TOPS_ListType13FieldType   = 30,
  TOPS_ListType14FieldType   = 31,
  TOPS_ListType15FieldType   = 32,
  TOPS_ListType16FieldType   = 33,
  TOPS_PrivateFieldType      = 34,
  TOPS_TaskDateFieldType     = 35,
  TOPS_CompleteDateFieldType = 36,
  TOPS_SynchronizerKeyFieldType = 37
}TOPS_DataFieldType;

typedef struct TOPS_RecordObjectsFieldDefinitionDef
{
  TOPS_Byte         DataType;  // TOPS_DataFieldTypes (used TOPS_Byte because C
                               // enumerated type is a WORD not BYTE).
  TOPS_Byte         FixedFlag; // 0 = fixed, 1 = type can be changed by PC
  TOPS_FieldDefName Name;      // name of field
} TOPS_FieldDefinition;

typedef struct TOPS_RecordObjectsFieldDefinitionRecordDef
{
  DWORD Length;
  TOPS_FieldDefinition Fields[];  // array of fields
}TOPS_FieldDefinitionRecord;

/*******************************************************************************
// Application Record Table record definition.
// There is only 1 Application Record Table.
// It is used to get/put the data record of an application in the PDA-100.
// For the Application Record Table the table id is constant (0x8001).
// The record id corresponds to the Application Id.
// (All app records must have App Info defined at the beginning of the data.
//  App Info is read by doing a get of the App Info Table, App Info is set
//  by doing a put of the App Record Table.)
*******************************************************************************/

typedef struct TOPS_AppRecDef
{
  TOPS_Application Application;
}TOPS_ApplicationRecord;

/*******************************************************************************
// Application Info Table record definition.
// There is only 1 Application Info Record Table.
// It is used to get the info of an application in the PDA-100.
// For the Field Definition Table the table id is constant (0x8002).
// The record id corresponds to the Application Id.
*******************************************************************************/

#define MAX_APPLICATION_NAME_LENGTH      256
#define MAX_APPLICATION_VERSION_LENGTH   128
#define MAX_APPLICATION_ICON_LENGTH      5*1024
#define MAX_APPLICATION_COPYRIGHT_LENGTH 128

typedef struct TOPS_ApplicationInfoRecordDef
{
  TOPS_Byte       AppType;
  TOPS_Byte       DataTableFlag;
  TOPS_Date       CreationDate;
  TOPS_Time       CreationTime;
  TOPS_Word       Checksum;
  TOPS_Text255    Name;
  TOPS_Text255    Version;
  TOPS_Icon       Icon;
  TOPS_Text127    Copyright;
  TOPS_Unassigned Reserved2;
} TOPS_ApplicationInfoRecord;

/*******************************************************************************
// Application Preference Table record definition.
// There is only 1 Application Record Table.
// It is used to get/put the preference record of an application in the PDA-100.
// For the Application Preference Table the table id is constant (0x8005).
// The record id corresponds to the Table Id
*******************************************************************************/

typedef struct TOPS_AppPrefRecDef
{
  TOPS_ApplicationPreference Preference;
}TOPS_ApplicationPreferenceRecord;

/*******************************************************************************
// PDA List Table record definition.
// There is only 1 PDA List Table.
// It is used to get/set list info in the PDA-100.
// For the PDA List Table the table id is constant (0x8003).
// The record id corresponds to the List Record Id which ranges from 0 to 15.
*******************************************************************************/

typedef struct TOPS_ListRecordDef
{
  TOPS_ListName  ListName;
  TOPS_ListItems Item1;
  TOPS_ListItems Item2;
  TOPS_ListItems Item3;
  TOPS_ListItems Item4;
  TOPS_ListItems Item5;
  TOPS_ListItems Item6;
  TOPS_ListItems Item7;
  TOPS_ListItems Item8;
  TOPS_ListItems Item9;
  TOPS_ListItems Item10;
  TOPS_ListItems Item11;
  TOPS_ListItems Item12;
  TOPS_ListItems Item13;
  TOPS_ListItems Item14;
  TOPS_ListItems Item15;
  TOPS_ListItems Item16;
} TOPS_ListRecord;

/*******************************************************************************
// Category Label Table record definition.
// There is only 1 Category Label Table.
// It is used to get/set categories in the PDA-100.
// For the Category Label Table the table id is constant (0x8004).
// The record id corresponds to the table id whose category labels are desired.
*******************************************************************************/

typedef struct TOPS_CategoryLabelRecordDef
{
  TOPS_CategoryLabelType SingularTitle;
  TOPS_CategoryLabelType PluralTitle;
  TOPS_CategoryLabelType Label1Name;
  TOPS_Byte              Label1Id;
  TOPS_CategoryLabelType Label2Name;
  TOPS_Byte              Label2Id;
  TOPS_CategoryLabelType Label3Name;
  TOPS_Byte              Label3Id;
  TOPS_CategoryLabelType Label4Name;
  TOPS_Byte              Label4Id;
  TOPS_CategoryLabelType Label5Name;
  TOPS_Byte              Label5Id;
  TOPS_CategoryLabelType Label6Name;
  TOPS_Byte              Label6Id;
  TOPS_CategoryLabelType Label7Name;
  TOPS_Byte              Label7Id;
  TOPS_CategoryLabelType Label8Name;
  TOPS_Byte              Label8Id;
}TOPS_CategoryLabelRecord;

/*******************************************************************************
// Table Status Table record definition.
// There is only 1 Application Record Table.
// It is used to get/put synchronization status of data tables.
// For the Table Status Table the table id is constant (0x8007).
// The record id corresponds to the Table Id.
*******************************************************************************/

typedef struct TOPS_TableStatusRecDef
{
  TOPS_TableStatus Status;
}TOPS_TableStatusRecord;

/*******************************************************************************
// Dictionary Table record definition.
// There is only 1 Dictionary Record Table.
// It is used to get/put the JustType User Dictionary.
// For the Dictionary Table the table id is constant (0x8008).
*******************************************************************************/

typedef struct TOPS_DictionaryRecDef
{
  TOPS_Dictionary Dictionary;
}TOPS_DictionaryRecord;

/*******************************************************************************
// Language Bundle Table record definition.
// There is only 1 Language Bundle Record Table.
// It is used to get/put the language specific data including the
// JustType Main Dictionary.
// For the Language Bundle Table the table id is constant (0x8009).
*******************************************************************************/

typedef struct TOPS_LanguageBundleRecDef
{
  TOPS_LanguageBundle LanguageBundle;
}TOPS_LanguageBundleRecord;


/*******************************************************************************
// Patch Table record definition.
// There is only 1 Patch Record Table.
// It is used to put a patch to the PDA.
// For the Patch Table the table id is constant (0x800A).
*******************************************************************************/

typedef struct TOPS_PatchRecDef
{
  TOPS_Patch Patch;
}TOPS_PatchRecord;



/*******************************************************************************
// The union of all possible TOPS record types.
*******************************************************************************/


typedef union TOPS_RecordDef
{
  TOPS_FieldDefinitionRecord  FieldDefinition;
  TOPS_RecordTableRecords     RecordTable;
  TOPS_TableInfoRecord        TableInfo;
  TOPS_ListRecord             ListTable;
  TOPS_ApplicationInfoRecord  AppInfoTable;
  TOPS_ApplicationRecord      AppTable;
  TOPS_ApplicationPreferenceRecord AppPrefTable;
  TOPS_CategoryLabelRecord    CategoryLabelTable;
  TOPS_TableStatusRecord      TableStatus;
  TOPS_DictionaryRecord       DictionaryTable;
  TOPS_LanguageBundleRecord   LanguageBundleTable;
  TOPS_Patch                  PatchTable;
} TOPS_Record;


//
// TOPS API Function Prototypes Overview
//
// The following function prototypes define the interface of the
// Table Oriented Protocol for Synchronization (TOPS) API.
// The TOPS API provides the data structure definitions and the functions
// necessary to communicate with the PDA-100.  The TOPS API uses the I/O API
// for handling the lower level protocol details.
//
// As the name implies the TOPS protocol is Table Oriented.  This means that
// all data communication to the PDA-100 is via a Table. Different tables are
// accessed by their associated Table Id.   Tables consist of one or more
// records.  The records in the tables can be accessed via Record Id's.

// TOPS is the protocol layer above the OBEX layer.
//
// All functions in this API return TRUE, FALSE, or UNEXPECTED RESPONSE.
// If the function returns TRUE, the function call was successful.
// If the function returns FALSE, the function call was unsuccessful.
// If the function returns UNEXPECTED RESPONSE then the PDA-100 is reporting
// an error and a call to TOPS_ReadUnexpectedResponse should be made to get
// the error information.
// Additional error information can be retrieved by examining the I/O API's
// Error Buffer.  The Error Buffer is a circular buffer that is written to
// by the software whenever an error occurs.  The include file IO_ERRORS.H
// contains more information about the I/O API Error Buffer.
//

// Function values returned by TOPS API Functions.
typedef enum TOPS_ReturnValuesDef
{
   TOPS_Failure                 = 0,
   TOPS_Success                 = 1,
   TOPS_UnexpectedResponse      = 2
}TOPS_ReturnValues;

// Table Types used as parameter on API calls.

typedef enum TOPS_TableTypesDef
{
   // TOPS record tables can range from 0-0x3fff.  Tables 0-15 are defined for
   // all handhelds.
   TOPS_FirstRecordTable        = 0,    // lowest record table #
   TOPS_AddressTable            = 0,    // address table is table 0
   TOPS_MemoTable               = 1,    // memo table is is table 1
   TOPS_ScheduleTable           = 2,
   TOPS_TaskTable               = 3,
   TOPS_ExpenseTable            = 4,
   TOPS_SketchTable             = 5,
   TOPS_Reserved1               = 6,
   TOPS_Reserved2               = 7,
   TOPS_Data1Table              = 8,
   TOPS_Data2Table              = 9,
   TOPS_Data3Table              = 10,
   TOPS_Data4Table              = 11,
   TOPS_Data5Table              = 12,
   TOPS_Data6Table              = 13,
   TOPS_Data7Table              = 14,
   TOPS_Data8Table              = 15,
   TOPS_FirstCustomAppTable     = 128,
   TOPS_LastCustomAppTable      = 256,
   TOPS_LastRecordTable         = 0x3fff, // highest record table #
   // Non-Record Tables
   TOPS_TableInfoTable          = 0x8000,
   TOPS_ApplicationRecordTable  = 0x8001,
   TOPS_ApplicationInfoTable    = 0x8002,
   TOPS_ListTable               = 0x8003,
   TOPS_CategoryLabelTable      = 0x8004,
   TOPS_ApplicationPreferenceTable = 0x8005,
   TOPS_FieldDefTable           = 0x8006,
   TOPS_TableStatusTable        = 0x8007,
   TOPS_DictionaryTable         = 0x8008,
   TOPS_LanguageBundleTable     = 0x8009,
   TOPS_PatchTable              = 0x800A,
   TOPS_LastTable
}TOPS_TableType;


// Definition of the bits in the status words of the records of a table.
typedef enum TOPS_StatusDef
{
   TOPS_Private                 = (1<<8),
   TOPS_FieldDefModified        = (1<<5),
   TOPS_DateModified            = (1<<4),
   TOPS_DeleteRecordTag         = (1<<3), //(Deletes record too if it exists)
   TOPS_RecordDeleted           = (1<<2),
   TOPS_RecordModified          = (1<<1),
   TOPS_RecordNew               = (1<<0)
}TOPS_Status;

// Definition of a Record Id. The record id value is dependent on the
// table type.  For example, the record id for the Application Table is an
// Application Id, the record id for Table Info Table is a Table Id
typedef DWORD TOPS_RecordId;



// FUNCTION PROTOTYPES:


// TOPS_BeginSession is the function that is used to start the communication
// session.  This function does the initialization of the API.
// If this function is not called, or it completes with a
// FALSE value, all subsequent calls to the API will return FALSE.
//
// Input Parameters:
//   p_port - This parameter tells the software what logical com port to
//   use for communication to the handheld if p_device is a COM device.
//   Typically this is COM4.  This
//   value depends on what logical com port the computer has setup for
//   IRDA communications.  This value is NOT the same as the actual physical
//   com port used for the IRDA connection. If p_device does not specify
//   a COM device then this parameter is ignored.
//
//   p_device - This parameter tells the software what the physical device
//   type is for the connection.  For example, it could be an IRDA device,
//   or it could be a COM port device, or a MODEM, etc.  The include
//   file devices.h has the values defined for this input parameter.
//
//   p_id_string_ptr - This parameter points to a block of memory that will
//   receive the PDA-100 indentification character string returned by the
//   handheld.  This string will be null terminated and
//   contains version information that should be examined to ensure that it is
//   compatible with the caller.
//   The buffer this parameter points at must be large enough to contain the
//   id and the null terminator.
//

// Return Values:
//   The function returns TRUE if all initialization succeeded, and the
//   software successfully initiated a connect sequence with the handheld.
//   If this software returns TRUE, it indicates that the API has
//   established a connection with the handheld and that subsequent API
//   function calls should successfully execute.  If this function returns
//   FALSE it indicates that a connection could not be established with
//   the handheld.  The I/O API Error Buffer can be examined to
//   retrieve additional error information.
//
//
// Usage Notes:
//   1.  This API was developed to allow a sesssion to be initiated from both
//       the PC and from the handheld.
//       TOPS_BeginSession will create the connection to the handheld. This
//       will cause automatic power-on of the handheld if it is connected via
//       the wire.  Since the handheld can also initiate a session, the PC
//       software must poll the handheld in order to determine when the
//       handheld is initiating a session. TOPS_BeginSession can not be used
//       to poll the handheld because each time it is called it would turn on
//       the handheld, this would waste the handheld battery and look strange
//       with the handheld going on and off.  Because of this, the function
//       TOPS_CheckComm should be used to poll for a handheld initiated session.
//       TOPS_CheckComm will not turn on the handheld because it uses the IRDA
//       sniff protocol to detect the presence of the handheld. See
//       TOPS_CheckComm for more details on it's usage. If the session is
//       initiated from the PC, TOPS_CheckComm shouldn't be called, and
//       TOPS_BeginSession can be called directly.

TOPS_ReturnValues DLLEXPORT TOPS_BeginSession(int p_port,
                                             IO_DeviceIdType p_device,
                                             char *p_id_string);




// TOPS_EndSession is the function that does the shutdown of the
// TOPS API.  This includes doing the OBEX disconnect sequence for the PDA-100.
//
// This must be the last API routine that is called.
//
//
// Input Parameters:  NONE
//
//
// Return Values:
//   The function returns TRUE if the disconnect packet sent to the handheld
//   was transmitted successfully.
//   The function returns FALSE if the disconnect packet sent to the handheld
//   was unsuccessful or a disconnect was attempted when not connected.  The
//   I/O API Error Buffer can be examined to retrieve additional error
//   information.
//   Regardless of whether this function returns TRUE or FALSE, the IO API will
//   be shutdown.
//

TOPS_ReturnValues DLLEXPORT TOPS_EndSession(void);




// TOPS_GetTags is the function that is used to get the tags of the
// records of a table. A tag is simply a two word data structure that
// contains the record id and the status of the record. The status of the
// record tells if the record is new, modified, deleted, and in some tables
// whether or not it is an exclusive access record.
// Since the records of Table Info Table
// contain the statuses of all the other tables, doing a GetTags of the
// Table Info Table is a quick way to get the statuses of all the tables.
// TOPS_BeginSession must be called before this function will properly
// execute.
//
// Input Parameters:
//   p_table_type - This parameter tells the software which table to get
//   the tags of all the records from.
//   p_selector - This parameter allows the user to further specify which
//   tags to return.  This parameter allows the caller to get:
//     1. all the tags,
//     2. get just the tags of modified records,
//     3. just the tags of new records,
//     4. just the tags of deleted records.
//     5. any combination of the above.
//   p_num_returned_tags - This parameter points at a memory location that will
//   receive the actual number of tags that were returned by the handheld.
//   p_returned_tags_ptr - This parameter points at a buffer that will
//   receive the returned tags.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.
//


// Definition of the tag list paramter.
typedef struct TOPS_TagDef
{
   TOPS_RecordId Id;
   WORD          Reserved;
   WORD          Status;    // can't make type TOPS_Status because enum is
                            // type int which is 4 bytes
}TOPS_TagType;

typedef TOPS_TagType TOPS_TagList[]; // an array of tags

// Definition of the Selector parameter.
typedef enum TOPS_Selector
{
   TOPS_GetPrivate              = (1<<8),
   TOPS_GetAll                  = (1<<3),
   TOPS_GetDeleted              = (1<<2),
   TOPS_GetModified             = (1<<1),
   TOPS_GetNew                  = (1<<0)
}TOPS_Selector;

TOPS_ReturnValues DLLEXPORT TOPS_GetTags(TOPS_TableType p_table_type,
                                         TOPS_Selector  p_selector,
                                         unsigned int  *p_num_of_returned_tags,
                                         TOPS_TagList  *p_returned_tags_ptr);



// TOPS_PutTags is the function that is used to set the statuses of the
// records of a table. A tag is simply a two word data structure that
// contains the record id and the status of the record. The status of the
// record tells if the record is new, modified, deleted, and in some tables
// whether or not it is an exclusive access record.
// Since the records of Table Info Table
// contain the statuses of all the other tables, doing a PutTags of the
// Table Info Table is a quick way to delete/create one or more tables.
// TOPS_BeginSession must be called before this function will properly
// execute.
//
// Input Parameters:
//   p_table_type - This parameter identifies which table will have its
//   record tags changed.
//   p_num_of_tags - This parameter identifies the number of tags to change.
//   This parameter must be equal to the number of tags pointed to by the
//   input parameter p_tags_ptr.
//   p_tags_ptr - This parameter points at a list of the new tags for
//   the specifed records and table.  In the list, there is one tag per
//   record to modify.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.
//

TOPS_ReturnValues DLLEXPORT TOPS_PutTags(TOPS_TableType p_table_type,
                                           WORD           p_num_tags,
                                           TOPS_TagList  *p_tags_ptr);





// TOPS_GetRecord is the function that is used to get a record from a table.
//
// Input Parameters:
//   p_table_type - This parameter tells the software which table to get
//   the record from.
//   p_record_id - The record id to get.
//   p_returned_rec_status_ptr - This parameter points at a memory location that
//   will receive the status of the returned record.
//   p_returned_rec_ptr - This parameter points to the record structure that
//   will be filled with the returned record.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

TOPS_ReturnValues DLLEXPORT TOPS_GetRecord( TOPS_TableType p_table_type,
                                           TOPS_RecordId  p_record_id,
                                           WORD          *p_returned_rec_status_ptr,
                                           TOPS_Record   *p_returned_rec_ptr);



// TOPS_PutRecord is the function that is used to put a record to a table.
//
// Input Parameters:
//   p_table_type - This parameter tells the software which table to put
//   the record to.   
//   p_record_id - The record id to put.
//   p_record_status - The status of the put record.
//   p_record_ptr - Pointer to the record to put.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

TOPS_ReturnValues DLLEXPORT TOPS_PutRecord( TOPS_TableType p_table_type,
                                           TOPS_RecordId  p_record_id,
                                           WORD           p_record_status,
                                           TOPS_Record   *p_record_ptr );


// TOPS_GetUnexpectedResponse is the function that is used to retrieve the
// unexpected response value from the handheld.  If one of the TOPS API
// functions return the unexpected response value (2), then the user
// should call this function to get the unexpected response data.
//
// Input Parameters:
//   p_error_category_ptr - Pointer to the location to receive the error
//   category from the handheld.
//   p_error_id_ptr - Pointer to the locationt to receive the error id from
//   the handheld.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.

// Definition of return parameter values
typedef enum TOPS_ErrorCategoryDef
{
 CommonError          = 0,
 RequestSpecificError = 1   // currently no request specific errors identified
} TOPS_ErrorCategory;

typedef enum TOPS_CommonErrorDef
{
  Password_Required   = 0,
  Object_Not_Found    = 1,
  Insufficient_Memory = 2,
  Invalid_OBEX_Opcode = 3,
  Id_Unavailable      = 4,
  Bad_Usage           = 5,
  Bad_Object_Size     = 6
}TOPS_ErrorId;


TOPS_ReturnValues DLLEXPORT TOPS_GetUnexpectedResponse(
                                    TOPS_ErrorCategory   *p_error_category_ptr,
                                    TOPS_ErrorId         *p_error_id_ptr);


 // Sys Info

typedef enum TOPS_SysInfoItemsDef
{
  // Category A Read/Write Item 0
  TOPS_DatabaseStatusItem=0,
  // Category A Read/Write, Item 1
  TOPS_PC_Storage_A_Item=1,
  // Category A Read/Write, Item 2
  TOPS_HH_ClockItem=2,
  // Category A Read/Write, Item 3
  TOPS_CityTimeSettingsItem=3,
  // Category A Read/Write, Item 4
  TOPS_DailyAlarmItem=4,
  // Category A Read/Write, Item 5
  TOPS_SystemSettingsItem=5,
  // Category A Read/Write, Item 6
  TOPS_DisplayFormatItem=6,
  // Category A Read/Write, Item 7
  TOPS_OwnerInfoItem=7,
  // Category A Read/Write, Item 8
  TOPS_OptimizationSettingsItem=8,
  // Category A Read/Write, Item 9
  TOPS_PrivateLabelItem=9,  // text for "private" checkbox
  // Category A Read/Write, Item 10
  TOPS_CalendarAppPreferencesItem=10,
  // Category A Read/Write, Item 11
  TOPS_TaskAppPreferencesItem=11,
  // Category A Read/Write, Item 12
  TOPS_AddressAppPreferencesItem=12,
  // Category A Read/Write Item 13
  TOPS_JustTypeDictionaryStatusItem=13,
  // Category B Read/Write, Item 0
  TOPS_HandheldIdItem=14,
  // Category B Read/Write, Item 1
  TOPS_PC_Storage_B_Item=15,
  // Category B Read/Write, Item 2
  TOPS_TZandDSTDatabaseEntryItem=16,
  // Category B Read/Write, Item 3
  TOPS_SyncMessageItem=17, // message to display during synchronization
  // Category B Read/Write, Item 4
  TOPS_SplashScreenItem=18,
  // Category C Read/Write, Item 0
  TOPS_PasswordItem=19,
  //*** Read-Only Items:
  // Category A Read-Only, Item 0
  TOPS_ModelInfoItem=20,
  // Category A Read-Only, Item 1
  TOPS_MemoryCapacityItem=21,
  // Category A Read-Only, Item 2
  TOPS_MemoryAvailableItem=22,
  // Category A Read-Only, Item 3
  TOPS_LocalTZandDSTItem=23,
  // Category A Read-Only, Item 4
  TOPS_City1to4TZandDSTItem=24,
  // Category A Read-Only, Item 5
  TOPS_JustTypeDictionarySizeItem=25,
  // Category A Read-Only, Item 6
  TOPS_LanguageBundleItem=26,

  // All Category A Items
  TOPS_GroupedItem=27,
  // Category B Read-Only Item 0
  TOPS_UserRequestedSyncItem=28,
  // Category B Read-Only Item 1
  TOPS_UserCancelledSyncItem=29,
  // Category D Read-Only Item 0
  TOPS_NewRecordIdItem=30,
  // Category D Read-Only Item 1
  TOPS_NewApplicationIdItem=31,
  TOPS_MaxSysInfoItem
}TOPS_SysInfoItems;


 // Read/write structures
 //   Theses are the structures that can be both written and read by the
 //   PC software.

 // Category A Read/Write

 // Category A Read/Write, Item 0
typedef BYTE TOPS_DatabaseStatus[16];

// Category A Read/Write, Item 1
typedef BYTE TOPS_PC_Storage_A[64];

typedef struct TOPS_HH_ClockDataAndTimeDef
{ // Category A Read/Write, Item 2
  TOPS_Date Date;
  TOPS_Time Time;
}TOPS_HH_Clock;
typedef struct TOPS_CityTimeSettings
{ // Category A Read/Write, Item 3
  WORD LocalIndex;
  WORD City1Index;
  WORD City2Index;
  WORD City3Index;
  WORD City4Index;
}TOPS_CityTimeSettings;
typedef struct TOPS_DailyAlarmDef
{ // Category A Read/Write, Item 4
  BYTE Hour; // (0..23)
  BYTE Minute; // (0..59)
  BYTE On_Off; // 0=off, 1=on
}TOPS_DailyAlarm;
typedef struct TOPS_SystemSettingsDef
{ // Category A Read/Write, Item 5
  BYTE PowerDownMinutes; // (1,2, or 3 minutes)
  BYTE BacklightOnTimer; // (0=5 seconds, 1=10s, 2=15s)
  BYTE AlarmSound;       // (0=no, 1=yes)
  BYTE PenTone;          // (0=no, 1=yes)
  BYTE StartupScreen;    // (0=don't show, 1=show default, 2=show graphic)
}TOPS_SystemSettings;

typedef struct TOPS_DisplayFormatDef
{ // Category A Read/Write, Item 6
  BYTE DateDisplay;  // 0=M/D/Y, 1=D/M/Y
  BYTE TimeDisplay;  // 0=12hr,  1=24hr
  BYTE NumberDisplay; // 0=1,234.56   1=1.234,56
  BYTE StartOfWeek;   // 0=Sunday,  1=Monday
}TOPS_DisplayFormat;

typedef struct TOPS_OwnerInfoDef
{ // Category A Read/Write, Item 7
  BYTE Name[32];
  BYTE Note[160];
}TOPS_OwnerInfo;

typedef struct TOPS_OptimizationSettingsDef
{ // Category A Read/Write, Item 8
  BYTE Frequency;          //0=manually, 1=daily, 2=weekly
  BYTE HourToOptimize;     // (0..23)
  BYTE DayToOptimize;      // (0..6) 0 = Sunday
}TOPS_OptimizationSettings;

// Category A Read/Write, Item 9
typedef BYTE TOPS_PrivateLabel[16];  // text for "private" checkbox

typedef struct TOPS_CalendarAppPreferencesDef
{ // Category A Read/Write, Item 10
  BYTE StartDayAt;      // (0..18)
  BYTE EventTimeStyle;  // 0=Start time/end time, 1=Start time and duration
  BYTE DefaultDuration; // 0=15min, 1=30min, 2=1hour
  BYTE TimeSlot;        // 0=15min, 1=30min, 2=1hour
  BYTE AdvanceNotice;   // 0=0min,1=5,2=10,3=15,4=30,5=45,6=1hour,7=1.5,8=2hours
}TOPS_CalendarAppPreferences;

typedef struct TOPS_TaskAppPreferencesDef
{ // Category A Read/Write, Item 11
  BYTE MaxAlphaPriority;   //1=No Numeric Priority, 2=B,3=C,4=D,5=E
  BYTE MaxNumericPriority; //1=No Numeric Priority, 2=2,3=3.....9=9
  BYTE ShowPriorityInList; // 0=no, 1=yes
  BYTE ShowDueDateInList;  // 0=no, 1=yes
}TOPS_TaskAppPreferences;

typedef struct TOPS_AddressAppPreferencesDef
{  // Category A Read/Write, Item 12
  BYTE AddressFormat; // 0=NorthAmerican, 1=European
}TOPS_AddressAppPreferences;

typedef struct TOPS_JustTypeDictionaryStatusDef
{  // Category A Read/Write, Item 13
  WORD NewWords;
  BYTE CorruptionFlag; // 0=Good, 1=Corrupt
}TOPS_JustTypeDictionaryStatus;

// Category B Read/Write

// Category B Read/Write, Item 0
typedef BYTE TOPS_HandheldId[16];

// Category B Read/Write, Item 1
typedef BYTE TOPS_PC_Storage_B[64];



// Special info for daylight savings time data:
// 1. When month is 0, the following two bytes stands for day of year
//   (in lo-byte, hi-byte format). Leap day is counted in calculation
//   for day of year.
// 2. When month is not 0 and week is 0 then month and date stand for normal
//    calendar day.
// 3. When month <> 0 and week <> 0  then this specifies day d of week w of
//    month m. The day d must be between 1 (Sunday) and 7. The week w must
//    be between 1 and 5; week 1 is the first week in which day d occurs,
//    and week 5 specifies the last d day in the month.

typedef struct TOPS_DSTDateDef
{
  BYTE Month;
  BYTE Week;
  BYTE Day;
}TOPS_DSTDate;

typedef struct TOPS_DSTTimeDef
{
  BYTE Hour;    // (0..23)
  BYTE Minutes; // (0..59)
}TOPS_DSTTime;

typedef struct TOPS_TZandDSTDef
{
  BYTE HourOffset;   // Timezone hour offset from GMT (0..12)
  BYTE MinuteOffset; // Timezone minute offset from GMT (0..59)
  BYTE DSTChange;    // DST change in minutes (-60..+60)
  TOPS_DSTDate StartDate;
  TOPS_DSTTime StartTime;
  TOPS_DSTDate EndDate;
  TOPS_DSTTime EndTime;
  BYTE PC_Storage[2];
  BYTE CityName[20];
}TOPS_TZandDST;

// Category B Read/Write, Item 2
typedef struct TOPS_TZandDSTDatabaseEntryDef
{
  TOPS_TZandDST Entries[100];
}TOPS_TZandDSTDatabaseEntry;

// Category B Read/Write, Item 3
typedef struct TOPS_SyncMessageDef
{
  BYTE SyncMessage[64];        // message to display during synchronization
  BYTE EndOfSyncMessage[64];   // message displayed after synchronization
  BYTE SyncResultsMessage[64]; // message describing result of sync
}TOPS_SyncMessage;


// The image is a sequence of rows. The first row has the top line of the image.
// Each row has just enough bytes to hold the pixels for the width of an image.
// Any remaining unused bits are filled with 0. The number of rows is equal to
// the height of the image in pixels.
// if you plan to display the owner info over the splash screen at startup,
// then choose or create an image that will still look good if the lower 1/3rd
// of the screen is covered.

// Category B Read/Write, Item 4
typedef struct TOPS_SplashScreenDef
{
  BYTE Image[4800]; // image
}TOPS_SplashScreen;

// Category C Read/Write, Item 0
typedef BYTE TOPS_Password[8];





//*********************************************************************
// Read only structures.
//   The following structures can only be read by the PC software
//*********************************************************************
// Category A Read-Only

// Category A Read-Only, Item 0
typedef char TOPS_ModelInfo[sizeof("PDA100 VM.m.P.p")-1];
// M : major version #
// m : minor version #
// P :  major patch #
// p :  minor patch #
// For example, each of the units in the first shipment would have Model Info
// of  "PDA100 V1.0.0.0". If they received patch version 2.4, then it would
// be "PDA100 V1.0.2.4"

// Category A Read-Only, Item 1
typedef struct TOPS_MemoryCapacityDef
{
  DWORD TOPS_ROMSize;
  DWORD TOPS_SRAMSize;
  DWORD TOPS_FlashSize;
}TOPS_MemoryCapacity;
// Category A Read-Only, Item 2
typedef struct TOPS_MemoryAvailableDef
{
  DWORD TOPS_BytesAvailable;
  DWORD TOPS_BytesDeleted;
  DWORD TOPS_BytesActive;
}TOPS_MemoryAvailable;
// Category A Read-Only, Item 3
typedef TOPS_TZandDST TOPS_LocalTZandDST;
// Category A Read-Only, Item 4
typedef TOPS_TZandDST TOPS_City1to4TZandDST[4];
// Category A Read-Only, Item 5
typedef WORD TOPS_JustTypeDictionarySize;
// Category A Read-Only, Item 6
typedef TOPS_Byte TOPS_LanguageBundleSysInfo; // 0=none loaded, 1=English,2=French,3=German,4=Spanish

// Category B Read-Only

typedef BYTE TOPS_UserRequestedSync;
typedef BYTE TOPS_UserCancelledSync;

// Category D Read-Only
typedef DWORD TOPS_NewRecordId;
typedef WORD TOPS_NewApplicationId;

// Category D Items also require parameters on the call to TOPS_GetSysInfo.

// Parameter definition for getting new record id
typedef struct TOPS_NewRecIdParmDef
{
  WORD RecordSize;  // size of the new record
  WORD TableId;     // table of the new record
}TOPS_NewRecIdParm;
// Parameter definition for getting new application id
typedef struct TOPS_NewAppIdParmDef
{
  WORD TOPS_NewAppSize;  // size of new application
}TOPS_NewAppIdParm;

typedef union TOPS_SysInfoParmsDef
{
  TOPS_NewRecIdParm RecordIdParm;
  TOPS_NewAppIdParm AppIdParm;
}TOPS_SysInfoParms;


// Category A Grouped Object (Read Only)
typedef struct TOPS_GroupedObjectDef
{
  // Category A Read/Write Item 0
  TOPS_DatabaseStatus DatabaseStatus;
  // Category A Read/Write, Item 1
  TOPS_PC_Storage_A    PC_Storage;
  // Category A Read/Write, Item 2
  TOPS_HH_Clock HH_Clock;
  // Category A Read/Write, Item 3
  TOPS_CityTimeSettings CityTimeSettings;
  // Category A Read/Write, Item 4
  TOPS_DailyAlarm Alarm;
  // Category A Read/Write, Item 5
  TOPS_SystemSettings SystemSettings;
  // Category A Read/Write, Item 6
  TOPS_DisplayFormat DisplayFormat;
  // Category A Read/Write, Item 7
  TOPS_OwnerInfo OwnerInfo;
  // Category A Read/Write, Item 8
  TOPS_OptimizationSettings OptimizationSettings;
  // Category A Read/Write, Item 9
  TOPS_PrivateLabel PrivateLabel;  // text for "private" checkbox
  // Category A Read/Write, Item 10
  TOPS_CalendarAppPreferences CalendarAppPreferences;
  // Category A Read/Write, Item 11
  TOPS_TaskAppPreferences TaskPreferences;
  // Category A Read/Write, Item 12
  TOPS_AddressAppPreferences AddressAppPreferences;
  // Category A Read/Write, Item 13
  TOPS_JustTypeDictionaryStatus JustTypeStatus;
  // Category A Read-Only, Item 0
  TOPS_ModelInfo ModelInfo;
  // Category A Read-Only, Item 1
  TOPS_MemoryCapacity MemoryCapacity;
  // Category A Read-Only, Item 2
  TOPS_MemoryAvailable MemoryAvailable;
  // Category A Read-Only, Item 3
  TOPS_LocalTZandDST TOPS_LocalTZandDST;
  // Category A Read-Only, Item 4
  TOPS_City1to4TZandDST City1to4TZandDST;
  // Category A Read-Only, Item 5
  TOPS_JustTypeDictionarySize JustTypeSize;
  // Category A Read-Only, Item 6
  TOPS_LanguageBundleSysInfo LanguageBundle;
}TOPS_GroupedObject;



typedef union TOPS_ReadWriteSysInfoDef
{
  // Category A Read/Write Item 0
  TOPS_DatabaseStatus DatabaseStatus;
  // Category A Read/Write, Item 1
  TOPS_PC_Storage_A  PC_StorageA;
  // Category A Read/Write, Item 2
  TOPS_HH_Clock HH_Clock;
  // Category A Read/Write, Item 3
  TOPS_CityTimeSettings CityTimeSettings;
  // Category A Read/Write, Item 4
  TOPS_DailyAlarm DailyAlarm;
  // Category A Read/Write, Item 5
  TOPS_SystemSettings SystemSettings;
  // Category A Read/Write, Item 6
  TOPS_DisplayFormat DisplayFormat;
  // Category A Read/Write, Item 7
  TOPS_OwnerInfo OwnerInfo;
  // Category A Read/Write, Item 8
  TOPS_OptimizationSettings OptimizationSettings;
  // Category A Read/Write, Item 9
  TOPS_PrivateLabel PrivateLabel;  // text for "private" checkbox
  // Category A Read/Write, Item 10
  TOPS_CalendarAppPreferences CalendarAppPreferences;
  // Category A Read/Write, Item 11
  TOPS_TaskAppPreferences TaskAppPreferences;
  // Category A Read/Write, Item 12
  TOPS_AddressAppPreferences AddressAppPreferences;
  // Category A Read/Write, Item 13
  TOPS_JustTypeDictionaryStatus JustTypeStatus;
  // Category B Read/Write, Item 0
  TOPS_HandheldId HandheldId;
  // Category B Read/Write, Item 1
  TOPS_PC_Storage_B PC_StorageB;
  // Category B Read/Write, Item 2
  TOPS_TZandDSTDatabaseEntry TZandDSTDatabaseEntry;
  // Category B Read/Write, Item 3
  TOPS_SyncMessage SyncMessage; // message to display during synchronization
  // Category B Read/Write, Item 4
  TOPS_SplashScreen SplashScreen;
  // Category C Read/Write, Item 0
  TOPS_Password Password;
}TOPS_ReadWriteSysInfo;

typedef union TOPS_ReadOnlyDef
{
  // Category A Read-Only, Item 0
  TOPS_ModelInfo ModelInfo;
  // Category A Read-Only, Item 1
  TOPS_MemoryCapacity MemoryCapacity;
  // Category A Read-Only, Item 2
  TOPS_MemoryAvailable MemoryAvailable;
  // Category A Read-Only, Item 3
  TOPS_LocalTZandDST LocalTZandDST;
  // Category A Read-Only, Item 4
  TOPS_City1to4TZandDST City1to4TZandDST;
  // Category A Read-Only, Item 5
  TOPS_JustTypeDictionarySize JustTypeSize;
  // Category A Read-Only, Item 6
  TOPS_LanguageBundleSysInfo LanguageBundle;
  // All Category A Items
  TOPS_GroupedObject GroupedObject;
  // Category B Read-Only Item 0
  TOPS_UserRequestedSync UserRequestedSync;
  // Category B Read-Only Item 1
  TOPS_UserCancelledSync UserCancelledSync;
  // Category D Read-Only Item 0
  TOPS_NewRecordId NewRecordId;
  // Category D Read-Only Item 1
  TOPS_NewApplicationId NewApplicationId;
}TOPS_ReadOnlySysInfo;

typedef union TOPS_SysInfoDef
{
  TOPS_ReadWriteSysInfo ReadWrite;
  TOPS_ReadOnlySysInfo  ReadOnly;
}TOPS_SysInfo;


// TOPS_PutSysInfo is the function that is used to put read/write SysInfo Items
//
// Input Parameters:
//   p_item_type - This parameter identifies what type item to put.
//   p_item_ptr - Pointer to the item to put.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

TOPS_ReturnValues DLLEXPORT TOPS_PutSysInfo( TOPS_SysInfoItems      p_item_type,
                                            TOPS_ReadWriteSysInfo *p_item_ptr );

// TOPS_GetSysInfo is the function that is used to get SysInfo Items
//
// Input Parameters:
//   p_item_type - This parameter identifies what type item to get.
//   p_parameters - This is the pointer to the parameter(s) for the get.
//   p_item_ptr - Pointer to memory location to receive the SysInfo.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

TOPS_ReturnValues DLLEXPORT TOPS_GetSysInfo( TOPS_SysInfoItems  p_item_type,
                                            TOPS_SysInfoParms *p_parameters,
                                            TOPS_SysInfo      *p_item_ptr );


// TOPS_PutCommand is the function that is used to send commands
//
// Input Parameters:
//   p_item_type - This parameter identifies the command.
//   p_parm_ptr - This is the pointer to the parameter(s) to send. See the
//                TOPS document for a description of the parameters.
//
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

typedef enum TOPS_CommandTypeDef
{
  TOPS_UnlockWithPassword   = 0,
  TOPS_CompactMemory        = 1,
  TOPS_SyncMessages         = 2,
  TOPS_DeleteUserDictionary = 3,
  TOPS_MaxCommand
}TOPS_CommandType;

#define TOPS_UnlockWithPasswordParameterSize   8
#define TOPS_CompactMemoryParameterSize        0
#define TOPS_SyncMessagesParameterSize         64*3
#define TOPS_DeleteUserDictionaryParameterSize 0
#define TOPS_MaxCmdParameterSize               64*3
typedef BYTE TOPS_CommandParameter[TOPS_MaxCmdParameterSize];

TOPS_ReturnValues DLLEXPORT TOPS_PutCommand(TOPS_CommandType       p_item_type,
                                            TOPS_CommandParameter *p_parm_ptr );

// TOPS_SetupCallback is the function that is used to setup a routine that will
// be called by the API when it is in a time consuming loop.  The callback
// routine must return TRUE or FALSE.  If the callback returns FALSE the API
// will stop its current execution and return.
//
// Input Parameters:
//   p_IO_Callbackfunction - Pointer to the callback function.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.

TOPS_ReturnValues DLLEXPORT TOPS_SetupCallback(int (_USERENTRY *p_IO_Callbackfunction)(void));

// TOPS_CancelCallback is the function that is used to remove a callback that
// was setup using TOPS_SetupCallback.
//
// Input Parameters:
//   None.
//
// Return Values:
//   This function can return 0, 1, or 2.
//   0 indicates a failure.
//   1 indicates the function executed successfully.
//   2 indicates an unexpected response from the handheld.



TOPS_ReturnValues DLLEXPORT TOPS_CancelCallback(void);


// TOPS_SetupDisconnectWarningCallback is the function that is used to setup
// a routine that will be called by the API when the IRLAP layer of the TI
// stack has not received any response from the handheld. This could be due
// to the handheld being removed from the cradle, or the cable removed, etc.
// The API calls the callback with a parameter indicating the current
// disconnect count.  IRLAP sends the first Warning when the count reaches
// 3, therefore the first number received will be 3.  If the handheld starts
// responding, the callback will be called with a value of 0xFFFF.  If the
// handheld never returns, the IRLAP layer will timeout after the IRLAP
// negotiatied timeout, and this function will return 0xFFFE.   This
// function is intended to allow the API user to pop up a message to the end
// product user when an obstruction in the data path occurs. The API user knows
// that the message can be removed when it receives the 0xFFFF or 0xFFFE value.
// Data path obstruction in wired communication this is not as probable as in
// a true IR usage.
//
// Input Parameters:
//   p_IO_Callbackfunction - Pointer to the callback function.
//
// Return Values:
//   NONE
//
// Callback Input Parameters:
//   integer that will get the current disconnect count

void DLLEXPORT TOPS_SetupDisconnectWarningCallback(void (_USERENTRY *p_IO_Callbackfunction)(int));

// TOPS_CancelDisconnectWarningCallback is the function that is used to remove
// a callback that was setup using TOPS_SetupDisconnectWarningCallback.
//
// Input Parameters:
//   None.
//
// Return Values:
//   None.

void DLLEXPORT TOPS_CancelDisconnectWarningCallback(void);


// TOPS_CheckComm is the function that is used to determine if there is
// a handheld ready to communicate. This routine should be called
// repeatedly until it returns TRUE, and then TOPS_BeginSession should be
// called with the same values for the port and device.
//
// Input Parameters:
//   p_port - This parameter tells the software what com port to
//   use for checking for communication to the handheld. Typically if the
//   Win95 stack is to be used (for IR) this value is COM4, if the TI stack
//   is to be used (for Wired Communication) this value is COM1 or COM2.
//   p_device - Identifies which connection to check, e.g. wired or IRDA.
//   This parameter essentially dictates which stack is going to be used and
//   the method by which a connection will be checked. There are 2 major
//   choices for p_device, IO_ComPort, and IO_IRDA:
//     When the input parameter is IO_ComPort it specifies communication
//   through the com port (ie the TI stack is used), then this function
//   initiates the low power IrDA sniffing algorithms.  Subsequent calls to
//   this function then check to see if a sniff detection has occurred. If a
//   detection  has occurred then this function returns TOPS_CheckCommSuccess,
//   and the user should then call TOPS_BeginSession to start the handheld
//   initiated session.  Note that this function only returns
//   TOPS_CheckCommSuccess when the handheld has attempted
//   to initiate a session, it is OK to call TOPS_BeginSession at anytime if
//   the PC session is to be initiated by the PC.  I.E. you don't have to wait
//   for this function to return TOPS_CheckCommSuccess before initiating a
//   session from the PC.
//     When the input paramter is IO_IRDA it specifies communication through
//   the IR Port, in this case the Win95 stack is intended to be used.  The
//   function will attempt to connect and if the connect attempt fails then
//   the function returns TOPS_CheckCommFindFailure. If it succeeds the
//   function returns TOPS_CheckCommSuccess, and the user should then call
//   TOPS_BeginSession.  Note that the Win95 stack has no way to initiate
//   a sniff, this means that when this function returns TOPS_CheckCommSuccess
//   the handheld has been connected to and can be communicated with regardless
//   of whether or not the handheld user has initiated a connection on the
//   handheld unit.
//
// Return Values:
//   0 - No connection available, call this function again at a later time.
//   1 - A connection is available, so it is safe to call TOPS_BeginSession.
//   2 - A true error has occurred, most likely the port is already in use.
//
// Usage Notes:
//   1.  There is nothing that prevents the user from using the TI Stack for
//       communication over the IR port rather than the serial port, (or for
//       using the Win95 stack for the serial port rather than the IR port).
//       However the original intent of the TI stack was to provide a
//       stand-alone stack for the wired port such that the Win95 stack and
//       the TI Stack could function simultaneosly on different ports.

// Function values returned by TOPS_CheckComm.
typedef enum TOPS_CheckCommReturnValuesDef
{
   TOPS_CheckCommFindFailure  = 0,
   TOPS_CheckCommSuccess      = 1,
   TOPS_CheckCommError        = 2
}TOPS_CheckCommReturnValues;

TOPS_CheckCommReturnValues DLLEXPORT TOPS_CheckComm(BYTE p_port,
                                                    IO_DeviceIdType p_device);

// restore previous alignment
#ifdef __BORLANDC__
#pragma option -a-.                                  // Borland restore
#else
#pragma pack( pop, alignment_before_tops_includes ) // MFC restore
#endif

#ifdef __cplusplus
}
#endif
#endif

//---------------------------------------------------------------------------
//  End of File: tops_pda.h
//---------------------------------------------------------------------------
