/***********************************************************************
*
* getdisk - Gets the disk image from the TI computer. Use in
*           conjunction with the 990 program ddisk.asm
*
* Changes:
*      06/30/04   DGP   Original
*
***********************************************************************/
 
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <termios.h>

#define NORMAL	0
#define ABORT	16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define MAXDISKS 7

typedef struct
{
   char *model;
   int cyls;
   int heads;
   int sectrk;
   int bytsec;
   int overhead;
} disk_types;


disk_types disks[MAXDISKS] =
{
   { "WD900",  815, 24, 38, 256,  0 },
   { "WD500",  150,  4, 32, 256,  0 },
   { "DS80",   803,  5, 61, 256,  0 },
   { "DS50",   815,  5, 38, 288,  0 },
   { "DS25",   408,  5, 38, 288,  0 },
   { "DS10",   408,  2, 20, 288, 96 },
   { "FD1000",  77,  2, 26, 288,  0 },
};

struct termios cmdtty, runtty;
int fd;

/*
** Terminal routines borrowed from Bub Supnik's simh
*/

int
ttinit (FILE *ifd)
{
   fd = fileno(ifd);
   if (tcgetattr (fd, &cmdtty) < 0) return -1;		/* get old flags */
   runtty = cmdtty;
   runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
   runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
   runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
   runtty.c_cc[VINTR] = 0;				/* interrupt */
   runtty.c_cc[VQUIT] = 0;				/* no quit */
   runtty.c_cc[VERASE] = 0;
   runtty.c_cc[VKILL] = 0;
   runtty.c_cc[VEOF] = 0;
   runtty.c_cc[VEOL] = 0;
   runtty.c_cc[VSTART] = 0;				/* no host sync */
   runtty.c_cc[VSUSP] = 0;
   runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
   runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   runtty.c_cc[VLNEXT] = 0;
#endif
   runtty.c_cc[VMIN] = 0;				/* no waiting */
   runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
   runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   runtty.c_cc[VSTATUS] = 0;
#endif
   return 0;
}

int
ttrunstate (void)
{
   runtty.c_cc[VINTR] = 0;
   if (tcsetattr (fd, TCSAFLUSH, &runtty) < 0) return -1;
   return 0;
}

int
ttcmdstate (void)
{
   if (tcsetattr (fd, TCSAFLUSH, &cmdtty) < 0) return -1;
   return 0;
}

int
ttclose (void)
{
   return ttcmdstate ();
}

int
poll_kbd (void)
{
   int status;
   unsigned char buf[1];

   status = read (fd, buf, 1);
   if (status != 1) return -1;
   else return (buf[0]);
}

/***********************************************************************
* dskwriteint - Write an integer.
***********************************************************************/

static int
dskwriteint (FILE *fd, int num)
{

   if (fputc ((num >> 24) & 0xFF, fd) < 0) return (-1);
   if (fputc ((num >> 16) & 0xFF, fd) < 0) return (-1);
   if (fputc ((num >> 8) & 0xFF, fd) < 0) return (-1);
   if (fputc (num & 0xFF, fd) < 0) return (-1);
   return (0);
}

/***********************************************************************
* dskreadint - Read an integer.
***********************************************************************/

static int
dskreadint (FILE *fd)
{
   int r;
   int i;

   r = 0;
   for (i = 0; i < 4; i++)
   {
      int c;
      if ((c = fgetc (fd)) < 0) return (-1);
      r = (r << 8) | (c & 0xFF);
   }
   return (r);
}

/***********************************************************************
* Main program.
***********************************************************************/

int
main(int argc, char **argv)
{
   int  *memory;
   FILE *iterm;
   FILE *oterm;
   FILE *ofd;
   int track = 0;
   int diskndx;
   int disksize;
   int i, k;
   int c;
   int tracksize;

   /*
   ** Check args
   */

   if (argc != 3)
   {
      fprintf (stderr, "usage: getdisk model file\n");
      exit (1);
   }

   /*
   ** Check disk model
   */

   for (diskndx = 0; diskndx < MAXDISKS; diskndx++)
   {
      if (!strcmp (disks[diskndx].model, argv[1])) break;
   }
   if (diskndx == MAXDISKS)
   {
      printf ("getdisk: Unknown disk model: %s\n", argv[1]);
      exit (ABORT);
   }

   /*
   ** Calculate the size of the disk and track for display
   */

   tracksize = (disks[diskndx].overhead + disks[diskndx].bytsec) *
	       disks[diskndx].sectrk;
   disksize = tracksize * disks[diskndx].heads * disks[diskndx].cyls;

   printf ("%s: disksize = %d, tracksize = %d\n\n",
	    argv[1], disksize, tracksize);

   /*
   ** Calculate the size of the disk and track less overhead for transfer
   */

   tracksize = disks[diskndx].bytsec * disks[diskndx].sectrk;
   disksize = tracksize * disks[diskndx].heads * disks[diskndx].cyls;

   /*
   ** Allocate the needed memory
   */

   if ((memory = (int *)malloc (disksize * sizeof(int))) == NULL)
   {
      printf ("getdisk: Can't malloc memory\n");
      exit (ABORT);
   }

   /*
   ** Open the disk image file
   */

   if ((ofd = fopen (argv[2], "wb")) == NULL)
   {
      perror ("Can't open file");
      exit (ABORT);
   }

   /*
   ** Write the disk geometry information
   */

   dskwriteint (ofd, disks[diskndx].cyls);
   dskwriteint (ofd, disks[diskndx].heads);
   dskwriteint (ofd, (disks[diskndx].overhead << 8) | disks[diskndx].sectrk);
   dskwriteint (ofd, disks[diskndx].bytsec);

   /*
   ** Set up the port
   */

   if ((iterm = fopen ("/dev/ttyS0", "rb")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }
   if ((oterm = fopen ("/dev/ttyS0", "w")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }

   ttinit (iterm);

   /*
   ** Tell the ddisk program to go
   */

   fputc ('\n', oterm);

   ttrunstate();

   /*
   ** Read the entire disk to memory
   */

   for (k = tracksize, i = 0; i < disksize; i++)
   {
      while ((c = poll_kbd()) == -1) ;
      memory[i] = c;
      k--;
      if (k == 0)
      {
	 printf ("track %d\n", track++);
	 k = tracksize;
	 /*
	 ** Send go ahead for next track
	 */
	 fputc ('\n', oterm);
      }
   }
   fputc ('\n', oterm); /* Send go ahead to finish */
   ttcmdstate();
   ttclose();

   /*
   ** Dump the disk image
   */

   for (k = disks[diskndx].bytsec, i = 0; i < disksize; i++)
   {
      if (fputc (memory[i], ofd) < 0)
      {
         perror ("Write error");
	 goto DIE;
      }

      /*
      ** If overhead is specified, pad out that amount
      */

      if (disks[diskndx].overhead)
      {
	 k--;
	 if (k == 0)
	 {
	    for (k = 0; k < disks[diskndx].overhead; k++)
	       if (fputc (0, ofd) < 0)
	       {
		  perror ("Write error");
		  goto DIE;
	       }
	    k = disks[diskndx].bytsec;
	 }
      }
   }

   /*
   ** Clean up and quit
   */

DIE:
   fclose (iterm);
   fclose (oterm);
   fclose (ofd);
   return (NORMAL);

}
