/***********************************************************************
*
* DUMPDISK - Dumps a TI990 tiline disk image file.
*
* Changes:
*      07/20/04   DGP   Adapted from dumpfpy.
*
***********************************************************************/
 
#include <stdio.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define DSKOVERHEAD 16
#define MAXDISKS 9

typedef struct
{
   char *model;
   int cyls;
   int heads;
   int sectrk;
   int bytsec;
   int overhead;
} disk_types;


disk_types disks[MAXDISKS+1] =
{
   { "WD900",  815, 24,  38, 256,  0 },
   { "WD500",  150,  4,  32, 256,  0 },
   { "DS300",  803, 19,  61, 256,  0 },
   { "DS200",  815, 19,  38, 288,  0 },
   { "DS80",   803,  5,  61, 256,  0 },
   { "DS50",   815,  5,  38, 288,  0 },
   { "DS25",   408,  5,  38, 288,  0 },
   { "DS10",   408,  2,  20, 288, 96 },
   { "FD1000",  77,  2,  26, 288,  0 },
   { "UNKNOWN",  0,  0,   0,   0,  0 }
};

#define HEXDUMP(file,ptr,size)	\
{\
   int jjj;\
   int iii;\
   char *tp;\
   char *cp;\
   for (iii = 0, tp = (char *)(ptr), cp = (char *)(ptr); iii < (size); )\
   {\
      fprintf ((file), "%04X  ", iii);\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (cp < ((char *)(ptr)+(size)))\
	 {\
	    fprintf ((file), "%02.2X", *cp++ & 0xFF);\
	    if (cp < ((char *)(ptr)+(size)))\
	    {\
	       fprintf ((file), "%02.2X ", *cp++ & 0xFF);\
	    }\
	    else\
	    {\
	       fprintf ((file), "   ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "     ");\
	 }\
	 iii += 2;\
      }\
      fprintf ((file), "   ");\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (tp < ((char *)(ptr)+(size)))\
	 {\
	    if (isprint(*tp))\
	       fprintf ((file), "%c", *tp);\
	    else\
	       fprintf ((file), ".");\
	    tp++;\
	    if (tp < ((char *)(ptr)+(size)))\
	    {\
	       if (isprint(*tp))\
		  fprintf ((file), "%c ", *tp);\
	       else\
		  fprintf ((file), ". ");\
	       tp++;\
	    }\
	    else\
	    {\
	       fprintf ((file), "  ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "   ");\
	 }\
      }\
      fprintf ((file), "\n");\
   }\
}

/***********************************************************************
* dskreadint - Read an integer.
***********************************************************************/

static int
dskreadint (FILE *fd)
{
   int r;
   int i;

   r = 0;
   for (i = 0; i < 4; i++)
   {
      int c;
      if ((c = fgetc (fd)) < 0) return (-1);
      r = (r << 8) | (c & 0xFF);
   }
   return (r);
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *fname;
   int dskloc;
   int i;
   int j;
   int cyls, heads, sectrk, overhead, sectlen;
   int tracklen;
   int head, cyl;
   int start, end;
   char buf[288];
   
   fname = NULL;
   cyl = 0;
   head = 0;
   start = 0;
   end = 0;
   for (i = 1; i < argc; i++)
   {
      switch (i)
      {
      case 1:
         fname = argv[i];
	 break;
      case 2:
         cyl = atoi(argv[i]);
	 break;
      case 3:
         head = atoi(argv[i]);
	 break;
      case 4:
         start = atoi(argv[i]);
	 end = start;
	 break;
      case 5:
         end = atoi(argv[i]);
	 break;
      default:
      USAGE:
	 fprintf (stderr,
	    "usage: dumpdisk file [cyl head [startsec [endsec]]]\n");
	 exit (1);
      }
   }
   if (fname == NULL) goto USAGE;

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (fname, "rb")) == NULL)
   {
      perror ("Can't open disk file");
      exit (1);
   }

   /*
   ** Read the disk geometry
   */

   if ((cyls = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (ABORT);
   }
   if ((heads = dskreadint (fd)) < 0) 
   {
      perror ("Can't read disk geometry");
      exit (ABORT);
   }
   if ((dskloc = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (ABORT);
   }
   sectrk = dskloc & 0xFF;
   overhead = (dskloc >> 8) & 0xFF;
   if ((sectlen = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (ABORT);
   }

   tracklen = sectrk * (sectlen + overhead);

   for (i = 0; i < MAXDISKS; i++)
   {
      if (disks[i].cyls == cyls &&
	  disks[i].heads == heads &&
	  disks[i].sectrk == sectrk &&
	  disks[i].bytsec == sectlen &&
	  disks[i].overhead == overhead)
         break;
   }
   if (i == MAXDISKS)
   {
      printf ("*Unknown Disk model*\n");
   }
	       
   printf ("%s disk geometry:\n", disks[i].model);
   printf ("   cyls      = %d\n", cyls);
   printf ("   heads     = %d\n", heads);
   printf ("   sectrk    = %d\n", sectrk);
   printf ("   sectlen   = %d\n", sectlen);
   printf ("   overhead  = %d\n", overhead);
   printf ("   tracklen  = %d\n", tracklen);
   printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

   if (cyl >= cyls)
   {
      printf ("Invalid disk cylinder specified, abort\n");
      exit (ABORT);
   }
   if (head >= heads)
   {
      printf ("Invalid disk head specified, abort\n");
      exit (ABORT);
   }
   if (start >= sectrk)
   {
      printf ("Invalid disk start sector specified, abort\n");
      exit (ABORT);
   }
   if (end >= sectrk)
   {
      printf ("Invalid disk end sector specified, abort\n");
      exit (ABORT);
   }

   for (j = start; j <= end; j++)
   {
      /*
      ** Calculate location
      */

      dskloc = (tracklen * head) + (tracklen * heads * cyl) +
	 ((overhead + sectlen) * j) + DSKOVERHEAD;

#ifdef DEBUG
      printf ("   cyl = %d, head = %d, sector = %d\n",
	       cyl, head, start);
      printf ("   dskloc = %d\n", dskloc);
#endif

      /*
      ** Go to the specified location on disk
      */

      if (fseek (fd, dskloc, SEEK_SET) < 0)
      {
	 perror ("seek failed");
	 exit (ABORT);
      }

      /*
      ** Read sector
      */

      for (i = 0; i < sectlen; i++)
	 buf[i] = fgetc (fd);
      /*
      ** Dump it
      */
      printf ("\ncyl %d head %d sect %d dskloc = %d:\n", cyl, head, j, dskloc);
      HEXDUMP (stdout, buf, sectlen);
   }

   return (NORMAL);
}
