/***********************************************************************
*
* simops.c - Simulate instruction operations for the TI 990 computer.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/22/03   DGP   Fixed XOP REG function.
*   07/08/03   DGP   Fixed status on Subtract and indexing with negative
*                    values.
*   07/09/03   DGP   Forced PC and WP to be on even boundry.
*   07/13/03   DGP   Forced word and address access to even boundry.
*   08/07/03   DGP   Fixed carry on Add and Add Byte.
*                    Fixed MPY and DIV (had signed problems).
*                    Fixed B Rn (branch to register).
*   08/12/03   DGP   Added Map file instructions and priv checks.
*   11/26/03   DGP   Fixup OVER & CARRY status checking.
*   12/10/03   DGP   Streamline TILINE & ROM access with 20 bits.
*   01/26/04   DGP   Added memory size option.
*   04/20/04   DGP   Changed SLA to detect sign change during shift.
*   05/06/04   DGP   Fix LREX to run in 990/10 mode.
*   06/14/04   DGP   Added floating point instructions.
*   06/16/04   DGP   Added extended /12 instructions.
*   07/13/04   DGP   Added /12 memory protection checks
*   03/30/05   DGP   Fixed XOP for Mapping.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 opcreg;	/* The old program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */
extern uint32 mpcreg;	/* Mapped PC */
extern int breakaddr;   /* The break point address */
extern long pancount;	/* Instructions executed for panel */
extern unsigned long instcount;

extern int run;
extern int idle;
extern int runled;
extern int idleled;
extern int model;
extern int breakflag;
extern int enablepanel;
extern int mapenabled;
extern int intsenabled;
extern int traceenable;
extern int tracestart;
extern int traceend;
extern int tracelim[2];
extern int deferint;
extern int traceit;
extern uint16 mapcrudata;
extern uint16 curmapindex;
extern uint16 errcrudata;
extern uint16 curinst;
extern uint32 memlen;
extern uint32 maplatch;
extern long delayclock;
extern char view[MAXVIEW][81];
extern FILE *tracefd;
extern uint32 tracemem[16];
extern uint32 errtrace[16];
extern int errindex;
extern int trcindex;

extern uint8 memory[MEMSIZE];
extern MapFile mapfile[MAPSIZE];

static void runinst (uint16);
static int use_LDD_map = FALSE;
static int use_LDS_map = FALSE;
static int use_map_cnt = 0;
static int tiline = FALSE;
static int inhibitwrite = FALSE;

/***********************************************************************
* decodeinst - Decode instructions.
***********************************************************************/

static int
decodeinst (uint16 inst)
{
   if (inst & 0xF000)
   {
      if (inst & 0xC000) return TYPE_1;
      if ((inst & 0xF800) == 0x3000) return TYPE_4;
      if ((inst & 0xF000) == 0x3000) return TYPE_9;
      if ((inst & 0xF000) == 0x2000) return TYPE_3;
      return TYPE_2;
   }
   else if (inst & 0x0F00)
   {
      if ((inst & 0x0F00) == 0x0F00) return TYPE_6;
      else if ((inst & 0x0F00) == 0xE00)
      {
	 if ((inst & 0x0FF0) == 0x0E00) return TYPE_15;
	 if ((inst & 0x0FC0) == 0x0E00) return TYPE_12;
	 return TYPE_6;
      }
      else if ((inst & 0x0F00) == 0x0D00) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0C00)
      {
	 if ((inst & 0x0FFF) == 0x0C0F) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C0E) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C08) return TYPE_11;
	 if ((inst & 0x0FFE) == 0x0C0C) return TYPE_17;
	 if ((inst & 0x0FF8) == 0x0C08) return TYPE_14;
	 if ((inst & 0x0FF0) == 0x0C00) return TYPE_7;
	 if ((inst & 0x0FC0) == 0x0C00) return TYPE_16;
	 return TYPE_6;
      }
      else if ((inst & 0x0C00) == 0x0800) return TYPE_5;
      else if ((inst & 0x0C00) == 0x0400) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0300)
      {
	 if (inst == 0x0300) return TYPE_8;
	 if ((inst & 0x0FF0) == 0x03F0) return TYPE_21;
	 if ((inst & 0x0FE0) == 0x0320) return TYPE_10;
	 return TYPE_7;
      }
      else if ((inst & 0x0F00) == 0x0200) return TYPE_8;
      else if ((inst & 0x0F00) == 0x0100) return TYPE_6;
   }
   else if (inst & 0x00F0)
   {
      if ((inst & 0x00F0) == 0x00B0) return TYPE_8;
      else if ((inst & 0x00F0) == 0x0070) return TYPE_18;
      else if ((inst & 0x0040) == 0x0040) return TYPE_12;
      else if ((inst & 0x00F0) == 0x0020)
      {
	 if ((inst & 0x00FF) == 0x0021) return TYPE_20;
	 if ((inst & 0x00FF) == 0x0022) return TYPE_20;
	 if ((inst & 0x00FF) == 0x002B) return TYPE_19;
	 if ((inst & 0x00FC) == 0x002C) return TYPE_7;
	 if ((inst & 0x00FC) == 0x0028) return TYPE_11;
	 return TYPE_11;
      }
      else if ((inst & 0x00F0) == 0x0010)
      {
	 if ((inst & 0x00FE) == 0x001E) return TYPE_11;
	 return TYPE_13;
      }
      return TYPE_18;
   }
   return (TYPE_ILL);
}


/***********************************************************************
* tracememory - Trace memory data
***********************************************************************/

void
tracememory (uint32 code, uint32 ma)
{
   if (model < 12) return;

   errtrace[errindex] = code | (ma & 0x1FFFFE);
   errindex++;
   if (errindex == 16) errindex = 0;
}

/***********************************************************************
* mapaddr - Map address to a 22 bit physical address.
***********************************************************************/

uint32
mapaddr (uint16 pa, uint16 mf, uint16 prot, int instaccess)
{
   int limreg;
   uint32 ma;
   uint16 latchmask;
   uint16 limit;
   uint16 base;

   if (!mapenabled)
   {
      ma = pa;
      if (pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	    tiline = TRUE;
         return (ma);
      }
   }

   else
   {

      if (mf >= MAPSIZE)
      {
	 inhibitwrite = TRUE;
	 if (GET_MASK >= ERRINT)
	 {
	    errcrudata |= ERRMAP;
	    geninterrupt (ERRINT);
	 }
#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	 fprintf (stderr, "%d: Map %d error pa = >%04X\n", instcount, mf, pa);
#endif
	 return (0xFFFFFFFF);
      }
      
      if ((mf == 0) && (pa >= TILINESTART))
      {
	 ma = TPCSSTART + pa;
	 if (pa < TILINEEND)
	    tiline = TRUE;
         return (ma);
      }

      if ((pa & 0xFFE0) <= (~mapfile[mf].l1 & 0xFFE0))
      {
	 limreg = 1;
	 limit = mapfile[mf].l1;
	 base = mapfile[mf].b1;
	 latchmask = MAP_LATCH1;
      }

      else if ((pa & 0xFFE0) <= (~mapfile[mf].l2 & 0xFFE0))
      {
	 limreg = 2;
	 limit = mapfile[mf].l2;
	 base = mapfile[mf].b2;
	 latchmask = MAP_LATCH2;
      }

      else if ((pa & 0xFFE0) <= (~mapfile[mf].l3 & 0xFFE0))
      {
	 limreg = 3;
	 limit = mapfile[mf].l3;
	 base = mapfile[mf].b3;
	 latchmask = MAP_LATCH3;
      }

      else
      {
#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	 int wr = mapaddr (wpreg, GET_MAP, MEM_NOPROT, FALSE);
	 int pr = mapaddr (pcreg-2, GET_MAP, MEM_NOPROT, FALSE);
	 fprintf (stderr, "%d: Map %d error >L3 pa = >%04X pa&>FFE0 = >%04X\n",
		  instcount, mf, pa, pa & 0xFFE0);
	 fprintf (stderr,
	       "   pcreg = >%04X(%06X), wpreg = >%04X(%06X), statreg = >%04X\n",
		  pcreg-2, pr, wpreg, wr, statreg);
	 fprintf (stderr, "Code:\n");
	 HEXDUMP (stderr, &memory[pr-16], 32, pr-16);
	 fprintf (stderr, "WS:\n");
	 HEXDUMP (stderr, &memory[wr], 32, wr);
	 fprintf (stderr,
	    "   mapfile[%d].l1 = >%04X ~l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	       mf, mapfile[mf].l1, (~mapfile[mf].l1 & 0xFFE0),
	       mf, mapfile[mf].b1);
	 fprintf (stderr,
	    "   mapfile[%d].l2 = >%04X ~l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	       mf, mapfile[mf].l2, (~mapfile[mf].l2 & 0xFFE0),
	       mf, mapfile[mf].b2);
	 fprintf (stderr,
	    "   mapfile[%d].l3 = >%04X ~l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	       mf, mapfile[mf].l3, (~mapfile[mf].l3 & 0xFFE0),
	       mf, mapfile[mf].b3);
#endif
	 inhibitwrite = TRUE;
	 if (!maplatch)
	 {
	    maplatch = pa;
#ifdef DEBUGMAPPER
            fprintf (stderr, "   maplatch = >%04X\n", maplatch);
#endif
	 }
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_MAPERR, pa);
	    errcrudata |= ERRMAP;
	    geninterrupt (ERRINT);
	 }
	 return (0xFFFFFFFF);
      }

      ma = base & 0xFFFF;
      ma = (((ma << 5) + pa) & 0x1FFFFF);
#ifdef DEBUGMAPPER
      if (traceit)
	 fprintf (tracefd,
    "\n%d: Map %d <L%d pa = >%04X, base = >%04X, ~limit = >%04X, ma = %06X\n",
		  instcount, mf, limreg, pa, base, ~limit & 0xFFE0, ma);
#endif
      if (!maplatch && ((mf == 2) || (mapcrudata & latchmask)))
      {
	 maplatch = ma;
#ifdef DEBUGMAPPER
            fprintf (stderr,
		  "%d:   latchmask = >%04X, maplatch = >%06X, pa = >%04X\n",
		     instcount, latchmask, maplatch, pa);
#endif
      }
      if (model >= 12)
      {
	 if (IS_PROT && ((limit & 0x3) & prot))
	 {
#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	    fprintf (stderr,
		     "%d: ProtViolation Map %d L%d prot = >%X limit = >%X\n",
		     instcount, mf, limreg, prot, limit & 0x3);
#endif
	    inhibitwrite = TRUE;
	    if (!maplatch)
	    {
	       maplatch = ma;
#ifdef DEBUGMAPPER
            fprintf (stderr, "   maplatch = >%06X\n", ma);
#endif
	    }
	    if (GET_MASK >= ERRINT)
	    {
	       errcrudata |= prot == 0x2 ? ERREXEC : ERRWRIT;
	       tracememory (prot == 0x2 ? TRACE_EXECVIO : TRACE_WRITVIO, ma);
	       geninterrupt (ERRINT);
	    }
	    return (0xFFFFFFFF);
	 }
      }

      if (ma >= memlen)
      {
#if 0
	 /*
	 ** Check if the OS is addressing the TILINE using MF 2 
	 ** It's looking for ECC or Cache controllers
	 */

	 if (!IS_NONPRV && mf == 2 && ma >= (TPCSSTART + TILINESTART))
	 {
	    if (ma < (TPCSSTART + TILINEEND))
	       tiline = TRUE;
	    return (ma);
	 }
#endif

	 /*
	 ** Otherwise, a true memory exception
	 */

#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	 fprintf (stderr,
"%d: >memlen Map %d <L%d pa = >%04X, base = >%04X, ~limit = >%04X, ma = %06X\n",
		  instcount, mf, limreg, pa, base, ~limit & 0xFFE0, ma);
	 fprintf (stderr, "   stat = >%04X\n", statreg);
#endif
	 inhibitwrite = TRUE;
	 if (!maplatch)
	 {
	    maplatch = ma;
#ifdef DEBUGMAPPER
            fprintf (stderr, "   maplatch = >%06X\n", ma);
#endif
	 }
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_TLTIME, ma);
	    errcrudata |= ERRTIT;
	    geninterrupt (ERRINT);
	 }
	 return (0xFFFFFFFF);
      }

   }

   return (ma);

}

/***********************************************************************
* getinst - Get an Instruction.
***********************************************************************/

uint16
getinst ()
{
   uint32 ma;

   if (model < 10)
   {
      ma = pcreg;
      if (model == 5)
      {
         if (ma >= ROMSTART)
	    ma += TPCSSTART;
      }
   }
   else
   {
      if (use_LDD_map || use_LDS_map) curmapindex = 2;
      else curmapindex = GET_MAP;
      ma = mapaddr (pcreg, GET_MAP, MEM_NOEXEC, TRUE);
      if (ma == 0xFFFFFFFF) return (0xFFFF);
      tracememory (TRACE_INSTFETCH, ma);
   }
   mpcreg = ma;

   return (GETMEM0 (ma));
}

/***********************************************************************
* getreg - Get a register value.
***********************************************************************/

uint16
getreg (uint16 r)
{
   uint16 pa;
   uint32 ma;

   pa = wpreg + (r * 2);
   if (model < 10)
   {
      ma = pa;
   }
   else
   {
      ma = mapaddr (pa, GET_MAP, MEM_NOPROT, FALSE);
      if (ma == 0xFFFFFFFF) return (0xFFFF);
      tracememory (TRACE_WSACCESS, ma);
   }

   return (GETMEM0 (ma));
}

/***********************************************************************
* putreg - Put a value into a register.
***********************************************************************/

void
putreg (uint16 r, uint16 v)
{
   uint16 pa;
   uint32 ma;

   pa = wpreg + (r * 2);
   if (model < 10)
   {
      ma = pa;
   }
   else
   {
      ma = mapaddr (pa, GET_MAP, MEM_NOWRITE, FALSE);
      tracememory (TRACE_WSACCESS | TRACE_WSRW, ma);
   }

   if (!inhibitwrite)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* getmem - Get a word from memory.
***********************************************************************/

uint16
getmem (uint16 pa, int srcdst)
{
   int tr;
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (model < 10)
   {
      ma = pa & 0xFFFE;
      if (model == 5)
      {
	 if (pa >= TILINESTART)
	 {
	    ma += TPCSSTART;
	    if (pa < TILINEEND)
	       tiline = TRUE;
	 }
      }
   }
   else
   {
      mr = 0;
      if (srcdst >= 0)
      {
	 if (srcdst) mr = use_LDS_map;
	 else mr = use_LDD_map;
      }
      if (mr) mf = 2;
      else mf = GET_MAP;
      ma = mapaddr (pa & 0xFFFE, mf, MEM_NOPROT, FALSE);
      if (ma == 0xFFFFFFFF) return (0xFFFF);
   }

   if (tiline)
   {
      tr = gettiline (ma, FALSE);
      tracememory (TRACE_TLACCESS, ma);
      return (tr & 0xFFFF);
   }

   return (GETMEM0 (ma));
}

/***********************************************************************
* getmemb - Get a byte from memory.
***********************************************************************/

uint8
getmemb (uint16 pa, int srcdst)
{
   int tr;
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (model < 10)
   {
      ma = pa;
      if (model == 5)
      {
	 if (pa >= TILINESTART)
	 {
	    ma += TPCSSTART;
	    if (pa < TILINEEND)
	       tiline = TRUE;
	 }
      }
   }
   else
   {
      mr = 0;
      if (srcdst) mr = use_LDS_map;
      else mr = use_LDD_map;
      if (mr) mf = 2;
      else mf = GET_MAP;
      ma = mapaddr (pa, mf, MEM_NOPROT, FALSE);
      if (ma == 0xFFFFFFFF) return (0xFF);
   }

   if (tiline)
   {
      tracememory (TRACE_TLACCESS, ma);
      tr = gettiline (ma, TRUE);
      return (tr & 0xFF);
   }

   return (GETMEMB0 (ma));
}

/***********************************************************************
* putmem - Put a word in memory.
***********************************************************************/

void
putmem (uint16 pa, uint16 v, int srcdst)
{
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (model < 10)
   {
      if (model == 5 && (pa >= TILINESTART && pa < TILINEEND))
      {
         tiline = TRUE;
	 puttiline (TPCSSTART + pa, v, FALSE);
      }
      else if (pa < memlen)
      {
	 ma = pa & 0xFFFE;
	 PUTMEM0 (ma, v);
      }
   }
   else
   {
      mr = 0;
      if (srcdst) mr = use_LDS_map;
      else mr = use_LDD_map;
      if (mr) mf = 2;
      else mf = GET_MAP;

      ma = mapaddr (pa & 0xFFFE, mf, MEM_NOWRITE, FALSE);
      if (ma == 0xFFFFFFFF) return;
      if (ma >= (TPCSSTART + ROMSTART)) return;

      if (tiline)
      {
	 tracememory (TRACE_TLRW | TRACE_TLACCESS, ma);
	 puttiline (ma, v, FALSE);
      }
      else if (!inhibitwrite)
      {
	 PUTMEM0 (ma, v);
      }
   }

}

/***********************************************************************
* putmemb - Put a byte in memory.
***********************************************************************/

void
putmemb (uint16 pa, uint8 v, int srcdst)
{
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (model < 10)
   {
      ma = pa;
      if (model == 5 && (pa >= TILINESTART && pa < TILINEEND))
      {
	 uint16 uv = v;
         tiline = TRUE;
	 puttiline (TPCSSTART + pa, uv, TRUE);
      }
      else if (ma < memlen)
      {
	 PUTMEMB0 (ma, v);
      }
   }
   else
   {
      mr = 0;
      if (srcdst) mr = use_LDS_map;
      else mr = use_LDD_map;
      if (mr) mf = 2;
      else mf = GET_MAP;

      ma = mapaddr (pa, mf, MEM_NOWRITE, FALSE);
      if (ma == 0xFFFFFFFF) return;
      if (ma >= (TPCSSTART + ROMSTART)) return;

      if (tiline)
      {
	 uint16 uv = v;

	 tracememory (TRACE_TLRW | TRACE_TLACCESS, ma);
	 puttiline (ma, uv, TRUE);
      }
      else if (!inhibitwrite)
      {
	 PUTMEMB0 (ma, v);
      }
   }

}

/***********************************************************************
* getaddr - Get an effective address.
***********************************************************************/

uint16
getaddr (uint16 sr, uint16 st, int inc)
{
   uint16 sa = 0;

   switch (st)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", sr);
      sa = wpreg + (sr * 2);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr);
      break;

   case 2:

      sa = GETINST;
      if (traceit)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%04X", sa);
	 else
	    fprintf (tracefd, " @>%04X(%d)", sa, sr);
      }
      pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr);
      if (inc) PUTREG (sr, sa + 2);
      break;
   }
   if (traceit)
      fprintf (tracefd, " sa=>%04X", sa);
   return (sa);
}

/***********************************************************************
* getbyte - Get a byte operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

uint8
getbyte (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint8 sval;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      sval = (GETREG (sr) & 0xFF00) >> 8;
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr);
      sval = GETMEMB (sa, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%04X", sa);
	 else
	    fprintf (tracefd, " @>%04X(%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sval = GETMEMB (sa, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr);
      sval = GETMEMB (sa, srcdst);
      if (inc) PUTREG (sr, sa + 1);
      break;
   }
   return (sval);
}

/***********************************************************************
* getword - Get a word operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

uint16
getword (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint16 sval;

   switch (st)
   {
   case 0:

      if (traceit && inc)
	 fprintf (tracefd, " R%d", sr);
      sval = GETREG (sr);
      break;

   case 1:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d", sr);
      sa = GETREG (sr);
      sval = GETMEM (sa & 0xFFFE, srcdst);
      break;

   case 2:

      sa = GETINST;
      if (traceit && inc)
      {
	 if (sr == 0)
	    fprintf (tracefd, " @>%04X", sa);
	 else
	    fprintf (tracefd, " @>%04X(R%d)", sa, sr);
      }
      if (inc) pcreg += 2;
      if (sr != 0) (int16)sa += (int16)GETREG (sr);
      sval = GETMEM (sa & 0xFFFE, srcdst);
      break;

   case 3:

      if (traceit && inc)
	 fprintf (tracefd, " *R%d+", sr);
      sa = GETREG (sr);
      sval = GETMEM (sa & 0xFFFE, srcdst);
      if (inc) PUTREG (sr, sa + 2);
      break;
   }
   return (sval);
}

/***********************************************************************
* putaddr - Bump address by count
***********************************************************************/

void
putaddr (uint16 sr, uint16 st, uint16 inc)
{
   if (st == 3)
   {
      PUTREG (sr, GETREG (sr) + inc);
   }
}

/***********************************************************************
* putbyte - Put a byte operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putbyte (uint16 dr, uint16 dt, uint8 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      da = (GETREG (dr) & 0x00FF) | (dval << 8);
      PUTREG (dr, da);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr);
      PUTMEMB (da, dval, srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%04X", da);
	 else
	    fprintf (tracefd, " @>%04X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      PUTMEMB (da, dval, srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr);
      PUTREG (dr, da + 1);
      PUTMEMB (da, dval, srcdst);
      break;
   }
}

/***********************************************************************
* putword - Put a word operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

void
putword (uint16 dr, uint16 dt, uint16 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      if (traceit)
	 fprintf (tracefd, " R%d", dr);
      PUTREG (dr, dval);
      break;

   case 1:

      if (traceit)
	 fprintf (tracefd, " *R%d", dr);
      da = GETREG (dr);
      PUTMEM (da, dval, srcdst);
      break;

   case 2:

      da = GETINST;
      if (traceit)
      {
	 if (dr == 0)
	    fprintf (tracefd, " @>%04X", da);
	 else
	    fprintf (tracefd, " @>%04X(%d)", da, dr);
      }
      pcreg += 2;
      if (dr != 0) (int16)da += (int16)GETREG (dr);
      PUTMEM (da, dval, srcdst);
      break;

   case 3:

      if (traceit)
	 fprintf (tracefd, " *R%d+", dr);
      da = GETREG (dr);
      PUTREG (dr, da + 2);
      PUTMEM (da, dval, srcdst);
      break;
   }
}

/***********************************************************************
* chkaddword - Check addition for carry and overflow
***********************************************************************/

static void
chkaddword (int32 acc, int16 sv1, int16 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((acc & 0xFFFF) < (sv1 & 0xFFFF)) SET_CARRY;

   if (((sv1 & 0x8000) == (sv2 & 0x8000)) &&
       ((sv2 & 0x8000) != (acc & 0x8000))) SET_OVER;
}

/***********************************************************************
* chkaddbyte - Check addition for carry and overflow
***********************************************************************/

static void
chkaddbyte (int32 acc, int8 bsv1, int8 bsv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((acc & 0xFF) < (bsv1 & 0xFF)) SET_CARRY;

   if (((bsv1 & 0x80) == (bsv2 & 0x80)) &&
       ((bsv2 & 0x80) != (acc & 0x80))) SET_OVER;
}

/***********************************************************************
* chksubword - Check subtraction for carry and overflow
***********************************************************************/

static void
chksubword (int32 acc, int16 sv1, int16 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((uint16)sv2 >= (uint16)sv1) SET_CARRY;

   if (((sv1 & 0x8000) != (sv2 & 0x8000)) &&
       ((sv2 & 0x8000) != (acc & 0x8000))) SET_OVER;
}

/***********************************************************************
* chksubbyte - Check subtraction for carry and overflow
***********************************************************************/

static void
chksubbyte (int32 acc, int8 bsv1, int8 bsv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((uint8)bsv2 >= (uint8)bsv1) SET_CARRY;

   if (((bsv1 & 0x80) != (bsv2 & 0x80)) &&
       ((bsv2 & 0x80) != (acc & 0x80))) SET_OVER;
}

/***********************************************************************
* chkparity - Check parity of a byte.
***********************************************************************/

static void
chkparity (uint8 bval)
{
   int c;
   int i;

   for (c = 0, i = 0; i < 8; i++)
   {
      c = c + (bval & 0x01);
      bval = bval >> 1;
   }
   if (c & 0x01) SET_ODDP;
   else CLR_ODDP;
}

/***********************************************************************
* cmpbytezero - Compare byte value to zero.
***********************************************************************/

void
cmpbytezero (uint8 bval)
{
   if (bval > 0) SET_LGT;
   else CLR_LGT;
   if ((int8)bval > 0) SET_AGT;
   else CLR_AGT;
   if (bval == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* cmpwordzero - Compare value to zero.
***********************************************************************/

void
cmpwordzero (uint16 val)
{
   if (val > 0) SET_LGT;
   else CLR_LGT;
   if ((int16)val > 0) SET_AGT;
   else CLR_AGT;
   if (val == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* proc1op - Process type 1 operations.
***********************************************************************/

static int
proc1op (uint16 inst)
{
   int32 acc;
   uint16 st;
   uint16 sr;
   uint16 val;
   uint16 cval;
   uint16 dt;
   uint16 dr;
   uint16 op;
   int16 sv1, sv2;
   uint16 uv1;
   uint8 bval;
   uint8 bcval;
   int8 bsv1, bsv2;
   uint8 buv1;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   dr = (inst & 0x03C0) >> 6;
   dt = (inst & 0x0C00) >> 10;
   op = (inst & 0xF000) >> 12;

   switch ((inst & 0xF000) >> 12)
   {
   case 4: /* SZC */

      if (traceit) fprintf (tracefd, "SZC");
      cval = getword (sr, st, TRUE, SRC);
      uv1 = getword (dr, dt, FALSE, DST);
      val = uv1 & (~cval);
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " >%04X & >%04X = >%04X", cval, uv1, val);
      break;

   case 5: /* SZCB */

      if (traceit) fprintf (tracefd, "SZCB");
      bcval = getbyte (sr, st, TRUE, SRC);
      buv1 = getbyte (dr, dt, FALSE, DST);
      bval = buv1 & (~bcval);
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      if (traceit)
	 fprintf (tracefd, " >%02X & >%02X = >%02X", bcval, buv1, bval);
      break;

   case 6: /* S */

      if (traceit) fprintf (tracefd, "S");
      sv1 = (int16)getword (sr, st, TRUE, SRC);
      acc = sv1;
      sv2 = (int16)getword (dr, dt, FALSE, DST);
      acc = sv2 - acc;
      uv1 = acc & 0xFFFF;
      acc = (int16)uv1;
      chksubword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      if (traceit)
         fprintf (tracefd, " >%04X - >%04X -> >%04X", sv2, sv1, acc);
      break;

   case 7: /* SB */

      if (traceit) fprintf (tracefd, "SB");
      bsv1 = (int8)getbyte (sr, st, TRUE, SRC);
      acc = bsv1;
      bsv2 = (int8)getbyte (dr, dt, FALSE, DST);
      acc = bsv2 - acc;
      buv1 = acc & 0xFF;
      acc = (int8)buv1;
      chksubbyte (acc, bsv1, bsv2);
      bval = acc &0xFF;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      if (traceit)
         fprintf (tracefd, " >%02X - >%02X -> >%02X", bsv2, bsv1, bval);
      break;

   case 8: /* C */

      if (traceit) fprintf (tracefd, "C");
      val = getword (sr, st, TRUE, SRC);
      cval = getword (dr, dt, TRUE, DST);
      if (traceit) fprintf (tracefd, " >%04X == >%04X", val, cval);
      if (val > cval) SET_LGT;
      else CLR_LGT;
      if ((int16)val > (int16)cval) SET_AGT;
      else CLR_AGT;
      if (val == cval) SET_EQ;
      else CLR_EQ;
      break;

   case 9: /* CB */

      if (traceit) fprintf (tracefd, "CB");
      bval = getbyte (sr, st, TRUE, SRC);
      bcval = getbyte (dr, dt, TRUE, DST);
      if (traceit) fprintf (tracefd, " >%02X == >%02X", bval, bcval);
      if (bval > bcval) SET_LGT;
      else CLR_LGT;
      if ((int8)bval > (int8)bcval) SET_AGT;
      else CLR_AGT;
      if (bval == bcval) SET_EQ;
      else CLR_EQ;
      chkparity (bval);
      break;

   case 10: /* A */

      if (traceit) fprintf (tracefd, "A");
      sv1 = (int16)getword (sr, st, TRUE, SRC);
      acc = sv1;
      sv2 = (int16)getword (dr, dt, FALSE, DST);
      acc = acc + sv2;
      uv1 = acc & 0xFFFF;
      acc = (int16)uv1;
      chkaddword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      if (traceit)
         fprintf (tracefd, " >%04X + >%04X -> >%04X", sv1, sv2, acc);
      break;

   case 11: /* AB */

      if (traceit) fprintf (tracefd, "AB");
      bsv1 = (int8)getbyte (sr, st, TRUE, SRC);
      acc = bsv1;
      bsv2 = (int8)getbyte (dr, dt, FALSE, DST);
      acc = acc + bsv2;
      buv1 = acc & 0xFF;
      acc = (int8)buv1;
      chkaddbyte (acc, bsv1, bsv2);
      bval = acc & 0xFF;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      if (traceit)
         fprintf (tracefd, " >%04X + >%04X -> >%04X", bsv1, bsv2, bval);
      break;

   case 12: /* MOV */

      if (traceit) fprintf (tracefd, "MOV");
      val = getword (sr, st, TRUE, SRC);
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      break;

   case 13: /* MOVB */

      if (traceit) fprintf (tracefd, "MOVB");
      bval = getbyte (sr, st, TRUE, SRC);
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      if (traceit)
	 fprintf (tracefd, " bval=>%02X", bval);
      break;

   case 14: /* SOC */

      if (traceit) fprintf (tracefd, "SOC");
      cval = getword (sr, st, TRUE, SRC);
      uv1 = getword (dr, dt, FALSE, DST);
      val = cval | uv1;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " >%04X | >%04X = >%04X", cval, uv1, val);
      break;

   case 15: /* SOCB */

      if (traceit) fprintf (tracefd, "SOCB");
      bcval = getbyte (sr, st, TRUE, SRC);
      buv1 = getbyte (dr, dt, FALSE, DST);
      bval = bcval | buv1;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      if (traceit)
	 fprintf (tracefd, " >%02X | >%02X = >%02X", bcval, buv1, bval);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc2op - Process type 2 operations.
***********************************************************************/

static int
proc2op (uint16 inst)
{
   int16 wpc;
   int8 disp;

   disp = inst & 0x00FF;
   wpc = (pcreg >> 1) & 0x7FFF;

   switch ((inst & 0x0F00) >> 8)
   {
   case 0: /* JMP */

      if (traceit) fprintf (tracefd, "JMP >%X", disp & 0xFF);
      break;

   case 1: /* JLT */

      if (traceit) fprintf (tracefd, "JLT >%X", disp & 0xFF);
      if (!IS_AGT && !IS_EQ) break;
      return (0);

   case 2: /* JLE */

      if (traceit) fprintf (tracefd, "JLE >%X", disp & 0xFF);
      if (!IS_LGT || IS_EQ) break;
      return (0);

   case 3: /* JEQ */

      if (traceit) fprintf (tracefd, "JEQ >%X", disp & 0xFF);
      if (IS_EQ) break;
      return (0);

   case 4: /* JHE */

      if (traceit) fprintf (tracefd, "JHE >%X", disp & 0xFF);
      if (IS_LGT || IS_EQ) break;
      return (0);

   case 5: /* JGT */

      if (traceit) fprintf (tracefd, "JGT >%X", disp & 0xFF);
      if (IS_AGT) break;
      return (0);

   case 6: /* JNE */

      if (traceit) fprintf (tracefd, "JNE >%X", disp & 0xFF);
      if (!IS_EQ) break;
      return (0);

   case 7: /* JNC */

      if (traceit) fprintf (tracefd, "JNC >%X", disp & 0xFF);
      if (!IS_CARRY) break;
      return (0);

   case 8: /* JOC */

      if (traceit) fprintf (tracefd, "JOC >%X", disp & 0xFF);
      if (IS_CARRY) break;
      return (0);

   case 9: /* JNO */

      if (traceit) fprintf (tracefd, "JNO >%X", disp & 0xFF);
      if (!IS_OVER) break;
      return (0);

   case 10: /* JL */

      if (traceit) fprintf (tracefd, "JL >%X", disp & 0xFF);
      if (!IS_LGT && !IS_EQ) break;
      return (0);

   case 11: /* JH */

      if (traceit) fprintf (tracefd, "JH >%X", disp & 0xFF);
      if (IS_LGT && !IS_EQ) break;
      return (0);

   case 12: /* JOP */

      if (traceit) fprintf (tracefd, "JOP >%X", disp & 0xFF);
      if (IS_ODDP) break;
      return (0);

   case 13: /* SBO */

      if (traceit) fprintf (tracefd, "SBO >%X", disp & 0xFF);
      if (traceit) fprintf (tracefd, " BASE >%04X", R12);
      setbitone (inst);
      return (0);

   case 14: /* SBZ */

      if (traceit) fprintf (tracefd, "SBZ >%X", disp & 0xFF);
      if (traceit) fprintf (tracefd, " BASE >%04X", R12);
      setbitzero (inst);
      return (0);

   case 15: /* TB */

      if (traceit) fprintf (tracefd, "TB >%X", disp & 0xFF);
      if (traceit) fprintf (tracefd, " BASE >%04X", R12);
      testbit (inst);
      return (0);

   default:
      return (-1);
   }

   /*
   ** Fall through, take the jump
   */

   if ((uint8)disp == 0xFF)
   {
      idle = TRUE;
      if (GET_MASK == 0) run = FALSE;
   }
   else
      opcreg = pcreg;

   wpc += disp;
   pcreg = (wpc << 1) & 0xFFFE;
   return (0);
}

/***********************************************************************
* proc3op - Process type 3 operations.
***********************************************************************/

static int
proc3op (uint16 inst)
{
   uint16 st;
   uint16 sr;
   uint16 reg;
   uint16 val;
   uint16 cval;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   reg = (inst & 0x03C0) >> 6;
   op = (inst & 0x1C00) >> 10;

   switch (op)
   {
   case 0: /* COC */

      if (traceit) fprintf (tracefd, "COC");
      val = getword (sr, st, TRUE, SRC);
      if (traceit)
         fprintf (tracefd, " R%d", reg);
      cval = val & GETREG (reg);
      if (cval == val) SET_EQ;
      else CLR_EQ;
      if (traceit)
	 fprintf (tracefd, " val=>%04X & cval=>%04X", val, GETREG(reg));
      break;

   case 1: /* CZC */

      if (traceit) fprintf (tracefd, "CZC");
      val = getword (sr, st, TRUE, SRC);
      if (traceit)
         fprintf (tracefd, " R%d", reg);
      cval = val & (~GETREG (reg));
      if (cval == val) SET_EQ;
      else CLR_EQ;
      if (traceit)
	 fprintf (tracefd, " val=>%04X & cval=>%04X", val, GETREG(reg));
      break;

   case 2: /* XOR */

      if (traceit) fprintf (tracefd, "XOR");
      val = getword (sr, st, TRUE, SRC);
      if (traceit)
         fprintf (tracefd, " R%d", reg);
      val ^= GETREG (reg);
      PUTREG (reg, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X ^ cval=>%04X",
		  val, (~GETREG(reg) & 0xFFFF));
      break;

   case 3: /* XOP */

      if (traceit) fprintf (tracefd, "XOP");
      val = wpreg;
      cval = getaddr (sr, st, TRUE);
      if (traceit)
         fprintf (tracefd, " %d", reg);
      op = statreg;
      SET_XOP;
      CLR_NONPRV; /* Privleged */
      CLR_MAP; /* Use Map 0 */
      CLR_OVERI;
      CLR_WCS;
      wpreg = GETMEM0 (reg * 4 + 0x0040) & 0xFFFE;
      PUTREG (11, cval);
      cval = pcreg;
      mpcreg = pcreg = GETMEM0 (reg * 4 + 0x0042) & 0xFFFE;
      PUTREG (13, val);
      PUTREG (14, cval);
      PUTREG (15, op);
      deferint = 1;
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc4op - Process type 4 operations.
***********************************************************************/

static int
proc4op (uint16 inst)
{
   uint16 sr;
   uint16 st;
   uint16 cnt;
   uint16 dat;
   uint8 bdat;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;

   switch ((inst & 0x0400) >> 10)
   {
   case 0: /* LDCR */

      if (traceit) fprintf (tracefd, "LDCR");
      if (cnt > 1 && cnt < 9)
      {
         bdat = getbyte (sr, st, TRUE, SRC);
	 if (traceit) fprintf (tracefd, ",%d DAT >%02X", cnt, bdat);
	 cmpbytezero (bdat);
	 chkparity (bdat);
	 dat = bdat;
      }
      else
      {
         dat = getword (sr, st, TRUE, SRC);
	 if (traceit) fprintf (tracefd, ",%d DAT >%04X", cnt, dat);
	 cmpwordzero (dat);
      }
      if (traceit) fprintf (tracefd, " BASE >%04X", R12);
      loadcru (inst, dat);
      break;

   case 1: /* STCR */

      if (traceit) fprintf (tracefd, "STCR");
      dat = storecru (inst);
      if (cnt > 1 && cnt < 9)
      {
         bdat = (uint8)dat;
	 putbyte (sr, st, bdat, SRC);
	 if (traceit) fprintf (tracefd, ",%d DAT >%02X", cnt, bdat);
	 cmpbytezero (bdat);
	 chkparity (bdat);
      }
      else
      {
	 putword (sr, st, dat, SRC);
	 if (traceit) fprintf (tracefd, ",%d DAT >%04X", cnt, dat);
	 cmpwordzero (dat);
      }
      if (traceit) fprintf (tracefd, " BASE >%04X", R12);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc5op - Process type 5 operations.
***********************************************************************/

static int
proc5op (uint16 inst)
{
   uint32 acc;
   uint16 reg;
   uint16 cnt;
   uint16 val;
   uint16 sbit;

   reg = inst & 0x000F;
   cnt = (inst & 0x00F0) >> 4;
   if (cnt == 0)
   {
      cnt = R0 & 0x000F;
      if (cnt == 0) cnt = 16;
   }
   val = GETREG (reg);

   switch ((inst & 0x0300) >> 8)
   {
   case 0: /* SRA */

      if (traceit) fprintf (tracefd, "SRA R%d %d", reg, cnt);
      acc = val;
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      acc = acc << 16;
      (int32)acc = (int32)acc >> cnt;
      if (acc & 0x00008000) SET_CARRY;
      else CLR_CARRY;
      val = (acc >> 16) & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      break;

   case 1: /* SRL */

      if (traceit) fprintf (tracefd, "SRL R%d %d", reg, cnt);
      acc = val;
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      acc = acc << 16;
      acc = acc >> cnt;
      if (acc & 0x00008000) SET_CARRY;
      else CLR_CARRY;
      val = (acc >> 16) & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      break;

   case 2: /* SLA */

      if (traceit) fprintf (tracefd, "SLA R%d %d", reg, cnt);
      CLR_OVER;
      acc = val;
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      sbit = acc & 0x8000;
      while (cnt)
      {
	 acc = acc << 1;
	 if (sbit != (acc & 0x8000)) SET_OVER;
	 sbit = acc & 0x8000;
         cnt--;
      }
      if (acc & 0x10000) SET_CARRY;
      else CLR_CARRY;
      val = acc & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      break;

   case 3: /* SRC */

      if (traceit) fprintf (tracefd, "SRC R%d %d", reg, cnt);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      while (cnt)
      {
	 uint16 ob;

         ob = val & 0x0001;
	 val = val >> 1;
	 if (ob)
	 {
	    val = val | 0x8000;
	    SET_CARRY;
	 }
	 else CLR_CARRY;
	 cnt--;
      }
      PUTREG (reg, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " val=>%04X", val);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc6op - Process type 6 operations.
***********************************************************************/

static int
proc6op (uint16 inst)
{
   int32 acc;
   int32 lval;
   int32 rval;
   uint16 sr;
   uint16 st;
   uint16 val;
   uint16 sval;
   uint16 xinst;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;

   if ((inst & 0x0F00) >= 0x0C00) /* Floating point instructions */
   {
      return (proc6fltop (inst));
   }
   else if ((inst & 0x0F00) == 0x0100) 
   {
      switch ((inst & 0x00C0) >> 6)
      {
      case 1: /* BIND */

	 if (traceit) fprintf (tracefd, "BIND");
	 if (model < 11) return (-1);
	 val = getword (sr, st, TRUE, SRC);
	 pcreg = val & 0xFFFE;
         break;

      case 2: /* DIVS */

	 if (traceit) fprintf (tracefd, "DIVS");
	 if (model < 11) return (-1);
	 rval = (int32)R0 << 16 | (int32)R1;
	 lval = (int32)getlong (sr, st, TRUE, SRC);
	 if (lval <= rval)
	 {
	    SET_OVER;
	 }
	 else
	 {
	    CLR_OVER;
	    acc = rval / lval;
	    val = acc & 0xFFFF;
	    PUTREG (0, val);
	    acc = rval % lval;
	    sval = acc & 0xFFFF;
	    PUTREG (1, sval);
	    cmpwordzero (val);
	 }
         break;

      case 3: /* MPYS */

	 if (traceit) fprintf (tracefd, "MPYS");
	 if (model < 11) return (-1);
	 lval = (int32)getword (sr, st, TRUE, SRC);
	 rval = (int32)R0;
	 acc = rval * lval;
	 PUTREG (0, (acc >> 16) & 0xFFFF);
	 PUTREG (1, acc & 0xFFFF);
	 cmplongzero (acc);
         break;

      default:
         return (-1);
      }
   }
   else switch ((inst & 0x03C0) >> 6)
   {
   case 0: /* BLWP */

      if (traceit) fprintf (tracefd, "BLWP");
      val = wpreg;
      if (st == 0)
      {
	 if (traceit)
	    fprintf (tracefd, " R%d", sr);
	 sval = GETREG (sr+1);
         wpreg = GETREG (sr) & 0xFFFE;
      }
      else
      {
	 sval = getaddr (sr, st, TRUE);
	 wpreg = GETMEM (sval, SRC) & 0xFFFE;
	 sval = GETMEM (sval+2, SRC);
      }
      PUTREG (13, val);
      PUTREG (14, pcreg);
      PUTREG (15, statreg);
      opcreg = pcreg;
      pcreg = sval & 0xFFFE;
      deferint = 1;
      break;

   case 1: /* B */

      if (traceit) fprintf (tracefd, "B");
      val = getaddr (sr, st, TRUE);
      opcreg = pcreg;
      pcreg = val & 0xFFFE;
      break;

   case 2: /* X */

      if (traceit) fprintf (tracefd, "X");
      val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, SRC);
      if (traceit) fprintf (tracefd, "\n ");
      runinst (xinst);
      break;

   case 3: /* CLR */

      if (traceit) fprintf (tracefd, "CLR");
      putword (sr, st, 0, SRC);
      break;

   case 4: /* NEG */

      if (traceit) fprintf (tracefd, "NEG");
      sval = val = getword (sr, st, FALSE, SRC);
      val = -(int16)val;
      if (val == 0) SET_CARRY;
      else CLR_CARRY;
      if (sval == 0x8000) SET_OVER;
      else CLR_OVER;
      if (traceit)
         fprintf (tracefd, " ~>%04X + 1 -> >%04X", sval, val);
      goto SAVE_CHECK_ZERO;

   case 5: /* INV */

      if (traceit) fprintf (tracefd, "INV");
      val = ~getword (sr, st, FALSE, SRC);
      goto SAVE_CHECK_ZERO;

   case 6: /* INC */

      if (traceit) fprintf (tracefd, "INC");
      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 1;
      goto CHECK_INC_STATUS;

   case 7: /* INCT */

      if (traceit) fprintf (tracefd, "INCT");
      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 2;

CHECK_INC_STATUS:
      acc += sval;
      if (traceit)
         fprintf (tracefd, "  >%04X + %d -> >%04X", val, sval, acc);
      chkaddword (acc, (int16)val, (int16)sval);
      val = acc & 0xFFFF;
      goto SAVE_CHECK_ZERO;


   case 8: /* DEC */

      if (traceit) fprintf (tracefd, "DEC");
      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 1;
      goto CHECK_DEC_STATUS;

   case 9: /* DECT */

      if (traceit) fprintf (tracefd, "DECT");
      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 2;

CHECK_DEC_STATUS:
      acc -= sval;
      if (traceit)
         fprintf (tracefd, "  >%04X - %d -> >%04X", val, sval, acc);
      chksubword (acc, (int16)sval, (int16)val);
      val = acc & 0xFFFF;

SAVE_CHECK_ZERO:
      putword (sr, st, val, SRC);
      cmpwordzero (val);
      break;

   case 10: /* BL */

      if (traceit) fprintf (tracefd, "BL");
      val = getaddr (sr, st, TRUE);
      PUTREG (11, pcreg);
      opcreg = pcreg;
      pcreg = val & 0xFFFE;
      break;

   case 11: /* SWPB */

      if (traceit) fprintf (tracefd, "SWPB");
      val = getword (sr, st, FALSE, SRC);
      sval = val >> 8;
      sval |= (val & 0xFF) << 8;
      putword (sr, st, sval, SRC);
      break;

   case 12: /* SETO */

      if (traceit) fprintf (tracefd, "SETO");
      putword (sr, st, 0xFFFF, SRC);
      break;

   case 13: /* ABS */

      if (traceit) fprintf (tracefd, "ABS");
      CLR_OVER;
      CLR_CARRY;
      sval = val = getword (sr, st, FALSE, SRC);
      cmpwordzero (val);
      if ((int16)val < 0)
      {
	 if (val == 0x8000) SET_OVER;
	 val = -(int16)val;
      }
      if (traceit)
         fprintf (tracefd, " abs(>%04X) -> >%04X", sval, val);
      putword (sr, st, val, SRC);
      break;

   case 14: /* LDS */

#ifdef DEBUGLDSLDD
      tracefd = stderr;
      traceenable = TRUE;
      traceit = TRUE;
      tracestart = 0;
      fprintf (tracefd, "%d: PC %04X(%06X) WP %04X ST %04X INST %04X ",
	    instcount, pcreg-2, mpcreg-2, wpreg, statreg, inst);
#endif
      if (traceit) fprintf (tracefd, "LDS");
      if (model < 10) return (-1);
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV LDS\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      sval = val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l3 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b3 = xinst;

      use_LDS_map = TRUE;
      use_map_cnt = 2;
#if defined(DEBUGMAPPER) || defined(DEBUGLDSLDD)
      fprintf (stderr, "\n%d: LDS val = >%04X\n", instcount, sval);
      fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	    2, mapfile[2].l1,
	    2, mapfile[2].b1);
      fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	    2, mapfile[2].l2,
	    2, mapfile[2].b2);
      fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	    2, mapfile[2].l3,
	    2, mapfile[2].b3);
#endif
      deferint = 1;
      break;

   case 15: /* LDD */

#ifdef DEBUGLDSLDD
      tracefd = stderr;
      traceenable = TRUE;
      traceit = TRUE;
      tracestart = 0;
      fprintf (tracefd, "%d: PC %04X(%06X) WP %04X ST %04X INST %04X ",
	    instcount, pcreg-2, mpcreg-2, wpreg, statreg, inst);
#endif
      if (traceit) fprintf (tracefd, "LDD");
      if (model < 10) return (-1);
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV LDD\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      sval = val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b1 = xinst;
      val += 2;

      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l3 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b3 = xinst;

      use_LDD_map = TRUE;
      use_map_cnt = 2;
#if defined(DEBUGMAPPER) || defined(DEBUGLDSLDD)
      fprintf (stderr, "\n%d: LDD val = >%04X\n", instcount, sval);
      fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	    2, mapfile[2].l1,
	    2, mapfile[2].b1);
      fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	    2, mapfile[2].l2,
	    2, mapfile[2].b2);
      fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	    2, mapfile[2].l3,
	    2, mapfile[2].b3);
#endif
      deferint = 1;
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc7op - Process type 7 operations.
***********************************************************************/

static int
proc7op (uint16 inst)
{
   uint16 val;

   switch ((inst & 0x00F0) >> 4)
   {
   case 0: /* Floating point instructions */

      return (proc7fltop (inst));

   case 2:
      if (model < 12) return (-1);
      switch (inst & 0x000F)
      {
      case 13: /* EMD */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = EMD\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%d: ERRINT requested: nonPRIV EMD\n", instcount);
#endif
	    if (GET_MASK >= ERRINT)
	    {
	       tracememory (TRACE_PRIVINST, mpcreg-2);
	       errcrudata |= ERRPIN;
	       geninterrupt (ERRINT);
	    }
	    return (0);
	 }
         break;

      case 14: /* EINT */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = EINT\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%d: ERRINT requested: nonPRIV EINT\n", instcount);
#endif
	    if (GET_MASK >= ERRINT)
	    {
	       tracememory (TRACE_PRIVINST, mpcreg-2);
	       errcrudata |= ERRPIN;
	       geninterrupt (ERRINT);
	    }
	    return (0);
	 }
	 intsenabled = TRUE;
	 deferint = 1;
         break;

      case 15: /* DINT */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = DINT\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%d: ERRINT requested: nonPRIV DINT\n", instcount);
#endif
	    if (GET_MASK >= ERRINT)
	    {
	       tracememory (TRACE_PRIVINST, mpcreg-2);
	       errcrudata |= ERRPIN;
	       geninterrupt (ERRINT);
	    }
	    return (0);
	 }
	 intsenabled = FALSE;
         break;

      default:
         return (-1);
      }
      break;

   case 4: /* IDLE */

      if (traceit) fprintf (tracefd, "IDLE");
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV IDLE\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      idle = TRUE;
      idleled = TRUE;
      if (GET_MASK == 0) run = FALSE;
      instcount--;
      pcreg -= 2;
      break;

   case 6: /* RSET */

      if (traceit) fprintf (tracefd, "RSET");
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV RSET\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      SET_MASK (0);
      stopclk ();
      errcrudata = 0;
      mapcrudata = 0;
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: RSET mapcrudata = >%04X\n",
	       instcount, mapcrudata);
#endif
      maplatch = 0;
      mapenabled = FALSE;
      intsenabled = TRUE;
      resetdev();
      break;

   case 8: /* RTWP */

      if (traceit) fprintf (tracefd, "RTWP");
      val = R13 & 0xFFFE;
      pcreg = R14 & 0xFFFE;
      if (IS_NONPRV)
      {
	 uint16 st;
         st = statreg & 0x01DF;
	 st |= R15 & 0xFE20;
         statreg = st;
      }
      else
	 statreg = R15;
      wpreg = val;
      if (traceit)
	 fprintf (tracefd, " wp=>%04X pc=>%04X st=>%04X",
		  wpreg, pcreg, statreg);
      break;

   case 10: /* CKON */

      if (traceit) fprintf (tracefd, "CKON");
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV CKON\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      startclk (TRUE);
      break;

   case 12: /* CKOF */

      if (traceit) fprintf (tracefd, "CKOF");
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV CKOF\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      stopclk ();
      break;

   case 14: /* LREX */

      if (traceit) fprintf (tracefd, "LREX");
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV LREX\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      val = wpreg;
      CLR_MAP;
      if (model < 5)
	 wpreg = GETMEM0 (0xFFFC) & 0xFFFE;
      else
	 wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
      PUTREG (13, val);
      PUTREG (14, pcreg);
      PUTREG (15, statreg);
      if (model < 5)
	 pcreg = GETMEM0 (0xFFFE) & 0xFFFE;
      else
	 pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;
      SET_MASK (0);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc8op - Process type 8 operations.
***********************************************************************/

static int
proc8op (uint16 inst)
{
   int32 acc;
   uint16 r;
   uint16 val;
   uint16 cval;
   uint16 tval;
   int32 sv1, sv2;

   r = inst & 0x000F;

   if ((inst & 0xFFF0) == 0x00B0) /* BLSK */
   {
      if (model < 12) return (-1);
      val = GETINST;
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = BLSK r = %d addr = >%04X\n", r, val);
#endif
      cval = GETREG(r) - 2;
      PUTMEMB (cval, ((pcreg + 2) >> 8) & 0xFF, SRC);
      PUTMEMB (cval+1, (pcreg + 2) & 0xFF, SRC);
      PUTREG (r, cval);
      pcreg = (val & 0xFFFE);
   }
   else if (inst == 0x0300) /* LIMI */
   {
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV LIMI\n", instcount);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_PRIVINST, mpcreg-2);
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
         return (0);
      }
      SET_MASK (GETINST);
      if (traceit) fprintf (tracefd, "LIMI %d", GET_MASK);
      pcreg += 2;
   }

   else switch ((inst & 0x00F0) >> 4)
   {
   case 0: /* LI */

      val = GETINST;
      pcreg += 2;
      if (traceit) fprintf (tracefd, "LI R%d >%04X", r, val);
      PUTREG (r, val);
      cmpwordzero (val);
      break;

   case 2: /* AI */

      sv1 = GETINST;
      pcreg += 2;
      cval = sv1;
      if (traceit) fprintf (tracefd, "AI R%d >%04X", r, cval);
      acc = sv1;
      tval = GETREG (r);
      sv2 = tval;
      acc = acc + sv2;
      val = acc & 0xFFFF;
      acc = (int16)val;
      chkaddword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      PUTREG (r, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " >%04X + >%04X = >%04X", cval, tval, val);
      break;

   case 4: /* ANDI */

      cval = GETINST;
      pcreg += 2;
      if (traceit) fprintf (tracefd, "ANDI R%d >%04X", r, cval);
      tval = GETREG (r);
      val = tval & cval;
      PUTREG (r, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " >%04X & >%04X = >%04X", cval, tval, val);
      break;

   case 6: /* ORI */

      cval = GETINST;
      pcreg += 2;
      if (traceit) fprintf (tracefd, "ORI R%d >%04X", r, cval);
      tval = GETREG (r);
      val = tval | cval;
      PUTREG (r, val);
      cmpwordzero (val);
      if (traceit)
	 fprintf (tracefd, " >%04X | >%04X = >%04X", cval, tval, val);
      break;

   case 8: /* CI */

      cval = GETINST;
      pcreg += 2;
      if (traceit) fprintf (tracefd, "CI R%d >%04X", r, cval);
      val = GETREG (r);
      if (val > cval) SET_LGT;
      else CLR_LGT;
      if ((int16)val > (int16)cval) SET_AGT;
      else CLR_AGT;
      if (val == cval) SET_EQ;
      else CLR_EQ;
      if (traceit)
	 fprintf (tracefd, " >%04X == >%04X", cval, val);
      break;

   case 10: /* STWP */

      if (traceit) fprintf (tracefd, "STWP R%d", r);
      PUTREG (r, wpreg);
      break;

   case 12: /* STST */

      if (traceit) fprintf (tracefd, "STST R%d", r);
      PUTREG (r, statreg);
      break;

   case 14: /* LWPI */

      val = wpreg;
      wpreg = GETINST & 0xFFFE;
      if (traceit) fprintf (tracefd, "LWPI >%04X", wpreg);
      pcreg += 2;
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc9op - Process type 9 operations.
***********************************************************************/

static int
proc9op (uint16 inst)
{
   uint32 acc;
   uint16 val;
   uint16 st;
   uint16 sr;
   uint16 r;
   uint16 uval;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   r = (inst & 0x03C0) >> 6;
   op = (inst & 0x0400) >> 10;

   switch ((inst & 0x0400) >> 10)
   {
   case 0: /* MPY */

      if (traceit) fprintf (tracefd, "MPY");
      val = getword (sr, st, TRUE, SRC);
      if (traceit)
	 fprintf (tracefd, " R%d", r);
      acc = GETREG (r);
      acc = acc * (uint32)val;
      uval = (acc >> 16) & 0xFFFF;
      PUTREG (r, uval);
      uval = acc & 0xFFFF;
      PUTREG (r+1, uval);
      break;

   case 1: /* DIV */

      if (traceit) fprintf (tracefd, "DIV");
      val = getword (sr, st, TRUE, SRC);
      if (traceit)
	 fprintf (tracefd, " R%d", r);
      if (val <= GETREG (r))
      {
         SET_OVER;
      }
      else
      {
	 CLR_OVER;
	 acc = GETREG (r);
	 acc = (acc << 16) | GETREG (r+1);
	 PUTREG (r, acc / val);
	 PUTREG (r+1, acc % val);
      }
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc10op - Process type 10 operations.
***********************************************************************/

static int
proc10op (uint16 inst)
{
   uint16 r;
   uint16 mf;
   uint16 addr;
   uint16 xinst;

   r =  inst & 0x000F;
   mf = (inst & 0x0010) >> 4;
   if (traceit) fprintf (tracefd, "LMF R%d,%d", r, mf);
   if (model < 10) return (-1);
   if (model >= 10 && IS_NONPRV)
   {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: ERRINT requested: nonPRIV LMF\n", instcount);
#endif
      if (GET_MASK >= ERRINT)
      {
	 tracememory (TRACE_PRIVINST, mpcreg-2);
	 errcrudata |= ERRPIN;
	 geninterrupt (ERRINT);
      }
      return (0);
   }
   addr = GETREG (r);
   xinst = GETMEM (addr, NOLD);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l1 = xinst;

   addr += 2;
   xinst = GETMEM (addr, NOLD);
   mapfile[mf].b1 = xinst;

   addr += 2;
   xinst = GETMEM (addr, NOLD);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l2 = xinst;

   addr += 2;
   xinst = GETMEM (addr, NOLD);
   mapfile[mf].b2 = xinst;

   addr += 2;
   xinst = GETMEM (addr, NOLD);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l3 = xinst;

   addr += 2;
   xinst = GETMEM (addr, NOLD);
   mapfile[mf].b3 = xinst;

#ifdef DEBUGMAPPER
   fprintf (stderr, "%d: LMF %d,%d, addr = >%04X\n",
	    instcount, r, mf, GETREG (r));
   fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	 mf, mapfile[mf].l1,
	 mf, mapfile[mf].b1);
   fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	 mf, mapfile[mf].l2,
	 mf, mapfile[mf].b2);
   fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	 mf, mapfile[mf].l3,
	 mf, mapfile[mf].b3);
#endif
   return (0);
}

/***********************************************************************
* runinst - Run an instruction.
***********************************************************************/

static void 
runinst (uint16 inst)
{
   int type;

   if (traceenable && pcreg >= tracelim[0] && pcreg <= tracelim[1] &&
      instcount >= tracestart && GET_MASK != 4)
      traceit = TRUE;
   else
      traceit = FALSE;

   if (traceit)
      fprintf (tracefd, "%ld: PC %04X(%06X) WP %04X ST %04X INST %04X ",
	    instcount, pcreg-2, mpcreg-2, wpreg, statreg, inst);

   if (inst != IDLEINST) idleled = FALSE;
   curinst = inst;
   type = decodeinst (inst);

#if defined(DEBUGINSTR)
   screenposition (16,1);
   clearline ();
   printf ("INST = %04X TYPE = %d", inst, type);
#endif

   deferint = 0;
   switch (type)
   {
   case TYPE_1:

      if (proc1op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_2:

      if (proc2op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_3:

      if (proc3op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_4:

      if (proc4op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_5:

      if (proc5op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_6:

      if (proc6op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_7:

      if (proc7op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_8:

      if (proc8op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_9:

      if (proc9op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_10:

      if (proc10op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_11:

      if (proc11op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_12:

      if (proc12op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_13:

      if (proc13op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_14:

      if (proc14op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_15:

      if (proc15op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_16:

      if (proc16op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_17:

      if (proc17op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_18:

      if (proc18op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_19:

      if (proc19op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_20:

      if (proc20op (inst) < 0) goto ILL_INST;
      break;

   case TYPE_21:

      if (proc21op (inst) < 0) goto ILL_INST;
      break;

   default:
ILL_INST:
      if (model >= 10)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: Illegal instruction: inst = >%04X, pc = >%04X\n",
		  instcount, inst, pcreg-2);
         run = FALSE;
         runled = FALSE;
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    tracememory (TRACE_ILLOP, mpcreg-2);
	    errcrudata |= ERRILO;
	    geninterrupt (ERRINT);
	 }
      }
#ifdef DEBUGERRINT
      else
      {
	 sprintf (view[0], "Illegal instruction: inst = >%04X, pc = >%04X",
		  inst, pcreg-2);
      }
#endif
   }

   instcount++;
   if (traceit)
      fprintf (tracefd, "\n");

   if (use_map_cnt)
   {
#ifdef DEBUGLDSLDD
      fprintf (stderr, "use_map_cnt = %d\n", use_map_cnt);
#endif
#ifdef DEBUGMAPPER1
      tracefd = stderr;
      traceenable = TRUE;
      tracestart = 0;
#endif
      use_map_cnt--;
      if (use_map_cnt == 0)
      {
	 use_LDS_map = FALSE;
	 use_LDD_map = FALSE;
#ifdef DEBUGLDSLDD
	 traceenable = FALSE;
#endif
#ifdef DEBUGMAPPER1
	 traceenable = FALSE;
#endif
      }
   }
   inhibitwrite = FALSE;

}

/***********************************************************************
* stepprog - Step through a program.
***********************************************************************/

void 
stepprog (void)
{
   uint16 inst;

   inst = GETINST;
   pcreg = (pcreg + 2) & 0xFFFE;
   mpcreg = (mpcreg + 2) & 0x1FFFFE;
   runinst (inst);
}

/***********************************************************************
* runprog - Run a program.
***********************************************************************/

void
runprog (void)
{
   ttrunstate ();

   while (run)
   {
      idle = FALSE;
      runled = TRUE;
      stepprog ();
      if (breakflag && mpcreg == breakaddr)
      {
         run = FALSE;
         runled = FALSE;
      }
      else
      {
	 if (enablepanel && pancount) panel ();
	 if (!deferint)
	    checkinterrupts ();
      }
      if (run && delayclock)
      {
         long i, j;
	 j = delayclock;
	 for (i = 0; i < j; i++) ;
      }
   }

   ttcmdstate ();
}
