/***********************************************************************
*
* simio.c - IO processing for the TI 990 Simulator.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/20/03   DGP   Added interrupt support.
*   07/01/03   DGP   Fixed floppy unit on STCR of status.
*   07/08/03   DGP   Changed FLOPPY to return status if unit not present.
*   07/09/03   DGP   Forced PC and WP to be on even boundry for interrupts.
*   07/15/03   DGP   Added Card Reader support.
*   07/23/03   DGP   Added ASCII/EBCDIC support for IBM OS/390.
*   08/07/03   DGP   Fixed PANEL support such that selftest and load works.
*   08/12/03   DGP   Added gen/clr/chk interrupt routine and /10 priv checks.
*   11/06/03   DGP   Put Floppy stuff in simfpy.c
*   11/07/03   DGP   Added VDT911 support.
*   11/12/03   DGP   Added TILINE support.
*   12/02/03   DGP   Improved card reader support.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/29/04   DGP   Added devreset function.
*   12/03/04   DGP   Added CRU address adjustment.
*   04/01/05   DGP   Changed IDLE wait time.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (_WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"
#include "simcra.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */

extern int run;
extern int idle;
extern int runled;
extern int idleled;
extern int faultled;
extern int devcnt;
extern int model;
extern int mapenabled;
extern int intsenabled;
extern int pancount;
extern long delayclock;
extern unsigned long instcount;
extern char view[MAXVIEW][81];

extern uint16 errcrudata;
extern uint16 mapcrudata;
extern uint16 curmapindex;
extern uint32 maplatch;
extern uint8 memory[MEMSIZE];
extern MapFile mapfile[MAPSIZE];
extern Device devices[MAXDEVICES];

extern uint32 tracemem[16];
extern uint32 errtrace[16];
extern int errindex;
extern int trcindex;
extern int trclatch;
extern int traceenable;
extern int tracestart;
extern int traceend;
extern int tracelim[2];
extern int traceit;
extern FILE *tracefd;

static int clockactive = FALSE;
static int clockthreadactive = FALSE;
static uint16 intrequest = 0;
static int sim_int_char = 005;

#if defined(USS) || defined(OS390)
/*
** ASCII to EBCDIC conversion table. 
*/

static unsigned char ascebc[256] =
{
 /*00  NL    SH    SX    EX    ET    NQ    AK    BL */
      0x00, 0x01, 0x02, 0x03, 0x37, 0x2D, 0x2E, 0x2F,
 /*08  BS    HT    LF    VT    FF    CR    SO    SI */
      0x16, 0x05, 0x15, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
 /*10  DL    D1    D2    D3    D4    NK    SN    EB */
      0x10, 0x11, 0x12, 0x13, 0x3C, 0x3D, 0x32, 0x26,
 /*18  CN    EM    SB    EC    FS    GS    RS    US */
      0x18, 0x19, 0x3F, 0x27, 0x1C, 0x1D, 0x1E, 0x1F,
 /*20  SP     !     "     #     $     %     &     ' */
      0x40, 0x5A, 0x7F, 0x7B, 0x5B, 0x6C, 0x50, 0x7D,
 /*28   (     )     *     +     ,     -    .      / */
      0x4D, 0x5D, 0x5C, 0x4E, 0x6B, 0x60, 0x4B, 0x61,
 /*30   0     1     2     3     4     5     6     7 */
      0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
 /*38   8     9     :     ;     <     =     >     ? */
      0xF8, 0xF9, 0x7A, 0x5E, 0x4C, 0x7E, 0x6E, 0x6F,
 /*40   @     A     B     C     D     E     F     G */
      0x7C, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
 /*48   H     I     J     K     L     M     N     O */
      0xC8, 0xC9, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,
 /*50   P     Q     R     S     T     U     V     W */
      0xD7, 0xD8, 0xD9, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
 /*58   X     Y     Z     [     \     ]     ^     _ */
      0xE7, 0xE8, 0xE9, 0xAD, 0xE0, 0xBD, 0x5F, 0x6D,
 /*60   `     a     b     c     d     e     f     g */
      0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
 /*68   h     i     j     k     l     m     n     o */
      0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
 /*70   p     q     r     s     t     u     v     w */
      0x97, 0x98, 0x99, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,
 /*78   x     y     z     {     |     }     ~    DL */
      0xA7, 0xA8, 0xA9, 0xC0, 0x4F, 0xD0, 0xA1, 0x07,
 /*80                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*88                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*90                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*98                   CENT                        */
      0x6F, 0x6F, 0x6F, 0x4A, 0x6F, 0x6F, 0x6F, 0x6F,
 /*A0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*A8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*B0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*B8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*C0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*C8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*D0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*D8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*E0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*E8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*F0                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
 /*F8                                               */
      0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0xFF
};

/*
** EBCDIC to ASCII conversion table. 
*/

static unsigned char ebcasc[256] =
{
 /*00  NU    SH    SX    EX    PF    HT    LC    DL */
      0x00, 0x01, 0x02, 0x03, 0x3F, 0x09, 0x3F, 0x7F,
 /*08              SM    VT    FF    CR    SO    SI */
      0x3F, 0x3F, 0x3F, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
 /*10  DE    D1    D2    TM    RS    NL    BS    IL */
      0x10, 0x11, 0x12, 0x13, 0x14, 0x0A, 0x08, 0x3F,
 /*18  CN    EM    CC    C1    FS    GS    RS    US */
      0x18, 0x19, 0x3F, 0x3F, 0x1C, 0x1D, 0x1E, 0x1F,
 /*20  DS    SS    FS          BP    LF    EB    EC */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x0A, 0x17, 0x1B,
 /*28              SM    C2    EQ    AK    BL       */
      0x3F, 0x3F, 0x3F, 0x3F, 0x05, 0x06, 0x07, 0x3F,
 /*30              SY          PN    RS    UC    ET */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x04,
 /*38                    C3    D4    NK          SU */
      0x3F, 0x3F, 0x3F, 0x3F, 0x14, 0x15, 0x3F, 0x1A,
 /*40  SP                                           */
      0x20, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*48             CENT    .     <     (     +     | */
      0x3F, 0x3F, 0x9B, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
 /*50   &                                           */
      0x26, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*58               !     $     *     )     ;     ^ */
      0x3F, 0x3F, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
 /*60   -     /                                     */
      0x2D, 0x2F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*68               |     ,     %     _     >     ? */
      0x3F, 0x3F, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
 /*70                                               */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*78         `     :     #     @     '     =     " */
      0x3F, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
 /*80         a     b     c     d     e     f     g */
      0x3F, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
 /*88   h     i           {                         */
      0x68, 0x69, 0x3F, 0x7B, 0x3F, 0x3F, 0x3F, 0x3F,
 /*90         j     k     l     m     n     o     p */
      0x3F, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
 /*98   q     r           }                         */
      0x71, 0x72, 0x3F, 0x7D, 0x3F, 0x3F, 0x3F, 0x3F,
 /*A0         ~     s     t     u     v     w     x */
      0x3F, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
 /*A8   y     z                       [             */
      0x79, 0x7A, 0x3F, 0x3F, 0x3F, 0x5B, 0x3F, 0x3F,
 /*B0                                               */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*B8                                 ]             */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x5D, 0x3F, 0x3F,
 /*C0   {     A     B     C     D     E     F     G */
      0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
 /*C8   H     I                                     */
      0x48, 0x49, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*D0   }     J     K     L     M     N     O     P */
      0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
 /*D8   Q     R                                     */
      0x51, 0x52, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*E0   \           S     T     U     V     W     X */
      0x5C, 0x3F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
 /*E8   Y     Z                                     */
      0x59, 0x5A, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*F0   0     1     2     3     4     5     6     7 */
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
 /*F8   8     9                                     */
      0x38, 0x39, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0xFF
};
#define TOASCII(a) ebcasc[(a)]
#define TOEBCDIC(a) ascebc[(a)]
#else
#define TOASCII(a) (a)
#define TOEBCDIC(a) (a)
#endif

/*
** Terminal routines borrowed from Bub Supnik's simh
*/

#if defined (_WIN32)
static volatile int sim_win_ctlc = 0;

void win_handler (int sig)
{
sim_win_ctlc = 1;
return;
}

int ttinit (void)
{
return 0;
}

int ttrunstate (void)
{
if (signal (SIGINT, win_handler) == SIG_ERR) return -1;
SetThreadPriority (GetCurrentThread (), THREAD_PRIORITY_BELOW_NORMAL);
return 0;
}

int ttcmdstate (void)
{
SetThreadPriority (GetCurrentThread (), THREAD_PRIORITY_NORMAL);
return 0;
}

int ttclose (void)
{
return 0;
}

int sim_os_poll_kbd (void)
{
   int c;

   if (sim_win_ctlc)
   {
	sim_win_ctlc = 0;
	signal (SIGINT, win_handler);
	return 003;
   }
   if (!kbhit ()) return EOF;
   c = _getch ();
   if ((c & 0177) == '\b') c = 0177;
   if ((c & 0177) == sim_int_char)
   {
      run = FALSE;
      return EOF;
   }
   return c;
}

int sim_os_putchar (int32 c)
{
if (c != 0177)
	_putch (c & 0x7F);
return 0;
}
#endif

#if defined(UNIX)
struct termios cmdtty, runtty;

int ttinit (void)
{
if (!isatty (fileno (stdin))) return 0;			/* skip if !tty */
if (tcgetattr (0, &cmdtty) < 0) return -1;		/* get old flags */
runtty = cmdtty;
runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
runtty.c_cc[VINTR] = sim_int_char;			/* interrupt */
runtty.c_cc[VQUIT] = 0;					/* no quit */
runtty.c_cc[VERASE] = 0;
runtty.c_cc[VKILL] = 0;
runtty.c_cc[VEOF] = 0;
runtty.c_cc[VEOL] = 0;
runtty.c_cc[VSTART] = 0;				/* no host sync */
runtty.c_cc[VSUSP] = 0;
runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
runtty.c_cc[VLNEXT] = 0;
#endif
runtty.c_cc[VMIN] = 0;					/* no waiting */
runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
runtty.c_cc[VSTATUS] = 0;
#endif
return 0;
}

int ttrunstate (void)
{
if (!isatty (fileno (stdin))) return 0;			/* skip if !tty */
runtty.c_cc[VINTR] = sim_int_char;			/* in case changed */
if (tcsetattr (0, TCSAFLUSH, &runtty) < 0) return -1;
return 0;
}

int ttcmdstate (void)
{
if (!isatty (fileno (stdin))) return 0;			/* skip if !tty */
if (tcsetattr (0, TCSAFLUSH, &cmdtty) < 0) return -1;
return 0;
}

int ttclose (void)
{
return ttcmdstate ();
}

int sim_os_poll_kbd (void)
{
int status;
unsigned char buf[1];

status = read (0, buf, 1);
if (status != 1) return EOF;
if (buf[0] == sim_int_char)
{
   run = FALSE;
   return (EOF);
}
buf[0] = TOASCII (buf[0]);
return (buf[0]);
}

int sim_os_putchar (uint8 out)
{
char c;

c = TOEBCDIC (out & 0x7F);
write (1, &c, 1);
return 0;
}
#endif

static int 
poll_kbd (Device *dev, int delay)
{
   int c;
#if defined(UNIX)
   int ifd;
   struct timeval tv;
   fd_set readfds;
#endif

   if (delay > 0)
   {
#if defined(_WIN32)
      Sleep ((unsigned long) (delay) / 100L);
      c = 1;
#endif
#if defined(UNIX)
      ifd = fileno (dev->fd);
      FD_ZERO (&readfds);

      FD_SET (ifd, &readfds);

      tv.tv_sec = 0;
      tv.tv_usec = delay;
      if ((c = select (ifd+1,
		       &readfds,
		       NULL,
		       NULL,
		       &tv)) < 0)
      {
	 if (ERRNO != EINTR && ERRNO != EAGAIN)
	    PERROR ("poll_kbd: select failed");
      }
#endif
   }
   else c = 1;

   if (c > 0)
   {
      if (dev->fd == stdin)
      {
	 c = sim_os_poll_kbd ();
	 if (c == 0x03)
	 {
	    dev->cbreak = TRUE;
	    c = EOF;
	 }
      }
      else
      {
	 c = TOASCII (fgetc (dev->fd));
      }
   }
   else
      c = EOF;

   return (c);
}

/***********************************************************************
* geninterrupt - Generate interrupt
***********************************************************************/

int
geninterrupt (uint16 lvl)
{
#ifdef DEBUGERRINT
   if (lvl == ERRINT)
      fprintf (stderr,
	       "%d: ERRINT requested: level = >%X, errcrudata = >%04X\n",
	       instcount, GET_MASK, errcrudata);
#endif
   if (model == 12 && lvl == ERRINT && GET_MASK > 1)
   {
      int i, j;
      
      trclatch = TRUE;
      trcindex = 0;
      j = errindex;
      for (i = 0; i < 16; i++)
      {
	 j --;
	 if (j < 0) j = 15;
         tracemem[i] = errtrace[j];
#ifdef DEBUGERRINT
	 fprintf (stderr, "trc[%2d] = err[%2d] = %08X\n", i, j, tracemem[i]);
#endif
      }
   }
   intrequest = intrequest | (1 << lvl);
   return (0);
}

/***********************************************************************
* clrinterrupt - Clear interrupt
***********************************************************************/

static void
clrinterrupt (uint16 lvl)
{
   intrequest = intrequest & ~(1 << lvl);
}

/***********************************************************************
* chkinterrupt - Check interrupt
***********************************************************************/

static int
chkinterrupt (uint16 lvl)
{
   uint16 i;

   if (lvl > 0 && !intsenabled) return (0);
   i = intrequest & (1 << lvl);
#ifdef DEBUGINTS
   if (i)
      fprintf (stderr,"chkinterrupt: lvl = %d TRUE\n", lvl);
#endif
   return (i);
}

/***************************************************************************
* smallsleep - sleep in U seconds
* NOTE: Depending on the system the sleep time can vary widely.
***************************************************************************/

void
smallsleep (long Usec)
{
#if defined(UNIX)
#if defined(LINUX) || defined(SOLARIS)
#include <unistd.h>
   struct timespec req, rem;
#ifdef DEBUGCLOCK
#include <time.h>
   static clock_t starttime;
   static clock_t curtime;
   static int ticks;
   static int first = TRUE;

   if (first)
   {
      first = FALSE;
      ticks = 0;
      starttime = time (NULL) + 1;
   }
#endif

   req.tv_sec = 0;
   req.tv_nsec = Usec * 1000;

   nanosleep (&req, &rem);
#ifdef DEBUGCLOCK
   ticks++;
   curtime = time(NULL);
   if (curtime == starttime)
   {
      fprintf (stderr, "CLOCK: ticks = %d\n\r", ticks);
      starttime = curtime + 1;
      ticks = 0;
   }
#endif

#else
   struct timeval tv;

   tv.tv_sec = 0;
   tv.tv_usec = Usec;
   if (select (1,
	       NULL,
	       NULL,
	       NULL,
	       &tv) < 0)
   {
      if (ERRNO != EINTR && ERRNO != EAGAIN)
	 PERROR ("smallsleep: select failed");
   }
#endif
#endif /* UNIX */

#if defined(_WIN32)
   Sleep ((unsigned long) (Usec) / 100L);
#endif /* _WIN32 */

#if defined(OS2)
   DosSleep ((unsigned long) (Usec) / 100L);
#endif /* OS2 */

}

/***********************************************************************
* clock_thread - run the system clock
***********************************************************************/

static void
clock_thread (void)
{
   while (TRUE)
   {
      smallsleep (8300);
      if (clockactive)
	 geninterrupt (CLOCKINT);
      pancount = TRUE;
   }
}

/***********************************************************************
* finddev - Find attached device.
***********************************************************************/

static int
finddev (uint16 devaddr)
{
   int i;

   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].devaddr == devaddr && devices[i].fd != NULL) break;
      if (devices[i].cdevaddr == devaddr && devices[i].fd != NULL) break;
   }

   if (i == devcnt)
   {
#if 0
      run = FALSE;
#endif
#if 1
#ifdef DEBUGFINDDEV
      fprintf (stderr, "Device for devaddr %04X is not attached\n", devaddr);
#endif
      return (-1);
#endif
   }
   return (i);
}

/***********************************************************************
* findtype - Find attached device by type.
***********************************************************************/

static int
findtype (int type, char *dt)
{
   int i;

   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].type == type && devices[i].fd != NULL &&
	  !strcmp (devices[i].dev, dt) ) return (i);
   }

   return (-1);
}

/***********************************************************************
* attachit - Attach a file to a device (do the dirty work).
***********************************************************************/

static void
attachit (char *bp, int type, char *mode, char *dev)
{
   FILE *nfd;
   char *attype;
   int i;
   int protect;
   int devaddr;
   int interrupt;

   /*
   ** Check for device protect 
   */

   protect = FALSE;
   if (type == DISK || type == TAPE)
      if (*bp == 'r' || *bp == 'R')
      {
	 bp++;
	 protect = TRUE;
      }

   /*
   ** Get device address
   */

   bp = getnum (bp, &devaddr);

   /*
   ** Sanity check device interrupt
   */

   bp = getnum (bp, &interrupt);
   if (interrupt > 0)
   {
      if (model == 4 && (interrupt < 2 || interrupt > 7))
      {
	 sprintf (view[0], "attach: illegal interrupt %d for 990/4",
		  interrupt);
	 return;
      }
      else
      {
	 if (interrupt < 2 || interrupt > 15)
	 {
	    sprintf (view[0], "attach: illegal interrupt %d for 990/%d",
		     interrupt, model);
	    return;
	 }
      }
   }

   /*
   ** Sanity check device address
   */

   switch (type)
   {
   case TAPE:
   case DISK:
   case VDT945:
      if (devaddr < TILINESTART || devaddr > TILINEEND)
      {
	 sprintf (view[0],
		  "attach: illegal TILINE device address %d",
		  devaddr);
	 return;
      }
      break;

   default:
      if (devaddr > CRUEND)
      {
	 sprintf (view[0],
		  "attach: illegal CRU device address %d",
		  devaddr);
	 return;
      }
   }

   while (isspace (*bp)) bp++;
   if (type == VDT911)
   {
      attype = "port";
      nfd = v911open (bp, mode);
   }
   else
   {
      attype = "filename";
      if (!strcmp (bp, "NULL"))
      {
	 nfd = DEV_NULL;
      }
      else
      {
	 nfd = fopen (bp, mode);
      }
   }
   if (nfd == NULL)
   {
      sprintf (view[0], "attach: %s open failed: %s",
	       dev, strerror (ERRNO));
      sprintf (view[1], "%s = %s", attype, bp);
      return;
   }

   if ((i = findtype (type, dev)) >= 0)
   {
      if (!(devices[i].fd == stdin || devices[i].fd == stdout) &&
	 devices[i].fd != DEV_NULL)
	    fclose (devices[i].fd);
   }
   else
   {
      i = devcnt;
      devcnt++;
   }

   memset ((char *)&devices[i], '\0', sizeof (Device));
   devices[i].fd = nfd;
   devices[i].devaddr = devaddr;
   devices[i].type = type;
   devices[i].select = 0;
   devices[i].count = 0;
   devices[i].unit = 0;
   devices[i].intenabled = FALSE;
   devices[i].intlvl = interrupt;
   strcpy (devices[i].file, bp);
   strcpy (devices[i].dev, dev);

   switch (type)
   {
   case DISK:
      devices[i].unit = (1 << (3 - (dev[1] - '0')));
      devices[i].info.dskinfo.writeprotect = protect;
      dskopen (&devices[i]);
#ifdef DEBUGDSK
      fprintf (stderr, "   unit = %02X\n", devices[i].unit);
#endif
      break;

   case TAPE:
      devices[i].unit = (1 << (3 - (dev[1] - '0')));
      devices[i].info.tapeinfo.writeprotect = protect;
      tapeopen (&devices[i]);
#ifdef DEBUGTAPE
      fprintf (stderr, "   unit = %02X\n", devices[i].unit);
#endif
      break;

   case VDT945:
      devices[i].unit = (1 << (3 - (dev[1] - '0')));
      v945open (&devices[i]);
#ifdef DEBUGV945
      fprintf (stderr, "   unit = %02X\n", devices[i].unit);
#endif
      break;

   case FLOPPY:
      devices[i].cdevaddr = devaddr + 0x20;
      devices[i].unit = dev[1] - '0';
#ifdef DEBUGFLOPPY
      fprintf (stderr, "Attach floppy, unit = %d, cru = %04x to %s\n",
	       devices[i].unit, devices[i].devaddr, bp);
#endif
      break;

   case VDT911:
      devices[i].cdevaddr = devaddr + 0x10;
      devices[i].unit = atoi (bp);
      v911start (&devices[i]);
      break;

   case CARDREADER:
      devices[i].info.cardinfo.cardeof = FALSE;
      devices[i].info.cardinfo.cardndx = 100;
      break;

   default: ;
   }

}

/***********************************************************************
* attachdev - Attach a file to a device.
***********************************************************************/

void
attachdev (char *bp)
{
   char fdev[10];

   while (isspace (*bp)) bp++;
   switch (*bp++)
   {
      case 'c':
	 switch (*bp++)
	 {
	 case 'i': /* Console input */
	    attachit (bp, CONIN, "r", "CI");
	    break;

	 case 'o': /* Console output */
	    attachit (bp, CONOUT, "w", "CO");
	    break;

	 case 'r': /* Card reader */
	    attachit (bp, CARDREADER, "r", "CR");
	    break;

	 default: 
	    goto BADDEV;
	 }
         break;

      case 'd': /* Disk */
	 if (!isdigit (*bp)) goto BADDEV;
	 sprintf (fdev, "D%c", *bp++);
	 attachit (bp, DISK, "r+b", fdev);
         break;

      case 'f': /* Floppy disk */
	 if (!isdigit (*bp)) goto BADDEV;
	 sprintf (fdev, "F%c", *bp++);
	 attachit (bp, FLOPPY, "r+b", fdev);
         break;

      case 'm': /* MagTape */
	 if (!isdigit (*bp)) goto BADDEV;
	 sprintf (fdev, "M%c", *bp++);
	 attachit (bp, TAPE, "r+b", fdev);
         break;

      case 'p': /* Printer */
	 attachit (bp, PRINTER, "w", "PR");
         break;

      case 's': /* Serial */
	 attachit (bp, SERIAL, "r+", "SE");
         break;

      case 't': /* 945 vdt */
	 if (!isdigit (*bp)) goto BADDEV;
	 sprintf (fdev, "T%c", *bp++);
	 attachit (bp, VDT945, "r+", fdev);
	 break;

      case 'v': /* 911 vdt */
	 if (!isdigit (*bp)) goto BADDEV;
	 sprintf (fdev, "V%c", *bp++);
	 attachit (bp, VDT911, "r+", fdev);
	 break;

      default:
BADDEV:
         sprintf (view[0], "Invalid device %s, devices are:", bp);
	 sprintf (view[1], " ci = console input  co = console output");
	 sprintf (view[2], " p  = printer        fn = floppy n=0-3");
	 sprintf (view[3], " cr = card reader    vn = 911 VDT n=0-3");
	 sprintf (view[4], " dn = DISK n=0-3     mn = TAPE n=0-3");
	 sprintf (view[5], "usage: a device[R] devaddr intlvl file");
	 break;
   }
}

/***********************************************************************
* resetdev - Reset devices for RSET.
***********************************************************************/

void
resetdev (void)
{
   Device *dev;
   int i;

   intrequest = 0; /* Reset all interrupts */

   for (i = 0; i < devcnt; i++)
   {
      dev = &devices[i];
      switch (dev->type)
      {
      case TAPE:
         tapereset (dev);
	 break;
      case DISK:
         dskreset (dev);
	 break;
      case VDT945:
         v945reset (dev);
	 break;
      default: ;
      }
   }
}

/***********************************************************************
* setbitone - Set CRU bit to one.
***********************************************************************/

void
setbitone (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;

   disp = inst & 0x00FF;
   devaddr = R12;

#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif

   /*
   ** Adjust CRU address
   */

   if (disp > 15)
   {
      while (disp > 15)
      {
         disp -= 16;
	 devaddr += 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }
   else if (disp < 0)
   {
      while (disp < 0)
      {
         disp += 16;
	 devaddr -= 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }

   /*
   ** Check for privleged CRU address
   */

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr + disp >= PRVCRU)
      {
#ifdef DEBUGERRINT
         fprintf (stderr, "%d: Priv CRU: cru = %04X, disp = %02x\n",
		  instcount, devaddr, disp);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
	 return;
      }
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      switch (disp)
      {
      case 10:
	 runled = TRUE;
	 faultled = FALSE;
	 break;
      case 11:
	 faultled = TRUE;
	 break;
      case 12:
	 errcrudata &= ~ ERRTIP; /* 990/4 compatible */
	 break;
      default: ;
      }
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X SBO %d\n", instcount, devaddr, disp);
#endif
      switch (disp)
      {
      case 0:
	 if (model == 12)
	 {
	    trclatch = FALSE;
	    trcindex++;
	    if (trcindex == 32) trcindex = 0;
	 }
	 break;
      case 3:
	 mapenabled = TRUE;
	 break;
      case 4:
	 errcrudata &= ~ ERRMAP;
      case 5:
      case 6:
      case 7:
	 maplatch = 0;
	 break;
      default: ;
      }
      if (disp < 8)
	 mapcrudata |= 1 << disp;
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }

   /*
   ** Error control
   */

   if (devaddr == ERRCRU && model >= 10)
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%d: CRU %04X SBO %d MASK %4.4X VAL %4.4X\n",
	       instcount, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16)
      {
	 errcrudata &= ~(1 << disp);
	 if (GET_MASK >= ERRINT && errcrudata && disp > 12)
	    geninterrupt (ERRINT);
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model >= 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      return;
   }
   dev = &devices[i];
   switch (dev->type)
   {
   case CONIN:
   case CONOUT:
#ifdef DEBUGCON
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      if (dev->fd == DEV_NULL) return;

      if (disp == 14)
      {
         i = findtype (CONIN, "CI");
	 dev = &devices[i];
	 dev->intenabled = TRUE;
         i = findtype (CONOUT, "CO");
	 dev = &devices[i];
	 dev->intenabled = TRUE;
	 if (GET_MASK >= dev->intlvl)
	    geninterrupt (dev->intlvl);
      }
      break;

   case VDT911:
      v911seto (dev, disp);
      break;

   case FLOPPY:
      fpyseto (dev, disp);
      break;

   case PRINTER:
#ifdef DEBUGPRINTER
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
   case SERIAL:
      if (disp == 14)
      {
	 dev->intenabled = TRUE;
	 if (GET_MASK >= dev->intlvl)
	    geninterrupt (dev->intlvl);
      }
      break;

   case CARDREADER:
#ifdef DEBUGCARD
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      if (disp == 13) /* Enable Column interrupt */
      {
	 dev->intenabled = TRUE;
	 dev->select = TRUE;
	 geninterrupt (dev->intlvl);
      }
      break;

   default: ;
   }
   return;
}

/***********************************************************************
* setbitzero - Set CRU bit to zero.
***********************************************************************/

void
setbitzero (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;

   disp = inst & 0x00FF;
   devaddr = R12;

#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif

   /*
   ** Adjust CRU address
   */

   if (disp > 15)
   {
      while (disp > 15)
      {
         disp -= 16;
	 devaddr += 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }
   else if (disp < 0)
   {
      while (disp < 0)
      {
         disp += 16;
	 devaddr -= 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }

   /*
   ** Check for privleged CRU address
   */

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr + disp >= PRVCRU)
      {
#ifdef DEBUGERRINT
         fprintf (stderr, "%d: Priv CRU: cru = %04X, disp = %02x\n",
		  instcount, devaddr, disp);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
	 return;
      }
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      switch (disp)
      {
      case 11:
	 faultled = FALSE;
	 break;
      case 12:
	 errcrudata &= ~ ERRTIP; /* 990/4 compatible */
	 break;
      default: ;
      }
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X SBZ %d\n", instcount, devaddr, disp);
#endif
      switch (disp)
      {
      case 0:
	 if (model == 12)
	 {
	    trclatch = FALSE;
	    trcindex++;
	    if (trcindex == 32) trcindex = 0;
	 }
	 break;
      case 3:
	 mapenabled = FALSE;
	 maplatch = 0;
	 break;
      case 4:
	 errcrudata &= ~ ERRMAP;
	 maplatch = MAPPER | (mapcrudata & 0x7); /* The diag wants this?? */
	 break;
      case 5:
      case 6:
      case 7:
	 maplatch = 0;
	 break;
      default: ;
      }
      if (disp < 8)
	 mapcrudata &= ~(1 << disp);
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && model >= 10)
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%d: CRU %04X SBZ %d MASK %4.4X VAL %4.4X\n",
	       instcount, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16)
      {
	 errcrudata &= ~(1 << disp);
	 if (GET_MASK >= ERRINT && errcrudata && disp > 12)
	    geninterrupt (ERRINT);
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model >= 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      return;
   }
   dev = &devices[i];
   switch (dev->type)
   {
   case CONIN:
   case CONOUT:
#ifdef DEBUGCON
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      if (dev->fd == DEV_NULL) return;

      if (disp == 14)
      {
         i = findtype (CONIN, "CI");
	 dev = &devices[i];
	 dev->intenabled = FALSE;
         i = findtype (CONOUT, "CO");
	 dev = &devices[i];
	 dev->intenabled = FALSE;
	 clrinterrupt (dev->intlvl);
      }
      break;

   case VDT911:
      v911setz (dev, disp);
      break;

   case FLOPPY:
      fpysetz (dev, disp);
      break;

   case PRINTER:
#ifdef DEBUGPRINTER
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
   case SERIAL:
      if (disp == 14)
      {
	 dev->intenabled = FALSE;
	 clrinterrupt (dev->intlvl);
      }
      break;

   case CARDREADER:
#ifdef DEBUGCARD
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      switch (disp)
      {
      case 9: /* Clear presence interrupt */
      case 14: /* Clear column interrupt */
	 if (dev->select)
	 {
	    dev->intenabled = TRUE;
	    geninterrupt (dev->intlvl);
	 }
	 break;
      case 13: /* Disable Column interrupt */
	 dev->intenabled = FALSE;
	 dev->select = FALSE;
	 geninterrupt (dev->intlvl);
	 break;
      default: ;
      }
      break;

   default: ;
   }

   return;
}

/***********************************************************************
* testbit - Test CRU bit.
***********************************************************************/

void
testbit (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;

   disp = inst & 0x00FF;
   devaddr = R12;

#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif

   /*
   ** Adjust CRU address
   */

   if (disp > 15)
   {
      while (disp > 15)
      {
         disp -= 16;
	 devaddr += 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }
   else if (disp < 0)
   {
      while (disp < 0)
      {
         disp += 16;
	 devaddr -= 0x20;
      }
#ifdef DEBUGCRU
      fprintf (stderr, "  adjusted %04X %d\n", devaddr, disp);
#endif
   }

   /*
   ** Check for privleged CRU address
   */

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr + disp >= PRVCRU)
      {
#ifdef DEBUGERRINT
         fprintf (stderr, "%d: Priv CRU: cru = %04X, disp = %02x\n",
		  instcount, devaddr, disp);
#endif
	 if (GET_MASK >= ERRINT)
	 {
	    errcrudata |= ERRPIN;
	    geninterrupt (ERRINT);
	 }
	 return;
      }
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      CLR_EQ;
      if (disp == 11) SET_EQ;
      if (disp == 14) SET_EQ;
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X TB %d\n", instcount, devaddr, disp);
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      if (disp < 8 && (mapcrudata & (1 << disp))) SET_EQ;
      else CLR_EQ;
      return;
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && model >= 10)
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%d: CRU %04X TB  %d MASK %4.4X VAL %4.4X",
	       instcount, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16 && (errcrudata & (1 << disp)))
      {
	 SET_EQ;
#ifdef DEBUGERRCRU
	 fprintf (stderr, " EQUAL\n");
#endif
      }
      else
      {
	 CLR_EQ;
#ifdef DEBUGERRCRU
	 fprintf (stderr, " NOT EQUAL\n");
#endif
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model >= 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      return;
   }
   dev = &devices[i];

   if (dev->type == CONIN || dev->type == CONOUT)
   {
#ifdef DEBUGCON
      fprintf (stderr, "CRU %04X TB %d", devaddr, disp);
#endif
      if (dev->fd == DEV_NULL) return;

      switch (disp)
      {
      case 0:  /* TTY */
      case 12: /* RRQ */
      case 15: /* INT */
         i = findtype (CONIN, "CI");
	 break;
      case 8: /* XMIT */
      case 11: /* WRQ */
      case 13: /* DCD */
      case 14: /* DSR */
      default:
         i = findtype (CONOUT, "CO");
      }
      dev = &devices[i];
   }

   switch (dev->type)
   {
   case CONIN:
      if (disp == 0) /* TTYI */
      {
         if (dev->cbreak) CLR_EQ;
	 else SET_EQ;
	 dev->cbreak = FALSE;
      }
      else if (disp == 12 || disp == 15) /* RRQ | INT */
      {
	 CLR_EQ;
	 if (!dev->intenabled)
	 {
	    int c;
	    c = poll_kbd (dev, 0);
	    if (c != EOF)
	    {
	       dev->inchar = c;
	       SET_EQ;
	    }
	 }
	 else if (dev->select)
	 {
	    SET_EQ;
	 }
      }
#ifdef DEBUGCON
      fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
      break;

   case PRINTER:
#ifdef DEBUGPRINTER
      fprintf (stderr, "CRU %04X TB %d", devaddr, disp);
#endif
   case SERIAL:
      switch (disp)
      {
      case 11: /* WRQ */
      case 13: /* DCD */
      case 14: /* DSR */
	 SET_EQ;
	 break;
      default:
	 CLR_EQ;
      }
#ifdef DEBUGPRINTER
      fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
      break;

   case CONOUT:
      switch (disp)
      {
      case 11: /* WRQ */
      case 13: /* DCD */
      case 14: /* DSR */
	 SET_EQ;
	 break;
      default:
	 CLR_EQ;
      }
#ifdef DEBUGCON
      fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
      break;

   case VDT911:
      v911tb (dev, disp);
      break;

   case FLOPPY:
      fpytb (dev, disp);
      break;

   case CARDREADER:
#ifdef DEBUGCARD
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      switch (disp)
      {
	 case 15: /* Column Ready */
	    SET_EQ;
	    break;
         default: ;
	    CLR_EQ;
      }
      break;

   default: ;
   }
   return;
}

/***********************************************************************
* loadcru - Load CRU.
***********************************************************************/

void
loadcru (uint16 inst, uint16 dat)
{
   Device *dev;
   int i;
   uint16 reg;
   uint16 tag;
   uint16 cnt;
   uint16 devaddr;
   uint8 ch;
#ifdef DEBUGPRINTER
   uint8 ch1;
#endif

   reg = inst & 0x000F;
   tag = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;

   devaddr = R12;

#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      ch = (uint8)dat;
      lights = (lights << 8) | ch;
      return;
   }

   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X LDCR CNT %d DAT %04X\n",
	       instcount, devaddr, cnt, dat);
#endif
      mapcrudata = (mapcrudata & 0xFFF8) | (dat & 0x7);
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && model >= 10)
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%d: CRU %04X LDCR CNT %d DAT %04X\n",
	       instcount, devaddr, cnt, dat);
#endif
      errcrudata = dat;
      return;
   }

   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model >= 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0) return;
   if (devices[i].type == CONIN)
   {
      if ((i = findtype (CONOUT, "CO")) < 0) return;
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case PRINTER:
#ifdef DEBUGPRINTER
      fprintf (stderr, "CRU %04X LDCR CNT %d", devaddr, cnt);
      if (cnt > 1 && cnt < 9)
      {
	 ch = dat & 0xFF;
	 if (!isprint (ch)) ch = '.';
	 fprintf (stderr, " DATA %02X(%c)\n", dat, ch);
      }
      else
      {
	 ch = (dat >> 8) & 0xFF;
	 if (!isprint (ch)) ch = '.';
	 ch1 = dat & 0xFF;
	 if (!isprint (ch1)) ch1 = '.';
	 fprintf (stderr, " DATA %04X(%c%c)\n", dat, ch, ch1);
      }
#endif
   case SERIAL:
      ch = dat & 0xFF;
      if (isprint (ch) || (ch > 0x07 && ch < 0x0D))
	 fputc (TOEBCDIC (ch), dev->fd);
      geninterrupt (dev->intlvl);
      break;

   case CONOUT:
#ifdef DEBUGCON
      if (dat >= 0x20 && dat <= 0x7E)
	 fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X(%c)\n",
	       devaddr, cnt, dat, dat);
      else
	 fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X\n", devaddr, cnt, dat);
#endif
      if (dev->fd == DEV_NULL) return;
      ch = (uint8)dat;
      if (ch > 0)
      {
	 if (dev->fd == stdout)
	    sim_os_putchar (ch);
	 else
	    fputc (TOEBCDIC (ch), dev->fd);
      }
      if (dev->intenabled && (GET_MASK >= dev->intlvl))
	 geninterrupt (dev->intlvl);
      break;

   case VDT911:
      v911ldcr (dev, cnt, dat);
      break;

   case FLOPPY:
      fpyldcr (dev, cnt, dat);
      break;

   case CARDREADER:
#ifdef DEBUGCARD
      fprintf (stderr, "CRU %04X LDCR CNT %d DATA %04X\n", devaddr, cnt, dat);
#endif
      break;

   default: ;
   }

   return;
}

/***********************************************************************
* storecru - Store CRU.
***********************************************************************/

uint16
storecru (uint16 inst)
{
   Device *dev;
   int i;
   uint16 reg;
   uint16 tag;
   uint16 cnt;
   uint16 devaddr;
   uint16 dat;

   reg = inst & 0x000F;
   tag = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;

   dat = 0;
   devaddr = R12;

#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X STCR CNT %d ", devaddr, cnt);
#endif

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n", devaddr, cnt, 0);
#endif
#ifdef DEBUGCRU
      fprintf (stderr, "DAT %d\n", 0);
#endif
      return (0);
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
      uint16 retval;

      if (model < 12)
      {
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "%d: CRU %04X STCR CNT %d DAT %04X\n",
		  instcount, devaddr, cnt, mapcrudata);
#endif
#ifdef DEBUGCRU
	 fprintf (stderr, "DAT %04X\n", mapcrudata);
#endif
	 switch (mapcrudata & 0x7)
	 {
	    case 0:
	       retval = mapfile[curmapindex].b1;
	       break;
	    case 1:
	       retval = mapfile[curmapindex].b2;
	       break;
	    case 2:
	       retval = mapfile[curmapindex].b3;
	       break;
	    case 3:
	       retval = mapfile[curmapindex].l1;
	       break;
	    case 4:
	       retval = mapfile[curmapindex].l2;
	       break;
	    case 5:
	       retval = mapfile[curmapindex].l3;
	       break;
	    case 6:
	       retval = maplatch & 0xFFFE;
	       break;
	    case 7:
	       retval = (maplatch >> 16) & 0x001F;
	 }
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "   RETVAL >%04X\n", retval);
#endif
      }
      else
      {
         int i = trcindex >> 1;
	 if (trcindex & 1)
	    retval = tracemem[i] & 0xFFFF;
	 else
	    retval = (tracemem[i] >> 16) & 0xFFFF;
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n",
		  devaddr, cnt, retval);
#endif
      }
      return (retval);
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && model >= 10)
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%d: CRU %04X STCR CNT %d DAT %04X\n",
	       instcount, devaddr, cnt, errcrudata);
#endif
#ifdef DEBUGCRU
      fprintf (stderr, "DAT %04X\n", errcrudata);
#endif
      return (errcrudata);
   }

   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model >= 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n",
	       devaddr, cnt, errcrudata);
#endif
      return(0);
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0) return (-1);
   if (devices[i].type == CONOUT) 
   {
      if ((i = findtype (CONIN, "CI")) < 0) return (-1);
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case PRINTER:
#ifdef DEBUGPRINTER
      fprintf (stderr, "CRU %04X STCR CNT %d\n", devaddr, cnt);
#endif
      break;

   case SERIAL:
   case CONIN:
      if (dev->fd == DEV_NULL) return(0);
      dev->select = FALSE;
      dat = dev->inchar;
#ifdef DEBUGCON
      if (dev->type == CONIN)
      {
	 if (dat >= 0x20 && dat <= 0x7E)
	    fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X(%c)\n",
		  devaddr, cnt, dat, dat);
	 else
	    fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X\n",
		  devaddr, cnt, dat);
      }
#endif
      break;

   case VDT911:
      dat = v911stcr (dev, cnt);
      break;

   case FLOPPY:
      dat = fpystcr (dev, cnt);
      break;

   case CARDREADER:
      {
	 int cdat;

	 /*
	 ** If record exhausted, read another
	 */

	 if (dev->info.cardinfo.cardndx >= 80)
	 {
	    int i1, o1;
	    int len;
	    char temp1[1024];
	    char temp[1024];

	    /*
	    ** Read record
	    */

	    if (!(dev->info.cardinfo.cardeof) && dev->fd)
	    {
	       if (fgets (temp1, 1024, dev->fd) == NULL)
	       {
		  strcpy (temp1, "/*\n");
		  dev->info.cardinfo.cardeof = TRUE;
	       }
	    }
	    else
	    {
	       strcpy (temp1, "/*\n");
	    }

	    /*
	    ** Detab input record
	    */

	    len = strlen (temp1);
	    for (i1 = o1 = 0; i1 < len; i1++)
	    {
	       if (temp1[i1] == '\t')
	       {
	          int o2;

	          for (o2 = 8 - (o1 % 8); o2; o2--)
		  {
		     temp[o1++] = ' ';
		  }
	       }
	       else
	       {
	          temp[o1++] = temp1[i1];
	       }
	    }
	    temp[o1] = '\0';

	    /*
	    ** Pad with blanks
	    */

	    len = strlen (temp);
	    if (len < 80)
	    {
	       len --;
	       while (len < 80)
	       {
		  temp[len++] = ' ';
	       }
	    }
	    temp[80] = '\0';

#ifdef DEBUGCARD
	    fprintf (stderr, "CARDREC:\n");
	    HEXDUMP (stderr, temp, 80, 0);
#endif
	    strcpy (dev->info.cardinfo.carddat, temp);
	    dev->info.cardinfo.cardndx = 0;

	 }

	 /*
	 ** Get a character from the record
	 */

	 cdat =
	    TOASCII (dev->info.cardinfo.carddat[dev->info.cardinfo.cardndx]);
	 dev->info.cardinfo.cardndx ++;

	 /*
	 ** Translate to card code
	 */

	 dat = tocard[cdat];
#ifdef DEBUGCARD
	 fprintf (stderr, "CRU %04X STCR CNT %d DATA %02X(%c) - %04X\n",
		  devaddr, cnt, cdat, cdat, dat);
#endif
      }
      break;

   default: ;
   }
#ifdef DEBUGCRU
   fprintf (stderr, "DAT %04X\n", dat);
#endif
   return (dat);
}

/***********************************************************************
* gettiline - Get TILINE memory.
***********************************************************************/

int
gettiline (uint32 addr, int bflg)
{
   Device *dev;
   int i;
   int dcrndx;
   uint16 devaddr;

   devaddr = addr & 0xFFF0;
   dcrndx = addr & 0x000F;

#ifdef ECCADDR
   if (devaddr == ECCADDR)
   {
      if (bflg)
      {
	 i = GETMEMB0 (addr);
      }
      else
      {
	 i = GETMEM0 (addr);
      }
#ifdef DEBUGECC
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   val = %04X\n", i);
#endif
      return (i);
   }
#endif

#ifdef DEBUGTILINE1
   fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
   fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif

   if ((i = finddev (devaddr)) < 0)
   {
#if defined(DEBUGERRINT) || defined(DEBUGTILINE)
      fprintf (stderr, "%d: No TILINE dev-get: devaddr = %04X\n",
	       instcount, devaddr);
#endif
      if (GET_MASK >= ERRINT)
      {
	 errcrudata |= ERRTIT;
	 geninterrupt (ERRINT);
      }
      return (0xFFFF);
   }

   dev = &devices[i];

   if (bflg)
   {
      i = GETMEMB0 (addr);
   }
   else
   {
      i = GETMEM0 (addr);
   }

#ifdef DEBUGTAPE
   if (dev->type == TAPE)
   {
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   val = %04X\n", i);
   }
#endif

#ifdef DEBUGDSK
   if (dev->type == DISK)
   {
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   val = %04X\n", i);
   }
#endif

#ifdef DEBUGV945
   if (dev->type == VDT945)
   {
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   val = %04X\n", i);
   }
#endif

#ifdef DEBUGTILINE1
   fprintf (stderr, "   val = %04X\n", i);
#endif

   return (i & 0xFFFF);
}

/***********************************************************************
* puttiline - Put TILINE memory.
***********************************************************************/

int
puttiline (uint32 addr, uint16 v, int bflg)
{
   Device *dev;
   int i;
   int intlevel;
   int start;
   int attn;
   int bit0;
   int dcrndx;
   uint16 devaddr;

   devaddr = addr & 0xFFF0;
   dcrndx = addr & 0x000F;

#ifdef ECCADDR
   if (devaddr == ECCADDR)
   {
      if (bflg)
      {
	 PUTMEMB0 (addr, v);
      }
      else
      {
	 PUTMEM0 (addr, v);
      }
#ifdef DEBUGECC
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	       addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif
      return (0);
   }
#endif

#ifdef DEBUGTILINE1
   fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	 addr, v, bflg);
   fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif

   if ((i = finddev (devaddr)) < 0)
   {
#if defined(DEBUGERRINT) || defined(DEBUGTILINE)
      fprintf (stderr, "%d: No TILINE dev-put: devaddr = %04X\n",
	       instcount, devaddr);
#endif
      if (GET_MASK >= ERRINT)
      {
	 errcrudata |= ERRTIT;
	 geninterrupt (ERRINT);
      }
      return (0);
   }

   dev = &devices[i];
   attn = FALSE;
   start = FALSE;
   bit0 = FALSE;
   intlevel = 0;

   if (bflg)
   {
      if (dcrndx == 1)
      {
	 if (v & 0xF)
	    attn = TRUE;
      }
      else if (dcrndx == 14)
      {
	 start = TRUE;
	 bit0 = v & 0x80 ? TRUE : FALSE;
      }
      PUTMEMB0 (addr, v);
   }
   else
   {
      if (dcrndx == 0)
      {
	 if (v & 0xF)
	    attn = TRUE;
      }
      else if (dcrndx == 14)
      {
	 start = TRUE;
	 bit0 = v & 0x8000 ? TRUE : FALSE;
      }
      PUTMEM0 (addr, v);
   }

#ifdef DEBUGV945
   if (dev->type == VDT945)
   {
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	    addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
   }
#endif
#ifdef DEBUGTAPE
   if (dev->type == TAPE)
   {
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	    addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
   }
#endif
#ifdef DEBUGDSK
   if (dev->type == DISK)
   {
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	 addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   start = %s, bit0 = %s, attn = %s\n", 
	       start ? "TRUE" : "FALSE",
	       bit0 ? "TRUE" : "FALSE",
	       attn ? "TRUE" : "FALSE");
   }
#endif

   if (start && !bit0)
   {
      switch (dev->type)
      {
      case TAPE:
	 intlevel = tapedocmd (dev);
	 break;

      case DISK:
	 intlevel = dskdocmd (dev);
	 break;

      case VDT945:
	 intlevel = v945docmd (dev);
	 break;

      default: ;
      }

      if (intlevel > 0)
	 geninterrupt (intlevel);
   }
   else if (attn)
   {
      geninterrupt (dev->intlvl);
   }

   return (0);
}

/***********************************************************************
* startclk - Start clock.
***********************************************************************/

void
startclk (int intenable)
{
#if defined(UNIX)
   pthread_t thread_handle;
#endif

#ifdef DEBUGCKINST
   fprintf (stderr, "CKON called\n");
#endif
   /*
   ** First call starts the thread
   */

   if (!clockthreadactive)
   {
      clockthreadactive = TRUE;
#if defined(UNIX)
      if (pthread_create (&thread_handle,
			  NULL,
			  (void *(*)(void *)) clock_thread,
			  NULL) < 0)
#endif
#if defined(_WIN32)
      if (_beginthread (clock_thread,
			NULL,
			NULL) < 0)
#endif
      {
	 PERROR ("startclk: Can't start clock thread");
      }
   }

   if (intenable)
      clockactive = TRUE;
}

/***********************************************************************
* stopclk - Stop clock.
***********************************************************************/

void
stopclk (void)
{
#ifdef DEBUGCKINST
   fprintf (stderr, "CKOF called\n");
#endif
   clockactive = FALSE;
   clrinterrupt (CLOCKINT);
}

/***********************************************************************
* boot - Boot from device.
***********************************************************************/

int
boot (char *device)
{
   Device *dev;
   uint16 lp;
   int i;

   run = FALSE;
   lp = LOADADDRESS;

   while (isspace (*device)) device++;
   if (islower (*device)) *device = toupper (*device);

   for (i = 0; i < devcnt; i++)
   {
      if (!strcmp (devices[i].dev, device)) break;
   }

   if (i == devcnt)
   {
      sprintf (view[0], "Boot device %s is not attached", device);
      return (-1);
   }

   dev = &devices[i];

   if (*device == 'F') /* Boot from FLOPPY */
   {
      return (fpyboot (dev));
   }
   else if (*device == 'D') /* Boot from TILINE disk */
   {
      return (dskboot (dev));
   }
   else if (*device == 'M') /* Boot from TILINE tape */
   {
      return (tapeboot (dev));
   }
   else
   {
      sprintf (view[0], "Boot from device %s is not supported", device);
      return (-1);
   }

   return (0);
}

/***********************************************************************
* checkinterrupts - If any pending and not masked, take it.
***********************************************************************/

void 
checkinterrupts (void)
{
   Device *dev;
   int curlvl;
   static int scancount = 0;
   int i;
   int keyhit;
   uint16 curinst;

#ifdef DEBUGCKINST
   fprintf (stderr, "checkinterrupts called: idle = %s, lvl = %d\n",
	    idle ? "TRUE" : "FALSE", GET_MASK);
#endif
   /*
   ** If mask == 0, return.
   */

   if (!GET_MASK) return;

   do
   {
      scancount++;
      curinst = GETINST;

      /*
      ** Check the keyboards for interrupts.
      */

      keyhit = FALSE;
      if (idle || scancount > 10000)
      {
	 if ((i = findtype (CONIN, "CI")) >= 0)
	 {
	    int c;

	    dev = &devices[i];
	    if (dev->fd != DEV_NULL && dev->intenabled)
	    {
	       c = poll_kbd (dev, scancount > 10000 ? 10 : 1000);
	       if (c != EOF)
	       {
		  dev->select = TRUE;
		  keyhit = TRUE;
		  dev->inchar = c;
		  if (GET_MASK >= dev->intlvl)
		     geninterrupt (dev->intlvl);
	       }
	       else if (run == FALSE) return;
	    }
	    else
	    {
	       c = sim_os_poll_kbd();
	       if (run == FALSE) return;
	    }
	 }

	 for (i = 0; i < devcnt; i++)
	 {
	    if (devices[i].dev[0] == 'V')
	    {
	       dev = &devices[i];
	       if (dev->info.vdtinfo.kbint)
	       {
		  dev->info.vdtinfo.kbint = FALSE;
		  keyhit = TRUE;
#ifdef DEBUG911TASK
		  fprintf (stderr, "Interrupt 911 keyboard\n");
#endif
		  if (dev->intenabled)
		     geninterrupt (dev->intlvl);
	       }
	    }
	 }
	 scancount = 0;
      }

      /*
      ** If we have interrupts, process them.
      */

      if (intrequest)
      {
	 curlvl = GET_MASK;
	 for (i = 0; i <= curlvl; i++)
	 {
	    if (chkinterrupt (i))
	    {
	       uint16 opc, npc;
	       uint16 owp, nwp;
	       uint16 ost;

	       if (curinst == IDLEINST) pcreg += 2;
	       ost = statreg;
	       opc = pcreg;
	       owp = wpreg;
	       clrinterrupt (i);
	       SET_MASK (i-1);
	       CLR_NONPRV;
	       CLR_MAP;
	       CLR_XOP;
	       nwp = GETMEM0 (i * 4) & 0xFFFE;
	       npc = GETMEM0 (i * 4 + 2) & 0xFFFE;
	       pcreg = npc;
	       wpreg = nwp;
#ifdef DEBUGERRINT
	       if (pcreg == 0)
	       {
		  fprintf (stderr,
			"%d: INVALID interrupt trap: wpreg = %d, pcreg = %d\n",
			instcount, wpreg, pcreg);
		  fprintf (stderr, "   curlvl = %X, i = %d\n", curlvl, i);
		  fprintf (stderr, "   owp = %04X, opc = %04X, ost = %04X\n",
			   owp, opc, ost);
		  run = FALSE;
		  return;
	       }
#endif
	       PUTREG (13, owp);
	       PUTREG (14, opc);
	       PUTREG (15, ost);
	       idle = FALSE;
	       return;
	    }
	 }
      }

      /*
      ** If nothing to do snooze a bit
      */

      if (idle && !scancount && !keyhit)
	 smallsleep(1);

   } while (idle);
}
