/***********************************************************************
*
* sim990.c - Simulator for the TI 990 computer.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/20/03   DGP   Added interrupt support.
*   12/09/03   DGP   Changed ROM load to >FC00.
*   12/10/03   DGP   Added 20 bit addresses for ROM and TILINE.
*   01/26/04   DGP   Added memory size option.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/01/04   DGP   Added /10A cpu, model = 11.
*   04/01/05   DGP   Changed signal handling.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#if defined(UNIX)
#include <unistd.h>
#endif

#include "simdef.h"
#include "screen.h"

uint16 pcreg;			/* The program PC */
uint16 opcreg;			/* The old program PC */
uint16 statreg;			/* The program status register */
uint16 wpreg;			/* The program Workspace Pointer */
uint16 lights;			/* The panel lights */
uint32 mpcreg;			/* Mapped PC */
int breakaddr;			/* Break point address */
long pancount;			/* Instructions executed for panel */
unsigned long instcount;	/* Instructions executed */

int run = FALSE;		/* CPU running */
int idle = FALSE;		/* CPU running IDLE or JXX $ */
int runled = FALSE;		/* CPU running LED */
int idleled = FALSE;		/* CPU idle LED */
int faultled = FALSE;		/* CPU fault LED */
int enablepanel = FALSE;	/* Enable panel */
int devcnt = 0;			/* Device attached count */
int model = 4;			/* Which model */
int breakflag = FALSE;		/* No breakpoint enabled */
int traceenable = FALSE;	/* No trace enabled */
int traceit = FALSE;		/* No trace enabled */
int mapenabled = FALSE;		/* Memory mapping enabled */
int intsenabled = TRUE;		/* Interrupts enabled */
int deferint = 0;		/* Defer interrupts inst count */
int tracelim[2];		/* Trace memory limits */
int tracestart = 0;		/* Trace instruction count start */
int traceend = 0;		/* Trace instruction count end */
uint16 mapcrudata = 0;		/* Memory map file CRU >1FA0 */
uint16 curmapindex = 0;		/* Current map index */
uint16 errcrudata = 0;		/* ERR CRU >1FC0 data */
uint16 curinst = 0;		/* Current instruction */
uint32 maplatch = 0;		/* Memory map latch */
uint32 memlen = 56*1024;	/* Length of system memory */
long delayclock = 0;		/* Delay clock interval */
char romfile[256];		/* ROM file name */
char view[MAXVIEW][81];		/* Messages panel */
uint8 memory[MEMSIZE];		/* The CPU memory */
Device devices[MAXDEVICES];	/* The attached devices */
MapFile mapfile[MAPSIZE];	/* Memory Mapping Files */
FILE *tracefd = NULL;		/* Trace file fd */
uint32 tracemem[16];		/* Error Trace Memory locked */
uint32 errtrace[16];		/* Error Trace Memory /12 only */
int errindex = 0;		/* Error Trace Memory index */
int trcindex = 0;		/* Error Trace Memory locked index */
int trclatch = FALSE;		/* Error Trace latch */


static int dispaddr = 0;	/* Display area address */
static char input[80];
static char oinput[80];

static char *models[] =
{
   "4", "5", "", "", "", "9", "10", "10A", "12"
};

/***********************************************************************
* sigintr - Process interrupt signal.
***********************************************************************/

void
sigintr (int sig)
{
   run = FALSE;
   signal (SIGINT, sigintr);
}

/***********************************************************************
* puthelp - Put help on screen.
***********************************************************************/

static void
puthelp (void)
{
   screenposition (14,1);
   clearline ();
   printf ("  a - attach  b - boot    c - clrBrk  d - display g - go");
   screenposition (15,1);
   clearline ();
   printf ("  h - help    k - break   l - load    m - modify  n - next");
   screenposition (16,1);
   clearline ();
   printf ("  p - set pc  q - quit    r - reset   s - step    w - set wp");
   screenposition (17,1);
   clearline ();
   printf ("  z - panel");
   screenposition (18,1);
   clearline ();
   printf ("  ");
}

/***********************************************************************
* putview - Puts messages on screen.
***********************************************************************/

static void
putview (int interactive)
{
   int i;

   for (i = 0; i < MAXVIEW; i++)
   {
      if (interactive)
      {
	 if (!enablepanel && view[i][0] == '\0') return;
	 screenposition (14+i,1);
	 clearline ();
      }
      if (view[i][0])
      {
	 printf ("%s", view[i]);
	 if (!interactive) printf ("\n");
      }
      view[i][0] = '\0';
   }
}

/***********************************************************************
* getnum - Get a number from the command.
***********************************************************************/

char *
getnum (char *bp, int *val)
{
   char *cp;
   int i = 0;

   while (isspace (*bp)) bp++;
   cp = bp;
   if (*bp == HEXSYM)
   {
      i = strtol (++bp, &cp, 16);
   }
   else if (isdigit (*bp) || *bp == '-' || *bp == '+')
   {
      i = strtol (bp, &cp, 10);
   }
   *val = i;
   return (cp);
}

/***********************************************************************
* panel - Put up the panel.
***********************************************************************/

void
panel (void)
{
   char c1, c2;

   c1 = DISPMEM >> 8;
   c2 = DISPMEM & 0xFF;
   if (!isprint (c1)) c1 = '.';
   if (!isprint (c2)) c2 = '.';

   screenposition (1,1);
   clearline ();
   printf ("%s %s      /%s CPU %dK",
	 INTRO, VERSION, models[model-4], memlen/1024);

   screenposition (3,1);
   printf (" PC  %04X  WP  %04X  ST  %04X  FP  %04X",
	    pcreg, wpreg, statreg, lights);

   screenposition (5,1);
   if (model < 10)
   {
      printf (" DISP  %04X %04X %c%c    %-5.5s %-5.5s %-5.5s",
	       dispaddr, DISPMEM, c1,c2,
	       runled ? "RUN" : " ",
	       idleled ? "IDLE" : " ",
	       faultled ? "FAULT" : " ");

      screenposition (6,1);
#ifdef SHOWOLDPC
      printf (" BREAK %04X  OPC %04X    COUNT %ld",
	       breakaddr,
	       opcreg,
	       instcount);
#else
      printf (" BREAK %04X  COUNT %ld",
	       breakaddr,
	       instcount);
#endif
   }
   else
   {
      printf (" DISP  %06X %04X %c%c  %-5.5s %-5.5s %-5.5s",
	       dispaddr, DISPMEM, c1,c2,
	       runled ? "RUN" : " ",
	       idleled ? "IDLE" : " ",
	       faultled ? "FAULT" : " ");

      screenposition (6,1);
#ifdef SHOWOLDPC
      printf (" BREAK %06X  OPC %04X  COUNT %ld",
	       breakaddr,
	       opcreg,
	       instcount);
#else
      printf (" BREAK %06X  COUNT %ld",
	       breakaddr,
	       instcount);
#endif
   }

   screenposition (8,1);
   printf ("Workspace:");
   screenposition (9,1);
   printf (" R00 %04X  R01 %04X  R02 %04X  R03 %04X", R0, R1, R2, R3);
   screenposition (10,1);
   printf (" R04 %04X  R05 %04X  R06 %04X  R07 %04X", R4, R5, R6, R7);
   screenposition (11,1);
   printf (" R08 %04X  R09 %04X  R10 %04X  R11 %04X", R8, R9, R10, R11);
   screenposition (12,1);
   printf (" R12 %04X  R13 %04X  R14 %04X  R15 %04X", R12, R13, R14, R15);
   screenposition (14,1);
   fflush (stdout);
   pancount = FALSE;

}

/***********************************************************************
* clearall - Clear out system.
***********************************************************************/

static int 
clearall ()
{
   FILE *rom;
   int i;
   uint32 lp;
   uint16 val;
   char filename[256];

   /*
   ** Clear registers
   */

   pcreg = 0;
   statreg = 0;
   wpreg = 0;
   lights = 0;
   errcrudata = 0;
   mapcrudata = 0;
   mapenabled = FALSE;

   /*
   ** Clear memory
   */

   memset (&memory, '\0', sizeof (memory));
   
   /*
   ** Load ROM
   */

   if (romfile[0])
   {
      strcpy (filename, romfile);
   }
   else
   {
      int rommod = model;

      if (model == 11) rommod = 10;
      sprintf (filename, "ti990-%d.rom", rommod);
   }

   if ((rom = fopen (filename, "rb")) == NULL)
   {
      sprintf (view[0], "clearall: ROM open failed: %s",
	    strerror (ERRNO));
      sprintf (view[1], "filename: %s", filename);
      return (-1);
   }

   lp = ROMSTART;
   if (model > 4)
      lp += TPCSSTART;

   for (i = 0; i < 1024; i+=2)
   {
      int c;

      if ((c = fgetc (rom)) == EOF)
      {
	 sprintf (view[0], "clearall: ROM read1 failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", filename);
	 return (-1);
      }
      val = (c & 0xFF) << 8;
      if ((c = fgetc (rom)) == EOF)
      {
	 sprintf (view[0], "clearall: ROM read2 failed: %s",
	       strerror (ERRNO));
	 sprintf (view[1], "filename: %s", filename);
	 return (-1);
      }
      val = val | (c & 0xFF);
      PUTMEM0 (lp, val);
      lp += 2;
   }
   fclose (rom);

   instcount = 0;
   clearscreen ();

   /*
   ** Initialize map registers
   */

   if (model >= 10)
   {
      for (i = 0; i < MAPSIZE; i++)
      {
	 mapfile[i].l1 = 0;
	 mapfile[i].b1 = 0;
	 mapfile[i].l2 = 0;
	 mapfile[i].b2 = 0;
	 mapfile[i].l3 = 0;
	 mapfile[i].b3 = 0;
      }
      maplatch = 0;
      errcrudata = 0;
      mapcrudata = 0;
      mapenabled = FALSE;
      intsenabled = TRUE;
   }

   return (0);
}

/***********************************************************************
* docommand - Do user commands.
***********************************************************************/

static int
docommand (int interactive)
{
   char *bp;
   int i;
   int norun;
   int action;

   bp = input;
   action = NULL_ACTION;
#ifdef DEBUGCMD
   printf ("docmommand: entered: input = '%s'\n", input);
#endif

   switch (*bp++)
   {
   case '\0':
   case '#': /* Comment */
      break;

   case 'C':
      bp = getnum (bp, &action);
      delayclock = action;
      action = NULL_ACTION;
      break;

   case 'a': /* Attach device */
      if (*bp)
	 attachdev (bp);
      if (interactive && view[0][0] == '\0')
      {
	 sprintf (view[0], "DEV  ADDR INT  FILE");
	 for (i = 0; i < devcnt; i++)
	 {
	    if (devices[i].fd)
	       sprintf (view[i+1], "%s   %04X %2d  %s",
		  devices[i].dev, devices[i].devaddr,
		  devices[i].intlvl, devices[i].file);
	 }
      }
      oinput[0] = '\0';
      break;

   case 'b': /* Boot */
      if (*bp == 'n') /* No start option */
      {
	 bp++;
	 norun = TRUE;
      }
      else norun = FALSE;
      if (!*bp)
      {
	 strcpy (view[0], "You must specify a boot device");
      }
      else if (boot (bp) == 0)
      {
	 if (!norun && run)
	 {
	    action = RUN_ACTION;
	    /*signal (SIGINT, sigintr);*/
	    runprog ();
	    /*signal (SIGINT, SIG_DFL);*/
#ifdef DEBUGCMD
	    printf ("docmommand: runprog returns\n");
#endif
	 }
      }
      oinput[0] = '\0';
      break;

   case 'c': /* Clear breakpoint */
      breakflag = FALSE;
      break;

   case 'd': /* Display memory */
      bp = getnum (bp, &action);
      dispaddr = action;
      action = DISP_ACTION;
      if (!enablepanel)
	 printf (" (%06X) = %04X\n", dispaddr, DISPMEM);
      oinput[0] = '\0';
      break;

   case 'e':
      bp = getnum (bp, &tracelim[0]);
      break;

   case 'f':
      bp = getnum (bp, &tracelim[1]);
      break;

   case 'g': /* Go - run program */
      action = RUN_ACTION;
      run = TRUE;
      /*signal (SIGINT, sigintr);*/
      runprog ();
      /*signal (SIGINT, SIG_DFL);*/
#ifdef DEBUGCMD
	    printf ("docmommand: runprog returns\n");
#endif
      oinput[0] = '\0';
      break;

   case 'h': /* Help */
      puthelp ();
      action = HELP_ACTION;
      oinput[0] = '\0';
      break;

   case 'k': /* set breaKpoint */
      bp = getnum (bp, &action);
      breakaddr = action;
      breakflag = TRUE;
      oinput[0] = '\0';
      break;

   case 'l': /* Load - use binloader */
      if (*bp == 'n') /* No start option */
      {
	 bp++;
	 norun = TRUE;
      }
      else norun = FALSE;
      bp = getnum (bp, &action);
      if (binloader (bp, action) == 0)
      {
	 if (!norun && run)
	 {
	    action = RUN_ACTION;
	    /*signal (SIGINT, sigintr);*/
	    runprog ();
	    /*signal (SIGINT, SIG_DFL);*/
#ifdef DEBUGCMD
	    printf ("docmommand: runprog returns\n");
#endif
	 }
	 else action = LOAD_ACTION;
      }
      run = FALSE;
      oinput[0] = '\0';
      break;

   case 'm': /* Modify memory */
      bp = getnum (bp, &action);
      PUTMEM0 (dispaddr, action);
      action = DISP_ACTION;
      if (!enablepanel)
	 printf (" (%06X) = %04X\n", dispaddr, DISPMEM);
      oinput[0] = '\0';
      break;

   case 'n': /* display Next memory location */
      dispaddr += 2;
      action = DISP_ACTION;
      if (!enablepanel)
	 printf (" (%06X) = %04X\n", dispaddr, DISPMEM);
      break;

   case 'p': /* set PC */
      bp = getnum (bp, &action);
      pcreg = action;
      action = PC_ACTION;
      oinput[0] = '\0';
      break;

   case 'q': /* Quit */
      action = QUIT_ACTION;
      break;

   case 'r': /* Reset system */
      clearall ();
      action = CLEAR_ACTION;
      oinput[0] = '\0';
      break;

   case 's': /* Step program */
      action = STEP_ACTION;
      run = FALSE;
      stepprog ();
      break;

   case 't': /* Trace program */
      traceenable = traceenable ? FALSE : TRUE;
      if (traceenable)
      {
	 bp = getnum (bp, &tracestart);
	 while (isspace (*bp)) bp++;
	 if ((tracefd = fopen (bp, "w")) == NULL)
	 {
	    traceenable = FALSE;
	    sprintf (view[0], "Can't open trace file: %s",
		  strerror (ERRNO));
	    sprintf (view[1], "filename: %s", bp);
	 }
      }
      else
      {
	 fclose (tracefd);
	 tracefd = NULL;
      }
      break;

   case 'w': /* set WP */
      bp = getnum (bp, &action);
      wpreg = action;
      action = WP_ACTION;
      oinput[0] = '\0';
      break;

   case 'z': /* it's Zee panel */
      enablepanel = enablepanel ? FALSE : TRUE;
      if (enablepanel)
      {
	 clearscreen ();
	 panel ();
      }
      break;

   default: puthelp ();
   }
   return (action);
}

/***********************************************************************
* readconfig - Read config file.
***********************************************************************/

static int
readconfig (char *config)
{
   FILE *fd;

   if ((fd = fopen (config, "r")) == NULL)
   {
      sprintf (view[0], "readconfig: open failed: %s",
	    strerror (ERRNO));
      sprintf (view[1], "filename: %s", config);
      return (-1);
   }
   while (fgets (input, sizeof (input), fd))
   {
      *(strchr (input,'\n')) = '\0';
      docommand (FALSE);
      if (view[0][0] != '\0')
      {
	 putview (FALSE);
	 break;
      }
   }
   fclose (fd);
   return (0);
}

/***********************************************************************
* main - Main Procedure.
***********************************************************************/

int
main (int argc, char **argv)
{
   char *bp;
   char *pp;
   char *config = NULL;
   int action;
   int i;

   /*
   ** Process args
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'C': /* delay Clock interval */
	    i++;
	    delayclock = atoi (argv[i]);
	    break;

	 case 'c': /* Config file spec */
	    i++;
	    config = argv[i];
	    break;

	 case 'm': /* CPU model */
	    i++;
	    model = 0;
	    for (pp = argv[i]; *pp; pp++)
	    {
	       if (isdigit(*pp))
	       {
		  model = (model * 10) + (*pp - '0');
	       }
	       else
	       {
	          if (*pp == 'a' || *pp == 'A')
		  {
		     if (model == 10) model = 11;
		     else goto BAD_MODEL;
		     break;
		  }
		  else
		  {
		  BAD_MODEL:
		     printf ("Invalid model specification %s\n", argv[i]);
		     exit (ABORT);
		  }
	       }
	    }
	    if (!(model ==  4 || model ==  5 || model == 9 ||
		  model == 10 || model == 11 || model == 12))
	    {
	       printf ("Unsupported CPU model: %s\n", argv[i]);
	       goto USAGE;
	    }
	    if (model >= 10 && memlen == 56*1024) /* new default for /10-/12 */
	       memlen = 256*1024;
	    break;

         case 'p': /* Panel enable */
            enablepanel = TRUE;
            break;

	 case 'r': /* ROM file spec */
	    i++;
	    strcpy (romfile, argv[i]);
	    break;

	 case 's': /* Size of memory */
	    i++;
	    memlen = 0;
	    for (pp = argv[i]; *pp; pp++)
	    {
	       if (isdigit(*pp))
	       {
		  memlen = (memlen * 10) + (*pp - '0');
	       }
	       else
	       {
	          if (*pp == 'm' || *pp == 'M')
		  {
		     memlen *= (1024*1024);
		     break;
		  }
		  else if (*pp == 'k' || *pp == 'K')
		  {
		     memlen *= 1024;
		     break;
		  }
		  else
		  {
		     printf ("Invalid memory specification %s\n", argv[i]);
		     exit (ABORT);
		  }
	       }
	    }
	    break;

         default:
      USAGE:
	    printf (
	 "usage:  sim990 [-c config][-m model][-p][-r romfile][-s memsize]\n");
	    return (ABORT);
         }
      }
   }

   /*
   ** Check memory size for CPU type
   */

   switch (model)
   {
      case 4:
      case 9:
         if (memlen > 56*1024)
	 {
	    printf ("Memory size %d too large for 990/%s CPU\n",
		     memlen, models[model-4]);
	    return (ABORT);
	 }
	 break;
      case 5:
         if (memlen > 64*1024)
	 {
	    printf ("Memory size %d too large for 990/%s CPU\n",
		     memlen, models[model-4]);
	    return (ABORT);
	 }
	 break;
      case 11:
         if (memlen > 512*1024)
	 {
	    printf ("Memory size %d too large for 990/%s CPU\n",
		     memlen, models[model-4]);
	    return (ABORT);
	 }
	 break;
      case 10:
      case 12:
	 if (memlen > 2*1024*1024)
	 {
	    printf ("Memory size %d too large for 990/%s CPU\n",
		     memlen, models[model-4]);
	    return (ABORT);
	 }
	 break;
   }

   /*
   ** Clear system
   */

   clearall ();
   tracelim[0] = 0;
   tracelim[1] = 0xFFFE;

   /*
   ** Set up initial devices
   */

   for (i = 0; i < MAXDEVICES; i++)
   {
      memset (&devices[i], '\0', sizeof (Device));
   }

   devices[0].type = CONIN;
   devices[0].fd = stdin;
   devices[0].devaddr = TERMINAL;
   devices[0].cdevaddr = 0;
   devices[0].intlvl = TERMINT;
   strcpy (devices[0].dev, "CI");
   strcpy (devices[0].file, "stdin");

   devices[1].type = CONOUT;
   devices[1].fd = stdout;
   devices[1].devaddr = TERMINAL;
   devices[1].cdevaddr = 0;
   devices[1].intlvl = TERMINT;
   strcpy (devices[1].dev, "CO");
   strcpy (devices[1].file, "stdout");

   devcnt = 2;

   ttinit ();

   /*
   ** Fire up clock thread
   */

   startclk (FALSE);

   /*
   ** Print intro
   */

   if (!enablepanel)
      printf ("%s %s: /%s CPU %dK\n",
	       INTRO, VERSION, models[model-4], memlen/1024);

   signal (SIGINT, sigintr);

   /*
   ** Config file specified, do the commands.
   */

   if (config)
   {
      readconfig (config);
   }

   /*
   ** Command loop
   */

   input[0] = oinput[0] = '\0';
   action = NULL_ACTION;

   while (action != QUIT_ACTION)
   {
      if (enablepanel) panel ();

      if (view[0][0] != '\0') putview (TRUE);

      if (enablepanel)
      {
	 screenposition (23,1);
	 clearline ();
	 printf (":%s", oinput);
	 screenposition (23,2);
      }
      else
         printf ("\n:");

      action = NULL_ACTION;
      if (fgets (input, sizeof (input), stdin) != NULL)
      {
	 *(strchr (input,'\n')) = '\0';

	 if (input[0] == '\0')
	    strcpy (input, oinput);
	 else
	    strcpy (oinput, input);
	 putview (TRUE);

	 action = docommand (TRUE);
      }
      else
      {
         printf ("fget returned NULL\n");
      }

   }

   /*
   ** Close open devices
   */

   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].fd != NULL && devices[i].fd != DEV_NULL)
      {
	 if (devices[i].dev[0] != 'V')
	 {
	    if (!(devices[i].fd == stdin || devices[i].fd == stdout))
	       fclose (devices[i].fd);
	 }
	 else
	 {
	    CLOSE ((int)devices[i].fd);
	 }
      }
   }

   ttclose ();
   screenposition (25,1);
   return (NORMAL);
}
