/***********************************************************************
*
* asmpass1 - Pass 1 for the TI 990 assembler.
*
* Changes:
*   05/21/03   DGP   Original.
*   07/02/03   DGP   Change hanging byte methods.
*   07/10/03   DGP   Mark inital REF as absolute.
*   07/15/03   DGP   Adjust symbol value on an EVEN.
*   08/13/03   DGP   Redo the way ops are handled.
*                    Added /12 instructions.
*   05/20/04   DGP   Added literal support.
*   05/25/04   DGP   Added Long REF/DEF pseudo ops.
*   06/07/04   DGP   Added LONG, QUAD, FLOAT and DOUBLE pseudo ops.
*	
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "asmdef.h"

extern int pc;
extern int model;
extern int genxref;
extern int xrefgened;
extern int rightmargin;
extern int linenum;
extern int rorgpc;
extern int symbolcount;
extern int absolute;
extern int asmskip;
extern int indorg;
extern int errcount;
extern int errnum;
extern int p1errcnt;
extern int pgmlength;
extern int codelength;
extern char inbuf[MAXLINE];
extern char errline[10][120];
extern SymNode *symbols[MAXSYMBOLS];
extern ErrTable p1error[MAXERRORS];
extern char ttlbuf[TTLSIZE+2];
extern char idtbuf[IDTSIZE+2];

static FILE *tmpfd;
static char cursym[32];

static int bytecnt;
static int litlen;
static int relocatable = TRUE;

/***********************************************************************
* p1literal - Process literal.
***********************************************************************/

static char *
p1literal (char *bp)
{
   pc += 2;
   bp++;
   if (*bp == 'E' || *bp == 'L')
      litlen += 4;
   else if (*bp == 'D' || *bp == 'Q')
      litlen += 8;
   else
      litlen += 2;
   return (bp);
}

/***********************************************************************
* p112mop - One word opcode 2 word memory operand processor.
***********************************************************************/

static void
p112mop (OpCode *op, char *bp)
{
   while (isspace (*bp)) bp++;
   if (*bp == MEMSYM)
   {
      pc += 2;
   }
   else if (*bp == LITERALSYM)
   {
      bp = p1literal (bp);
   }
   while (*bp && *bp != ',') bp++;
   if (*bp == ',')
   {
      bp++;
      if (*bp == MEMSYM)
      {
	 pc += 2;
      }
   }
   pc += 2;
}

/***********************************************************************
* p111mop - One word opcode 1 word memory operand processor.
***********************************************************************/

static void
p111mop (OpCode *op, char *bp)
{
   while (isspace (*bp)) bp++;
   if (*bp == MEMSYM)
   {
      pc += 2;
   }
   else if (*bp == LITERALSYM)
   {
      bp = p1literal (bp);
   }
   pc += 2;
}

/***********************************************************************
* p111iop - One word opcode 1 word immediate operand processor.
***********************************************************************/

static void
p111iop (OpCode *op, char *bp)
{
   if (op->opmod != 2) pc += 2;
   pc += 2;
}

/***********************************************************************
* p110mop - One word opcode 0 word memory operand processor.
***********************************************************************/

static void
p110mop (OpCode *op, char *bp)
{
   pc += 2;
}

/***********************************************************************
* p122mop - Two word opcode 2 word memory operand processor.
***********************************************************************/

static void
p122mop (OpCode *op, char *bp)
{
   while (isspace (*bp)) bp++;
   if (*bp == MEMSYM)
   {
      pc += 2;
   }
   else if (*bp == LITERALSYM)
   {
      bp = p1literal (bp);
   }
   while (*bp && *bp != ',') bp++;
   if (*bp == ',')
   {
      bp++;
      if (*bp == MEMSYM)
      {
	 pc += 2;
      }
   }
   pc += 4;
}

/***********************************************************************
* p122cop - Two word opcode 2 word memory operand processor with condition.
***********************************************************************/

static void
p122cop (OpCode *op, char *bp)
{
   while (isspace (*bp)) bp++;
   while (*bp && *bp != ',') bp++;
   if (*bp == ',')
   {
      bp++;
      if (*bp == MEMSYM)
      {
	 pc += 2;
      }
      else if (*bp == LITERALSYM)
      {
	 bp = p1literal (bp);
      }
      while (*bp && *bp != ',') bp++;
      if (*bp == ',')
      {
	 bp++;
	 if (*bp == MEMSYM)
	 {
	    pc += 2;
	 }
      }
   }
   pc += 4;
}

/***********************************************************************
* p121mop - Two word opcode 1 word memory operand processor.
***********************************************************************/

static void
p121mop (OpCode *op, char *bp)
{
   while (isspace (*bp)) bp++;
   if (*bp == MEMSYM)
   {
      pc += 2;
   }
   else if (*bp == LITERALSYM)
   {
      bp = p1literal (bp);
   }
   pc += 4;
}

/***********************************************************************
* p120mop - Two word opcode 0 word memory operand processor.
***********************************************************************/

static void
p120mop (OpCode *op, char *bp)
{
   pc += 4;
}

/***********************************************************************
* p1pop - Process Pseudo ops.
***********************************************************************/


static void
p1pop (OpCode *op, char *bp)
{
   SymNode *s;
   char *cp;
   char *token;
   int tokentype;
   int val;
   char term;

   switch (op->opvalue)
   {

   case AORG_T:
      if (!absolute && !indorg) rorgpc = pc;
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 val = 0;
      }
      absolute = TRUE;
      relocatable = FALSE;
      indorg = FALSE;
      pc = val;
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = val;
	 s->relocatable = FALSE;
      }
      break;

   case ASMIF_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      asmskip = val ? FALSE : TRUE;
      break;

   case ASMELS_T:
      asmskip = asmskip ? FALSE : TRUE;
      break;

   case ASMEND_T:
      asmskip = FALSE;
      break;

   case BES_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 val = 0;
      }
      pc += val;
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
	 s->relocatable = relocatable;
      }
      break;

   case BSS_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 val = 0;
      }
      pc += val;
      break;

   case BYTE_T:
      do {
	 bp = exprscan (bp, &val, &term, &relocatable, 2, TRUE, 0);
	 bytecnt++;
	 pc++;
      } while (term == ',');
      break;

   case COPY_T:
      tmpfd = opencopy (bp, tmpfd);
      break;

   case DATA_T:
      if (pc & 0x0001) pc++;
#ifdef DEBUGPC
      printf ("DATA pc = %04X\n", pc);
#endif
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
      }
      do {
	 bp = exprscan (bp, &val, &term, &relocatable, 2, TRUE, 0);
	 pc += 2;
#ifdef DEBUGPC
	 printf ("    pc = %04X\n", pc);
#endif
      } while (term == ',');
      break;

   case DEF_T:
      break;

   case DORG_T:
      if (!absolute && !indorg) rorgpc = pc;
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 val = 0;
      }
      absolute = TRUE;
      indorg = TRUE;
      pc = val;
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = val;
	 s->relocatable = FALSE;
      }
      break;

   case DOUBLE_T:
   case QUAD_T:
      if (pc & 0x0001) pc++;
#ifdef DEBUGPC
      printf ("DOUBLE/QUAD pc = %04X\n", pc);
#endif
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
      }
      do {
	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 pc += 8;
#ifdef DEBUGPC
	 printf ("    pc = %04X\n", pc);
#endif
      } while (term == ',');
      break;

   case DXOP_T:
      bp = tokscan (bp, &token, &tokentype, &val, &term);
      if (tokentype == SYM)
      {
	 char temp[20];

	 sprintf (temp, "!!%s", token);
	 s = symlookup (temp, TRUE, TRUE);
	 if (term == ',')
	 {
	    bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
	    if (val < 0 || val > 15)
	    {
	       errcount++;
	       p1error[p1errcnt].errorline = linenum;
	       p1error[p1errcnt].errortext = (char *)malloc (120);
	       sprintf (p1error[p1errcnt].errortext, "Invalid DXOP value: %d",
			val);
	       p1errcnt++;
	       val = 0;
	    }
	    s->value = val;
	    s->relocatable = FALSE;
	    s->xop = TRUE;
	 }
      }
      else
      {
	 errcount++;
	 p1error[p1errcnt].errorline = linenum;
	 p1error[p1errcnt].errortext = (char *)malloc (120);
	 sprintf (p1error[p1errcnt].errortext, "Invalid DXOP symbol: %s",
		  token);
	 p1errcnt++;
      }
      break;

   case END_T:
      codelength = pc;
      pgmlength = pc + litlen;
      break;

   case EQU_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
#ifdef DEBUGPC
      printf ("EQU pc = %04X, val = %04X\n", pc, val);
#endif
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = val;
	 s->relocatable = relocatable;
      }
      break;

   case EVEN_T:
      if (pc & 0x0001) pc++;
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
	 s->relocatable = relocatable;
      }
      break;

   case FLOAT_T:
   case LONG_T:
      if (pc & 0x0001) pc++;
#ifdef DEBUGPC
      printf ("FLOAT/LONG pc = %04X\n", pc);
#endif
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
      }
      do {
	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 pc += 4;
#ifdef DEBUGPC
	 printf ("    pc = %04X\n", pc);
#endif
      } while (term == ',');
      break;

   case IDT_T:
      if (!idtbuf[0])
      {
	 while (isspace(*bp)) bp++;
	 if (*bp == '\'')
	 {
	    cp = ++bp;
	    while (*bp != '\'' && *bp != '\n') bp++;
	 }
	 *bp = '\0';
	 if (strlen(idtbuf) > IDTSIZE)
	 {
	    *(cp+IDTSIZE) = '\0';
	 }
	 strcpy (idtbuf, cp);
      }
      break;

   case LDEF_T:
      break;

   case LOAD_T:
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    p1error[p1errcnt].errorline = linenum;
	    p1error[p1errcnt].errortext = (char *)malloc (120);
	    sprintf (p1error[p1errcnt].errortext, "LOAD requires symbol: %s",
		     token);
	    p1errcnt++;
	    return;
	 }
	 if (strlen(token) > MAXSYMLEN)
	 {
	    token[MAXSYMLEN] = '\0';
	 }
	 if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	 {
	    s = symlookup (token, TRUE, FALSE);
	 }
	 s->value = 0;
	 s->relocatable = FALSE;
	 s->load = TRUE;
      } while (term == ',');
      break;

   case LREF_T:
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    p1error[p1errcnt].errorline = linenum;
	    p1error[p1errcnt].errortext = (char *)malloc (120);
	    sprintf (p1error[p1errcnt].errortext, "LREF requires symbol: %s",
		     token);
	    p1errcnt++;
	    return;
	 }
	 if (strlen(token) > MAXSYMLEN)
	 {
	    token[MAXSYMLEN] = '\0';
	 }
	 if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	 {
	    s = symlookup (token, TRUE, FALSE);
	 }
	 s->value = 0;
	 s->relocatable = FALSE;
	 s->external = TRUE;
	 s->longsym = TRUE;
      } while (term == ',');
      break;

   case NOP_T:
   case RT_T:
      if (pc & 0x0001) pc++;
      pc += 2;
      break;

   case OPTION_T:
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype == SYM)
	 {
	    int j = strlen (token);
	    if (!(strncmp (token, "XREF", j))) genxref = TRUE;
	 }
	 else if (tokentype == DECNUM)
	 {
	    if (val == 10) model = 10;
	    else if (val == 12) model = 12;
	 }
      } while (term == ',');
      break;

   case PAGE_T:
      break;

   case REF_T:
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    p1error[p1errcnt].errorline = linenum;
	    p1error[p1errcnt].errortext = (char *)malloc (120);
	    sprintf (p1error[p1errcnt].errortext, "REF requires symbol: %s",
		     token);
	    p1errcnt++;
	    return;
	 }
	 if (strlen(token) > MAXSYMLEN)
	 {
	    token[MAXSYMLEN] = '\0';
	 }
	 if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	 {
	    s = symlookup (token, TRUE, FALSE);
	 }
	 s->value = 0;
	 s->relocatable = FALSE;
	 s->external = TRUE;
      } while (term == ',');
      break;

   case RORG_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      indorg = FALSE;
      absolute = FALSE;
      relocatable = TRUE;
      if (val == 0)
	 pc = rorgpc;
      else
         pc = val;
      if (cursym[0])
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 s->value = pc;
	 s->relocatable = TRUE;
      }
      break;

   case SETRM_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val >= 10 && val <= 80)
	 rightmargin = val;
      else
      {
	 errcount++;
	 p1error[p1errcnt].errorline = linenum;
	 p1error[p1errcnt].errortext = (char *)malloc (120);
	 sprintf (p1error[p1errcnt].errortext, "Invalid SETRM value: %d",
		  val);
	 p1errcnt++;
      }
      break;

   case SREF_T:
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    p1error[p1errcnt].errorline = linenum;
	    p1error[p1errcnt].errortext = (char *)malloc (120);
	    sprintf (p1error[p1errcnt].errortext, "SREF requires symbol: %s",
		     token);
	    p1errcnt++;
	    return;
	 }
	 if (strlen(token) > MAXSYMLEN)
	 {
	    token[MAXSYMLEN] = '\0';
	 }
	 if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	 {
	    s = symlookup (token, TRUE, FALSE);
	 }
	 s->value = 0;
	 s->relocatable = FALSE;
	 s->sref = TRUE;
      } while (term == ',');
      break;

   case TEXT_T:
      bp = tokscan (bp, &token, &tokentype, &val, &term);
      if (term == '-')
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
      if (tokentype == STRING)
      {
	 bytecnt = strlen(token);
	 pc += bytecnt;
      }
      break;

   case TITL_T:
      if (!ttlbuf[0])
      {
	 while (isspace(*bp)) bp++;
	 if (*bp == '\'')
	 {
	    cp = ++bp;
	    while (*bp != '\'' && *bp != '\n') bp++;
	 }
	 *bp = '\0';
	 strcpy (ttlbuf, cp);
      }
      break;

   case UNL_T:
      break;

   case XVEC_T:
      pc += 4;
      break;

   default: ;
   }
}

/***********************************************************************
* asmpass1 - Pass 1
***********************************************************************/

int
asmpass1 (FILE *infd)
{
   char *token;
   int status = 0;
   int val;
   int tokentype;
   char term;

#ifdef DEBUG
   printf ("asmpass1: Entered\n");
#endif

   /*
   ** Rewind the input.
   */

   tmpfd = infd;
   if (fseek (tmpfd, 0, SEEK_SET) < 0)
   {
      perror ("asm990: Can't rewind temp file");
      return (-1);
   }

   /*
   ** Process the source.
   */

   pc = 0;
   rorgpc = 0;
   litlen = 0;

   absolute = FALSE;
   asmskip = FALSE;
   indorg = FALSE;

   linenum = 0;
   ttlbuf[0] = '\0';
   idtbuf[0] = '\0';

DOREAD:
   while (fgets(inbuf, MAXLINE, tmpfd))
   {
      char *bp;

#ifdef DEBUG
      printf ("in = %s", inbuf);
#endif
      errnum = 0;
      errline[0][0] = '\0';
      linenum++;
      xrefgened = FALSE;
      bp = inbuf;

      /*
      ** If not a comment, then process.
      */

      if (*bp != COMMENTSYM)
      {
	 SymNode *s;
	 OpCode *op;

	 /*
	 ** If label present, add to symbol table.
	 */

         if (isalpha(*bp) || *bp == '$' || *bp == '_')
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    strcpy (cursym, token);
	    if (strlen(token) > MAXSYMLEN)
	    {
	       cursym[MAXSYMLEN] = '\0';
	       token[MAXSYMLEN] = '\0';
	    }
	    if (!asmskip)
	    {
	       if ((s = symlookup (token, TRUE, FALSE)) == NULL)
	       {
		  errcount++;
		  p1error[p1errcnt].errorline = linenum;
		  p1error[p1errcnt].errortext = (char *)malloc (120);
		  sprintf (p1error[p1errcnt].errortext, "Duplicate symbol: %s",
			   token);
		  p1errcnt++;
		  status = -1;
	       }
	    }
	    if (term == '\n') continue;
	 }
	 else 
	 {
	    cursym[0] = '\0';
	    while (isspace (*bp)) bp++;
	 }

	 /*
	 ** Scan off opcode.
	 */

	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (!token[0]) continue;
#ifdef DEBUGPC
	 printf ("INST = %s, pc = %04X\n", token, pc);
#endif

	 op = oplookup (token);
	 if (asmskip)
	 {
	    if (op != NULL && op->optype == TYPE_P)
	       switch (op->opvalue)
	       {
	       case ASMELS_T:
	       case ASMEND_T:
	          p1pop (op, bp);
		  break;

	       default: ;
	       }
	 }
	 else if (op != NULL)
	 {
	    while (*bp && isspace(*bp)) bp++;

	    bytecnt = 0;
	    if ((op->optype != TYPE_P) && (pc & 0x0001)) pc++;
	    switch (op->optype)
	    {

	       case TYPE_1:
	          p112mop (op, bp);
		  break; 

	       case TYPE_2:
	       case TYPE_5:
	       case TYPE_7:
	       case TYPE_10:
	       case TYPE_18:
	          p110mop (op, bp);
		  break; 

	       case TYPE_3:
	       case TYPE_6:
	       case TYPE_4:
	       case TYPE_9:
	          p111mop (op, bp);
		  break; 

	       case TYPE_8:
	          p111iop (op, bp);
		  break; 

	       case TYPE_11:
	       case TYPE_12:
	       case TYPE_16:
	       case TYPE_19:
	       case TYPE_21:
	          p122mop (op, bp);
		  break; 

	       case TYPE_20:
	          p122cop (op, bp);
		  break; 

	       case TYPE_13:
	       case TYPE_14:
	       case TYPE_15:
	          p121mop (op, bp);
		  break; 

	       case TYPE_17:
	          p120mop (op, bp);
		  break; 

	       case TYPE_P:
	          p1pop (op, bp);
	          break;

	       default: ;
	    }
	 }
	 else
	 {
	    /* print errors in pass2 */
	    status = -1;
	 }
	 
      }

   }
   if ((tmpfd = closecopy (tmpfd)) != NULL) goto DOREAD;

   return (status);
}
