/***********************************************************************
*
* asmpass2.c - Pass 2 for the TI 990 assembler.
*
* Changes:
*   05/21/03   DGP   Original.
*   06/15/03   DGP   Added registerarg function.
*   06/23/03   DGP   Added date/time and creator to EOF record.
*   07/01/03   DGP   Fixed TEXT/BYTE processing.
*                    Fixed negative REF/DEF equ'd values.
*   07/02/03   DGP   Change hanging byte methods.
*                    Fixed register expressions.
*   07/08/03   DGP   Moved get date/time to always get it.
*   07/10/03   DGP   Fixed EVEN to use absolute for new org.
*   07/17/03   DGP   Fixed BSS & BES hanging byte.
*   08/12/03   DGP   Revert to one word per line listing.
*                    Add type 10 opcode (LMF).
*   08/13/03   DGP   Added /12 instructions.
*   12/01/03   DGP   Insure pgmlength is 16 bits in IDT.
*   04/08/04   DGP   Added "cassette" mode and orgout.
*   04/09/04   DGP   Added object checksum.
*   05/20/04   DGP   Added literal support.
*   05/25/04   DGP   Added Long REF/DEF pseudo ops.
*   06/07/04   DGP   Added LONG, QUAD, FLOAT and DOUBLE pseudo ops.
*	
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "asmdef.h"

extern int pc;
extern int model;
extern int rorgpc;
extern int genxref;
extern int xrefgened;
extern int symbolcount;
extern int asmskip;
extern int errcount;
extern int errnum;
extern int absolute;
extern int cassette;
extern int linenum;
extern int indorg;
extern int p1errcnt;
extern int pgmlength;
extern int codelength;
extern char errline[10][120];
extern char inbuf[MAXLINE];
extern SymNode *symbols[MAXSYMBOLS];
extern ErrTable p1error[MAXERRORS];

char ttlbuf[TTLSIZE+2];
char idtbuf[IDTSIZE+2];

static FILE *tmpfd;
static int linecnt = MAXLINE;
static int curp1err = 0;
static int objrecnum = 0;
static int objcnt = 0;
static int pagenum = 0;
static int printed = FALSE;
static int listing = TRUE;
static int listmode;
static char datebuf[48];
static char pcbuf[6];
static char objbuf[80];
static char objrec[82];
static char opbuf[32];
static char cursym[32];
static char lstbuf[MAXLINE];
static struct tm *timeblk;

static uint16 byteacc;
static int bytecnt;
static int ckpt = -1;

static int orgout = FALSE;
static int bunlist = FALSE;
static int dunlist = FALSE;
static int munlist = FALSE;
static int tunlist = FALSE;
static int allsymbols = FALSE;

/***********************************************************************
* printheader - Print header on listing.
***********************************************************************/

static void
printheader (FILE *lstfd)
{
   if (listmode && listing)
   {
      if (linecnt > LINESPAGE)
      {
	 linecnt = 0;
	 if (pagenum) fputc ('\f', lstfd);
	 fprintf (lstfd, H1FORMAT,
		  VERSION,
		  idtbuf,
		  datebuf,
		  ++pagenum);
	 fprintf (lstfd, H2FORMAT, ttlbuf); 
      }
   }
}

/***********************************************************************
* printline - Print line on listing.
*
*0        1         2         3         4
*1234567890123456789012345678901234567890
*
* oooo  FFFF  LLLL  TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
* oooo  FFFF  
*
***********************************************************************/

static void
printline (FILE *lstfd)
{
   if (listmode && listing)
   {
      printheader (lstfd);
      fprintf (lstfd, L1FORMAT, pcbuf, opbuf,
	       linenum, lstbuf);
      linecnt++;
   }
}

static void
printdata (FILE *lstfd, int opc)
{
   if (listmode && listing)
   {
      printheader (lstfd);
      sprintf (pcbuf, PCFORMAT, opc);
      fprintf (lstfd, L2FORMAT, pcbuf, opbuf);
      linecnt++;
   }
}

static void
printlit (FILE *lstfd, int opc, char *lit)
{
   if (listmode && listing)
   {
      printheader (lstfd);
      sprintf (pcbuf, PCFORMAT, opc);
      fprintf (lstfd, LITFORMAT, pcbuf, opbuf, lit);
      linecnt++;
   }
}

/***********************************************************************
* printsymbols - Print the symbol table.
***********************************************************************/

static void
printsymbols (FILE *lstfd)
{
   int i, j;
   int longformat;
   char type;

   if (!listmode) return;

   linecnt = MAXLINE;
   longformat = FALSE;
   j = 0;

   for (i = 0; i < symbolcount; i++) if (symbols[i]->longsym) longformat = TRUE;

   for (i = 0; i < symbolcount; i++)
   {
      printheader (lstfd);

      if (genxref && !linecnt)
      {
	 if (longformat)
	    fprintf (lstfd, LONGXREFHEADER);
	 else
	    fprintf (lstfd, XREFHEADER);
	 linecnt = 2;
      }

      if (!(symbols[i]->xop) && (symbols[i]->symbol[0] != LITERALSYM))
      {
	 SymNode *sym = symbols[i];

	 if (sym->global) type = 'G';
	 else if (sym->external) type = 'E';
	 else if (sym->load) type = 'L';
	 else if (sym->sref) type = 'S';
	 else if (sym->relocatable) type = '\'';
	 else type = ' ';

	 if (longformat)
	 {
	    fprintf (lstfd, LONGSYMFORMAT,
		     sym->symbol,
		     sym->value & 0xFFFF,
		     type);
	    if (!genxref) j = 6;
	 }
	 else
	 {
	    fprintf (lstfd, SYMFORMAT,
		     sym->symbol,
		     sym->value & 0xFFFF,
		     type);
	    j++;
	 }

	 if (genxref)
	 {
	    int refcnt;
	    XrefNode *xref = sym->xref_head;

	    j = 0;
	    if (longformat)
	       refcnt = 5;
	    else
	       refcnt = 9;
	    fprintf (lstfd, " %4d ", symbols[i]->line);
	    while (xref)
	    {
	       if (j >= refcnt)
	       {
		  fprintf (lstfd, "\n");
		  printheader (lstfd);
	          if (!linecnt)
		  {
		     if (longformat)
			fprintf (lstfd, LONGXREFHEADER);
		     else
			fprintf (lstfd, XREFHEADER);
		     linecnt = 2;
		  }
		  fprintf (lstfd, "                      ");
		  linecnt++;
		  j = 0;
	       }
	       fprintf (lstfd, " %4d ", xref->line);
	       xref = xref->next;
	       j++;
	    }
	    j = 6;
	 }

	 if (j >= 5)
	 {
	    fprintf (lstfd, "\n");
	    linecnt++;
	    j = 0;
	 }
      }
   }
   fprintf (lstfd, "\n");
}

/***********************************************************************
* punchfinish - Punch a record with sequence numbers.
***********************************************************************/

static void
punchfinish (FILE *outfd)
{
   if (objcnt)
   {
      int i;
      short int cksum;
      char temp[10];

      cksum = 0;
      for (i = 0; i < objcnt; i++) cksum += objrec[i];
      cksum += CKSUM_TAG;
      cksum = -cksum;
      sprintf (temp, "%c%04X%c", CKSUM_TAG, cksum & 0xFFFF, EOR_TAG);
      strncpy (&objrec[objcnt], temp, 6);
      objcnt += 6;

      if (cassette)
      {
	 strcpy (&objrec[objcnt], "\n");
      }
      else
      {
	 sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
      }
      fputs (objrec, outfd);
      memset (objrec, ' ', sizeof(objrec));
      objcnt = 0;
   }
}

/***********************************************************************
* punchrecord - Punch an object value into record.
***********************************************************************/

static void 
punchrecord (FILE *outfd)
{
   int len = strlen (objbuf);

   if (!indorg)
   {
      if (objcnt+len >= (cassette ? CHARSPERCASREC : CHARSPERREC))
      {
	 punchfinish (outfd);
      }
      if (!orgout && !absolute)
      {
	 char temp[10];
	 sprintf (temp, OBJFORMAT, RELORG_TAG, 0);
	 strncpy (&objrec[objcnt], temp, 5);
	 objcnt += 5;
	 orgout = TRUE;
      }
      strncpy (&objrec[objcnt], objbuf, len);
      objcnt += len;
      objbuf[0] = '\0';
   }
}

/***********************************************************************
* punchsymbols - Punch REF and DEF symbols.
***********************************************************************/

static void
punchsymbols (FILE *outfd)
{
   int i;

   for (i = 0; i < symbolcount; i++)
   {
      if (symbols[i]->external)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LREFFORMAT,
		  symbols[i]->relocatable ? LRELEXTRN_TAG : LABSEXTRN_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, REFFORMAT,
		  symbols[i]->relocatable ? RELEXTRN_TAG : ABSEXTRN_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         
         punchrecord (outfd);
      }
      else if (symbols[i]->global)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LDEFFORMAT,
		  symbols[i]->relocatable ? LRELGLOBAL_TAG : LABSGLOBAL_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, DEFFORMAT,
		  symbols[i]->relocatable ? RELGLOBAL_TAG : ABSGLOBAL_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         punchrecord (outfd);
      }
      else if (symbols[i]->load)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LREFFORMAT,
		  LLOAD_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, REFFORMAT,
		     LOAD_TAG,
		     0, symbols[i]->symbol);
         
         punchrecord (outfd);
      }
      else if (symbols[i]->sref)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LREFFORMAT,
		  symbols[i]->relocatable ? LRELSREF_TAG : LABSSREF_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, REFFORMAT,
		  symbols[i]->relocatable ? RELSREF_TAG : ABSSREF_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         
         punchrecord (outfd);
      }
      else if (allsymbols)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LDEFFORMAT,
		  symbols[i]->relocatable ? LRELSYMBOL_TAG : LABSSYMBOL_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	 sprintf (objbuf, DEFFORMAT,
	       symbols[i]->relocatable ? RELSYMBOL_TAG : ABSSYMBOL_TAG,
	       symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         punchrecord (outfd);
         
      }
   }
}

/***********************************************************************
* puncheof - Punch EOF mark.
***********************************************************************/

static void
puncheof (FILE *outfd)
{
   char temp[80];

   punchfinish (outfd);
   objrec[0] = EOFSYM;
   if (cassette)
   {
      strcpy (&objrec[1], "\n");
   }
   else
   {
      sprintf (temp, "%-8.8s  %02d/%02d/%02d  %02d:%02d:%02d    ASM990 %s",
	       idtbuf,
	       timeblk->tm_mon+1, timeblk->tm_mday, timeblk->tm_year - 100,
	       timeblk->tm_hour, timeblk->tm_min, timeblk->tm_sec,
	       VERSION);
      strncpy (&objrec[7], temp, strlen(temp));
      sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
   }
   fputs (objrec, outfd);
}

/***********************************************************************
* outputdataword - Output a word of data.
***********************************************************************/

static int
outputdataword (FILE *outfd, FILE *lstfd, int opc, int val, int first)
{
   sprintf (opbuf, DATA1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);
   pc += 2;
   printed = TRUE;
   if (first)
   {
      printline (lstfd);
   }
   else
   {
      if (!dunlist)
	 printdata (lstfd, opc);
   }
   strcpy (opbuf, OPBLANKS);
   return (pc);
}

/***********************************************************************
* outputlitword - Ouput a literal word value.
***********************************************************************/

static void 
outputlitword (FILE *outfd, FILE *lstfd, int d, int first, char *sym)
{
   printed = TRUE;
   if (listmode)
   {
      sprintf (opbuf, OP1FORMAT, d, ' ');
      if (first)
	 printlit (lstfd, pc, sym);
      else if (!dunlist)
	 printdata (lstfd, pc);
   }
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, d);
   punchrecord (outfd);
   pc += 2;
}

/***********************************************************************
* processliterals - Process literals.
***********************************************************************/

static int
processliterals (FILE *outfd, FILE *lstfd, int listmode)
{
   char *cp;
   int i;
   int lrelo;
   int literalcount = 0;
   int escapecount = 0;
   int status = 0;
   int datapunched;

#ifdef DEBUGLITERAL
   printf ("processliterals: entered: listmode = %s\n",
	   listmode ? "TRUE" : "FALSE");
#endif

   for (i = 0; i < symbolcount; i++)
   {
      if (symbols[i]->symbol[0] == LITERALSYM) literalcount++;
   }
   escapecount = literalcount * 2; /* just in case */

#ifdef DEBUGLITERAL
   printf ("   literalcount = %d\n", literalcount);
#endif

   while (literalcount)
   {
      if (--escapecount == 0) break;
      for (i = 0; i < symbolcount; i++)
      {
	 if (symbols[i]->symbol[0] == LITERALSYM && symbols[i]->value == pc)
	 {
	    t_uint64 qd;
	    unsigned long ld;
	    int d;

#ifdef DEBUGLITERAL
	    printf ("   literal = %s, value = %04X, pc = %04X\n",
	       symbols[i]->symbol, symbols[i]->value, pc);
#endif
	    lrelo = FALSE;
	    cp = &symbols[i]->symbol[1];
	    errline[0][0] = '\0';
	    printed = FALSE;
	    datapunched = FALSE;

	    switch (*cp)
	    {
	    case '@': /* Address literal */
	       {
		  SymNode *s;

		  cp++;
		  if ((s = symlookup (cp, FALSE, FALSE)) == NULL)
		  {
		     sprintf (errline[0], "Literal symbol %s not found", cp);
		     d = 0;
		  }
		  else
		  {
		     d = s->value;
		     lrelo = s->relocatable;
		  }
	       }
	       break;

	    case 'B': /* Byte decimal */
	       cp++;
	       d = atoi (cp) << 8;
	       break;

	    case 'D': /* Extended precision float */
	       cp++;
	       qd = cvtdouble (cp);

	       d = (int)((qd >> 48) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, TRUE, symbols[i]->symbol);
	       d = (int)((qd >> 32) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       d = (int)((qd >> 16) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       d = (int)(qd & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       datapunched = TRUE;
	       break;
	       
	    case 'F': /* Float */
	       cp++;
	       ld = cvtfloat (cp);

	       d = (int)((ld >> 16) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, TRUE, symbols[i]->symbol);
	       d = (int)(ld & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       datapunched = TRUE;
	       break;

	    case 'L': /* Long decimal */
	       cp++;
	       ld = atoi (cp);

	       d = (int)((ld >> 16) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, TRUE, symbols[i]->symbol);
	       d = (int)(ld & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       datapunched = TRUE;
	       break;

	    case 'O': /* Octal word */
	       cp++;
	       d = strtol (cp, NULL, 8);
	       break;

	    case 'Q': /* Quad decimal */
	       cp++;
#ifdef WIN32
	       sscanf (cp, "%I64d", &qd);
#else
	       sscanf (cp, "%lld", &qd);
#endif

	       d = (int)((qd >> 48) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, TRUE, symbols[i]->symbol);
	       d = (int)((qd >> 32) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       d = (int)((qd >> 16) & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       d = (int)(qd & 0xFFFF);
	       outputlitword (outfd, lstfd, d, FALSE, symbols[i]->symbol);
	       datapunched = TRUE;
	       break;

	    case 'W': /* Word decimal */
	       cp++;
	       d = atoi (cp);
	       break;

	    case 'X': /* Hex word */
	       cp++;
	       d = strtol (cp, NULL, 16);
	       break;

	    default:
	       sprintf (errline[0], "Invalid literal: %s", cp);
	       d = 0;
	    }

	    if (errline[0][0])
	    {
	       errcount++;
	       if (listmode)
	       {
		  fprintf (lstfd, "ERROR: %s\n", errline[0]);
		  linecnt++;
	       }
	       else
	       {
		  fprintf (stderr, "asm990: %d: %s\n",
			 linenum, errline[0]);
	       }
	       status = -1;
	    }

	    if (!datapunched)
	    {
	       d &= 0xFFFF;

	       sprintf (objbuf, OBJFORMAT,
			lrelo ? RELDATA_TAG : ABSDATA_TAG, d);

	       if (listmode && !printed)
	       {
		  sprintf (opbuf, OP1FORMAT, d, lrelo ? '\'' : ' ');
		  printlit (lstfd, pc, symbols[i]->symbol);
	       }
	       punchrecord (outfd);

	       pc += 2;
	    }

	    literalcount--;
	    break;
	 }
      }
   }
   return (status);
}

/***********************************************************************
* generalarg - Process general operand arg.
***********************************************************************/

static char *
generalarg (OpCode *op, char *bp, int *ts, int *tr, int *saddr, int *srelo,
int *spc, char *term, int pcinc, int src)
{
   char *cp;
   int laddr = -1;
   int lr = 0;
   int lrelo = 0;
   int val;
   int lpc = 0;
   char lterm;

   while (isspace(*bp)) bp++;
   lpc = *spc;
   *ts = 0;
   if (*bp == MEMSYM)
   {
      char *ep;
      bp++;
      *ts = 0x2;
      lpc += 2;
      cp = bp;
      while (*cp && !isspace(*cp) && *cp != ',') cp++;
      ep = cp;
      /*
      ** Isolate parens for indexing, there may be parens in addr expr.
      */
      if (*(cp-1) == ')')
      {
	 cp --;
	 *cp = '\0';
	 cp --;
	 while (*cp != '(') cp--;
	 *cp++ = '\0';
         cp = exprscan (cp, &lr, &lterm, &val, 2, FALSE, 0);
      }
      bp = exprscan (bp, &laddr, &lterm, &lrelo, 2, FALSE, pcinc);
      laddr = laddr & 0xFFFF;
      lterm = *ep++;
      bp = ep;
   }
   else if (*bp == LITERALSYM)
   {
      if (src)
      {
	 char *ep;

	 *ts = 0x2;
	 lpc += 2;
	 cp = bp;
	 while (*cp && !isspace(*cp) && *cp != ',') cp++;
	 ep = cp;
	 bp = exprscan (bp, &laddr, &lterm, &lrelo, 2, FALSE, pcinc);
	 laddr = laddr & 0xFFFF;
	 lterm = *ep++;
	 bp = ep;
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++],
		  "Literal not allowed for destination operand");
      }
   }
   else
   {
      if (*bp == '*')
      {
	 bp++;
	 *ts = 0x1;
	 cp = bp;
	 while (!isspace(*cp) && *cp != ',') cp++;
	 if (*(cp-1) == '+')
	 {
	    *ts = 0x3;
	    *(cp-1) = '\0';
	    bp = exprscan (bp, &lr, &lterm, &val, 2, FALSE, 0);
	    lterm = *cp++;
	    bp = cp;
	 }
	 else
	    bp = exprscan (bp, &lr, &lterm, &val, 2, FALSE, 0);
      }
      else
	 bp = exprscan (bp, &lr, &lterm, &val, 2, FALSE, 0);
   }

   if (lr < 0 || lr > 15)
   {
      errcount++;
      sprintf (errline[errnum++], "Invalid register value: %d", lr);
      lr = 0;
   }

   *saddr = laddr;
   *tr = lr;
   *srelo = lrelo;
   *spc = lpc;
   *term = lterm;

   return (bp);
}

/***********************************************************************
* registerarg - Process register operand arg.
***********************************************************************/

static char *
registerarg (char *bp, int *reg, char *term)
{
   int rd;
   int junk;
   char lt;

   while (isspace(*bp)) bp++;
   if (*bp == MEMSYM || *bp == LITERALSYM)
   {
	 errcount++;
	 sprintf (errline[errnum++], "Not a register: %s", bp);
	 rd = 0;
   }
   else
   {

      bp = exprscan (bp, &rd, &lt, &junk, 2, FALSE, 0);
      if (rd < 0 || rd > 15)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid register value: %d", rd);
	 rd = 0;
      }
   }
   *reg = rd;
   *term = lt;
   return (bp);
}

/***********************************************************************
* countarg - Process count operand arg.
***********************************************************************/

static char *
countarg (char *bp, int *cnt, int size, char *type, char *term)
{
   int ct;
   int junk;
   char lt;

   while (isspace(*bp)) bp++;
   if (*bp == MEMSYM || *bp == LITERALSYM)
   {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid %s field: %s", type, bp);
	 ct = 0;
   }
   else
   {

      bp = exprscan (bp, &ct, &lt, &junk, 2, FALSE, 0);
      if (ct < 0 || ct > size)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid %s value: %d", type, ct);
	 ct = 0;
      }
   }
   *cnt = ct;
   *term = lt;
   return (bp);
}

/***********************************************************************
* p201op - Process type 1 operation code.
***********************************************************************/

static void
p201op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int td = 0;
   int ts = 0;
   int rd = 0;
   int rs = 0;
   int daddr = -1;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 2,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 2,
		       FALSE);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }

   val = op->opvalue | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 2 + spc;
}

/***********************************************************************
* p202op - Process type 2 operation code.
***********************************************************************/

static void
p202op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int addr = 0;
   int val;
   int relocatable;
   int disp;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);

   bp = exprscan (bp, &addr, &term, &relocatable, 2, FALSE, 0);
   if (op->opmod == 0) /* A Jump instruction */
   {
      if (*bp == MEMSYM || *bp == LITERALSYM)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Not a valid jump operand: %s", bp);
	 disp = 0;
      }
      else
      {
	 disp = addr/2 - (pc+2)/2;
	 if (disp < -128 || disp > 128)
	 {
	       errcount++;
	       sprintf (errline[errnum++], "Jump displacement too large: %d",
			disp);
	       disp = 0;
	 }
      }
   }
   else	/* A CRU instruction */
   {
      disp = addr;
   }
   val = op->opvalue | (disp & 0xFF);
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   pc += 2;
}

/***********************************************************************
* p203op - Process type 3 operation code.
***********************************************************************/

static void
p203op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int rd = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, 2, TRUE);
   if (term == ',')
   {
      bp = registerarg (bp, &rd, &term);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing register operand");
   }

   val = op->opvalue | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 2 + spc;
}

/***********************************************************************
* p204op - Process type 4 operation code.
***********************************************************************/

static void
p204op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int count = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, 2, TRUE);
   if (term == ',')
   {
      bp = countarg (bp, &count, 15, "transfer count", &term);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing count operand");
   }

   val = op->opvalue | (count << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 2 + spc;
}

/***********************************************************************
* p205op - Process type 5 operation code.
***********************************************************************/

static void
p205op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int r = 0;
   int c = 0;
   int val;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   bp = registerarg (bp, &r, &term);
   if (term == ',')
   {
      bp = countarg (bp, &c, 15, "shift count", &term);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing shift count operand");
   }

   val = op->opvalue | (c << 4) | r;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   pc += 2;
}

/***********************************************************************
* p206op - Process type 6 operation code. 
***********************************************************************/

static void
p206op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, 2, TRUE);

   val = op->opvalue | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 2 + spc;
}

/***********************************************************************
* p207op - Process type 7 operation code. 
***********************************************************************/

static void
p207op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   sprintf (pcbuf, PCFORMAT, pc);
   sprintf (opbuf, OP1FORMAT, op->opvalue, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, op->opvalue);
   pc += 2;
}

/***********************************************************************
* p208op - Process type 8 operation code. 
***********************************************************************/

static void
p208op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int r = 0;
   int i = 0;
   int relo;
   int val;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   val = op->opvalue;

   if (op->opmod == 1) /* Instructions without register operand */
   {
      sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
      goto NOREG;
   }

   bp = registerarg (bp, &r, &term);
   val = op->opvalue | r;
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);

   if (op->opmod == 2) /* Instructions with only register operand */
   {
      sprintf (opbuf, OP1FORMAT, val, ' ');
      pc += 2;
      return;
   }

   if (term == ',')
   {
NOREG:
      bp = exprscan (bp, &i, &term, &relo, 2, FALSE, 2);
      if (i < -65634 || i > 65535)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid immediate value: %d", i);
	 i = 0;
      }
      i &= 0xFFFF;
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing immediate operand");
   }

   punchrecord (outfd);
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (opbuf, OP1FORMAT, i, relo ? '\'' : ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, relo ? RELDATA_TAG : ABSDATA_TAG, i);
   pc += 4;
}

/***********************************************************************
* p209op - Process type 9 operation code. 
***********************************************************************/

static void
p209op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int rd = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, 2, TRUE);
   if (term == ',')
   {
      bp = registerarg (bp, &rd, &term);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing register operand");
   }

   val = op->opvalue | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0)
   {
      printed = TRUE;
      printline (lstfd);
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 2 + spc;
}

/***********************************************************************
* p210op - Process type 10 operation code. 
***********************************************************************/

static void
p210op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int rs = 0;
   int rd = 0;
   int junk;
   int val;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   bp = registerarg (bp, &rs, &term);
   if (term == ',')
   {
      if (*bp == MEMSYM || *bp == LITERALSYM)
      {
	    errcount++;
	    sprintf (errline[errnum++], "Not a map file: %s", bp);
	    rd = 0;
      }
      else
      {

	 bp = exprscan (bp, &rd, &term, &junk, 2, FALSE, 0);
	 if (rd < 0 || rd > 1)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Invalid map file value: %d", rd);
	    rd = 0;
	 }
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing map file operand");
   }

   val = op->opvalue | (rd << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   pc += 2 ;
}

/***********************************************************************
* p211op - Process type 11 operation code.
***********************************************************************/

static void
p211op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int td = 0;
   int ts = 0;
   int rd = 0;
   int rs = 0;
   int bc = 0;
   int daddr = -1;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 4,
		       FALSE);
      if (term == ',')
      {
	 bp = countarg (bp, &bc, 15, "byte count", &term);
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (bc << 12) | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p212op - Process type 12 operation code.
***********************************************************************/

static void
p212op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int td = 0;
   int ts = 0;
   int rd = 0;
   int rs = 0;
   int bc = 0;
   int ck = -1;
   int daddr = -1;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 4,
		       FALSE);
      if (term == ',')
      {
	 bp = countarg (bp, &bc, 15, "byte count", &term);
	 if (term == ',')
	 {
	    bp = registerarg (bp, &ck, &term);
	 }
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }
   if (ck < 0)
   {
      if (ckpt < 0)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Missing checkpoint operand");
	 ck = 0;
      }
      else ck = ckpt;
   }

   val = op->opvalue | ck;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (bc << 12) | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p213op - Process type 13 operation code.
***********************************************************************/

static void
p213op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int bc = 0;
   int sc = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = countarg (bp, &bc, 15, "byte count", &term);
      if (term == ',')
      {
	 bp = countarg (bp, &sc, 15, "shift count", &term);
      }
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (bc << 12) | (sc << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p214op - Process type 14 operation code.
***********************************************************************/

static void
p214op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int pos = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = countarg (bp, &pos, 0x3FF, "position", &term);
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (pos << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p215op - Process type 15 operation code.
***********************************************************************/

static void
p215op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int pos = 0;
   int wid = 0;
   int saddr = -1;
   int srelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      if (*bp == '(')
      {
	 bp++;
	 bp = countarg (bp, &pos, 15, "position", &term);
	 if (term == ',')
	 {
	    char *cp;

	    cp = bp;
	    while (*bp && *bp != ')') bp++;
	    if (*bp != ')')
	    {
	       errcount++;
	       sprintf (errline[errnum++], "Invalid width operand");
	    }
	    else
	    {
	       *bp = '\0';
	       bp = countarg (cp, &wid, 15, "width", &term);
	    }
	 }
	 else
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Missign width operand");
	 }
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid position operand");
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing position operand");
   }

   val = op->opvalue | wid;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (pos << 12) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p216op - Process type 16 operation code.
***********************************************************************/

static void
p216op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int ts = 0;
   int rs = 0;
   int td = 0;
   int rd = 0;
   int pos = 0;
   int wid = 0;
   int saddr = -1;
   int daddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 4,
		       FALSE);
      if (term == ',')
      {
	 if (*bp == '(')
	 {
	    bp++;
	    bp = countarg (bp, &pos, 15, "position", &term);
	    if (term == ',')
	    {
	       char *cp;

	       cp = bp;
	       while (*bp && *bp != ')') bp++;
	       if (*bp != ')')
	       {
		  errcount++;
		  sprintf (errline[errnum++], "Invalid width operand");
	       }
	       else
	       {
		  *bp = '\0';
		  bp = countarg (cp, &wid, 15, "width", &term);
	       }
	    }
	    else
	    {
	       errcount++;
	       sprintf (errline[errnum++], "Missing width operand");
	    }
	 }
	 else
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Invalid position operand");
	 }
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Missing position operand");
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }

   val = op->opvalue | wid;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (pos << 12) | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p217op - Process type 17 operation code.
***********************************************************************/

static void
p217op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int addr = 0;
   int disp = 0;
   int r = 0;
   int c = 0;
   int relocatable;
   int val;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   bp = exprscan (bp, &addr, &term, &relocatable, 2, FALSE, 0);
   if (*bp == MEMSYM || *bp == LITERALSYM)
   {
      errcount++;
      sprintf (errline[errnum++], "Not a valid jump operand: %s", bp);
      disp = 0;
   }
   else
   {
      disp = addr/2 - (pc+4)/2;
      if (disp < -128 || disp > 128)
      {
	    errcount++;
	    sprintf (errline[errnum++], "Jump displacement too large: %d",
		     disp);
	    disp = 0;
      }
   }
   if (term == ',')
   {
      bp = countarg (bp, &c, 15, "count", &term);
      if (term == ',')
      {
	 bp = registerarg (bp, &r, &term);
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Missing register operand");
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing count operand");
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (c << 12) | (r << 8) | (disp & 0xFF);
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   pc += 4;
}

/***********************************************************************
* p218op - Process type 18 operation code.
***********************************************************************/

static void
p218op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int r = 0;
   int val;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   bp = registerarg (bp, &r, &term);

   val = op->opvalue | r;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   pc += 2;
}

/***********************************************************************
* p219op - Process type 19 operation code.
***********************************************************************/

static void
p219op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int td = 0;
   int ts = 0;
   int rd = 0;
   int rs = 0;
   int daddr = -1;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 4,
		       FALSE);
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p220op - Process type 20 operation code.
***********************************************************************/

static void
p220op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   char *token;
   int tokentype;
   int td = 0;
   int ts = 0;
   int rd = 0;
   int rs = 0;
   int cond = -1;
   int daddr = -1;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = tokscan (bp, &token, &tokentype, &val, &term);
   if (tokentype == SYM)
   {
      if (!strcmp (token, "EQ")) cond = 0;
      else if (!strcmp (token, "NE")) cond = 1;
      else if (!strcmp (token, "HE")) cond = 2;
      else if (!strcmp (token, "L")) cond = 3;
      else if (!strcmp (token, "GE")) cond = 4;
      else if (!strcmp (token, "LT")) cond = 5;
      else if (!strcmp (token, "LE")) cond = 6;
      else if (!strcmp (token, "H")) cond = 7;
      else if (!strcmp (token, "LTE")) cond = 8;
      else if (!strcmp (token, "GT")) cond = 9;
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid condition code: %s", token);
	 cond = 0;
      }
   }
   else if (tokentype == DECNUM)
   {
      cond = val;
   }
   if (cond < 0 || cond > 9)
   {
      errcount++;
      sprintf (errline[errnum++], "Invalid condition code: %s", token);
      cond = 0;
   }
   if (term == ',')
   {
      bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		       TRUE);
      if (term == ',')
      {
	 bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc+4,
			  FALSE);
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Missing destination operand");
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing source operand");
   }

   val = op->opvalue;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (cond << 12) | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p221op - Process type 21 operation code.
***********************************************************************/

static void
p221op (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   int td = 0;
   int rd = 0;
   int cd = 0;
   int daddr = -1;
   int ts = 0;
   int rs = 0;
   int cs = 0;
   int saddr = -1;
   int srelo;
   int drelo;
   int val;
   int spc;
   char term;

   sprintf (pcbuf, PCFORMAT, pc);
   spc = 0;
   bp = generalarg (op, bp, &ts, &rs, &saddr, &srelo, &spc, &term, spc + 4,
		    TRUE);
   if (term == ',')
   {
      bp = generalarg (op, bp, &td, &rd, &daddr, &drelo, &spc, &term, spc + 4,
		       FALSE);
      if (term == ',')
      {
         bp = countarg (bp, &cs, 15, "source count", &term);
	 if (term == ',')
	 {
	    bp = countarg (bp, &cd, 15, "destination count", &term);
	 }
      }
   }
   else
   {
      errcount++;
      sprintf (errline[errnum++], "Missing destination operand");
   }

   val = op->opvalue | cd;
   printed = TRUE;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printline (lstfd);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   val = (cs << 12) | (td << 10) | (rd << 6) | (ts << 4) | rs;
   sprintf (opbuf, OP1FORMAT, val, ' ');
   printdata (lstfd, pc + 2);
   sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, val);
   punchrecord (outfd);

   if (saddr >= 0 && daddr >= 0)
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 6);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (saddr < 0 && daddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, daddr, drelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, drelo ? RELDATA_TAG : ABSDATA_TAG, daddr);
      punchrecord (outfd);
   }
   else if (daddr < 0 && saddr >= 0) 
   {
      sprintf (opbuf, OP1FORMAT, saddr, srelo ? '\'' : ' ');
      printdata (lstfd, pc + 4);
      sprintf (objbuf, OBJFORMAT, srelo ? RELDATA_TAG : ABSDATA_TAG, saddr);
      punchrecord (outfd);
   }
   pc += 4 + spc;
}

/***********************************************************************
* p2pop - Process Pseudo operation code.
***********************************************************************/

static int
p2pop (OpCode *op, char *bp, FILE *outfd, FILE *lstfd)
{
   SymNode *s;
   char *cp;
   char *token;
   int i, j, k, l;
   int opc;
   int val;
   int oval;
   int relocatable;
   int tokentype;
   char term;
   char temp[10];


   sprintf (pcbuf, PCFORMAT, pc);

   /*
   ** If word based op and "hanging" byte, punch it.
   */

   switch (op->opvalue)
   {
   case AORG_T:
   case ASMIF_T:
   case ASMELS_T:
   case ASMEND_T:
   case BES_T:
   case BSS_T:
   case DATA_T:
   case DORG_T:
   case DOUBLE_T:
   case END_T:
   case EVEN_T:
   case FLOAT_T:
   case LONG_T:
   case NOP_T:
   case QUAD_T:
   case RT_T:
      if (bytecnt)
      {
	 byteacc = (byteacc << 8) | 0;
	 sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, byteacc);
	 punchrecord (outfd);
	 bytecnt = 0;
      }
      break;
   default: ;
   }

   switch (op->opvalue)
   {

   case AORG_T:
      if (!absolute && !indorg) rorgpc = pc;
      cp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid AORG value: %d", val);
	 val = 0;
      }
      indorg = FALSE;
      absolute = TRUE;
      pc = val;
      sprintf (pcbuf, PCFORMAT, pc);
      sprintf (objbuf, OBJFORMAT, ABSORG_TAG, val);
      orgout = TRUE;
      break;

   case ASMIF_T:
      strcpy (pcbuf, PCBLANKS);
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      asmskip = val ? FALSE : TRUE;
      break;

   case ASMELS_T:
      strcpy (pcbuf, PCBLANKS);
      asmskip = asmskip ? FALSE : TRUE;
      break;

   case ASMEND_T:
      strcpy (pcbuf, PCBLANKS);
      asmskip = FALSE;
      break;

   case BES_T:
   case BSS_T:
      bp = exprscan (bp, &val, &term, &relocatable, 1, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid storage value: %d", val);
	 val = 0;
      }
      pc += val;
      sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
      break;

   case BYTE_T:
      i = j = k = l = 0;
      opc = pc;
      do {
	 bp = exprscan (bp, &val, &term, &relocatable, 1, FALSE, 0);
	 val &= 0xFF;

	 byteacc = byteacc << 8 | val;
	 bytecnt++;
	 sprintf (temp, CHAR1FORMAT, val);
	 strncpy (&opbuf[j], temp, 2);
	 j += 2;
	 pc++;
	 i++;
	 if (bytecnt == 2)
	 {
	    sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, byteacc);
	    punchrecord (outfd);
	    bytecnt = 0;
	    byteacc = 0;
	 }
	 printed = TRUE;
	 if (l++ == 0)
	 {
	    printline (lstfd);
	    opc = pc;
	 }
	 else
	 {
	    if (!bunlist)
	       printdata (lstfd, opc);
	    opc = pc;
	 }
	 strcpy (opbuf, OPBLANKS);
	 j = k = 0;
      
      } while (term == ',');

      break;

   case CKPT_T:
      strcpy (pcbuf, PCBLANKS);
      ckpt = -1;
      bp = registerarg (bp, &val, &term);
      if (errnum == 0) ckpt = val;
      break;

   case COPY_T:
      strcpy (pcbuf, PCBLANKS);
      printed = TRUE;
      printline (lstfd);
      tmpfd = opencopy (bp, tmpfd);
      break;
      
   case DATA_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      opc = pc;
      i = 0;
      do {
	 bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
	 if (val < -65534 || val > 65535)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Invalid DATA value: %d", val);
	 }
	 val &= 0xFFFF;
	 sprintf (opbuf, DATA1FORMAT, val, relocatable ? '\'' : ' ');
	 sprintf (objbuf, OBJFORMAT,
		  relocatable ? RELDATA_TAG : ABSDATA_TAG, val);
	 punchrecord (outfd);
	 pc += 2;
	 printed = TRUE;
	 if (i++ == 0)
	 {
	    printline (lstfd);
	 }
	 else
	 {
	    if (!dunlist)
	       printdata (lstfd, opc);
	 }
	 opc = pc;
	 strcpy (opbuf, OPBLANKS);
      } while (term == ',');
      break;

   case DEF_T:
      strcpy (pcbuf, PCBLANKS);
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "DEF requires symbol: %s", token);
	 }
	 else
	 {
	    if (strlen (token) > MAXSYMLEN)
	       token[MAXSYMLEN] = '\0';

	    if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	    {
	       errcount++;
	       sprintf (errline[errnum++], "DEF undefined: %s", token);
	    }
	    else
	    {
	       s->global = TRUE;
	    }
	 }
      } while (term == ',');
      break;

   case DOUBLE_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      opc = pc;
      i = 0;
      do {
	 t_uint64 d;

	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 d = cvtdouble (cp);
	 val =  (int)((d >> 48) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, i == 0 ? TRUE : FALSE);
	 val =  (int)((d >> 32) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 val =  (int)((d >> 16) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 val =  (int)(d & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 i++;
      } while (term == ',');
      break;

   case DORG_T:
      if (!absolute && !indorg) rorgpc = pc;
      cp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      if (val < 0 || val > 65535)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid DORG value: %d", val);
	 val = 0;
      }
      absolute = TRUE;
      indorg = TRUE;
      pc = val;
      sprintf (pcbuf, PCFORMAT, pc);
      break;

   case DXOP_T:
   case LOAD_T:
   case REF_T:
   case LREF_T:
   case SETRM_T:
   case SREF_T:
      strcpy (pcbuf, PCBLANKS);
      break;

   case END_T:
      strcpy (pcbuf, PCBLANKS);
      bp = tokscan (bp, &token, &tokentype, &val, &term);
      if (tokentype == SYM)
      {
	 if (strlen (token) > MAXSYMLEN)
	 {
	    token[MAXSYMLEN] = '\0';
	 }
	 if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Undefined symbol: %s", token);
	 }
	 else
	 {
	    sprintf (opbuf, ADDRFORMAT, s->value, s->relocatable ? '\'' : ' ');
	    sprintf (objbuf, OBJFORMAT,
		  (!absolute && s->relocatable) ? RELENTRY_TAG : ABSENTRY_TAG,
		  s->value);
	 }
      }
      return (1);
      break;

   case EQU_T:
      strcpy (pcbuf, PCBLANKS);
      if (cursym[0] == '\0')
      {
         errcount++;
	 strcpy (errline[errnum++], "EQU requires a label");
      }
      else
      {
	 s = symlookup (cursym, FALSE, TRUE);
	 sprintf (opbuf, ADDRFORMAT, s->value & 0xFFFF,
		  s->relocatable ? '\'' : ' ');
      }
      break;

   case EVEN_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      break;

   case FLOAT_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      opc = pc;
      i = 0;
      do {
	 long d;

	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 d = cvtfloat (cp);
	 val =  (d >> 16) & 0xFFFF;
	 opc = outputdataword (outfd, lstfd, opc, val, i == 0 ? TRUE : FALSE);
	 val =  d & 0xFFFF;
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 i++;
      } while (term == ',');
      break;

   case IDT_T:
      strcpy (pcbuf, PCBLANKS);
      while (*bp && isspace(*bp)) bp++;
      if (*bp == '\'')
      {
	 cp = ++bp;
	 while (*bp != '\'' && *bp != '\n') bp++;
      }
      *bp = '\0';
      if (strlen(idtbuf) > IDTSIZE)
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid IDT value: %s", cp);
	 *(cp+IDTSIZE) = '\0';
      }
      strcpy (idtbuf, cp);
      break;

   case LDEF_T:
      strcpy (pcbuf, PCBLANKS);
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype != SYM)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "LDEF requires symbol: %s", token);
	 }
	 else
	 {
	    if (strlen (token) > MAXSYMLEN)
	       token[MAXSYMLEN] = '\0';

	    if ((s = symlookup (token, FALSE, FALSE)) == NULL)
	    {
	       errcount++;
	       sprintf (errline[errnum++], "LDEF undefined: %s", token);
	    }
	    else
	    {
	       s->global = TRUE;
	       s->longsym = TRUE;
	    }
	 }
      } while (term == ',');
      break;

   case LIST_T:
      strcpy (pcbuf, PCBLANKS);
      listing = TRUE;
      break;

   case LONG_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      opc = pc;
      i = 0;
      do {
	 long d;

	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 if (*cp == '>')
	 {
	    cp++;
	    d = strtol (cp, NULL, 16);
	 }
	 else
	 {
	    d = strtol (cp, NULL, 10);
	 }
	 val =  (d >> 16) & 0xFFFF;
	 opc = outputdataword (outfd, lstfd, opc, val, i == 0 ? TRUE : FALSE);
	 val =  d & 0xFFFF;
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 i++;
      } while (term == ',');
      break;

   case NOP_T:
   case RT_T:
      sprintf (opbuf, OP1FORMAT, op->opmod, ' ');
      sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, op->opmod);
      pc += 2;
      break;

   case OPTION_T:
      strcpy (pcbuf, PCBLANKS);
      do {
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (tokentype == SYM)
	 {
	    j = strlen (token);
	    if (!(strncmp (token, "BUNLIST", j))) bunlist = TRUE;
	    else if (!(strncmp (token, "DUNLIST", j))) dunlist = TRUE;
	    else if (!(strncmp (token, "MUNLIST", j))) munlist = TRUE;
	    else if (!(strncmp (token, "TUNLIST", j))) tunlist = TRUE;
	    else if (!(strncmp (token, "NOLIST", j))) listmode = FALSE;
	    else if (!(strncmp (token, "SYMT", j))) allsymbols = TRUE;
	    else if (!(strncmp (token, "XREF", j))) ;
	    else goto BADOPTIONTOKEN;
	 }
	 else if (tokentype == DECNUM)
	 {
	    if (val == 10) model = 10;
	    else if (val == 12) model = 12;
	    else
	    {
	       sprintf (token, "%d", val);
	       goto BADOPTIONTOKEN;
	    }
	 }
	 else
	 {
	 BADOPTIONTOKEN:
	    errcount++;
	    sprintf (errline[errnum++], "Invalid OPTION token: %s", token);
	 }
      } while (term == ',');
      break;

   case PAGE_T:
      strcpy (pcbuf, PCBLANKS);
      printed = TRUE;
      printline (lstfd);
      linecnt = MAXLINE;
      break;

   case QUAD_T:
      opc = pc;
      if (pc & 0x0001) pc++;
      sprintf (pcbuf, PCFORMAT, pc);
      if (pc != opc)
      {
	 sprintf (objbuf, OBJFORMAT, absolute ? ABSORG_TAG : RELORG_TAG, pc);
	 punchrecord (outfd);
      }
      opc = pc;
      i = 0;
      do {
	 t_uint64 d;

	 while (*bp && isspace(*bp)) bp++;
	 cp = bp;
	 while (*bp != ',' && !isspace(*bp)) bp++;
	 term = *bp;
	 *bp++ = '\0';
	 if (*cp == '>')
	 {
	    cp++;
#ifdef WIN32
	    sscanf (cp, "%I64x", &d);
#else
	    sscanf (cp, "%llx", &d);
#endif
	 }
	 else
	 {
#ifdef WIN32
	    sscanf (cp, "%I64d", &d);
#else
	    sscanf (cp, "%lld", &d);
#endif
	 }
	 val =  (int)((d >> 48) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, i == 0 ? TRUE : FALSE);
	 val =  (int)((d >> 32) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 val =  (int)((d >> 16) & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 val =  (int)(d & 0xFFFF);
	 opc = outputdataword (outfd, lstfd, opc, val, FALSE);
	 i++;
      } while (term == ',');
      break;

   case RORG_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      indorg = FALSE;
      absolute = FALSE;
      if (val == 0)
      {
	 pc = rorgpc;
      }
      else
      {
	 if (val < 0 || val > 65535)
	 {
	    errcount++;
	    sprintf (errline[errnum++], "Invalid RORG value: %d", val);
	    val = 0;
	 }
         pc = val;
      }
      sprintf (pcbuf, PCFORMAT, pc);
      sprintf (objbuf, OBJFORMAT, RELORG_TAG, pc);
      orgout = TRUE;
      break;

   case TEXT_T:
      bp = tokscan (bp, &token, &tokentype, &val, &term);
      oval = 0;
      if (term == '-')
      {
	 oval = 1;
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
      }
      if (tokentype == STRING)
      {
	 strcpy (opbuf, OPBLANKS);

	 bp = token;
	 i = l = 0;
	 val = 0;
	 opc = pc;
	 while (*bp)
	 {
	    if (oval && *(bp+1) == '\0') *bp = (1 - *bp) & 0xFF;
	    byteacc = byteacc << 8 | (*bp & 0xFF);
	    sprintf (opbuf, CHAR1FORMAT, (*bp++ & 0xFF));
	    pc++;
	    bytecnt++;
	    i++;
	    if (bytecnt == 2)
	    {
	       sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, byteacc);
	       punchrecord (outfd);
	       byteacc = 0;
	       bytecnt = 0;
	    }
	    i = 0;
	    printed = TRUE;
	    if (l++ == 0)
	    {
	       printline (lstfd);
	       opc = pc;
	    }
	    else
	    {
	       if (!tunlist)
	       printdata (lstfd, opc);
	       opc = pc;
	    }
	    strcpy (opbuf, OPBLANKS);
	 }
      }
      else
      {
	 errcount++;
	 sprintf (errline[errnum++], "Invalid TEXT string: %s", token);
      }
      break;

   case TITL_T:
      strcpy (pcbuf, PCBLANKS);
      while (isspace(*bp)) bp++;
      if (*bp == '\'')
      {
	 cp = ++bp;
	 while (*bp != '\'' && *bp != '\n') bp++;
      }
      *bp = '\0';
      strcpy (ttlbuf, cp);
      break;

   case UNL_T:
      strcpy (pcbuf, PCBLANKS);
      listing = FALSE;
      break;

   case XVEC_T:
      bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      sprintf (opbuf, DATA1FORMAT, val, relocatable ? '\'' : ' ');
      printed = TRUE;
      printline (lstfd);
      sprintf (objbuf, OBJFORMAT,
	       relocatable ? RELDATA_TAG : ABSDATA_TAG, val);
      punchrecord (outfd);

      if (term == ',')
      {
	 bp = exprscan (bp, &val, &term, &relocatable, 2, FALSE, 0);
      }
      else
      {
	 relocatable = TRUE;
         val = pc + 4;
      }
      sprintf (opbuf, DATA1FORMAT, val, relocatable ? '\'' : ' ');
      if (!dunlist)
	 printdata (lstfd, pc + 2);
      sprintf (objbuf, OBJFORMAT,
	       relocatable ? RELDATA_TAG : ABSDATA_TAG, val);
      punchrecord (outfd);
      pc += 4;
      break;

   default: ;
   }
   return (0);
}

/***********************************************************************
* asmpass2 - Pass 2 
***********************************************************************/

int
asmpass2 (FILE *infd, FILE *outfd, int listingmode, FILE *lstfd)
{
   char *bp;
   char *token;
   int status = 0;
   int done = 0;
   int val;
   int tokentype;
   time_t curtime;
   char term;

#ifdef DEBUG
   printf ("asmpass2: Entered\n");
#endif

   /*
   ** Get current date/time.
   */

   listmode = listingmode;
   curtime = time(NULL);
   timeblk = localtime(&curtime);
   strcpy (datebuf, ctime(&curtime));
   *strchr (datebuf, '\n') = '\0';

   /*
   ** Rewind input file.
   */

   tmpfd = infd;
   if (fseek (tmpfd, 0, SEEK_SET) < 0)
   {
      perror ("asm990: Can't rewind temp file");
      return (-1);
   }

   /*
   ** Process the source.
   */

   memset (objrec, ' ', sizeof(objrec));

   pc = 0;
   rorgpc = 0;
   absolute = FALSE;
   asmskip = FALSE;
   indorg = FALSE;

   orgout = TRUE;
   sprintf (objbuf, IDTFORMAT, IDT_TAG, pgmlength & 0xFFFF, idtbuf);
   punchrecord (outfd);
   orgout = FALSE;

   byteacc = 0;
   bytecnt = 0;
   linenum = 0;

DOREAD:
   while (!done && fgets (inbuf, MAXLINE, tmpfd))
   {
#ifdef DEBUG
      printf ("in = %s", inbuf);
#endif
      linenum++;
      xrefgened = FALSE;
      printed = FALSE;
      errline[0][0] = '\0';
      errnum = 0;
      objbuf[0] = '\0';
      strcpy (lstbuf, inbuf);
      bp = inbuf;

      strcpy (pcbuf, PCBLANKS);
      strcpy (opbuf, OPBLANKS);

      if (*bp != COMMENTSYM)
      {
	 OpCode *op;

	 /*
	 ** If label present, scan it off.
	 */

         if (isalpha(*bp) || *bp == '$' || *bp == '_')
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    strcpy (cursym, token);
	    if (strlen(token) > MAXSYMLEN)
	    {
	       cursym[MAXSYMLEN] = '\0';
	       token[MAXSYMLEN] = '\0';
	    }
	    if (term == '\n') goto PRINTLINE;
	 }
	 else 
	 {
	    cursym[0] = '\0';
	    while (isspace (*bp)) bp++;
	 }

	 /*
	 ** Scan off opcode.
	 */

	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (!token[0]) goto PRINTLINE;

	 /*
	 ** Process according to type.
	 */

	 op = oplookup (token);
	 if (asmskip)
	 {
	    if (op != NULL && op->optype == TYPE_P)
	       switch (op->opvalue)
	       {
	       case ASMELS_T:
	       case ASMEND_T:
	          p2pop (op, bp, outfd, lstfd);
		  break;

	       default: ;
	       }
	 }
	 else if (op != NULL)
	 {
	    while (*bp && isspace(*bp)) bp++;

	    if (op->optype != TYPE_P)
	    {
	       if (bytecnt)
	       {
		  byteacc = (byteacc << 8) | 0;
		  sprintf (objbuf, OBJFORMAT, ABSDATA_TAG, byteacc);
		  punchrecord (outfd);
		  bytecnt = 0;
	       }
	       if (pc & 0x0001) pc++;
	    }

	    switch (op->optype)
	    {

	       case TYPE_1:
		  p201op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_2:
		  p202op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_3:
		  p203op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_4:
		  p204op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_5:
		  p205op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_6:
		  p206op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_7:
		  p207op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_8:
		  p208op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_9:
		  p209op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_10:
		  p210op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_11:
		  p211op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_12:
		  p212op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_13:
		  p213op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_14:
		  p214op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_15:
		  p215op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_16:
		  p216op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_17:
		  p217op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_18:
		  p218op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_19:
		  p219op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_20:
		  p220op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_21:
		  p221op (op, bp, outfd, lstfd);
	          break;

	       case TYPE_P:
	          done = p2pop (op, bp, outfd, lstfd);
	    }
	 }
      }

      /*
      ** Write out a print line.
      */
PRINTLINE:
      if (!printed)
      {
         printline (lstfd);
	 printed = FALSE;
      }

      /*
      ** Write an object buffer.
      */

      if (objbuf[0])
      {
         punchrecord (outfd);
      }

      /*
      ** Process errors
      */

      if (errnum)
      {
	 int i;

	 for (i = 0; i < errnum; i++)
	 {
	    if (listmode)
	    {
	       fprintf (lstfd, "ERROR: %s\n", errline[i]);
	       linecnt++;
	    }
	    else
	    {
	       fprintf (stderr, "asm990: %d: %s\n", linenum, errline[i]);
	    }
	    errline[i][0] = '\0';
	 }
	 status = -1;
      }

      /*
      ** Check for pass 1 error for this line
      */

      if (p1errcnt && p1error[curp1err].errorline == linenum)
      {
	 if (listmode)
	 {
	    fprintf (lstfd, "ERROR: %s\n", p1error[curp1err].errortext);
	    linecnt++;
	 }
	 else
	 {
	    fprintf (stderr, "asm990: %d: %s\n",
		     linenum, p1error[curp1err].errortext);
	 }
	 p1errcnt--;
	 curp1err++;
	 status = -1;
      }
   }
   if ((tmpfd = closecopy (tmpfd)) != NULL) goto DOREAD;

   if (!done)
   {
      errcount++;
      if (listmode)
      {
         fprintf (lstfd, "ERROR: No END record\n");
      }
      else
      {
	 fprintf (stderr, "asm990: %d: No END record\n", linenum);
      }
      status = -1;
   }
   else
   {

      /*
      ** Process literals
      */

      if (processliterals (outfd, lstfd, listmode) < 0) status = -1;

      /*
      ** Punch REF and DEF entries
      */

      punchsymbols (outfd);
      puncheof (outfd);

      /*
      ** Print symbol table
      */

      if (listmode)
      {
         printsymbols (lstfd);
      }
   }

   return (status);
}
