/* @(#)psio.c 9.8 88/01/19 SMI	 */

/*
 * Copyright (c) 1985 by Sun Microsystems, Inc.
 */

/*-
	input side of the stdio for NeWS

	psio.c, Mon Jan  5 16:21:31 1987

		James Gosling,
		Sun Microsystems
 */

/*
 * The psio package is essentially equivalent to stdio except that it has some
 * extensions for NeWS.  The extensions are that it supports reading from
 * strings, it supports the non-blocking escape required by the lwp mechanism,
 * it supports "reading ahead" past a protected part of the input buffer, it
 * supports a hook to the output file for automatic flushing, and it supports
 * more file descriptors than stdio.  [many stdio implementations only allow
 * one iob per fd; with read/write descriptors, two are needed]
 */

#include "psio.h"
#include <fcntl.h>
#ifndef SYSVREF
#include <sys/time.h>
#else
#include <stropts.h>
#include <poll.h>
#endif
#ifdef REF
#include <sys/types.h>
#include <ref/config.h>
#endif
#include <errno.h>
    extern int  errno;

#define INITBUF(p) if ((p)->bufsiz == 0) initbuf(p)

PSFILE     *psio_stdin;
PSFILE     *psio_stdout;
PSFILE     *psio_stderr;
int         psio_bufsiz;

/*
 * psio_noneverblock & psio_block_delay are not public.  They are included
 * as tuning variables for the PSNVRBLK feature.  They are include because
 * this feature was included late in the beta cycle of NeWS 1.1 in order
 * to fix a major problem with the server hanging.  These variable are
 * serving as a bailout mechanism if PSNVRBLK proves to be a bad idea
 * while in the field.  THESE MAY DISAPPEAR IN THE NEXT RELEASE.
 */
int psio_noneverblock;	/* A 1 value will disable PSNVRBLK */
struct timeval psio_block_delay = {0, 500000};
			/* Retry delay interval (1/2 sec) */
int psio_retries = 7;	/* Retry times */

static
initbuf(p)
    register PSFILE *p;
{
    if (p->bufsiz == 0) {
	if (psio_bufsiz <= 0 || psio_bufsiz >= 16 * PSINBUFSIZ)
	    psio_bufsiz = PSINBUFSIZ;
	p->bufsiz = psio_bufsiz;
	p->base = (unsigned char *) malloc(p->bufsiz);
	p->flag |= PSMYBUF;
    }
}

/* before calling psio_filbuf, either set or clear the PSBLOCKOK bit */

psio_filbuf(p)
    register PSFILE *p;
{
    register    avail;

    psio_clearblocked(p);
    if (p->flag & (PSSTRG | PSEOF | PSERR | PSWRITE)) {
	p->flag |= PSEOF;
	return -1;
    }
    INITBUF(p);
    while (1) {
	avail = p->bufsiz - p->protected;
	if (avail <= 0)
	    return -1;
	p->ptr = p->base + p->protected;
	if (p->outputside && psio_needsflush(p->outputside))
	    psio_flush(p->outputside);
	p->cnt = read(p->file, p->ptr, avail);
	if (p->cnt > 0)
	    return psio_getc(p);
	if (p->cnt == 0) {
	    p->flag |= PSEOF;
	    return -1;
	}
	if (errno == EINTR)
	    continue;
	if (errno == EWOULDBLOCK) {
	    if (psio_blockok(p)) {
		sleep(1);		/* Resort to polling */
		continue;
	    } else {
		psio_setblocked(p);
		return -1;
	    }
	}
	p->flag |= PSERR;
	return -1;
    }
}

PSFILE     *
psio_open(name, mode)
    char *name;
    register char *mode;
{
    register    fd;
    switch (mode[0]) {
    case 'a':
	fd = open(name, 1);
	if (fd >= 0) {
	    lseek(fd, 0, 2);
	    break;
	}
    case 'w':
	fd = creat(name, 0644);
	break;
    case 'r':
	fd = open(name, 0);
	break;
    default:
	fd = -1;
    }
    return psio_fdopen(fd, mode);
}

PSFILE     *
psio_fdopen(fd, mode)
    char *mode;
{
    register PSFILE *f;
    if (fd < 0)
	return 0;
    f = (PSFILE *) malloc(sizeof(PSFILE));
    bzero(f, sizeof *f);
    f->flag = mode && mode[0] == 'w' ? PSWRITE : PSREAD;
    if (fd <= 2)
	f->flag |= PSLINEBUF;
    f->file = fd;
    return f;
}

PSFILE     *
psio_sopen(string, length, mode)
    unsigned char *string;
    char *mode;
{
    register PSFILE *f = (PSFILE *) malloc(sizeof(PSFILE));
    bzero(f, sizeof *f);
    f->cnt = f->bufsiz = length;
    f->base = f->ptr = string;
    f->flag = PSSTRG | (mode[0] == 'w' ? PSWRITE : PSREAD);
    f->file = -1;
    return f;
}

psio_close(f)
    register PSFILE *f;
{
    if (f == 0)
	return -1;
    if ((f->flag & PSSTRG) == 0) {
	if ((f->flag & PSWRITE) && f->bufsiz > 0)
	    psio_flush(f);
	if (f->base && (f->flag & PSMYBUF))
	    free(f->base);
	close(f->file);
    }
    free(f);
}

psio_nodropclose(f)
    register PSFILE *f;
{
    if (f == 0)
	return -1;
    if ((f->flag & PSSTRG) == 0) {
	if ((f->flag & PSWRITE) && f->bufsiz > 0)
	    psio_flush(f);
	if (f->base && (f->flag & PSMYBUF))
	    free(f->base);
#ifdef SYSVREF
	if (t_close(f->file) < 0)
	    close(f->file);
#else
	close(f->file);
#endif
    }
    f->flag |= PSEOF;
    f->base = 0;
    f->ptr = 0;
    f->cnt = 0;
    f->file = -1;
}

psio_read(dest, sz0, n, p)
    register char *dest;
    register PSFILE *p;
    int         sz0,
                n;
{
    register    ret = 0,
                sz,
                c;
    while (--n >= 0) {
	for (sz = sz0; --sz >= 0;) {
	    if ((c = psio_getc(p)) < 0)
		return ret;
	    *dest++ = c;
	}
	ret++;
    }
    return ret;
}

psio_write(src, sz0, n, p)
    register char *src;
    register PSFILE *p;
    int         sz0,
                n;
{
    register    ret = 0,
                sz,
                c;
    while (--n >= 0) {
	for (sz = sz0; --sz >= 0;)
	    if (psio_putc(*src++, p) < 0)
		return ret;
	ret++;
    }
    return ret;
}

psio_ungetc(c, p)
    register PSFILE *p;
{
    if (c == EOF || (p->flag & PSREAD) == 0)
	return EOF;
    if (p->bufsiz == 0)
	initbuf(p);
    if (p->ptr <= p->base)
	if (p->ptr == p->base && p->cnt == 0)
	    ++p->ptr;
	else
	    return EOF;
    *--p->ptr = c;
    ++p->cnt;
    return c;
}

psio_flush(p)
    register PSFILE *p;
{
    int never_block = (psio_neverblock(p) && !psio_noneverblock);

    if (p->flag & (PSSTRG | PSEOF | PSERR | PSREAD)) {
	p->flag |= PSERR | PSEOF;
	return -1;
    }
    if (p->bufsiz == 0)
	initbuf(p);
    else if (p->ptr > p->base) {	/* Buffer contains output */
	register unsigned char *pos = p->base;
	register    left = p->ptr - p->base;
	register    retry = 0;
	int	    errno_write;
	int	    fcntl_flags;

	if (never_block) {
	    /* Remember original flags */
	    if ((fcntl_flags = fcntl(p->file, F_GETFL, 0)) < 0)
	        perror("fcntl");
	}
	while (left > 0) {
	    register    written;

	    if (never_block) {
	        /* Set to be non-blocking */
		if (fcntl(p->file, F_SETFL, fcntl_flags | O_NDELAY) < 0)
		    perror("fcntl");
	    }
	    errno = 0;
	    written = write(p->file, pos, left);
	    errno_write = errno;
	    if (never_block) {
	        /* Restore original flags ASAP */
		if (fcntl(p->file, F_SETFL, fcntl_flags) < 0)
		    perror("fcntl");
	    }
	    if (written <= 0)
		switch (errno_write) {
		case EWOULDBLOCK:
		    if (++retry > psio_retries) {
			/* When print message, make sure that don't block */
		        fcntl_flags = fcntl(2, F_GETFL, 0);
			fcntl(2, F_SETFL, fcntl_flags | O_NDELAY);
			write(2,"NeWS: output dropped to avoid deadlock\n", 39);
			fcntl(2, F_SETFL, fcntl_flags);
			left = 0;
			break;
		    }
		    psio_wait(psio_block_delay);
		    break;
		case EINTR:
		    break;
		default:
		    p->flag |= PSERR;
		    return -1;
		}
	    else {
		left -= written;
		pos += written;
	    }
	}
    }
    p->ptr = p->base;
    p->cnt = p->bufsiz;
    return 0;
}

static
psio_wait(wait_tv)
    struct timeval wait_tv;
{
#ifdef SYSVREF
    struct pollfd fd[1];
#else
    int bits;
#endif
    int n = 0;
    register int i;

    /*
     * Wait wait_tv.  If n != 0 then got interrupted and try again.
     * This routine can wait up to 2*wait_tv and as little as almost
     * no time.
     */
    for (i = 0; n == 0 && i < 2; i++) {
#ifndef SYSVREF
    	bits = 0;
	n = select(0, &bits, &bits, &bits, &wait_tv);
#else
	n = poll(fd, 0, (wait_tv.tv_sec * 1000) + (wait_tv.tv_usec / 1000));
#endif
    }
}

psio_flushbuf(c, p)
    register PSFILE *p;
{
    psio_flush(p);
    if (p->flag & (PSSTRG | PSERR | PSEOF | PSREAD))
	return -1;
    p->cnt--;
    return *p->ptr++ = c;
}
