#ifndef lint
static char sccsid[] = "@(#)tftpd.c	4.11 (Berkeley) 7/2/83";
#endif

/*
 * Trivial file transfer protocol server.
 *
 * Modified at Caltech to run with safe user and group ids,
 * to have a login directory, to expand ~ in file names, and
 * to log attempts.
 *
 * Dan Whelan, November 1, 1983
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/file.h>

#include <netinet/in.h>

#include <arpa/tftp.h>

#include <signal.h>
#include <stdio.h>
#include <strings.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>
#include <setjmp.h>

#ifdef SAFEUSER
#include <pwd.h>
#endif SAFEUSER

#ifdef SAFEGROUP
#include <grp.h>
#endif SAFEGROUP

#define	TIMEOUT		2

extern	int errno;
struct	sockaddr_in sin = { AF_INET };
int	f;
int	rexmtval = TIMEOUT;
int	maxtimeout = 5*TIMEOUT;
char	buf[BUFSIZ];
int	reapchild();
char	*lclname;			/* expanded local file name */
long	time(), tloc;			/* time of day */
int	mypid;				/* process pid */

#ifdef SAFEUSER
int	SafeUid;
#endif SAFEUSER
#ifdef SAFEGROUP
int	SafeGid;
#endif SAFEGROUP

main(argc, argv)
	char *argv[];
{
	struct sockaddr_in from;
	register struct tftphdr *tp;
	register int n;
	struct servent *sp;

#ifdef SAFEUSER
    struct passwd *p;
#endif SAFEUSER

#ifdef SAFEGROUP
    struct group *g;

    g = (struct group *) getgrnam(SAFEGROUP);
    if (g == NULL) {
	fprintf(stderr,"Can't get gid for group '%s'\n",SAFEGROUP);
	exit(-1);
    }
    /* Set real gid to SAFEGROUP, effective gid to opr */
    SafeGid = g->gr_gid;
    if (setregid(SafeGid, -1) != 0) {
	fprintf(stderr,"Can't set gid to '%s'\n",SAFEGROUP);
	exit(-1);
    }
#endif SAFEGROUP

#ifdef SAFEUSER
    p = (struct passwd *) getpwnam(SAFEUSER);
    if (p == NULL) {
	fprintf(stderr,"Can't get uid for user '%s'\n",SAFEUSER);
	exit(-1);
    }
    /* Set real uid to SAFEUSER, effective uid to root */
    SafeUid = p->pw_uid;
    if (setreuid(SafeUid, -1) != 0) {
	fprintf(stderr,"Can't set uid to '%s'\n",SAFEUSER);
	exit(-1);
    }
#endif SAFEUSER

	sp = getservbyname("tftp", "udp");
	if (sp == 0) {
		fprintf(stderr, "tftpd: udp/tftp: unknown service\n");
		exit(1);
	}
	sin.sin_port = sp->s_port;
#ifndef DEBUG
	if (fork())
		exit(0);
	for (f = 0; f < 10; f++)
		(void) close(f);
	(void) open("/", 0);
	(void) dup2(0, 1);
	(void) dup2(0, 2);
	{ int t = open("/dev/tty", 2);	
	  if (t >= 0) {
		ioctl(t, TIOCNOTTY, (char *)0);
		(void) close(t);
	  }
	}
#endif
	signal(SIGCHLD, reapchild);

	time(&tloc);
	log("#\n#    TFTP server starting at %s#\n", ctime(&tloc));

	for (;;) {
		int fromlen;

		f = socket(AF_INET, SOCK_DGRAM, 0);
		if (f < 0) {
			perror("tftpd: socket");
			log("tftpd: socket call failed\n");
			sleep(5);
			continue;
		}
		if (setsockopt(f, SOL_SOCKET, SO_REUSEADDR, 0, 0) < 0) {
			perror("tftpd: setsockopt (SO_REUSEADDR)");
			log("tftpd: setsockopt (SO_REUSEADDR)\n");
		}
		sleep(1);			/* let child do connect */
		while (bind(f, (caddr_t)&sin, sizeof (sin), 0) < 0) {
			perror("tftpd: bind");
			log("tftpd: bind call failed\n");
			sleep(5);
		}
		do {
			fromlen = sizeof (from);
			n = recvfrom(f, buf, sizeof (buf), 0,
			    (caddr_t)&from, &fromlen);
		} while (n <= 0);
		tp = (struct tftphdr *)buf;
		tp->th_opcode = ntohs(tp->th_opcode);
		if (tp->th_opcode == RRQ || tp->th_opcode == WRQ) {
			if (fork() == 0)
				tftp(&from, tp, n);
		}
		else
		    log("tftpd: Bad TFTP OPCODE (%x), len = %d\n",
		        tp->th_opcode, n);
		(void) close(f);
	}
}

reapchild()
{
	union wait status;

	while (wait3(&status, WNOHANG, 0) > 0)
		;
}

int	validate_access();
int	sendfile(), recvfile();

struct formats {
	char	*f_mode;
	int	(*f_validate)();
	int	(*f_send)();
	int	(*f_recv)();
} formats[] = {
	{ "netascii",	validate_access,	sendfile,	recvfile },
	{ "ascii",	validate_access,	sendfile,	recvfile },
	{ "octet",	validate_access,	sendfile,	recvfile },
	{ "binary",	validate_access,	sendfile,	recvfile },
#ifdef notdef
	{ "mail",	validate_user,		sendmail,	recvmail },
#endif
	{ 0 }
};

int	fd;			/* file being transferred */

/*
 * Handle initial connection protocol.
 */
tftp(client, tp, size)
	struct sockaddr_in *client;
	struct tftphdr *tp;
	int size;
{
	register char *cp;
	int first = 1, ecode, n_bytes;
	register struct formats *pf;
	char *filename, *mode;

	/* Save the PID and starting time */
	mypid = getpid();
	time(&tloc);
	log("%d: TFTP slave process forked at %s", mypid, ctime(&tloc));

	if (connect(f, (caddr_t)client, sizeof (*client), 0) < 0) {
		perror("connect");
		log("%d: Can't connect to slave socket\n", mypid);
		exit(1);
	}
	filename = cp = tp->th_stuff;
again:
	while (cp < buf + size) {
		if (*cp == '\0')
			break;
		cp++;
	}
	if (*cp != '\0') {
		nak(EBADOP);
		exit(1);
	}
	if (first) {
		mode = ++cp;
		first = 0;
		goto again;
	}
	for (cp = mode; *cp; cp++)
		if (isupper(*cp))
			*cp = tolower(*cp);
	for (pf = formats; pf->f_mode; pf++)
		if (strcmp(pf->f_mode, mode) == 0)
			break;

	/*
	 * Log that we received a TFTP request
	 *
	 */

	if (tp->th_opcode == WRQ)
                log("%d: WRQ \"%s\" (%s) from %08.8x\n", mypid, filename,
		    pf->f_mode, client->sin_addr.s_addr);
	else 
                log("%d: RRQ \"%s\" (%s) from %08.8x\n", mypid, filename,
		    pf->f_mode, client->sin_addr.s_addr);


	/*
	 * Attempt to perform a TFTP transfer
	 *
	 */

	if (pf->f_mode == 0) {
		nak(EBADOP);
		exit(1);
	}
	ecode = (*pf->f_validate)(filename, client, tp->th_opcode);
	if (ecode) {
		nak(ecode);
		exit(1);
	}
	if (tp->th_opcode == WRQ) {
		n_bytes = (*pf->f_recv)(pf);
		log("%d: Received %d bytes\n", mypid, n_bytes);
	}
	else {
		n_bytes = (*pf->f_send)(pf);
		log("%d: Transfered %d bytes\n", mypid, n_bytes);
	}
	exit(0);
}

/*
 * Validate file access.
 * Expand file name.
 * Require file to exist and be
 * readable or writable.
 */
validate_access(file, client, mode)
	char *file;
	struct sockaddr_in *client;
	int mode;
{
	char *c;

	/* Translate the file name so ~ works */
        lclname = (char *) ftrans(file);
	if (*lclname == '\0') {
	    return(ENOTFOUND);
	}

	/* Check the files accessability */
	if (mode == RRQ) {
	    if (access(lclname, F_OK | R_OK) < 0) return(EACCESS);
	}
	else {
	    if (access(lclname, W_OK) < 0) {

		/* File doesn't exist, Check directory accessibility */
		if ((c = rindex(lclname, '/')) != NULL) {
		    *c = '\0';
		    if (access(lclname, F_OK | W_OK) < 0) return(EACCESS);
		    *c = '/';
		}
		else
		    if (access(".", F_OK | W_OK) < 0) return(EACCESS);
	    }
	    else
	    	return(EEXISTS);
	}

	/* Open the file */
	umask(0777);
	fd = open(lclname, mode == RRQ ? O_RDONLY : 
		(O_WRONLY | O_CREAT | O_EXCL), 0666);
	if (fd < 0)
		return (errno + 100);
	
#ifdef SAFEUSER
#ifdef SAFEGROUP	
	/* If we are writing this file, change the owner and group */
	if (mode == WRQ)
		fchown(fd, SafeUid, SafeGid);
#endif SAFEGROUP
#endif SAFEUSER

	return (0);
}

int	timeout;
jmp_buf	timeoutbuf;

timer()
{

	timeout += rexmtval;
	if (timeout >= maxtimeout) {
	    log("%d: Timed out after %d attempts.\n", mypid, timeout);
	    exit(1);
	}
	longjmp(timeoutbuf, 1);
}

/*
 * Send the requested file.
 */
sendfile(pf)
	struct format *pf;
{
	register struct tftphdr *tp;
	register int block = 1, size, n, n_bytes = 0;

	signal(SIGALRM, timer);
	tp = (struct tftphdr *)buf;
	do {
		size = read(fd, tp->th_data, SEGSIZE);
		if (size < 0) {
			nak(errno + 100);
			goto abort;
		}
		n_bytes += size;
		tp->th_opcode = htons((u_short)DATA);
		tp->th_block = htons((u_short)block);
		timeout = 0;
		(void) setjmp(timeoutbuf);
		if (write(f, buf, size + 4) != size + 4) {
			perror("tftpd: write");
			log("%d: Can't write to net.\n", mypid);
			goto abort;
		}
		do {
			alarm(rexmtval);
			n = read(f, buf, sizeof (buf));
			alarm(0);
			if (n < 0) {
				perror("tftpd: read");
				log("%d: Can't read from net.\n", mypid);
				goto abort;
			}
			tp->th_opcode = ntohs((u_short)tp->th_opcode);
			tp->th_block = ntohs((u_short)tp->th_block);
			if (tp->th_opcode == ERROR) {
				log("%d: Received TFTP ERROR.\n", mypid);
				goto abort;
			}
		} while (tp->th_opcode != ACK || tp->th_block != block);
		block++;
	} while (size == SEGSIZE);
abort:
	(void) close(fd);
	return(n_bytes);
}

/*
 * Receive a file.
 */
recvfile(pf)
	struct format *pf;
{
	register struct tftphdr *tp;
	register int block = 0, n, size, n_bytes = 0;

	signal(SIGALRM, timer);
	tp = (struct tftphdr *)buf;
	do {
		timeout = 0;
		tp->th_opcode = htons((u_short)ACK);
		tp->th_block = htons((u_short)block);
		block++;
		(void) setjmp(timeoutbuf);
		if (write(f, buf, 4) != 4) {
			perror("tftpd: write");
			log("%d: Can't write to net.\n", mypid);
			goto abort;
		}
		do {
			alarm(rexmtval);
			n = read(f, buf, sizeof (buf));
			alarm(0);
			if (n < 0) {
				perror("tftpd: read");
				log("Can't read from net.\n", mypid);
				goto abort;
			}
			tp->th_opcode = ntohs((u_short)tp->th_opcode);
			tp->th_block = ntohs((u_short)tp->th_block);
			if (tp->th_opcode == ERROR) {
				log("%d: Received TFTP ERROR.\n", mypid);
				goto abort;
			}
		} while (tp->th_opcode != DATA || block != tp->th_block);
		size = write(fd, tp->th_data, n - 4);
		if (size < 0) {
			nak(errno + 100);
			goto abort;
		}
		n_bytes += size;
	} while (size == SEGSIZE);
abort:
	tp->th_opcode = htons((u_short)ACK);
	tp->th_block = htons((u_short)(block));
	(void) write(f, buf, 4);
	(void) close(fd);
	return(n_bytes);
}

struct errmsg {
	int	e_code;
	char	*e_msg;
} errmsgs[] = {
	{ EUNDEF,	"Undefined error code" },
	{ ENOTFOUND,	"File not found" },
	{ EACCESS,	"Access violation" },
	{ ENOSPACE,	"Disk full or allocation exceeded" },
	{ EBADOP,	"Illegal TFTP operation" },
	{ EBADID,	"Unknown transfer ID" },
	{ EEXISTS,	"File already exists" },
	{ ENOUSER,	"No such user" },
	{ -1,		0 }
};

/*
 * Send a nak packet (error message).
 * Error code passed in is one of the
 * standard TFTP codes, or a UNIX errno
 * offset by 100.
 */
nak(error)
	int error;
{
	register struct tftphdr *tp;
	int length;
	register struct errmsg *pe;
	extern char *sys_errlist[];

	tp = (struct tftphdr *)buf;
	tp->th_opcode = htons((u_short)ERROR);
	tp->th_code = htons((u_short)error);
	for (pe = errmsgs; pe->e_code >= 0; pe++)
		if (pe->e_code == error)
			break;
	if (pe->e_code < 0)
		pe->e_msg = sys_errlist[error - 100];
	strcpy(tp->th_msg, pe->e_msg);
	length = strlen(pe->e_msg);
	tp->th_msg[length] = '\0';
	length += 5;
	if (write(f, buf, length) != length)
		perror("nak");

	/* Log the error */
	log("%d: Error #%d: %s\n", mypid, error, pe->e_msg);
}


/*
 * log -- Log an error message in the log file
 *
 */

log(fmt, args)
char *fmt;
{
    FILE *f_log;

#ifdef LOGFILE
    if ((f_log = fopen(LOGFILE, "a+")) == NULL) {
	perror("log_mess: Problems accessing the logfile.");
    }
    _doprnt(fmt, &args, f_log);
    fclose(f_log);
#endif
}

