
/*
 * filedate - show or set access/modify times of files
 * (partly scavenged from Unix "date" program)
 *
 * the -m flag says to display the modification time of the file(s)
 * the -a flag says to display the access time of the file(s)
 * the -M flag takes the following argument as a new modification time
 * the -A flag takes the following argument as a new access time
 * if both -a/-m and -A/-M are given, the old value is displayed
 * before setting the new value.
 * the -u flag says use GMT instead of local time
 *
 * Times are in the same form as used by date(1):
 *    yymmddhhmm[.ss]
 * where the seconds are always displayed but need not be entered
 *
 * The displayed values are given in the form
 *   mtime=nnn   atime=nnn
 * so they can be used in sh(1) scripts with eval.
 *
 * If the -c flag is given, the output format is instead
 *   set mtime=nnn    atime=nnn
 * for csh(1) scripts.
 *
 * Copyright MCMLXXXV Stanford University
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>

static	int	dmsize[12] =
    { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

static char *usage = "usage: filedate [-a] [-m] \
[-A yymmddhhmm[.ss]] [-M yymmddhhmm[.ss]] [-F oldfile] [-u] file...\n";

time_t old_mtime, old_atime, new_mtime, new_atime;
struct  timezone time_zone;
int uflag;      /* Use GMT */
int cflag;      /* Format output for csh(1) */
int backonly;   /* Set time backwards only (not to the future) */
char *format_time();

main(argc, argv)
	int argc;
	char *argv[];
{
    int i;
    int show_mtime, show_atime, set_mtime, set_atime; /* Flags */
    int have_old_times; /* True if we have read the old values from file */
    int newflags;       /* Starting to read a new set of flags */
    int rc;         /* Return code */
    struct  timeval tv;

    newflags = 1;
    gettimeofday(&tv, &time_zone);  /* All we need is the time zone */
    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    /*
	     * Reset these for each new set of files, e.g.
	     * filedate -a file1 -m file2
	     */
	    if (newflags)
	    {
		newflags = 0;
		show_mtime = show_atime = set_mtime = set_atime =
		    have_old_times = 0;
	    }
	    switch(argv[i][1])
	    {
	      case 'm':     /* Show modification time */
		show_mtime = 1;
		break;

	      case 'a':     /* Show access time */
		show_atime = 1;
		break;

	      case 'M':     /* Set modification time */
		if (gtime(argv[++i], &new_mtime) != 0)
		    exit(1);
		set_mtime = 1;
		break;

	      case 'A':     /* Set access time */
		if (gtime(argv[++i], &new_atime) != 0)
		    exit(1);
		set_atime = 1;
		break;

	      case 'F':     /* Set times from file */
		if (getfiletimes(argv[++i]))
		    exit(1);
		set_mtime = set_atime = 1; /* Set both times */
		new_mtime = old_mtime;
		new_atime = old_atime;
		break;

	      case 'c':         /* Print "set" first, for csh */
		cflag = 1;
		break;

	      case 'u':         /* Use GMT instead of local time */
		uflag = 1;
		break;

	      case 'b':         /* Allow dates to be set backwards only */
		backonly = 1;
		break;

	      default:
		printf(usage);
		exit(1);
	    }
	}
	else                     /* This is a file name */
	{
	    newflags = 1;       /* Re-initialize if we get more flags */
	    if (show_mtime || show_atime)
	    {
		if (getfiletimes(argv[i])) /* Get current values */
		    rc = 1;         /* Error */
		else
		{
		    have_old_times = 1;
		    if (cflag)
			printf("set ");
		    if (show_mtime)
		    {
			printf("mtime=%s", format_time(old_mtime));
		    }
		    if (show_atime)
		    {
			if (show_mtime)
			    printf("        ");
			printf("atime=%s", format_time(old_atime));
		    }
		    printf("\n");
		}
	    }

	    /* -b is meaningless if neither -M, -A, nor -F is given */
	    if (set_mtime || set_atime)
	    {
		/* If they aren't both set, have to get old values first */
		/* Also need old values if changing the time to the
		   future is disallowed. */
		if (backonly || !(set_mtime && set_atime))
		{
		    if (!have_old_times)
			getfiletimes(argv[i]);
		    /* Don't update the time if either:
		     * 1. -M not specified, or
		     * 2. -M is specified, and -b is specified, and
		     *    the time specified with -M is newer than the
		     *    current one.
		     */
		    if (!set_mtime ||
			(backonly && new_mtime > old_mtime))
			new_mtime = old_mtime;
		    /* Same for access time */
		    if (!set_atime ||
			(backonly && new_atime > old_atime))
			new_atime = old_atime;
		}
		if (setfiletimes(argv[i])) /* Set times */
		    rc  = 1;    /* Indicate error */
	    }
	}
    }
    return rc;
}


gtime(str, tv)  /* Convert string in yymmddhhmm.ss form to time */
    char *str;   /* String in yymmddhhmm.ss form */
    time_t *tv;  /* Pointer to place to put time in seconds */
{
    int year, month;
    int day, hour, mins, secs;
    char timestring[15];
    register char *cp; /* Pointer to current position */
    int len;
    time_t val;         /* Value to return */
    int i;

    len = strlen(str);
    if (len > 13)
    {
	fprintf(stderr, "Date specification too long\n");
	return 1;
    }

    strcpy(timestring, str);

    /* Work backwards from end of string, putting nulls as we
       go along to terminate the next atoi */

    /* Look for optional ".ss" (seconds) */
    if ((cp = (char *)index(timestring, '.')) != 0)
    {
	if (len != 13 || cp != &timestring[10])
	{
	    fprintf(stderr, "Date specification ill formed\n");
	    return 1;  /* Period not in the right place */
	}
	secs = atoi(cp+1);
	*cp = '\0';
    }
    else      /* If none, start at end of string */
    {
	if (len != 10)
	{
	    fprintf(stderr, "Date specification wrong length\n");
	    return 1;
	}
	secs=0;
	cp = &timestring[len];
    }

    cp -= 2;
    mins = atoi(cp);
    *cp = '\0';
    cp -= 2;
    hour = atoi(cp);
    *cp = '\0';
    cp -= 2;
    day = atoi(cp);
    *cp = '\0';
    cp -= 2;
    month = atoi(cp);
    *cp = '\0';
    cp -= 2;
    year = atoi(cp);

    /* The rest of this routine comes from the Unix "date" program */
    if (month < 1 || month > 12)
    {
	fprintf(stderr, "Month %d out of range in %s\n", month, str);
	return 1;
    }
    if (day < 1 || day > 31)
    {
	fprintf(stderr, "Day %d out of range in %s\n", day, str);
	return 1;
    }
    if (mins < 0 || mins > 59)
    {
	fprintf(stderr, "Minutes %d out of range in %s\n", mins, str);
	return 1;
    }
    if (secs < 0 || secs > 59)
    {
	fprintf(stderr, "Seconds %d out of range in %s\n", secs, str);
	return 1;
    }
    if (year < 70)
    {
	fprintf(stderr, "Year %d out of range in %s\n", year, str);
	return 1;
    }
    if (hour == 24) {
	    hour = 0;
	    day++;
    }
    if (hour < 0 || hour > 23)
    {
	fprintf(stderr, "Hour out of range in %s\n", str);
	return 1;
    }
    val = 0;
    year += 1900;
    for (i = 1970; i < year; i++)
	    val += dysize(i);
    /* Leap year */
    if (dysize(year) == 366 && month >= 3)
	    val++;
    while (--month)
	    val += dmsize[month-1];
    val += day-1;
    val = 24*val + hour;
    val = 60*val + mins;
    val = 60*val + secs;
    if (uflag == 0)         /* Adjust for time zone */
    {
	    val += (long)time_zone.tz_minuteswest*60;
	    /* now fix up local daylight time */
	    if (localtime(&val)->tm_isdst)
		    val -= 60*60;
    }
    *tv = val;
    return 0;
}


setfiletimes(filename)
char *filename;
{
    struct timeval tvp[2];

    tvp[0].tv_sec = new_atime;
    tvp[0].tv_usec = 0;
    tvp[1].tv_sec = new_mtime;
    tvp[1].tv_usec = 0;
    if (utimes(filename, tvp) == -1)
    {
	perror(filename);
	return 1;
    }
    return 0;
}


getfiletimes(filename)
    char *filename;
{
    struct stat statbuf;

    if (lstat(filename, &statbuf) == -1)
    {
	perror(filename);
	return 1;
    }
    old_atime = statbuf.st_atime;
    old_mtime = statbuf.st_mtime;
    return 0;
}



char *
format_time(timeval)
    time_t timeval;
{
    static char buf[100];
    struct tm *tm;

    tm = uflag ? gmtime(&timeval) : localtime(&timeval);
    sprintf(buf, "%02d%02d%02d%02d%02d.%02d",
	tm->tm_year, tm->tm_mon+1, tm->tm_mday,
	tm->tm_hour, tm->tm_min, tm->tm_sec);
    return buf;
}
