#define	HZ 60
/*
 * enstat.c
 *
 * Print enet status & statistics: 4.2BSD version
 *
 * HISTORY:
 * 17 October 1984	Jeff Mogul	Stanford
 *	- Prints RecvCount field; this comes out as part of
 * 	the "Filters" listing, even though it should probably be
 *	with the "AllDescriptors" list, because it fits better on
 *	the output lines.
 *
 * 3 October 1984	Jeff Mogul	Stanford
 *	- Prints state of "enOneCopy", if present
 *
 * 17 February 1984	Jeff Mogul	Stanford
 *	- Fixes from "decvax!genrad!grkermit!masscomp!clyde!watmath!arhwhite"
 *	to:
 *		make "k" option work (using core dumps instead of kmem);
 *
 * 15 February 1984	Jeff Mogul	Stanford
 *	- Added printout of underlying device name
 *
 * 20 December 1983	Jeffrey Mogul	Stanford
 *	- added unit number options ("01234567")
 *
 * 2 December 1983	Jeffrey Mogul	Stanford
 *	- added printout of new endevp structure, new "p" flag
 *
 * Jeffrey Mogul	Stanford	28 April 1983
 *
 * Derived from:
 *	static char *sccsid = "@(#)pstat.c	4.9 (Berkeley) 5/7/81";
 * and from
 *	modifications thereto by Mike Accetta @ CMU
 */

#define mask(x) (x&0377)
#define	clear(x) ((int)(x)&0x7FFFFFFF)

#include <stdio.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/enet.h>
#include <sys/enetdefs.h>
#include <sys/vm.h>
#include <machine/pte.h>

#include <net/if.h>

#include <nlist.h>


char   *fcore = "/dev/kmem";
char   *fnlist = "/vmunix";
int     fc;

struct nlist    nl[] = {

#define	SENSTATE	(0)
    {
	"_enState"
    },
#define	SENQELTS	(1)
    {
	"_enQueueElts"
    },
#define	SENFREEQ	(2)
    {
	"_enFreeq"
    },
#define	SENUNITS	(3)
    {
	"_enUnits"
    },
#define	SENFREEQMIN 	(4)
    {
	"_enFreeqMin"
    },
#define	SENSCAVENGES	(5)
    {
	"_enScavenges"
    },
#define	SENDEBUG	(6)
    {
	"_enDebug"
    },
#define	SENXFREEQ	(7)
    {
	"_enXfreeq"
    },
#define	SENALLOC	(8)
    {
	"_enAllocCount"
    },
#define	SENETINFO	(9)
    {
	"_enet_info"
    },
#define	X_SYSMAP	(10)
    {
	"_Sysmap"
    },
#define	X_SYSSIZE	(11)
    {
	"_Syssize"
    },
#define	SENONECOPY	(12)
    {
	"_enOneCopy"
    },

    0,
};

int     kflg = 0;
int     Verbose = 0;
int     Counts = 0;
int     FreeStats = 0;
int     Descriptors = 0;
int     Filters = 0;
int     QueueElements = 0;
int 	Parameters = 0;
int	UnitMask = 0;

main (argc, argv)
char  **argv;
{
    register char  *argp;
    int     something = 0;

    argc--, argv++;
    while (argc > 0 && **argv == '-') {
	argp = *argv++;
	argp++;
	argc--;
	while (*argp++)
	    switch (argp[-1]) {

		case 'k': 
		    kflg++;
		    fcore = "/vmcore";
		    break;

		case 'v': 
		    Verbose++;
		    break;

		case 'c': 
		    Counts++;
		    something++;
		    break;

		case 's': 
		    FreeStats++;
		    something++;
		    break;

		case 'd': 
		    Descriptors++;
		    something++;
		    break;

		case 'f': 
		    Filters++;
		    something++;
		    break;

		case 'q': 
		    QueueElements++;
		    something++;
		    break;

		case 'p':
		    Parameters++;
		    something++;
		    break;

		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		    UnitMask |= (1<<(argp[-1] - '0'));
		    break;

		default:
		    fprintf(stderr,
		    	"Usage: enstat [-kvcsdfqp01234567] [nlist [core]]\n");
		    fprintf(stderr,
    "\t(c=Counts, s=FreeStats, d=Descriptors, f=filters, q=QueueElts,\n");
		    fprintf(stderr,
    "\t p=Parameters); 0-7 specifies unit\n");
		    exit(1);
	    }
    }
    if (!something) {		/* if none given then use all */
	Counts = FreeStats = Descriptors = Filters = QueueElements = 1;
	Parameters = 1;
    }
    if (argc > 0) {		/* different system image */
	fnlist = argv[0];
	argv++;
	argc--;
    }
    if (argc > 0)
	fcore = argv[0];

    if (UnitMask == 0)		/* no specific units, do them all */
    	UnitMask = 0xFFFFFFFF;

    if ((fc = open (fcore, 0)) < 0) {
	printf ("Can't find %s\n", fcore);
	exit (1);
    }

    nlist (fnlist, nl);
    if (nl[0].n_type == 0) {
	printf ("no namelist\n");
	exit (1);
    }

    doenet ();
}

/* copied from enet./sys/vaxif/enet.c */
struct enet_info {
	struct ifnet *ifp;
};

#define	NEN	4	/*
			 * enough for a good long while, 8 is the maximum
			 * possible with 32 minors/unit
			 */

doenet () {
    struct enPacket enQueueElts[ENPACKETS];
    struct enQueue  enFreeq;
    struct enState  enState[NEN];
    struct enet_info enet_info[NEN];
    int     enFreeqMin,
            enScavenges,
            enDebug,
	    enUnits,
	    enOneCopy;
    int     enAllocCount;
    register    loc;
    register struct enOpenDescriptor   *op;
    register struct enPacket   *p;
    register int    i;
    register struct enState *enStatep;

    klseek (fc, (long) nl[SENUNITS].n_value, 0);
    read (fc, &enUnits, sizeof (enUnits));

    klseek (fc, (long) nl[SENSTATE].n_value, 0);
    read (fc, enState, enUnits * sizeof (struct enState));
    klseek (fc, (long) nl[SENETINFO].n_value, 0);
    read (fc, enet_info, enUnits * sizeof (struct enet_info));
    klseek (fc, (long) nl[SENQELTS].n_value, 0);
    read (fc, enQueueElts, sizeof (enQueueElts));
    klseek (fc, (long) nl[SENFREEQ].n_value, 0);
    read (fc, &enFreeq, sizeof (enFreeq));
    klseek (fc, (long) nl[SENFREEQMIN].n_value, 0);
    read (fc, &enFreeqMin, sizeof (enFreeqMin));
    klseek (fc, (long) nl[SENSCAVENGES].n_value, 0);
    read (fc, &enScavenges, sizeof (enScavenges));
    klseek (fc, (long) nl[SENDEBUG].n_value, 0);
    read (fc, &enDebug, sizeof (enDebug));
    klseek (fc, (long) nl[SENALLOC].n_value, 0);
    read (fc, &enAllocCount, sizeof (enAllocCount));
    if (nl[SENONECOPY].n_type) {
	klseek (fc, (long) nl[SENONECOPY].n_value, 0);
	read (fc, &enOneCopy, sizeof (enOneCopy));
    }
    else {
	enOneCopy = -1;
    }

    if (FreeStats) {
	printf ("Scavenges:\t%d\n", enScavenges);
	printf ("Allocated:\t%d\n", enAllocCount);
	printf ("Freeq(%x): %d (Min %d) [%x,%x]\n",
		 (nl[SENFREEQ].n_value),
		enFreeq.enQ_NumQueued, enFreeqMin,
		 (enFreeq.enQ_F),  (enFreeq.enQ_B));
	if (enOneCopy >= 0)
	    printf("OneCopy:\t%d\n", enOneCopy);
	printf ("\n");
    }

    for (enStatep = enState; enStatep < &enState[enUnits]; enStatep++) {
	int unit = enStatep - enState;
	
	if ((UnitMask & (1<<unit)) == 0)	/* skip this unit */
	    continue;
	
	if (enDesq.enQ_F == 0)	/* probably not needed any more */
	    continue;
	if (Counts || Parameters || Descriptors || Filters) {
	    printf ("enet%d:\n\n", unit);
	}
	if (Counts) {
	    printf ("Xcnt:\t\t%-8d\tRcnt:\t\t%-8d\n", enXcnt, enRcnt);
	    printf ("Xdrops:\t\t%-8d\tRdrops:\t\t%-8d\n",
		    enXdrops, enRdrops);
	    printf ("\n");
	}
	if (Parameters) {
	    struct endevp *devp = &enDevParams;
	    printf ("Device type: ");
	    switch (devp->end_dev_type) {
		case ENDT_3MB:
		     printf("3Mb");
		     break;
		case ENDT_BS3MB:
		     printf("byte-swapping 3Mb");
		     break;
		case ENDT_10MB:
		     printf("10Mb");
		     break;
		default:
		     printf("unknown (%d)", devp->end_dev_type);
		     break;
	    }
	    PrintIFP(enet_info[unit].ifp);
	    printf("\n");
	    printf("Address Length: %d\tHeader Length: %d\tMTU: %d\n",
    		devp->end_addr_len, devp->end_hdr_len, devp->end_MTU);
	    printf("Interface Address: ");
	    PrintAddress(devp->end_addr, devp->end_addr_len);
	    printf("\nBroadcast Address: ");
	    PrintAddress(devp->end_broadaddr, devp->end_addr_len);
	    printf("\n\n");
	}

	if (Descriptors) {
	    printf ("Desq(%x): %d open files [%x,%x]:\n",
		     (((caddr_t)&enDesq
		    		- (caddr_t)enState + nl[SENSTATE].n_value)),
		    enDesq.enQ_NumQueued,
		     (enDesq.enQ_F),  (enDesq.enQ_B));
	    printf ("AllDescriptors:\n");
	    printf ("       LOC    USED   LINK-QUEUE STATE WAIT-QUEUE NQ'D TOUT SIGN   PROC(PID)");
	    printf ("\n");
	    loc = ((caddr_t)enAllDescriptors - (caddr_t)enState)
	    					+ nl[SENSTATE].n_value;
	    for (i = 0, op = enAllDescriptors;
	    		op < &enAllDescriptors[ENMAXOPENS];
			op++, loc += sizeof (enAllDescriptors[0]), i++) {
		if (Verbose == 0 && enOpenFlag[i] == 0)
		    continue;
		printf ("%2d%c %8.1x ", i,
			op -> enOD_Flag & ENKERNEL ? 'K' : ' ',
			loc | 0x80000000);
		printf ("  %s ", enOpenFlag[i] ? "yes" : "no ");
		popdes (op);
	    }
	}

	if (Filters) {
	    printf ("\nFilters:\n");
	    printf ("      LOC     COUNT PRI LEN FILTER");
	    printf ("\n");
	    loc = ((caddr_t) enAllDescriptors - (caddr_t) enState)
	    					+ nl[SENSTATE].n_value;
	    for (i = 0, op = enAllDescriptors;
	    		op < &enAllDescriptors[ENMAXOPENS];
			op++, loc += sizeof (enAllDescriptors[0]), i++) {
		if (Verbose == 0 && enOpenFlag[i] == 0)
		    continue;
		printf ("%2d %8.1x ", i, loc | 0x80000000);
		printf("%7d ", op->enOD_RecvCount);
		pfilt (&op -> enOD_OpenFilter);
	    }
	}
    }
    if (QueueElements) {
	printf ("\n");
	printf ("QueueElts:\n");
	printf ("   LOC     LINK-QUEUE  FUNC  DATA  COUNT   REF");
	printf ("\n");
	loc = nl[SENQELTS].n_value;
	for (p = enQueueElts; p < &enQueueElts[ENPACKETS];
				p++, loc += sizeof (enQueueElts[0])) {
	    printf ("%8.1x ", loc | 0x80000000);
	    pqelt (p);
	}
    }
}

popdes (op)
register struct enOpenDescriptor   *op;
{
    static char *states[] = {
	" wait", "timed", " tout"
    };
    int     num;
    int     head;
    int	    to;

    num = (Verbose) ? ENMAXWAITING : op -> enOD_Waiting.enWQ_NumQueued;
    head = (Verbose) ? 0 : op -> enOD_Waiting.enWQ_Head;

    printf (" %5x", clear (op -> enOD_Link.F));
    printf (" %5x", clear (op -> enOD_Link.B));
    if ((int) op -> enOD_RecvState <= 2)
	printf (" %-5.5s", states[(int) op -> enOD_RecvState]);
    else
	printf (" %3d  ", op -> enOD_RecvState);
    if (num)
	pwq (&op -> enOD_Waiting, head);
    else
	printf (" %-10s", "");
    printf (" %1d", op -> enOD_Waiting.enWQ_NumQueued);
    printf ("(%1d)", op -> enOD_Waiting.enWQ_MaxWaiting);
    to = op->enOD_Timeout;
    if ((to > HZ*60) && (Verbose == 0)) {
	to /= (HZ * 60);
	if (to > 60) {
	    to /= 60;
	    if (to > 999)
	    	printf(" long");
	    else
	    	printf(" %3h", to);
	}	
	else
    	    printf(" %3dm", to);
    }
    else
	printf (" %4d", op -> enOD_Timeout);
    printf (" %3d ", op -> enOD_SigNumb);
    printf (" %6x", clear (op -> enOD_SigProc));
    printf ("(%d)", op -> enOD_SigPid);
    printf ("\n");
    while (--num > 0) {
	enNextWaitQueueIndex (head);
	printf ("%-37s", "");
	pwq (&op -> enOD_Waiting, head);
	printf ("\n");
    }
}


pwq (wq, idx)
register struct enWaitQueue *wq;
{

    int     tail = wq -> enWQ_Tail;

    enPrevWaitQueueIndex (tail);
    printf (" %3s", (idx == wq -> enWQ_Head) ? " ^ " : "   ");
    printf ("%5x", clear (wq -> enWQ_Packets[idx]));
    printf ("%s", (idx == tail) ? " $" : "  ");

}

pfilt (fp)
register struct enfilter   *fp;
{

    int     w,
            arg,
            op,
            i;
    unsigned short *wp;

    printf ("%3d ", fp -> enf_Priority);
    printf ("%3d ", fp -> enf_FilterLen);

    i = 0;
    for (wp = &fp -> enf_Filter[0];
    			wp < &fp -> enf_Filter[fp -> enf_FilterLen]; wp++) {
	w = *wp;
	arg = (w & (0xffff >> ((sizeof (short) * 8) - ENF_NBPO)));
	op = (w & (0xffff << ENF_NBPA));
	switch (arg) {
	    default: 
		if (arg < ENF_PUSHWORD)
		    printf ("BADPUSH%d", arg);
		else
		    printf ("PUSHWORD+%d", arg - ENF_PUSHWORD);
		break;
	    case ENF_PUSHLIT: 
		printf ("PUSHLIT");
		break;
	    case ENF_PUSHZERO: 
		printf ("PUSHZERO");
		break;
	    case ENF_NOPUSH: 
		if (op == ENF_NOP)
		    printf ("NOP");
		break;
	}
	if (arg != ENF_NOPUSH && op != ENF_NOP)
	    printf ("|");
	switch (op) {
	    case ENF_AND: 
		printf ("AND");
		break;
	    case ENF_OR: 
		printf ("OR");
		break;
	    case ENF_XOR: 
		printf ("XOR");
		break;
	    case ENF_EQ: 
		printf ("EQ");
		break;
	    case ENF_NEQ: 
		printf ("NEQ");
		break;
	    case ENF_LT: 
		printf ("LT");
		break;
	    case ENF_LE: 
		printf ("LE");
		break;
	    case ENF_GT: 
		printf ("GT");
		break;
	    case ENF_GE: 
		printf ("GE");
		break;
	    case ENF_NOP: 
		break;
	    case ENF_COR:
	    	printf("COR");
		break;
	    case ENF_CAND:
	    	printf("CAND");
		break;
	    case ENF_CNOR:
	    	printf("CNOR");
		break;
	    case ENF_CNAND:
	    	printf("CNAND");
		break;
	    default: 
		printf ("OP%d", op);
		break;
	}
	printf (",");
	if (arg == ENF_PUSHLIT) {
	    i++;
	    printf ("%.1o,", *++wp);
	}
	if (++i >= 4) {
	    i = 0;
	    if (&wp[1] < &fp -> enf_Filter[fp -> enf_FilterLen])
		printf ("\n\t\t\t    ");
	}
    }
    printf ("\n");

}

pqelt (qp)
register struct enPacket   *qp;
{

    printf (" %5x", clear (qp -> enP_Link.F));
    printf (" %5x", clear (qp -> enP_Link.B));
    printf (" %5x", clear (qp -> enP_Func));
    printf (" %5x", clear (qp -> enP_Data));
    printf (" %5d", qp -> enP_ByteCount);
    printf (" %5d", qp -> enP_RefCount);
    printf ("\n");

}

PrintAddress(a, l)
unsigned char *a;
int l;
{
	if (l == 1) {	/* one-byte addresses in octal */
	    if (*a)
	    	printf("0");
	    printf("%o", *a);
	    return;
	}
	while (l-- > 0) {
	    printf("%x", *a++);
	    if (l > 0)
	    	printf(".");
	}
}

PrintIFP(ifp)
struct ifnet *ifp;
{
	struct ifnet ifnet;
	char name[16];
	
	printf("\tInterface: ");

	if (ifp == 0) {
	    printf("[ifp == 0?]");
	    return;
	}
	
	klseek(fc, (long) ifp, 0);
	read(fc, &ifnet, sizeof(struct ifnet));
	
	klseek(fc, (long)ifnet.if_name, 0);
	read(fc, name, sizeof(name));
	name[15] = 0;
	
	printf("%s%d", name, ifnet.if_unit);
	
	if ((ifnet.if_flags&IFF_UP) == 0)
	    printf(" not up");
}

/*
 * Stolen from ps as is
 */
klseek(fd, loc, off)
	int fd;
	long loc;
	int off;
{

	/*
	 * If this is a dump, then the kernel isn't doing the page
	 * mapping for us.  Simulate it.
	 */
	if (kflg && (loc & 0x80000000) != 0) {
		long v;
		long addr;
		struct pte pte;

		loc &= 0x7fffffff;
		v = btop(loc);
 		if(v >= nl[X_SYSSIZE].n_value) {
 			printf("address botch %x\n", loc);
 			return;
 		}
 		addr = (long)((struct pte *)nl[X_SYSMAP].n_value + v);
 		lseek(fd, addr&0x7fffffff, 0);
 		if(read(fd, (char *)&pte, sizeof(pte)) != sizeof(pte)) {
 			printf("Error reading kmem for pte at %x\n", addr);
 			return;
 		}
 		if (pte.pg_v == 0 && (pte.pg_fod || pte.pg_pfnum == 0)) {
 			printf("pte bad for %x\n", addr);
 			return;
 		}
 		loc = (long)ptob(pte.pg_pfnum) + (loc & PGOFSET);
 	}
	(void) lseek(fd, (long)loc, off);
}
