/*
 * pupprint.c - print out a PUP packet
 *
 * Bill Nowicki September 1982
 *
 * with some stuff from Jeff Mogul's pupwatch program
 *
 * Modified:
 * 22 June 1983	  Jeffrey Mogul
 *	- complains if Pup checksums are bad
 */

# include <stdio.h>
# include <pup/puplib.h>
# include <pup/pupconstants.h>
# include <pup/puppacket.h>
# include <pup/leaf.h>


PupPrint( buf, out, count)
  struct PupPacket *buf;/* pointer to packet */
  FILE *out;		/* file to print on */
  int count;		/* size of packet in bytes  */
  {
    /*
     * decode a PUP packet onto given file
     */
    struct Port src, dst;
    int length;
    char *PupTypeName();
    ushort checksum();
    int checksumwrd;
    ushort pkcksum;

# ifdef VAX
    /* swab (&buf->PupSrc, &buf->PupSrc, sizeof(buf->PupSrc));
    swab (&buf->PupDst, &buf->PupDst, sizeof(buf->PupDst));
    swab (&buf->PupLength, &buf->PupLength, sizeof(buf->PupLength)); */
    fprintf (out, "\nMicroVAX netwatch does not examine PUP packets at the present time\n");
    return;
# endif VAX
    
    PortUnpack( &buf->PupSrc, &src);
    PortUnpack( &buf->PupDst, &dst);
    length = buf->PupLength - PUPACKOVER;

    fprintf( out, "PUP Type %s Length %d (%d Data)\n", 
    	PupTypeName(buf->PupType, dst.socket, src.socket),
	buf->PupLength, length);
    
    if (count < buf->PupLength )
      {
        fprintf( out, "Short packet, actual size=%d, PUP Length=%d\n",
		count, buf->PupLength);
      }

    fprintf( out, "From "); PortPrint( &src, out );
    fprintf( out, " to ");  PortPrint( &dst, out );
    fprintf( out, " ID 0%o", buf->PupID );

    /* check checksum */
    checksumwrd = roundup2(length);
    pkcksum = *((ushort *)&(buf->PupData[checksumwrd]));
    if ((pkcksum != NOCKSUM) && (pkcksum !=
    		checksum(buf, checksumwrd + PUPACKOVER - 2))) {
	fprintf(out, " *BAD CHECKSUM*");
    }

    fprintf( out, "\n" );

    switch (buf->PupType)
     {
       case BSP_ACK:
          fprintf(out, "Bytes/Pup: %d, Pups: %d, Total bytes: %d\n",
	      *(short *)(buf->PupData), *(short *)(buf->PupData+2),
	      *(short *)(buf->PupData+4)	);
       	  break;

       case RFC:
	   fprintf(out,"Connection ");
       case NAMERESP:
       case ADDREQ:
           PortUnpack( buf->PupData, &dst );
	   fprintf(out,"Port: ");
           PortPrint( &dst, out);
           fprintf( out, "\n" );
	   break;
	   
       case ERRORPUP:
           PrintErrorPUP(out, buf->PupData, length );
	   break;

       case SEQUIN:
       	  PrintSequin(out, buf->PupID, length, buf->PupData);
	  break;

       case GWINFREP:
           PrintGatewayInfo(out, buf->PupData, length );
	   break;

	case NETDIRVERSION:	
	    fprintf( out, "Version number %d\n",
	    	buf->PupData[0]*256+buf->PupData[1]);
	    break;

       	default:
          if (length>0)
            {
	        int count;
		extern int PrintData;

	        fprintf( out, "Data: ");
		for (count=0; count<length && (count<64 || PrintData); count++)
	  	    PrettyPutchar( buf->PupData[count], out);
		if (count<length) fprintf(out,"...");
        	fprintf( out, "\n");
	   }
      }
  }


PortPrint(prt, out)
  struct Port *prt;
  FILE *out;
{
	fprintf( out, "%o#%o#",prt->net,prt->host);
	
	switch(prt->socket) {

	case TELNET:
		fprintf( out, "Telnet");
		break;
	case GATEWAYINFO:
		fprintf( out, "GatewayInfo");
		break;
	case FTP:
		fprintf( out, "FTP");
		break;
	case MISCSERVICES:
		fprintf( out, "MiscServices");
		break;
	case ECHOSERVER:
		fprintf( out, "EchoServer");
		break;
	case MAIL:
		fprintf( out, "Mail");
		break;
	case EFTPSERVER:
		fprintf( out, "EftpServer");
		break;
	case EARSSTATUS:
		fprintf( out, "EarsStatus");
		break;
	case LEAF:
		fprintf( out, "Leaf");
		break;
	
	default:
		fprintf( out, "%o",prt->socket);
		break;
	}
}


char *PupTypeName(pt,dsock,ssock)
  unsigned char pt;
  unsigned long dsock;
  unsigned long ssock;
 {
    /*
     * Given a Pup type and a pair of sockets, PupTypeName
     * tries to come up with a human-sensible name for the pup
     * type.
     */
  static char numtype[100];

  if (pt < 0100)
    {	/* Registered pup type */
	
	switch (pt)
	  {
	    case ECHOME:	return("Echome");
	    case IMANECHO:	return("ImAnEcho");
	    case IMABADECHO:	return("ImABadEcho");
	    case ERRORPUP: 	return("Error");
	    case RFC:		return("RFC");
	    case RTP_ABORT: 	return("RTP Abort");
	    case RTP_END:	return("RTP End");
	    case RTP_ENDR: 	return("RTP EndReply");
	    case BSP_DATA: 	return("BSP Data");
	    case BSP_ADATA:	return("BSP AData");
	    case BSP_ACK:	return("BSP Ack");
	    case BSP_MARK:	return("BSP Mark");
	    case BSP_INTR:	return("BSP Interrupt");
	    case BSP_INTRR:	return("BSP InterruptReply");
	    case BSP_AMARK:	return("BSP AMark");

	    case EFTPDATA:	return("Eftp Data");
	    case EFTPACK:	return("Eftp Ack");
	    case EFTPEND:	return("Eftp End");
	    case EFTPABORT:	return("Eftp Abort");
	    
	    default:
		sprintf(numtype,"[strange registered type %o]",pt);
		return(numtype);
	    }
	}


	if ((ssock == MISCSERVICES) ||(dsock == MISCSERVICES)) 
	{
		/* MiscServices pup type */

	 switch (pt) 
	   {
	    case STIMEREQ:    	return("string time request");
	    case STIMERESP:    	return("string time reply");
	    case ATIMEREQ:    	return("Alto time request");
	    case ATIMERESP:    	return("Alto time response");
	    case NAMEREQ:	return("Name lookup request");
	    case NAMERESP:	return("Name lookup response");
	    case ADDREQ:	return("Address lookup request");
	    case ADDRESP:	return("Address lookup response");
	    case LOOKUPERR:	return("Lookup Error");
	    case MMAILREQ:    	return("Msg-style Mail Check Request");
	    case LMAILREQ:    	return("Laurel-style Mail Check Request");
	    case NEWMAIL:	return("Mail Check - New Mail Exists");
	    case NONEWMAIL:	return("Mail Check - No New Mail");
	    case NOSUCHMBOX:	return("Mail Check - No Such Mailbox");
	    case WHRUSREQ:    	return("Where is User request");
	    case WHRUSRESP:    	return("Where is User response");
	    case WHRUSERR:    	return("Where is User Error");
	    case NETDIRVERSION:	return("Net Directory Version");
	    case SENDNETDIR:	return("Send Net Directory");
	    case BOOTFILEREQ:  	return("Boot File Request");
	    case KISSOFDEATH:  	return("Kiss of Death");
	    case AUTHREQ:    	return("User authentication request");
	    case AUTHPOSRESP:  	return("User authentication positive response");
	    case AUTHNEGRESP:  	return("User authentication negative response");
	    case BOOTSTRQ:    	return("Boot Stats request");
	    case BOOTSTRP:    	return("Boot Stats reply");
	    case BOOTDREQ:    	return("Boot directory request");
	    case BOOTDREP:    	return("Boot directory reply");

	    case SENDUMSG:	return("Send User Message");
	    case SENDUACK:	return("Send User Message Ack");
	    case SENDUERR:    	return("Send User Message Error");
	    case SUNBOOTREQ:   	return("Sun Boot request");
	    case SUNBOOTDREQ:  	return("Sun Boot directory request");
	    case SUNBOOTDREP:  	return("Sun Boot directory reply");
	    
	    default:
	    	sprintf(numtype,"[unknown MiscServices type %o]",pt);
		return(numtype);
	    }
	}

	if ((ssock == GATEWAYINFO) ||(dsock == GATEWAYINFO)) 
	 {
	  switch (pt) 
	   {
	    case GWINFREQ:    	return("GateWay Information Request");
	    case GWINFREP:    	return("GateWay Information Reply");

	    default:
	    	sprintf(numtype,"[unknown GatewayInfo type %o]",pt);
		return(numtype);
	    }
	}
	
	if ((ssock == EARSSTATUS) ||(dsock == EARSSTATUS)) {
	
	 switch (pt) 
	  {

	    case EARSTATREQ:	return("Ears Status request");
	    case EARSTATREP:	return("Ears Status reply");
	    default:	    	sprintf(numtype,"[unknown Ears-Status type %o]",pt);
				return(numtype);
	  }
	}

	if ((ssock == LEAF) || (dsock == LEAF)) 
	 {
	  switch (pt) 
	   {
	    case SEQUIN:	return("Sequin");
	
	    default:	sprintf(numtype,"[unknown Leaf/Sequin type %o]",pt);
			return(numtype);
	    }
	 }
	
	sprintf(numtype,"[unknown unregistered type %o]",pt);
	return(numtype);

}

char *PrintLeafMode(m)
    unsigned short m;
  {
  	/*
	 * Format a Leaf open mode
	 *
	 */
    static char buf[256];

    strcpy(buf,"");
    if (m & LEAFOPENR) strcat(buf,"Read, ");
    if (m & LEAFOPENW) strcat(buf,"Write, ");
    if (m & LEAFOPENEX) strcat(buf,"Extend, ");
    if (m & LEAFOPENMULT) strcat(buf,"Multiple, ");
    if (m & LEAFOPENC) strcat(buf,"Create, ");
    if (m & LEAFOPENMW) strcat(buf,"Multiple Writers, ");
    switch ( (m & 03000) >> 9)
      {
        case 0: strcat(buf,"No Version, "); break;
	case 1: strcat(buf,"Old Version, "); break;
	case 2: strcat(buf,"Next or Old Version, "); break;
	case 3: strcat(buf,"Any Version, "); break;
      }
    switch ( (m & 0600) >> 7)
      {
        case 0: strcat(buf,"No Default"); break;
	case 1: strcat(buf,"Lowest Default"); break;
	case 2: strcat(buf,"Highest Default"); break;
	case 3: strcat(buf,"Next Default"); break;
      }
    return(buf);
  }


char *PrintLeafString(f,buf)
    FILE *f;
    register char *buf;
  {
  	/*
	 * Print out a leaf string (length plus bytes)
	 */
    int len, odd;
    
    len = *buf++ & 255;
    len = (len << 8) + (*buf++ & 255);
    odd = (len & 1);
    while (len--)
      {
        PrettyPutchar(*buf++,f);
      }
    if (odd) buf++;
    return(buf);
  }

PrintLeafOpen(f,buf)
    FILE *f;
    register char *buf;
  {
  	/*
	 * Print out a leaf open string set
	 */
    int len;
    
    fprintf(f,"User Name: ");
    buf = PrintLeafString(f,buf);
    
    len = *buf++ & 255;
    len = (len << 8) + (*buf++ & 255);
    fprintf(f," Password: (%d)",len);
    buf += roundup2(len);

    fprintf(f," Connect Name: ");
    buf = PrintLeafString(f,buf);

    len = *buf++ & 255;
    len = (len << 8) + (*buf++ & 255);
    fprintf(f," Connect Password: (%d)",len);
    buf += roundup2(len);

    fprintf(f," File Name: ");
    buf = PrintLeafString(f,buf);
  }


PrintSequin(f,id,length,buf)
    FILE *f;
    char *buf;
    union SequinID id;
  {
  	/*
	 * print out a sequin packet
	 */
     register union SequinData *lp;
     
     fprintf(f,"Sendseq=%d, Recvseq=%d, Allocate=%d, SequinOP=",
       id.codes.Sendseq, id.codes.Recvseq, id.codes.Allocate );
     switch (id.codes.Control)
       {
	  case SEQUINDATA:	fprintf(f,"Data");	break;
	  case SEQUINACK:	fprintf(f,"Ack");	break;
	  case SEQUINNOP:	fprintf(f,"Nop");	break;
	  case SEQUINRESTART:	fprintf(f,"Restart");	break;
	  case SEQUINCHECK:	fprintf(f,"Check");	break;
	  case SEQUINOPEN:	fprintf(f,"Open");	break;
	  case SEQUINBREAK:	fprintf(f,"Break");	break;
	  case SEQUINCLOSE:	fprintf(f,"Close");	break;
	  case SEQUINCLOSED:	fprintf(f,"Closed");	break;
	  case SEQUINDESTROY:	fprintf(f,"Destroy");	break;
	  case SEQUINDALLYING:	fprintf(f,"Dallying");	break;
	  case SEQUINQUIT:	fprintf(f,"Quit");	break;
	  case SEQUINBROKEN:	fprintf(f,"Broken");	break;
	  case SEQUINDEAD:	fprintf(f,"Dead");	break;
	  default:		fprintf(f,"Strange op %d",id.codes.Control);
       }
     fprintf(f,"\n");

     while (length>0)
      {
       int size;

       	lp = (union SequinData *)buf; 
 	size = (int) ( lp->Logical.LeafOp & 01777);
    	switch (lp->Logical.LeafOp & 076000)
         {
	  case LEAFERROR:	
	  	fprintf(f,"Leaf Error: ");
	  	PrintLeafError(f,lp->Logical.LeafData.Error.Subcode);
		break;

	  case LEAFOPEN:
		fprintf(f,"Leaf Open handle=%d, mode=%d (%s)\n", 
			lp->Logical.LeafData.Open.FileHandle,
			lp->Logical.LeafData.Open.LeafOpenMode,
			PrintLeafMode(lp->Logical.LeafData.Open.LeafOpenMode));
		PrintLeafOpen(f,lp->Logical.LeafData.Open.Openbytes);
	  	break;

	  case LEAFCLOSE:	
		fprintf(f,"Leaf Close, handle=%d", 
			lp->Logical.LeafData.FileHandle);
	  	break;

	  case LEAFDELETE:
		fprintf(f,"Leaf Delete, handle=%d", 
			lp->Logical.LeafData.FileHandle);
	  	break;

	  case LEAFREAD:	
		fprintf(f,"Leaf Read, handle=%d, addr=%d, length=%d", 
			lp->Logical.LeafData.Read.FileHandle,
			lp->Logical.LeafData.Read.Address,
			lp->Logical.LeafData.Read.ReadLength);
		break;

	  case LEAFWRITE:
		fprintf(f,"Leaf Write, handle=%d, addr=%d, length=%d", 
			lp->Logical.LeafData.Write.FileHandle,
			lp->Logical.LeafData.Write.Address,
			lp->Logical.LeafData.Write.WriteLength);
		break;

	  case LEAFPARAMS:	
	  	fprintf(f,"Leaf Params, max Pup Data length=%d, \n",
			lp->Logical.LeafData.Params.MaxPupDataLength);
	  	fprintf(f,"File Lock timeout=%d, Connection timeout=%d",
			lp->Logical.LeafData.Params.FileLockTimeout,
			lp->Logical.LeafData.Params.ConnectionTimeout);
	  	break;

	  case LEAFERRORANS:	
	  	fprintf(f,"Leaf Error Answer: ");
		PrintLeafError(f,lp->Logical.LeafData.Error.Subcode);
	  	break;

	  case LEAFOPENANS:	
	  	fprintf(f,"Leaf Open Answer, handle=%d, length=%d\n",
			lp->Logical.LeafData.OpenA.FileHandle,
			lp->Logical.LeafData.OpenA.Length);
		break;

	  case LEAFRESET:	fprintf(f,"Leaf Reset");		break;
	  case LEAFNOOP:	fprintf(f,"Leaf Noop");			break;
	  case LEAFLENGTH:	fprintf(f,"Leaf Length");		break;
	  case LEAFTRUNCATE:	fprintf(f,"Leaf Truncate");		break;
	  case LEAFCLOSEANS:	fprintf(f,"Leaf Close Answer");		break;
	  case LEAFDELETEANS:	fprintf(f,"Leaf Delete Answer");	break;
	  case LEAFLENGTHANS:	fprintf(f,"Leaf Length Answer");	break;
	  case LEAFTRUNCATEANS:	fprintf(f,"Leaf Truncate Answer");	break;
	  case LEAFREADANS:	fprintf(f,"Leaf Read Answer");		break;
	  case LEAFWRITEANS:	fprintf(f,"Leaf Write Answer");		break;
	  case LEAFRESETANS:	fprintf(f,"Leaf Reset Answer");		break;
	  case LEAFNOOPANS:	fprintf(f,"Leaf Noop Answer");		break;
	  case LEAFPARAMSANS:	fprintf(f,"Leaf Params Answer");	break;
         }
	fprintf(f," of %d bytes\n",size );
	buf += size;
        length -= size;
	if ( (lp->Logical.LeafOp & 01777)==0) break;
     }
  }

PrintLeafError(f,code)
    FILE *f;
    short code;
  {
    switch(code)
      {
    case IllegalLookupControl:	fprintf(f,"Illegal Lookup Control "); break;
    case Namemalformed:		fprintf(f,"Name Malformed "); break;
    case IllegalChar:		fprintf(f,"Illegal Char "); break;
    case IllegalStar:		fprintf(f,"Illegal Star "); break;
    case IllegalVersion:	fprintf(f,"Illegal Version "); break;
    case NameTooLong:		fprintf(f,"Name Too Long "); break;
    case IllegalDIFAccess:	fprintf(f,"Illegal DIFAccess "); break;
    case FileNotFound:		fprintf(f,"File Not Found "); break;
    case AccessDenied:		fprintf(f,"Access Denied "); break;
    case FileBusy:		fprintf(f,"File Busy "); break;
    case DirNotFound:		fprintf(f,"Directory Not Found "); break;
    case AllocExceeded:		fprintf(f,"Allocation Exceeded "); break;
    case FileSystemFull:	fprintf(f,"File System Full "); break;
    case CreateStreamFailed:	fprintf(f,"Create Stream Failed "); break;
    case FileAlreadyExists:	fprintf(f,"File Already Exists "); break;
    case FileUndeletable:	fprintf(f,"File Undeletable "); break;
    case Username:		fprintf(f,"Username "); break;
    case Userpassword:		fprintf(f,"User Password "); break;
    case FilesOnly:		fprintf(f,"Files Only "); break;
    case ConnectName:		fprintf(f,"Connect Name "); break;
    case ConnectPassword:	fprintf(f,"Connect Password "); break;
    case BrokenLeaf:		fprintf(f,"Broken Leaf "); break;
    case BuddingLeaf:		fprintf(f,"Budding Leaf "); break;
    case BadHandle:		fprintf(f,"Bad Handle "); break;
    case LeafFileTooLong:	fprintf(f,"Leaf File TooLong "); break;
    case IllegalLeafTruncate:	fprintf(f,"Illegal Leaf Truncate "); break;
    case AllocLeafVMem:		fprintf(f,"Alloc Leaf VMem "); break;
    case IllegalLeafRead:	fprintf(f,"Illegal Leaf Read "); break;
    case IllegalLeafWrite:	fprintf(f,"Illegal Leaf Write "); break;
    default:		
    		fprintf(f,"subcode %d", code);
    }
  }

PrintErrorPUP(out, buf, length )
    FILE *out;
    struct PupPacket *buf;
    int length;
  {
  	/*
	 * decode an error PUP
	 */
    struct Port src, dst;
    int badLength, count;
    char *PupTypeName();
    
    PortUnpack( &buf->PupSrc, &src);
    PortUnpack( &buf->PupDst, &dst);
    badLength = buf->PupLength - PUPACKOVER;

    fprintf( out, "Erroneous PUP was of Type %s Length %d (%d Data)\n", 
    	PupTypeName(buf->PupType, dst.socket, src.socket),
	buf->PupLength, badLength);
    
    fprintf( out, "From "); PortPrint( &src, out );
    fprintf( out, " to ");  PortPrint( &dst, out );
    fprintf( out, " ID 0%o", buf->PupID );

    length -= PUPACKOVER - 2;
    fprintf( out, " Error codes %o,%o\n", 
    		*(short *)buf->PupData, *(short *)(buf->PupData+2) );
    fprintf( out, "Error string: ");
    for (count=4; count<length && count<64; count++)
	  	    PrettyPutchar( buf->PupData[count], out);
    fprintf( out, "\n" );
  }


PrintGatewayInfo(out, buf, length )
    FILE 	*out;
    unsigned char *buf;
    int		length;
  {
  	/*
	 * decode a routing packet
	 */
    while (length>0)
      {
	fprintf(out," len->%d ", length);
        fprintf(out,"Target net: 0%o ", buf[0]);
	if (buf[1]==0 && buf[2]==0 && buf[3]==0)
	  fprintf(out,"Directly connected\n");
	else
          fprintf(out,"Gateway net: 0%o, Host: 0%o, Hops: %d\n",
	    buf[1], buf[2], buf[3]);
        length -= 4;
	buf += 4;
      }
  }
