/*
 * V Kernel - Copyright (c) 1984 by David Cheriton, Tim Mann
 *
 * Header file describing Sun-3 memory mapping and onboard device
 *   addresses.  See the Sun-3 Architecture Manual (if you can get one)
 *   for details.  Some of the definitions may only be valid for "machine
 *   type 1", the Multibus-based Sun-2 implementation.
 *
 * This file uses current SMI terminology, which differs from older V
 *   terminology.  In particular, PMEG = "page map entry group" is
 *   equivalent to the old term "physical segment".
 */

#ifndef MEMORY
#define MEMORY

/* Where the interrupt vectors live (as set by the boot prom) */
#define VECTOR_BASE 0x0FE60C00

typedef unsigned long	Bit32;
typedef unsigned short	Bit16;
typedef unsigned char	Bit8;
typedef long		sBit32;
typedef short		sBit16;
typedef char		sBit8;

/* Routines to read and write function-code-3 space, exported by sun2mem.c */
extern Bit8 Fc3ReadBit8(/*addr*/);		/* read an 8-bit quantity */
extern Bit16 Fc3ReadBit16(/*addr*/);		/* read a 16-bit quantity */
extern Bit32 Fc3ReadBit32(/*addr*/);		/* read a 32-bit quantity */
extern void Fc3WriteBit8(/*addr, value*/);	/* write an 8-bit quantity */
extern void Fc3WriteBit16(/*addr, value*/);	/* write a 16-bit quantity */
extern void Fc3WriteBit32(/*addr, value*/);	/* write a 32-bit quantity */

/* CPU Layer (Architecure manual, chapter 4) */
/* Addresses and formats for various things in "function code 3" space.
 *  These are accessed using the MOVS instruction with the
 *  value 3 in the function code registers */

#define SPACE(space)(space<<28)

/* - Identification PROM (read only)*/
#define IDP_FORMAT		0	/* format byte */
#define IDP_MACHINE_TYPE	1	/* machine type byte */
#define IDP_ENET_ADDRESS	2	/* start of Enet address (6 bytes) */
#define IDP_DATE		8	/* start of manuf. date (4 bytes) */
#define IDP_SERIAL_NUMBER	12	/* start of serial number (3 bytes) */
#define IDP_CHECKSUM		15	/* checksum (1 byte) */
#define IDP_RESERVED		16	/* reserved field (16 bytes) */

/* - Diagnostic LED register (1 byte, write only) */
#define DIAGNOSTIC_REGISTER	SPACE(0x7)

/* - Bus error register (16 bits, read only) */
#define BUS_ERROR_REGISTER	SPACE(0x6)
typedef union
  {
    struct
      {
	Bit8 invalid:1;		/* Access to an invalid address */
	Bit8 protection:1;	/* Protection violation */
	Bit8 timeout:1;		/* Bus timeout (on and off board) */
	Bit8 VMEbus:1;		/* VMEbus access error */
	Bit8 FPAbus:1;		/* FPA error (FPA doesn't interrupt) */
	Bit8 FPAenable:1;	/* Access to FPA when en.fpa = 0 */
	Bit8 unused:1;
        Bit8 watchdog:1;	/* Watchdog or User reset (optional) */
      }
    f;
    Bit8 u;			/* Whole thing as an unsigned quantity */
  }
BusErrorRegister;

/* System Enable register (8 bits, read/write) */
#define SYSTEM_ENABLE_REGISTER	SPACE(0x4)
typedef struct
  {
    Bit8 enableBoot:1;		/* 0 = boot ; 1 = normal */
    Bit8 enableFPP:1;		/* enable floating point coprocessor */
    Bit8 enableDVMA:1;		/* enable VME DVMA */
    Bit8 enableCache:1;		/* enable external cache */
    Bit8 enabelVideo:1;		/* enable video output to monitor */
    Bit8 enableCopy:1;		/* enable copy update mode to monitor */
    Bit8 enableFPA:1;		/* enable floating point accelerator */
    Bit8 diag:1;		/* State of diagnostic switch */
  }
SystemEnableRegister;

/* - MMU entries for virtual address v in the current address space */
#define PageMapAddr(v)		(SPACE(0x1)|(((Bit32)(v))&0x0fffe000))
#define SegMapAddr(v)		(SPACE(0x2)|(((Bit32)(v))&0x0ffe0000))
#define CONTEXT_REGISTER	SPACE(0x3)

/* MMU hardware characteristics (manual, chapter 5) */

/* Size of a page, a segment, and a context (address space) */
#define PAGE_SIZE	0x2000		/*   8 Kbytes */
#define SEG_SIZE	0x20000		/* 128 Kbytes */
#define CONTEXT_SIZE	0x10000000	/* 256 Mbytes */

#define NUM_CONTEXTS	8		/* Total number of contexts */
#define	NUM_PMEGS	256		/* Number of PMEGs available */

#define SEGS_PER_CTX	(CONTEXT_SIZE/SEG_SIZE)	/*  */
#define PGS_PER_SEG	(SEG_SIZE/PAGE_SIZE)	/*  */

/* - Structure of the context register */
typedef union
  {
    /* Fields */
    struct
      {
        Bit8 supervisor;	/* Supervisor state context number */
        Bit8 user;		/* User state context number */
      }
    f;
    Bit16 u;	/* Whole thing as an unsigned quantity */
  }
ContextRegister;

#define MAP_NOT_RESIDENT (-1)	/* an invalid context register value */

/* - Segment map entry */
typedef Bit8 SegMapEntry;

/* - Page map entry */
typedef union
  {
    /* Fields */
    struct
      {
        Bit32 valid:1;		/* 1 = Valid page */
	Bit32 protection:3;	/* write, system, dont_cache */
	Bit32 type:2;		/* Page type; see below */
	Bit32 accessed:1;	/* Page has been accessed */
	Bit32 modified:1;	/* Page has been modified */
	Bit32 reserved:5;
	Bit32 pagenum:19;	/* Page number (physical address) */
      }
    f;
    Bit32 u;	/* Whole thing as an unsigned integer */
  }
PageMapEntry;
	
/* Protection bits */
#define SYSTEM_ACCESS		0x4
#define PAGE_WRITE_ACCESS	0x2
#define	DONT_CACHE		0x1

/* Page types */
#define ONBOARD_MEM	0	/* "Onboard" (P2 bus) */
#define ONBOARD_IO	1	/* Onboard devices */
#define VME_16		2	/* VMEbus 16 bit data space */
#define VME_32		3	/* VMEbus 32 bit data space */

/* - Structure of a virtual address */
typedef union
  {
    /* Fields */
    struct
      {
	Bit32	unused:4;
	Bit32	segment:11;	/* Segment number */
	Bit32	page:4;		/* Page number within segment */
	Bit32	byte:13;	/* Byte number */
      }
    f;
    char *p;		/* Whole thing as a character pointer */
    Bit32 u;		/* Whole thing as an unsigned 32-bit quantity */
  }
VirtualAddress;


/* Extract the page number (for the pagenum field of a page table entry)
 *   from a physical address */
#define ExtractPageNumber(p) ((p)>>13)	/* effectively same as p%PAGE_SIZE */
/* Extract the segment number from an address */
#define ExtractSegNumber(p) ((p)>>17)	/* effectively same as p%SEG_SIZE */

/*
 * Physical addresses for things in onboard memory space
 *   These are addressed through page map entries of type ONBOARD_MEM.
 */
#define P_ONBOARD_RAM		0
#define P_ONBOARD_RAM_LIMIT	0x800000	/* No more than 8 Mbytes (?)*/
#define P_BW_FRAMEBUFFER	0xFF000000	/* "new" frame buffer */

/* Addresses for things in onboard mappable device space
 *   These are addressed through page map entries of type ONBOARD_IO.
 */
#define P_KBD_MOUSE_UART	0x00000000
#define P_SERIAL_PORT		0x00020000
#define P_EEPROM		0x00040000
#define P_REAL_TIME_CLOCK	0x00060000
#define P_MEMORY_ERROR_REGS	0x00080000
#define P_INTERRUPT_REG		0x000A0000
#define P_INTEL_ETHERNET	0x000C0000
#define P_COLOR_MAP		0x000E0000
#define P_EPROM			0x00100000
#define P_AMD_ETHERNET		0x00120000
#define P_SCSI			0x00140000
#define P_IOX			0x001A0000
#define P_ENCRYPTION_CHIP	0x001C0000

/* Conventional virtual addresses.
 */
#define KERNEL_START	0x2000		/* Kernel is loaded at this address */
#define KERNEL_LIMIT	0x20000		/* end of kernel */
#define TEAM_START 	KERNEL_LIMIT	/* Start of team space */
#define TEAM_LIMIT	0x0FE00000	/* Team size limit */
#define XFERSEG		0x0FF80000	/* Used by kernel for moving things */
#define PD_SEG		0x0FFA0000	/* 128k for process descriptors */
#define PROFILESEG	0x0FFC0000	/* 128k for profile data */
#define ENET_DATA_SEG   0x0FFE0000	/* 128k for buffer/data space*/
#define TESTPAGE	PROFILESEG

#define V_SUN3_FRAMEBUFFER	0x0FE20000
#define SUN3_FRAMEBUFFER_SIZE	0x020000
#define V_KBD_MOUSE_UART	0x0FE00000
#define V_SERIAL_PORT		0x0FE02000
#define V_EEPROM		0x0FE04000
#define V_REAL_TIME_CLOCK	0x0FE06000
#define V_MEMORY_ERROR_REGS	0x0FE08000
#define V_INTERRUPT_REG		0x0FE0A000
#define V_INTEL_ETHERNET	0x0FE0C0000
#define V_COLOR_MAP		0x0FE0E0000
#define V_AMD_ETHERNET		0x0FE120000
#define V_SCSI			0x0FE140000
#define V_IOX			0x0FE1A0000
#define V_ENCRYPTION_CHIP	0x0FE1C0000
#define V_EPROM			0x0FEF00000

#define V_DVMA_START	0xF00000	/* Virtual address of DVMA space */
#define V_DVMA_LIMIT	0xF40000

/* end conventional addresses */


/* Page map entry used to indicate an unmapped page.  Note that the 
 * valid bit is not set!
 */
#define NULL_PAGE 0

/* Segment map entry used to indicate an unused segment.  This pmeg
 *   is filled with NULL_PAGE page entries.
 */
#define NULL_PMEG (NUM_PMEGS-1)

/* Round an address up to the nearest page boundary */
#define uptopage(addr)\
  ( ((((Bit32) (addr)) - 1) & (~(PAGE_SIZE-1))) + PAGE_SIZE )
/* Round an address up to the nearest segment boundary */
#define uptoseg(addr)\
  ( ((((Bit32) (addr)) - 1) & (~(SEG_SIZE-1))) + SEG_SIZE )
/* Round an address down to the nearest page boundary */
#define downtopage(addr)\
  ( ((Bit32) (addr)) & ~(PAGE_SIZE-1) )
/* Round an address down to the nearest segment boundary */
#define downtoseg(addr)\
  ( ((Bit32) (addr)) & ~(SEG_SIZE-1) )

/* Read and write MMU registers */
#define SetPageMap(addr, value)\
  Fc3WriteBit32(PageMapAddr(addr), (Bit32)(value))
#define SetSegMap(addr, value)\
  Fc3WriteBit8(SegMapAddr(addr), (Bit8)(value))
#define GetPageMap(addr)\
  Fc3ReadBit32(PageMapAddr(addr))
#define GetSegMap(addr)\
  Fc3ReadBit8(SegMapAddr(addr))
#define SetContextNum(num)\
  Fc3WriteBit16(CONTEXT_REGISTER, (Bit16)(num))
#define SetContextReg(cxt)\
  Fc3WriteBit16(CONTEXT_REGISTER, (Bit16)(cxt).u)
#define GetContextReg(cxt)\
  (cxt)->u = Fc3ReadBit16(CONTEXT_REGISTER)
#define GetBusErrorReg()\
  Fc3ReadBit16(BUS_ERROR_REGISTER)

/* This kernel normally sets both contexts the same */
#define SetBothContexts(ctx)\
  Fc3WriteBit16( CONTEXT_REGISTER, (((Bit8)(ctx))<<8) | ((Bit8)(ctx)) )
/* ...except in a few cases during initialization */
#define SetSuperContext(ctx)\
  Fc3WriteBit8(CONTEXT_REGISTER, (Bit8)(ctx))
#define SetUserContext(ctx)\
  Fc3WriteBit8(CONTEXT_REGISTER+1, (Bit8)(ctx))

/* Modes for Probe() function -- see memory.c */
#define PROBE_READ 1
#define PROBE_INIT 2
#define PROBE_PRESERVE 3

#endif
