/*
 * enwatch.c - record data on Internet Datagrams
 *
 * Bill Nowicki September 1982
 *
 * Copyright (c) 1983 Stanford University
 *
 * This source code can be copied ONLY if all changes and improvements
 * are promptly sent back to the Author.
 *
 *
 * October 1983
 *	- Split out device-dependent parts (this is 3Mbit version)
 *	- Added XNS protocol
 *
 * See RCS log for history.
 *
 */

# define ENET3MEG

/*
 * Luckily, the PUP and XNS packet types are the same on both 10 and 3Mbit 
 * Ethernet.  Some of the others are different, so watch out!
 */

# include <Venviron.h>
# include <Vio.h>
# include "ip.h"
# include "Vikc.h"
# include "xns.h"
# include "chaos.h"

# include <m68enet.h>
# include <pup/puplib.h>
# include <pup/pupconstants.h>
# include <pup/puppacket.h>

# ifndef PUP
# define PUP ETHERNET_TYPE_PUP
# endif PUP

#define BUFFERS 64	/* number of Network buffers */
#define SIZE 1600	/* size in bytes of each buffer */
int MaxBuffers = BUFFERS;
int BufferSize = SIZE;

char Buffers[BUFFERS][SIZE];

int FirstBuffer = 0;		/* index of firrst buffer */
int LastBuffer = 0;		/* index of next usable buffer */

int TickFlag = 1;		/* print exclamation on each packet */
int PupFlag = 0;		/* listen to PUPs */
int IPFlag = 0;			/* listen to IPs */
int VFlag = 1;			/* listen to V kernel packets */
int XNSFlag = 0;		/* listen to XNS packets */
int CHAOSFlag = 0;		/* listen to ChaosNet packets */
int OtherFlag = 0;		/* listen to other random types */
int OrFlag = 0;			/* OR source and dest filters instead of AND */
int PrintData = 0;		/* do not print data in packets */
int StartTime = 0;		/* K_ticks when we started */
int HitFlag;			/* a key was hit */
int Annoying = 0;		/* Always print option list */

unsigned char SourceFilter[256];/* software source filter */
unsigned char DestFilter[256];	/* hardware destination filter */

char userName[64], cwd[128];

short *myAddress;

EtherInit( addr )
short *addr;
  {
  }

char *Buffer(i)
  {
    return(Buffers[i]);
  }

EnetSetFilter()
  {
  	/*
	 * Set the Ethernet hardware receive filter
	 * to be desired watch settings.
	 * First turn them all off, then turn requested ones on.
	 */
    register short address;
    
    EIStatus = Init1+FilterData0+Loopback0;
    for (address=0; address<256; address++)
      EIAddress = address;
    EIStatus = Init1+FilterData1+Loopback0;
    for (address=0; address<256; address++)
      if (DestFilter[address] || OrFlag) EIAddress = address;
    EIStatus = Init0+Loopback0;
  }


EnetNormalFilter()
  {
  	/*
	 * reset the Ethernet hardware receive filter
	 * to be normal (broadcasts plus my host)
	 */
    register short address;
    
    EIStatus = Init1+FilterData0+Loopback0;
    for (address=0; address<256; address++)
      EIAddress = address;
    address = EIAddress;
    EIStatus = Init1+FilterData1+Loopback0;
    EIAddress = address;
    EIAddress = 0;
    EIStatus = Init0+Loopback0;
  }
  


ReadPacket()
  {
    /*
     * Read a packet into the next buffer.
     * returns true if we one matched the filter.
     */
     
    register unsigned short *buf = (unsigned short *)Buffers[LastBuffer];
    register short count;
    register unsigned short *end = (unsigned short *)(Buffers[LastBuffer+1]);
    register short *eid0 = &EIData0;
    short blackHole;

    *(int *)buf = K_ticks() - StartTime;
    buf += 2;
    while ( (count = *eid0) & QueueEmptyBit)
      if (KeyHit()) return(0);
    *buf++ = count;
    count &= CountMask;
    if (count==0) return(0);

    do  
       { *buf++ = *eid0; }
    while (--count>0 && buf<end);

    while (count-->0)
      blackHole = *eid0;
    buf = (unsigned short *)Buffers[LastBuffer];
    buf += 4;
      /*
       * now we have read the packet, throw away Ethernet types
       * that we do not want.
       */
    switch (*buf)
      {
        case PUP:	if (!PupFlag)	return(0);	break;
		/*
		 * Slight violation of layering here:
		 * if Printdata is off, ignore ICMPs.
		 * I hate to do this, but otherwise they really
		 * clutter up the logs.
		 */
		/* (Violation commented out by S. Deering March 1986) */

        case IP:	if (!IPFlag /* || ( (buf[5] & 255)==1 && !PrintData) */)
					return(0);	break;
        case XNS:	if (!XNSFlag)	return(0);	break;
	case CHAOS3MB:	if (!CHAOSFlag)	return(0);	break;
        case NSToPup:	if (!XNSFlag)	return(0);	break;
	case V_KERNEL_PACKET:
	case XV_KERNEL_PACKET:
			if (!VFlag)	return(0);	break;
	default:	if (!OtherFlag)	return(0);
      }
      	/*
	 * check that the sending host is in our software filter 
	 */
    buf--;
    if (SourceFilter[ (*buf & 0377) ]==0 && 
         ( !OrFlag || DestFilter[ (*buf >> 8) & 0377 ]==0) ) return(0);
    LastBuffer++;
    if (LastBuffer>=MaxBuffers) LastBuffer %= MaxBuffers;
    if (LastBuffer==FirstBuffer) 
      {
        FirstBuffer++;
	FirstBuffer %= MaxBuffers;
      }
    return(1);
  }

EtherReady() { return( EIStatus & RxRcvd ); }

PrintBuffer( buf, out )
 register unsigned short *buf;
  {
    /*
     * Decode and print the buffer indicated on the
     * given file.
     */
    short rxStatus;
    unsigned char source, dest;
    int count;    
    
    fprintf( out, "\n%d.%02ds ",
    	(*(int *)buf)/1000, ( (*(int *)buf)/10)%100 );
    buf += 2;
    rxStatus = *buf++;
    count = rxStatus & CountMask;    
    source = *buf & 0377;
    dest = *buf >>8;
    fprintf( out, "[0%o --> 0%o]  ", source, dest );
    buf++;

    if ( (rxStatus & CRCerrorBit) || (rxStatus & OverflowBit) ||
         (*buf != PUP && *buf != IP && *buf != XNS && *buf != PupToNS &&
	  *buf != V_KERNEL_PACKET && *buf != XV_KERNEL_PACKET &&
	  *buf != CHAOS3MB) )
      {
	int i = 0;

    	fprintf( out, "Ethernet Status: ");
    	if (rxStatus & CRCerrorBit) fprintf( out, "CRC Error, ");
    	if (rxStatus & OverflowBit) fprintf( out, "Overflow, ");
    	fprintf( out, "%d words, ", count );
	if (!PrintData && count > 16) count = 16;
	fprintf( out, " %04x", buf[-1] & 0xFFFF );
	while (++i < count)
	    {
	      if (i % 16 == 0) fprintf( out, "\n");
	      fprintf( out, " %04x", *buf++ & 0xFFFF );
	    }
	fprintf( out, "\n");
	return;
      }

      	/*
	 * Subtract off the Ethernet encapsulation overhead:
	 * 	One word for source and Destination.
	 * 	One word for Ethernet packet type.
	 * 	One word for Ethernet CRC
	 * Then convert from words to bytes.
	 */
    count -= 3;
    count *= 2;
    if (*buf == PUP) 			PupPrint( ++buf, out, count);
    if (*buf == XNS) 			XNSPrint( ++buf, out, count);
    if (*buf == CHAOS3MB)		CHAOSPrint( ++buf, out, count);
    if (*buf == NSToPup) 		NSToPupPrint( ++buf, out, count);
    if (*buf == IP)  			IPPrint(  ++buf, out, count);
    if (*buf == V_KERNEL_PACKET)  	VPrint(  ++buf, out, count);
    if (*buf == XV_KERNEL_PACKET)	xVPrint(  ++buf, out, count);
   }

PrintSpecifications()
    {
    if (OrFlag)
	printf("c - Conjuctive mode - Currently OR (Disjuctive mode)\n");
      else
	printf("o - OR (Disjuctive) mode - Currently AND (Conjuctive mode)\n");
    printf("e - Look at Everybody\n");
    printf("2 - Look at two hosts\n");
    printf("n - Look at nobody\n");
    printf("a - Add host to filter\n");
    printf("d - Delete host from filter\n");
    printf("p - Print out filters\n");
    printf("Return or space goes back to main menu\n");
    }

HostSpecify()
  {
	/*
	 * specify hosts to watch
	 */
    register short host;
    int inputHost;

    while (1)
      {
	if (Annoying)
	  {
	    PrintSpecifications();
	    printf("Enter suboption: ");
	  }
	else
	    printf("Host suboption (? for list): ");
	switch(getchar())
	  {
	    case '?':
		printf("\n");
		PrintSpecifications();
		break;

	    case 'e':
	        printf("\nSelecting EVERYbody\n");
    		for (host=0; host<256; host++)
		  {
        	    SourceFilter[host] = 1;
		    DestFilter[host] = 1;
		  }
		break;
		
	    case 'n':
	        printf("\nSelecting Nobody\n");
    		for (host=0; host<256; host++)
		  {
        	    SourceFilter[host] = 0;
		    DestFilter[host] = 0;
		  }
		break;
		
		
	    case 'a':
	        printf("\nAdd host (octal):");
		scanf( "%o", &inputHost );
        	SourceFilter[inputHost] = 1;
		DestFilter[inputHost] = 1;
		break;		
		

	    case 'o':
	        printf("\nOR mode, or disjuctive,  source OR dest\n");
		OrFlag = 1;
		break;
		

	    case 'c':
	        printf("\nAND mode, or conjuctive, source AND dest\n");
		OrFlag = 0;
		break;

	    case 'd':
	        printf("\nDelete host (octal):");
	        fflush(stdin);
		scanf( "%o", &inputHost );
        	SourceFilter[inputHost] = 0;
		DestFilter[inputHost] = 0;
		break;		
		
	    case '2':
	        printf("\nSelecting Two hosts (and nobody else!)\n");
    		for (host=0; host<256; host++)
		  {
        	    SourceFilter[host] = 0;
		    DestFilter[host] = 0;
		  }
	        printf("First host (octal):");
		scanf( "%o", &inputHost );
        	SourceFilter[inputHost] = 1;
		DestFilter[inputHost] = 1;
	        printf("Second host (octal):");
		scanf( "%o", &inputHost );
        	SourceFilter[inputHost] = 1;
		DestFilter[inputHost] = 1;
		printf("\n");
		break;		
		
	    case 'p':
	        printf("\nReceiving packets from: ");
    		for (host=0; host<256; host++)
        	    if (SourceFilter[host]) printf("0%o ", host);
		printf("\n %s to: ", OrFlag ? "OR" : "AND");
    		for (host=0; host<256; host++)
        	    if (DestFilter[host]) printf("0%o ", host);
		printf("\n");
		break;
	    
	    case ' ':
	    case '\n':
	    case '\r':
	        printf( "\nBack to main menu\n");
	    	return;

    
	    default:
	        printf("\nUnknown option!\n");
		break;
	  }
      }
    
  }

#define FLAG( state, flag, meaning ) \
printf("  %s %c %s\n", (state) ? " ON" : "OFF", flag, meaning);

PrintOptions()
  {
    short address = EIAddress;
    printf("\n3Mb Netwatch [ local address 0%o, %d ]\n", address, address);
    FLAG(CHAOSFlag,	'k', "Chaos packets");
    FLAG(IPFlag,	'i', "IP packets");
    FLAG(PupFlag,	'p', "Pup packets");
    FLAG(VFlag,		'v', "V kernel packets");
    FLAG(XNSFlag,	'x', "XNS packets");
    FLAG(OtherFlag,	'o', "Other packets");
    FLAG(PrintData,	'd', "Print all packet data");
    FLAG(TickFlag,	'!', "Report ! when packets are received");
    FLAG(Annoying,	'm', "Print annoying option menus all the time");
    printf("      t Type incoming packets (^S to pause)\n");
    printf("      r Record incoming packets to buffer\n");
    printf("      b Display buffer (^S to pause)\n");
    printf("      s Display buffer, skipping initial packets\n");
    printf("      h Modify host filter\n");
    printf("      w Write buffer to a file\n");
    printf("      l Login to network\n");
    printf("      c Change directory\n");
    printf("      q Quit\n");
  }

#define TOGGLE(flag,string) \
flag = !flag; \
printf("\n%s %s packets\n", ((flag) ? "Print" : "Ignore"), string);

main()
  {
    register short host;
    int j;
    extern int sp;		/* stack pointer */

    sp = K_getmemsize()-4;

    setecho(0);
    NoEcho(); Raw();			/* no input buffering */
    
    strcpy(userName, "unknown");
    cwd[0] = 0;
    for (host=0; host<256; host++)
      {
        SourceFilter[host] = 1;
	DestFilter[host] = 1;
      }

    putchar('\f');
    PrintOptions();

    while (1)
      {
 	if (Annoying)
	  {
	    PrintOptions();
	    printf("Enter option: ");
	  }
	else printf("Option (? for list): ");
	switch ( getchar() )
	 {
	  case 'k': TOGGLE(CHAOSFlag, "Chaos"); break;
	  case 'i': TOGGLE(IPFlag, "IP"); break;
	  case 'p': TOGGLE(PupFlag, "PUP"); break;
	  case 'v': TOGGLE(VFlag, "V kernel"); break;
	  case 'x': TOGGLE(XNSFlag, "XNS"); break;
	  case 'o': TOGGLE(OtherFlag, "other"); break;
	    
	  case 'd': PrintData = !PrintData; break;
	  case '!': TickFlag = !TickFlag; break;
	  case 'm': Annoying = !Annoying; printf("\n"); break;

	  case 't': netwatch(1); break;
	  case 'r': netwatch(0); break;
	  case 'b': showbuffer(0); break;
	  case 's': showbuffer(1); break;

	  case 'h': printf("\n"); HostSpecify(); break;

	  case 'l': login(); break;
	  case 'c': cd(); break;
	  case 'w': WriteFile(); break;

	  case 'q': printf("\nQuit the program\n"); return;
	  case '?': PrintOptions(); break;

	  case ' ': case '\n': case '\r': break;

	  default: printf( "\nIllegal option!\n"); continue;
	 }
      }
    
  }

cd()
  {
    char dir[128];

    Cooked(); Echo();
    printf("\nDirectory (user: %s cwd: %s): ", userName, cwd[0]?cwd:"<none>");
    gets(dir);
    if (dir[0])
	if (dir[0] != '[')
	    printf("Directories must start with a '['.\n");
	else
	    strcpy(cwd, dir);
    NoEcho(); Raw();
  }


PrettyPutchar (ch, fp)
register unsigned char ch;
FILE *fp;
  {
    fprintf (fp, ((ch>=0x20) && (ch<=0x7E)) ? "%c" : "`%02x", ch);
  }
