/************************************************/
/*						*/
/*	       (C)  COPYRIGHT 1983		*/
/*		BOARD OF TRUSTEES		*/
/*	LELAND STANFORD JUNIOR UNIVERSITY	*/
/*	  STANFORD, CA. 94305, U. S. A.		*/
/*						*/
/************************************************/
 
/*
 *  spline.c
 *
 *  This file contains the definitions and code necessary to generate
 *  arbitrary spline curves of order 2 though 5.  The actual blending
 *  functions are precomputed (see splinegen.c) and evaluated, so that
 *  floating point computation can be avoided as much as possible at
 *  run time.  Note that data points are control points, not true knots.
 *  Actually values of the blending function are kept in XYsplinedefs.c.
 *  The only user-callable routines are XYspline() and XYfillSpline();
 *  NB:  If you update XYspline, update XYfillSpline too!!
 *
 *  Update History:
 *	March 1983 -- Written by David Kaelbling.
 */
 
#include <Vioprotocol.h>
#include "Vgts.h"
#include "sdf.h"
#include "vgt.h"
#include "splines.h"
  
  
/* Imports */
extern int XYbrush();
extern int XYline();
extern int XYfillInit();
extern int XYfillLine();
extern int XYfillScreen();
extern short XYsplineRes;
extern unsigned short spline2[];
extern unsigned short spline3[];
extern unsigned short spline4[];
extern unsigned short spline5[];
    
 
/* Exports */
extern int XYspline();
extern int XYfillSpline();
    
    
/* Local Definitions */
static POINT knotvec[5];	/* knot vector cache buffer. */
static short scan;		/* scan index in knot array. */

/*
 *  InitKnots will initialize the knot vector and the scan variable.
 *  Scan is used to determine which knot to move next into the cache
 *  (knotvec), based upon the end conditions of the spline.
 *
 *  In addition, the gauge variable is set to the reflect the resolution
 *  of this particular spline.  It is the number of subintervals * order
 *  to skip during evaluation.
 */
 
static InitKnots( closed, order, numVert, vertex, view, gauge, dx, dy )
	BOOLEAN closed;
	unsigned short order;
	short numVert;
	POINT *vertex;
	View *view;
	short *gauge;
	register short dx, dy;
  {
    register short i;
    
    
    /* For a closed spline, simply include the first 'order' knots. */
    /* For an open spline, duplicate the first data point 'order' times. */
    if (closed)
      {
        for (i = 0;  i < order;  i++)
          {
	    scan = ((numVert<<1) - order + i) % numVert;
            knotvec[i].x = XcvtToScreen( vertex[scan].x + dx, view );
	    knotvec[i].y = YcvtToScreen( vertex[scan].y + dy, view );
          }
      }
    else
      {
        for (i = 0;  i < order;  i++)
          {
            knotvec[i].x = XcvtToScreen( vertex[0].x + dx, view );
	    knotvec[i].y = YcvtToScreen( vertex[0].y + dy, view );
          }
      }
    scan = 0;
    
    /* Set the gauge of the spline */
    i = (view->zoom + 8) >> 1;
    i = max(i, 2);
    i = min(i, XYsplineRes);
    *gauge = ((order << XYsplineRes) >> i) - order;
  }

/*
 *  NewKnots will shift the knot vector to the left by one, and
 *  insert a new data point in at the end.  If closed end conditions
 *  are used, the vertex points are simply cycled.  Otherwise, the
 *  last end point is duplicated 'order' (actually as often as necessary)
 *  times.
 */
 
static NewKnots( numVert, vertex, knotNum, closed, order, view, dx, dy )
	short numVert;
	POINT *vertex;
	int knotNum;
	BOOLEAN closed;
	unsigned short order;
	View *view;
	register short dx, dy;
  {
    register short i;
    
    /* Shift existing data points */
    for (i = 1;  i < order;  i++)
      {
        knotvec[i-1].x = knotvec[i].x;
	knotvec[i-1].y = knotvec[i].y;
      }
    
    /* Insert the new point (if open and at end, don't advance scan) */
    knotvec[order-1].x = XcvtToScreen( vertex[scan].x + dx, view );
    knotvec[order-1].y = YcvtToScreen( vertex[scan].y + dy, view );
    if ((knotNum > (order - 2)) || (closed))
      {
	if (++scan >= numVert)
	    scan -= numVert;
      }
  }

/*
 *  XYspline()
 *       
 *  Generate an open or closed spline curve of order 2, 3, 4, or 5.
 *  NB:  If you find a bug here, you've found one in XYfillSpline too!!
 */
 
XYspline( numpoints, vertex, order, closed, clip, nib, view, dx, dy )
	short numpoints;
	POINT *vertex;
	short order;
	BOOLEAN closed;
	struct SubView *clip;
	enum Nib nib;
	View *view;
	register short dx, dy;
  {
    unsigned int x, y;		/* Working end-points. */
    short newx, newy, oldx, oldy;	/* End-points of segment. */
    int knotNum, v;		/* Used to step through the knots */
    short knot;			/* Stepped for each knot in the support */
    int weight;			/* Current weighting factor */
    unsigned short *Blend;	/* Weighting function */
    short gauge;		/* Gauge of the spline */
    
    if (numpoints <= 0)
      {
        printf("XYSPLINE:  Illegal number of points %d\n", numpoints);
        return( ERROR );
      }
      
    switch (order)
      {
	case 2:  Blend = spline2;  break;
        case 3:  Blend = spline3;  break;
        case 4:  Blend = spline4;  break;
        case 5:  Blend = spline5;  break;
        default: printf("XYSPLINE:  Illegal order %d.\n", order);
		 return( ERROR );
      }
      
    /* Initialize the knot cache */
    if (InitKnots( closed, order, numpoints, vertex, view, &gauge, dx, dy )
		== ERROR)
        return( ERROR );
    
    /* If an open spline, first interval is trivial */
    if (!closed)
      {
	knotNum = numpoints + order + (order == 2) - 4;
        oldx = knotvec[0].x;
	oldy = knotvec[0].y;
	NewKnots( numpoints, vertex, knotNum, closed, order, view, dx, dy );
	NewKnots( numpoints, vertex, knotNum, closed, order, view, dx, dy );
      }
    else
      {
        knotNum = numpoints;
	oldx = oldy = -1;
      }
    
    /* Iterate over the range of interesting knots */
    while (knotNum-- > !closed)
      {
	/* Get knots, and compute on subintervals */
	NewKnots( numpoints, vertex, knotNum, closed, order, view, dx, dy );
	v = 0;
	while (v <= (order << XYsplineRes))
	  {	
	    /* Combine the weighted knot values */
	    x = y = 1<<15;
	    knot = order;
	    while (knot--)
	      {
	        weight = Blend[v++];
	        x += weight * knotvec[knot].x;
	        y += weight * knotvec[knot].y;
	      }
	    v += gauge;
	
	    /* If a new result, do something */
	    newx = x >> 16;  newy = y >> 16;
	    if ((newx != oldx) || (newy != oldy))
	      {
                if (oldy != -1)
	            XYline( oldx, oldy, newx, newy, XYbrush, clip, nib );
	        oldx = newx;
	        oldy = newy;
	      }
	  }
      }
    
    /* If open, last interval is trivial too. */
    if (!closed)
	XYline( oldx, oldy, knotvec[order-1].x, knotvec[order-1].y,
		   XYbrush, clip, nib );
	
    return( 0 );
  }

/*
 *  XYfillSpline()
 *       
 *  Fill a closed spline of order 2, 3, 4, or 5.  Note that this is a copy
 *  of XYspline, suitably modified, so MIRROR ALL UPDATES!!!  Modifications
 *  are the removal of code dealing with open splines, and calling the
 *  fill routines.  NB:  Do not use XYdrawLine() here!!
 */
 
XYfillSpline( numpoints, vertex, order, pattern, opaque, clip, view, dx, dy )
	short numpoints;
	POINT *vertex;
	short order;
	enum Pattern pattern;
	unsigned short opaque;
	struct SubView *clip;
	View *view;
	short dx, dy;
  {
    unsigned int x, y;		/* Working end-points. */
    short newx, newy, oldx, oldy;	/* End-points of segment. */
    int knotNum, v;		/* Used to step through the knots */
    short knot;			/* Stepped for each knot in the support */
    int weight;			/* Current weighting factor */
    unsigned short *Blend;	/* Weighting function */
    short gauge;		/* Gauge of the spline */
    BOOLEAN closed = TRUE;
    
    if (numpoints <= 0)
      {
        printf("XYFILLSPLINE:  Illegal number of points %d\n", numpoints);
	return( ERROR );
      }
      
    switch (order)
      {
	case 2:  Blend = spline2;  break;
        case 3:  Blend = spline3;  break;
        case 4:  Blend = spline4;  break;
        case 5:  Blend = spline5;  break;
        default: printf("XYFILLSPLINE:  Illegal order %d.\n", order);
		 return( ERROR );
      }
      
    /* Initialize */
    if (XYfillInit( clip ) == ERROR)	return( ERROR );
    if (InitKnots( closed, order, numpoints, vertex, view, &gauge, dx, dy )
		== ERROR)
        return( ERROR );
    
    /* Iterate over the range of interesting knots */
    knotNum = numpoints;
    oldx = oldy = -1;
    while (knotNum--)
      {
	/* Get knots, and compute on subintervals */
	NewKnots( numpoints, vertex, knotNum, closed, order, view, dx, dy );
	v = 0;
	while (v <= (order << XYsplineRes))
	  {	
	    /* Combine the weighted knot values */
	    x = y = 1<<15;
	    knot = order;
	    while (knot--)
	      {
	        weight = Blend[v++];
	        x += weight * knotvec[knot].x;
	        y += weight * knotvec[knot].y;
	      }
	    v += gauge;
	
	    /* If a new result, do something */
	    newx = x >> 16;  newy = y >> 16;
	    if ((newx != oldx) || (newy != oldy))
	      {
                if (oldy != -1)
	            XYfillLine( oldx, oldy, newx, newy, clip );
	        oldx = newx;
	        oldy = newy;
	      }
	  }
      }
	
    return( XYfillScreen( pattern, opaque ) );
  }
