/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*------------------------------------------------------------------------

                    STRUCTURED DISPLAY FILE MANAGER

This module contains the basic routines for managing a structured
display file.  The module was designed to support a network
implementation of the SILT VLSI design-aid system, but has fairly
general applicability to any needs for a structured graphical data
base.

The display file is a structured representation of a list of graphical
objects such as rectangles, points, lines, and symbols, the basic
structured units.  This list is traversed by a window manager to
display the graphics objects appropriately.  This package only builds
the structure.  Other routines are necessary to display it.

The basic unit in the display file is the display record.  This record
is composed of fields indicating the type of the object represented and
other data describing the particular attributes of this realization of
the object. Also included in the display records are link fields to
facilitate the structured representation.

The symbol definition record type is the header record for a circular
list of records which collectively represent a "symbol".  These records
are linked together by the rightSib field in the display records.  The
list may include calls to other symbols, which will be expanded by the
window manager when asked to display the symbol being defined.  A
symbol call is represented by a record type of "symbol call" which has
a link to the appropriate symbol definition record placed in its
leftChild field.

This structure is interfaced via the routines given here.  Procedures
are given for defining new symbols, calling already existent symbols,
inserting primitive graphical objects into a symbol's definition list,
deleting symbol definitions, and editing symbol definitions by
inserting or deleting records from its defining list.

A set of lower level routines including those routines necessary for
maintaining a hash table for locating named symbols is presented
elsewhere (hash.c).  These routines are used by the
procedures presented here, and should be of no interest to the user.
The one exception is a Find(item) procedure which returns a
DisplayRecord pointer which gives the location of the display record with
the given name.

                                  Rocky Rhodes
                                  February 18, 1982
------------------------------------------------------------------------
*/

/* Modifications:  Tom Davis added code to keep track of the minimum
 * and maximum dimensions of the symbols. March 26, 1982.
 *
 * Bill Nowicki August 1982
 *	- Moved SdfTrace here from yaleinssym.c
 *	- AddItem now takes optional char * parameter
 *	- Added InquireItem() and InquireCall
 *	- return zero on error instead of printf-ing
 *	- Keep track of redrawing coordinates
 *	- Added CreateSDF and DeleteSDF calls
 *
 * David Kaelbling March 1983
 *	- Updated to include SDF_SPLINE type.
 *
 * Gus Fernandez February 1986
 *	- Improved error reporting, added SDF_OUTLINE optimization.
 */

#include <Vgts.h>
#include <Vgtp.h>
#include "sdf.h"
#include "arc.h"
#include "bitmaps.h"
#include <Vfont.h>
#include "splines.h"

extern short Debug, RowOrder;
extern DisplayRecord *Find();


SdfTableType SdfTable[NumberSDF];


short InitSDF()
  {
    /*
     * Initializes the SDF Table
     */
     register SdfTableType *sdf = SdfTable;
     register int i;
     
     for (i=0; i<NumberSDF; i++, sdf++)
       {
         sdf->type=UNUSED;
      }
  }


short CreateSDF()
  {
    /*
     * Returns the number of an available SDF. by finding an unused
     * entry in the SdfTable.
     */
     register SdfTableType *sdf = SdfTable;
     register short i;
     
     for (i=0; i<NumberSDF; i++, sdf++)
       {
         if (sdf->type!=UNUSED) continue;
	 sdf->type = GRAPHICS;
	 InitAttributeTables(sdf);
         if (Debug & DebugVgtsSdf)  
	     dprintf("Creating sdf %d\n", i);
         return(i);
      }
    return(-1);
  }



DeleteSDF(i,andStrings)
  short i;
  int andStrings;
    {
      /*
       * Remove all symbol definitions from the indicated SDF.
       * if andStrings is true, all the strings will be freed as well.
       */
       register SdfTableType *sdf = SdfTable + i;
       if (Debug & DebugVgtsSdf)  
	  dprintf("Delete sdf %d %s including strings\n", i, 
		  andStrings ? "" : "not");
       
       if (i<0 || i >= NumberSDF ) return(-1);
       if (sdf->type==UNUSED) return(i);
       FreeSdfAll(i,andStrings);
       sdf->type = UNUSED;
       return(i);
    }

/* DefineSymbol is called to enter item
in the structured display file.  A display record is created containing
this name and a pointer to some text information describing the symbol.
This entry is also added to the hash tables to enable future calls to
find the proper items.  A list of items should follow this definition
item, and all will be linked together using the rightSib field of the
display records.  The list is terminated by a call to EndSymbol which
also fills in the other fields of the symbol definition entry.  */

DefineSymbol(sdf, item, name)
  short sdf;
  short item;
  DisplayRecord *name;	/* name is actually a char pointer */
{
    DisplayRecord *Insert();
    register SdfTableType *s = SdfTable + sdf;
   
    if (Debug & DebugVgtsSdf)  
	dprintf("Defining symbol %d (%s) in sdf %d\n", item, name, sdf);

    if (s->definingSymbol != NULL) {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** DefineSymbol %d called when symbol %d was already open\n",item,s->definingSymbol->item);
	EndSymbol(sdf,s->definingSymbol->item,0);
    }

    if ( item != 0 )
    if ( Find( sdf, item ) != NULL )
      {
	if ( Debug & ( DebugVgtsSdf | DebugVgtsErrors ))
	    dprintf("VGTS ERROR*** DefineSymbol: item %d already exists\n",
		    item);
	DeleteSymbol( sdf, item );
     }

    s->definingSymbol = s->currentItem = Insert (sdf, item);
    s->needToClear = FALSE;
    s->outlineFlag = 0;
    if (s->currentItem==NULL) return(0);
 /* Create a display record with this name and add to hash table. */
    s->currentItem->type = SDF_SYMBOL_DEF;
    s->currentItem->rightSib = s->definingSymbol;
    s->currentItem->leftChild = name;
    s->currentItem->item = item;
    s->openXmin = s->openYmin = MaxCoord;
    s->openXmax = s->openYmax = -MaxCoord;
    s->redrawXmin = s->redrawYmin = MaxCoord;
    s->redrawXmax = s->redrawYmax = -MaxCoord;
    return (item);
}

/*
 * EndSymbol is called at the end of a list of items defining a
 * symbol.  It closes the named symbol to further additions and fills in
 * some needed information about the symbol that might not have been
 * available at the initial call to DefineSymbol. 
 */

EndSymbol(sdf, item, vgt)
  short sdf;
  short item;
  unsigned char vgt;
  {
    register SdfTableType *s = SdfTable + sdf;


    if (s->definingSymbol==NULL)
      {
	if (Debug & (DebugVgtsErrors | DebugVgtsSdf)) 
	    dprintf("VGTS ERROR*** EndSymbol %d called with with no symbol open\n",item);
	    return(0);
      }
    if (s->definingSymbol->item != item && 
        (Debug & (DebugVgtsErrors | DebugVgtsSdf)))
      {
	dprintf("VGTS ERROR*** EndSymbol called with wrong item\n");
	dprintf("   called with %d. Should have been %d.\n",
		item,s->definingSymbol->item);
      }

    if (Debug & DebugVgtsSdf)  
	dprintf("End symbol %d in vgt %d\n", item, vgt);

    s->definingSymbol->xmin = s->openXmin;
    s->definingSymbol->xmax = s->openXmax;
    s->definingSymbol->ymin = s->openYmin;
    s->definingSymbol->ymax = s->openYmax;
    s->definingSymbol->typedata = vgt;
    s->currentItem = NIL;
    s->definingSymbol = NIL;

    if (vgt) 
      {
        if (s->needToClear)
	    if (s->outlineFlag>0)
    	         RedrawVGTOutline( vgt, s->redrawXmin, s->redrawXmax, 
	    		    s->redrawYmin, s->redrawYmax);
	    else
    	         RedrawVGT( vgt, s->redrawXmin, s->redrawXmax, 
	    		    s->redrawYmin, s->redrawYmax);
	 else
    	    RedrawPartialVGT( vgt, s->redrawXmin, s->redrawXmax, 
	    		    s->redrawYmin, s->redrawYmax, s->addList);
      }
    return(item);
  }


/* 
 * EditSymbol(item) opens item for editing by
 * initializing the fields s->definingSymbol and s->currentItem to
 * the symbol definition item and the last item in the circular definition
 * list before the symbol definition, respectively.  This command has the
 * effect of calling DefineSymbol, inserting all the already existing
 * entries to the definition list, and waiting for additional insertions
 * or deletions. The editing process is ended the same way the initial
 * definition process was ended - a call to EndSymbol. 
 */
EditSymbol(sdf, item)
  short sdf;
  short   item;
{
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf)  dprintf("Edit symbol %d, sdf %d\n", item,sdf);

    if (s->definingSymbol != NULL) {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** EditSymbol %d called when symbol %d was already open\n",item,s->definingSymbol->item);
	EndSymbol(sdf,s->definingSymbol->item,0);
    }

    s->definingSymbol = Find(sdf, item);

    if (s->definingSymbol == NIL) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** EditSymbol: item %d does not exist\n",item);
	return(0);
    }
    if (s->definingSymbol->type != SDF_SYMBOL_DEF) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** EditSymbol: item %d is not a symbol\n",item);
	return(0);
    }
    s->currentItem = s->definingSymbol;
    s->needToClear = FALSE;
    s->outlineFlag = 2;

    while (s->currentItem != NIL &&
		s->currentItem->rightSib != NIL &&
		s->currentItem->rightSib->type != SDF_SYMBOL_DEF)
	s->currentItem = s->currentItem->rightSib;
	 /* Set s->currentItem to the last item in the symbol. */

    s->openXmin = s->definingSymbol->xmin;
    s->openXmax = s->definingSymbol->xmax;
    s->openYmin = s->definingSymbol->ymin;
    s->openYmax = s->definingSymbol->ymax;
    s->redrawXmin = s->redrawYmin = MaxCoord;
    s->redrawXmax = s->redrawYmax = -MaxCoord;
    s->addList = NULL;
    return(item);
}


/* 
 * DeleteSymbol(item) is used to "unexpand" the definition of a
 * symbol.  The hash table entry will remain and the symbol definition
 * item will remain, in order to prevent dangling references to this
 * symbol by other unknown referees, but all the items within the
 * definition list will be deleted. 
 */
DeleteSymbol(sdf, item)
  short sdf;
  short   item;
{
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf)  
	dprintf("Delete symbol %d in sdf %d calls...\n...", item, sdf);

    if (!EditSymbol(sdf, item)) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("Error above caused by invalid DeleteSymbol call");
	return(0);
    }

    while ((s->definingSymbol->rightSib)
    	&& (s->definingSymbol->rightSib->type!=SDF_SYMBOL_DEF))
      {
	/*
	 * Traverse the circular symbol definition list, deleting everything
         * along the way. 
	 */
        if (Debug & DebugVgtsSdf)  
	    dprintf("...");
	DeleteItem(sdf, s->definingSymbol->rightSib->item);
      }
    if (Debug & DebugVgtsSdf)  
	dprintf("...");
    EndSymbol(sdf, item, 0);
    return(item);
 }


/* Given a display record, store new item data into it, computing bounding
 * box and making other modifications as needed.
 */
short PutItem(sdf, disprec, type, x, y, x2, y2, typedata, data, changeFlag)
  short sdf;
  register DisplayRecord *disprec;
  unsigned char   type;
  short	x, y, x2, y2;
  short typedata;	/* number of points; radius; attribute value */
  char	*data;
  unsigned char changeFlag;
  /* True iff the entry is being changed rather than being written for the
   * first time.
   */
  {
    register SdfTableType *s = SdfTable + sdf;

    if ( type<1 || type>SDF_MaxCode) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** invalid item type %d\n",type);
	return(0);
    }

    s->outlineFlag = 0;
    disprec->x = x;
    disprec->y = y;
    disprec->xmin = x;
    disprec->ymin = y;
    disprec->xmax = x2;  /* may be changed farther on down */
    disprec->ymax = y2;  /* may be changed farther on down */
    disprec->typedata = typedata;
    disprec->type = type;
    
    switch (type)
      {
	  /*
	   * Set up some of the fields to known values, to prevent
	   * users from messing us up.  Compute bounding boxes.
	   */
	case SDF_TEXT:
	    SetUpText(disprec, &data, changeFlag);
	    break;

	case SDF_VERTICAL_LINE:
    		disprec->xmax = x + 1;
		break;

	case SDF_HORIZONTAL_LINE:
    		disprec->ymax = disprec->ymin;
		disprec->ymin--;
		break;

	case SDF_POINT:
		disprec->ymax = y + 1;
		disprec->xmax = x + 1;
		break;

	case SDF_GENERAL_LINE:
		if (x == x2)
		  {
		    disprec->type = SDF_VERTICAL_LINE;
		    disprec->xmax = x+1;
		    if (y > y2)
		      {
			disprec->ymin = y2;
			disprec->ymax = y;
		      }
		    break;
		  }
		if (y == y2)
		  {
		    disprec->type = SDF_HORIZONTAL_LINE;
		    disprec->ymin = disprec->ymax-1;
		    if (x > x2)
		      {
		        disprec->xmin = x2;
			disprec->xmax = x;
		      }
		    break;
		  }
    		if (x > x2)
		  { /* Swap points so (xmin,ymin) is on the left */
		    disprec->xmin = x2;
		    disprec->ymin = y2;
		    disprec->xmax = x;
		    disprec->ymax = y;
		  }
		if (disprec->ymin > disprec->ymax)
		  {
		    register temp;
		    disprec->type = SDF_DownLine;
		    temp = disprec->ymin;
		    disprec->ymin = disprec->ymax;
		    disprec->ymax = temp;
		  }
		 else
		  {
		    disprec->type = SDF_UpLine;
		  }
		break;
		
	case SDF_RASTER:
	    SetUpRaster(disprec, &data, changeFlag);
	    break;

	case SDF_SPLINE:
	    SplineBoundingBox(disprec, data);
	    break;

	case SDF_CIRCLE:	/* typedata is radius */
	case SDF_FILL_CIRCLE:
	    disprec->xmin = x - typedata;
	    disprec->ymin = y - typedata;
	    disprec->xmax = x + typedata;
	    disprec->ymax = y + typedata;
	    break;

	case SDF_POLYMARKER:
	    PolyBoundingBox(disprec, data);
	    disprec->xmin -= 3;	/* allow for the size of the marker */
	    disprec->ymin -= 3;
	    disprec->xmax += 3;
	    disprec->ymax += 3;
	    break;

	case SDF_POLYLINE:
	case SDF_FILL_AREA:
	    PolyBoundingBox(disprec, data);
	    break;

	case SDF_POLYARC:
	case SDF_FILL_POLYARC:
	    PolyarcBoundingBox(disprec, data);
	    break;

      }  /* end switch */

    /* update bounding box of region to redraw */
    if (disprec->type != SDF_SYMBOL_CALL)
      {
        s->openXmin = min (s->openXmin, disprec->xmin);
        s->openYmin = min (s->openYmin, disprec->ymin);
        s->openXmax = max (s->openXmax, disprec->xmax);
        s->openYmax = max (s->openYmax, disprec->ymax);
        s->redrawXmin = min (s->redrawXmin, disprec->xmin);
        s->redrawYmin = min (s->redrawYmin, disprec->ymin);
        s->redrawXmax = max (s->redrawXmax, disprec->xmax);
        s->redrawYmax = max (s->redrawYmax, disprec->ymax);
      }

    /* Enter any special, type-specific data: */
    if (data || !changeFlag)
        disprec->leftChild = (DisplayRecord *)data;

    if (Debug & DebugVgtsSdf) 
	 SdfTrace(disprec,-6,0);

    return (disprec->item);
  }


/*
 *  AddItem is the main routine for adding a item to the list of items
 * defining a symbol.  The routine is called with all the necessary
 * information for the display record, and a display record is created
 * with these characteristics.  The global name is used to enter this
 * location in the hash table for future reference.  The global variable
 * "s->currentItem" is used to remember the last item added to the current
 * symbol definition list, and is updated by this routine. 
 */
AddItem(sdf, item, x, x2, y, y2, typedata, type, data)
  short sdf;
  short item;
  unsigned char type;
  short	x, y, x2, y2;
  short typedata;	/* number of points; radius; attribute value */
  char	*data;
  {
	/*
	 * Create new display record, insert entry in hash table, and link it
	 *  to the chain of items defined for the currently open symbol. 
	 * if the item number is zero, we don't bother putting it into
	 * the hash table.
	 */
    register DisplayRecord *disprec;
    DisplayRecord *Insert ();
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf) 
	 dprintf("   Adding item %d, type %d\n", item, type);

    if (s->definingSymbol==NULL) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** AddItem %d called with no symbol open\n",item);
	return(0);
    }
    if (s->currentItem == NIL) return(0);
    if ( type<1 || type>SDF_MaxCode) return(0);

    /* Verify some of the user's data for SDF_SPLINE */
    if (type == SDF_SPLINE)
    {
#ifdef  LITTLE_ENDIAN
	SwapSpline(data);       /* Byte-swap */
#endif
        if (!VerifySpline(data)) return(0);
    }

    disprec = Insert(sdf, item);
    if (disprec==NULL) return(0);
    s->currentItem->rightSib = disprec;
    disprec->rightSib = s->definingSymbol;
    disprec->item = item;

    item = PutItem(sdf, disprec, type, x, y, x2, y2, typedata, data, FALSE);

    s->currentItem = disprec;
    if (s->addList==NULL) s->addList = disprec;

    return(item);
  }


/*
 * ChangeItem is the only procedure that (possibly) alters the database outside
 * the currently open symbol.  It merely changes the parameters of an already
 * existent item.  
 */
ChangeItem(sdf, item, x, x2, y, y2, typedata, type, data)
  short sdf;
  short item;
  unsigned char type;
  short	x, y, x2, y2;
  short typedata;	/* number of points; radius; attribute value */
  char	*data;
  {
    register    DisplayRecord *disprec;
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf) 
	 dprintf("   Changing item %d, type %d\n", item, type);

    disprec = Find(sdf, item);
    if (!disprec)  {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** ChangeItem: item %d does not exist\n",item);
	return(0);
    }

    s->redrawXmin = min( disprec->xmin, s->redrawXmin);
    s->redrawYmin = min( disprec->ymin, s->redrawYmin);
    s->redrawXmax = max( disprec->xmax, s->redrawXmax);
    s->redrawYmax = max( disprec->ymax, s->redrawYmax);
    s->needToClear = TRUE;

    if (disprec->type != SDF_SYMBOL_CALL)
      {
	disprec->item = item;
	return(PutItem(sdf, disprec, type, x, y, x2, y2, typedata, data, TRUE));
      }
    else
     {
	return (0);
      }
  }


DeleteItem(sdf, item)
  short sdf;
  short   item;
{
	/*
	 * DeleteItem: will delete item from the currently open symbol
	 * definition, if possible.  The display record will be freed,
	 * and item will be stricken from the hash table. Symbol
	 * calls may be deleted just like any other item, but an entire symbol
	 * definition must be deleted by using the DeleteSymbol routine.
	 */

    DisplayRecord *disprec, *saveRec;
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf) 
	 dprintf("   Deleting item %d\n", item);

    if (!s->definingSymbol) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** DeleteItem %d: no symbol is open\n",item);
	return(0);			/* No symbol is open. Can't delete */
    }

    if (disprec=Find(sdf, item))
      {
        s->needToClear = TRUE;
	if ((disprec->type != SDF_OUTLINE))
	   s->outlineFlag = 0;
	if ((disprec->type == SDF_SYMBOL_CALL) && (disprec->leftChild))
	  {
	    s->redrawXmin = min (s->redrawXmin,
		    disprec->xmin + disprec->leftChild->xmin);
	    s->redrawXmax = max (s->redrawXmax,
		    disprec->xmin + disprec->leftChild->xmax);
	    s->redrawYmin = min (s->redrawYmin,
		    disprec->ymin + disprec->leftChild->ymin);
	    s->redrawYmax = max (s->redrawYmax,
		    disprec->ymin + disprec->leftChild->ymax);
	  }
	else
	  {
	    s->redrawXmin = min (s->redrawXmin, disprec->xmin);
	    s->redrawXmax = max (s->redrawXmax, disprec->xmax);
	    s->redrawYmin = min (s->redrawYmin, disprec->ymin);
	    s->redrawYmax = max (s->redrawYmax, disprec->ymax);
	  }
      }
    else {
       if (item && (Debug & (DebugVgtsSdf | DebugVgtsErrors)))
	    dprintf("VGTS WARNING*** DeleteItem: Item %d does not exist\n",item);
    }

    disprec = s->definingSymbol;	/* Start the search for item at the
				   symbol definition item. */

    s->openXmin = s->openYmin = MaxCoord;/* need to re-compute these */
    s->openXmax = s->openYmax = -MaxCoord;

    while ((disprec->rightSib) && (disprec->rightSib->type != SDF_SYMBOL_DEF))
    /* Search until back to the beginning of the circular list. */
    {
	if (disprec->rightSib->item == item)
	  {
	    if ((disprec->rightSib->type != SDF_OUTLINE))
	        s->outlineFlag = 0 ;
	    else
		s->outlineFlag-- ; /* Allow only one DeleteItem call! */
	    saveRec = disprec->rightSib;
	    if (saveRec == NIL)
		return;
	    DeleteHash(sdf, item);/* Delete hash table entry for item. */
	    if ( disprec->rightSib == s->addList )
		s->addList = disprec->rightSib->rightSib;
	    disprec->rightSib = disprec->rightSib->rightSib;
	    FreeLeftChild( saveRec );
	    ReturnDisplayRecord( saveRec );
  	  }
	if ((disprec->rightSib) && (disprec->rightSib->type!=SDF_SYMBOL_DEF))
	    disprec = disprec->rightSib;
	if ((disprec->type == SDF_SYMBOL_CALL) && (disprec->leftChild))
	  {
	    s->openXmin = min (s->openXmin,
		    disprec->xmin + disprec->leftChild->xmin);
	    s->openXmax = max (s->openXmax,
		    disprec->xmin + disprec->leftChild->xmax);
	    s->openYmin = min (s->openYmin,
		    disprec->ymin + disprec->leftChild->ymin);
	    s->openYmax = max (s->openYmax,
		    disprec->ymin + disprec->leftChild->ymax);
	  }
	else
	  {
	    s->openXmin = min (s->openXmin, disprec->xmin);
	    s->openXmax = max (s->openXmax, disprec->xmax);
	    s->openYmin = min (s->openYmin, disprec->ymin);
	    s->openYmax = max (s->openYmax, disprec->ymax);
	  }
      }
    s->currentItem = disprec;	/* s->currentItem should point to the last
				   item in the circular list. */
    return(item);
}


AddCall(sdf, item, xoffset, yoffset, symbolName)
  short sdf;
  short   item,
        xoffset,
        yoffset,
        symbolName;
{
/*
 * AddCall is called to insert an instance of one symbol in the
 * definition list of another symbol.  The hashed directory is consulted
 * and a dummy entry is made for this symbol if the definition is not
 * already there. Any future attempts to define this symbol will use this
 * dummy item as the start of the symbol definition list, maintaining
 * consistency with any already existing calls to the symbol.  
 */
    DisplayRecord *symbolDef;
    unsigned char   typedata = 0;
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf) 
	 dprintf("   Adding call %d @(%d, %d) calling %d\n", item, xoffset,yoffset,symbolName);

    AddItem(sdf, item, xoffset, 0, yoffset, 0,
	    typedata, SDF_SYMBOL_CALL, NULL);
    if (s->currentItem == NIL) {
       if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("Error above caused by invalid AddCall\n",item);
	return;
    }
    s->currentItem->leftChild = symbolDef = Find(sdf, symbolName);
    if (symbolDef==NULL) return(0);

	 /*
	  * A symbol call is the same as a AddItem, except the leftChild 
	  * of the created display record is linked to the symbol definition
	  * item of the called symbol type. 
	  */

    s->openXmin = min (xoffset + symbolDef->xmin, s->openXmin);
    s->openYmin = min (yoffset + symbolDef->ymin, s->openYmin);
    s->openXmax = max (xoffset + symbolDef->xmax, s->openXmax);
    s->openYmax = max (yoffset + symbolDef->ymax, s->openYmax);
    s->redrawXmin = min (xoffset + symbolDef->xmin, s->redrawXmin);
    s->redrawYmin = min (yoffset + symbolDef->ymin, s->redrawYmin);
    s->redrawXmax = max (xoffset + symbolDef->xmax, s->redrawXmax);
    s->redrawYmax = max (yoffset + symbolDef->ymax, s->redrawYmax);
    return(item);
}


InquireItem(sdf, item, xmin, xmax, ymin, ymax, typedata, type, string)
    short sdf;
    short item, *xmin, *xmax, *ymin, *ymax;
    unsigned char   *type;
    unsigned char   *typedata;
    char	*string;
  {
	/*
	 * InquireItem -- return the values in the SDF record indicated.
	 * returns non-zero if the item was found.
	 */
    register DisplayRecord *disp;
    register SdfTableType *s = SdfTable + sdf;

    if (Debug & DebugVgtsSdf) 
	 dprintf("Inquire item %d, sdf %d\n", item, sdf);

    disp = Find(sdf, item);
    if (disp)
      {
       if (xmin) *xmin = disp->xmin;
       if (xmax) *xmax = disp->xmax;
       if (ymin) *ymin = disp->ymin;
       if (ymax) *ymax = disp->ymax;
       if (typedata) *typedata = disp->typedata;
       if (type) *type = disp->type;
       if (string) strcpy( string, disp->leftChild );
       return(item);
      }
    if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	dprintf("VGTS ERROR*** InquireItem: Item %d does not exist\n",item);
    return(0);
  }


short InquireCall(sdf, item)
    short sdf;
    short item;
  {
	/*
	 * InquireCall -- return the name of the symbol called by the item.
	 * returns zero if item is not found or not a symbol call.
	 */
    register DisplayRecord *disp;

    if (Debug & DebugVgtsSdf) 
	dprintf("Inquire call %d, sdf %d\n", item, sdf);

    disp = Find(sdf, item);
    if (disp && disp->type == SDF_SYMBOL_CALL)
      {
        disp = disp->leftChild;
        if (disp) return(disp->item);
      }
    if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	dprintf("InquireCall: item %d does not exist or is not a symbol call\n",item);
    return(0);
  }


#define abs1(a) (a>=0?a:-a - 1)
/* debug trace of a structured display file. */
SdfTrace(sdfPtr, indent, all)
  DisplayRecord *sdfPtr;
  short indent, all;
{
  short i;
  char string[200];
  char c;				/* MMT */
  char GetInputChar();

  while (1)
   {
    if (sdfPtr == 0)
        return;
    if (indent >= 0) {			/* GAF */
       dprintf("next?");
       c = GetInputChar();		/* MMT */
       if (c == 'n')
           return;
    }
    for (i = 0; i < abs1(indent); i++)	/* tab in */
        dputchar(' ');

    switch (sdfPtr->type)
      {
	case SDF_SYMBOL_CALL: strcpy(string, "SDF_SYMBOL_CALL"); break;
	case SDF_SYMBOL_DEF: strcpy(string, "SDF_SYMBOL_DEF"); break;
	case SDF_FILLED_RECTANGLE: strcpy(string, "SDF_FILLED_RECTANGLE"); break;
	case SDF_HORIZONTAL_LINE: strcpy(string, "SDF_HORIZONTAL_LINE"); break;
	case SDF_VERTICAL_LINE: strcpy(string, "SDF_VERTICAL_LINE"); break;
	case SDF_POINT: strcpy(string, "SDF_POINT"); break;
	case SDF_SIMPLE_TEXT: sprintf(string, "SDF_SIMPLE_TEXT (%s)",sdfPtr->leftChild); break;
	case SDF_GENERAL_LINE: strcpy(string, "SDF_GENERAL_LINE"); break;
	case SDF_OUTLINE: strcpy(string, "SDF_OUTLINE"); break;
	case SDF_HORIZONTAL_REF: strcpy(string, "SDF_HORIZONTAL_REF"); break;
	case SDF_VERTICAL_REF: strcpy(string, "SDF_VERTICAL_REF"); break;
	case SDF_SEL_HORIZ_REF: strcpy(string, "SDF_SEL_HORIZ_REF"); break;
	case SDF_SEL_VERT_REF: strcpy(string, "SDF_SEL_VERT_REF"); break;
	case SDF_TEXT: sprintf(string, "SDF_TEXT (%s)",((TextStructure *)sdfPtr->leftChild)->string); break;
	case SDF_RASTER: strcpy(string, "SDF_RASTER"); break;
	case SDF_SPLINE: strcpy(string, "SDF_SPLINE"); break;
	case SDF_POLYMARKER: strcpy(string, "SDF_POLYMARKER"); break;
	case SDF_POLYLINE: strcpy(string, "SDF_POLYLINE"); break;
	case SDF_FILL_AREA: strcpy(string, "SDF_FILL_AREA"); break;
	case SDF_CIRCLE: strcpy(string, "SDF_CIRCLE"); break;
	case SDF_FILL_CIRCLE: strcpy(string, "SDF_FILL_CIRCLE"); break;
	case SDF_POLYARC: strcpy(string, "SDF_POLYARC"); break;
	case SDF_FILL_POLYARC: strcpy(string, "SDF_FILL_POLYARC"); break;
	case SDF_FILL_PATTERN: strcpy(string, "SDF_FILL_PATTERN"); break;
	case SDF_FILL_AREA_OPACITY: strcpy(string, "SDF_FILL_AREA_OPACITY"); break;
	case SDF_FILL_BORDER_STYLE: strcpy(string, "SDF_FILL_BORDER_STYLE"); break;
	case SDF_FILL_AREA_COLOR: strcpy(string, "SDF_FILL_AREA_COLOR"); break;
	case SDF_MARKER_TYPE: strcpy(string, "SDF_MARKER_TYPE"); break;
	case SDF_MARKER_COLOR: strcpy(string, "SDF_MARKER_COLOR"); break;
	case SDF_POLYLINE_COLOR: strcpy(string, "SDF_POLYLINE_COLOR"); break;
	case SDF_TEXT_COLOR: strcpy(string, "SDF_TEXT_COLOR"); break;
	case SDF_FONT_INDEX: strcpy(string, "SDF_FONT_INDEX"); break;
	case SDF_COLOR: strcpy(string, "SDF_COLOR"); break;
	case SDF_InternalText: strcpy(string, "SDF_InternalText"); break;
	case SDF_UpLine: strcpy(string, "SDF_UpLine"); break;
	case SDF_DownLine: strcpy(string, "SDF_DownLine"); break;
        case SDF_FREE: strcpy( string,  "Free!?!?"); break;

        default:
    	    sprintf( string, "***Undefined (%d)", sdfPtr->type );
    	    break;
      }
    
    dprintf("%s: Num: %d typedata: %d, (%d, %d)-(%d, %d)\n",
    	string,
    	sdfPtr->item,
    	sdfPtr->typedata,
    	sdfPtr->xmin,
    	sdfPtr->ymin,
    	sdfPtr->xmax,
    	sdfPtr->ymax);

    if (!all) return;

    if (sdfPtr->type == SDF_SYMBOL_CALL )
        SdfTrace(sdfPtr->leftChild, indent>=0?indent + 4:indent-4);
#if 0
    else
        dprintf("\"%s\"\n", sdfPtr->leftChild ?
	      (char *) sdfPtr->leftChild : "NULL" );
#endif 0
    if (sdfPtr->rightSib==NULL)
       {
         dprintf( "Right Sibling pointer is NULL!\n");
	 return;
       }
    sdfPtr = sdfPtr->rightSib;
    if (sdfPtr->type == SDF_SYMBOL_DEF) return;
  }
}


#ifdef  LITTLE_ENDIAN
/*
 * Byte-swap a spline.  Data is always sent in big-endian order.
 * THIS ROUTINE MUST BE CHANGED IF THE DEFINITION OF SPLINE IS CHANGED.
 */
SwapSpline(item)
  register SPLINE *item;
  {
    /* Make sure numvert is swapped before it is used in the
	multiplication below */
    ByteSwapShortInPlace(&item->order, 4);
    ByteSwapLongInPlace(&item->nib, 4);
    ByteSwapShortInPlace(&item->border, 8);
    ByteSwapLongInPlace(&item->pat, 4);
    /* numvert is the number of points; they are located consecutively
       in memory, starting at head */
    ByteSwapShortInPlace(&item->head, item->numvert * sizeof item->head);
}
#endif LITTLE_ENDIAN


/* Check that a spline structure is valid, return 1 if OK, 0 otherwise */
int VerifySpline(item)
  register SPLINE *item;
  {
    if ((item->order <= 0) || (item->order >= 6))
      {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	  dprintf("VGTS ERROR*** VerifySpline: bad order %d\n", item->order);
	return( 0 );
      }
    if (item->numvert < 1)
      {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	  dprintf("VGTS ERROR*** VerifySpline: bad numvert %d\n", item->numvert);
	return( 0 );
      }
    else if (item->numvert == 1)
      {
	item->order = 1;
      }
    if (((int) item->nib < 0) || ((int) item->nib >= (int) NibEnd))
      {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** VerifySpline: Invalid nib %d.\n\r", item->nib );
	return( 0 );
      }
    if (((int) item->pat < 0) || ((int) item->pat >= (int) PatEnd))
      {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** VerifySpline: Invalid fill pattern %d.\n\r", item->pat );
	return( 0 );
      }
    if (item->filled && !(item->closed))
      {
	if (Debug & (DebugVgtsSdf | DebugVgtsErrors))
	    dprintf("VGTS ERROR*** VerifySpline: Filled but not closed\n");
	return( 0 );
      }
    return(1);
  }


SetUpText(disprec, stringPtr, changeFlag)
    DisplayRecord *disprec;
    char **stringPtr;
    unsigned char changeFlag;
  {
    TextStructure *text;
    short xOrigin, yOrigin;

    if (changeFlag && (disprec->type == SDF_TEXT))
      { /* reuse the existing structure */
	if (( text = (TextStructure*) disprec->leftChild ) == NULL )
	    text = (TextStructure *) malloc( sizeof( TextStructure ));
	else if ( text->string )
	    free( text->string );
      }
    else /* create a new structure */
	text = (TextStructure*)malloc(sizeof(TextStructure));
    if (text == NULL)
	return (0);

    xOrigin = disprec->xmin; yOrigin = disprec->ymin;
    text->string = *stringPtr;
    text->length = (*stringPtr ? strlen(*stringPtr) : 0);
    TextSize(text, disprec->typedata,
	     &disprec->xmin, &disprec->xmax, &disprec->ymin, &disprec->ymax);
    text->xskip = xOrigin - disprec->xmin;
    text->descent = yOrigin - disprec->ymin;
    *stringPtr = (char*)text; /* now points to the structure, not the string */
  }


SetUpRaster(disprec, rasterData, changeFlag)
    DisplayRecord *disprec;
    BitUnit **rasterData;
    unsigned char changeFlag;
  {
    if (*rasterData==NULL) 
	return(0);
    if (RowOrder == 0) /* temporary old stuff for sun-100 */
      { 
	register MemRaster *raster;

	if (changeFlag && (disprec->type == SDF_RASTER))
	  { /* reuse the existing structure */
	    if (( raster = (MemRaster *) disprec->leftChild ) == NULL )
		raster = (MemRaster *) malloc( sizeof( MemRaster ));
	    else if ( (BitUnit *)raster->start )
	        free( (BitUnit *)raster->start );
	  }
        else /* create a new structure */
	    raster = (MemRaster *)(malloc(sizeof(MemRaster)));
	    
	if (raster==NULL) return(0);
	raster->where = (disprec->typedata & InverseRaster)
	    |HeadRaster|MemoryRaster|LinkedRaster;
	raster->next = raster;
	raster->start = (unsigned short *)*rasterData;
	raster->width = disprec->xmax - disprec->xmin + 1;
	raster->height = disprec->ymax - disprec->ymin + 1;
	raster->stride = raster->height;
	raster->bitOffset = 0;
	*rasterData = (BitUnit *)raster;/* now points to the structure */
      }
    else
      {   
	register VRaster *raster;

	if (changeFlag && (disprec->type == SDF_RASTER))
	  { /* reuse the existing structure */
	    if (( raster = (VRaster *)disprec->leftChild ) == NULL )
		raster = (VRaster *) malloc( sizeof( VRaster ));
	    else if ( raster->start )
		free( raster->start );
	  }
        else /* create a new structure */
	    raster = (VRaster *)(malloc(sizeof(VRaster)));
	    
	if (raster==NULL) return(0);
	raster->start = *rasterData;
	Options(*raster) =
	    (disprec->typedata & InverseRaster) ? RasterInverse : 0;
	raster->bBox.h = disprec->xmax - disprec->xmin + 1;
	raster->bBox.v = disprec->ymax - disprec->ymin + 1;
	raster->origin.h = 0; raster->origin.v = 0;
	raster->stride =
	  ((raster->bBox.h+(BitWordLen-1)) >> BitWordLog) * BitAlign;
	raster->xoffset = 0;
	*rasterData = (BitUnit *)raster;/* now points to the structure */
      }
    disprec->xmax++; disprec->ymax++;
    disprec->xmin--; disprec->ymin--;
  }


SplineBoundingBox(disprec, sptr)
  DisplayRecord *disprec;
  SPLINE *sptr;
  {
    register temp;
    register POINT *pptr;
    register short xm, ym;
    
    xm = disprec->xmin;
    ym = disprec->ymin;
    disprec->xmin = disprec->xmax = (sptr ? sptr->head.x += xm : 0);
    disprec->ymin = disprec->ymax = (sptr ? sptr->head.y += ym : 0);
    pptr = (sptr ? &(sptr->head) : NULL);
    if (pptr)
	for (temp = 1;  temp < sptr->numvert;  temp++)
	  {	
	    pptr[temp].x += xm;
	    pptr[temp].y += ym;
	    disprec->xmax = max( pptr[temp].x, disprec->xmax );
	    disprec->xmin = min( pptr[temp].x, disprec->xmin );
	    disprec->ymax = max( pptr[temp].y, disprec->ymax );
	    disprec->ymin = min( pptr[temp].y, disprec->ymin );
	  }
    disprec->ymax += 10;  /* Random fudging to take into account	*/
    disprec->xmax += 10;  /*  the width of the pens.  (These are	*/
    disprec->ymin -= 10;  /*  in World coordinates, and nibs have	*/
    disprec->xmin -= 10;  /*  constant size in screen coords.)	*/
    
    /* Convert to relative coordinates */
    if (sptr)
	for (temp = 0;  temp < sptr->numvert;  temp++)
	  {
	    pptr[temp].x -= disprec->xmin;
	    pptr[temp].y -= disprec->ymin;
	  }
  }


/* The client program passes "poly" data in terms of point coordinates in
 * the local symbol coordinate system.  What we store is a set of
 * bounding box coordinates in the local symbol coordinate system, and a
 * set of RELATIVE point values, relative to the first point given.
 * Here we compute that bounding box and relativize the data points.
 */
PolyBoundingBox(disprec, data)
  register DisplayRecord *disprec;
  SdfPoint *data;
  {
    register SdfPoint *p = data;
    register short xmin, xmax, ymin, ymax;
    register int i;
    short x, y;

    x = p->x;
    y = p->y;
    disprec->x = x;
    disprec->y = y;

    xmin = ymin = 32767;
    xmax = ymax = -32767;
    /* compute bounding box */
    for (i=0; i < disprec->typedata; i++)
      {
	if (p->x < xmin)  xmin = p->x;
	if (p->x > xmax)  xmax = p->x;
	if (p->y < ymin)  ymin = p->y;
	if (p->y > ymax)  ymax = p->y;
	p++;
      }
    disprec->xmin = xmin;
    disprec->xmax = xmax;
    disprec->ymin = ymin;
    disprec->ymax = ymax;

    /* Adjust all the coordinates, to be relative to the lower left corner
     * of the bounding box.  That is going to make MoveItem trivial to
     * implement.
     */
    p = data;
    for (i=0; i < disprec->typedata; i++)
      {
	p->x -= x;
	p->y -= y;
	p++;
      }
  }


PolyarcBoundingBox(disprec, data)
  register DisplayRecord *disprec;
  Arcdata *data;
  {
    register Arcdata *p = data;
    register short xmin, xmax, ymin, ymax;
    register int i;
    short x, y;

    x = p->x;
    y = p->y;
    disprec->x = x;
    disprec->y = y;

    xmin = ymin = 32767;
    xmax = ymax = -32767;
    /* compute bounding box */
    for (i=0; i < disprec->typedata; i++)
      {
	if (p->x < xmin)  xmin = p->x;
	if (p->x > xmax)  xmax = p->x;
	if (p->y < ymin)  ymin = p->y;
	if (p->y > ymax)  ymax = p->y;
	p++;
      }
    disprec->xmin = xmin;
    disprec->xmax = xmax;
    disprec->ymin = ymin;
    disprec->ymax = ymax;

    /* Adjust all the coordinates, to be relative to the lower left corner
     * of the bounding box.  That is going to make MoveItem trivial to
     * implement.
     */
    p = data;
    for (i=0; i < disprec->typedata; i++)
      {
	p->x -= x;
	p->y -= y;
	p->cx -= x;
	p->cy -= y;
	p++;
      }
  }
