/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * FILE:  mousehit.c 
 *
 * Original Author: Linda Gass
 * Bill Nowicki August 1982
 *	- Changed sdf pointer to display number in FindSelectedObject
 *	- Removed Yale dependencies in DisambiguateSelection
 * David Kaelbling April 1983
 *	- updated to handle splines.
 * Gustavo Fernandez 4/23/85
 *	- Added support in HitSearch for rasters and weak support for other
 *	  types by checking the bounding box.
 *                                                                       
 * This set of functions detects the object(s) that the mouse hit, if any. 
 * The main routine, FindSelectedObject takes as input:                      
 * 	(1) The x and y coordinates of the mouse hit, ALREADY CONVERTED TO 
 *    	    the appropriate WORLD coordinates.                             
 *	(2) vgt Index associated with the window in which the 
 * 	    hit occurred.                                                  
 *	(3) The search type.  This is the type of object we expected to    
 *	    hit. (eg- a line, rectangle, symbol call, etc.)                
 *	    ==> If any new search types are added, a new constant must be  
 * 	    added to the search type constants defined below.(see constant 
 *	    definitions)                                                   
 * FindSelectedObject returns a list of minimum area objects of type Search- 
 * Type which were hit.  This list can have 3 states when returned:        
 *	(1) empty - this means nothing of type SearchType was hit.         
 *	(2) contains one element - this means that we found exactly one    
 *	    hit of type SearchType which has minimum area.                 
 *	(3) contains more than one element - this means we hit several     
 *	    IDENTICAL objects of type SearchType which all had minimum     
 *	    area.  In this case it's up to the application program to      
 *	    decide which object it will accept as the hit.                 
 * The HitList has a maximum number of objects it can return.              
 * ==> This maximum is stored as a constant in MAX HIT OBJECTS.            
 *                                                                         
 * FindSelectedObject performs two passes of the SDF if necessary.  Each     
 * object in the SDF has a bounding box and on the first pass, we check if 
 * the hit is directly contained within the bounding box, except in the    
 * case of lines.  For lines we extend the bounding box of the line by a   
 * certain number of pixels at the endpoints and by a certain number of    
 * pixels at the sides.  Then we test if the hit is directly contained     
 * within this expanded bounding box.  On the second pass, depending upon  
 * what type of object we are examining (except for lines), we expand its  
 * bounding box by a certain number of pixels, and we test for a hit       
 * directly within this expanded bounding box.  In the case of lines, if   
 * they were not hit during the first pass then we ignore them during the  
 * second pass.                                                            
 * ==> The number of pixels to expand by are declared as constants below   
 *     in the constant definition section.                                 
 *                                                                         
 **************************************************************************
 */


#include <Vgts.h>
#include "sdf.h"
#include "vgt.h"

/****************************** CONSTANT DEFINITIONS ***********************/

#define MAXLONG 017777777777       /* Octal constant for maximum positive */
				   /* integer that a long integer can    */
				   /* hold (on this machine 32 bits)      */
#define FirstPass 1
#define SecondPass 2

/********** Pixel Expansions ***********/

#define SYM_RANGE 2
#define RECT_RANGE 2
#define TEXT_RANGE 2
#define LINE_END_RANGE 2
#define LINE_SIDE_RANGE 3
#define REF_POINT_RANGE 2
#define SPLINE_RANGE 2
#define RASTER_RANGE 2
#define OTHER_RANGE 0

# define abs(x) (x<0 ? -x : x)
	/*
	 * some day someone should do an integer square root routine!
	 */
extern double sqrt();

/*************************** GLOBAL DECLARATIONS *************************/

static int MinArea, MinWidth;    /* used to keep track of the minimum area */
				 /* and width of hit objects found so far  */
				 /* They are declared static because they  */
				 /* are "locals" of this object file only. */
LISTTYPE HitList;         /* This list contains the minimum area objects */
			  /* which have been hit.                       */
			  /* the list information structure which keeps */
			  /* track of the head of the list and the num- */
			  /* ber of elements in the list.               */


/**************************************************************************/




/******************** STORAGE MANAGEMENT FUNCTIONS *************************/

/* The following storage management functions operate on a stack-like data 
structure which keeps track of storage allocated.  

The HitList is a linked list of MinElement records.  The underlying data 
structure of the HitList is a fixed size array of MinElement records which
is maintained as a stack.  We can keep "adding" new records to the stack
(list) until it becomes full.  Requesting new records is essentially the
same as pushing items onto the stack; (ie - each new record "created" takes
up space on the stack.)  When records are no longer needed, their space is 
reclaimed.  Reclaiming space essentially amounts to popping items off the
stack, thereby creating more space on the stack so that new records can be
"created".

The storage management functions are GetMinRecord and Reclaim.  No error
checking (eg- checks for exceeding storage limits) is done in these functions.
This is for efficiency (speed) considerations.  Error checks must be made in
higher level routines.

/********************* Declarations for Storage functions ******************/

#define MAX_HIT_OBJECTS 10

static MINREC Storage[MAX_HIT_OBJECTS];
static MINPTR StoragePtr = Storage;  /* Initialize to point at */
		                     /* beginning of storage array. */


/**************************************************************************
 *                                                                
 *                           Reclaim                                 
 *                                                                        
 * This macro takes the place of the function Reclaim which reclaims       
 * storage space.  All this macro does is reset the storage pointer to its
 * initial (empty) value.
 *                                                                         
 *                                                                         
 * INPUT:  none
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED:  StoragePtr                                           
 * CALLED BY:  CreateNewList                                               
 * CALLS ON:   none                                                        
 *                                                                         
 **************************************************************************
 */

#define Reclaim() StoragePtr = Storage
 

/**************************************************************************
 *                                                                         
 *                         GetMinRecord                                    
 *                                                                         
 * This macro returns a pointer to a MinElement record.  It also updates
 * the storage space pointer to point at the next free record.             
 *                                                                         
 * IMPORTANT ASSUMPTION:  This function is not called unless we are        
 *        GUARANTEED not to exceed our storage limits.  The reason a check 
 *        is not done in this function is because of speed considerations. 
 *        We dont want to spend the time calling a function only to find   
 *        out that we cant to what we wanted to do.                        
 *                                                                         
 * INPUT:  none                                                            
 * OUTPUT: pointer to next available MinElement record.                    
 * GLOBALS MODIFIED: StoragePtr                                            
 * CALLED BY:  CreateNewList, AddToList                                    
 * CALLS ON:   none                                                        
 *                                                                         
 **************************************************************************
 */

#define GetMinRecord() (StoragePtr++) /* return pointer to rec and autoinc */


/************************* LIST MAINTENANCE FUNCTIONS **********************/

/* The list maintenance functions are built upon the storage management
 * functions.  There is no function for deleting an item from a list as
 * this is never done.  Instead there is a function for creating a new
 * list with one item in it.  This function takes care of deleting all the
 * items on the old list.  There is a function for adding items to the
 * list.  The list has a maximum number of items it may contain but no
 * error chekcs are done in these functions for exceeding the list
 * maximum.  This must be done immediately before the call to a list
 * maintenance function.  We have constrained that the number of items in
 * the list does not exceed the amount of storage space we have allocated
 * for list elements.  By making the maximum size of the list be the same
 * as the maximum storage available, we guarantee that our storage will
 * never be exceeded.
 */


/**************************************************************************
 *                                                                         
 *                           CreateNewList                                 
 *                                                                         
 * This function creates a new HitList containing one MinElement record.   
 *                                                                         
 * INPUT:  ItemPtr - pointer to the SDF record which contains the infor-   
 *                   mation about the object we just hit.  We are creating 
 *                   a new list with this object being the 1st thing in it.
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED: HitList                                               
 * CALLED BY:  KeepTrackMin                                                
 * CALLS ON:   GetMinRecord, Reclaim                                       
 *                                                                         
 **************************************************************************
 */

CreateNewList(ItemPtr, edgeset)
  short edgeset;
  DisplayRecord *ItemPtr;
{
    /* Throw away the old list and start fresh */
    Reclaim();
    HitList.Header = GetMinRecord();
    /* when we get to here we have a list containing one element */
    HitList.NumOfElements = 1;
    HitList.Header->item = ItemPtr->item;
    HitList.Header->edgeset = edgeset;
    HitList.Header->next = NIL;

}  /* end CreateNewList */

/**************************************************************************
 *                                                                         
 *                            AddToList                                    
 *                                                                         
 * This function adds a MinElement record to the Hit List.                 
 *                                                                         
 * IMPORTANT ASSUMPTION:  We will not try to add more records than the     
 *           maximum allowable on our hit list.  We guarantee this BEFORE  
 *           we call this function.                                        
 *                                                                         
 * INPUT:  ItemPtr - pointer to the SDF record which contains the infor-   
 *                   mation about the object we just hit.  We are adding   
 *                   this object to the list of objects we have hit.       
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED: HitList                                               
 * CALLED BY:  KeepTrackMin                                                
 * CALLS ON:   GetMinRecord                                                
 *                                                                         
 **************************************************************************
 */

AddToList(ItemPtr, selectSet)
short selectSet;
DisplayRecord *ItemPtr;
{
MINPTR tempRec;

    tempRec = GetMinRecord();
    tempRec->item = ItemPtr->item;
    tempRec->edgeset = selectSet;
    tempRec->next = HitList.Header;
    HitList.Header = tempRec;
    ++HitList.NumOfElements;
}



/**************************************************************************
 *                                                                         
 *                           KeepTrackMin                                  
 *                                                                         
 * INPUT:  ItemPtr - pointer to the SDF record which contains an object    
 *                   being checked for minimum area.                       
 *         Height -  the height of the object ItemPtr is pointing to.      
 *         Width  -  the width of the object ItemPtr is pointing to.       
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED: MinArea, MinWidth                                     
 * CALLED BY:  CheckRectangle, CheckLine                                   
 * CALLS ON:   CreateNewList, AddToList                                    
 *                                                                         
 **************************************************************************
 */

KeepTrackMin(ItemPtr, Height, Width, edgeset)
short edgeset;
DisplayRecord *ItemPtr;
int Height, Width;
{
/* local declarations */
int newArea, newWidth;

    newArea = abs(Height * Width);
    if (newArea < MinArea)
	{
	MinArea = newArea;
	MinWidth = Width;
	/* make this new object the first thing on our Hit list */
	CreateNewList(ItemPtr, edgeset);
 	}
    else if (newArea == MinArea)
	{
	/* if our new object is shorter and fatter than the current one */
	/* then pick the new one as minimum & make it the first on list */
	if (Width > MinWidth)
	    {
	    MinWidth = Width;  /* note: we need not update the area */
	    CreateNewList(ItemPtr, edgeset);
	    } /* end if newWidth */
        else if (Width == MinWidth)
	    {
	    /* now we wish to add this to the chain of minimum hits */
	    /* this hit and the previous hits are identical.        */
	    if (HitList.NumOfElements == 0) 
 		printf("\nKeepTrackMin:  Error in HitList\n");
	    if (HitList.NumOfElements < MAX_HIT_OBJECTS)
		AddToList(ItemPtr, edgeset);
	    } /* end if Width == */
        } /* end if newArea == MinArea */
    } /* end KeepTrackMin */
	    





/**************************************************************************
 *                                                                         
 *                            CheckRectangle                               
 *                                                                         
 * This function is used for determining if a rectangle was hit or if text 
 * which for all practical purposes is treated like a rectangle, was hit.  
 * Depending on which pass this is, we take two different approaches to    
 * testing for a hit:                                                      
 *     Pass One:  Check if we hit the rectangle exactly.                   
 *     Pass Two:  Extend the bounding box by the value in PixelRange and   
 *                then test for an exact hit within this enlarged rectangle
 *                                                                         
 * INPUT:  sdf:     pointer to the actual object we are considering.       
 *         ItemPtr: pointer to the record which contains the dimensions of 
 *                  the object we are considering.  See (%) below.         
 *         Xtrans, Ytrans:  the amount the object we are examining has     
 *                  been translated by.                                    
 *         PassNum: Either FirstPass or SecondPass.                        
 *         PixelRange:  The amount by which to expand the bounding box of  
 *                  this particular object.                                
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED:  none                                                 
 * CALLED BY: HitSearch                                                    
 * CALLS ON:  KeepTrackMin                                                 
 *                                                                         
 * (%)In most cases sdf and ItemPtr will be the same pointer.  The only 
 * time when these two pointers differ is when we are considering a symbol 
 * call.  Then sdf should be the pointer to the actual symbol call but  
 * ItemPtr will be the pointer to the symbol definition which contains the 
 * dimensions of the bounding box.                                         
 *                                                                         
 **************************************************************************
 */

CheckRectangle(Xhit, Yhit, sdf, ItemPtr,Xtrans,Ytrans,PassNum,PixelRange)
  int Xhit, Yhit;
  register DisplayRecord *sdf;
  DisplayRecord *ItemPtr;
  int Xtrans,Ytrans;
  char PassNum;
  char PixelRange;
{
char directHit;
int height, width;
short edgeset;
int	Sxmin, Sxmax, Symin, Symax;

    edgeset = 0;
    Sxmin = sdf->xmin + Xtrans;
    Sxmax = sdf->xmax + Xtrans;
    Symin = sdf->ymin + Ytrans;
    Symax = sdf->ymax + Ytrans;

    switch (PassNum)
	{
	case FirstPass:
	    directHit = ((Xhit<= Sxmax)&&
			 (Xhit>= Sxmin)&&
			 (Yhit<= Symax)&&
			 (Yhit>= Symin));
	    if (directHit)
		{
		if (Xhit == Sxmax) edgeset |= RightEdge;
		if (Xhit == Sxmin) edgeset |= LeftEdge;
		if (Yhit == Symax) edgeset |= TopEdge;
		if (Yhit == Symin) edgeset |= BottomEdge;
		}
	    break;
	case SecondPass:
	    directHit = ((Xhit<= Sxmax + PixelRange)&&
			 (Xhit>= Sxmin - PixelRange)&&
			 (Yhit<= Symax + PixelRange)&&
			 (Yhit>= Symin - PixelRange));
/*	    if (directHit)
		{
		if (Xhit >= Sxmax) edgeset |= RightEdge;
		if (Xhit <= Sxmin) edgeset |= LeftEdge;
		if (Yhit >= Symax) edgeset |= TopEdge;
		if (Yhit <= Symin) edgeset |= BottomEdge;
		}
 */
	    break;
	default:
	    directHit = FALSE;
	    printf("\nCheckRectangle:  Pass Number not equal to 1 or 2\n");
	    break;
	} /* end switch */
    if (directHit)
	{
	height = abs(sdf->ymax - sdf->ymin);
	width  = abs(sdf->xmax - sdf->xmin);
	KeepTrackMin(ItemPtr,height,width, edgeset);
	}  /* end if directHit */

} /* end CheckRectangle */

/* CheckRefPoint:  similar to CheckRectangle, above, but works only
 * for reference points.  Since it is restricted, it does not require
 * all of the parameters that the general CheckRectangle does.
 */

CheckRefPoint(Xhit, Yhit, sdf,Xtrans,Ytrans)
int Xhit, Yhit;
DisplayRecord *sdf;
int Xtrans,Ytrans;
{
int height, width;
BOOLEAN directHit;

    switch (sdf->type)
	{
	case SDF_HORIZONTAL_REF:
	case SDF_SEL_HORIZ_REF:
	    directHit = (abs(Yhit-(sdf->ymax+sdf->ymin)/2-Ytrans)
				<= REF_POINT_RANGE);
	    break;
	case SDF_VERTICAL_REF:
	case SDF_SEL_VERT_REF:
	    directHit = (abs(Xhit-(sdf->xmax+sdf->xmin)/2-Xtrans)
				<= REF_POINT_RANGE);
	    break;
	default:
	    directHit = FALSE;
	    break;
	} /* end switch */
    if (directHit)
	{
	height = abs(sdf->ymax - sdf->ymin);
	width  = abs(sdf->xmax - sdf->xmin);
	KeepTrackMin(sdf,height,width, 0);
	}  /* end if directHit */

} /* end CheckRefPoint */

/**************************************************************************
 *                                                                         
 *                            CheckLine                                    
 *                                                                         
 * This function is used for determining if a line (either horizontal or   
 * vertical but NOT general) is hit.  It is only called on the first pass. 
 * This is because on the first pass we aren't expecting the line to be    
 * hit exactly so we expand its bounding box by a constant number of pixels
 * at its endpoints and a constant number of pixels at its sides and then  
 * we check for a hit inside this expanded bounding box.  If a line was not
 * hit on the first pass through, we dont need to check it on the second   
 * pass through.                                                           
 *                                                                         
 * INPUT:  sdf:  pointer to the actual object we are considering.       
 *         Xtrans, Ytrans:  the amount the object we are examining has     
 *                  been translated by.                                    
 *         LineType:  either Horizontal or vertical.  This indicates to    
 *                  the function it should interpret the line.             
 * OUTPUT: none                                                            
 * GLOBALS MODIFIED:  none                                                 
 * CALLED BY: HitSearch                                                    
 * CALLS ON:  KeepTrackMin                                                 
 *                                                                         
 **************************************************************************
 */

CheckLine(Xhit, Yhit, sdf, Xtrans, Ytrans, LineType)
int Xhit, Yhit;
register DisplayRecord *sdf;
int Xtrans, Ytrans;
char LineType;

{
char directHit;
int height,width;

switch (LineType)
    {
    case SDF_UpLine:
	if ( Xhit > (sdf->xmax + Xtrans + LINE_END_RANGE) ||
	     Xhit < (sdf->xmin + Xtrans - LINE_END_RANGE) ||
	     Yhit > (sdf->ymax + Ytrans + LINE_END_RANGE) ||
	     Yhit < (sdf->ymin + Ytrans - LINE_END_RANGE) )
	   directHit = 0;
	else
	   directHit = Distance( sdf->xmin+Xtrans, sdf->ymin+Ytrans,
	   		sdf->xmax+Xtrans, sdf->ymax+Ytrans, 
			Xhit, Yhit) <= LINE_SIDE_RANGE;
	break;

    case SDF_DownLine:
	if ( Xhit > (sdf->xmax + Xtrans + LINE_END_RANGE) ||
	     Xhit < (sdf->xmin + Xtrans - LINE_END_RANGE) ||
	     Yhit > (sdf->ymax + Ytrans + LINE_END_RANGE) ||
	     Yhit < (sdf->ymin + Ytrans - LINE_END_RANGE) )
	   directHit = 0;
	else
	   directHit = Distance( sdf->xmin+Xtrans, sdf->ymax+Ytrans,
	   		sdf->xmax+Xtrans, sdf->ymin+Ytrans, 
			Xhit, Yhit) <= LINE_SIDE_RANGE;
	break;

    case SDF_HORIZONTAL_LINE:
	directHit = ((Xhit <= (sdf->xmax + Xtrans + LINE_END_RANGE)) &&
		     (Xhit >= (sdf->xmin + Xtrans - LINE_END_RANGE)) &&
		     (Yhit <= (sdf->ymax + Ytrans + LINE_SIDE_RANGE)) &&
		     (Yhit >= (sdf->ymin + Ytrans - LINE_SIDE_RANGE)));
	break;

    case SDF_VERTICAL_LINE:
	directHit = ((Xhit <= (sdf->xmax + Xtrans + LINE_SIDE_RANGE)) &&
		     (Xhit >= (sdf->xmin + Xtrans - LINE_SIDE_RANGE)) &&
		     (Yhit <= (sdf->ymax + Ytrans + LINE_END_RANGE)) &&
		     (Yhit >= (sdf->ymin + Ytrans - LINE_END_RANGE)));
	break;

    default:
	printf("\nCheckLine:  LineType undefined\n");
	break;
    } /* end switch */

if (directHit)
    {
    /* Now we must compute the height and the width of the line. Normally 
     * lines have width or height zero (depending on which type of line   
     * it is - horiz. or vert.)  But in order for the line to have an     
     * area, this zero dimension must be made = 1.                        
     */

    switch (LineType)
	{
	case SDF_HORIZONTAL_LINE:
	    height = 1;
	    width = abs(sdf->xmax - sdf->xmin);
	    break;

	case SDF_VERTICAL_LINE:
	    height = abs(sdf->ymax - sdf->ymin);
	    width = 1;
	    break;

	case SDF_UpLine:
	case SDF_DownLine:
	    height = (sdf->ymax - sdf->ymin)*(sdf->ymax - sdf->ymin) +
	             (sdf->xmax - sdf->xmin)*(sdf->xmax - sdf->xmin);
	    height = sqrt( (double)height );
	    width = 1;
	    break;

	default:
	    printf("\nCheckLine:  LineType undefined\n");
	    break;
	} /* end switch */

    KeepTrackMin(sdf, height, width, 0);
    } /* end if directHit */
} /* end CheckLine */


static Distance(x1, y1, x2, y2, x, y)
    /*
     * return the distance from the line through (x1,y1) (x2,y2)
     * to the point (x,y).
     * Try to avoid dividing by zero; if a**2 + b**2 is zero,
     * that means we have two identical points????
     */
    {
      int a, b;
      int denom;
      
      a = y2 - y1;
      b = x1 - x2;
      denom = sqrt( (double)(a*a + b*b) );
      if (denom == 0) return( 32767 );
      return( abs( (a*x + b*y + x1*y2 - x2*y1)/denom ) );
    }


/**************************************************************************
 *                                                                         
 *                            HitSearch                                    
 *                                                                         
 **************************************************************************
 */

HitSearch(Xhit, Yhit, sdf, SearchType, Xtrans, Ytrans, PassNum)
int Xhit, Yhit;
register DisplayRecord *sdf;
char SearchType;
int Xtrans, Ytrans;
char PassNum;
{
   /* local variable declarations */
    int symXtrans, symYtrans;
    char directHit;
    DisplayRecord *sdfSym;


    if (sdf == NIL)
	return;
    /* otherwise we scan throught the circularly linked lists until we 
     * have checked all the objects associated with the symbol call  
     * we are currently looking at. 
     */

    while (sdf->type != SDF_SYMBOL_DEF)
	{
	if (sdf->item != 0)
	  {
	    /* We dont want to check this object if its name = 0.  
	    /* This means its not hittable.                        */

	    if (sdf->type == SDF_SYMBOL_CALL)
	      {
	        /* Check if the hit is in the bounding box of the symbol.  
	         * In order to do this for a symbol call, we  must first   
	         * obtain the translation for all the items in the symbol  
	         * definition to follow.  Then we must examine the bounding
	         * box of the symbol definition in its translated form.    
		 */
	        symXtrans = Xtrans + sdf->xmin;
	        symYtrans = Ytrans + sdf->ymin;

	        /* now lets get at the symbol definition */
	        sdfSym = sdf->leftChild;
		if (sdfSym == NIL)
		    return;
		if ((SearchType == JustSymbols)
		                || (SearchType == All))
		    CheckRectangle(Xhit, Yhit, sdfSym, sdf,
				symXtrans, symYtrans, PassNum, SYM_RANGE);
		if (SearchType != JustSymbols)
		    {
		    /* Now check out the expansion on the definition */
		    HitSearch(Xhit, Yhit, sdfSym->rightSib, SearchType,
				symXtrans, symYtrans, PassNum /* TRD */);
		    }
		} /* end if SDF_SYMBOL_CALL */
	    else
		{
		switch (sdf->type)
		    {
		    case SDF_FILLED_RECTANGLE:
			if ((SearchType == JustRects) || (SearchType == All))
			    CheckRectangle(Xhit, Yhit,
					sdf,sdf,Xtrans,
					Ytrans,PassNum,RECT_RANGE);
			break;

		    case SDF_HORIZONTAL_LINE:
			if ((PassNum==FirstPass) && ((SearchType == JustHoriz)
			    || (SearchType==AllLines) || (SearchType == All)))
			    CheckLine(Xhit, Yhit,sdf,Xtrans,
					Ytrans,SDF_HORIZONTAL_LINE);
			break;

		    case SDF_VERTICAL_LINE:
			if ((PassNum==FirstPass) && ((SearchType==JustVerts)
			    || (SearchType==AllLines) || (SearchType == All)))
			    CheckLine(Xhit, Yhit, sdf,
					Xtrans,Ytrans,SDF_VERTICAL_LINE);
			break;

		    case SDF_POINT:
			/* can a point be hit?  */
			break;

		    case SDF_TEXT:
		    case SDF_SIMPLE_TEXT:
			/* we treat Text as a rectangle */
			if ((SearchType == JustText) || (SearchType == All))
			    CheckRectangle(Xhit, Yhit,sdf,
				sdf,Xtrans,Ytrans,PassNum,TEXT_RANGE);
			break;

		    case SDF_UpLine:
    		    case SDF_DownLine:
			if ((PassNum==FirstPass) && (
			    (SearchType==AllLines) || (SearchType == All) ) )
			    CheckLine(Xhit, Yhit, sdf,
					Xtrans,Ytrans,sdf->type);
			break;

		    case SDF_OUTLINE:	/* Never want to find this */
			break;

		    case SDF_VERTICAL_REF:
		    case SDF_SEL_VERT_REF:
			if ((SearchType == JustRefPt) || (SearchType == All))
			    CheckRefPoint(Xhit, Yhit, sdf,
				Xtrans, Ytrans);
			break;

		    case SDF_HORIZONTAL_REF:
		    case SDF_SEL_HORIZ_REF:
			if ((SearchType == JustRefPt) || (SearchType == All))
			    CheckRefPoint(Xhit, Yhit, sdf,
				Xtrans, Ytrans);
			break;

		    case SDF_SPLINE:
			/* Treat splines like rectangles */
			if ((SearchType == JustSplines) || (SearchType == All))
			    CheckRectangle(Xhit, Yhit,sdf,
				sdf,Xtrans,Ytrans,PassNum,SPLINE_RANGE);
			break;

		    case SDF_RASTER:
			if ((SearchType == JustRasters) || (SearchType == All))
			    CheckRectangle(Xhit, Yhit,
					sdf,sdf,Xtrans,
					Ytrans,PassNum,RASTER_RANGE);
			break;

		    default:
			if ((SearchType == JustTheRest) || (SearchType == All))
			    CheckRectangle(Xhit, Yhit,
					sdf,sdf,Xtrans,
					Ytrans,PassNum,OTHER_RANGE);
			break;
		    } /*end switch on sdf->type */
		} /*end else */
	    } /* end if sdf->item != 0 */
	sdf = sdf->rightSib;
	if (sdf == NIL)
	    return;
 	} /* end while */
} /* end HitSearch */


/*************************************************************************
 *                                                                        
 *                         InitSearch                                     
 *                                                                        
 * This function initializes all the global variables and list structures 
 * used in finding the results of a mouse hit.                            
 *                                                                        
 * INPUT:  none                                                           
 * OUTPUT: none                                                           
 * GLOBALS MODIFIED:  MinArea, MinWidth, HitList                          
 * CALLED BY: FindSelectedObject                                            
 * CALLS ON:  none                                                        
 *                                                                        
 *************************************************************************
 */

InitSearch()
{
    MinArea = MAXLONG;  /* initialize the minimum area to be the largest  */
			 /* integer possible so that on the first compar-  */
			 /* ison made, the first real area computed will   */
			 /* automatically become the minimum area.         */
    MinWidth = 0;        /* Can actually be initialized to anything.       */
    HitList.Header = NIL;
    HitList.NumOfElements = 0;
    StoragePtr = Storage;	/* TRD */

}  /* end InitSearch */

/*************************************************************************
 *									  
 *                        FindSelectedObject                                
 *									  
 * This is the main driver routine. 
 */

LISTTYPE FindSelectedObject(sdf, Xhit, Yhit, display, SearchType)
  short sdf;
  int Xhit, Yhit;  /* the coordinates of the mouse hit, already converted 
                   /* to WORLD coordinates.                               */
  short display;
  char SearchType;
{
    LISTTYPE DisambiguateSelection();
    register DisplayRecord *sdfPtr = VGTtable[display].topSymbol;

    InitSearch();
    if (sdfPtr != NIL)
      {
	HitSearch(Xhit, Yhit, sdfPtr->rightSib, SearchType, 0, 0, FirstPass);
	if (HitList.NumOfElements == 0)
	    HitSearch(Xhit, Yhit, sdfPtr->rightSib,SearchType, 0, 0, SecondPass);
      }
    return(DisambiguateSelection(sdf, HitList));
}


/* DisambiguateSelection:  This routine asks the user to disambiguate
 * if the original selection is ambiguous.  When it is called, it assumes
 * that list points to a list of objects of the same area and having the
 * same aspect ratio.  It selects the one with the greatest y-value.  If
 * more than one has the same greatest y-value, then the left-most object
 * is selected.  It there is still ambiguity, the user is consulted.  The
 * value returned is a list -- exactly the same as what FindSelectedObject
 * would return;
 */

LISTTYPE DisambiguateSelection (sdfin, list)
short sdfin;
LISTTYPE list;
{
short ymax, xmin;
MINPTR cons, saveCons;
BOOLEAN duplicateFlag;
char *ans, userPrompt[64], *TtyGetPromptedString ();
register DisplayRecord *sdf;
DisplayRecord *Find();

if (list.NumOfElements < 2)
    return(list);	/* bullet-proofing! */

ymax = -MaxCoord;
cons = list.Header;
while (cons != NIL)
  {
    sdf = Find(sdfin, cons->item);
    if (sdf)
      {
        if (sdf->ymax > ymax)
          {
	    duplicateFlag = FALSE;
	    saveCons = cons;
	    ymax = sdf->ymax;
          }
        else if (sdf->ymax == ymax)
	    duplicateFlag = TRUE;
      }
    cons = cons->next;
  }
if (!duplicateFlag)
  {
    list.Header = saveCons;
    list.NumOfElements = 1;
    return(list);
  }

xmin = MaxCoord;
cons = list.Header;
while (cons != NIL)
  {
    sdf = Find(sdfin, cons->item);
    if (sdf)
     {
      if (sdf->ymax == ymax && sdf->xmin < xmin)
       {
	duplicateFlag = FALSE;
	saveCons = cons;
	xmin = sdf->xmin;
       }
      else if (sdf->ymax == ymax && sdf->xmin == xmin)
  	duplicateFlag = TRUE;
     }
    cons = cons->next;
    }
if (!duplicateFlag)
    {
    list.Header = saveCons;
    list.NumOfElements = 1;
    return(list);
    }

#if 0
/*
 * This is old cruft - see below - GAF
 */

/* To reach this point, there must be more than one object, all of
 * which are identical in length, width, and screen position.  We must
 * cycle through the list and ask the user about each one in turn.
 */

cons = list.Header;
while (cons != NIL)
 {
    sdf = Find(sdfin, cons->item);
    if (sdf && sdf->ymax == ymax && sdf->xmin == xmin)
      {
	BuildPrompt(userPrompt, sdf);
	ans = TtyGetPromptedString(userPrompt, TRUE);
	while (*ans == ' ') ans++;
	if ((*ans == 'y') || (*ans == 'Y'))
	  {
	    list.Header = cons;
	    list.NumOfElements = 1;
	    return(list);
	  }
	}
    cons = cons->next;
    }

/* if we reach this point, the user has rejected all possibilities --
 * just return an empty list.
 */

list.Header = NIL;
list.NumOfElements = 0;
#endif 0

/* Actually, asking the user is a really bad idea as this occurrs so rarely
   that the user never expects the silly prompt in the VGTS window. Simply
   return the first item on the list - GAF */

    TtyMessage( "Ambiguous mousehit selection" );
    list.NumOfElements = 1;
    return(list);
}

#if 0
/* BuildPrompt: builds the prompt to ask the user whether this is
 * \\really// the item she wishes to select.
 */

BuildPrompt(userPrompt, dispRecPtr)
DisplayRecord *dispRecPtr;
char *userPrompt;
{
    char *name;
    short gName;

    switch (dispRecPtr->type)
     {
      case SDF_FILLED_RECTANGLE:
        sprintf(userPrompt, "Select rectangle? (y/n)");
	return;

      case SDF_SYMBOL_CALL:
	dispRecPtr = dispRecPtr->leftChild;
	if (dispRecPtr == NULL || (name = (char *)dispRecPtr->leftChild)==NULL)
	    name = "unknown symbol";
	sprintf(userPrompt, "Select instance of %s? (y/n)", name);
	return;

      case SDF_HORIZONTAL_REF:
      case SDF_VERTICAL_REF:
      case SDF_SEL_HORIZ_REF:
      case SDF_SEL_VERT_REF:
	if ((name = (char *)dispRecPtr->leftChild)==NULL)
	    name = "";
	sprintf(userPrompt, "Select reference line %s? (y/n)", name);
	return;
	
      case SDF_SPLINE:
	sprintf(userPrompt, "Select spline object? (y/n)");
	return;

      case SDF_RASTER:
	sprintf(userPrompt, "Select raster? (y/n)");
	return;

      default:
	sprintf(userPrompt, "Select misc. item? (y/n)" );
    }
}

#endif 0
